/***
 *** Source for PIC programmer
 *** originally: prog84
 ***
 *** Added support for serial programmer, lr 02-apr-96
 ***/

#define EXTERN
#include "prog84.h"

/*** 26-mar-95 Luigi Rizzo: fixed a missing loadconf in programID()
 *** 01-apr-96 Luigi Rizzo: added clear device (-z)
 ***/
int argc;
char **argv;
char **argp;

const char *usage = 
    "Usage:\n"
    "\t-x filename      read hex intel file & program\n"
    "\t-p filename      read pictools format file & program\n"
    "\t-P speed         apply power and run serial port at given speed\n"
    "\t-Q               remove power from the device\n"
    "\t-a               verify after programming each location\n"
    "\t-F               force program even if value is already correct\n"
    "\t-i NN NN NN NN   program ID locations\n"
    "\t-z		fully clear the device (program & config)\n"
    "\t-v               be more verbose\n"
    "\t-h               print this help message & exit\n";

/* types of verification */
#define FORCE  0x1
#define VERIFY_AFTER   0x2

/* Argument processing functions */
long int shiftNum(char *);
FILE *shiftFile(char *);

int picPC; /* what we think the PIC's pc is right now */

int	progMode= 0 ;/* default: program */

typedef char BOOL;
BOOL progUp; /* has the programmer been initialized? */
int verify; /* what kinds of verification we do */
int errors = 0;  /* how many programming errors */
int prog_w=0, read_w=0; /* total programmed and read words */

/* programming routines */
void programID(unsigned int *);
void programPicTools(FILE *);
void programIntel(FILE *);
void programFuse(unsigned int);
void programInit();

int handleSerial(int speed)
{
    static int cycle=0;
    int base, a, d;
    programInit();
    base=lpb.base;
    if (base!=0x3f8 && base!=0x2f8 && base!=0x3e8 && base != 0x2e8) {
	printf("Sorry, no serial port connected\n");
	return 0;
    }
    d = 115200/ speed ;
    printf("Serial port at %d baud (0x%04x)\n", speed, d);
    if (d < 1 || d > 0xffff) {
	printf("sorry, unsupported speed\n");
	return 0;
    }
    out_byte(base+4, 1 ); /* MCR: only set DTR */
    a= 3 ; /* 8 data, 1 stop, no par, ... */
    out_byte(base+3, a | 0x80);
    out_byte(base, d & 0xff);
    out_byte(base+1, (d>>8) & 0xff);
    out_byte(base+3, a);
    for ( ;; ) {
	int c;
	char d[2];
	d[1]='\0';
	d[0]='\0';
	if (!(in_byte(base+5) & 0x1) ) {
	    continue;
	}
	c=in_byte(base);
	if ( (c>=' ' && c<= 0x7f) || c==13 ||c==10 || c==8 || c==9) d[0]=c;
	cycle++;
	if (verbose) printf("rx [%6d] [0x%02x] 0x%02x %s\n", cycle,
		in_byte(base+5), c, d );
	else printf(d);
	fflush(stdout);
    }
}

void programInit()
{
    if (!progUp) {
	progSetup();
	progUp = 1;
    }
    power_and_reset();
    picPC = 0;
    progMode=0;
}

/***
 *** jumps to a given address, possibly resetting the programmer.
 ***/
void
jumpTo(unsigned addr)
{

    /***
     *** requests < 0x2100 mean working on program
     *** > 2100 work on data. Data can only be done if were in data
     *** or previously were reset.
     ***/
    if (    !progUp ||	/*** was not active ***/
	    addr < picPC ||	/*** must go backwards ***/
	    (addr >= 0x2100 && (picPC !=0 && picPC < 0x2100) ) )
	programInit();
    /***
     *** at this point, picPC is certainly <= addr
     ***/
    
    if (addr >= 0x2100 && picPC==0)
	picPC= 0x2100;	/*** doing data ***/
    else if (picPC < 0x2000 && addr >= 0x2000) {
	loadConfiguration(0x3FFF);
	picPC = 0x2000;
    }

    for ( ; picPC < addr; picPC++)
	incAddr();
}

/***
 *** command "x" : clear protection fuse
 ***/
void
ResetPic()
{
    jumpTo(0x2007);
    pipeBits(SC_CMD1,6);
    pipeBits(SC_CMD7,6);
    doProgram();
    pipeBits(SC_CMD1,6);
    pipeBits(SC_CMD7,6);
}

int
main(int largc, char *largv[])
{
    char *cp;
    int i;
    unsigned int id[4];
    unsigned int f;
    FILE *fp;

    verbose = 0;
    argc = largc;
    argv = largv;
    argp = largv + 1;

    read_w=0;
    prog_w=0;
    while ((cp = *(argp++))) {
	if (*cp != '-') {
	    fprintf(stderr, "%s: %s is not an option\n", *argv, cp);
	    exit(1);
	}

	switch(cp[1]) {
	case 'P':
	    progSetup();
	    handleSerial(shiftNum("baud rate"));
	    break;
	case 'Q':
	    programInit();
	    progShutdown();
	    exit(0);
	case 'v':
	    verbose ++;
	    if(verbose > 1) p84_verbose ++;
	    break;
	case 'i':
	    for(i=0; i<4; i++) id[i] = shiftNum("id number");
	    programID(id);
	    break;
	case 'c':
	    f = shiftNum("configuration word");
	    programFuse(f);
	    break;
	case 'z':
	    ResetPic();
	    break;
	case 'x':
	    fp = shiftFile("hex intel format file");
	    programIntel(fp);
	    fclose(fp);
	    break;
	case 'p':
	    fp = shiftFile("pictools format file");
	    programPicTools(fp);
	    fclose(fp);
	    break;
	case 'h':
	    printf(usage);
	    exit(0);
	case 'a':
	    verify |= VERIFY_AFTER;
	    break;
	case 'F':
	    verify |= FORCE;
	    break;
	default:
	    fprintf(stderr, "%s: unknown option \"%s\"\n", *argv, cp);
	    exit(1);
	}
    }

    if (argc < 2) printf(usage);

    if(progUp) progShutdown();

    printf("Read %d cells, programmed %d\n", read_w, prog_w);
    return 0;
}

	/* Programming utilities */

void programID(unsigned int *id)
{
    int i;

    jumpTo(0x2000);

    if (verbose) {
	printf("writing config [");
	fflush(stdout);
    }
    loadConfiguration(id[0]);
    for (i = 0; i < 4; i++) {
	char *s=" ";	/* default message */
	unsigned int w=readProg();
	read_w++;
	printf("ID %d : have 0x%04x, want 0x%04x\n", i, w, id[i]); 
	if (w == id[i] && ! (verify & FORCE) ) {
	} else {
	    loadProg(id[i]); /* needed! XXX */
	    doProgram();
	    if ((verify & VERIFY_AFTER) && (readProg() != id[i])) {
		s="X";
		errors ++;
	    } else {
		s=".";
	    }
	}
	if (verbose) printf(s);
	if (i < 3) {
	    if (verbose) fflush(stdout);
	    incAddr();
	}
    }
    if (verbose) printf("]\n");
    picPC = 0x2003;
}

/* fuses */
#define WDT_FUSE 0x04  /* watchdog timer */
#define PUT_FUSE 0x08  /* power-up timer */
#define CP_FUSE  0x10  /* code protect */
#define FUSE_MASK 0x1F /* only these bits are implemented */

void
programFuse(unsigned int f)
{
    unsigned int w=readProg();

    jumpTo(0x2007);

    if ((w & FUSE_MASK) == (f & FUSE_MASK)) {
	if (verbose) printf("fuses not changed; not re-programmed\n");
	return;
    }

    if (verbose) printf("writing fuses\n");

    loadProg(f);
    doProgram();

    if (verify & VERIFY_AFTER) {
	int ver = readProg();
	if ((ver & FUSE_MASK) != (f & FUSE_MASK)) {
	    fprintf(stderr, "writing fuses: wrote %04x, verified as %04x\n",
		f, ver);
	    errors ++;
	}
    }
}

void
programData(unsigned int addr, int count, unsigned int *buf)
{
    unsigned int w;

again:
    jumpTo(addr);

    if (verbose) {
	printf("%d words at data offset 0x%04x [", count, addr);
	fflush(stdout);
    }
    while (count) {
	char *s=" ";
	w=readData() & 0xff;
	read_w++;
	if ( w == *buf && ! (verify & FORCE) ) {
	} else {
	    if (verbose) printf("0x%04x: want 0x%02x have 0x%02x\n",
			picPC, *buf, w);
	    loadData(*buf);
	    doProgram();
	    s=".";
	}
	if (verbose) {
	    printf(s);
	    fflush(stdout);
	}
	if (readData() & 0xff  != *buf) {
	    printf("\bX");
	    fflush(stdout);
	    errors ++;
	    programInit();
	    goto again;
	}
	incAddr();
	picPC ++;
	count --;
	buf ++;
    }

    if (verbose) { 
	printf("]\n");
    }
}

void
programProg(unsigned int addr, int count, unsigned int *buf)
{
    unsigned int w;

again:
    jumpTo(addr);

    if (verbose) {
	printf("%2d words at offset %04x [", count, addr);
	fflush(stdout);
    }
    while (count) {
	char *s=" ";
	w=readProg();
	read_w++;
	if ( w == *buf && ! (verify & FORCE) ) {
	} else {
	    loadProg(*buf);
	    doProgram();
	    s=".";
	}
	if (verbose) {
	    printf(s);
	    fflush(stdout);
	}
	if (readProg() != *buf) {
	    printf("\bX");
	    fflush(stdout);
	    errors ++;
	    programInit();
	    goto again;
	}
	incAddr();
	picPC ++;
	count --;
	buf ++;
    }

    if (verbose) { 
	printf("]\n");
    }
}

void
programIntel(FILE *fp)
{
    char buf[512];
    unsigned ch, cl, ll, bl, bh, base, len, type;
    unsigned code_buf[32];
    int i;

    while (fgets(buf, 512, fp)) {
	if (verbose) printf("--- read <%s>\n",buf);
	if (buf[0]!=':') { /* print bad line */
	    continue;
	}
	sscanf(buf+1,"%02x%02x%02x%02x",&ll,&bh,&bl, &type);
	len=  (ll) /2;
	base= ((bh<<8) + bl) /2;
	if (base < 0x2100) {
	    /*** normal code ***/
	    for (i=0; i<len; i++) {
		sscanf(buf+9+i*4,"%02x%02x",&cl, &ch);
		code_buf[i]= (ch<<8)+cl;
	    }
	    if (verbose) printf("Writing %d words at addr 0x%04x\n",len, base);
	    if (len>0) programProg(base, len, code_buf);
	} else {
	    for (i=0; i<len; i++) {
		sscanf(buf+9+i*4,"%02x%02x",&cl, &ch);
		code_buf[i]= cl;
	    }
	    if (verbose) printf("Writing %d words at addr 0x%04x\n",len, base);
	    if (len>0) programData(base, len, code_buf);
	}
	if (type == 1) break;
    }
}

void
programPicTools(FILE *fp)
{
    BOOL have_fuses, have_id;
    unsigned id[4];
    unsigned fuses;
    char buf[512];
    unsigned code_addr = 0;
    int i;

    fuses = 0xFFF;

    while (fgets(buf, 512, fp)) {
	switch(*buf) {
	case 0:
	case '\n':
	case '\r':
	case '#':
	    break;
	case 'I':
	    if (sscanf(buf+1, "%i %i %i %i",
		    &id[0], &id[1], &id[2], &id[3]) != 4) {
		fprintf(stderr,"error reading ID numbers\n");
		exit(2);
	    }
	    have_id = 1;
	case 'A':
	    sscanf(buf+1, "%x", &code_addr);
	    break;
	case 'D':
	    {
	    unsigned code_buf[32];
	    int code_count;
	    char *cp;

	    code_count = strtol(buf+1, &cp, 10);
	    if (cp == buf+1) {
		 fprintf(stderr,"error reading data byte count\n");
		 exit(2);
	    }
	    for (i=0; i<code_count; i++)
	        code_buf[i] = strtol(cp, &cp, 16);
	
	    programProg(code_addr, code_count, code_buf);
	    code_addr += code_count;
	    break;
	    }
	case 'T':
	    if (atoi(buf+1) != 84) {
		fprintf(stderr, "I'm a 16C84 programmer, but this file as %s",
		    buf);
		exit(2);
	    }
	    break;
	case 'W':
	    have_fuses = 1;
	    fuses &= ~ WDT_FUSE;
	    if (atoi(buf+1)) fuses |= WDT_FUSE;
	    break;
	case 'U':
	    have_fuses = 1;
	    fuses &= ~ PUT_FUSE;
	    if (atoi(buf+1)) fuses |= PUT_FUSE;
	    break;
	case 'P':
	    have_fuses = 1;
	    fuses &= ~ CP_FUSE;
	    if(atoi(buf+1)) fuses |= CP_FUSE;
	    break;
	case 'C':
	    i = atoi(buf+1);
	    have_fuses = 1;
	    fuses = ( fuses & ~3 ) | ( atoi(buf+1) & 3 );
	    break;
	case 'S':
	    printf("[ignoring pictools file checksum]");
	    break;
	default:
	    printf("warning: unrecognized line: %s", buf);
	    break;
	}
    }

    if (have_id)
	programID(id);

    if (have_fuses)
	programFuse(fuses);
}


/* Arg parsing functions */

long int shiftNum(char *what)
{
    char *end;
    long int val;

    if (!*argp) {
	fprintf(stderr, "%s: arglist ends abruptly\n", *argv);
	exit(1);
    }

    val = strtol(*argp, &end, 0);
    if (end == *argp) {
	fprintf(stderr, "%s: expecting %s, got \"%s\"\n", *argv, what, *argp);
	exit(1);
    }
    argp++;
    return val;
}

FILE *shiftFile(char *what)
{
    FILE *fp;
    extern char *strerror();

    if (!*argp) {
	fprintf(stderr, "%s: arglist ends abruptly\n", *argv);
	exit(1);
    }

    fp = fopen(*argp, "r");
    if (!fp) {
	fprintf(stderr, "%s: can't open %s %s: %s\n",
	*argv, what, *argp, strerror(errno));
	exit(2);
    }
    argp++;
    return fp;
}
