
#ifndef PAC207USB_H
#define PAC207USB_H
/****************************************************************************
#	 	Pixart PAC207BCA library                                    #
# 		Copyright (C) 2005 Thomas Kaiser thomas@kaiser-linux.li     #
#               Copyleft (C) 2005 Michel Xhaard mxhaard@magic.fr            #
#                                                                           #
# This program is free software; you can redistribute it and/or modify      #
# it under the terms of the GNU General Public License as published by      #
# the Free Software Foundation; either version 2 of the License, or         #
# (at your option) any later version.                                       #
#                                                                           #
# This program is distributed in the hope that it will be useful,           #
# but WITHOUT ANY WARRANTY; without even the implied warranty of            #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
# GNU General Public License for more details.                              #
#                                                                           #
# You should have received a copy of the GNU General Public License         #
# along with this program; if not, write to the Free Software               #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA #
#                                                                           #
****************************************************************************/

/*
 * These are data written to the register 0x0001 starting
 * at offset 0x02.. 0x19 and 0x40..0x4b
 * I _think_ these coincide with the individual registers.
 */
enum {
	REG_ID0		= 0x00,	/* sensor identifier */
	REG_ID1		= 0x01,	/* sensor identifier */
	REG_PXCK	= 0x02,	/* pixel clock */
	REG_EXPOSURE	= 0x03,	/* range: 0x12 (fast) 0x24 (slow) */
	REG_BOH4	= 0x04,	/* ??? */
	REG_BRIGHTNESS	= 0x08,	/* brightness */
	REG_CONTRAST	= 0x0e,	/* contrast - also global gain */
	REG_POWER	= 0x0f,	/* power control */
	REG_BIAS	= 0x11,	/* analog bias */
	REG_LOAD	= 0x13,	/* sensor load */
	REG_START	= 0x40,	/* start/stop isoc pipe */
	REG_FORMAT	= 0x41,	/* format, led, compression */
	REG_RATE	= 0x42,	/* rate control ? */
	REG_BAL_SIZE	= 0x4a,	/* balance size (?) */
	REG_RAM_TEST	= 0x4b,	/* sram test value */
};

static __u8 pac207_sensor_init[][8] = {
    {0x10, 0x12, 0x0d, 0x12, 0x0c, 0x01, 0x29, 0xf0},	//2 
  // {0x10, 0x24, 0x06, 0x12, 0x0c, 0x01, 0x29, 0xf0},	//2 increase the times exposure decrease frame rate
    {0x00, 0x64, 0x64, 0x64, 0x04, 0x10, 0xF0, 0x30},	//a reg_10 digital gain Red Green Blue Ggain
    {0x00, 0x00, 0x00, 0x70, 0xA0, 0xF8, 0x00, 0x00},	//12
    {0x00, 0x00, 0x32, 0x00, 0x96, 0x00, 0xA2, 0x02},	//40
                {0x32, 0x00, 0x96, 0x00, 0xA2, 0x02, 0xAF, 0x00},	//42 reg_66 rate control
};

static __u8 PacReg72[] = { 0x00, 0x00, 0x36, 0x00 };	//48 reg_72 Rate Control end BalSize_4a =0x36

/*******************     Camera Interface   ***********************/
static __u16 pac207_getbrightness(struct usb_spca50x *spca50x);
static __u16 pac207_getcontrast(struct usb_spca50x *spca50x);
static void pac207_setbrightness(struct usb_spca50x *spca50x);
static void pac207_setcontrast(struct usb_spca50x *spca50x);
static int pac207_init(struct usb_spca50x *spca50x);
static void pac207_start(struct usb_spca50x *spca50x);
static void pac207_stop(struct usb_spca50x *spca50x);
static int pac207_config(struct usb_spca50x *spca50x);
static void pac207_shutdown(struct usb_spca50x *spca50x);
static void pac207_setAutobright(struct usb_spca50x *spca50x);
/*******************     Camera Private     ***********************/
static void pac207_reg_write(struct usb_device *dev, __u16 index,
			     __u16 value);
static void pac207_reg_read(struct usb_device *dev, __u16 index,
			    __u8 * buffer);

#define pac207RegRead(dev,req,value,index,buffer,length)        \
        usb_rd_vend_int(dev,req,value,index,buffer,length)
#define pac207RegWrite(dev,req,value,index,buffer,length)       \
         usb_wr_vend_int(dev,req,value,index,buffer,length)

/***************************** Implementation ****************************/
/*
 * Probably req = 00 means write a single value at index,
 * 	req = 01 means write the buffer sized.
 */
static void pac207_reg_read(struct usb_device *dev, __u16 index, __u8 * buffer)
{
    pac207RegRead(dev, 0x00, 0x00, index, buffer, 1);
    return;
}

static void pac207_reg_write(struct usb_device *dev, __u16 index, __u16 value)
{
    pac207RegWrite(dev, 0x00, value, index, NULL, 0);
    return;
}

/* push data to the sensor */
static void pac207_push(struct usb_spca50x *spca50x)
{
    pac207_reg_write(spca50x->dev, 0x13, 0x01);	//load registers to sensor (Bit 0, auto clear)
    pac207_reg_write(spca50x->dev, 0x1c, 0x01);	//not documented
}

static __u16 pac207_getbrightness(struct usb_spca50x *spca50x)
{
    __u8 brightness = 0;
    pac207_reg_read(spca50x->dev, 0x0008, &brightness);
    spca50x->brightness = brightness << 8;
    return spca50x->brightness;
}

static void pac207_setbrightness(struct usb_spca50x *spca50x)
{
    __u8 brightness = spca50x->brightness >> 8;

    pac207_reg_write(spca50x->dev, 0x0008, brightness);
    pac207_push(spca50x);
}

static __u16 pac207_getcontrast(struct usb_spca50x *spca50x)
{
    __u8 contrast = 0;
    pac207_reg_read(spca50x->dev, 0x000e, &contrast);
    spca50x->contrast = contrast << 11;
    return spca50x->contrast;
}

static void pac207_setcontrast(struct usb_spca50x *spca50x)
{
    __u8 contrast = spca50x->contrast >> 11;
    pac207_reg_write(spca50x->dev, 0x000e, contrast);
    pac207_push(spca50x);
}

static int pac207_init(struct usb_spca50x *spca50x)
{
    __u8 id[] = { 0, 0 };
    pac207_reg_write(spca50x->dev, 0x41, 0x00);	//Turn of LED
    pac207_reg_read(spca50x->dev, 0x0000, &id[0]);
    pac207_reg_read(spca50x->dev, 0x0001, &id[1]);
    id[0] = ((id[0] & 0x0F) << 4) | ((id[1] & 0xf0) >> 4);
    id[1] = id[1] & 0x0f;
    PDEBUG(2, " Pixart Sensor ID 0x%02X Chips ID 0x%02X !!\n", id[0], id[1]);
    if (id[0] != 0x27 || id[1] != 0x00)
	return -ENODEV;

    return 0;
}

/*
 * Fill the table of formats.
 * This camera has 2 native formats - CIF (mode=0) and QCIF (mode=1),
 * the remaining ones are obtained by cropping.
 */
static void set_pac207SIF(struct usb_spca50x *spca50x)
{
    memset(spca50x->mode_cam, 0x00, TOTMODE * sizeof(struct mwebcam));
    spca50x->mode_cam[CIF].width = 352;
    spca50x->mode_cam[CIF].height = 288;
    spca50x->mode_cam[CIF].t_palette = P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[CIF].pipe = 1023;
    spca50x->mode_cam[CIF].method = 0;
    spca50x->mode_cam[CIF].mode = 0;

    spca50x->mode_cam[QCIF].width = 176;
    spca50x->mode_cam[QCIF].height = 144;
    spca50x->mode_cam[QCIF].t_palette = P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[QCIF].pipe = 1023;
    spca50x->mode_cam[QCIF].method = 0;	/* native */
    spca50x->mode_cam[QCIF].mode = 1;

#if 0
    spca50x->mode_cam[SIF].width = 320;
    spca50x->mode_cam[SIF].height = 240;
    spca50x->mode_cam[SIF].t_palette = P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[SIF].pipe = 1023;
    spca50x->mode_cam[SIF].method = 1;	/* crop */
    spca50x->mode_cam[SIF].mode = 0;

    spca50x->mode_cam[QPAL].width = 192;
    spca50x->mode_cam[QPAL].height = 144;
    spca50x->mode_cam[QPAL].t_palette = P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[QPAL].pipe = 1023;
    spca50x->mode_cam[QPAL].method = 1;	/* crop */
    spca50x->mode_cam[QPAL].mode = 0;

    spca50x->mode_cam[QSIF].width = 160;
    spca50x->mode_cam[QSIF].height = 120;
    spca50x->mode_cam[QSIF].t_palette = P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[QSIF].pipe = 1023;
    spca50x->mode_cam[QSIF].method = 1;
    spca50x->mode_cam[QSIF].mode = 1;
#endif
    return;
}

/*
 * Configurations for the em2820

alt 0 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 0:   0 attr 0x1 interval 1
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 1 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 0:1024 attr 0x1 interval 1	1024
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 2 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 1: 724 attr 0x1 interval 1	1448
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 3 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 1:1024 attr 0x1 interval 1	2048
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 4 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 2: 768 attr 0x1 interval 1	2304
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 5 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 2: 860 attr 0x1 interval 1	2580
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 6 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 2: 964 attr 0x1 interval 1	2892
      [2] endp 4 size 0: 512 attr 0x2 interval 1
alt 7 [0] endp 1 size 0:   1 attr 0x3 interval 11
      [1] endp 2 size 2:1024 attr 0x1 interval 1	3072
      [2] endp 4 size 0: 512 attr 0x2 interval 1

 */

/*** a trace of accesses to the em28xx regs

time	rw	data

	r10	00
	alt5
	r0c	0
	w20	0
	w22	0
	r12	26
	w12	26
	r0c	0
	w0c	0
	r08	f7
	w08	f7
	w06	40
	r0f	07
	w0f	08
	w15	20
	w16	20
	w17	20
	w18	00
	w19	00
	w1a	00
	w23	00
	w24	00
	w26	00
	w13	00
	w12	27
	w0c	10
	w27	34

20584	w10	0d
	r11	00
	w11	00
	r0c	10
	w28	1c
	w29	84
	w2a	14
	w2b	64
	w1c	00
	w1d	00
	w1e	a0	width
	w1f	78	height
	r1b	00
	w1b	00
	r1b	00
	w1b	00
	we0	00 00 00
	we3	00 00 00
	we6	00 00 00
	we9	00 00 00
	wec	00 00 00
	wef	00 00 00
	wf2	00 00 00
	wf5	00 00 00
	wf8	00 00 00
	wfb	00 00 00
	w13	0c
	r0f	08
	w0f	08
	wba	0d 00 01
	r05	00
	wba	0d 00 00
	r05	00

	wba	01 00 08
	r05	00
	wba	02 00 14
	r05	00
	wba	03 01 e0
	r05	00
	wba	04 02 80
	r05	00
	wba	05 00 01

20598	...
20623	r27	34
	w27	34
	r0c	10
	w0c	10
	r12	27
	w12	67
	w22	1c
	w20	13
	r0c	10
	w0c	10
	reset pipe
	get current frame
	setup isoc (3 times)

***/
/* em28xx registers */
#define CHIPID_REG      0x0a
#define USBSUSP_REG     0x0c    /* */

#define AUDIOSRC_REG    0x0e
#define XCLK_REG        0x0f

#define VINMODE_REG     0x10
#define VINCTRL_REG     0x11
#define VINENABLE_REG   0x12    /* */

#define GAMMA_REG       0x14
#define RGAIN_REG       0x15
#define GGAIN_REG       0x16
#define BGAIN_REG       0x17
#define ROFFSET_REG     0x18
#define GOFFSET_REG     0x19
#define BOFFSET_REG     0x1a

#define OFLOW_REG       0x1b
#define HSTART_REG      0x1c
#define VSTART_REG      0x1d
#define CWIDTH_REG      0x1e
#define CHEIGHT_REG     0x1f

#define YGAIN_REG       0x20
#define YOFFSET_REG     0x21
#define UVGAIN_REG      0x22
#define UOFFSET_REG     0x23
#define VOFFSET_REG     0x24
#define SHARPNESS_REG   0x25

#define COMPR_REG       0x26
#define OUTFMT_REG      0x27

#define XMIN_REG        0x28
#define XMAX_REG        0x29
#define YMIN_REG        0x2a
#define YMAX_REG        0x2b

#define HSCALELOW_REG   0x30
#define HSCALEHIGH_REG  0x31
#define VSCALELOW_REG   0x32
#define VSCALEHIGH_REG  0x33

#define AC97LSB_REG     0x40
#define AC97MSB_REG     0x41
#define AC97ADDR_REG    0x42
#define AC97BUSY_REG    0x43

/* em202 registers */
#define MASTER_AC97     0x02
#define VIDEO_AC97      0x14

/* register settings */
#define EM28XX_AUDIO_SRC_TUNER  0xc0
#define EM28XX_AUDIO_SRC_LINE   0x80

#define _W(reg, buf, len)	 em28xx_write_regs(dev, reg, buf, len)
#define _R(reg)	 em28xx_read_reg(dev, reg)

static int em28xx_outfmt_set_yuv422(struct usb_device *dev)
{
	_W(OUTFMT_REG, "\x34", 1);
	_W(VINMODE_REG, "\x10", 1);
	_W(VINCTRL_REG, "\x11", 1);
	return 0;
}

static int em2820_init(struct usb_spca50x *spca50x)
{
    int i;
    struct usb_device *dev = spca50x->dev;

    printf("%s: to be done\n", __FUNCTION__);

    _W(0x08, "\x7d", 1); // reset through GPIO?
    /* Sets I2C speed to 100 KHz */
    _W(0x06, "\x40", 1);
    em28xx_outfmt_set_yuv422(spca50x->dev);
    printf("chipid 0x%x\n", _R(CHIPID_REG) );
    printf("format 0x%x\n", _R(OUTFMT_REG) );
    _W(CWIDTH_REG, "\xa0", 1);
    _W(CHEIGHT_REG, "\x78", 1);
    printf("cwidth 0x%x\n", _R(CWIDTH_REG) );
    printf("cheight 0x%x\n", _R(CHEIGHT_REG) );

    _W(VINENABLE_REG, "\x67", 1);	// video in enable ? */
    _W(0x08, "\xf7", 1); // boh
    _W(0x0c, "\x10", 1); // start?
    _W(0x0c, "\x10", 1); // start?
    i = 0; /* silence compiler */
#if 0
    for (i=0; i < 255; i++) {
	printf("em28xx reg 0x%02x -> 0x%02x\n",
		i, em28xx_read_reg(spca50x->dev, i));
    }
#endif
    return 0;
}

static int em2820_config(struct usb_spca50x *spca50x)
{
    printf("%s: XXX not complete yet\n", __FUNCTION__);
    memset(spca50x->mode_cam, 0x00, TOTMODE * sizeof(struct mwebcam));
    spca50x->mode_cam[CIF].width = 352;
    spca50x->mode_cam[CIF].height = 288;
    spca50x->mode_cam[CIF].t_palette = P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[CIF].pipe = 2580;	/* alt 5 */
    spca50x->mode_cam[CIF].method = 0;
    spca50x->mode_cam[CIF].mode = 0;

    spca50x->mode_cam[VGA].width = 352;
    spca50x->mode_cam[VGA].height = 288;
    spca50x->mode_cam[VGA].t_palette = P_RAW | P_YUV420 | P_RGB32 | P_RGB24 | P_RGB16;
    spca50x->mode_cam[VGA].pipe = 2580;	/* alt 5 */
    spca50x->mode_cam[VGA].method = 0;
    spca50x->mode_cam[VGA].mode = 0;

    return 0;
}

static int pac207_config(struct usb_spca50x *spca50x)
{

    PDEBUG(2, "Find Sensor PAC207");
    PWC_SC(spca50x)->pwc_info.sensor = SENSOR_PAC207;
    set_pac207SIF(spca50x);
    pac207_reg_write(spca50x->dev, 0x41, 0x00);	// 00 Bit_0=Image Format, Bit_1=LED, Bit_2=Compression test mode enable
    pac207_reg_write(spca50x->dev, 0x0f, 0x00);	//Power Control
    pac207_reg_write(spca50x->dev, 0x11, 0x30);	//Analog Bias

    return 0;
}

static void pac207_start(struct usb_spca50x *spca50x)
{

    __u8 buffer;
    __u8 mode;

spca50x->compress=1;
    pac207_reg_write(spca50x->dev, 0x0f, 0x10);	//Power control (Bit 6-0)
    pac207RegWrite(spca50x->dev, 0x01, 0, 0x0002, pac207_sensor_init[0], 8);
    pac207RegWrite(spca50x->dev, 0x01, 0, 0x000a, pac207_sensor_init[1], 8);
    pac207RegWrite(spca50x->dev, 0x01, 0, 0x0012, pac207_sensor_init[2], 8);

    pac207RegWrite(spca50x->dev, 0x01, 0, 0x0040, pac207_sensor_init[3], 8);
    pac207RegWrite(spca50x->dev, 0x01, 0, 0x0042, pac207_sensor_init[4], 8);
    pac207RegWrite(spca50x->dev, 0x01, 0, 0x0048, PacReg72, 4);
    if (spca50x->compress) {
	// pac207_reg_write(spca50x->dev, 0x4a, 0x88);	//Compression Balance size 0x88
	pac207_reg_write(spca50x->dev, 0x4a, 0x88);	//cOMPression Balance size 0x88
    } else {
	pac207_reg_write(spca50x->dev, 0x4a, 0xff);	//Compression Balance size
    }
    pac207_reg_write(spca50x->dev, 0x4b, 0x00);	//Sram test value
    pac207_push(spca50x);

    pac207_reg_write(spca50x->dev, 0x41, 0x02);	//Image Format (Bit 0), LED (Bit 1), Compression test mode enable (Bit 2)

    if (spca50x->mode) {	/* 176x144 */
	pac207_reg_read(spca50x->dev, 0x41, &buffer);
	mode = buffer | 0x01;
	//mode = buffer | 0x00;
	pac207_reg_write(spca50x->dev, 0x41, mode);	//Set mode
	pac207_reg_write(spca50x->dev, 0x02, 0x04);	//PXCK = 12MHz /n
	pac207_reg_write(spca50x->dev, 0x0e, 0x0f);	//PGA global gain (Bit 4-0)
	PDEBUG(1, "pac207_start mode 176x144, mode = %x", mode);
    } else {			/* 352x288 */
	pac207_reg_read(spca50x->dev, 0x41, &buffer);
	mode = buffer & 0xfe;
	pac207_reg_write(spca50x->dev, 0x41, mode);	//Set mode
	if (spca50x->compress) {
	    pac207_reg_write(spca50x->dev, 0x02, 0x04);	//PXCK = 12MHz / n 04
	} else {
	    pac207_reg_write(spca50x->dev, 0x02, 0x0a);	//PXCK = 12MHz / n	/* XXX was 0a */
	}
	// pac207_reg_write(spca50x->dev, 0x0e, 0x04);	//PGA global gain (Bit 4-0) higher=brighter
	pac207_reg_write(spca50x->dev, 0x0e, 0x0f);	//PGA global gain (Bit 4-0)
	PDEBUG(1, "pac207_start mode 352x288, mode = %x", mode);
    }
    pac207_push(spca50x);
    udelay(1000);
    pac207_reg_write(spca50x->dev, 0x40, 0x01);	//Start ISO pipe

//      pac207_setbrightness(spca50x);
    return;
}

static void pac207_stop(struct usb_spca50x *spca50x)
{
    pac207_reg_write(spca50x->dev, 0x40, 0x00);	//Stop ISO pipe
    pac207_reg_write(spca50x->dev, 0x41, 0x00);	//Turn of LED
    pac207_reg_write(spca50x->dev, 0x0f, 0x00);	//Power Control
    return;
}

static void pac207_shutdown(struct usb_spca50x *spca50x)
{
    pac207_reg_write(spca50x->dev, 0x41, 0x00);	//Turn of LED
    pac207_reg_write(spca50x->dev, 0x0f, 0x00);	//Power Control
    return;
}

#ifdef SPCA5XX_ENABLE_REGISTERPLAY
/* XXX i think this code is bigus, does it use global vars ? */
static void pac207_RegRead(struct usb_spca50x *spca50x)
{
    __u8 buffer;
    RegAddress = RegAddress & 0xff;
    pac207_reg_read(spca50x->dev, RegAddress, &buffer);
    RegValue = buffer;
    PDEBUG(0, "pac207_ReadReg, Reg 0x%02X value = %x", RegAddress,
	   RegValue);
    return;
}

static void pac207_RegWrite(struct usb_spca50x *spca50x)
{
    __u8 buffer;

    RegAddress = RegAddress & 0xff;
    buffer = RegValue & 0xff;
    pac207_reg_write(spca50x->dev, RegAddress, buffer);
    pac207_push(spca50x->dev);
    PDEBUG(0, "pac207_WriteReg,Reg 0x%02X value = %x", RegAddress, buffer);
    return;
}
#endif
	
#define BLIMIT(bright) (__u8)((bright>0x1a)?0x1a:((bright < 4)? 4:bright))

static void pac207_setAutobright(struct usb_spca50x *spca50x)
{
    unsigned long flags;
    __u8 luma = 0;
    __u8 luma_mean = 128;
    __u8 luma_delta = 20;
    __u8 spring = 5;
    __u8 Pxclk;
    int Gbright = 0;
    
    
    pac207_reg_read(spca50x->dev, 0x02, &Pxclk);
    Gbright = Pxclk;
    spin_lock_irqsave(&spca50x->v4l_lock, flags);
    luma = spca50x->avg_lum;
    spin_unlock_irqrestore(&spca50x->v4l_lock, flags);
    
    PDEBUG(2, "Pac207 lumamean %d", luma);
    if ((luma < (luma_mean - luma_delta)) ||
	(luma > (luma_mean + luma_delta))) {
	Gbright += ((luma_mean - luma) >> spring);
	Gbright = BLIMIT(Gbright);
	PDEBUG(2, "Pac207 Gbright %d", Gbright);
	pac207_reg_write(spca50x->dev, 0x02,(__u8) Gbright);
	pac207_reg_write(spca50x->dev, 0x13, 0x01);	//load registers to sensor (Bit 0, auto clear)
	pac207_reg_write(spca50x->dev, 0x1c, 0x01);	//not documented
    }
}
#undef BLIMIT

			/* SPCA5XX_ENABLE_REGISTERPLAY */
#endif				// PAC207USB_H
