/* This file is a part of SecureDevice 1.4
   Copyright (C) 1994 by Max Loewenthal and Arthur Helwig
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <dir.h>
#include <time.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>
#include <stdio.h>
#include <conio.h>
#include <alloc.h>
#include "md5.h"
#include "globals.h"
#include "usuals.h"
#include "common.h"

#define MinorName "MKVOLUME"
#define MinorVer "2.01"
#define KEYLEN KeySize/2
#define MAXPWDLEN 100
#define SBUFSIZE 50

#define NOF_SYSERRORS 9

#define errWriteFile 1
#define errReadFile 2
#define errOpenFile 3
#define errSeekFile 4
#define errHideFile 5
#define errEraseFile 6
#define errGetFilePos 7
#define errGetCurDir 8
#define errChangeDir 9
#define errGetDFree 10
#define errCantFindFile 11
#define errGetDPB 12
#define errSectRead 13

#define errPassphraseMismatch 14
#define errWrongPass 15
#define errUserAbort 16
#define errInvalidName 17
#define errInvalidSize 18
#define errSizeTooSmall 19
#define errTooManyClusters 20
#define errNotinRootDir 21
#define errInvalidSSize 22
#define errNotEnoughMem 23

char *errmsg[]={"This is not an error! Should not occur",
                "Error writing data to file",
                "Error reading data from file",
                "Error opening file",
                "Error seeking info file",
                "Error hiding file",
                "Error erasing file",
                "Error getting file position",
                "Error getting current directory",
                "Error changing directory",
                "Error getting diskfree information",
                "Error: Can't find file",
                "Error getting DPB",
                "Error reading sector",
                "Error: Passphrases don't match",
                "Error: Wrong passphrase",
                "Error: Operation aborted by user",
                "Error: Invalid Filename",
                "Error: Invalid size",
                "Error: Size is too small",
                "Error: More than 65530 data clusters",
                "Error: File must be located in the root directory",
                "Error: Target drive must have 512 bytes per sector",
                "Error: Not enough heap memory available"};

extern far void IdeaCFB(void far *iv,void far *key,void far *plain,
         void far *cipher,unsigned len);
extern far void IdeaCFBx(void far *iv,void far *key,void far *cipher,
         void far *plain,unsigned len);

sectortype *sectbuffer;
FILE *F;                               /* The data file to-be-created */
word16 Error=0;

void err(int errnum)
{  if(errnum <= NOF_SYSERRORS)
     perror(errmsg[errnum]);
     else
     printf("%s\n",errmsg[errnum]);
   exit(1);
}

void calcIV(word16 myiv[], word16 iv[], word32 sectnum)
{ myiv[0]=iv[0] ^ low16(sectnum);
  myiv[1]=iv[1] ^ hi16(sectnum);
  myiv[2]=iv[2] ^ low16(sectnum);
  myiv[3]=iv[3] ^ hi16(sectnum);
}


int writesector(int n)

{ if(fwrite(sectbuffer,SectorSize,n,F)!=n)
    Error=errWriteFile;
  return(Error);
}

int crypt_n_write(word16 key[],struct DISK_INFO di)
{ fpos_t Fpos;
  word32 sectnum;
  word16 myiv[4];
  char *dummy;

  if(fgetpos(F,&Fpos))
    {  Error = errGetFilePos;
       return(Error);
    }

  sectnum=(Fpos / SectorSize);

  calcIV (myiv,
          di.IV,
          sectnum);

  IdeaCFB(myiv,key,dummy,dummy,1);
  IdeaCFB(myiv,key,sectbuffer,sectbuffer,1+SectorSize/8);

  return(writesector(1));
}

int getfilename(const char *s,struct FILE_INFO *finfo)
{  int flags;
   strncpy(finfo->fullname,s,sizeof(finfo->fullname));
   strupr(finfo->fullname);
   flags=fnsplit(finfo->fullname,finfo->drive,finfo->dir,
                 finfo->file,finfo->ext);
   if(flags & WILDCARDS || !(flags & FILENAME))
     Error=errInvalidName;
   if(!(flags & DRIVE))
     {  finfo->drive[0]=getdisk()+'A';
        finfo->drive[1]=':';
        finfo->drive[2]='\0';
     }
   if(!(flags & DIRECTORY))
     { finfo->dir[0]='\\';
       if(getcurdir(finfo->drive[0]-'A'+1,&finfo->dir[1]))
         Error=errGetCurDir;
     }
  return(Error);
}

int getsize(const char *s,word32 *size,struct FILE_INFO const finfo)
{  if(!strcmpi(s,"ALL"))
     {  struct dfree diskfree;
        getdfree(finfo.drive[0]-'A'+1,&diskfree);
        if(diskfree.df_sclus == 0xFFFF)
          Error=errGetDFree;
        *size=(word32)diskfree.df_avail*
              diskfree.df_bsec*
              diskfree.df_sclus;
     }
   else
     {  *size=strtoul(s,NULL,0);
        if(*size==0)
          Error=errInvalidSize;
     }
  return(Error);
}

int getdriveinfo(struct FILE_INFO const finfo,struct DISK_INFO *di)
{ struct fatinfo diskinfo;

  if(strcmp(finfo.dir,"\\"))
    Error=errNotinRootDir;

  getfat (finfo.drive[0]-'A'+1,&diskinfo);
  if(diskinfo.fi_bysec != SectorSize)
    Error=errInvalidSSize;

  di->clustsize=diskinfo.fi_sclus;

  return(Error);
}

int calculate (word32 size,struct SETUP_INFO const si,struct DISK_INFO *di)
{ char ok;
  di->totalsects=size / si.sectsize;

  /* There should be room to store 1 BOOT, 1 FAT and the ROOT */
  if(di->totalsects < 1 + si.rootsects + 1 * si.noffats)
    {  Error=errSizeTooSmall;
       return(Error);
    }

  di->sectspfat=0;
  do
  { di->sectspfat++;
    di->datasects = di->totalsects - 1 - si.rootsects - si.noffats*di->sectspfat;
    di->dataclust = di->datasects / di->clustsize;
    di->FAT12     = di->dataclust <= 0xFF6;
    ok = di->dataclust <= (word32)di->sectspfat * si.sectsize * 2 / (di->FAT12?3:4);
  } while(!ok);

  di->IV[0] = rand16;
  di->IV[1] = rand16;
  di->IV[2] = rand16;
  di->IV[3] = rand16;

  if(di->dataclust>0xFFF0)
    Error=errTooManyClusters;

  return(Error);
}

void getpasswd(char *p,word16 maxlen)
{ word16 i=0;
  char c;
  while(i<(maxlen-1))
    {  c=getch();
       switch (c)
         { case 0x0d:
             printf("\n");
             *p='\0';
             return;
           case '\b':
             if(i>0)
               { i--;
                 p--;
                 printf("\b \b");
               }
             break;
           default:
             printf("*");
             *p++=c;
             i++;
       }
    }
}

/*      Compute IDEA encryption subkeys Z */
void en_key_idea(word16 *userkey, word16 *Z)
{ unsigned i,j;
  word16 *Y=Z;
  /*
   * shifts
   */
  for (j=0; j<8; j++)
     Z[j] = *userkey++;

  for (i=0; j<KEYLEN; j++)
  { i++;
    Z[i+7] = Z[i & 7] << 9 | Z[i+1 & 7] >> 7;
    Z += i & 8;
    i &= 7;
  }
  for(i=0;i<KEYLEN;i++)
     Y[i]^=0x0dae;
}        /* en_key_idea */

int getkey(word16 key[])
{  char pass1[MAXPWDLEN];
   char pass2[MAXPWDLEN];
   MD5_CTX md5buf;
   char hash[16];

   printf("Enter passphrase: ");
   getpasswd(pass1,MAXPWDLEN);
   printf("\nRe-enter passphrase: ");
   getpasswd(pass2,MAXPWDLEN);
   if(strcmp(pass1,pass2))
     Error=errPassphraseMismatch;
   MD5Init(&md5buf);
   MD5Update(&md5buf,pass1,strlen(pass1));
   MD5Final(hash,&md5buf);
   burn(pass1);
   burn(pass2);
   en_key_idea ((word16 *)hash, key);
   burn(hash);
   return(Error);
}

int createbootsec(struct SETUP_INFO const si,struct DISK_INFO const di,
                  word16 key[])
{ struct BOOT boot;
  char *dummy;
  word16 myiv[4];
  word16 signature[4];
  int i;
  boot.jmp[0]=0xEB;
  boot.jmp[1]=0;
  boot.jmp[2]=0;
  strncpy(boot.oem,"SECDEV  ",8);
  boot.sectsize    = si.sectsize;
  boot.clustsize   = di.clustsize;
  boot.ressects    = 1;
  boot.fatcnt      = si.noffats;
  boot.rootsize    = si.rootentries;
  boot.totsects    = (di.totalsects<=0xFFFF?di.totalsects:0);
  boot.meddesc     = 0xF8;
  boot.sectspfat   = di.sectspfat;
  boot.sectsptrack = 8;
  boot.nofheads    = 1;
  boot.hidden      = 0;
  boot.totsectsl   = di.totalsects;
  boot.physdrv     = 0;
  boot.reserved    = 0;
  boot.signature   = 0x29;
  boot.serial      = rand32;
  memset(boot.label,' ',sizeof(boot.label));
  if(di.FAT12)
    strncpy(boot.fatid,"FAT12   ",8); else
    strncpy(boot.fatid,"FAT16   ",8);
  memcpy(boot.iv,di.IV,sizeof(di.IV));
  for(i=1;i<4;i++) signature[i]=rand16;
  signature[0]=0x1234;
  for(i=0;i<4;i++) myiv[i]=~di.IV[i];
  IdeaCFB(myiv,key,dummy,dummy,1);
  IdeaCFB(myiv,key,signature,boot.check,1+1);

  burn(sectbuffer[0]);
  memcpy(sectbuffer[0],&boot,sizeof(boot));

  printf("Writing bootsector...\n");

  Error=writesector(1);
  return(Error);
}

int createroot(struct SETUP_INFO si,
               struct DISK_INFO di, word16 key[])
{ int i;
  printf("Writing root directory...\n");
  for(i=0;i<si.rootsects && Error==0;i++)
    { burn(sectbuffer[0]);
      Error=crypt_n_write(key,di);
    }
  return(Error);
}

int createfats(struct SETUP_INFO si,
               struct DISK_INFO di, word16 key[])
{ int i,j;
  printf("Writing FATs...\n");
  for (i=0;i<si.noffats;i++)
    { burn(sectbuffer[0]);
      if(di.FAT12)
        { sectbuffer[0][0]=0xF8;
          sectbuffer[0][1]=0xFF;
          sectbuffer[0][2]=0xFF;
        }
      else
        { sectbuffer[0][0]=0xF8;
          sectbuffer[0][1]=0xFF;
          sectbuffer[0][2]=0xFF;
          sectbuffer[0][3]=0xFF;
        }
      Error=crypt_n_write(key,di);
      for(j=1;j<di.sectspfat && Error==0;j++)
        { burn(sectbuffer[0]);
          Error=crypt_n_write(key,di);
        }
    }
  return(Error);
}

int createdatasects(struct DISK_INFO const di)
{ word32 i;
  int j;
  printf("About to write %ld data sectors...\n",di.datasects);
  for(i=0;i<SBUFSIZE;i++)
    for(j=0;j<SectorSize;j++)
      sectbuffer[i][j]=random(256);

  for(i=0;i<di.datasects && Error==0;)
   { printf("Writing sector %ld\r",i);
     j=SBUFSIZE;
     if((i+j)>di.datasects) j=di.datasects-i;
     Error=writesector(j);
     i+=j;
   }
  printf("Written %ld sectors\n",i);
  return(Error);
}

int makevolumefile(struct FILE_INFO const fi, struct SETUP_INFO const si,
                   struct DISK_INFO const di, word16 key[])
{ if((F=fopen(fi.fullname,"w+b"))==NULL)
    {  Error=errOpenFile;
       return(Error);
    }

  if(!(createbootsec(si,di,key)))
   if(!(createfats(si,di,key)))
    if(!(createroot(si,di,key)))
      createdatasects(di);

  fclose(F);

  return(Error);
}

int readsect(char Drv,word16 Sect)
{ struct { word32 startsect;
           word16 Count;
           char far *DTA;
         } CBlock;
  int error=0;
  if(_osmajor<4)
      asm { mov    al,Drv
            sub    al,'A'
            mov    cx,1
            mov    dx,Sect
            mov    bx,OFFSET sectbuffer
            push   bp
            push   si
            push   di
            int    25h
            pop    ax
            pop    di
            pop    si
            pop    bp
            adc    error,0
          }
      else { CBlock.startsect=Sect;
             CBlock.Count=1;
             CBlock.DTA=&sectbuffer[0][0];
             asm { push ds
                   mov    al,Drv
                   sub    al,'A'
                   mov    cx,-1
                   lea    bx,CBlock
                   push   ss
                   pop    ds
                   push   bp
                   push   si
                   push   di
                   int    25h
                   pop    ax
                   pop    di
                   pop    si
                   pop    bp
                   pop    ds
                   adc    error,0
                 }
           }
  if(error) Error=errSectRead;
  return(Error);
}

int FollowChain(word16 *n,char Drive,word16 StartClust)
{ struct DPB far *dpbptr;
  union REGS regs;
  struct SREGS segregs;
  boolean FAT12;
  int FatInBuf=-1;
  word16 PrevClust=-1,CurClust,Part1,Part2;
  int FatSect,InFat;

  *n=0;
  regs.h.ah = 0x32;
  regs.h.dl = Drive-'A'+1;
  intdosx(&regs, &regs, &segregs);
  if(regs.h.al!=0)
    Error=errGetDPB;

  dpbptr=MK_FP(segregs.ds,regs.x.bx);
  FAT12=(dpbptr->NofClust<=0xFF6);

  CurClust=StartClust;

  if(FAT12)
    while(CurClust<=0xFF6 && Error==0)
      { if(CurClust!=PrevClust+1) (*n)++;
        PrevClust=CurClust;
        FatSect=((long)CurClust*3/2) / dpbptr->sectsize;
        InFat=((long)CurClust*3/2) % dpbptr->sectsize;
        if(FatInBuf!=FatSect)
          Error=readsect(Drive,dpbptr->ReservedSects+FatSect);
        FatInBuf=FatSect;
        Part1=sectbuffer[0][InFat++];
        if(InFat>=dpbptr->sectsize)
          {  InFat=0;
             FatSect++;
             if(FatInBuf!=FatSect && Error==0)
               Error=readsect(Drive,dpbptr->ReservedSects+FatSect);
             FatInBuf=FatSect;
          }
        Part2=sectbuffer[0][InFat];

        if(odd(CurClust))
          CurClust=Part1 >>4 | (Part2 << 4);
        else
          CurClust=Part1 | ((Part2 & 0xF) << 8);
      }
    else
      while(CurClust<=0xFFF6 && Error==0)
        { if(CurClust!=PrevClust+1) (*n)++;
          PrevClust=CurClust;
          FatSect=((long)CurClust*2) / dpbptr->sectsize;
          InFat=((long)CurClust*2) % dpbptr->sectsize;
          if(FatInBuf!=FatSect)
            Error=readsect(Drive,dpbptr->ReservedSects+FatSect);
          FatInBuf=FatSect;
          CurClust=sectbuffer[0][InFat] | (sectbuffer[0][InFat+1] << 8);
        }
  return(Error);
}

int checkfrag (word16 *n,struct FILE_INFO const f)
{ struct fcb FCB;
  struct DTA far *DTAPtr;
  word16 PrevClust,Clust;
  char OldPath[MAXPATH],NewPath[MAXPATH];
  union REGS regs;
  struct SREGS segregs;

  if(getcurdir(f.drive[0]-'A'+1,OldPath))
    { Error=errGetCurDir;
      return(Error);
    }

  strcpy(NewPath,"X:\\");
  NewPath[0]=f.drive[0];
  if(chdir(NewPath))
    { Error = errChangeDir;
      return(Error);
    }

  FCB.fcb_drive = f.drive[0]-'A'+1;
  memset(FCB.fcb_name,' ',sizeof(FCB.fcb_name));
  memset(FCB.fcb_ext,' ',sizeof(FCB.fcb_ext));
  memcpy(FCB.fcb_name,f.file,strlen(f.file));
  if(f.ext[0]=='.')
    memcpy(FCB.fcb_ext,&f.ext[1],strlen(&f.ext[1]));
    else
    memcpy(FCB.fcb_ext,f.ext,strlen(f.ext));

  regs.h.ah = 0x11;
  regs.x.dx = FP_OFF(&FCB);
  segregs.ds = FP_SEG(&FCB);
  intdosx(&regs, &regs, &segregs);
  if(regs.h.al!=0)
    { Error=errCantFindFile;
      return(Error);
    }

  regs.h.ah = 0x2F;
  intdosx(&regs,&regs,&segregs);
  DTAPtr=MK_FP(segregs.es,regs.x.bx);

  Error = FollowChain(n,f.drive[0],DTAPtr->Clust);

  strcpy(NewPath,"X:\\");
  NewPath[0]=f.drive[0];
  strcat(NewPath,OldPath);
  if(chdir(NewPath))
    if(Error==0)
      Error=errChangeDir;
  return(Error);
}

int hidefile (char *fname)
{  if(_dos_setfileattr(fname,FA_HIDDEN | FA_SYSTEM | FA_RDONLY))
     Error = errHideFile;
   return(Error);
}

void summarize(struct SETUP_INFO const si, struct DISK_INFO const di)
{ printf("\nCreating volume:\n");
  printf("Root entries          :%d\n",si.rootentries);
  printf("Root sectors          :%d\n",si.rootsects);
  printf("FAT type              :%d bit\n",(di.FAT12?12:16));
  printf("# Sectors in 1 FAT    :%d\n",di.sectspfat);
  printf("# FAT copies          :%d\n",si.noffats);
  printf("Sectors per cluster   :%d\n",di.clustsize);
  printf("# data clusters       :%lu\n\n",di.dataclust);
}


void givewarning(void)
{  printf("This file is too fragmented for use. If you still attempt to use it,\n\
you'll get a 'General Failure'-error if you try to access it. Unfragment your\n\
hard disk first, then manually make your file hidden, system and readonly.\n");
   printf("Then you can normally use the file.\n");
}

void givehelp(void)
{ printf("Usage: MKVOLUME <filename> <filesize | all>\n\
You will be prompted for a passphrase that will give you access\n\
to the newly created volume.\n\n\
The file that will be created MUST be located in the root directory\n\
of the drive. After creation, it will be marked readonly, hidden and\n\
system file, to prevent it from being accidently deleted, or moved.\n\n");
  exit(0);
}


void main(int argc,char *argv[])
{ struct FILE_INFO finfo;
  struct SETUP_INFO setup_inf={SectorSize,112,7,2};
  struct DISK_INFO disk_inf;
  word16 key[KEYLEN];
  word32 size;
  word16 NofFragments;

  printf("%s %s's %s %s\n",MajorName,MajorVer,MinorName,MinorVer);
  printf("Written by %s\n\n",AuthorName);

  randomize();
  if(argc<3) givehelp();

  if(getfilename(argv[1],&finfo))
     err(Error);
  if(getsize(argv[2],&size,finfo))
     err(Error);
  if(getdriveinfo(finfo,&disk_inf))
     err(Error);
  if(calculate(size,setup_inf,&disk_inf))
     err(Error);
  if(getkey(key))
     err(Error);

  summarize(setup_inf,disk_inf);

  if((sectbuffer=malloc(SBUFSIZE*SectorSize))==NULL)
    { Error=errNotEnoughMem;
      err(Error);
    }

  if(makevolumefile(finfo,setup_inf,disk_inf,key))
     err(Error);

  printf("Ready. Checking fragmentation...\n");

  if(checkfrag (&NofFragments,finfo))
     err(Error);

  printf("File is fragmented into %d pieces.\n\n",NofFragments);
  if(NofFragments>MaxNofParts)
    givewarning();
    else
    {  printf("Hiding file...\n");
       if(hidefile(argv[1]))
         err(Error);
    }

  free(sectbuffer);

  printf("Done! To access this volume: place the name of this file in your CONFIG.SYS\n");
  printf("after the SECDEV.SYS command. Refer to the documentation for more info.\n\n");

}
