#ifndef _syncdir_h_
#define _syncdir_h_

#include "os.h"
#include <setjmp.h>

#include <stdio.h>              
#include <limits.h>

#ifdef _DEBUG
#define UNUSED(x) x
#else
#define UNUSED(x)
#endif

#ifdef FALSE
#undef FALSE
#endif

#ifdef TRUE
#undef TRUE
#endif

#ifdef BOOL
#undef BOOL
#endif

typedef enum
{
  FALSE = 0,
  TRUE
} BOOL;

typedef enum
{
  EXECMODE_NORMAL, /* do not cancel       */
  EXECMODE_CANCEL, /* complete current op */
  EXECMODE_ABORT   /* stop now!           */
} EXECMODE;

void     execModeSet(EXECMODE c);
EXECMODE execModeGet(void);

typedef enum
{
  SYNCDIR_ERROR_NONE = 0, /* none */
  SYNCDIR_ERROR_PARSE,    /* bad parameter to parser */
  SYNCDIR_ERROR_LOGFILE,  /* cannot open/create log file */
  SYNCDIR_ERROR_SOURCE,   /* source not a directory */
  SYNCDIR_ERROR_DEST,     /* destination not a directory */
  SYNCDIR_ERROR_MEMORY,   /* out of memory */
  SYNCDIR_ERROR_TEMP,     /* temp file error */
  SYNCDIR_ERROR_OVERFLOW  /* temp file overflow */
} SYNCDIR_ERROR;

typedef enum
{
  ACTION_NONE,
  ACTION_ADD,
  ACTION_COPY,
  ACTION_DELETE,
  ACTION_UPDATE,
  ACTION_ERROR,   /* source and dest are not both files or directories */
  ACTION_AMBIG    /* source and dest have same date but different sizes, */
                  /*   and '/c' was not used */
} ACTION;

/*
   these are set by the parser
*/
#define OPT_ADD           0x00000001
#define OPT_BOTH          0x00000002
#define OPT_COPY          0x00000004
#define OPT_DELETE        0x00000008
#define OPT_DELETE_DIR    (0x00000010 | OPT_DELETE)
#define OPT_FORCE_ADD     0x00000020
#define OPT_FORCE_COPY    0x00000040
#define OPT_FORCE_DELETE  0x00000080
#define OPT_FORCE_ALL     (OPT_FORCE_ADD | OPT_FORCE_COPY | OPT_FORCE_DELETE)
#define OPT_NOACTION      0x00000100
#define OPT_QUIET         0x00000200
#define OPT_RECURSE       0x00000400
#define OPT_VERBOSE       0x00000800
#define OPT_UPDATE        0x00001000
#define OPT_LOG_ERRORS    0x00002000
#define OPT_RESET_ACCESS  0x00004000
#define OPT_RESET_ARCHIVE 0x00008000

/*
   these are set during execution
*/
#define OPT_NEVER_ADD    0x00010000
#define OPT_NEVER_COPY   0x00020000
#define OPT_NEVER_DELETE 0x00040000

#define MALLOC(type)            (type *) malloc(sizeof(type))
#define VMALLOC(type, sz)       (type *) malloc(sizeof(type) + (sz))
#define REALLOC(type, base, ct) (type *) realloc(base, (ct) * sizeof(type))

/********************************************************************
 **
 ** these must be modified for each system
 **
 ********************************************************************/
typedef struct tagFILEINFO
{
  OS_FILEINFO   osfi;    /* O/S specific file info (defined in os.h)
  /*
     do not touch anything below here!
  */
  int                  refCt;   /* reference count (used by fileInfoDup(...)) */
  struct tagFILETABLE *owner;   /* table to which i'm linked                  */
  BOOL                 isDir;   /* TRUE if this is a directory entry          */
  UINT32               size;    /* file size */
  ACTION               action;  /* what am i to do with this? */
  int                  nameSz;  /* # of bytes in [name] */
  char                 name[1]; /* this must come last */
} FILEINFO;

/********************************************************************
 **
 ** misc. structures
 **
 ********************************************************************/

typedef struct tagACTIONSTATS
{
  UINT32 tCt;   /* total files     */
  UINT32 tSz;   /* total size      */
  UINT32 cCt;   /* completed files */
  UINT32 cSz;   /* completed size  */
} ACTIONSTATS;

typedef struct tagFILETABLE
{
  ACTIONSTATS stAdd;
  ACTIONSTATS stDelete;
  ACTIONSTATS stCopy;

  UINT32      entryCt; /* total # of entries in this table */
  char        name[MAX_PATH];
} FILETABLE;

/********************************************************************
 **
 ** globals
 **
 ********************************************************************/
extern char       includeList[2048];
extern char       excludeList[2048];
extern unsigned   granularity;
extern char       logFileName[MAX_PATH];
extern FILETABLE *source;
extern FILETABLE *dest;
extern FILE      *logFile;
extern jmp_buf    jmpbuf;

/********************************************************************
 **
 ** FILEINFO functions
 **
 ********************************************************************/
FILEINFO       *fileInfoAlloc(FILETABLE *owner, const char *name);
void            fileInfoFree(FILEINFO *inf);
BOOL            fileInfoIsDirectory(const FILEINFO *inf);
BOOL            fileInfoSetAction(FILEINFO *inf, ACTION act);
FILEINFO       *fileInfoDup(FILEINFO *inf);

void            fileInfoUpdateStats(FILEINFO *inf);
/*
  member access functions
*/
UINT32          fileInfoSize(FILEINFO *inf);
BOOL            fileInfoIsDir(FILEINFO *inf);
ACTION          fileInfoAction(FILEINFO *inf);
const char     *fileInfoName(FILEINFO *inf);
const char     *fileInfoFullName(FILEINFO *inf);

/********************************************************************
 **
 ** FILETABLE functions
 **
 ********************************************************************/
FILETABLE       *fileTableAlloc(const char *name);
void             fileTableFree(FILETABLE *tbl);
BOOL             fileTableAdd(FILETABLE *tbl, FILEINFO *inf);
FILEINFO        *fileTableGetFirst(FILETABLE *tbl);
FILEINFO        *fileTableGetNext(FILETABLE *tbl);
/********************************************************************
 **
 ** misc functions
 **
 ********************************************************************/
int         syncdir(int argc, char **argv);
void        optionSet(UINT32 remove, UINT32 add);
BOOL        optionTest(UINT32 flags);
BOOL        optionTestOne(UINT32 flags);
BOOL        isInList(const char *list, const char *text);
BOOL        parse(int argc, char **argv);   /* parse command line */
BOOL        process(void); /* process lists to determine actions */
BOOL        execute(void); /* execute lists */
const char *actionString(ACTION act);

unsigned long elapsed(void); /* elapsed time in seconds */
unsigned long BPS(void);     /* bytes - per - second */

/********************************************************************
 **
 ** logging functions
 **
 ********************************************************************/
#define LOG_CANNOT_GET_FILE_INFO "Cannot get file stats"
#define LOG_CANNOT_SET_FILE_INFO "Cannot set file stats"
#define LOG_USER_CANCEL          "Operation canceled by user"

#define LOG_FN_OPEN   "OPEN"
#define LOG_FN_CREATE "CREATE"
#define LOG_FN_CLOSE  "CLOSE"
#define LOG_FN_READ   "READ"
#define LOG_FN_WRITE  "WRITE"
#define LOG_FN_DELETE "DELETE"
#define LOG_FN_GET_FS "GET FS"
#define LOG_FN_SET_FS "SET FS"
#define LOG_FN_USER   "CANCEL"

void logBegin(void);
void logComplete(void);
void logExecuteBegin(void);
void logExecuteComplete(void);
void logExecuteError(const char *file, const char *fn, const char *msg); /* local message */
void logExecuteAction(const char *src, const char *dst, ACTION act, BOOL exec);

/********************************************************************
 **
 ** UI functions
 ** note : these functions are only called if OPT_QUIET is not set
 **        (except showHelp and queryAction)
 **
 ********************************************************************/
#define HELP_NOT_ENOUGH_ARGS   "Not enough arguments"
#define HELP_NO_SOURCE         "No source directory"
#define HELP_BAD_SWITCH        "Unknown option"
#define HELP_BAD_FORCE_OPTION  "Unknown '/f' suboption"
#define HELP_BAD_DELETE_OPTION "Unknown '/d' suboption"
#define HELP_NO_B_A_D          "When using '/b', only choose one of '/a' or '/d'"
#define HELP_NO_B_C            "'/c' cannot be used with '/b'"
#define HELP_NO_C_U            "Only choose one of '/c' or '/u'"
#define HELP_BAD_GRANULARITY   "Bad granularity"

void showHelp(const char *str);
/*
   this is called during execute if '/f*' not specified
*/
BOOL queryAction(const char *src, const char *dst, ACTION act);
/*
   these are called from the corresponding ``log'' commands
   if OPT_QUIET is not specified
*/
void showBegin(void);
void showComplete(void);
/*
  file lists are being processed
*/
void showProcessBegin(UINT32 amt);
void showProcessUpdate(UINT32 amt, UINT32 complete);
void showProcessComplete(UINT32 amt);
/*
post processing is beginning/continuing
*/
void showPostProcessBegin(const char *dir, UINT32 amt);
void showPostProcessUpdate(const char *dir, UINT32 amt, UINT32 complete);
void showPostProcessComplete(const char *dir, UINT32 amt);
/*
called from the analogous log* commands
*/
void showExecuteBegin(void); /* execute begin     */
void showExecuteError(const char *file, const char *fn, const char *str, const char *logStr);
void showExecuteActionBegin(const char *src, const char *dst, ACTION act);
void showExecuteAction(const char *src, const char *dst, ACTION act, BOOL exec, const char *logStr);
void showExecuteActionComplete(const char *src, const char *dst, ACTION act);
void showExecuteComplete(void);
/*
   these are called from during copy
   if OPT_QUIET is not specified
*/
void showCopyBegin(const char *src, const char *dst, UINT32 toCopy);
void showCopyUpdate(const char *src, const char *dst, UINT32 toCopy, UINT32 amtCopied);
void showCopyComplete(const char *src, const char *dst, UINT32 copied);

/********************************************************************
 **
 ** platform specific functions
 **
 ********************************************************************/
int        sysError(void);

FILETABLE *fileTableInit(const char *dir, BOOL create);

BOOL       directoryScan(FILETABLE *tbl);
BOOL       directoryDelete(const char *name);
BOOL       directoryCreate(const char *name);

BOOL       fileDelete(const char *name);
FILEHANDLE fileOpen(const char *name);
FILEHANDLE fileCreate(const char *name);
BOOL       fileHandleIsValid(FILEHANDLE f);
FILE      *fileTempCreate(void);
BOOL       fileTempDestroy(FILE *f);
UINT32     fileLength(FILEHANDLE f);
UINT32     fileRead(FILEHANDLE f, void *dst, UINT32 len);
UINT32     fileWrite(FILEHANDLE f, const void *src, UINT32 len);
BOOL       fileClose(FILEHANDLE f);

int        fileInfoDateCompare(const FILEINFO *a, const FILEINFO *b);
int        fileInfoNameCompare(const FILEINFO *a, const FILEINFO *b, BOOL partial);

BOOL       fileAttrSet(const char *dst, const FILEATTR *attr);
BOOL       fileAttrGet(const char *src, FILEATTR *attr);
#endif
