/*****************************************************************************
 * FILE: signals.c							     *
 *									     *
 * DESC:								     *
 *	- signal handling						     *
 *	- exception handler						     *
 *	- DPMI 1.0 page fault handler					     *
 *									     *
 * Copyright (C) 1993,1994						     *
 *	Rainer Schnitker, Heeper Str. 283, 33607 Bielefeld		     *
 *	email: rainer@mathematik.uni-bielefeld.de			     *
 *									     *
 *****************************************************************************/

#include <string.h>
#include "PRINTF.H"
#include "DPMI.H"
#include "DPMI10.H"
#include "RMLIB.H"
#include "PROCESS.H"
#include "SIGNALS.H"
#include "START32.H"
#include "CDOSX32.H"
#include "EXCEP32.H"
#include "ADOSX32.H"
#include "COPY32.H"
#include "RSX.H"
#include "LOADPRG.H"
#include "KDEB.H"
#include "DOSERRNO.H"

#define SIGSA(no)   ((no)-1)
#define SIGMASK(no) (1L<<((no)-1))
#define SIGBLOCK    (~(SIGSA(SIGKILL)))

#define SA_NOTBSD   (SA_SYSV | SA_ACK)

/* local functions */
static int exception2signal(WORD);
static int do_signal(DWORD);
static void print_exception_exit(void);
static void print_regs_exception(void);

/* regs after exceptions */
REG386 regf;
extern char *sigtext[];

/*
** give back a signal no from a hardware exception fault no
*/
static int exception2signal(WORD faultno)
{
    int signal;

    switch (faultno) {
    case 0:
    case 2:
    case 4:
    case 5:
    case 6:
    case 8:
    case 10:
    case 15:
	signal = SIGILL;
	break;
    case 1:
    case 3:
	signal = SIGTRAP;
	break;
    case 7:
    case 16:
	signal = SIGFPE;
	break;
    case 9:
    case 11:
    case 12:
    case 13:
    case 14:
    case 17:
	signal = SIGSEGV;
	break;
    default:
	signal = SIGSEGV;
	break;
    }

    return signal;
}

/*
** set signal for one process
*/
int send_signal(NEWPROCESS * p, int signal)
{
    if (!p || signal < 0 || signal >= MAX_SIGNALS)
	return EMX_EINVAL;
    p->sig_raised |= SIGMASK(signal);
    return 0;
}

static void check_pending(int signum)
{
    struct sigaction *p;

    p = signum - 1 + npz->sigaction;
    if (p->sa_handler == SIG_IGN) {
	if (signum == SIGCHLD)
	    return;
	npz->sig_raised &= ~ SIGMASK(signum);
	    return;
    }
    if (p->sa_handler == SIG_DFL) {
	if (signum != SIGCHLD)
	    return;
	npz->sig_raised &= ~ SIGMASK(signum);
	return;
    }
}

int sys_sigaction(int signum, DWORD action, DWORD oldaction)
{
    struct sigaction new_sa, *p;

    if (signum<1 || signum>=MAX_SIGNALS || signum==SIGKILL)
	return -EMX_EINVAL;
    p = signum - 1 + npz->sigaction;
    if (action) {
	if (verify_illegal(npz, action, sizeof(action)))
	    return -EMX_EINVAL;
	cpy32_16(npz->data32sel, action, &new_sa, sizeof(struct sigaction));
	new_sa.sa_mask |= SIGMASK(signum);
	new_sa.sa_mask &= SIGBLOCK;
    }
    if (oldaction) {
	if (verify_illegal_write(npz, oldaction, sizeof(oldaction)))
	    return -EMX_EINVAL;
	cpy16_32(npz->data32sel, oldaction, p, sizeof(struct sigaction));
	}
    if (action) {
	*p = new_sa;
	check_pending(signum);
    }
    return 0;
}

int sys_sigpending(DWORD set_addr)
{
    DWORD set = npz->sig_blocked & npz->sig_raised;

    if (verify_illegal_write(npz, set, 4))
	return -EMX_EINVAL;
    store32(npz->data32sel, set_addr, set);
    return 0;
}

int sys_sigprocmask(int how, DWORD set, DWORD oset)
{
    sigset_t new_set, old_set = npz->sig_blocked;

    if (set) {
	if (verify_illegal(npz, set, sizeof(sigset_t)))
	    return -EMX_EINVAL;
	new_set = read32(npz->data32sel, set) & SIGBLOCK;

	switch (how) {
	    case SIG_BLOCK:
		npz->sig_blocked |= new_set;
		break;
	    case SIG_UNBLOCK:
		npz->sig_blocked &= ~new_set;
		break;
	    case SIG_SETMASK:
		npz->sig_blocked = new_set;
		break;
	    default:
		return -EMX_EINVAL;
	}
    }
    if (oset) {
	if (verify_illegal_write(npz, oset, sizeof(sigset_t)))
	    return -EMX_EINVAL;
	store32(npz->data32sel, oset, old_set);
    }
    return 0;
}

/*
** signal handler returned via syscall 0x10
*/
int signal_handler_returned(void)
{
    DWORD signal, sig_blocked;
    DWORD r;

    signal = read32(npz->data32sel, ESPORG);
    if (opt_printall)
	printf("return signal handler: %lX\n", signal);

    sig_blocked = read32(npz->data32sel, ESPORG + 4);

    /* unblock old signal mask, if not emx ack */
    if (!(npz->sigaction[SIGSA(signal)].sa_flags & SA_ACK))
	npz->sig_blocked = sig_blocked & ~SIGMASK(signal);

    r = read32(npz->data32sel, ESPORG + 8);	/* nop ; mov $7f10, %eax */
    if (r != 0x7F10b890) {
	puts("illegal frame");
	return 1;
    }
    r = read32(npz->data32sel, ESPORG + 12);	 /* int 0x21 */
    if (r != 0x21cd0000) {
	puts("illegal frame");
	return 1;
    }

    cpy32_16(npz->data32sel, ESPORG + 16, &(npz->regs), sizeof(REG386));

    if (AX == 0x7f0E) { 	/* back from raise() */
	EAX = ECX = 0;		/* put return values */
	back_from_syscall();
	return 1;
    }
    /*
    ** if we had a hardware exception, we zero the user handler, to
    ** prevent a exception loop to handler (may be changed in future)
    */
    if (signal == SIGSEGV || signal == SIGILL ||
	signal == SIGFPE || signal == SIGTRAP)
	(npz->sigaction[SIGSA(signal)]).sa_handler = 0;

    return 1;
}

/*
** called before return to user process
*/
int check_signals(void)
{
    do_signal(npz->sig_blocked);
    return 0;
}

#define SIGDFL_IGNORE	    0
#define SIGDFL_TERMINATE    1
#define SIGDFL_CORE	    2

static struct {
    char *text;
    char action;
} sigdfl[MAX_SIGNALS] = {
    { "SIGHUP",     SIGDFL_TERMINATE },
    { "SIGINT",     SIGDFL_TERMINATE },
    { "SIGQUIT",    SIGDFL_CORE },
    { "SIGILL",     SIGDFL_CORE },
    { "SIGTRAP",    SIGDFL_CORE },
    { "SIGABRT",    SIGDFL_CORE },
    { "SIGEMT",     SIGDFL_CORE },
    { "SIGFPE",     SIGDFL_CORE },
    { "SIGKILL",    SIGDFL_TERMINATE },
    { "SIGBUS",     SIGDFL_CORE },
    { "SIGSEGV",    SIGDFL_CORE },
    { "SIGSYS",     SIGDFL_CORE },
    { "SIGPIPE",    SIGDFL_TERMINATE },
    { "SIGALRM",    SIGDFL_TERMINATE },
    { "SIGTERM",    SIGDFL_TERMINATE },
    { "SIGUSR1",    SIGDFL_IGNORE },
    { "SIGUSR2",    SIGDFL_IGNORE },
    { "SIGCHLD",    SIGDFL_IGNORE },
    { "SIG19",      SIGDFL_IGNORE },
    { "SIG20",      SIGDFL_IGNORE },
    { "SIGBREAK",   SIGDFL_TERMINATE }
};

void setup_frame(DWORD address, DWORD oldmask, int signal)
{
    DWORD code;

    if (ESP == ESPORG)		/* build stack-frame,if not */
	ESP -= 12;

    /* save regs for handler return, put reg values on user stack */
    cpy16_32(npz->data32sel, ESP - sizeof(REG386),
	     &(npz->regs), sizeof(REG386));
    ESP -= sizeof(REG386);	/* sub register-frame */

    EIP = address;		/* sighandler address */

    ESP -= sizeof(long);	/* put code: int 0x21 */
    store32(npz->data32sel, ESP, 0x21cd0000);
    ESP -= sizeof(long);	/* put code: popl %eax ; movl $10, %eax */
    store32(npz->data32sel, ESP, 0x7F10b890);
    code = ESP;

    ESP -= sizeof(long);	/* put signal mask on user stack */
    store32(npz->data32sel, ESP, oldmask);
    ESP -= sizeof(long);	/* put signalno on user stack */
    store32(npz->data32sel, ESP, signal);
    ESP -= sizeof(long);	/* put return address: stack code */
    store32(npz->data32sel, ESP, code);

    ESPORG = ESP;
    ESP -= 12;			/* sub iret frame */
}

/*
** check signals settings , change eip to signal handler
*/
static int do_signal(DWORD oldmask)
{
    unsigned long mask = ~npz->sig_blocked;
    unsigned long handler_signal = 0;
    unsigned long signr;
    struct sigaction * sa;

    while ((signr = npz->sig_raised & mask)) {
	__asm__("bsf %2,%1\n\t"
		"btrl %1,%0"
		:"=m" (npz->sig_raised),"=r" (signr)
		:"1" (signr));
	sa = npz->sigaction + signr;
	signr++;
#ifdef CONFIG_KDEB
	if (!opt_kdeb)
#endif
	if ((npz->p_flags & PF_DEBUG) && signr != SIGKILL && signr != SIGCLD) {
	    npz->p_status = PS_STOP;
	    npz->wait_return = (signr << 8) | 127;
	    npz->p_flags |= PF_WAIT_WAIT;
	    npz->pptr->p_status = PS_RUN;
	    switch_context(npz->pptr);
	    if (npz->code32sel == 0) {
		puts("child killed");
		switch_context(npz->pptr);
		shut_down(0);
	    }
	    if (SIGMASK(signr) & npz->sig_blocked) {
		npz->sig_raised |= SIGMASK(signr);
		continue;
	    }
	    sa = npz->sigaction + signr - 1;
	}
	if (sa->sa_handler == SIG_IGN) {
	    unsigned dummy;
	    if (signr != SIGCHLD)
		continue;
	    /* check for SIGCHLD: wait for processes */
	    while (sys_wait(&dummy) > 0)
		;
	    continue;
	}
	if (sa->sa_handler == SIG_DFL) {
	    /* emx ignores SIGCLD, SIGCHLD, SIGUSR */

	    if (sigdfl[SIGSA(signr)].action == SIGDFL_IGNORE)
		continue;
	    else if (sigdfl[SIGSA(signr)].action == SIGDFL_CORE)
		if (!opt_nocore)
		    write_core_file(npz);

	    printf("\nProcess terminated by %s\n", sigdfl[SIGSA(signr)].text);

	    do_exit4c(signr);
	    return 1;
	}
	handler_signal |= 1 << (signr-1);
	mask &= ~sa->sa_mask;
    }

    if (!handler_signal)    /* no handler will be called - return 0 */
	return 0;

    if (opt_printall)
	printf("signals = %lX\n", handler_signal);

    signr = 1;
    sa = npz->sigaction;
    for (mask = 1 ; mask ; sa++,signr++,mask += mask) {
	if (mask > handler_signal)
	    break;
	if (!(mask & handler_signal))
	    continue;
        setup_frame(sa->sa_handler, oldmask | SIGMASK(signr), signr);
	if (sa->sa_flags & SA_SYSV)
	    sa->sa_handler = 0L;
	else
	    npz->sig_blocked |= sa->sa_mask;   /* set blocked */
	oldmask |= sa->sa_mask;
    }
    return 1;
}

long sys_signal(int signum, long handler)
{
    long old_handler;

    if (signum < 1 || signum >= MAX_SIGNALS || signum == SIGKILL)
	return -1;

    old_handler = npz->sigaction[SIGSA(signum)].sa_handler;

    if (handler == SIG_ACK) {
	npz->sig_blocked &= ~ SIGMASK(signum);
	return old_handler;
    }
    else if (handler != SIG_DFL && handler != SIG_IGN)
	if (verify_illegal(npz, handler, 4))
	    return -1;

    npz->sigaction[SIGSA(signum)].sa_handler = handler;
    npz->sigaction[SIGSA(signum)].sa_flags = 0;
    npz->sigaction[SIGSA(signum)].sa_mask = 0;

    if ((npz->uflags & 3) == 1) 	/* system V */
	npz->sigaction[SIGSA(signum)].sa_flags = SA_SYSV;
    else if ((npz->uflags & 3) == 2)	/* BSD */
	npz->sigaction[SIGSA(signum)].sa_flags = 0;
    else				/* old EMX */
	npz->sigaction[SIGSA(signum)].sa_flags = SA_ACK;

    return old_handler;
}

char *exceptext[] =
{
    "division by zero",
    "debug",
    "NMI",
    "breakpoint",
    "overflow",
    "bound check",
    "invalid opcode",
    "copro not availble",
    "double fault",
    "copro exception",
    "invalid TSS",
    "segment not present",
    "stack fault",
    "general protection",
    "page fault",
    "reserved",
    "copro error",
    "alignment error"
};


/*
** this function is called after hardware exceptions
*/

EXCEPTION_10 reg_info;

void myexcep13(void)
{				/* C exception handler */
    int signal;

#ifdef CONFIG_KDEB
    if (opt_kdeb && (WORD) regf.cs == code16sel && regf.faultno == 1) {
	memcpy(&(RSX_PROCESS.regs), &regf, sizeof(REG386));
	return KDEB_debug_handler();
    }
#endif

    if (opt_printall)
	printf("Exception %d\n", (WORD) regf.faultno);

    /* test if we have a error in kernel, abort rsx */
    /* future versions will just terminate the running process */

    if ((WORD) regf.cs == code16sel || (WORD) regf.ds == data16sel) {
	printf("Kernel fault at %X %lX\n", (WORD) regf.cs, regf.eip);
	printf("EAX=%08lX EBX=%08lX ECX=%08lX EDX=%08lX\n"
	       "EBP=%08lX ESP=%08lX  ESI=%08lX EDI=%08lX\n"
	       "CS=%04X DS=%04X ES=%04X SS=%04X\n",
	       regf.eax, regf.ebx, regf.ecx, regf.edx,
	       regf.ebp, regf.esp, regf.esi, regf.edi,
	 (WORD) regf.cs, (WORD) regf.ds, (WORD) regf.es, (WORD) regf.ss);
	if (dpmi10) {
	    printf("cr2 = %lX\n", reg_info.cr2);
	    printf("pte = %lX\n", reg_info.pte);
	}
	printf("User Registers:\n");
	npz->regs.faultno = regf.faultno;
	print_exception_exit();
    }

    /* user fault, copy saved regs to process table */
    memcpy(&(npz->regs), &regf, sizeof(REG386));

    signal = exception2signal(FAULTNO);

    if (signal != SIGTRAP) {
	printf("process %d get hardware fault %d (%s) at %lX\n",
	       npz->pid, FAULTNO, exceptext[FAULTNO], EIP);
	if (opt_printall) {
	    print_regs_exception();
	    if (dpmi10) {
		printf("cr2 = %lX\n", reg_info.cr2);
		printf("offset = %lX\n", reg_info.cr2 - npz->memaddress);
		printf("pte = %lX\n", reg_info.pte);
	    }
	}
    }
    send_signal(npz, signal);

    /* then, check_signal() is called (see excep32.asm) */
}

static void print_regs_exception(void)
{
    printf("selector=%lX  errbits: %X\n"
	   "cs:eip=%04X:%08lX eflags=%08lX\n"
	   "eax=%08lX ebx=%08lX ecx=%08lX edx=%08lX\n"
	   "ebp=%08lX esp=%08lX  esi=%08lX edi=%08lX\n"
	   "cs=%04X ds=%04X es=%04X ss=%04X fs=%04X gs=%04X\n",
	   ERR & ~7L, (WORD) ERR & 7,
	   CS, EIP, EFLAGS,
	   EAX, EBX, ECX, EDX,
	   EBP, ESP, ESI, EDI,
	   CS, DS, ES, SS, FS, GS);
}

static void print_exception_exit()
{
    printf("PROTECTION FAULT  %d :\n", FAULTNO);
    print_regs_exception();
    shut_down(3);
}


/*
** DPMI 1.0 support, damand paging
**
** only called, if start32.c sets page_fault() function
*/

/*
** commit page, if legal address
** page in text, data
** return 1, if real page-fault
*/

static unsigned char pagein_buffer[4096];

int swapper(void)
{
    DWORD offset;
    NEWPROCESS *proc;
    WORD page = 1 + 8;		/* commit & read/write */
    int handle;

    if ((WORD) reg_info.cs == code16sel) {
	/* copy in kernel, find current process */
	for (proc = &FIRST_PROCESS; proc <= &LAST_PROCESS; proc++) {
	    if (!proc->code32sel)
		continue;
	    if ((reg_info.cr2 > proc->memaddress) &&
		(reg_info.cr2 < proc->memaddress + proc->membytes))
		break;
	}
	if (proc > &LAST_PROCESS) {
	    if (opt_printall) {
		puts("swapper: cannot find process");
		printf("pagefault in %04X\n", (WORD) reg_info.cs);
		printf("cr2 %08lX\n", reg_info.cr2);
		printf("pte %X err %X\n", (WORD) reg_info.pte, (WORD) reg_info.error_code);
	    }
	    return 1;
	}
    } else
	proc = npz;

    offset = (reg_info.cr2 - proc->memaddress) & ~0xFFFL;

#if 0
    if (opt_printall) {
	printf("process %d : pagefault in %04X\n", proc->pid, (WORD) reg_info.cs);
	printf("cr2 %08lX, pageoffset %08lX\n", reg_info.cr2, offset);
	printf("pte %X err %X\n", (WORD) reg_info.pte, (WORD) reg_info.error_code);
	printf("memaddress = %lX handle = %lX\n", proc->memaddress, proc->memhandle);
    }
#endif

    if (proc->pid == 0)
	return 1;

    handle = (int) proc->filehandle;

    /* text */
    if (offset >= proc->text_start && offset < proc->text_end) {
	if ((WORD) reg_info.cs != code16sel && (reg_info.error_code & 2))
	    return 1;
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;		/* better:readonly */
	if (handle == 0)	/* forked process */
	    return 0;
	rm_lseek(handle, proc->text_off + (offset - proc->text_start), SEEK_SET);
	if (rm_read(handle, pagein_buffer, 4096) != 4096)
	    return 1;
	cpy16_32(proc->data32sel, offset, pagein_buffer, 4096L);
	page = 1;
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;
	return 0;
    } else
     /* bss */ if (offset >= proc->bss_start && offset < proc->bss_end) {
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;
	if (handle == 0)	/* forked process */
	    return 0;
	bzero32(proc->data32sel, offset, 4096L);
	return 0;
    } else
     /* data */ if (offset >= proc->data_start && offset < proc->data_end) {
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;
	if (handle == 0)	/* forked process */
	    return 0;
	rm_lseek(handle, proc->data_off + (offset - proc->data_start), SEEK_SET);
	if (rm_read(handle, pagein_buffer, 4096) != 4096)
	    return 1;
	cpy16_32(proc->data32sel, offset, pagein_buffer, 4096L);
	return 0;
    } else
     /* heap */ if (offset >= proc->init_brk && offset < proc->brk_value) {
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;
	if (handle == 0)	/* forked process */
	    return 0;
	if (proc->p_flags & PF_DJGPP_FILE)
	    bzero32(proc->data32sel, offset, 4096L);
	return 0;
    } else
     /* stack */ if (offset >= proc->brk_value && offset <= proc->membytes) {
	if (ModifyPageAttributes(proc->memhandle, offset, 1, &page))
	    return 1;
	return 0;
    } else
	return 1;
}
