/*
  ͻ
                                                                          
                     Mountains and Concrete Structures                    
                                                                          
                         by Christopher D. Watkins                        
                                                                          
  ͼ
*/

#include "stdio.h"
#include "stdlib.h"
#include "dos.h"
#include "math.h"
#include "defs.h"
#include "globals.h"
#include "mathb.h"
#include "graphb.h"

typedef char name[33];

FILE *textdiskfile;
name filename;

#define minmountaincoverage 0.85    /*  minimum mountainous coverage in percent  */
#define roughness 280    /*  landscape Roughness (feet) - scale z axis  */
#define numberlevels 3    /*  binary divisions of each axis  */
#define maxres 8    /*  2^NumberLevels  */
#define sc 27.0    /*  scale X axis and Y axis  */
#define perturb 0.0    /*  percentage of SC for X and Y perturbation  */
#define mntrotate 6.0    /*  rotation in degrees  */
#define mnttilt 22.0    /*  tilt in degrees  */

Word xresolution  = 640;
Word yresolution  = 480;
Byte numberofframes  = 1;

 /*     (2^NumberLevels)^(2 dimensions) * (2 triangles per parallelogram)     */

 /*   1 ->   8 triangles   */
 /*   2 ->  32 triangles   */
 /*   3 -> 128 triangles   */
 /*   4 -> 512 triangles   */

void writeheader()
{
  fprintf(textdiskfile,"ͻ\n");
  fprintf(textdiskfile,"                                                                        \n");
  fprintf(textdiskfile,"                   Mountains and Concrete Structures                    \n");
  fprintf(textdiskfile,"                                                                        \n");
  fprintf(textdiskfile,"                       by Christopher D. Watkins                        \n");
  fprintf(textdiskfile,"                                                                        \n");
  fprintf(textdiskfile,"ͼ\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"STATS\n");
  fprintf(textdiskfile,"   XRES    = %d\n",xresolution);
  fprintf(textdiskfile,"   YRES    = %d\n",yresolution);
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"FRAMES\n");
  fprintf(textdiskfile,"   NUMBER  = %d\n",numberofframes);
}

void writestudio()
{
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"ENVIRONMENT\n");
  fprintf(textdiskfile,"   LOCLWGT =    0.750    0.750    0.750\n");
  fprintf(textdiskfile,"   REFLWGT =    0.250    0.250    0.250\n");
  fprintf(textdiskfile,"   TRANWGT =    0.000    0.000    0.000\n");
  fprintf(textdiskfile,"   MINWGT  =    0.030    0.030    0.030\n");
  fprintf(textdiskfile,"   MAXWGT  =    1.000    1.000    1.000\n");
  fprintf(textdiskfile,"   RDEPTH  =    2\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"LAMPS\n");
  fprintf(textdiskfile,"   REFLAMP = FALSE\n");
  fprintf(textdiskfile,"   LAMPREF =    0.000    0.000    0.000\n");
  fprintf(textdiskfile,"   DISTEFF =    0.050\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"OBSERVER\n");
  fprintf(textdiskfile,"   FLENGTH =    3.500\n");
  fprintf(textdiskfile,"   OBSPOS  = -100.000 -700.000  300.000\n");
  fprintf(textdiskfile,"   ROTATE  = %3.3f\n",mntrotate);
  fprintf(textdiskfile,"   TILT    = %3.3f\n",mnttilt);
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"SKY\n");
  fprintf(textdiskfile,"   HORCOL  =    0.300    0.400    0.500\n");
  fprintf(textdiskfile,"   ZENCOL  =    0.700    0.800    1.000\n");
  fprintf(textdiskfile,"   CLOUDS  = FALSE\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"MATERIAL\n");
  fprintf(textdiskfile,"   TYPE    = WATER\n");
  fprintf(textdiskfile,"   TEXTURE = OCEANWAVES\n");
  fprintf(textdiskfile,"   AMBRFL  =    0.100    0.100    0.100\n");
  fprintf(textdiskfile,"   DIFRFL  =    0.100    0.400    0.250\n");
  fprintf(textdiskfile,"   SPCRFL  =    0.100    0.100    0.100\n");
  fprintf(textdiskfile,"   GLOSS   =    8.000\n");
  fprintf(textdiskfile,"   TRANS   =    0.000    0.000    0.000\n");
  fprintf(textdiskfile,"   INDEX   =    1.000\n");
  fprintf(textdiskfile,"   AMPL    =    0.400\n");
  fprintf(textdiskfile,"   PHASE   =    0.000\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"MATERIAL\n");
  fprintf(textdiskfile,"   TYPE    = MOUNTAINGRAY\n");
  fprintf(textdiskfile,"   TEXTURE = GRIT\n");
  fprintf(textdiskfile,"   AMBRFL  =    0.150    0.150    0.150\n");
  fprintf(textdiskfile,"   DIFRFL  =    0.850    0.850    0.850\n");
  fprintf(textdiskfile,"   SPCRFL  =    0.000    0.000    0.000\n");
  fprintf(textdiskfile,"   GLOSS   =    0.000\n");
  fprintf(textdiskfile,"   TRANS   =    0.000    0.000    0.000\n");
  fprintf(textdiskfile,"   INDEX   =    1.000\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"GROUND\n");
  fprintf(textdiskfile,"   MATL    = WATER\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"LAMP\n");
  fprintf(textdiskfile,"   LOC     =  550.000 -550.000  550.000\n");
  fprintf(textdiskfile,"   RADIUS  =  100.000\n");
  fprintf(textdiskfile,"   INTENS  =    0.970    0.970    0.930\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"LAMP\n");
  fprintf(textdiskfile,"   LOC     = -550.000 -550.000  550.000\n");
  fprintf(textdiskfile,"   RADIUS  =  100.000\n");
  fprintf(textdiskfile,"   INTENS  =    0.930    0.900    0.800\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
}

/*
  ͻ
                      Fractal Landscape Generator                         
  ͼ
*/

#define maximumlevel numberlevels    /*  maximum number of levels  */

float height[maxres+1][maxres+1];
float perturbx[maxres+1][maxres+1];
float perturby[maxres+1][maxres+1];

void clearheightbuffer()
{
  int i, j;

  for(i=0; i<=maxres; i++)
  {
    for(j=0; j<=maxres; j++)
	height[i][j]=0.0;
  }
}

void initperturbationbuffer()  /*  added to avoid uniformity of image  */
{
  int i, j;
  float o;

  for(i=0; i<=maxres; i++)
  {
    for(j=0; j<=maxres; j++)
    {
      perturbx[i][j]=0.0;
      perturby[i][j]=0.0;
    }
  }
  o=sc*perturb;
  for(i=0; i<=maxres; i++)
  {
    for(j=0; j<=maxres; j++)
    {
      perturbx[i][j]=Rand()*o;
      perturby[i][j]=Rand()*o;
    }
  }
}

int points, space;

void initialization()
{
  int i;

  randomize();
  points=1;
  for(i=1; i<=numberlevels; i++)
    points=points*2;
  space=1;
  for(i=1; i<=(maximumlevel-numberlevels); i++)
    space=space*2;
}

float hgt(int i, int j)
{
  return height[i * space][j * space];
}

float deviation;
float halfdeviation;
int skip, offset;

void interpolateheight(int i, int j, int i0, int j0, int i1, int j1)
{
  height[i * space][j * space] = (hgt(i0,j0) + hgt(i1,j1)) * 0.5 + Rand() * deviation - halfdeviation;
}

void assignheightsalongx()
{
  int x, y;
  int i, j;

  j = 0;
  do
  {
    i = offset;
    do
    {
      interpolateheight(i,j,i - offset,j,i + offset,j);
      x = points - i;
      y = points - j;
      interpolateheight(x,y,x + offset,y,x - offset,y);
      i = i + skip;
    }
    while (!((i > points)));
    j = j + skip;
  }
  while (!((j > points)));
}

void assignheightsalongy()
{
  int x, y;
  int i, j;

  i = points;
  do
  {
    j = offset;
    do
    {
      interpolateheight(i,j,i,j + offset,i,j - offset);
      x = points - i;
      y = points - j;
      interpolateheight(x,y,x,y - offset,x,y + offset);
      j = j + skip;
    }  while (!((j > i)));
    i = i - skip;
  }
  while (!(!(i > 0)));
}

void assignheightsalongdiagonal()
{
  int x, y;
  int i, j;
  int k, l;

  i = 0;
  do
  {
    j = offset;
    k = points - i;
    do
    {
      l = i + j;
      interpolateheight(l,j,l - offset,j - offset,l + offset,j + offset);
      x = points - l;
      y = points - j;
      interpolateheight(x,y,x + offset,y + offset,x - offset,y - offset);
      j = j + skip;
    }
    while (!((j > k)));
    i = i + skip;
  }
  while (!(!(i < points)));
}

void zerobufferedge()
{
  int i, j;

  for (i = 0; i <= maxres; i++)
  {
    height[i][0] = 0.0;
    height[i][maxres] = 0.0;

  }
  for (j = 0; j <= maxres; j++)
  {
    height[0][j] = 0.0;
    height[maxres][j] = 0.0;
  }
}

void calculatefractalheights()
{
  int n;

  printf("\n");
  halfdeviation = roughness;
  offset = points;
  for (n = 1; n <= numberlevels; n++)
  {
     deviation = halfdeviation;   /*     Roughness / 2^n          */
     halfdeviation = deviation / 2;   /*     Roughness / 2^n / 2      */
     skip = offset;   /*        Points / 2^n          */
     offset = skip / 2;   /*        Points / 2^n / 2      */
     printf("Pass Level %d --> ",n);
     assignheightsalongx();
     zerobufferedge();
     printf("  X");
     assignheightsalongy();
     zerobufferedge();
     printf("  Y");
     assignheightsalongdiagonal();
     zerobufferedge();
     printf("  Diag\n");
   }
   printf("\n");
}

void zeroheightbufferbelowwaterlevel()
{
  int i, j;

  for (i = 0; i <= maxres; i++)
  {
    for (j = 0; j <= maxres; j++)
    {
      if ((height[i][j] < 0.0))
	height[i][j] = 0.0;
    }
  }
}

Boolean done;

void checkpercentagemountain()
{
  int i, j;
  long cnt;
  float coverage;

  cnt = 0;
  for (i = 0; i <= maxres; i++)
  {
    for (j = 0; j <= maxres; j++)
    {
      if ((height[i][j] > 0.0))
	cnt = cnt + 1;
    }
  }
  if ((cnt > Round((SqrFP(maxres - 1) * minmountaincoverage))))
	/*  ^ Since we zero buffer edges to the water level  */
    done = true;
  else
    done = false;
  coverage = (100 * cnt / Round(SqrFP(maxres - 1)));
  printf("Mountainous Coverage = %3.3f%\n",coverage);
}

void writeouttriangles()
{
  long i, j;
  float os;

  os = (maxres / 2) * sc;
  for (i = 0; i <= maxres - 1; i++)
  {
    for (j = 0; j <= maxres - 1; j++)
    {
      fprintf(textdiskfile,"\n");
      fprintf(textdiskfile,"TRIANGLE\n");   /*  0 1 3  */
      fprintf(textdiskfile,"   LOC     = %3.3f %3.3f %3.3f\n",(perturbx[i][j] + i * sc - os),(perturby[i][j] + j * sc - os),height[i][j]);
      fprintf(textdiskfile,"   V1      = %3.3f %3.3f %3.3f\n",(perturbx[i][j] + sc),(perturby[i][j] + 0.0),(height[i + 1][j] - height[i][j]));
      fprintf(textdiskfile,"   V2      = %3.3f %3.3f %3.3f\n",(perturbx[i][j] + 0.0),(perturby[i][j] + sc),(height[i][j + 1] - height[i][j]));
      fprintf(textdiskfile,"   MATL    = MOUNTAINGRAY\n");
      fprintf(textdiskfile,"\n");
      fprintf(textdiskfile,"TRIANGLE\n");   /*  2 3 0  */
      fprintf(textdiskfile,"   LOC     = %3.3f %3.3f %3.3f\n",(perturbx[i][j] + (i + 1) * sc - os),(perturby[i][j] + (j + 1) * sc - os),height[i + 1][j + 1]);
      fprintf(textdiskfile,"   V1      = %3.3f %3.3f %3.3f\n",(perturbx[i][j] - sc),(perturby[i][j] + 0.0),(height[i][j + 1] - height[i + 1][j + 1]));
      fprintf(textdiskfile,"   V2      = %3.3f %3.3f %3.3f\n",(perturbx[i][j] + 0.0),(perturby[i][j] - sc),(height[i + 1][j] - height[i + 1][j + 1]));
      fprintf(textdiskfile,"   MATL    = MOUNTAINGRAY\n");
    }
  }
}

void displaymountain()
{
  Byte i, j;
  TDA pnt1, pnt2;
  Byte col;

  Init_Graphics(56);
  Init_Perspective(true,-65,-45,80,500);
  Init_Plotting(Round(mntrotate),Round(mnttilt));
  Put_Axis_And_Palette(true);
  Axis_And_Palette();
  for (i = 0; i <= maxres; i++)
  {
    for (j = 0; j <= maxres - 1; j++)
    {
      Vec(i * 10,j * 10,height[i][j] / 3.0,pnt1);
      Vec(i * 10,(j + 1) * 10,height[i][j + 1] / 3.0,pnt2);
      if ((height[i][j] == 0.0) && (height[i][j + 1] == 0.0))
	col = 107;
      else
	col = 251;
      Draw_Line_3D(pnt1,pnt2,col);
    }
  }
  for (j = 0; j <= maxres; j++)
  {
    for (i = 0; i <= maxres - 1; i++)
    {
      Vec(i * 10,j * 10,height[i][j] / 3.0,pnt1);
      Vec((i + 1) * 10,j * 10,height[i + 1][j] / 3.0,pnt2);
      if ((height[i][j] == 0.0) && (height[i + 1][j] == 0.0))
	col = 107;
      else
	col = 251;
      Draw_Line_3D(pnt1,pnt2,col);
    }
  }
}

void main()
{
  strcpy(filename,"MOUNT.RT");
  clrscr();
  printf("Generating %s File\n",filename);
  printf("\n");
  printf("By Christopher D. Watkins\n");
  printf("\n");
  printf("\n");
  printf("Writing %d frames\n",numberofframes);
  InitRand(42.116);   /*  selects which mountain  */
  initperturbationbuffer();
  do
  {
    clearheightbuffer();
    initialization();
    initperturbationbuffer();
    calculatefractalheights();
    zeroheightbufferbelowwaterlevel();
    checkpercentagemountain();
  }
  while (!(done));
  Wait_For_Key();
  displaymountain();
  textdiskfile=fopen(filename, "wt");
  writeheader();
  writestudio();
  writeouttriangles();
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"BOUNDINGSPHERETEST\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"BOUNDINGPRISMTEST\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"\n");
  fprintf(textdiskfile,"ENDFRAME\n");
  fclose(textdiskfile);
  Exit_Graphics();
}