COMMENT  */
From Inbar Raz - Chip detection

Most of this code is not originally mine. Some of it belongs to Intel,
some of it to other people whose name I forgot, since this was written
some time ago. Most of my work was coordinating the pieces together.

One missing thing - FPU detection. It's only needed to give a YES/NO
response, so you could use ANY co-processor detection routine instead.
The routine is called FPUType.

This was written originally for PASCAL, but is easily alterable.

Inbar Raz

COMMENT ENDS  /*

; ************************************************************************
;
; Procedure     CPUInfo(InfoRec:Pointer); Far; External;
;
; This routine indentifies the CPU of the system. Also, if the CPU
; supports the CPUID instruction, the routine returns additional CPU data,
; and sets the MSB of CPUType to 1 (or 080h).
;
; CPU Data block comprises of the following:
;
; CPUInfoType     = Record
;                     CPUType    : Byte; ; For Non-CPUID CPUs
;                     Model      : Byte; ; CPUID: Model
;                     Family     : Byte; ; CPUID: Family
;                     Stepping   : Byte; ; CPUID: Stepping
;                     Features   : Word; ; CPUID: Features
;                     Special    : Byte; ; Manufacturer (if determined)
;                   End;
;

; VendorStruc - Space to store the CPUID reply for Genuine Intel Text

VendorStruc     Struc
                        db 12 dup (?)
                Ends

; InfoRecStruc - Format of the data block to fill with information

InfoRecStruc    Struc
                  CPUType       db ?
                  CPUModel      db ?
                  Family        db ?
                  Stepping      db ?
                  Features      dw ?
                  Special       db ?
                Ends

CPUInfo         Proc    Pascal Far InfoRec:DWord

                Local   VendorID: VendorStruc, ID_Flag: Byte
                Local   CPUFound: Byte

; ****************
; * Routine Data *
; ****************

; CPUType values

f8088           EQU     00h
f8086           EQU     01h
fV20            EQU     02h
fV30            EQU     03h
f80188          EQU     04h
f80186          EQU     05h
f80286          EQU     06h
f80386SX        EQU     07h
f80386DX        EQU     08h
f80486SX        EQU     09h
f80486DX        EQU     0Ah
f80586          EQU     0Bh
f80686          EQU     0Ch
fUnknown        EQU     0Eh

; CPU Manufacturer values

fGenuineIntel   EQU     01h
fCyrix          EQU     02h
fAMD            EQU     03h
fUMC            EQU     04h

; CPUID opcode

CPUID           EQU     <dw 0A20Fh>

; CPUID Shifts and Masks

FAMILY_MASK     EQU     00F00h
FAMILY_SHIFT    EQU     8
MODEL_MASK      EQU     0F0h
MODEL_SHIFT     EQU     4
STEPPING_MASK   EQU     0Fh

; ****************
; * Routine Code *
; ****************

                push    bx
                push    cx
                push    dx
                push    di
                push    es
                pushf

                mov     ID_Flag,0               ; Vendor Undetermined

; 1. Seperate 808x/NEC/8018x from later CPUs.
;
;    On the above CPUs, bits 12-15 of the Flags register are always set.

                xor     ax,ax                   ; Zero flags register
                push    ax
                popf
                pushf                           ; Read flags back
                pop     ax
                and     ax,0F000h               ; If b12-b15 set, it's a 286+
                cmp     ax,0F000h
                jnz     ATandUp

; 1.1 Check for 8018x.
;
;     Check for 5-bit shift limit (8018x only).

                mov     bl,f80188

                mov     ax,0FFFFh
                mov     cl,33
                shl     ax,cl
                jnz     FindBusSize             ; Go if 8018x

; 1.2 Check for NEC or 808x.
;
;     Check for ZF activity after MUL (NEC ignores ZF).

                mov     bl,fV20

                xor     al,al                   ; Set ZF
                mov     al,40h                  ; Calc 64^2
                mul     al
                jz      FindBusSize             ; Go if NEC

                mov     bl,f8088                ; Must be an 808x

; 1.3 Determine specific chip (20/30, 88/86).
;
;     On V20 and x88, PIQ length is 6 (8-bit bus).
;     On V30 and x86, PIQ length is 4 (16-bit bus).

FindBusSize:    call    PiqCalc                 ; Find PIQ length
                cmp     dx,6
                je      FBSdone                 ; If PIQ=6, 8-bit bus
                inc     bl                      ; If PIQ=4, 16-bit bus

FBSdone:        jmp     CPUMaker

; 2. Check for 80286.
;
;    Bits 12-15 on the Flags register are always clear on the 80286.

ATandUp:        mov     bl,f80286

                mov     ax,07000h               ; Try to set NT, IOPL
                push    ax
                popf
                pushf                           ; Read it back
                pop     ax
                and     ax,07000h
                jnz     Not286

                jmp     CPUMaker                ; Go if 286

; 3. Check for 80386.
;
;    Test AC bit in EFLAGS (386 won't change).

Not286:         mov     bl,f80386SX

                P386

                mov     edx,esp                 ; Original ESP in EDX
                and     esp,NOT 3               ; Align stack to prevent fault

                pushfd                          ; Original EFLAGS in EAX
                pop     eax
                mov     ecx,eax                 ; EFLAGS => ECX
                xor     ecx,40000h              ; Flip AC flag
                push    ecx                     ; Store it
                popfd
                pushfd                          ; Read it back
                pop     ecx
                push    eax                     ; Restore EFLAGS
                popfd

                mov     esp,edx                 ; Restore ESP

                xor     ecx,eax                 ; Did AC bit flip?
                jne     Not386                  ; If so, 486+

; 3.1 Determine wether this 386 is an SX or a DX.
;
;     See if CR0 bit 4 (ET bit) can be flipped (386SX can't)

                mov     eax,cr0
                mov     ecx,eax                 ; Original CR0 into ECX
                xor     al,10h                  ; Flip bit
                mov     cr0,eax                 ; Store it
                mov     eax,cr0                 ; Read it back
                mov     cr0,ecx                 ; Restore CR0
                cmp     eax,ecx                 ; Did it flip?
                jz      CPUMaker                ; If not, it's a SX
                mov     bl,f80386DX             ; If so, it's a DX
                jmp     CPUMaker

; 4. Check for 80486.
;
;    Check for a NON-EMULATED FPU.

Not386:         mov     bl,f80486SX

                call    cs:FPUType              ; Get CoProcessor
                or      ax,ax
                je      CPUIDTest               ; No FPU - sx

                ; Yes, there IS a co-processor. HOWEVER, is it EMULATED?

                smsw    ax
                test    al,004h
                jne     CPUIDTest               ; Emulation. sx.

                mov     bl,f80486DX

; 5. Check for Pentium and up (including late 486DXs).
;
;    Starting from the Pentium, and a few late 486DXs, the CPUID opcode
;    is available. If bit 21 on the Extended Flags can be flipped, this
;    opcode is supported.

CPUIDTest:

                ; Remember - original flags already in ECX from ^^^^

                mov     eax,ecx                 ; EFLAGS => EAX
                xor     eax,200000h             ; Flip ID bit (1 SHL 21)
                push    eax                     ; Store it
                popfd
                pushfd                          ; Read it back
                pop     eax
                push    ecx                     ; Restore EFLAGS
                popfd

                mov     esp,edx                 ; Restore Stack

                xor     eax,ecx                 ; Test bit 15h
                bts     eax,15h
                jnc     CPUMaker                ; If so, CPUID not available

; 5.1 Retrieve CPUID data.
;
;     Execute CPUID instruction to determine vendor, family, model and
;     stepping.

                mov     CPUFound,bl             ; Save original CPU

; 5.1.1 CPU Vendor
;
;       On CPUID call, when EAX=0, EBX, ECX and EDX return the CPU vendor
;       string. In Intel's case, it's GenuineIntel.

                xor     eax,eax                 ; Set up input for CPUID
                                                ;  instruction

                CPUID                           ; Macro for CPUID instruction

                ; Is it in Intel CPU?

                cmp     ebx,'uneG'
                jne     NotIntel
                cmp     edx,'Ieni'
                jne     NotIntel
                cmp     ecx,'letn'
                jne     NotIntel

                mov     ID_Flag, fGenuineIntel  ; Genuine Intel

                jmp     short ContinueCPUID

NotIntel:

                ; Is it a Cyrix CPU?

                cmp     ebx,'iryC'
                jne     NotCyrix
                cmp     edx,'snIx'
                jne     NotCyrix
                cmp     ecx,'daet'
                jne     NotCyrix

                mov     ID_Flag, fCyrix         ; Cyrix Chip

                jmp     short ContinueCPUID

NotCyrix:

                ; Is it an AMD?

                cmp     ebx,'thuA'
                jne     NotAMD
                cmp     edx,'inte'
                jne     NotAMD
                cmp     ecx,'DMAc'
                jne     NotAMD

                mov     ID_Flag, fAMD

                jmp     short ContinueCPUID

NotAMD:

                ; Is it a UMC?

                cmp     ebx,'UMC '
                jne     NotUMC
                cmp     edx,ebx
                jne     NotUMC
                cmp     ecx,ebx
                jne     NotUMC

                mov     ID_Flag,fUMC

                jmp     short ContinueCPUID

NotUMC:

; 5.1.2 CPU Model, Stepping and Family

ContinueCPUID:  cmp     eax, 1                  ; Make sure 1 is a valid
                                                ;  input value for CPUID
                jl      CPUMaker                ; If not, jump to end

                xor     eax,eax                 ; Otherwise, use as input to
                                                ;  CPUID
                inc     eax                     ; And get stepping, model and
                                                ;  family

                CPUID

                les     di,InfoRec

                mov     es:[di].Stepping, al
                and     es:[di].Stepping, STEPPING_MASK ; isolate stepping info

                and     al, MODEL_MASK          ; isolate model info
                shr     al, MODEL_SHIFT
                mov     es:[di].CPUModel, al

                and     ax, FAMILY_MASK         ; mask everything but family
                shr     ax, FAMILY_SHIFT
                mov     es:[di].Family, al      ; set cpu_type with family

                mov     bl,CPUFound
                or      bl,080h                 ; Signal we've used CPUID

                mov     es:[di].Features,dx     ; save feature flag data

; 6. Identify CPU Manufacturer.
;

CPUMaker:      ; cmp     ID_Flag,0               ; Already identified?
               ; jne     CPUExit

; 6.1 Cyrix test.
;
;     Cyrix CPUs do NOT alter AF on XOR AX,AX, while INTEL's (or
;     compatibles) do.

                lahf
                or      ah,010h                 ; Set Auxilary Carry flag
                sahf

                xor     ax,ax                   ; Clears AF if Intel
                aaa                             ; Side effect: moving AF to
                                                ;  carry flag
                jnc     CPUExit                 ; This is in Intel
                mov     ID_Flag,fCyrix          ; Store CpuMaker return value

CPUExit:        les     di,InfoRec

                mov     es:[di].CPUType,bl
                mov     bh,ID_Flag
                mov     es:[di].Special,bh

                popf
                pop     es
                pop     di
                pop     dx
                pop     cx
                pop     bx

                ret

; ************************************************************************
;
; Prefetch Instruction Queue (PIQ) Measuring Procedure
;
; Returns: DX = Length of PIQ (0..64)
;
; AL, CX, DI destroyed
; Assumes ES = CS
;

PiqCalc         Proc    Near

MaxPIQ          EQU     64                      ; Maximum PIQ to check
OpIncDX         EQU     42h
OpNop           EQU     90h

                jmp     PiqCalcEntry

                ALIGN   16                      ; PIQ must start on para.

PiqStart:

                rep stosb
Piq:            db      MaxPIQ dup (OpIncDX)    ; Original code is INC DX

                ALIGN   2
                sti
                add     dx,2                    ; Adjust for 'REP STOSB'
                retn

PiqCalcEntry:

                cld
                xor     dx,dx                   ; Reset PIQ counter
                mov     cx,MaxPIQ               ; Reset PIQ to "INC DX"s
                mov     di,OFFSET Piq
                mov     al,OpIncDX
                rep     stosb

                mov     cx,MaxPIQ
                mov     di,OFFSET Piq
                mov     al,OpNop                ; Opcode for NOP
                cli
                jmp     PiqStart

PiqCalc         Endp

CPUInfo         Endp

