/***************************************************************************
** rol2snd.c
**
** This program converts the first three voices of an Adlib ROL file (as
** used with Visual Composer) into an AGI sound. At the present time it
** ignores volume settings except for when a voice isn't playing a note
** in which case it sets the attenutation to the maximum level. It adjusts
** for different ticks_in_beat and basic_tempo values, but it does not
** adjust for tempo events or pitch events. This may be supported at a
** later stage if the need arises.
**
** So, my advice to you is keep it simple and hopefully it might work ;)
**
** (c) Lance Ewing 1997, 1998
**
** Version 1.1 update by Anders M Olsson, May 21 1998
***************************************************************************/

#include <stdio.h>
#include <math.h>

#include "roltypes.h"

#define  NOTE_DATA_SIZE  10000

typedef struct {
   int dataLen;
   BYTE *buf;
} noteDataType;

noteDataType noteData[3];

int round(float number)
{
   int retVal;

   retVal = ((((number - (int)number) < 0.5)? floor(number) : ceil(number)));

   return (retVal);
}

int buildNoteData(FILE *input, int noteOffset)
{
   Rol_Header theHeader;
   BYTE2 numTempoEvents, lastNoteTicks, totalDuration, numInsEvents;
   BYTE2 numVolEvents, numPitchEvents, totalConverted, oldTotalConverted;
   int voiceNum, numOfNotes, freqDivisor, retError;
   NoteEvent note;
   BYTE *data;
   float durFactor;

   retError = 0;

   /* For some stupid reason DJGPP seems to want to align every member of
   ** a structure to the next 4-byte boundary. Thus the need to split the
   ** read up into a number of reads. */
   fread(&theHeader, 52, 1, input);
   fread(&theHeader.filler2, sizeof(char), 1, input);
   fread(&theHeader.music_mode, sizeof(char), 1, input);
   fread(&theHeader.filler3, sizeof(theHeader.filler3), 1, input);
   fread(&theHeader.basic_tempo, sizeof(float), 1, input);
   fread(&numTempoEvents, sizeof(BYTE2), 1, input);
   fseek(input, (numTempoEvents*6), SEEK_CUR);
   durFactor = (float)(480*7)/(float)(theHeader.ticks_in_beat *
      theHeader.basic_tempo);

   /* For each voice, store the note data in the relevant buffer */
   for (voiceNum=0; voiceNum<3; voiceNum++) {
      fseek(input, 15, SEEK_CUR);
      fread(&lastNoteTicks, sizeof(BYTE2), 1, input);
      numOfNotes = 0;
      totalDuration = 0;
      oldTotalConverted = totalConverted = 0;
      data = noteData[voiceNum].buf;
      while (totalDuration < lastNoteTicks) {
         fread(&note, sizeof(note), 1, input);
         totalDuration += note.duration;
         totalConverted = round(totalDuration*durFactor);
         *data++ = ((totalConverted - oldTotalConverted) & 0xFF);
         *data++ = (((totalConverted - oldTotalConverted) & 0xFF00) >> 8);
         oldTotalConverted = totalConverted;
         if (note.note) {
            note.note += noteOffset ;
            if (note.note < 45) retError = -1 ;
            freqDivisor = round((float)111860/
                               (440.0*exp((note.note-69)*log(2.0)/12.0)));
         }
         else
            freqDivisor = 0;
         *data++ = ((freqDivisor >> 4) & 0x3F);
         *data++ = (0x80 + ((voiceNum << 5) & 0x70) + (freqDivisor & 0x0F));
         if (freqDivisor == 0)
            *data++ = (0x8F | (((voiceNum << 5) | 0x10) & 0x70));
         else
            *data++ = (0x80 | (((voiceNum << 5) | 0x10) & 0x70));
         numOfNotes++;
      }
      noteData[voiceNum].dataLen = numOfNotes * 5;

      /* Skip the rest of the voice data */
      fseek(input, 15, SEEK_CUR);
      fread(&numInsEvents, sizeof(BYTE2), 1, input);
      fseek(input, (numInsEvents*14) + 15, SEEK_CUR);
      fread(&numVolEvents, sizeof(BYTE2), 1, input);
      fseek(input, (numVolEvents*6) + 15, SEEK_CUR);
      fread(&numPitchEvents, sizeof(BYTE2), 1, input);
      fseek(input, (numPitchEvents*6), SEEK_CUR);
   }
   return (retError) ;
}

void createSound(FILE *output)
{
   BYTE2 offset = 0x0008;
   const BYTE2 endTag = 0xFFFF;

   /* Write offsets to SOUND file */
   fwrite(&offset, sizeof(BYTE2), 1, output);
   offset += (noteData[0].dataLen + 2);
   fwrite(&offset, sizeof(BYTE2), 1, output);
   offset += (noteData[1].dataLen + 2);
   fwrite(&offset, sizeof(BYTE2), 1, output);
   offset += (noteData[2].dataLen + 2);
   fwrite(&offset, sizeof(BYTE2), 1, output);

   /* Write note data to SOUND file*/
   fwrite(noteData[0].buf, sizeof(BYTE), noteData[0].dataLen, output);
   fwrite(&endTag, sizeof(BYTE2), 1, output);
   fwrite(noteData[1].buf, sizeof(BYTE), noteData[1].dataLen, output);
   fwrite(&endTag, sizeof(BYTE2), 1, output);
   fwrite(noteData[2].buf, sizeof(BYTE), noteData[2].dataLen, output);
   fwrite(&endTag, sizeof(BYTE2), 1, output);
   fwrite(&endTag, sizeof(BYTE2), 1, output);
}

void main(int argc, char **argv)
{
   FILE *input, *output;

   if (argc != 3) {
      printf("ROL2SND v1.1\n");
      printf("Usage: rol2snd input.ROL sound-file\n");
   }
   else {
      if ((input = fopen(argv[1], "rb")) == NULL) {
         printf("Could not find given ROL file : %s\n", argv[1]);
         exit(1);
      }
      noteData[0].buf = (BYTE *)malloc(sizeof(char)*NOTE_DATA_SIZE);
      noteData[1].buf = (BYTE *)malloc(sizeof(char)*NOTE_DATA_SIZE);
      noteData[2].buf = (BYTE *)malloc(sizeof(char)*NOTE_DATA_SIZE);
      if ((noteData[0].buf == NULL) || (noteData[0].buf == NULL) ||
          (noteData[0].buf == NULL)) {
         printf("Could not allocate memory for note data.\n");
         exit(1);
      }
      if (buildNoteData(input,0)) {
         printf("MIDI note number below 45 detected.\n");
         printf("Retrying with song shifted up one octave.\n");
         fclose (input);
         if ((input = fopen(argv[1], "rb")) == NULL) {
            printf("Could not re-open given ROL file : %s\n", argv[1]);
            exit(1);
         }
         if (buildNoteData(input,12)) {
            printf("Some notes are still too low.\n");
            printf("Giving up.\n");
            exit(1);
         }
      }

      fclose(input);

      if ((output = fopen(argv[2], "wb")) == NULL) {
         printf("Could not create given SOUND file : %s\n", argv[2]);
         exit(1);
      }
      createSound(output);
      fclose(output);
      free(noteData[0].buf);
      free(noteData[1].buf);
      free(noteData[2].buf);
      printf("Success!\n");
   }
}

