/***************************************************************************
** PICTURE EDITOR
**
** picedit.c
***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <dir.h>

#include "general.h"
#include "agicode.c"

extern byte *vgaScreen;

typedef struct { int left, top, right, bottom; } rect;

/* Status Line and Interface Variables */

char tool[10], vColour, pColour;
int mouseX, mouseY;
const char hexNumbers[16] = "0123456789ABCDEF";
int brushSize, brushShape, brushTexture;
boolean priorityScreen=FALSE;
boolean brushActive=TRUE;

#define  CIRCLE  0
#define  SQUARE  1
#define  SPRAY   0
#define  SOLID   1

void updateScreen()
{
   if (priorityScreen) {
      showPriority();
   }
   else {
      showPicture();
   }
}

void drawStatusLine(char *t, char vC, char pC, int x, int y)
{
   char statusLine[41];

   strcpy(tool, t);
   vColour = vC;
   pColour = pC;
   mouseX = x;
   mouseY = y;

   sprintf(statusLine, "Tool:%-6s V:%-2c P:%-2c X=%-3d Y=%-3d Pri=%-2d",
	   tool, vColour, pColour, mouseX>>1, mouseY-9, ((mouseY-9)/12)+1);
   drawString(statusLine, 0, 0, 0, 15);
}

boolean justEntered = TRUE;

void drawMenuBar(int barOption, int itemOption)
{
   static int oldBar=-1;

   if (justEntered) { oldBar = -1; justEntered = FALSE; }

   if (oldBar != barOption)
      drawString(" Info File View Special                 ", 0, 0, 0, 15);

   switch (barOption) {
      case 0:
	 drawString("Info", 8, 0, 15, 0);
	 if (barOption != oldBar) {
	   drawFilledBox(0, 8, 55, 39, 15);
	   drawBox(2, 9, 53, 38, 0);
	   drawBox(3, 9, 52, 38, 0);
	 }
	 drawString("About", 8, 16, 0, 15);
	 drawString("Help", 8, 24, 0, 15);
	 switch (itemOption) {
	    case 0:
	       drawString("About", 8, 16, 15, 0);
	       break;
	    case 1:
	       drawString("Help", 8, 24, 15, 0);
	       break;
	 }
	 break;
      case 1:
	 drawString("File", 48, 0, 15, 0);
	 if (barOption != oldBar) {
	    drawFilledBox(40, 8, 151, 55, 15);
	    drawBox(42, 9, 149, 54, 0);
	    drawBox(43, 9, 148, 54, 0);
	 }
	 drawString("New Picture", 48, 16, 0, 15);
	 drawString("Save Picture", 48, 24, 0, 15);
	 drawString("Load Picture", 48, 32, 0, 15);
	 drawString("Quit", 48, 40, 0, 15);
	 switch (itemOption) {
	    case 0:
	       drawString("New Picture", 48, 16, 15, 0);
	       break;
	    case 1:
	       drawString("Save Picture", 48, 24, 15, 0);
	       break;
	    case 2:
	       drawString("Load Picture", 48, 32, 15, 0);
	       break;
	    case 3:
	       drawString("Quit", 48, 40, 15, 0);
	       break;
	 }
	 break;
      case 2:
	 drawString("View", 88, 0, 15, 0);
	 /* View\Edit data option */
	 if (barOption != oldBar) {
	    drawFilledBox(80, 8, 167, 32, 15);
	    drawBox(82, 9, 165, 31, 0);
	    drawBox(83, 9, 164, 31, 0);
	 }
	 drawString("View Data", 88, 16, 0, 15);
	 switch (itemOption) {
	    case 0:
	       drawString("View Data", 88, 16, 15, 0);
	       break;
	 }
	 break;
      case 3:
	 drawString("Special", 128, 0, 15, 0);
	 /* Show background image (for overdrawing) */
	 if (barOption != oldBar) {
	    drawFilledBox(120, 8, 223, 32, 15);
	    drawBox(122, 9, 221, 31, 0);
	    drawBox(123, 9, 220, 31, 0);
	 }
	 drawString("Background", 136, 16, 0, 15);
	 switch (itemOption) {
	    case 0:
	       drawString("Background", 136, 16, 15, 0);
	       if (backEnabled) drawChar(7, 119, 16, 4, 15);
	       break;
	 }
	 break;
   }

   oldBar=barOption;
}

#define   LEAVE    255
#define   QUIT     0x13
#define   NEWPIC   0x10
#define   ABOUT    0x00
#define   HELP     0x01
#define   SAVEPIC  0x11
#define   LOADPIC  0x12
#define   VIEWDATA 0x20
#define   BACKGRND 0x30

byte menuSystem()
{
   int menuBarOption=0, menuItemOption=0, ch, maxItem=2, minItem=0;
   boolean stillInMenu=TRUE;
   char maxItems[] = { 2, 4, 1, 1 };

   justEntered = TRUE;

   drawMenuBar(menuBarOption, menuItemOption);

   while (stillInMenu) {
      ch=getch();

      switch (ch) {
	 case 0x00:
	    ch=getch();
	    switch (ch) {
	       case 'H':
		  menuItemOption--;
		  if (menuItemOption < 0) menuItemOption=maxItem-1;
		  break;
	       case 'K':
		  menuBarOption--;
		  if (menuBarOption < 0) menuBarOption=3;
		  maxItem = maxItems[menuBarOption];
		  menuItemOption=0;
		  if (priorityScreen) {
		     showPriority();
		  }
		  else {
		     showPicture();
		  }
		  memset(MK_FP(0xA000, 8*320), 1, 320);
		  break;
	       case 'M':
		  menuBarOption++;
		  if (menuBarOption > 3) menuBarOption=0;
		  maxItem = maxItems[menuBarOption];
		  menuItemOption=0;
		  if (priorityScreen) {
		     showPriority();
		  }
		  else {
		     showPicture();
		  }
		  memset(MK_FP(0xA000, 8*320), 1, 320);
		  break;
	       case 'P':
		  menuItemOption++;
		  if (menuItemOption == maxItem) menuItemOption=0;
		  break;
	    }
	    break;

	 case 0x1B:
	    stillInMenu=FALSE;
	    break;

	 case 0x0D:
	    return ((menuBarOption*0x10) + menuItemOption);
	    break;
      }

      drawMenuBar(menuBarOption, menuItemOption);
   }

   return LEAVE;
}

/**************************************************************************
** showPos
**
** Purpose: To display the current buffer position in the relevant field
** on the interface. This functions also displays the next six PICTURE
** codes if the buffer position is somewhere other than at the end of the
** picture. This is handy when deleting an action.
**************************************************************************/
void showPos()
{
   char tempString[6], count;
   struct picCodeNode *temp;

   hideCursor();

   drawFilledBox(230, 179, 273, 187, 0);
   sprintf(tempString, "%5d", bufPos);
   drawString(tempString, 230, 180, 7, 0);
   strcpy(tempString, "");

   drawFilledBox(210, 189, 317, 197, 0);

   for (temp=picPos, count=0; ((count<6)&&(temp!=NULL)); count++, temp=temp->next) {
      sprintf(tempString, "%02X", temp->node);
      if (temp->node >= 0xF0)
	 drawString(tempString, 211+(count*16), 190, 4, 0);
      else
	 drawString(tempString, 211+(count*16), 190, 7, 0);
   }

   showCursor();
}

/**************************************************************************
** updateBrushPanel
**************************************************************************/
void updateBrushPanel()
{
   int circleColour, squareColour, solidColour, sprayColour;

   hideCursor();

   if (brushActive) {

      drawChar(24, 193, 180, 8, 7);
      drawChar(25, 193, 190, 8, 7);
      normline(201, 188, 209, 188, 8);
      normline(200, 178, 200, 198, 8);
      drawFilledBox(189, 179, 199, 197, 0);

      circleColour = ((brushShape == CIRCLE)? 15 : 8);
      squareColour = ((brushShape == SQUARE)? 15 : 8);
      solidColour = ((brushTexture == SOLID)? 15 : 8);
      sprayColour = ((brushTexture == SPRAY)? 15 : 8);

      drawChar(0x30+brushSize, 183, 184, 7, 0);
      drawChar(7, 160, 180, circleColour, 7);
      drawChar(254, 160, 189, squareColour, 7);
      drawChar(176, 171, 180, sprayColour, 7);
      drawChar(219, 171, 189, solidColour, 7);

      drawBox(167, 178, 188, 198, 8); 	/* Brush Panel */
      drawBox(188, 178, 209, 198, 8);
      normline(176, 178, 176, 198, 8);
   }
   else {
      drawFilledBox(167, 178, 209, 198, 7);
   }

   showCursor();
}

/**************************************************************************
** drawInterface
**
** Draws palette and buttons at the bottom of the screen.
**************************************************************************/
void drawInterface()
{
   int c;

   memset(MK_FP(0xA000, (unsigned)177*320), 7, 7360);

   drawString("Line", 3, 180, 8, 7);
   drawBox(1, 178, 34, 189, 8);			/* Tool Buttons */

   drawString("Pen", 36, 180, 8, 7);
   drawBox(34, 178, 59, 189, 8);

   drawString("Step", 61, 180, 8, 7);
   drawBox(59, 178, 93, 189, 8);

   drawString("Fill", 95, 180, 8, 7);
   drawBox(93, 178, 125, 189, 8);

   drawString("Brush", 127, 180, 8, 7);
   drawBox(125, 178, 167, 189, 8);

   drawBox(1, 189, 167, 198, 8);		/* Palette */
   normline(147, 189, 147, 198, 8);
   for (c=0; c<16; c++) {
      drawFilledBox(2+c*9, 190, 11+c*9, 197, c);
   }
   normline(151, 192, 152, 192, 8);  /* O */
   normline(151, 195, 152, 195, 8);
   normline(150, 193, 150, 194, 8);
   normline(153, 193, 153, 194, 8);
   normline(155, 192, 158, 192, 8);  /* F */
   normline(155, 192, 155, 195, 8);
   normline(155, 194, 157, 194, 8);
   normline(160, 192, 163, 192, 8);  /* F */
   normline(160, 192, 160, 195, 8);
   normline(160, 194, 162, 194, 8);

   drawBox(209, 178, 294, 188, 8);

   drawChar(174, 203, 180, 8, 7);
   drawChar(60, 214, 180, 8, 7);
   drawChar(62, 268, 180, 8, 7);
   drawChar(175, 277, 180, 8, 7);

   normline(220, 178, 220, 188, 8);
   normline(229, 178, 229, 188, 8);
   normline(283, 178, 283, 188, 8);
   normline(274, 178, 274, 188, 8);

   drawString("Del", 296, 180, 8, 7);
   drawBox(294, 178, 318, 188, 8);

   drawFilledBox(210, 189, 317, 197, 0);
   drawBox(209, 188, 318, 198, 8);

   drawFilledBox(230, 179, 273, 187, 0);

   showPos();
}



/**************************************************************************
** drawEditorScreen
**
** Displays the PICTURE editors screen.
**************************************************************************/
void drawEditorScreen()
{
   drawStatusLine("", 'o', 'o', 0, 0);
   memset(MK_FP(0xA000, 8*320), 1, 320);
   showPicture();
   brushSize = 0;
   brushShape = CIRCLE;
   brushTexture = SPRAY;
   drawInterface();
   updateBrushPanel();
}

void setRect(rect *aRect, int left, int top, int right, int bottom)
{
   aRect->left = left;
   aRect->right = right;
   aRect->top = top;
   aRect->bottom = bottom;
}

int ptInRect(int x, int y, rect testRect)
{
   return ((x >= testRect.left) && (x <= testRect.right) &&
	   (y >= testRect.top) && (y <= testRect.bottom));
}

void waitForRelease()
{
   int bStatus, hPos, vPos;

   do {
      getPosition(&bStatus, &hPos, &vPos);
   } while (bStatus > 0);
}

void messageBox(int left, int top, int right, int bottom)
{
   drawFilledBox(left, top, right, bottom, 15);
   drawBox(left+2, top+1, right-2, bottom-1, 4);
   drawBox(left+3, top+1, right-3, bottom-1, 4);
}

void getPath(char *path)
{
   int pathStart=0, ch, pathLength, pathPos;
   boolean finished=FALSE;
   char inputLine[40], blanks[31] = "                              ";

   pathLength = strlen(path);
   path[pathLength] = '_';
   path[pathLength+1] = 0;

   do {

      pathStart = 0;
      if (pathLength > 29) pathStart = pathLength - 29;

      strcpy(inputLine, &path[pathStart]);
      if (pathStart == 0)
	 strncat(inputLine, blanks, 29-pathLength);

      drawString(inputLine, 40, 125, 7, 0);

      ch = getch();
      switch (ch) {
	 case 0x1B:   /* ignore */
	 case '\t':
	    break;

	 case 0x00:
	    ch = getch();
	    break;

	 case '\b':
	    if (pathLength > 0) {
	       pathLength--;
	       path[pathLength] = '_';
	       path[pathLength+1] = 0;
	    }
	    else {
	       sound(1000);
	       delay(50);
	       nosound();
	    }
	    break;

	 case 0x0D:
	    path[pathLength] = 0;
	    finished = TRUE;
	    break;

	 default:
	    path[pathLength] = toupper(ch);
	    path[++pathLength] = '_';
	    path[pathLength+1] = 0;

	    break;
      }

   } while (!finished);
}

char picFileNames[300][20];
boolean badPath=FALSE;

void loadPicNames(char *path, int *numPicFiles)
{
   struct ffblk ffblk, dirblk;
   int done, picCount = 0, copyLength, drive, result;

   memset(picFileNames, 0, sizeof(picFileNames));

   if (path[1] == ':') {    /* The first part gives the drive */
      drive = path[1] - 'A';
      setdisk(drive);
   }

   if (chdir(&path[2]) != 0) {
      badPath = TRUE;
      messageBox(40, 65, 279, 120);
      drawString("There is no directory named", 48, 73, 0, 15);
      path[24] = 0;  /* Cut path short */
      drawString(path, 48, 89, 0, 15);
      drawString("Press ENTER to continue.", 48, 105, 0, 15);
      getch();
   }
   else {
      done = findfirst("picture.*",&ffblk,0);
      while (!done) {
	 copyLength = strlen(ffblk.ff_name);
	 strncpy(picFileNames[picCount],ffblk.ff_name, copyLength);
	 picCount++;
	 done = findnext(&ffblk);
      }

      *numPicFiles = picCount;
   }
}

void getFileName(char *path, int y)
{
   int pathStart=0, ch, pathLength, pathPos;
   boolean finished=FALSE;
   char inputLine[40], blanks[31] = "                              ";

   pathLength = strlen(path);
   path[pathLength] = '_';
   path[pathLength+1] = 0;

   do {

      pathStart = 0;
      if (pathLength > 29) pathStart = pathLength - 29;

      strcpy(inputLine, &path[pathStart]);
      if (pathStart == 0)
	 strncat(inputLine, blanks, 29-pathLength);

      drawString(inputLine, 40, y, 7, 0);

      ch = getch();
      switch (ch) {
	 case 0x1B:
	    strcpy(path, "");
	    finished = TRUE;
	    break;
	 case '\t':   /* ignore */
	    break;

	 case 0x00:
	    ch = getch();
	    break;

	 case '\b':
	    if (pathLength > 0) {
	       pathLength--;
	       path[pathLength] = '_';
	       path[pathLength+1] = 0;
	    }
	    else {
	       sound(1000);
	       delay(50);
	       nosound();
	    }
	    break;

	 case 0x0D:
	    path[pathLength] = 0;
	    finished = TRUE;
	    break;

	 default:
	    path[pathLength] = toupper(ch);
	    path[++pathLength] = '_';
	    path[pathLength+1] = 0;

	    break;
      }

   } while (!finished);
}

int validPicName(char *fileName)
{
   char testStr[10]="PICTURE.", aChar, picNum[10];
   boolean found = FALSE;
   int strPos = 0, picNumLength;

   if (strcmp(fileName, testStr) == 0) return 0;

   do {
      aChar = *fileName;

      if (found) {
	 picNum[strPos++] = aChar;
	 if (aChar == 0) break;
      }
      else {
	 if (aChar == 0) break;
	 if (aChar == testStr[strPos]) {
	    strPos++;
	    if (strPos == 8) {
	       found = TRUE;
	       strPos = 0;
	    }
	 }
	 else {
	    strPos = 0;
	    if (aChar == testStr[strPos]) strPos++;
	 }
      }
   } while(*fileName++);

   if (found) {
      picNumLength = strlen(picNum);
      if (picNumLength > 3) return 0;

      for (strPos=0; strPos<picNumLength; strPos++) {
	 if (!isdigit(picNum[strPos])) return 0;
      }

      return 1;
   }

   return 0;
}

void getSaveName(char *choice, int numPics, int top)
{
   int curPic=0, i, numSlots=10, ch, startPos, endPos, oldSlot;
   boolean stillSelecting = TRUE, tabPushed=FALSE;
   char blanks[3] = "  ", tempString[20], path[200]="PICTURE.";

   if (numPics < 10) numSlots = numPics;

   for (i=0; i<numSlots; i++) {
      drawChar('-', 40, top+48+(i*8), 0, 15);
      drawString(picFileNames[i], 56, top+48+(i*8), 0, 15);
   }
   oldSlot = top+48;

   drawString("                              ", 40, top+56+numSlots*8, 7, 0);

   if (numPics == 0) tabPushed = TRUE;

   if (!tabPushed) {
      do {

	 startPos = ((curPic > 9)? (curPic - 9) : 0);
	 for (i=0; i<numSlots; i++) {
	    strcpy(tempString, picFileNames[i+startPos]);
	    strcat(tempString, blanks);
	    drawString(tempString, 56, top+48+(i*8), 0, 15);
	 }
	 drawChar(26, 32, oldSlot, 15, 15);
	 drawChar(26, 32, top+48+((curPic-startPos)*8), 0, 15);
	 oldSlot = top+48+((curPic-startPos)*8);

	 ch = getch();
	 switch (ch) {
	    case '\t':
	       tabPushed = TRUE;
	       drawChar(26, 32, oldSlot, 15, 15);
	       stillSelecting = FALSE;
	       break;

	    case 0:
	       ch = getch();
	       switch (ch) {
		  case 'H':
		     curPic--;
		     if (curPic < 0) curPic = numPics - 1;
		     break;
		  case 'P':
		     curPic++;
		     if (curPic >= numPics) curPic = 0;
		     break;
	       }
	       break;

	    case 0x1B:
	       stillSelecting = FALSE;
	       strcpy(choice, "");
	       break;

	    case 0x0D:
	       stillSelecting = FALSE;
	       strcpy(choice, picFileNames[curPic]);
	       break;
	 }

      } while (stillSelecting);
   }

   if ((strcmp(choice, "") == 0) && !tabPushed) return;

   if (tabPushed) {
      getFileName(path, top+56+(numSlots*8));
      if (strcmp(path, "") == 0) return;
      strcpy(choice, path);
   }

   if (validPicName(choice)) {
      savePicture(choice);
   }
   else {
      updateScreen();
      messageBox(40, 69, 279, 116);
      drawString("Sorry, the picture name you ", 48, 77, 0, 15);
      drawString("specified is not acceptable.", 48, 85, 0, 15);
      drawString("The name must be PICTURE and", 48, 93, 0, 15);
      drawString("the extension from 0 - 999.", 48, 101, 0, 15);
      getch();
   }
}

void savePicHandler()
{
   int drive, numPicFiles=0, winHeight, top, result;
   char directory[300], disk[5], path[300];
   char choice[15]="";

   hideCursor();
   messageBox(32, 45, 287, 141);
   drawString("SAVE PICTURE", 114, 53, 0, 15);
   drawString("On which disk or in which", 40, 69, 0, 15);
   drawString("directory do you wish to save", 40, 77, 0, 15);
   drawString("this picture?", 40, 85, 0, 15);
   drawString("(For example, \"B:\" or", 40, 101, 0, 15);
   drawString("\"C:\\pictures\")", 40, 109, 0, 15);
   drawString("                              ", 40, 125, 7, 0);
   drive = getdisk();
   disk[0] = 'A' + drive;
   disk[1] = ':';
   disk[2] = 0;
   getcurdir(drive+1, directory);
   sprintf(path, "%s\\%s", disk, directory);
   getPath(path);
   updateScreen();

   loadPicNames(path, &numPicFiles);

   if (!badPath) {
      if (numPicFiles != 0) {
	 winHeight = ((numPicFiles <= 10)? (72 + numPicFiles*8) : 152);
	 top = 9 + ((168-winHeight)/2);

	 messageBox(32, top, 287, top+winHeight);
	 drawString("Use the arrow keys and ENTER", 40, top+8, 0, 15);
	 drawString("to overwrite, or press TAB to", 40, top+16, 0, 15);
	 drawString("specify a new picture name.", 40, top+24, 0, 15);
	 drawString("ESC to not save the picture.", 40, top+32, 0, 15);
	 drawChar(26, 32, top+48, 0, 15);
	 getSaveName(choice, numPicFiles, top);

      }
      else {
	 top = 75;
	 winHeight = 50;

	 messageBox(32, top, 287, top+winHeight);
	 drawString("Specify a new picture name.", 40, top+8, 0, 15);
	 drawString("ESC to not save the picture.", 40, top+16, 0, 15);
	 getSaveName(choice, numPicFiles, top-21);
      }
   }

   updateScreen();
   showCursor();
}

void getSelection(char *choice, int numPics, int top)
{
   int curPic=0, i, numSlots=10, ch, startPos, endPos, oldSlot;
   boolean stillSelecting = TRUE;
   char blanks[3] = "  ", tempString[20];

   if (numPics < 10) numSlots = numPics;

   for (i=0; i<numSlots; i++) {
      drawChar('-', 24, top+48+(i*8), 0, 15);
      drawString(picFileNames[i], 40, top+48+(i*8), 0, 15);
   }
   oldSlot = top+48;

   do {

      startPos = ((curPic > 9)? (curPic - 9) : 0);
      for (i=0; i<numSlots; i++) {
	 strcpy(tempString, picFileNames[i+startPos]);
	 strcat(tempString, blanks);
	 drawString(tempString, 40, top+48+(i*8), 0, 15);
      }
      drawChar(26, 16, oldSlot, 15, 15);
      drawChar(26, 16, top+48+((curPic-startPos)*8), 0, 15);
      oldSlot = top+48+((curPic-startPos)*8);

      ch = getch();
      switch (ch) {
	 case 0:
	    ch = getch();
	    switch (ch) {
	       case 'H':
		  curPic--;
		  if (curPic < 0) curPic = numPics - 1;
		  break;
	       case 'P':
		  curPic++;
		  if (curPic >= numPics) curPic = 0;
		  break;
	    }
	    break;

	 case 0x1B:
	    stillSelecting = FALSE;
	    strcpy(choice, "");
	    break;

	 case 0x0D:
	    stillSelecting = FALSE;
	    strcpy(choice, picFileNames[curPic]);
	    break;
      }

   } while (stillSelecting);
}

void loadPicHandler()
{
   int drive, numPicFiles=0, winHeight, top, result;
   char directory[300], disk[5], path[300];
   char choice[15];

   hideCursor();
   messageBox(32, 45, 287, 141);
   drawString("LOAD PICTURE", 114, 53, 0, 15);
   drawString("On which disk or in which", 40, 69, 0, 15);
   drawString("directory is the picture that", 40, 77, 0, 15);
   drawString("you want to load?", 40, 85, 0, 15);
   drawString("(For example, \"B:\" or", 40, 101, 0, 15);
   drawString("\"C:\\pictures\")", 40, 109, 0, 15);
   drawString("                              ", 40, 125, 7, 0);
   drive = getdisk();
   disk[0] = 'A' + drive;
   disk[1] = ':';
   disk[2] = 0;
   getcurdir(drive+1, directory);
   sprintf(path, "%s\\%s", disk, directory);
   getPath(path);
   updateScreen();

   loadPicNames(path, &numPicFiles);

   if (!badPath) {

      if (numPicFiles == 0) {
	 messageBox(48, 61, 271, 125);
	 drawString("There are no pictures to", 56, 69, 0, 15);
	 drawString("load in", 56, 77, 0, 15);
	 path[24] = 0;  /* Cut path short */
	 drawString(path, 56, 93, 0, 15);
	 drawString("Press ENTER to continue.", 56, 109, 0, 15);
	 getch();
      }
      else {
	 winHeight = ((numPicFiles <= 10)? (56 + numPicFiles*8) : 136);
	 top = 9 + ((168-winHeight)/2);

	 messageBox(16, top, 303, top+winHeight);
	 drawString("Use the arrow keys to select the", 24, top+8, 0, 15);
	 drawString("picture which you wish to load.", 24, top+16, 0, 15);
	 drawString("Press ENTER to load the picture,", 24, top+24, 0, 15);
	 drawString("ESC to not load the picture.", 24, top+32, 0, 15);
	 drawChar(26, 16, top+48, 0, 15);
	 getSelection(choice, numPicFiles, top);

	 if (strcmp(choice, "") != 0) {
	    freeList();
	    drawPicture();
	    priorityScreen = FALSE;
	    loadPicture(choice);    /* Load picture */
	    drawPicture();
	 }
      }

   }

   updateScreen();
   showCursor();

   badPath = FALSE;
}

void aboutHandler()
{
   hideCursor();
   messageBox(40, 69, 279, 116);
   drawString("Picture Edit was designed to", 48, 77, 0, 15);
   drawString("edit pictures used in Sierra", 48, 85, 0, 15);
   drawString("On-Lines AGI adventure game", 48, 93, 0, 15);
   drawString("system.", 48, 101, 0, 15);
   getch();
   updateScreen();
   showCursor();
}

void textScreen()
{
   asm {
      mov ah, 0
      mov al, 3
      int 10h
   }
}

void hideTextCursor()
{
   union REGS regs;

   regs.h.ah = 1;
   regs.h.ch = 0x20;
   regs.h.cl = 0x20;
   int86(16, &regs, &regs);
}

void helpHandler()
{
   hideCursor();
   textScreen();
   hideTextCursor();
   printf("                                PICTURE EDIT v1.2\n\n");
   printf(" ESC     Brings up the menu system. Use arrow keys to select. \n");
   printf(" TAB     Switch between visual\priority screens.\n\n");
   printf(" PALETTE\n\n");
   printf(" Right click: choose colour for visual screen.\n");
   printf(" Left click: choose colour for priority screen.\n");
   printf(" Off: Disable visual/priority drawing.\n\n");
   printf(" FEATURES\n\n");
   printf(" - Click on buffer position field to enter a specific position.\n");
   printf(" - Use navigation buttons to navigate through picture code buffer.\n");
   printf(" - Drawing actions can be inserted anywhere inside the picture.\n");
   printf(" - The delete button removes the drawing action that is currently being\n");
   printf("   shown in the data box (bottom right corner).\n");
   printf(" - The brush tool is set up using the buttons immediately to the right of\n");
   printf("   the off button.\n");
   printf(" - Right clicking on the picture surface turns the current tool off.\n");
   printf(" - Toggle the background option to enable tracing of PCX files (background\n");
   printf("   image is included as a command line parameter).");
   getch();

   clrscr();
   printf("                                PICTURE EDIT v1.2\n\n\n");
   printf(" SHORTCUT KEYS\n\n");
   printf(" L, l, F1: Activates line tool\n");
   printf(" P, p, F2: Activates pen tool\n");
   printf(" S, s, F3: Activates step tool\n");
   printf(" F, f, F4: Activates fill tool\n");
   printf(" B, b, F5: Activates brush tool\n\n");
   printf(" F10: Toggles background picture\n\n");
   printf(" Home: Start of picture code buffer\n");
   printf(" Left arrow: Back one drawing action\n");
   printf(" Right arrow: Forward one drawing action\n");
   printf(" End: End of picture code buffer\n\n");
   printf(" Del: Remove drawing action at current buffer position\n");

   getch();
   graphicsScreen();
   restorePal();
   drawInterface();
   updateBrushPanel();
   updateScreen();
   showCursor();
}

void viewData()
{
   struct picCodeNode *temp;
   int tempPos = 2, startPos = 0, i, ch, textPos=0, j;
   boolean stillViewing = TRUE;
   char *textBuf, tempString[5];

   if (picStart == NULL) return;

   hideCursor();
   textScreen();
   hideTextCursor();

   textBuf = (char *)malloc(sizeof(char)*(23*80*2));
   memset(textBuf, 0, 23*80*2);
   clrscr();
   printf("                                 PICTURE DATA");

   while (stillViewing) {
      textPos=0;

      /* Find position */
      temp = picStart;
      for (i=0; ((i<startPos) && (temp!=NULL)); i++) {
	 temp = temp->next;
      }

      for (i=0; i<440; i++) {
	 sprintf(tempString, "%02X..", temp->node);
	 if (temp->node >= 0xF0) {
	    for (j=0; j<2; j++) {
	       textBuf[textPos++] = tempString[j];
	       if (temp == picPos)
		  textBuf[textPos++] = 13;
	       else
		  textBuf[textPos++] = 4;
	    }
	    for (j=2; j<4; j++) {
	       textBuf[textPos++] = tempString[j];
	       textBuf[textPos++] = 8;
	    }
	 }
	 else {
	    for (j=0; j<2; j++) {
	       textBuf[textPos++] = tempString[j];
	       textBuf[textPos++] = 7;
	    }
	    for (j=2; j<4; j++) {
	       textBuf[textPos++] = tempString[j];
	       textBuf[textPos++] = 8;
	    }
	 }
	 temp = temp->next;
	 if (temp == NULL) {
	    textBuf[textPos++] = 'F';
	    textBuf[textPos++] = 12;
	    textBuf[textPos++] = 'F';
	    textBuf[textPos++] = 12;
	    textBuf[textPos++] = ' ';
	    textBuf[textPos++] = 0;
	    textBuf[textPos++] = ' ';
	    textBuf[textPos++] = 0;
	    break;
	 }
      }

      memcpy(MK_FP(0xB814, 0), textBuf, 3520);

      switch ((ch = getch())) {
	 case 0:
	    switch ((ch = getch())) {
	       case 'H':
		  startPos-=20;
		  break;
	       case 'P':
		  startPos+=20;
		  break;
	       case 'I':
		  startPos-=440;
		  break;
	       case 'Q':
		  startPos+=440;
		  break;
	       case 'K':
		  startPos--;
		  break;
	       case 'M':
		  startPos++;
		  break;
	    }
	    break;
	 case 0x1B:
	    stillViewing = FALSE;
	    break;
      }

      if ((startPos + 440) > (bufLen+1)) startPos = ((bufLen+1) - 440);
      if (startPos < 0) startPos = 0;
   }

   free(textBuf);
   graphicsScreen();
   drawInterface();
   updateBrushPanel();
   updateScreen();
   showCursor();
}

void addCode(byte code)   /* To the list */
{
   struct picCodeNode *temp;

   temp = (struct picCodeNode *)malloc(sizeof(picCodes));
   if (temp == 0) {
      printf("Memory allocation problem.");
      exit(0);
   }
   temp->node = code;
   dlstore(temp);
}

void addPatCode()
{
   int pat=0;

   pat = (brushSize & 0x07);
   if (brushShape == SQUARE) pat |= 0x10;
   if (brushTexture == SPRAY) pat |= 0x20;

   addCode(pat);
}

void adjustDisp(int *dX, int *dY)
{
   if (*dX > 6) *dX = 6;
   if (*dX < -6) *dX = -6;
   if (*dY > 7) *dY = 7;
   if (*dY < -7) *dY = -7;
}

void adjustStatusLine()
{
   struct picCodeNode *temp;
   char sV='o', sP='o';
   char sTool[10]="";
   int pCode;

   brushSize = 0;
   brushShape = CIRCLE;
   brushTexture = SPRAY;
   updateBrushPanel();

   for (temp=picStart; temp!=picPos; ) {

      switch (temp->node) {
	 case 0xF0:
	    temp = temp->next;
	    sV = hexNumbers[temp->node];
	    break;

	 case 0xF1:
	    picDrawEnabled = FALSE;
	    sV = 'o';
	    break;

	 case 0xF2:
	    temp = temp->next;
	    sP = hexNumbers[temp->node];
	    break;

	 case 0xF3:
	    priDrawEnabled = FALSE;
	    sP = 'o';
	    break;

	 case 0xF4:
	    strcpy(sTool, "Step");
	    break;

	 case 0xF5:
	    strcpy(sTool, "Step");
	    break;

	 case 0xF6:
	    strcpy(sTool, "Line");
	    break;

	 case 0xF7:
	    strcpy(sTool, "Pen");
	    break;

	 case 0xF8:
	    strcpy(sTool, "Fill");
	    break;

	 case 0xF9:
	    strcpy(sTool, "Style");
	    pCode = temp->next->node;
	    if (pCode & 0x10) brushShape = SQUARE;
	    else brushShape = CIRCLE;
	    if (pCode & 0x20) brushTexture = SPRAY;
	    else brushTexture = SOLID;
	    brushSize = (pCode & 0x07);
	    brushActive = TRUE;
	    updateBrushPanel();
	    break;

	 case 0xFA:
	    strcpy(sTool, "Brush");
	    break;
      }

      do {
	 temp = temp->next;
      } while ((temp->node < 0xF0) && (temp != NULL));
   }

   drawStatusLine(sTool, sV, sP, mouseX, mouseY);
}

/***************************************************************************
** setBufPos
**
** Purpose: To allow the user to go to a position in the picture buffer
** immediately without having to use the navigation buttons. The position
** will be set to the start of the drawing action that the given position
** lies within.
***************************************************************************/
void setBufPos()
{
   int ch, inputPos, inputValue;
   boolean finished=FALSE;
   char inputLine[6]="";
   struct picCodeNode *temp;


   /* Get value from user */

   inputPos = 0;

   do {
      drawFilledBox(230, 179, 273, 187, 0);
      drawString(inputLine, 231, 180, 7, 0);

      ch = getch();
      switch (ch) {
	 case 0x1B:   /* ignore */
	 case '\t':
	    break;

	 case 0x00:
	    ch = getch();
	    break;

	 case '\b':
	    if (inputPos > 0) {
	       inputPos--;
	       inputLine[inputPos] = 0;
	    }
	    else {
	       sound(1000);
	       delay(50);
	       nosound();
	    }
	    break;

	 case 0x0D:
	    inputLine[inputPos] = 0;
	    finished = TRUE;
	    break;

	 default:
	    if (inputPos < 5) {
	       inputLine[inputPos++] = toupper(ch);
	       inputLine[inputPos] = 0;
	    }
	    else {
	       sound(1000);
	       delay(50);
	       nosound();
	    }
	    break;
      }

   } while (!finished);


   /* Test input value */
   inputValue = atof(inputLine);
   if ((inputValue < 0) || (inputValue > bufLen)) return;

   if (inputValue == bufLen) {
      picPos = NULL;
      bufPos = inputValue;
      return;
   }

   /* Find given location */
   while (bufPos != inputValue) {
      if (inputValue < bufPos) {
	 if (picPos == NULL)
	    picPos = picLast;
	 else
	    picPos = picPos->prior;
	 bufPos--;
      }
      else {
	 picPos = picPos->next;
	 bufPos++;
      }
   }

   /* Find current action position */
   while (picPos->node < 0xF0) {
      picPos = picPos->prior;
      bufPos--;
   }
}


/* Some global variables used by the following functions */
int stepClicks=0, numClicks=0;
boolean firstClick = TRUE;

void home_proc()
{
   moveToStart();
   adjustStatusLine();
   priDrawEnabled = picDrawEnabled = FALSE;
   drawPicture();
   hideCursor();
   updateScreen();
   showCursor();
   firstClick = TRUE;
   numClicks = 0;
   stepClicks=0;
}

void left_proc()
{
   moveBackAction();
   adjustStatusLine();
   priDrawEnabled = picDrawEnabled = FALSE;
   drawPicture();
   hideCursor();
   updateScreen();
   showCursor();
   firstClick = TRUE;
   numClicks = 0;
   stepClicks=0;
}

void right_proc()
{
   moveForwardAction();
   adjustStatusLine();
   priDrawEnabled = picDrawEnabled = FALSE;
   drawPicture();
   hideCursor();
   updateScreen();
   showCursor();
   firstClick = TRUE;
   numClicks = 0;
   stepClicks=0;
}

void end_proc()
{
   moveToEnd();
   adjustStatusLine();
   priDrawEnabled = picDrawEnabled = FALSE;
   drawPicture();
   hideCursor();
   updateScreen();
   showCursor();
   firstClick = TRUE;
   numClicks = 0;
   stepClicks=0;
}

void back_tog_proc()
{
   if (backAvail) {
      backEnabled = (backEnabled? FALSE : TRUE);
      restorePal();
      hideCursor();
      updateScreen();
      showCursor();
   }
}

void tool_proc(char *toolName)
{
   drawStatusLine(toolName, vColour, pColour, mouseX, mouseY);
   numClicks=0;
   stepClicks=0;
   firstClick=TRUE;
}

void del_proc()
{
   removeAction();
   adjustStatusLine();
   priDrawEnabled = picDrawEnabled = FALSE;
   drawPicture();
   hideCursor();
   updateScreen();
   showPos();
   showCursor();
}

/***************************************************************************
**                          MAIN PROGRAM
***************************************************************************/
void main(int argc, char **argv)
{
   int newX=0, newY=0, oldX=0, oldY=0, buttonStatus=0, ch, oldDrive;
   rect picRect, menuRect, toolRect, brushPanel;
   rect lineButton, penButton, stepButton, fillButton, brushButton;
   rect paletteRect, offButton, circleRect, squareRect, sprayRect;
   rect solidRect, upButton, downButton, posRect, bufRect, statusRect;
   rect delRect, leftButton, rightButton, homeButton, endButton;
   boolean needToWait, stillEditing=TRUE;
   byte menuCmd;
   char oldDir[200], path[200];
   struct picCodeNode *temp;
   int clickX, clickY, dX, dY, disp;

   if (argc > 1) {
      vgaScreen = (byte *)malloc(55000);
      if (vgaScreen == NULL) {
	 textmode(C80);
	 printf("Error allocating memory for background picture.\n");
	 exit(1);
      }
      if (!LoadPCX(0, 0, argv[1])) {
	 printf("An error occured loading the background picture:\n");
	 printf("%s\n", argv[1]);
	 free(vgaScreen);
	 exit(1);
      }
      else {
	 backAvail = TRUE;
      }
   }

   addMode = INS_MODE;
   firstClick = TRUE;

   oldDrive = getdisk();
   getcurdir(oldDrive + 1, oldDir);

   initAGIScreen();

   drawEditorScreen();
   setRect(&picRect, 0, 9, 319, 176);

   drawPicture();
   showPicture();

   priDrawEnabled = picDrawEnabled = FALSE;
   initMouse();
   showCursor();

   setRect(&lineButton, 2, 179, 33, 188);
   setRect(&penButton, 35, 179, 58, 188);
   setRect(&stepButton, 60, 179, 92, 188);
   setRect(&fillButton, 94, 179, 124, 188);
   setRect(&brushButton, 126, 179, 166, 188);
   setRect(&paletteRect, 2, 190, 146, 197);
   setRect(&offButton, 148, 190, 166, 197);
   setRect(&circleRect, 168, 179, 175, 187);
   setRect(&squareRect, 168, 189, 175, 197);
   setRect(&sprayRect, 177, 179, 187, 187);
   setRect(&solidRect, 177, 189, 187, 197);
   setRect(&upButton, 201, 179, 208, 187);
   setRect(&downButton, 201, 189, 208, 197);
   setRect(&homeButton, 210, 179, 219, 187);
   setRect(&leftButton, 221, 179, 228, 187);
   setRect(&rightButton, 275, 179, 282, 187);
   setRect(&endButton, 284, 179, 293, 187);
   setRect(&brushPanel, 168, 179, 208, 197);
   setRect(&delRect, 295, 179, 317, 187);
   setRect(&posRect, 230, 179, 273, 187);
   setRect(&bufRect, 210, 189, 317, 187);
   setRect(&statusRect, 0, 0, 319, 7);

   while (stillEditing) {
      needToWait = TRUE;

      if (kbhit()) {
	 ch = getch();
	 switch (ch) {
	    case 0:
	       ch = getch();
	       switch (ch) {
		  case 'G': home_proc(); break;
		  case 'K': left_proc(); break;
		  case 'M': right_proc(); break;
		  case 'O': end_proc(); break;
		  case 0x3B: tool_proc("Line"); break;
		  case 0x3C: tool_proc("Pen"); break;
		  case 0x3D: tool_proc("Step"); break;
		  case 0x3E: tool_proc("Fill"); break;
		  case 0x3F: tool_proc("Brush"); break;
		  case 0x44: back_tog_proc(); break;
		  case 0x53: del_proc(); break;
	       }
	       break;

	    case 'l': case 'L': tool_proc("Line"); break;
	    case 'p': case 'P': tool_proc("Pen"); break;
	    case 's': case 'S': tool_proc("Step"); break;
	    case 'f': case 'F': tool_proc("Fill"); break;
	    case 'b': case 'B': tool_proc("Brush"); break;

	    case 0x09: /* TAB : Switches between the two screens */
	       priorityScreen ^= 1;
	       hideCursor();
	       updateScreen();
	       showCursor();
	       break;

	    case 0x1B: /* ESC: Activates the menu system */
	       hideCursor();
	       menuCmd = menuSystem();
	       updateScreen();
	       memset(MK_FP(0xA000, 8*320), 1, 320);
	       drawStatusLine(tool, vColour, pColour, newX, newY);
	       showCursor();
	       if (menuCmd == QUIT) stillEditing = FALSE;
	       if (menuCmd == NEWPIC) {
		  freeList();
		  drawPicture();
		  priorityScreen = FALSE;
		  priDrawEnabled = picDrawEnabled = FALSE;
		  drawStatusLine("", 'o', 'o', 0, 0);
		  hideCursor();
		  showPicture();
		  showPos();
		  showCursor();
	       }
	       if (menuCmd == LOADPIC) {
		  loadPicHandler();
		  adjustStatusLine();
		  priDrawEnabled = picDrawEnabled = FALSE;
		  showPos();
	       }
	       if (menuCmd == SAVEPIC) savePicHandler();
	       if (menuCmd == ABOUT) aboutHandler();
	       if (menuCmd == HELP) {
		  helpHandler();
		  drawStatusLine(tool, vColour, pColour, newX, newY);
	       }
	       if (menuCmd == VIEWDATA) {
		  viewData();
		  drawStatusLine(tool, vColour, pColour, newX, newY);
	       }
	       if (menuCmd == BACKGRND) back_tog_proc();
	       break;
	 }
      }

      getPosition(&buttonStatus, &newX, &newY);
      if ((newX != oldX) || (newY != oldY)) {
	 if (ptInRect(newX, newY, picRect)) {
	    hideCursor();
	    drawStatusLine(tool, vColour, pColour, newX, newY);
	    updateScreen();
	    if ((strcmp(tool, "Line") == 0) && (numClicks>0)) {
	       normline2(clickX>>1, clickY, newX>>1, newY, picColour);
	    }
	    if ((strcmp(tool, "Step") == 0) && (stepClicks > 0)) {
	       switch (stepClicks) {
		  case 1:
		     dX = ((newX>>1) - (clickX>>1));
		     dY = ((newY-9) - (clickY-9));
		     if (abs(dX) > abs(dY)) dY = 0;
		     else dX = 0;
		     normline2(clickX>>1, clickY,
			 (clickX>>1)+dX, clickY+dY, picColour);
		     break;

		  default:
		     dX = ((newX>>1) - (clickX>>1));
		     dY = ((newY-9) - (clickY-9));
		     if (stepClicks % 2) dX = 0;
		     else dY = 0;
		     normline2(clickX>>1, clickY,
			       (clickX>>1)+dX, clickY+dY, picColour);
		     break;
	       }
	    }
	    if ((strcmp(tool, "Pen") == 0) && (numClicks>0)) {
	       dX = ((newX>>1) - (clickX>>1));
	       dY = ((newY-9) - (clickY-9));
	       adjustDisp(&dX, &dY);
	       normline2(clickX>>1, clickY,
			 (clickX>>1)+dX, clickY+dY, picColour);
	    }
	 }
	 else {
	    if (newY > 9) updateScreen();
	 }
	 showCursor();
      }
      if (buttonStatus != 0) {
	 switch (buttonStatus) {
	    case 1:
	       if (ptInRect(newX, newY, picRect)) {

		  if (strcmp(tool, "Fill") == 0) {
		     agiFill(mouseX>>1, mouseY-9);
		     if (firstClick) {
			addCode(0xF8);
			firstClick = FALSE;
		     }
		     addCode(mouseX>>1);
		     addCode(mouseY-9);
		     showPos();
		  }
		  else if (strcmp(tool, "Line") == 0) {
		     switch (numClicks++) {
			case 0: break;
			case 1:
			   addCode(0xF6);
			   addCode(clickX>>1);
			   addCode(clickY-9);
			default:
			   addCode(newX>>1);
			   addCode(newY-9);
			   drawline(clickX>>1, clickY-9, newX>>1, newY-9);
			   break;
		     }
		     clickX = newX;
		     clickY = newY;
		     showPos();
		  }
		  else if (strcmp(tool, "Pen") == 0) {
		     dX = ((newX>>1) - (clickX>>1));
		     dY = ((newY-9) - (clickY-9));
		     adjustDisp(&dX, &dY);
		     switch (numClicks++) {
			case 0:
			   clickX = newX;
			   clickY = newY;
			   break;
			case 1:
			   addCode(0xF7);
			   addCode(clickX>>1);
			   addCode(clickY-9);
			default:
			   if (dX < 0) disp = (0x80 | ((((-1)*dX)-0) << 4));
			   else disp = (dX << 4);
			   if (dY < 0) disp |= (0x08 | (((-1)*dY)-0));
			   else disp |= dY;
			   addCode(disp);
			   drawline(clickX>>1, clickY-9,
				 (clickX>>1)+dX, (clickY-9)+dY);
			   clickX = clickX + (dX<<1);
			   clickY = clickY + dY;
			   break;
		     }
		     showPos();
		  }
		  else if (strcmp(tool, "Brush") == 0) {
		     if (numClicks == 0) {
			addCode(0xF9);
			addPatCode();
			addCode(0xFA);
		     }
		     numClicks++;

		     patCode = brushSize;
		     if (brushShape == SQUARE) patCode |= 0x10;
		     if (brushTexture == SPRAY) patCode |= 0x20;
		     patNum = ((rand() % 0xEE) >> 1) & 0x7F;
		     plotPattern(mouseX>>1, mouseY-9);
		     if (brushTexture == SPRAY) addCode(patNum<<1);
		     addCode(mouseX>>1);
		     addCode(mouseY-9);
		     showPos();
		  }
		  else if (strcmp(tool, "Step") == 0) {
		     switch (stepClicks) {
			case 0:
			   clickX = newX;
			   clickY = newY;
			   break;

			case 1:
			   dX = ((newX>>1) - (clickX>>1));
			   dY = ((newY-9) - (clickY-9));
			   if (abs(dX) > abs(dY)) {   /* X or Y corner */
			      dY = 0;
			      stepClicks++;
			      addCode(0xF5);
			      addCode(clickX>>1);
			      addCode(clickY-9);
			      addCode((clickX>>1)+dX);
			   }
			   else {
			      dX = 0;
			      addCode(0xF4);
			      addCode(clickX>>1);
			      addCode(clickY-9);
			      addCode((clickY-9)+dY);
			   }
			   drawline(clickX>>1, clickY-9,
				 (clickX>>1)+dX, (clickY-9)+dY);
			   clickX = clickX + (dX<<1);
			   clickY = clickY + dY;
			   break;

			default:
			   dX = ((newX>>1) - (clickX>>1));
			   dY = ((newY-9) - (clickY-9));
			   if (stepClicks % 2) {
			      dX = 0;
			      addCode((clickY-9)+dY);
			   }
			   else {
			      dY = 0;
			      addCode((clickX>>1)+dX);
			   }
			   drawline(clickX>>1, clickY-9,
				 (clickX>>1)+dX, (clickY-9)+dY);
			   clickX = clickX + (dX<<1);
			   clickY = clickY + dY;
			   break;
		     }
		     stepClicks++;
		     showPos();
		  }

		  hideCursor();
		  updateScreen();
		  showCursor();
		  needToWait = TRUE;
	       }

	       if (ptInRect(newX, newY, paletteRect)) {
		  picColour = getpix(newX, newY);
		  addCode(0xF0);
		  addCode(picColour);
		  drawStatusLine(tool, hexNumbers[picColour], pColour,
		     mouseX, mouseY);
		  picDrawEnabled = TRUE;
		  firstClick = TRUE;
		  numClicks = 0;
		  stepClicks = 0;
		  showPos();
	       }
	       if (ptInRect(newX, newY, offButton)) {
		  addCode(0xF1);
		  drawStatusLine(tool, 'o', pColour, mouseX, mouseY);
		  picDrawEnabled = FALSE;
		  firstClick = TRUE;
		  numClicks = 0;
		  stepClicks = 0;
		  showPos();
	       }

	       if (ptInRect(newX, newY, lineButton)) tool_proc("Line");
	       if (ptInRect(newX, newY, penButton)) tool_proc("Pen");
	       if (ptInRect(newX, newY, stepButton)) tool_proc("Step");
	       if (ptInRect(newX, newY, fillButton)) tool_proc("Fill");
	       if (ptInRect(newX, newY, brushButton)) tool_proc("Brush");

	       if (ptInRect(newX, newY, upButton)) {
		  brushSize--;
		  if (brushSize < 0) brushSize = 0;
		  updateBrushPanel();
		  numClicks=0;
	       }
	       if (ptInRect(newX, newY, downButton)) {
		  brushSize++;
		  if (brushSize > 7) brushSize = 7;
		  updateBrushPanel();
		  numClicks=0;
	       }
	       if (ptInRect(newX, newY, circleRect)) {
		  brushShape = CIRCLE;
		  updateBrushPanel();
		  numClicks=0;
	       }
	       if (ptInRect(newX, newY, squareRect)) {
		  brushShape = SQUARE;
		  updateBrushPanel();
		  numClicks=0;
	       }
	       if (ptInRect(newX, newY, sprayRect)) {
		  brushTexture = SPRAY;
		  updateBrushPanel();
		  numClicks=0;
	       }
	       if (ptInRect(newX, newY, solidRect)) {
		  brushTexture = SOLID;
		  updateBrushPanel();
		  numClicks=0;
	       }

	       if (ptInRect(newX, newY, homeButton)) home_proc();
	       if (ptInRect(newX, newY, endButton)) end_proc();
	       if (ptInRect(newX, newY, rightButton)) right_proc();
	       if (ptInRect(newX, newY, leftButton)) left_proc();
	       if (ptInRect(newX, newY, delRect)) del_proc();

	       if (ptInRect(newX, newY, posRect)) {
		  hideCursor();
		  setBufPos();
		  adjustStatusLine();
		  priDrawEnabled = picDrawEnabled = FALSE;
		  drawPicture();
		  hideCursor();
		  updateScreen();
		  showPos();
		  showCursor();
		  numClicks=0;
		  stepClicks=0;
		  firstClick=TRUE;
	       }

	       if (ptInRect(newX, newY, statusRect)) {
		  hideCursor();
		  menuCmd = menuSystem();
		  updateScreen();
		  memset(MK_FP(0xA000, 8*320), 1, 320);
		  drawStatusLine(tool, vColour, pColour, newX, newY);
		  showCursor();
		  if (menuCmd == QUIT) stillEditing = FALSE;
		  if (menuCmd == NEWPIC) {
		     freeList();
		     drawPicture();
		     priorityScreen = FALSE;
		     priDrawEnabled = picDrawEnabled = FALSE;
		     drawStatusLine("", 'o', 'o', 0, 0);
		     hideCursor();
		     showPicture();
		     showPos();
		     showCursor();
		  }
		  if (menuCmd == LOADPIC) {
		     loadPicHandler();
		     adjustStatusLine();
		     priDrawEnabled = picDrawEnabled = FALSE;
		     showPos();
		  }
		  if (menuCmd == SAVEPIC) savePicHandler();
		  if (menuCmd == ABOUT) aboutHandler();
		  if (menuCmd == HELP) {
		     helpHandler();
		     drawStatusLine(tool, vColour, pColour, newX, newY);
		  }
		  if (menuCmd == VIEWDATA) {
		     viewData();
		     drawStatusLine(tool, vColour, pColour, newX, newY);
		  }
		  if (menuCmd == BACKGRND) back_tog_proc();
	       }
	       break;

	    case 2:
	       if (ptInRect(newX, newY, picRect)) {
		  drawStatusLine("", vColour, pColour, mouseX, mouseY);
		  brushActive=TRUE;
		  updateBrushPanel();
		  numClicks=0;
		  stepClicks=0;
		  firstClick=TRUE;
	       }
	       if (ptInRect(newX, newY, paletteRect)) {
		  priColour = getpix(newX, newY);
		  addCode(0xF2);
		  addCode(priColour);
		  drawStatusLine(tool, vColour, hexNumbers[priColour],
		     mouseX, mouseY);
		  priDrawEnabled = TRUE;
		  firstClick = TRUE;
		  numClicks = 0;
		  stepClicks=0;
		  showPos();
	       }
	       if (ptInRect(newX, newY, offButton)) {
		  addCode(0xF3);
		  drawStatusLine(tool, vColour, 'o', mouseX, mouseY);
		  priDrawEnabled = FALSE;
		  firstClick = TRUE;
		  numClicks = 0;
		  stepClicks=0;
		  showPos();
	       }
	       break;
	 }
	 if (needToWait) waitForRelease();
      }

      oldX = newX;
      oldY = newY;
   }

   setdisk(oldDrive);
   sprintf(path, "\\%s", oldDir);
   chdir(path);

   freeList();
   free(vgaScreen);
   free(picture);
   free(priority);
   closeAGIScreen();
}

