{
    LogEdit1.pas - LOGIC Editor
    Copyright (C) 1997-1999 Peter Kelly <peter@area51.org.au>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit LogEdit1;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, Menus;

type
  TLogEditWin = class(TForm)
    StatusBar1: TStatusBar;
    MainMenu1: TMainMenu;
    FileMenu: TMenuItem;
    FileCompile: TMenuItem;
    N1: TMenuItem;
    FileClose: TMenuItem;
    SaveDialog1: TSaveDialog;
    EditMemo: TRichEdit;
    FileSave: TMenuItem;
    FileCompileRun: TMenuItem;
    FileOpen: TMenuItem;
    FileNew: TMenuItem;
    OpenDialog1: TOpenDialog;
    FileSaveAs: TMenuItem;
    EditMenu: TMenuItem;
    EditCut: TMenuItem;
    EditCopy: TMenuItem;
    EditPaste: TMenuItem;
    N2: TMenuItem;
    EditUndo: TMenuItem;
    N3: TMenuItem;
    EditFind: TMenuItem;
    FindDialog1: TFindDialog;
    EditFindAgain: TMenuItem;
    FileChangeLogNum: TMenuItem;
    N4: TMenuItem;
    FilePrint: TMenuItem;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FileCloseClick(Sender: TObject);
    procedure FileCompileClick(Sender: TObject);
    procedure FileSaveAsClick(Sender: TObject);
    procedure FileSaveClick(Sender: TObject);
    procedure FileCompileRunClick(Sender: TObject);
    procedure FileNewClick(Sender: TObject);
    procedure FileOpenClick(Sender: TObject);
    procedure EditMemoChange(Sender: TObject);
    procedure EditCutClick(Sender: TObject);
    procedure EditCopyClick(Sender: TObject);
    procedure EditPasteClick(Sender: TObject);
    procedure EditUndoClick(Sender: TObject);
    procedure EditFindClick(Sender: TObject);
    procedure EditFindAgainClick(Sender: TObject);
    procedure FindDialog1Find(Sender: TObject);
    procedure FileChangeLogNumClick(Sender: TObject);
    procedure FilePrintClick(Sender: TObject);
  private
    { Private declarations }
    function GetLogicNumber(NumberEditText:string) : integer;
    function SaveSource : boolean;
    function CompileThisLogic : boolean;
  public
    { Public declarations }
    LogEditorNum : word;
    LogicNum : byte;
    ContentsIsLogic : boolean;  // if false, contents is a text file
    TextFilename : string;
    NewLogic : boolean;
    Modified : boolean;
    function AskClose : boolean;
    procedure SaveTextFile;
  end;

const MaxLogEditWins = 20;

var
     LogEditWin: array[1..MaxLogEditWins] of TLogEditWin;
     LogEditWinUsed : array[1..MaxLogEditWins] of boolean;

function EditLogic(LogNum:byte;new:boolean) : integer;  // return value = editor window number
function EditTextFile(Filename:string) : integer;  // return value = editor window number

implementation

uses ResourceWin1, Defines, VolMan2, LogDecode, LogCompile, main1,
  GetResourceNum1;

{$R *.DFM}


{*************************************************************}
function TLogEditWin.AskClose : boolean;
{*************************************************************}
var SaveMessageResult : integer;
    WindowName : PChar;
begin
  AskClose := True;
  if Modified then
  begin
    GetMem(WindowName,Length(LogEditWin[LogEditorNum].Caption)+1);
    StrPCopy(WindowName,LogEditWin[LogEditorNum].Caption);
    if ContentsIsLogic then
    begin  // logic editor
      SaveMessageResult := Application.MessageBox('Do you want to save changes?',WindowName,MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then SaveSource
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then Modified := False;
    end
    else
    begin // text editor
      SaveMessageResult := Application.MessageBox('Do you want to save changes?',WindowName,MB_YESNOCANCEL);
      if SaveMessageResult = IDYES then FileSaveClick(Application)
      else if SaveMessageResult = IDCANCEL then AskClose := False;
      if (SaveMessageResult = IDYES) or (SaveMessageResult = IDNO) then Modified := False;
    end;
    FreeMem(WindowName,Length(LogEditWin[LogEditorNum].Caption)+1);
  end;
end;

{*************************************************************}
function EditLogic(LogNum:byte;new:boolean) : integer;
{*************************************************************}
var ResourceData : TResource;
    CurWinNum : byte;
    ThisWinNum : byte;
    ResourceLoadSuccess : boolean;
    LogicText : TStringList;
    TempStream : TMemoryStream;
begin
  EditLogic := 0;
  ThisWinNum := 0;
  if (not new) then
  begin
    CurWinNum := 0;
    for CurWinNum := 1 to MaxLogEditWins do
    if LogEditWinUsed[CurWinNum] and LogEditWin[CurWinNum].ContentsIsLogic
    and (not LogEditWin[CurWinNum].NewLogic) and (LogEditWin[CurWinNum].LogicNum = LogNum) then
      ThisWinNum := CurWinNum;
    if ThisWinNum > 0 then LogEditWin[ThisWinNum].Show;  // logic is already open in another window so set focus to it
  end;
  if ThisWinNum = 0 then
  begin  // the logic is not already open in another window
    for CurWinNum := 1 to MaxLogEditWins do
      if not LogEditWinUsed[CurWinNum] then ThisWinNum := CurWinNum;
    if ThisWinNum = 0 then
      ShowMessage('Error: Too many editors already open.')
    else
    begin
      LogEditWinUsed[ThisWinNum] := True;
      LogEditWin[ThisWinNum] := TLogEditWin.Create(Main);
      if Settings.LogicEditor.MaximizeEditorWindow then LogEditWin[ThisWinNum].WindowState := wsMaximized;
      LogEditWin[ThisWinNum].LogEditorNum := ThisWinNum;
      LogEditWin[ThisWinNum].LogicNum := LogNum;
      LogEditWin[ThisWinNum].ContentsIsLogic := True;
      LogEditWin[ThisWinNum].NewLogic := New;
      LogEditWin[ThisWinNum].FileCompile.Visible := True;
      LogEditWin[ThisWinNum].FileCompileRun.Visible := True;
      LogEditWin[ThisWinNum].FileNew.Visible := False;
      LogEditWin[ThisWinNum].FileOpen.Visible := False;
      LogEditWin[ThisWinNum].FileSaveAs.Visible := False;
      LogEditWin[ThisWinNum].FileChangeLogNum.Visible := True;
      LogEditWin[ThisWinNum].FileCompile.Enabled := LogEditWin[ThisWinNum].FileCompile.Visible;
      LogEditWin[ThisWinNum].FileCompileRun.Enabled := LogEditWin[ThisWinNum].FileCompileRun.Visible;
      LogEditWin[ThisWinNum].FileNew.Enabled := LogEditWin[ThisWinNum].FileNew.Visible;
      LogEditWin[ThisWinNum].FileOpen.Enabled := LogEditWin[ThisWinNum].FileOpen.Visible;
      LogEditWin[ThisWinNum].FileSaveAs.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Visible;
      LogEditWin[ThisWinNum].FileChangeLogNum.Enabled := LogEditWin[ThisWinNum].FileChangeLogNum.Visible;
      if LogEditWin[ThisWinNum].NewLogic then
      begin
        LogEditWin[ThisWinNum].Caption := 'Logic editor - New Logic';
      end
      else
      begin
        LogEditWin[ThisWinNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,Lognum);
        LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := 'Reading logic....';
        if FileExists(GameDir+'src\Logic'+IntToStr(LogNum)+'.txt') then
        begin  // source already exists, read that
          LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromFile(GameDir+'src\Logic'+IntToStr(LogNum)+'.txt');
          LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := '';
          LogEditWin[ThisWinNum].EditMemo.ReadOnly := False;
          LogEditWin[ThisWinNum].Modified := False;
        end
        else
        begin // source does not exist, must decode logic
          ResourceData := ReadResource(LOGIC,LogNum);
          if ResourceData.Size > 0 then
          begin
            LogicText := TStringList.Create;
            LogEditWin[ThisWinNum].StatusBar1.Repaint;
            if DecodeLogic(ResourceData,LogicText) then {DecodeLogic frees the memory after it has used it.}
            begin
              TempStream := TMemoryStream.Create;
              LogicText.SaveToStream(TempStream);
              TempStream.Position := 0;
              LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromStream(TempStream);
              TempStream.Free;
              LogEditWin[ThisWinNum].StatusBar1.Panels[0].Text := '';
              LogEditWin[ThisWinNum].EditMemo.ReadOnly := False;
              LogEditWin[ThisWinNum].Modified := False;
            end
            else
            begin
              ShowMessage('Error loading Logic. The resource may be corrupt.');
              LogEditWin[ThisWinNum].Close;
              LogEditWinUsed[ThisWinNum] := False;
            end;
            LogicText.Free;
          end  // if ResourceData.Size > 0
          else
          begin
            ShowMessage('Error: Could not load resource. The files may be open or the resource may be corrupt.');
            LogEditWin[ThisWinNum].Close;
            LogEditWinUsed[ThisWinNum] := False;
          end;
        end;
      end;  // if not LogEditWin[ThisWinNum].NewLogic
    end;
  end;  // if ThisWinNum = 0 (i.e. the logic is not already open in another window)
  EditLogic := ThisWinNum;
end;

{*************************************************************}
function EditTextFile(Filename:string) : integer;
{*************************************************************}
var
    CurWinNum : byte;
    ThisWinNum : byte;
begin
  EditTextFile := 0;
  ThisWinNum := 0;
  CurWinNum := 0;
  for CurWinNum := 1 to MaxLogEditWins do
  if LogEditWinUsed[CurWinNum] and (not LogEditWin[CurWinNum].ContentsIsLogic)
  and (Lowercase(LogEditWin[CurWinNum].TextFilename) = Lowercase(Filename)) then
    ThisWinNum := CurWinNum;
  if ThisWinNum > 0 then LogEditWin[ThisWinNum].Show;  // text file is already open in another window so set focus to it
  if ThisWinNum = 0 then
  begin  // the text file is not already open in another window
    for CurWinNum := 1 to MaxLogEditWins do
      if not LogEditWinUsed[CurWinNum] then ThisWinNum := CurWinNum;
    if ThisWinNum = 0 then
      ShowMessage('Error: Too many editors already open.')
    else
    begin
      if (Filename <> '') and (not CanAccessFile(Filename)) then
        ShowMessage('Error: Could not read file.')
      else
      begin
        LogEditWinUsed[ThisWinNum] := True;
        LogEditWin[ThisWinNum] := TLogEditWin.Create(Main);
        if Settings.LogicEditor.MaximizeEditorWindow then LogEditWin[ThisWinNum].WindowState := wsMaximized;
        LogEditWin[ThisWinNum].LogEditorNum := ThisWinNum;
        LogEditWin[ThisWinNum].ContentsIsLogic := False;
        LogEditWin[ThisWinNum].FileCompile.Visible := False;
        LogEditWin[ThisWinNum].FileCompileRun.Visible := False;
        LogEditWin[ThisWinNum].FileNew.Visible := True;
        LogEditWin[ThisWinNum].FileOpen.Visible := True;
        LogEditWin[ThisWinNum].FileSaveAs.Visible := True;
        LogEditWin[ThisWinNum].FileChangeLogNum.Visible := False;
        LogEditWin[ThisWinNum].FileCompile.Enabled := LogEditWin[ThisWinNum].FileCompile.Visible;
        LogEditWin[ThisWinNum].FileCompileRun.Enabled := LogEditWin[ThisWinNum].FileCompileRun.Visible;
        LogEditWin[ThisWinNum].FileNew.Enabled := LogEditWin[ThisWinNum].FileNew.Visible;
        LogEditWin[ThisWinNum].FileOpen.Enabled := LogEditWin[ThisWinNum].FileOpen.Visible;
        LogEditWin[ThisWinNum].FileSaveAs.Enabled := LogEditWin[ThisWinNum].FileSaveAs.Visible;
        LogEditWin[ThisWinNum].FileChangeLogNum.Enabled := LogEditWin[ThisWinNum].FileChangeLogNum.Visible;
        if Filename = '' then LogEditWin[ThisWinNum].Caption := 'Text editor'
        else
        begin
          LogEditWin[ThisWinNum].Caption := 'Text editor - ' + ExtractFileName(Filename);
          LogEditWin[ThisWinNum].EditMemo.Lines.LoadFromFile(Filename)
        end;
        LogEditWin[ThisWinNum].Modified := False;
        LogEditWin[ThisWinNum].TextFilename := Filename;
      end;
    end;  // if ThisWinNum > 0
  end;  // if ThisWinNum = 0 (i.e. the logic is not already open in another window)
  EditTextFile := ThisWinNum;
end;

{*************************************************************}
function TLogEditWin.GetLogicNumber(NumberEditText:string) : integer;
{*************************************************************}
var NewNum : byte;
    LogicIndex : integer;
begin
  GetLogicNumber := -1;
  GetResourceNum.WindowFunction := LogEditGetNum;
  GetResourceNum.NumberEdit.Text := NumberEditText;
  GetResourceNum.ShowModal;
  if GetResourceNum.OKPressed then
  begin
    NewNum := StrToInt(GetResourceNum.NumberEdit.Text);  // GetResourceNum verifies that this is a number between 0 and 255.
      LogicIndex := ResourceIndex(LOGIC,NewNum);
      if (LogicIndex < 0) or (AskYesNo('Change logic number','Logic '+IntToStr(NewNum)+' already exists. Do you wish to replace it?')) then
        GetLogicNumber := NewNum;
  end;
end;

{*************************************************************}
function TLogEditWin.SaveSource : boolean;
{*************************************************************}
var FoundDir : boolean;
    SourceFilename : string;
    NewLogicNum : integer;
begin
  SaveSource := False;

  if NewLogic then
  begin
    NewLogicNum := GetLogicNumber('');
    if NewLogicNum >= 0 then
    begin
      NewLogic := False;
      LogicNum := NewLogicNum;
      LogEditWin[LogEditorNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,LogicNum);
    end;
  end;

  if (not NewLogic) then
  begin
    FoundDir := FileExists(GameDir+'\'+SourceDir+'\nul');
    if not FoundDir then FoundDir := CreateDir(GameDir+'\'+SourceDir);
    if not FoundDir then
      ShowMessage('Error creating directory "'+GameDir+'\'+SourceDir+'". The source code could not be saved.')
    else
    begin
      SourceFilename := GameDir+SourceDir+'\Logic'+IntToStr(LogicNum)+'.txt';
      if FileExists(SourceFilename) and (not CanAccessFile(SourceFilename)) then
       ShowMessage('Error saving source code to "'+SourceFilename+'".')
      else
      begin
        EditMemo.Lines.SaveToFile(SourceFilename);
        SaveSource := True;
        Modified := False;
        StatusBar1.Panels[0].Text := 'File saved.';
        EditUndo.Enabled := False;
      end;
    end;
  end;
end;

{*************************************************************}
function TLogEditWin.CompileThisLogic : boolean;
{*************************************************************}
var CompiledLogic : TResource;
begin
  CompileThisLogic := False;
  CompiledLogic := CompileLogic(LogEditorNum);
  if CompiledLogic.Size > 0 then
  begin
    StatusBar1.Panels[0].Text := 'Compile successful.';
    AddResource(CompiledLogic,LOGIC,LogicNum); // AddResource will free the memory assigned to CompiledLogic
    CompileThisLogic := True;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FormClose(Sender: TObject; var Action: TCloseAction);
{*************************************************************}
begin
  LogEditWinUsed[LogEditorNum] := false;
  action := caFree;
end;

{*************************************************************}
procedure TLogEditWin.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
{*************************************************************}
begin
  CanClose := AskClose;
end;

{*************************************************************}
procedure TLogEditWin.FileCloseClick(Sender: TObject);
{*************************************************************}
begin
  Close;
end;

{*************************************************************}
procedure TLogEditWin.FileCompileClick(Sender: TObject);
{*************************************************************}
begin
  SaveSource;
  if not NewLogic then CompileThisLogic;
end;

{*************************************************************}
procedure TLogEditWin.SaveTextFile;
{*************************************************************}
begin
  EditMemo.Lines.SaveToFile(TextFilename);
  StatusBar1.Panels[0].Text := 'File saved.';
  Modified := False;
  EditUndo.Enabled := False;
end;

{*************************************************************}
procedure TLogEditWin.FileSaveAsClick(Sender: TObject);
{*************************************************************}
begin
  SaveDialog1.Filename := TextFilename;
  if SaveDialog1.Execute then
  begin
    if FileExists(SaveDialog1.Filename) and (not CanAccessFile(SaveDialog1.Filename)) then
      ShowMessage('Error: Could not write to file.')
    else
    begin
      TextFilename := SaveDialog1.Filename;
      LogEditWin[LogEditorNum].Caption := 'Text editor - ' + ExtractFileName(TextFilename);
      SaveTextFile;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileSaveClick(Sender: TObject);
{*************************************************************}
begin
  if ContentsIsLogic then SaveSource
  else
  begin
    if TextFilename <> '' then
    begin
      if FileExists(TextFilename) and (not CanAccessFile(TextFilename)) then
        ShowMessage('Error: Could not write to file "'+TextFilename+'".')
      else
      begin
        SaveTextFile;
      end;
    end
    else
    begin
      FileSaveAsClick(Sender);
      Modified := False;
      EditUndo.Enabled := False;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileCompileRunClick(Sender: TObject);
{*************************************************************}
begin
  SaveSource;
  if (not NewLogic) and CompileThisLogic then
    Main.GameRunClick(Sender);
end;

{*************************************************************}
procedure TLogEditWin.FileNewClick(Sender: TObject);
{*************************************************************}
begin
  if AskClose then
  begin
    TextFilename := '';
    LogEditWin[LogEditorNum].Caption := 'Text editor';
    EditMemo.Lines.Clear;
    StatusBar1.Panels[0].Text := '';
    Modified := False;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FileOpenClick(Sender: TObject);
{*************************************************************}
var CurWinNum : word;
    ThisWinNum : word;
begin
  if AskClose and OpenDialog1.Execute then
  begin
    if not CanAccessFile(OpenDialog1.Filename) then
      ShowMessage('Error: Could not read file.')
    else
    begin
      ThisWinNum := 0;
      for CurWinNum := 1 to MaxLogEditWins do
      if (CurWinNum <> LogEditorNum) and LogEditWinUsed[CurWinNum] and (not LogEditWin[CurWinNum].ContentsIsLogic)
      and (Lowercase(LogEditWin[CurWinNum].TextFilename) = Lowercase(OpenDialog1.Filename)) then
        ThisWinNum := CurWinNum;
      if ThisWinNum > 0 then
        ShowMessage('File is already open in another window.')
      else
      begin
        EditMemo.Lines.LoadFromFile(OpenDialog1.Filename);
        LogEditWin[LogEditorNum].Caption := 'Text editor - ' + ExtractFileName(OpenDialog1.Filename);
        TextFilename := OpenDialog1.Filename;
        StatusBar1.Panels[0].Text := '';
        Modified := False;
      end;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.EditMemoChange(Sender: TObject);
{*************************************************************}
begin
  if not Modified then Modified := True;
  StatusBar1.Panels[0].Text := '';
  EditUndo.Enabled := Modified and (SendMessage(EditMemo.Handle, EM_CANUNDO, 0, 0) <> 0);
end;

{*************************************************************}
procedure TLogEditWin.EditCutClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.CutToClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditCopyClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.CopyToClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditPasteClick(Sender: TObject);
{*************************************************************}
begin
  EditMemo.PasteFromClipboard;
end;

{*************************************************************}
procedure TLogEditWin.EditUndoClick(Sender: TObject);
{*************************************************************}
begin
  with EditMemo do
    if HandleAllocated then SendMessage(Handle, EM_UNDO, 0, 0);
end;

{*************************************************************}
procedure TLogEditWin.EditFindClick(Sender: TObject);
{*************************************************************}
begin
  FindDialog1.Execute;
end;

{*************************************************************}
procedure TLogEditWin.EditFindAgainClick(Sender: TObject);
{*************************************************************}
begin
  if FindDialog1.FindText = '' then FindDialog1.Execute
  else FindDialog1Find(Sender);
end;

{*************************************************************}
procedure TLogEditWin.FindDialog1Find(Sender: TObject);
{*************************************************************}
var FoundPos : longint;
    SearchType : TSearchTypes;
begin
  SearchType := [];
  if frWholeWord in FindDialog1.Options then SearchType := SearchType + [stWholeWord];
  if frMatchCase in FindDialog1.Options then SearchType := SearchType + [stMatchCase];
  FoundPos := EditMemo.FindText(FindDialog1.FindText,EditMemo.SelStart+1,Length(EditMemo.Text)-EditMemo.SelStart,SearchType);
  if FoundPos >= 0 then
  begin
    EditMemo.SelStart := FoundPos;
    EditMemo.SelLength := Length(FindDialog1.FindText);
  end
  else ShowMessage('Cannot find "'+FindDialog1.FindText+'".');
end;

{*************************************************************}
procedure TLogEditWin.FileChangeLogNumClick(Sender: TObject);
{*************************************************************}
var NewLogicNum : integer;
begin
  if NewLogic then NewLogicNum := GetLogicNumber('')
  else NewLogicNum := GetLogicNumber(IntToStr(LogicNum));
  if NewLogicNum >= 0 then
  begin
    if (not NewLogic) and (NewLogicNum = LogicNum) then
      ShowMessage('Logic number not changed!')
    else
    begin
      NewLogic := False;
      LogicNum := NewLogicNum;
      LogEditWin[LogEditorNum].Caption := 'Logic editor - ' + ResourceName(LOGIC,LogicNum);
      SaveSource;
      CompileThisLogic;
    end;
  end;
end;

{*************************************************************}
procedure TLogEditWin.FilePrintClick(Sender: TObject);
{*************************************************************}
var PrintCaption : string;
begin
  if ContentsIsLogic then
  begin
   if NewLogic then PrintCaption := 'New Logic'
   else PrintCaption := 'Logic '+IntToStr(LogicNum);
  end
  else
  begin
    if TextFilename = '' then PrintCaption := 'Untitled'
    else PrintCaption := ExtractFilename(TextFilename);
  end;
  EditMemo.Print(PrintCaption);
end;

end.
