/*
 * Copyright (c) 2002 by Ravi Iyengar [ravi.i@softhome.net]
 * Released under the GNU General Public License
 * See LICENSE for details.
 */

// ChildView.cpp : implementation of the CChildView class
//

#include "stdafx.h"
#include "Soundbox.h"
#include "ChildView.h"
#include "SoundResource.h"
#include "DataStream.h"
#include "LoopPointDlg.h"
#include "CuesDlg.h"
#include "ChannelsDlg.h"
#include "SoundPlayer.h"
#include "MidiPlayer.h"
#include "LicenseDlg.h"
#include <fstream>

#ifdef _DEBUG
#define new DEBUG_NEW
#endif


// CChildView

CChildView::CChildView()
: mLoaded(false), mPath(""), mPlaying(0), mPlayer(0), mFilter(0)
{
	mSound = new SoundResource();
	mDisplay.SetSound(mSound);
	mVUnit = HIWORD(GetDialogBaseUnits());
}

CChildView::~CChildView()
{
	if (mPlaying) {
		try {
			mPlayer->Stop();
			mPlayer->Cleanup();
		} catch (std::runtime_error &e) {
			::MessageBox(0, e.what(), "Player Error", MB_OK);
		}
		if (mPlayer->IsError()) {
			::MessageBox(0, mPlayer->GetError().c_str(), "Player Error", MB_OK);
		}
		delete mPlayer;
		mPlayer = 0;
		mPlaying = false;
	}
	delete mSound;
}


BEGIN_MESSAGE_MAP(CChildView, CWnd)
	ON_UPDATE_COMMAND_UI(ID_FILE_OPEN130, OnUpdateFileOpen130)
	ON_UPDATE_COMMAND_UI(ID_FILE_IMPORTMIDI, OnUpdateFileImportmidi)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE132, OnUpdateFileSave132)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVEAS, OnUpdateFileSaveas)
	ON_UPDATE_COMMAND_UI(ID_EDIT_LOOPPOINT, OnUpdateEditLooppoint)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CUES, OnUpdateEditCues)
	ON_UPDATE_COMMAND_UI(ID_EDIT_CHANNELPROPERTIES, OnUpdateEditChannelproperties)
	ON_COMMAND(ID_FILE_OPEN130, OnFileOpen130)
	ON_COMMAND(ID_FILE_IMPORTMIDI, OnFileImportmidi)
	ON_COMMAND(ID_FILE_SAVE132, OnFileSave132)
	ON_COMMAND(ID_FILE_SAVEAS, OnFileSaveas)
	ON_COMMAND(ID_EDIT_LOOPPOINT, OnEditLooppoint)
	ON_COMMAND(ID_EDIT_CUES, OnEditCues)
	ON_COMMAND(ID_EDIT_CHANNELPROPERTIES, OnEditChannelproperties)
	ON_COMMAND(ID_VIEW_CUES, OnViewCues)
	ON_UPDATE_COMMAND_UI(ID_VIEW_CUES, OnUpdateViewCues)
	ON_COMMAND(ID_VIEW_LOOPPOINT, OnViewLooppoint)
	ON_UPDATE_COMMAND_UI(ID_VIEW_LOOPPOINT, OnUpdateViewLooppoint)
	ON_COMMAND(ID_VIEW_NOTEONEVENTS, OnViewNoteonevents)
	ON_UPDATE_COMMAND_UI(ID_VIEW_NOTEONEVENTS, OnUpdateViewNoteonevents)
	ON_COMMAND(ID_VIEW_TIMEMARKERS, OnViewTimemarkers)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TIMEMARKERS, OnUpdateViewTimemarkers)
	ON_COMMAND(ID_VIEW_VOICECONTROLS, OnViewVoicecontrols)
	ON_UPDATE_COMMAND_UI(ID_VIEW_VOICECONTROLS, OnUpdateViewVoicecontrols)
	ON_WM_CREATE()
	ON_WM_SIZE()
	ON_MESSAGE(SBM_SETFILTER, OnSetFilter)
	ON_MESSAGE(SBM_SOUNDTICK, OnSoundTick)
	ON_MESSAGE(SBM_PLAYSTARTED, OnPlayStarted)
	ON_MESSAGE(SBM_PLAYSTOPPED, OnPlayStopped)
	ON_MESSAGE(SBM_SOUNDSIGNAL, OnSoundSignal)
	ON_MESSAGE(SBM_PLAY, OnPlay)
	ON_MESSAGE(SBM_STOP, OnStop)
	ON_COMMAND(ID_HELP_LICENSE, OnHelpLicense)
END_MESSAGE_MAP()



// CChildView message handlers

BOOL CChildView::PreCreateWindow(CREATESTRUCT& cs) 
{
	if (!CWnd::PreCreateWindow(cs))
		return FALSE;

	cs.dwExStyle |= WS_EX_CLIENTEDGE;
	cs.style &= ~WS_BORDER;
	cs.style |= WS_CLIPCHILDREN;
	cs.lpszClass = AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, 
		::LoadCursor(NULL, IDC_ARROW), reinterpret_cast<HBRUSH>(COLOR_WINDOW+1), NULL);

	return TRUE;
}

void CChildView::OnUpdateFileOpen130(CCmdUI *pCmdUI)
{
	// cannot load while playing
	pCmdUI->Enable(!mPlaying);
}

void CChildView::OnUpdateFileImportmidi(CCmdUI *pCmdUI)
{
	// cannot load while playing
	pCmdUI->Enable(!mPlaying);
}

void CChildView::OnUpdateFileSave132(CCmdUI *pCmdUI)
{
	// saving requires a loded resource
	pCmdUI->Enable(mLoaded);
}

void CChildView::OnUpdateFileSaveas(CCmdUI *pCmdUI)
{
	// saving requires a loded resource
	pCmdUI->Enable(mLoaded);
}

void CChildView::OnUpdateEditLooppoint(CCmdUI *pCmdUI)
{
	// editing requires a loaded resource and cannot occur during playback
	pCmdUI->Enable(mLoaded && !mPlaying);
}

void CChildView::OnUpdateEditCues(CCmdUI *pCmdUI)
{
	// editing requires a loaded resource and cannot occur during playback
	pCmdUI->Enable(mLoaded && !mPlaying);
}
 
void CChildView::OnUpdateEditChannelproperties(CCmdUI *pCmdUI)
{
	// editing requires a loaded resource and cannot occur during playback
	pCmdUI->Enable(mLoaded && !mPlaying);
}

void CChildView::OnFileOpen130()
{
	// get the resource to import
	CFileDialog openDlg(true, 0, 0, OFN_FILEMUSTEXIST | OFN_HIDEREADONLY);
	if (openDlg.DoModal() != IDOK) return;
	CString file = openDlg.GetPathName();

	// open the file and construct a DataStream around it
	std::fstream inStream;
	inStream.open(file, std::ios_base::in | std::ios_base::binary);
	if (!inStream.good()) {
		MessageBox("Unable to open file for loading", "Load Error");
		return;
	}
	DataStream inDataStream(&inStream);

	// load the resource
	try {
		mSound->Load(inDataStream);
		mPath = file;
		mLoaded = true;
		mPlayBar.SetPlayState(PLAYSTATE_STOPPED);
		mManager.SetResourcePath(file.Left(file.ReverseFind('\\')));
		CString title = "Soundbox - " + file;
		AfxGetMainWnd()->SetWindowText(title);
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Load Error");
	}

	mDisplay.PaintBuffer();
}

void CChildView::OnFileImportmidi()
{
	// get the SMF to import
	CFileDialog openDlg(true, 0, 0, OFN_FILEMUSTEXIST | OFN_HIDEREADONLY);
	if (openDlg.DoModal() != IDOK) return;
	CString file = openDlg.GetPathName();

	// open the file and construct a DataStream around it
	std::fstream inStream;
	inStream.open(file, std::ios_base::in | std::ios_base::binary);
	if (!inStream.good()) {
		MessageBox("Unable to open file for importing", "Import Error");
		return;
	}
	DataStream inDataStream(&inStream);

	// load the SMF
	try {
		mSound->ImportMidi(inDataStream);
		mPath = "";
		mLoaded = true;
		mPlayBar.SetPlayState(PLAYSTATE_STOPPED);
		mManager.SetResourcePath(file.Left(file.ReverseFind('\\')));
		AfxGetMainWnd()->SetWindowText("Soundbox - none");
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Import Error");
	}

	mDisplay.PaintBuffer();
}

void CChildView::OnFileSave132()
{
	// if we don't have a path, ask the user for one
	if (mPath == "") {
		CFileDialog saveDlg(false, 0, "SOUND.000");
		if (saveDlg.DoModal() != IDOK) return;
		mPath = saveDlg.GetPathName();
	}

	// open the file and construct a DataStream around it
	std::fstream outStream;
	outStream.open(mPath, std::ios_base::out | std::ios_base::binary);
	if (!outStream.good()) {
		MessageBox("Unable to open file for saving", "Save Error");
		return;
	}
	DataStream outDataStream(&outStream);

	// save the file
	try {
		mSound->Save(outDataStream);
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Save Error");
	}
}

void CChildView::OnFileSaveas()
{
	// ask the user for a file to save to
	CFileDialog saveDlg(false, 0, "SOUND.000");
	if (saveDlg.DoModal() != IDOK) return;
	mPath = saveDlg.GetPathName();

	// open the file and construct a DataStream around it
	std::fstream outStream;
	outStream.open(mPath, std::ios_base::out | std::ios_base::binary);
	if (!outStream.good()) {
		MessageBox("Unable to open file for saving", "Save Error");
		return;
	}
	DataStream outDataStream(&outStream);

	// save the file
	try {
		mSound->Save(outDataStream);
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Save Error");
	}
}

void CChildView::OnEditLooppoint()
{
	CLoopPointDlg dlg(mSound, this);
	dlg.DoModal();
	mDisplay.PaintBuffer();
}

void CChildView::OnEditCues()
{
	CCuesDlg dlg(mSound, this);
	dlg.DoModal();
	mDisplay.PaintBuffer();
}

void CChildView::OnEditChannelproperties()
{
	CChannelsDlg dlg(mSound, this);
	dlg.DoModal();
	mDisplay.PaintBuffer();
}

void CChildView::OnViewCues()
{
	mDisplay.SetShowCues(!mDisplay.GetShowCues());
}

void CChildView::OnUpdateViewCues(CCmdUI *pCmdUI)
{
	pCmdUI->SetCheck(mDisplay.GetShowCues());
}

void CChildView::OnViewLooppoint()
{
	mDisplay.SetShowLoopPoint(!mDisplay.GetShowLoopPoint());
}

void CChildView::OnUpdateViewLooppoint(CCmdUI *pCmdUI)
{
	pCmdUI->SetCheck(mDisplay.GetShowLoopPoint());
}

void CChildView::OnViewNoteonevents()
{
	mDisplay.SetShowNoteOns(!mDisplay.GetShowNoteOns());
}

void CChildView::OnUpdateViewNoteonevents(CCmdUI *pCmdUI)
{
	pCmdUI->SetCheck(mDisplay.GetShowNoteOns());
}

void CChildView::OnViewTimemarkers()
{
	mDisplay.SetShowTimeRules(!mDisplay.GetShowTimeRules());
}

void CChildView::OnUpdateViewTimemarkers(CCmdUI *pCmdUI)
{
	pCmdUI->SetCheck(mDisplay.GetShowTimeRules());
}

void CChildView::OnViewVoicecontrols()
{
	mDisplay.SetShowVoiceControls(!mDisplay.GetShowVoiceControls());
}

void CChildView::OnUpdateViewVoicecontrols(CCmdUI *pCmdUI)
{
	pCmdUI->SetCheck(mDisplay.GetShowVoiceControls());
}

int CChildView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;

	RECT rect = {0, 0, 0, 0};
	mDisplay.Create(0, 0, WS_CHILD | WS_VISIBLE, rect, this, 10002);
	mPlayBar.Create(0, 0, WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN, rect, this, 10001);

	return 0;
}

void CChildView::OnSize(UINT nType, int cx, int cy)
{
	CWnd::OnSize(nType, cx, cy);

	RECT rect, rect2;
	GetClientRect(&rect);

	rect2.top = 0;
	rect2.left = 0;
	rect2.right = rect.right;
	rect2.bottom = 3 * mVUnit;
	mPlayBar.MoveWindow(&rect2);

	rect2.top = rect2.bottom;
	rect2.bottom = rect.bottom;
	mDisplay.MoveWindow(&rect2);
}

LRESULT CChildView::OnSetFilter(WPARAM wParam, LPARAM lParam)
{
	mFilter = (int)lParam;
	mDisplay.SetFilter(mFilter);
	return 0;
}

LRESULT CChildView::OnSoundTick(WPARAM wParam, LPARAM lParam)
{
	mDisplay.SetPlayPosition((int)wParam);
	return 0;
}

LRESULT CChildView::OnSoundSignal(WPARAM wParam, LPARAM lParam)
{
	mPlayBar.SetSignal((int)wParam);
	return 0;
}

LRESULT CChildView::OnPlayStarted(WPARAM wParam, LPARAM lParam)
{
	mPlayBar.SetPlayState(PLAYSTATE_PLAYING);
	mPlaying = true;
	return 0;
}

LRESULT CChildView::OnPlayStopped(WPARAM wParam, LPARAM lParam)
{
	mDisplay.SetPlayPosition(-1);
	mPlayBar.SetPlayState(PLAYSTATE_STOPPED);

	if (mPlayer->IsError()) {
		MessageBox(mPlayer->GetError().c_str(), "Player Error", MB_OK);
	}
	mPlayer->Cleanup();
	delete mPlayer;
	mPlayer = 0;
	mPlaying = false;
	return 0;
}

LRESULT CChildView::OnPlay(WPARAM wParam, LPARAM lParam)
{
	if (mPlaying) ASSERT("Already playing");
	try {
		// Create and initilize the appropriate player
		mPlayer = 0;
		mPlayer = mManager.CreateAndInit(GetSafeHwnd(), (int)wParam);
		mPlayer->Start(mSound, mFilter);
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Player Error", MB_OK);
		if (mPlayer) delete mPlayer;
		mPlayer = 0;
	}
	return 0;
}

LRESULT CChildView::OnStop(WPARAM wParam, LPARAM lParam)
{
	if (!mPlaying) ASSERT("Not playing");
	try {
		mPlayer->Stop();
	} catch (std::runtime_error &e) {
		MessageBox(e.what(), "Player Error", MB_OK);
	}
	return 0;
}

void CChildView::OnHelpLicense()
{
	CLicenseDlg dlg(this);
	dlg.DoModal();
}
