/////////////////////////////////////////////////////
//
// SimpleStack.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides a definition for a very simple stack
//    template class; no bells and whistles
//
// Portability information:
//    the code in this file should be portable without
//    much difficulty
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "BLGExceptions.h"

/////////////////////////////////////////////////////
//
// template class CSimpleStack
//
/////////////////////////////////////////////////////

template <class T>
class CSimpleStack
{
public:
    /////////////////////////////////////////////////////
    //
    // constructor
    //
    /////////////////////////////////////////////////////
    //
    // Parameter nCapacity:
    //    specifies the maximum number of items that can
    //    be stored in the stack; this number cannot be
    //    changed after the stack has been instantiated,
    //    so set its capacity wisely
    //
    /////////////////////////////////////////////////////

    CSimpleStack(int nCapacity)
    {
        assert(nCapacity > 0);
        m_pTheStack = new T[nCapacity];

        if (!m_pTheStack)
        {
            throw CMallocFailedException();
        }

        m_nTop = 0;
        m_nCapacity = nCapacity;
    }

    /////////////////////////////////////////////////////
    //
    // destructor
    //
    /////////////////////////////////////////////////////

    virtual ~CSimpleStack()
    {
        delete [] m_pTheStack;
    }

    /////////////////////////////////////////////////////
    //
    // Push
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    puts an item on top of the stack
    // Parameter item:
    //    the item that is to be put on the stack
    //
    /////////////////////////////////////////////////////

    void Push(T item)
    {
        assert(!IsFull());
        m_pTheStack[m_nTop++] = item;
    }

    /////////////////////////////////////////////////////
    //
    // Pop
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    removes the topmost item from the stack and
    //    returns it
    // Return value:
    //    the stack's former topmost item
    //
    /////////////////////////////////////////////////////

    T Pop()
    {
        assert(!IsEmpty());
        return m_pTheStack[--m_nTop];
    }

    /////////////////////////////////////////////////////
    //
    // Peek
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    provides a look at what is on top of the stack
    //    without removing it from the stack
    // Return value:
    //    the stack's current topmost item
    //
    /////////////////////////////////////////////////////

    T Peek()
    {
        assert(!IsEmpty());
        return m_pTheStack[m_nTop - 1];
    }

    /////////////////////////////////////////////////////
    //
    // IsEmpty
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    reports whether the stack currently has no
    //    items
    // Return value:
    //    TRUE if there are no items on the stack;
    //    FALSE if there are any items on the stack
    //
    /////////////////////////////////////////////////////

    BOOL IsEmpty()
    {
        // this condition is correct, because m_nTop
        // actually stores the index where the next item
        // pushed onto the stack will go
        return m_nTop == 0;
    }

    /////////////////////////////////////////////////////
    //
    // IsFull
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    reports whether the stack is currently filled
    //    to capacity
    // Return value:
    //    TRUE if no more items can be stored on the
    //    stack; FALSE if more items can be stored on
    //    the stack
    //
    /////////////////////////////////////////////////////

    BOOL IsFull()
    {
        // this condition is correct, because m_nTop
        // actually stores the index where the next item
        // pushed onto the stack will go
        return m_nTop == m_nCapacity;
    }

private:
    // the array of items that represents the stack
    T* m_pTheStack;
    // the maximum number of items this stack object can hold
    int m_nCapacity;
    // the index of the current top of the stack + 1
    int m_nTop;
};