/////////////////////////////////////////////////////
//
// MessageBoxes.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    implements the interface for the various
//    custom message boxes of the BLG and contains
//    their dialog box procedures
//
// Portability information:
//    this file contains code written specifically
//    for a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "resource.h"
#include "BLGDefs.h"
#include "MessageBoxesEventHandlers.h"
#include "MessageBoxes.h"

/////////////////////////////////////////////////////
//
// globals
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in AGIBLG.cpp
extern HINSTANCE g_hInstance;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// declared in MainDlgUI.cpp
extern COptions g_options;
/////////////////////////////////////////////////////

char g_szMBControlName[MAX_CONTROL_NAME_LEN + 1];

/////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////

BOOL CALLBACK UnescapedQuoteMessageBoxDlgProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK EscapedQuoteMessageBoxDlgProc(HWND, UINT, WPARAM, LPARAM);

/////////////////////////////////////////////////////
//
// UnescapedQuoteMessageBox
//
/////////////////////////////////////////////////////
//
// Purpose:
//    provides a somewhat abstract interface to the
//    Unescaped Quote msg box so that modules that need
//    to use this dialog don't have to worry about
//    any of the details of displaying the dialog box;
//    this function also takes care of the details of
//    applying changes when the user clicks OK and
//    discarding changes when the user clicks Cancel
// Parameter hwndParent:
//    the window handle of the window that is to become
//    the message box's parent
// Parameter pszControlName:
//    a null-terminated string specifying the name of the
//    control containing a string with unescaped quotes;
//    must not be NULL
// Return value:
//    one of the various UQMB values defined in MessageBoxes.h
//    that indicates how the user responded to the message box
// Remarks:
//    calling this function does not guarantee that a message
//    box will be displayed...if the user has checked the
//    "Don't ask again" option, then this function will simply
//    return the user's remembered response
//
/////////////////////////////////////////////////////

int UnescapedQuoteMessageBox(HWND hwndParent, PSTR pszControlName)
{
    assert(pszControlName != NULL);
    int nResult;

    // before trying to display the dialog box, make sure the user hasn't
    // specified that they don't want to be asked about this anymore
    switch(g_options.m_nUnescapedQuoteResponse)
    {
    case OPT_UQR_ALWAYS_INCLUDE_QUOTES:
        return UQMB_INCLUDE_QUOTES;
    case OPT_UQR_ALWAYS_REMOVE_QUOTES:
        return UQMB_REMOVE_QUOTES;
    default:
        break;
    }

    strncpy(g_szMBControlName, pszControlName, MAX_CONTROL_NAME_LEN);
    g_szMBControlName[MAX_CONTROL_NAME_LEN] = '\0';

    nResult = DialogBox(g_hInstance, 
                        MAKEINTRESOURCE(IDD_DLG_UNESC_QUOTE),
                        hwndParent,
                        UnescapedQuoteMessageBoxDlgProc);

    return nResult;
}

/////////////////////////////////////////////////////
//
// UnescapedQuoteMessageBoxDlgProc
//
/////////////////////////////////////////////////////
//
// Purpose:
//    dialog box procedure for the Unescaped Quote
//    message box
// Parameters:
//    consult a Windows API reference for the meanings
//    of the parameters
// Return value:
//    TRUE if the BLG handled the event and Windows
//    doesn't need to; FALSE if Windows should handle
//    the event
// Remarks:
//    to prevent the window procedure from getting
//    too large, this function calls event handler
//    functions to handle specific messages
//
/////////////////////////////////////////////////////

BOOL CALLBACK UnescapedQuoteMessageBoxDlgProc(HWND hDlg, UINT uiMsg,
                                              WPARAM wParam, LPARAM lParam)
{
    switch(uiMsg)
    {
    case WM_COMMAND:
        return OnUnescapedQuoteMessageBoxWMCommand(hDlg, wParam, lParam);
    case WM_INITDIALOG:
        return OnUnescapedQuoteMessageBoxWMInitDialog(hDlg, wParam,
                                                      lParam);
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// EscapedQuoteMessageBox
//
/////////////////////////////////////////////////////
//
// Purpose:
//    provides a somewhat abstract interface to the
//    Escaped Quote msg box so that modules that need
//    to use this dialog don't have to worry about
//    any of the details of displaying the dialog box;
//    this function also takes care of the details of
//    applying changes when the user clicks OK and
//    discarding changes when the user clicks Cancel
// Parameter hwndParent:
//    the window handle of the window that is to become
//    the message box's parent
// Parameter pszControlName:
//    a null-terminated string specifying the name of the
//    control containing a string with unescaped quotes;
//    must not be NULL
// Return value:
//    one of the various EQMB values defined in MessageBoxes.h
//    that indicates how the user responded to the message box
// Remarks:
//    calling this function does not guarantee that a message
//    box will be displayed...if the user has checked the
//    "Don't ask again" option, then this function will simply
//    return the user's remembered response
//
/////////////////////////////////////////////////////

int EscapedQuoteMessageBox(HWND hwndParent, PSTR pszControlName)
{
    assert(pszControlName != NULL);
    int nResult;

    // before trying to display the dialog box, make sure the user hasn't
    // specified that they don't want to be asked about this anymore
    switch(g_options.m_nEscapedQuoteResponse)
    {
    case OPT_EQR_ALWAYS_REMOVE_BACKSLASH:
        return EQMB_REMOVE_BACKSLASH;
    case OPT_EQR_ALWAYS_INCLUDE_BACKSLASH:
        return EQMB_INCLUDE_BACKSLASH;
    default:
        break;
    }

    strncpy(g_szMBControlName, pszControlName, MAX_CONTROL_NAME_LEN);
    g_szMBControlName[MAX_CONTROL_NAME_LEN] = '\0';

    nResult = DialogBox(g_hInstance, 
                        MAKEINTRESOURCE(IDD_DLG_ESC_QUOTE),
                        hwndParent,
                        EscapedQuoteMessageBoxDlgProc);

    return nResult;

}

/////////////////////////////////////////////////////
//
// EscapedQuoteMessageBoxDlgProc
//
/////////////////////////////////////////////////////
//
// Purpose:
//    dialog box procedure for the Escaped Quote
//    message box
// Parameters:
//    consult a Windows API reference for the meanings
//    of the parameters
// Return value:
//    TRUE if the BLG handled the event and Windows
//    doesn't need to; FALSE if Windows should handle
//    the event
// Remarks:
//    to prevent the window procedure from getting
//    too large, this function calls event handler
//    functions to handle specific messages
//
/////////////////////////////////////////////////////

BOOL CALLBACK EscapedQuoteMessageBoxDlgProc(HWND hDlg, UINT uiMsg,
                                            WPARAM wParam, LPARAM lParam)
{
    switch(uiMsg)
    {
    case WM_COMMAND:
        return OnEscapedQuoteMessageBoxWMCommand(hDlg, wParam, lParam);
    case WM_INITDIALOG:
        return OnEscapedQuoteMessageBoxWMInitDialog(hDlg, wParam, lParam);
    }

    return FALSE;
}
