/////////////////////////////////////////////////////
//
// EdgeCodeAdvancedDlgUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    Edge Code Advanced dialog; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written specifically for
//    a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include "resource.h"
#include "BLGDefs.h"
#include "GeneralUI.h"
#include "MessageBoxes.h"
#include "EdgeCodeAdvancedDlgUI.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in EdgeCodeAdvancedDlgEventHandlers.cpp
extern HWND g_hwndEdgeCodeAdvancedDlg;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in EdgeCodeAdvancedDlg.cpp
extern CEdgeCodeInfo g_ecinfoTemp;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// TryToEnableEmptyCodeLeftEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    this function attempts to enable the Empty Code
//    for Left Edge check box; it should be called
//    whenever changes are made to the settings in
//    the dialog box that might change the applicability
//    of the check box
// Postconditions:
//    if the check box can be enabled, it is; if it cannot
//    be enabled, it is disabled
//
/////////////////////////////////////////////////////

void TryToEnableEmptyCodeLeftEdge()
{
    char szBuffer[MAX_PRINT_MSG_LEN + 1];
    HWND hwndEdit = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                               IDC_EDIT_LEFT_EDGE_MSG);
    HWND hwndCheck = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                IDC_CHECK_EMPTY_CODE_LEFT_EDGE);
    GetWindowText(hwndEdit, szBuffer, GetWindowTextLength(hwndEdit) + 1);
    
    if (g_ecinfoTemp.m_nLeftGotoRoom == NO_GOTO_ROOM &&
        strlen(szBuffer) == 0)
    {
        EnableWindow(hwndCheck, TRUE);
    }
    else
    {
        EnableWindow(hwndCheck, FALSE);
    }
}

/////////////////////////////////////////////////////
//
// TryToEnableEmptyCodeRightEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    this function attempts to enable the Empty Code
//    for Right Edge check box; it should be called
//    whenever changes are made to the settings in
//    the dialog box that might change the applicability
//    of the check box
// Postconditions:
//    if the check box can be enabled, it is; if it cannot
//    be enabled, it is disabled
//
/////////////////////////////////////////////////////

void TryToEnableEmptyCodeRightEdge()
{
    char szBuffer[MAX_PRINT_MSG_LEN + 1];
    HWND hwndEdit = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                               IDC_EDIT_RIGHT_EDGE_MSG);
    HWND hwndCheck = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                IDC_CHECK_EMPTY_CODE_RIGHT_EDGE);
    GetWindowText(hwndEdit, szBuffer, GetWindowTextLength(hwndEdit) + 1);
    
    if (g_ecinfoTemp.m_nRightGotoRoom == NO_GOTO_ROOM &&
        strlen(szBuffer) == 0)
    {
        EnableWindow(hwndCheck, TRUE);
    }
    else
    {
        EnableWindow(hwndCheck, FALSE);
    }
}

/////////////////////////////////////////////////////
//
// TryToEnableEmptyCodeBottomEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    this function attempts to enable the Empty Code
//    for Bottom Edge check box; it should be called
//    whenever changes are made to the settings in
//    the dialog box that might change the applicability
//    of the check box
// Postconditions:
//    if the check box can be enabled, it is; if it cannot
//    be enabled, it is disabled
//
/////////////////////////////////////////////////////

void TryToEnableEmptyCodeBottomEdge()
{
    char szBuffer[MAX_PRINT_MSG_LEN + 1];
    HWND hwndEdit = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                               IDC_EDIT_BOTTOM_EDGE_MSG);
    HWND hwndCheck = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                IDC_CHECK_EMPTY_CODE_BOTTOM_EDGE);
    GetWindowText(hwndEdit, szBuffer, GetWindowTextLength(hwndEdit) + 1);
    
    if (g_ecinfoTemp.m_nBottomGotoRoom == NO_GOTO_ROOM &&
        strlen(szBuffer) == 0)
    {
        EnableWindow(hwndCheck, TRUE);
    }
    else
    {
        EnableWindow(hwndCheck, FALSE);
    }
}

/////////////////////////////////////////////////////
//
// TryToEnableEmptyCodeHorizonEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    this function attempts to enable the Empty Code
//    for Horizon Edge check box; it should be called
//    whenever changes are made to the settings in
//    the dialog box that might change the applicability
//    of the check box
// Postconditions:
//    if the check box can be enabled, it is; if it cannot
//    be enabled, it is disabled
//
/////////////////////////////////////////////////////

void TryToEnableEmptyCodeHorizonEdge()
{
    char szBuffer[MAX_PRINT_MSG_LEN + 1];
    HWND hwndEdit = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                               IDC_EDIT_HORIZON_EDGE_MSG);
    HWND hwndCheck = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                IDC_CHECK_EMPTY_CODE_HORIZON_EDGE);
    GetWindowText(hwndEdit, szBuffer, GetWindowTextLength(hwndEdit) + 1);
    
    if (g_ecinfoTemp.m_nHorizonGotoRoom == NO_GOTO_ROOM &&
        strlen(szBuffer) == 0)
    {
        EnableWindow(hwndCheck, TRUE);
    }
    else
    {
        EnableWindow(hwndCheck, FALSE);
    }
}

/////////////////////////////////////////////////////
//
// OnEditLeftEdgeMsg
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications related to the Left Edge
//    Message edit box
// Parameter wParam:
//    the wParam that was sent to the dialog box
//    procedure with the WM_COMMAND message; contains
//    the notification code
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditLeftEdgeMsg(WPARAM wParam)
{
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditLeftEdgeMsgENUpdate(); 
    }
    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditLeftEdgeMsgENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    Left Edge Message edit box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditLeftEdgeMsgENUpdate()
{
    TryToEnableEmptyCodeLeftEdge();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditRightEdgeMsg
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications related to the Right Edge
//    Message edit box
// Parameter wParam:
//    the wParam that was sent to the dialog box
//    procedure with the WM_COMMAND message; contains
//    the notification code
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditRightEdgeMsg(WPARAM wParam)
{
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditRightEdgeMsgENUpdate();
    }
    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditRightEdgeMsgENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    Right Edge Message edit box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditRightEdgeMsgENUpdate()
{
    TryToEnableEmptyCodeRightEdge();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditBottomEdgeMsg
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications related to the Bottom Edge
//    Message edit box
// Parameter wParam:
//    the wParam that was sent to the dialog box
//    procedure with the WM_COMMAND message; contains
//    the notification code
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditBottomEdgeMsg(WPARAM wParam)
{
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditBottomEdgeMsgENUpdate();
    }
    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditBottomEdgeMsgENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    Bottom Edge Message edit box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditBottomEdgeMsgENUpdate()
{
    TryToEnableEmptyCodeBottomEdge();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEditHorizonEdgeMsg
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications related to the Horizon Edge
//    Message edit box
// Parameter wParam:
//    the wParam that was sent to the dialog box
//    procedure with the WM_COMMAND message; contains
//    the notification code
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditHorizonEdgeMsg(WPARAM wParam)
{
    switch(HIWORD(wParam))
    {
    case EN_UPDATE:
        return OnEditHorizonEdgeMsgENUpdate();
    }
    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnEditHorizonEdgeMsgENUpdate
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the EN_UPDATE notification for the
//    Horizon Edge Message edit box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEditHorizonEdgeMsgENUpdate()
{
    TryToEnableEmptyCodeHorizonEdge();
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnEdgeCodeAdvancedDlgOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles event for the Edge Code Advanced dialog's
//    OK button
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnEdgeCodeAdvancedDlgOK()
{
    int nUnescapedQuoteCount;
    int nMsgBoxResult;

    g_ecinfoTemp.m_bEmptyLeft = 
        IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                          IDC_CHECK_EMPTY_CODE_LEFT_EDGE);
    g_ecinfoTemp.m_bEmptyRight =
        IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                          IDC_CHECK_EMPTY_CODE_RIGHT_EDGE);
    g_ecinfoTemp.m_bEmptyBottom =
        IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                          IDC_CHECK_EMPTY_CODE_BOTTOM_EDGE);
    g_ecinfoTemp.m_bEmptyHorizon =
        IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                          IDC_CHECK_EMPTY_CODE_HORIZON_EDGE);

    HWND hwndControl;

    hwndControl = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                             IDC_EDIT_LEFT_EDGE_MSG);
    GetWindowText(hwndControl, g_ecinfoTemp.m_szLeftMessage,
                  MAX_PRINT_MSG_LEN + 1);
    nUnescapedQuoteCount = CountUnescapedQuotes(g_ecinfoTemp.m_szLeftMessage);

    if (nUnescapedQuoteCount != 0)
    {
        nMsgBoxResult = UnescapedQuoteMessageBox(g_hwndEdgeCodeAdvancedDlg,
                                                 "Left edge code message");

        switch(nMsgBoxResult)
        {
        case UQMB_INCLUDE_QUOTES:
            if ((nUnescapedQuoteCount + 
                strlen(g_ecinfoTemp.m_szLeftMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEdgeCodeAdvancedDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndControl);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_ecinfoTemp.m_szLeftMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            RemoveUnescapedQuotes(g_ecinfoTemp.m_szLeftMessage);
            break;
        case UQMB_USER_EDIT:
            SetFocus(hwndControl);
            return TRUE;
        }
    }

    // set the entry text to its modified value now so that if the user's
    // input is invalid or they cancel their OK click at a later prompt,
    // they won't have to see this prompt again
    SetWindowText(hwndControl, g_ecinfoTemp.m_szLeftMessage);

    hwndControl = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                             IDC_EDIT_RIGHT_EDGE_MSG);
    GetWindowText(hwndControl, g_ecinfoTemp.m_szRightMessage,
                  MAX_PRINT_MSG_LEN + 1);
    nUnescapedQuoteCount = CountUnescapedQuotes(g_ecinfoTemp.m_szRightMessage);

    if (nUnescapedQuoteCount != 0)
    {
        nMsgBoxResult = UnescapedQuoteMessageBox(g_hwndEdgeCodeAdvancedDlg,
                                                 "Right edge code message");

        switch(nMsgBoxResult)
        {
        case UQMB_INCLUDE_QUOTES:
            if ((nUnescapedQuoteCount + 
                strlen(g_ecinfoTemp.m_szRightMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEdgeCodeAdvancedDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndControl);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_ecinfoTemp.m_szRightMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            RemoveUnescapedQuotes(g_ecinfoTemp.m_szRightMessage);
            break;
        case UQMB_USER_EDIT:
            SetFocus(hwndControl);
            return TRUE;
        }
    }

    // set the entry text to its modified value now so that if the user's
    // input is invalid or they cancel their OK click at a later prompt,
    // they won't have to see this prompt again
    SetWindowText(hwndControl, g_ecinfoTemp.m_szRightMessage);

    hwndControl = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                             IDC_EDIT_BOTTOM_EDGE_MSG);
    GetWindowText(hwndControl, g_ecinfoTemp.m_szBottomMessage,
                  MAX_PRINT_MSG_LEN + 1);

    nUnescapedQuoteCount = 
        CountUnescapedQuotes(g_ecinfoTemp.m_szBottomMessage);

    if (nUnescapedQuoteCount != 0)
    {
        nMsgBoxResult = UnescapedQuoteMessageBox(g_hwndEdgeCodeAdvancedDlg,
                                                 "Bottom edge code message");

        switch(nMsgBoxResult)
        {
        case UQMB_INCLUDE_QUOTES:
            if ((nUnescapedQuoteCount + 
                strlen(g_ecinfoTemp.m_szBottomMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEdgeCodeAdvancedDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndControl);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_ecinfoTemp.m_szBottomMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            RemoveUnescapedQuotes(g_ecinfoTemp.m_szBottomMessage);
            break;
        case UQMB_USER_EDIT:
            SetFocus(hwndControl);
            return TRUE;
        }
    }

    // set the entry text to its modified value now so that if the user's
    // input is invalid or they cancel their OK click at a later prompt,
    // they won't have to see this prompt again
    SetWindowText(hwndControl, g_ecinfoTemp.m_szBottomMessage);


    hwndControl = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                             IDC_EDIT_HORIZON_EDGE_MSG);
    GetWindowText(hwndControl, g_ecinfoTemp.m_szHorizonMessage,
                  MAX_PRINT_MSG_LEN + 1);

    nUnescapedQuoteCount = 
        CountUnescapedQuotes(g_ecinfoTemp.m_szHorizonMessage);

    if (nUnescapedQuoteCount != 0)
    {
        nMsgBoxResult = UnescapedQuoteMessageBox(g_hwndEdgeCodeAdvancedDlg,
                                                 "Horizon edge code message");

        switch(nMsgBoxResult)
        {
        case UQMB_INCLUDE_QUOTES:
            if ((nUnescapedQuoteCount + 
                strlen(g_ecinfoTemp.m_szHorizonMessage)) > MAX_PRINT_MSG_LEN)
            {
                MessageBox(g_hwndEdgeCodeAdvancedDlg,
                           "String with escaped quotes is too big",
                           "Error", MB_OK);
                SetFocus(hwndControl);
                return TRUE;
            }

            ConvertUnescapedQuotesToEscapedQuotes(
                g_ecinfoTemp.m_szHorizonMessage);
            break;
        case UQMB_REMOVE_QUOTES:
            RemoveUnescapedQuotes(g_ecinfoTemp.m_szHorizonMessage);
            break;
        case UQMB_USER_EDIT:
            SetFocus(hwndControl);
            return TRUE;
        }
    }

    // set the entry text to its modified value now so that if the user's
    // input is invalid or they cancel their OK click at a later prompt,
    // they won't have to see this prompt again
    SetWindowText(hwndControl, g_ecinfoTemp.m_szHorizonMessage);

    EndDialog(g_hwndEdgeCodeAdvancedDlg, IDOK);
    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnCheckEmptyCodeLeftEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events for the Empty Code for Left Edge
//    check box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnCheckEmptyCodeLeftEdge()
{
    BOOL bChecked = IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                                      IDC_CHECK_EMPTY_CODE_LEFT_EDGE);
    HWND hwndEditLeftEdgeMsg = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                          IDC_EDIT_LEFT_EDGE_MSG);

    if (bChecked)
    {
        EnableWindow(hwndEditLeftEdgeMsg, FALSE);
        SetWindowText(hwndEditLeftEdgeMsg, "");
    }
    else
    {
        EnableWindow(hwndEditLeftEdgeMsg, TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnCheckEmptyCodeRightEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events for the Empty Code for Right Edge
//    check box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnCheckEmptyCodeRightEdge()
{
    BOOL bChecked = IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                                      IDC_CHECK_EMPTY_CODE_RIGHT_EDGE);
    HWND hwndEditRightEdgeMsg = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                           IDC_EDIT_RIGHT_EDGE_MSG);

    if (bChecked)
    {
        EnableWindow(hwndEditRightEdgeMsg, FALSE);
        SetWindowText(hwndEditRightEdgeMsg, "");
    }
    else
    {
        EnableWindow(hwndEditRightEdgeMsg, TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnCheckEmptyCodeBottomEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events for the Empty Code for Bottom Edge
//    check box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnCheckEmptyCodeBottomEdge()
{
    BOOL bChecked = IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                                      IDC_CHECK_EMPTY_CODE_BOTTOM_EDGE);
    HWND hwndEditBottomEdgeMsg = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                            IDC_EDIT_BOTTOM_EDGE_MSG);

    if (bChecked)
    {
        EnableWindow(hwndEditBottomEdgeMsg, FALSE);
        SetWindowText(hwndEditBottomEdgeMsg, "");
    }
    else
    {
        EnableWindow(hwndEditBottomEdgeMsg, TRUE);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnCheckEmptyCodeHorizonEdge
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events for the Empty Code for Horizon Edge
//    check box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnCheckEmptyCodeHorizonEdge()
{
    BOOL bChecked = IsCheckboxChecked(g_hwndEdgeCodeAdvancedDlg,
                                      IDC_CHECK_EMPTY_CODE_HORIZON_EDGE);
    HWND hwndEditHorizonEdgeMsg = GetDlgItem(g_hwndEdgeCodeAdvancedDlg,
                                             IDC_EDIT_HORIZON_EDGE_MSG);

    if (bChecked)
    {
        EnableWindow(hwndEditHorizonEdgeMsg, FALSE);
        SetWindowText(hwndEditHorizonEdgeMsg, "");
    }
    else
    {
        EnableWindow(hwndEditHorizonEdgeMsg, TRUE);
    }

    return TRUE;
}