/////////////////////////////////////////////////////
//
// DefineNameList.h
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    declaration of the CDefineNameList class
//
// Portability information:
//    the code in this file should be portable without
//    modification; if it is not, feel free to modify
//    it so that it is portable; do not add non-portable
//    code to this file
//
//    the constant _MAX_PATH will need to be defined
//    for ports of the Base Logic Generator; it should
//    be defined in BLGDefs.h
//
/////////////////////////////////////////////////////

// make sure this header can't be included twice
#if !defined(DEFINE_NAME_LIST_H_INCLUDED)
#define DEFINE_NAME_LIST_H_INCLUDED

// for some reason, the Visual C++ fstream virtual header
// doesn't define ofstream or ifstream, so fstream.h is needed
#include <fstream.h>
#include "BLGDefs.h"

/////////////////////////////////////////////////////
//
// class CDefineNameList
//
/////////////////////////////////////////////////////

class CDefineNameList
{
public:
    /////////////////////////////////////////////////////
    //
    // constructor
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    constructs a CDefineNameList object
    // Remarks:
    //    when a CDefineNameList object is created, the
    //    define names are intially set to the AGI Studio
    //    template game's define names
    //
    /////////////////////////////////////////////////////

    CDefineNameList();

    /////////////////////////////////////////////////////
    //
    // destructor
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    performs any necessary cleanup work for a
    //    CDefineNameList object before it is destroyed
    //
    /////////////////////////////////////////////////////

    virtual ~CDefineNameList();

    /////////////////////////////////////////////////////
    //
    // IsValidDefineName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    determines if a given string contains a valid
    //    AGI define name; a valid AGI define name
    //    contains letters, numbers, and the characters
    //    '.' and '_'
    // Parameter pszDefName:
    //    a pointer to the string to be tested; must
    //    not be NULL
    // Return value:
    //    TRUE if the string is a valid AGI define name;
    //    FALSE if not
    // Remarks:
    //    this member function is static; it can be called
    //    even if there is no current instance of the
    //    CDefineNameList class available
    //
    /////////////////////////////////////////////////////

    static BOOL IsValidDefineName(PSTR pszDefName);

    /////////////////////////////////////////////////////
    //
    // BecomeCopy
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    makes this CDefineNameList object into a copy of
    //    another CDefineNameList object; similar to a copy
    //    constructor, except it's not a constructor
    // Parameter deflistSrc:
    //    the CDefineNameList object that this CDefineNameList
    //    object is to become a copy of; this object is not
    //    changed
    //
    /////////////////////////////////////////////////////

    void BecomeCopy(CDefineNameList& deflistSrc);

    /////////////////////////////////////////////////////
    //
    // SetDesiredPath
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the path where this define name object
    //    would PREFER to be saved; this value might be
    //    ignored
    // Parameter pszDesiredPath:
    //    a string containing the desired path; must not
    //    be NULL and must be less than _MAX_PATH
    //
    /////////////////////////////////////////////////////

    void SetDesiredPath(PCSTR pszDesiredPath);

    /////////////////////////////////////////////////////
    //
    // GetDesiredPath
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the path where this define name object
    //    would prefer to be saved
    // Parameter psz:
    //    a pointer to a string buffer where the path
    //    string will go; must not be NULL and must be
    //    big enough to hold the path string
    // Parameter nMaxChars:
    //    the maximum number of characters of the path
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetDesiredPath(PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetDesiredPathLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the path where this 
    //    define name object would prefer to be saved
    // Return value:
    //    the number of characters in the path string,
    //    not including the null character
    //
    /////////////////////////////////////////////////////

    int GetDesiredPathLength();

    /////////////////////////////////////////////////////
    //
    // SaveToDesiredPath
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    instructs this CDefineNameList object to 
    //    save a representation of itself to its
    //    desired path
    // Return value:
    //    TRUE if the file is saved successfully;
    //    FALSE if not
    //
    /////////////////////////////////////////////////////

    BOOL SaveToDesiredPath();

    /////////////////////////////////////////////////////
    //
    // LoadFromDesiredPath
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    instructs this CDefineNameList object to load
    //    its data from its desired path
    // Return value:
    //    TRUE if the file is successfully loaded;
    //    otherwise FALSE
    //
    /////////////////////////////////////////////////////

    BOOL LoadFromDesiredPath();

    /////////////////////////////////////////////////////
    //
    // Load
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    instructs this CDefineNameList object to load
    //    its data from a file
    // Parameter pszPath:
    //    the path of the file from which this CDefineNameList
    //    object should load its data; must not be NULL
    // Return value:
    //    TRUE if the data is loaded successfully;
    //    otherwise FALSE
    //
    /////////////////////////////////////////////////////

    BOOL Load(PSTR pszPath);

    /////////////////////////////////////////////////////
    //
    // SetVarName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the define name for one of the AGI
    //    reserved variables
    // Parameter ulVarNum:
    //    number of the reserved variable whose name
    //    is to be set; must be less than RESERVED_VAR_COUNT,
    //    as defined in BLGDefs.h
    // Parameter pszNewName:
    //    the new name for the variable; must not be
    //    NULL and must not be longer than MAX_DEFINE_NAME_LEN
    //    characters
    //
    /////////////////////////////////////////////////////

    void SetVarName(ULONG ulVarNum, PCSTR pszNewName);

    /////////////////////////////////////////////////////
    //
    // SetFlagName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the define name for one of the AGI
    //    reserved flags
    // Parameter ulFlagNum:
    //    number of the reserved flag whose name
    //    is to be set; must be less than RESERVED_FLAG_COUNT,
    //    as defined in BLGDefs.h
    // Parameter pszNewName:
    //    the new name for the flag; must not be
    //    NULL and must not be longer than MAX_DEFINE_NAME_LEN
    //    characters
    //
    /////////////////////////////////////////////////////

    void SetFlagName(ULONG ulFlagNum, PCSTR pszNewName);

    /////////////////////////////////////////////////////
    //
    // SetStrName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the define name for one of the AGI
    //    reserved strings
    // Parameter ulStrNum:
    //    number of the reserved string whose name
    //    is to be set; must be less than RESERVED_STR_COUNT,
    //    as defined in BLGDefs.h
    // Parameter pszNewName:
    //    the new name for the string; must not be
    //    NULL and must not be longer than MAX_DEFINE_NAME_LEN
    //    characters
    //
    /////////////////////////////////////////////////////

    void SetStrName(ULONG ulStrNum, PCSTR pszNewName);

    /////////////////////////////////////////////////////
    //
    // SetConstName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the define name for one of the constants
    //    that is meaningful to the AGI interpreter
    // Parameter ulConstID:
    //    number of the constant whose name is to be set; 
    //    must be less than RESERVED_CONST_COUNT, as defined 
    //    in BLGDefs.h; the AGI_CONST... defines in 
    //    BLGDefs.h give the numbers corresponding
    //    to each AGI constant
    // Parameter pszNewName:
    //    the new name for the constant; must not be
    //    NULL and must not be longer than MAX_DEFINE_NAME_LEN
    //    characters
    //
    /////////////////////////////////////////////////////

    void SetConstName(ULONG ulConstID, PCSTR pszNewName);

    /////////////////////////////////////////////////////
    //
    // SetObjName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    sets the define name for one of the AGI
    //    reserved objects
    // Parameter ulObjNum:
    //    number of the reserved object whose name
    //    is to be set; must be less than RESERVED_OBJ_COUNT,
    //    as defined in BLGDefs.h
    // Parameter pszNewName:
    //    the new name for the object; must not be
    //    NULL and must not be longer than MAX_DEFINE_NAME_LEN
    //    characters
    //
    /////////////////////////////////////////////////////

    void SetObjName(ULONG ulObjNum, PCSTR pszNewName);

    /////////////////////////////////////////////////////
    //
    // GetVarName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the define name for one of the AGI
    //    reserved variables
    // Parameter ulVarNum:
    //    number of the reserved variable whose name
    //    is to be retrieved; must be less than RESERVED_VAR_COUNT,
    //    as defined in BLGDefs.h
    // Parameter psz:
    //    a pointer to a buffer that will receive the name
    //    of the variable; must not be NULL and must be
    //    large enough to hold the entire define name
    // Parameter nMaxChars:
    //    the maximum number of characters of the define
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetVarName(ULONG ulVarNum, PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetFlagName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the define name for one of the AGI
    //    reserved flags
    // Parameter ulFlagNum:
    //    number of the reserved flag whose name
    //    is to be retrieved; must be less than RESERVED_FLAG_COUNT,
    //    as defined in BLGDefs.h
    // Parameter psz:
    //    a pointer to a buffer that will receive the name
    //    of the flag; must not be NULL and must be
    //    large enough to hold the entire define name
    // Parameter nMaxChars:
    //    the maximum number of characters of the define
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetFlagName(ULONG ulFlagNum, PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetStrName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the define name for one of the AGI
    //    reserved strings
    // Parameter ulStrNum:
    //    number of the reserved string whose name
    //    is to be retrieved; must be less than RESERVED_STR_COUNT,
    //    as defined in BLGDefs.h
    // Parameter psz:
    //    a pointer to a buffer that will receive the name
    //    of the string; must not be NULL and must be
    //    large enough to hold the entire define name
    // Parameter nMaxChars:
    //    the maximum number of characters of the define
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetStrName(ULONG ulStrNum, PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetConstName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the define name for one of the constants
    //    that is meaningful to the AGI interpreter
    // Parameter ulConstID:
    //    number of the constant whose name is to be retrieved; 
    //    must be less than RESERVED_CONST_COUNT, as defined 
    //    in BLGDefs.h; the AGI_CONST... defines in 
    //    BLGDefs.h give the numbers corresponding
    //    to each AGI constant
    // Parameter psz:
    //    a pointer to a buffer that will receive the name
    //    of the constant; must not be NULL and must be
    //    large enough to hold the entire define name
    // Parameter nMaxChars:
    //    the maximum number of characters of the define
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetConstName(ULONG ulConstID, PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetObjName
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the define name for one of the AGI
    //    reserved objects
    // Parameter ulObjNum:
    //    number of the reserved object whose name
    //    is to be retrieved; must be less than RESERVED_OBJ_COUNT,
    //    as defined in BLGDefs.h
    // Parameter psz:
    //    a pointer to a buffer that will receive the name
    //    of the obj; must not be NULL and must be
    //    large enough to hold the entire define name
    // Parameter nMaxChars:
    //    the maximum number of characters of the define
    //    name that are to be retrieved, including the
    //    null character
    //
    /////////////////////////////////////////////////////

    void GetObjName(ULONG ulObjNum, PSTR psz, int nMaxChars);

    /////////////////////////////////////////////////////
    //
    // GetVarNameLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the define name currently
    //    assigned to one of the AGI variables
    // Parameter ulVarNum:
    //    number of the reserved variable whose name length
    //    is to be retrieved; must be less than RESERVED_VAR_COUNT,
    //    as defined in BLGDefs.h
    // Return value:
    //    the number of characters in the variable's define
    //    name, not including the null character
    //
    /////////////////////////////////////////////////////

    int GetVarNameLength(ULONG ulVarNum);

    /////////////////////////////////////////////////////
    //
    // GetFlagNameLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the define name currently
    //    assigned to one of the AGI flags
    // Parameter ulFlagNum:
    //    number of the reserved flag whose name length
    //    is to be retrieved; must be less than RESERVED_FLAG_COUNT,
    //    as defined in BLGDefs.h
    // Return value:
    //    the number of characters in the flag's define
    //    name, not including the null character
    //
    /////////////////////////////////////////////////////

    int GetFlagNameLength(ULONG ulFlagNum);

    /////////////////////////////////////////////////////
    //
    // GetStrNameLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the define name currently
    //    assigned to one of the AGI strings
    // Parameter ulStrNum:
    //    number of the reserved string whose name length
    //    is to be retrieved; must be less than RESERVED_STR_COUNT,
    //    as defined in BLGDefs.h
    // Return value:
    //    the number of characters in the string's define
    //    name, not including the null character
    //
    /////////////////////////////////////////////////////

    int GetStrNameLength(ULONG ulStrNum);

    /////////////////////////////////////////////////////
    //
    // GetConstNameLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the define name currently
    //    assigned to one of the AGI constants
    // Parameter ulConstID:
    //    number of the constant whose name length is to be 
    //    retrieved; must be less than RESERVED_CONST_COUNT, 
    //    as defined in BLGDefs.h; the AGI_CONST... defines in 
    //    BLGDefs.h give the numbers corresponding
    //    to each AGI constant
    // Return value:
    //    the number of characters in the constant's define
    //    name, not including the null character
    //
    /////////////////////////////////////////////////////

    int GetConstNameLength(ULONG ulConstID);

    /////////////////////////////////////////////////////
    //
    // GetObjNameLength
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    retrieves the length of the define name currently
    //    assigned to one of the AGI objects
    // Parameter ulStrNum:
    //    number of the reserved object whose name length
    //    is to be retrieved; must be less than RESERVED_OBJ_COUNT,
    //    as defined in BLGDefs.h
    // Return value:
    //    the number of characters in the object's define
    //    name, not including the null character
    //
    /////////////////////////////////////////////////////

    int GetObjNameLength(ULONG ulObjNum);

protected:
    char m_aszVarNames[RESERVED_VAR_COUNT][MAX_DEFINE_NAME_LEN + 1];
    char m_aszFlagNames[RESERVED_FLAG_COUNT][MAX_DEFINE_NAME_LEN + 1];
    char m_aszStrNames[RESERVED_STR_COUNT][MAX_DEFINE_NAME_LEN + 1];
    char m_aszConstNames[RESERVED_CONST_COUNT][MAX_DEFINE_NAME_LEN + 1];
    char m_aszObjNames[RESERVED_OBJ_COUNT][MAX_DEFINE_NAME_LEN + 1];

    char m_szDesiredPath[_MAX_PATH + 1];

    /////////////////////////////////////////////////////
    //
    // Load2Pt0DefineNameList
    //
    /////////////////////////////////////////////////////
    //
    // Purpose:
    //    loads a version 2.0 define name list into this
    //    CDefineNameList object; if changes to the BLG
    //    change the format of the define name list file,
    //    then a new version should be defined and another
    //    member function should carry out loading it; this
    //    function should not be modified in a significant
    //    way, so that old define name lists can continue
    //    to be loaded even if a new format is defined
    // Parameter file:
    //    an already open ifstream file from which to load
    //    the resource
    // Return value:
    //    TRUE if the entire file is read successfully;
    //    FALSE if the operation fails at any point; note
    //    that even if the operation fails, the function
    //    salvages as much of the file as it can
    //
    /////////////////////////////////////////////////////

    BOOL Load2Pt0DefineNameList(ifstream& file);
};

#endif // !defined(DEFINE_NAME_LIST_H_INCLUDED)