/////////////////////////////////////////////////////
//
// DefineNameDlgUI.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides event handlers for user-interface
//    events such as button presses, etc., for the
//    Define Name dialog; also provides functions
//    that update the user interface as necessary to
//    reflect the current state of the application
//
// Portability information:
//    this file contains code written specifically for
//    a Win32 build of the BLG
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include "resource.h"
#include "BLGDefs.h"
#include "GeneralUI.h"
#include "DefineNameList.h"
#include "EditDefineNameDlg.h"
#include "DefineNameDlgUI.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in DefineNameDlgEventHandlers.cpp
extern HWND g_hwndDefineNameDlg;
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
// defined in DefineNameDlg.cpp
extern CDefineNameList g_deflistTemp;
/////////////////////////////////////////////////////

int g_nPrevDefTypeSel;

/////////////////////////////////////////////////////
//
// OnListDefinedNames
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications from the Defined Names
//    list box
// Parameter wParam:
//    the wParam value that contains the notification
//    code indicating what type of event has occurred
// Return value:
//    TRUE if the event is handled; FALSE if not
//
/////////////////////////////////////////////////////

BOOL OnListDefinedNames(WPARAM wParam)
{
    // figure out what type of notification this is
    switch(HIWORD(wParam))
    {
    case LBN_SELCHANGE:
        // the user changed the selection
        return OnListDefinedNamesLBNSelChange();

    case LBN_DBLCLK:
        // the user double-clicked an item
        return OnListDefinedNamesLBNDblClick();
    }
    return FALSE;
}

BOOL OnListDefinedNamesLBNSelChange()
{
    int nCurSel;
    
    nCurSel = SendMessage(
        GetDlgItem(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES),
        LB_GETCURSEL, 0, 0);
    EnableEditButton(nCurSel != LB_ERR);

    return TRUE;
}

BOOL OnListDefinedNamesLBNDblClick()
{
    return OnButtonEditDefine();
}

void EnableEditButton(BOOL bEnable /* = TRUE */)
{
    EnableWindow(GetDlgItem(g_hwndDefineNameDlg, IDC_BUTTON_EDIT_DEFINE),
                 bEnable);
}

/////////////////////////////////////////////////////
//
// OnComboDefineType
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles notifications from the Define Type 
//    combo box
// Parameter wParam:
//    the wParam value that contains the notification
//    code indicating what type of event has occurred
// Return value:
//    TRUE if the event is handled; FALSE if not
//
/////////////////////////////////////////////////////

BOOL OnComboDefineType(WPARAM wParam)
{
    switch(HIWORD(wParam))
    {
    case CBN_SELCHANGE:
        return OnComboDefineTypeCBNSelChange();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnComboDefineTypeCBNSelChange
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles the CBN_SELCHANGE notification for
//    the Define Type combo box
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnComboDefineTypeCBNSelChange()
{
    char szBuffer[MAX_DEFINE_NAME_LEN + 1];
    int nCurSel = SendMessage(GetDlgItem(g_hwndDefineNameDlg,
                                         IDC_COMBO_DEFINE_TYPE),
                              CB_GETCURSEL, 0, 0);
    ULONG ul;

    EnableEditButton(FALSE);

    HWND hwndLB = GetDlgItem(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES);

    switch(g_nPrevDefTypeSel)
    {
    case 0:
        for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetVarName(ul, szBuffer);
        }
        break;

    case 1:
        for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetFlagName(ul, szBuffer);
        }
        break;

    case 2:
        for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetStrName(ul, szBuffer);
        }
        break;

    case 3:
        for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetConstName(ul, szBuffer);
        }
        break;
    case 4:
        for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetObjName(ul, szBuffer);
        }
        break;
    }

    g_nPrevDefTypeSel = nCurSel;

    // empty the list box
    SendMessage(GetDlgItem(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES),
                LB_RESETCONTENT, 0, 0);

    switch(nCurSel)
    {
    case 0:
        // vars selected
        for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
        {
            g_deflistTemp.GetVarName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
            ListBoxAddString(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES, 
                             szBuffer);
        }
        return TRUE;
    case 1:
        // flags selected
        for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
        {
            g_deflistTemp.GetFlagName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
            ListBoxAddString(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES, 
                             szBuffer);
        }
        return TRUE;
    case 2:
        // strings selected
        for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
        {
            g_deflistTemp.GetStrName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
            ListBoxAddString(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES, 
                             szBuffer);
        }
        return TRUE;
    case 3:
        // constants selected
        for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
        {
            g_deflistTemp.GetConstName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
            ListBoxAddString(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES, 
                             szBuffer);
        }
        return TRUE;
    case 4:
        // objects selected
        for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
        {
            g_deflistTemp.GetObjName(ul, szBuffer, MAX_DEFINE_NAME_LEN + 1);
            ListBoxAddString(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES,
                             szBuffer);
        }
        return TRUE;
    default:
        SendMessage(GetDlgItem(g_hwndDefineNameDlg, IDC_COMBO_DEFINE_TYPE),
                    CB_SETCURSEL, 0, 0);
        // call this function again, because Windows doesn't send a
        // notification when the combo box selection is changed with
        // a CB_SETCURSEL message
        return OnComboDefineTypeCBNSelChange();
    }

    return FALSE;
}

/////////////////////////////////////////////////////
//
// OnButtonEditDefine
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles events for the Edit Define button
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnButtonEditDefine()
{
    char szBuffer[MAX_DEFINE_NAME_LEN + 1];
    HWND hwndLB = GetDlgItem(g_hwndDefineNameDlg, 
                             IDC_LIST_DEFINED_NAMES);

    int nCurSel = SendMessage(hwndLB, LB_GETCURSEL, 0, 0);

    if (nCurSel == LB_ERR)
    {
        MessageBox(g_hwndDefineNameDlg,"You must select a define name to edit "
                                       "before clicking the Edit button.",
                   "Error", MB_OK);
        return TRUE;
    }

    SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(nCurSel), 
                reinterpret_cast<LPARAM>(szBuffer));

    if (EditDefineNameDlgDoModal(szBuffer) == IDOK)
    {
        SendMessage(hwndLB, LB_DELETESTRING, static_cast<WPARAM>(nCurSel), 0);
        SendMessage(hwndLB, LB_INSERTSTRING, static_cast<WPARAM>(nCurSel),
                    reinterpret_cast<LPARAM>(szBuffer));
        SendMessage(hwndLB, LB_SETCURSEL, static_cast<WPARAM>(nCurSel), 0);
    }

    return TRUE;
}

/////////////////////////////////////////////////////
//
// OnDefineNameDlgOK
//
/////////////////////////////////////////////////////
//
// Purpose:
//    handles event for the Define Name dialog's
//    OK button
// Return value:
//    TRUE if event is handled; FALSE otherwise
//
/////////////////////////////////////////////////////

BOOL OnDefineNameDlgOK()
{
    int nDefType = SendMessage(GetDlgItem(g_hwndDefineNameDlg, 
                                          IDC_COMBO_DEFINE_TYPE),
                                          CB_GETCURSEL, 0, 0);

    ULONG ul;
    char szBuffer[MAX_DEFINE_NAME_LEN + 1];
    HWND hwndLB = GetDlgItem(g_hwndDefineNameDlg, IDC_LIST_DEFINED_NAMES);

    switch(nDefType)
    {
    case 0:
        for (ul = 0; ul < RESERVED_VAR_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetVarName(ul, szBuffer);
        }
        break;
    case 1:
        for (ul = 0; ul < RESERVED_FLAG_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetFlagName(ul, szBuffer);
        }
        break;
    case 2:
        for (ul = 0; ul < RESERVED_STR_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetStrName(ul, szBuffer);
        }
        break;
    case 3:
        for (ul = 0; ul < RESERVED_CONST_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetConstName(ul, szBuffer);
        }
        break;
    case 4:
        for (ul = 0; ul < RESERVED_OBJ_COUNT; ul++)
        {
            SendMessage(hwndLB, LB_GETTEXT, static_cast<WPARAM>(ul),
                        reinterpret_cast<LPARAM>(szBuffer));
            g_deflistTemp.SetObjName(ul, szBuffer);            
        }
    }

    EndDialog(g_hwndDefineNameDlg, IDOK);
    return TRUE;
}