/////////////////////////////////////////////////////
//
// Clipboard.cpp
//
// Original author:
//    Joel McCormick
//
// Purpose of this file:
//    provides the implementation of the abstract
//    clipboard interface; here is where operating
//    system dependent code relating to the clipboard
//    mechanism should go
//
// Portability information:
//    ports to different operating systems will need
//    to provide their own implementation of the clipboard
//    mechanism; currently, the clipboard mechanism
//    is defined for Win32 only
//
/////////////////////////////////////////////////////

/////////////////////////////////////////////////////
//
// includes
//
/////////////////////////////////////////////////////

#include <cassert>
#include "BLGExceptions.h"
#include "Clipboard.h"

/////////////////////////////////////////////////////
//
// global variables
//
/////////////////////////////////////////////////////

#if defined(WIN32)

/////////////////////////////////////////////////////
// defined in MainDlgEventHandlers.h

// in a Win32 build, need the main window handle to tell
// Windows which window is trying to get ownership of
// the clipboard
extern HWND g_hwndMainDlg;

/////////////////////////////////////////////////////

#endif // defined(WIN32)

/////////////////////////////////////////////////////
//
// WriteToClipboardSupported
//
/////////////////////////////////////////////////////
//
// Purpose:
//    indicates whether the current build of the
//    software supports the clipboard mechanism; this
//    is used by the source code generation module
//    to determine whether a write to the clipboard
//    is possible; it may also be used by the user
//    interface to determine whether the option
//    of writing to the clipboard may be selected
//    as the only means of generating a source file
// Return value:
//    TRUE if the clipboard mechanism is supported;
//    FALSE if it is not
//
/////////////////////////////////////////////////////

BOOL WriteToClipboardSupported()
{
#if defined(WIN32)

    /////////////////////////////////////////////////////
    //
    // begin Win32 specific implementation of
    // WriteToClipboardSupported
    //
    /////////////////////////////////////////////////////

    return TRUE;

    /////////////////////////////////////////////////////
    //
    // end Win32 specific implementation of
    // WriteToClipboardSupported
    //
    /////////////////////////////////////////////////////

#endif // defined(WIN32)


}

/////////////////////////////////////////////////////
//
// WriteTextToClipboard
//
/////////////////////////////////////////////////////
//
// Purpose:
//    provides an abstraction of the clipboard
//    mechanism so that the source code generation
//    module doesn't have to worry about the specifics
//    of how a particular operating system's clipboard
//    mechanism is used
// Parameter pszText:
//    a pointer to a null-terminated string that is
//    to be written to the clipboard; must not be
//    NULL
//
/////////////////////////////////////////////////////

void WriteTextToClipboard(PSTR pszText)
{
    assert(pszText != NULL);
    assert(WriteToClipboardSupported());

#if defined(WIN32)

    /////////////////////////////////////////////////////
    //
    // begin Win32-specific implementation of
    // WriteTextToClipboard
    //
    /////////////////////////////////////////////////////

    // a handle to global memory (memory that can be shared
    // between applications)
    HGLOBAL hglobal;
    // a pointer to the memory allocated for the clipboard text
    PSTR pszClipboardMem;
    // the length of the text to write to the clipboard
    int nStrLen = strlen(pszText);

    // don't even bother if the string has no text
    if (nStrLen <= 0)
    {
        return;
    }

    // allocate a global shared memory region
    hglobal = GlobalAlloc(GHND | GMEM_SHARE, nStrLen + 1);

    if (!hglobal)
    {
        throw CMallocFailedException();
    }

    // get a pointer to the shared memory
    pszClipboardMem = reinterpret_cast<PSTR>(GlobalLock(hglobal));

    if (!pszClipboardMem)
    {
        throw CMallocFailedException();
    }

    // copy the text to the shared memory
    strcpy(pszClipboardMem, pszText);

    // release the shared memory
    GlobalUnlock(hglobal);

    // try to get the clipboard
    if (!OpenClipboard(g_hwndMainDlg))
    {
        throw COpenClipboardFailedException();
    }

    // remove the current contents of the clipboard
    EmptyClipboard();

    // tell the clipboard to take text data from the shared memory region
    SetClipboardData(CF_TEXT, hglobal);
    // release the clipboard so that other apps can use it
    CloseClipboard();

    /////////////////////////////////////////////////////
    //
    // end Win32-specific implementation of
    // WriteTextToClipboard
    //
    /////////////////////////////////////////////////////

#endif // defined(WIN32)

}
