/* SGSDK.H */

/******* Common definitions and typedefs ***********************************/

#define VOID                void

#define FAR             _far
#define NEAR                _near
#define PASCAL              _pascal
#define CDECL               _cdecl

/****** Simple types & common helper macros *********************************/

typedef int                 BOOL;
#define FALSE               0
#define TRUE                1

typedef unsigned char       BYTE;
typedef unsigned short      WORD;
typedef unsigned long       DWORD;

typedef unsigned int        UINT;

typedef signed long         LONG;

#define LOBYTE(w)           ((BYTE)(w))
#define HIBYTE(w)           ((BYTE)(((UINT)(w) >> 8) & 0xFF))

#define LOWORD(l)           ((WORD)(DWORD)(l))
#define HIWORD(l)           ((WORD)((((DWORD)(l)) >> 16) & 0xFFFF))

#define MAKELONG(low, high) ((LONG)(((WORD)(low)) | (((DWORD)((WORD)(high))) << 16)))

#define max(a,b)            (((a) > (b)) ? (a) : (b))
#define min(a,b)            (((a) < (b)) ? (a) : (b))

#define NO_ERROR        0

/* Mixer/Codec general */
#define MONO            0
#define STEREO          1
#define STEP8           0
#define STEP16          1
#define ON              1
#define OFF             0

/* Mixer recording source */
#define MICROPHONE      0
#define CD              1
#define SPARE           2
#define LINE_IN         3

/* Mixer filter */
#define LOWFILTER       0
#define HIGHFILTER      1
#define FILTER_ON       0
#define FILTER_OFF      1

/* Mixer volume */
#define MASTERVOL       0
#define VOICEVOL        1
#define FMVOL           2
#define CDVOL           3
#define LINE_INVOL      4
#define MICVOL          5
#define WAVESYNVOL      6
#define SPEAKERVOL      7
#define BASSVOL         0
#define TREBLEVOL       1

/* Mixer gain */
#define RECGAIN         0
#define PLAYGAIN        1

/* Codec input source */
#define CODEC_LINE      0
#define CODEC_AUX1      1
#define CODEC_MIC       2
#define CODEC_LINEDAC   3

/* Codec data format */
#define PCM             1
#define ALAW            6
#define MULAW           7
#define ADPCM_IMA       17
#define _8BIT           0
#define _16BIT          1

/* Operating mode of sound card */
#define SB8MODE         0       // SB 8bit stereo mode
#define MSSMODE         1       // MSS mode
#define DMA8MODE        2       // 8bit DMA mode
#define DMA16MODE       3       // 16bit DMA mode

/*  Creative Voice file header structure  */
typedef struct
{
	char       id[20];
	unsigned   voice_offset;
	unsigned   version;
	unsigned   check_code;
} VOCHDR;

/* External Global variables */
extern WORD sg_voice_status;

/* Global variables */
BYTE    gbCardMode = SB8MODE;
//WORD    gwMSSPort = 0x530;
//WORD    gwCODECPort = 0x534;

/********************/
/* Common functions */
/********************/

// Get Library Version
// parameters: none
// return high byte:major version and low byte:minor version
extern unsigned sgGetVocLibVersion(void);

// Get Library Version
// parameters: none
// return high byte:major version and low byte:minor version
extern unsigned sgGetVocdLibVersion(void);

// Get Library Version
// parameters: none
// return high byte:major version and low byte:minor version
extern unsigned sgGetWavLibVersion(void);

// Get Library Version
// parameters: none
// return high byte:major version and low byte:minor version
extern unsigned sgGetMixLibVersion(void);

// Set Base I/O Address
// parameters: Base I/O Address
// return 0 for success and non-zero for failure
extern int sgSetBaseAddress(unsigned IOAddress);

// Set Interrupt for DMA
// parameters: Interrupt Number
// return 0 for success and non-zero for failure
extern int sgSetIRQ(unsigned IRQNumber);

// Set DMA channel [SBPRO compatible]
// parameters: DMA Channel
// return none
extern void sgSetDMA(unsigned DMAChannel);

// Set Sound Card Mode
// parameters: mode - Sound Card Mode
//              SB8MODE   - SB 8-bit Stereo
//              MSSMODE   - Microsoft's Windows Sound System
//              DMA8MODE  - 8-bit DMA mode
//              DMA16MODE - 16-bit DMA mode
extern int sgSetMode(BYTE mode);

/****************************/
/* Voice functions (Memory) */
/****************************/

// Initialise Hardware
// parameters: none
// return 0 for success and non-zero for failure
extern int sgDSPvm_init(void);

// Switch DAC Speaker ON/OFF
// parameters: 0 - OFF
//             1 - ON
// return none
extern void sgDSPvm_speaker(int flag);

// Start Voice Output
// parameters: Pointer to Voice Data Buffer
// return 0 for success and non-zero for failure
extern int sgDSPvm_output(char far *lpBuf);

// Start Voice Input
// parameters: Pointer to Voice Data Buffer, buffer size and sampling rate
// return 0 for success and non-zero for failure
extern int sgDSPvm_input(char far *lpBuf,
		  unsigned long BufSize, unsigned sampling_rate);

// Stop Voice I/O
// parameters: none
// return none
extern void sgDSPvm_stop(void);

// Terminate Library Function
// parameters: none
// return none
extern void sgDSPvm_terminate(void);

// Pause Voice Output
// parameters: none
// return none
extern void sgDSPvm_pause(void);

// Continue Voice Output
// parameters: none
// return none
extern void sgDSPvm_continue(void);

// Start Voice Output from Extended Memory
// parameters: XMS handle and offset
// return 0 for success and non-zero for failure
extern int sgDSPvm_outputxms(unsigned xms_handle, unsigned long xms_offset);

// Start Voice Input to Extended Memory
// parameters: XMS handle and offset, sampling rate and buffer size
// return 0 for success and non-zero for failure
extern int sgDSPvm_inputxms(unsigned xms_handle, unsigned sampling_rate,
		     unsigned long xms_offset, unsigned buffer_size);

/**************************/
/* Voice functions (Disk) */
/**************************/

// Initialise Hardware
// parameters: none
// return 0 for success and non-zero for failure
extern int sgDSPvd_init(void);

// Switch DAC Speaker ON/OFF
// parameters: 0 - OFF
//             1 - ON
// return none
extern void sgDSPvd_speaker(int flag);

// Start Voice Output
// parameters: Voice File Handle
// return 0 for success and non-zero for failure
extern int sgDSPvd_output(int Handle);

// Start Voice Input
// parameters: Voice File Handle, sampling rate and channel
// return 0 for success and non-zero for failure
extern int sgDSPvd_input(int Handle, unsigned sampling_rate, int channel);

// Stop Voice I/O
// parameters: none
// return none
extern void sgDSPvd_stop(void);

// Terminate Library Function
// parameters: none
// return none
extern void sgDSPvd_terminate(void);

// Pause Voice Output
// parameters: none
// return none
extern void sgDSPvd_pause(void);

// Continue Voice Output
// parameters: none
// return none
extern void sgDSPvd_continue(void);

/******************/
/* Wave functions */
/******************/

// Initialise Hardware
// parameters: none
// return 0 for success and non-zero for failure
int sgDSPwd_init(void);

// Switch DAC Speaker ON/OFF
// parameters: 0 - OFF
//             1 - ON
// return none
void sgDSPwd_speaker(int flag);

// Start Wave Output
// parameters: Wave File Handle
// return 0 for success and non-zero for failure
int sgDSPwd_output(int Handle);

// Start Wave Input
// parameters: Wave File Handle, sampling rate and channel
// return 0 for success and non-zero for failure
int sgDSPwd_input(int Handle, unsigned sampling_rate, int channel);

// Stop Wave I/O
// parameters: none
// return none
void sgDSPwd_stop(void);

// Terminate Library Function
// parameters: none
// return none
void sgDSPwd_terminate(void);

// Pause Wave Output
// parameters: none
// return none
void sgDSPwd_pause(void);

// Continue Wave Output
// parameters: none
// return none
void sgDSPwd_continue(void);

/*******************/
/* Mixer functions */
/*******************/

// Set Mixer On/Off Mode [SBPRO compatible]
// parameters: mixer mode 0 - OFF
//                        1 - ON
// return previous mixer mode
extern int sgDSPAux_Mix(int mode);

// Set 8- or 16-step Mode [SBPRO compatible]
// parameters: step mode 0 - 8
//                       1 - 16
// return previous step mode
extern int sgDSPAux_Step(int mode);

// Set AGC Mode [SBPRO compatible]
// parameters: step mode 0 - OFF
//                       1 - ON
// return previous AGC mode
extern int sgDSPAux_AGC(int mode);

// Set Mute Mode [SBPRO compatible]
// parameters: mute mode 0 - OFF
//                       1 - ON
// return previous mute mode
extern int sgDSPAux_UnMute(int mode);

// Set Volume for specified sources [SBPRO compatible]
// [NXPRO, NXPRO16]:
// parameters: volume source 0 - master  (0 - 15)
//                           1 - voice   (0 - 15)
//                           2 - FM      (0 - 15)
//                           3 - CD      (0 - 15)
//                           4 - line-in (0 - 15)
//                           5 - mic (use right level for value input) (0 - 7);
//             volume level for both left and right channels except
//             microphone mixing
// [PRO16]:
// parameters: volume source 0 - master           (0 - 63)
//                           1 - voice            (0 - 31)
//                           2 - FM               (0 - 31)
//                           3 - CD               (0 - 31)
//                           4 - line-in          (0 - 31)
//                           5 - mic              (0 - 31)
//                           6 - Wave Synthesizer (0 - 31)
//                           7 - PC Speaker       (0 - 31) (single channel only)
//             volume level for both left and right channels
//
// return previous volume level
extern int sgDSPAux_SetVolume(int source, BYTE leftlvl, BYTE rightlvl);

// Set Volume for Tone Control [SBPRO compatible]
// [NXPRO, NXPRO16]:
// parameters: tone control 0 - bass    (0 - 15)
//                          1 - treble; (0 - 15)
//             volume level (use rightlvl for value input)
// [PRO16]:
// parameters: tone control 0 - bass    (0 - 15)
//                          1 - treble; (0 - 15)
//             volume level (leftlvl and rightlvl for value inputs)
// return previous volume level
extern int sgDSPAux_ToneControl(int tone, BYTE leftlvl, BYTE rightlvl);

// Set Recording Mode [SBPRO compatible]
// parameters: recording mode 0 - mono
//                            1 - stereo
// return previous recording mode
extern int sgDSPAux_set_stereo(int mode);

// Set Recording Source [SBPRO compatible]
// parameters: recording score 0 - microphone
//                             1 - CD
//                             2 - microphone
//                             3 - line-in
// return previous source used
extern int sgDSPAux_set_input_source(int source);

// Set Recording Filter [SBPRO compatible]
// parameters: recording filter 0 - low filter
//                              1 - high filter
// return previous filter status
extern int sgDSPAux_set_input_freq(int frequency);

// Turn Recording Filter ON/OFF [SBPRO compatible]
// parameters: 0 - ON
//             1 - OFF
// return previous recording filter status
extern int sgDSPAux_ANFI_control(int option);
  
// Turn Output Filter ON/OFF [SBPRO compatible]
// parameters: 0 - ON
//             1 - OFF
// return previous output filter status
extern int sgDSPAux_DNFI_control(int option);

// Get Maximum and Minimum Recording Sampling Rates
// parameters: mode 0 - mono
//                  1 - stereo [SBPRO compatible]
// return maximum and minimum recording sampling rates
extern unsigned long sgDSPAux_get_ADC_range(int mode);

// Get Maximum and Minimum Output Sampling Rates
// parameters: mode 0 - mono
//                  1 - stereo [SBPRO compatible]
// return maximum and minimum output sampling rates
extern unsigned long sgDSPAux_get_DAC_range(int mode);

// Read Recording Filter Status [SBPRO compatible]
// parameters: none
// return current recording filter status
extern int sgDSPAux_get_ANFI(void);

// Read Output Filter Status [SBPRO compatible]
// parameters: none
// return current output filter status
extern int sgDSPAux_get_DNFI(void);

// Set Gain for specified sources [SBPRO compatible]
// parameters: gain source 0 - recording
//                         1 - playback
//             volume level for both left and right channels
//
// return previous gain level
extern int sgDSPAux_SetGain(int source, BYTE leftlvl, BYTE rightlvl);

/**************************/
/* AD1848 Codec functions */
/**************************/

// Get Library Version
// parameters: none
// return high byte:major version and low byte:minor version
extern unsigned sgGetCodecLibVersion(void);

// Set Wave Port Base I/O Address
// parameters: Wave Port I/O Address (0x530, 0x604, 0xE80 or 0xF40)
// return 0 for success and -1 for failure
extern int sgSetCodecBaseAddress(unsigned WavePortAddress);

// Verify Codec Hardware
// parameters: Codec Port - 0x534, 0x608, 0xE84, 0xF44
//
// return 0 for success and -1 for failure
extern int sgVerifyCodecHardware(int CodecPort);

// Set Digital Mix Control Register
// parameters: digital mix enable (DME)
//              0 - OFF
//              1 - ON
//             digital mix attenuation (DMA)
//              0 - 63 (0 dB to -94.5 dB)
// return previous value: DME (lower byte) and DMA (high byte) if success
//        -1 if fail
extern int sgCodecAux_Mix(BYTE mix_enable, BYTE mix_attenuation);

// Set DMA Base Count Register
// (also used for CS4231 playback DMA count register under mode 2 operation)
// parameters: base count value: 0 - 65535
// return none
extern void sgCodecAux_DMABlock(unsigned int basecount);

// Get AD1848/CS4231 Codec Chip Revision ID
// parameters: none
// return AD1848 revision ID: J - 09h, K - 0Ah if success
//        CS4231 revision ID: 8x - Mode 1, 0Cx - Mode 2 if success;
//        -1 if fail
extern int sgCodecAux_get_RevisionID(void);

// Set Interrupt Pin Enable
// parameters: interrupt enable
//              0 - OFF
//              1 - ON
// return 0 if success and -1 if fail
extern int sgCodecAux_IRQEnable(int interrupt_enable);

// Set Input Source
// parameters: input score 0 - line
//                         1 - auxiliary #1
//                         2 - microphone
//                         3 - line post-mixed DAC output
// return previous source used if success
//        -1 if fail
extern int sgCodecAux_set_input_source(int source);

// Set Input Source Level
// parameters: input gain select (IG)
//               0 - 15 (0 dB to 22.5 dB)
//             input microphone gain enable (MGE)
//               0 - OFF
//               1 - ON (+20 dB)
// return previous value:
//          upper word (left channel) and lower word (right channel)
//          IG (low byte) and MGE (high byte) if success
//        -1 if fail
extern long sgCodecAux_set_input_level(BYTE leftgain, BYTE leftmic_enable,
				       BYTE rightgain, BYTE rightmic_enable);

// Get Input Source Level
// parameters: none
// return current value:
//          upper word (left channel) and lower word (right channel)
//          gain (low byte) and microphone gain enable (high byte) if success
//        -1 if fail
extern long sgCodecAux_get_input_level(void);

// Set Aux #1 Input Level
// parameters: auxiliary input #1 attenuate select (X1A)
//               J-grade: 0 - 15 (0 dB to 22.5 dB)
//               K-grade: 0 - 31 (12 dB to -34.5 dB)
//             auxiliary #1 mute (MX1)
//               0 - OFF
//               1 - ON
// return previous value:
//          upper word (left channel) and lower word (right channel)
//          X1A (low byte) and MX1 (high byte) if success
//        -1 if fail
extern long sgCodecAux_set_aux1_level(BYTE leftatten, BYTE leftmute_enable,
				      BYTE rightatten, BYTE rightmute_enable);

// Get Aux #1 Input Level
// parameters: none
// return current value:
//          upper word (left channel) and lower word (right channel)
//          attenuation (low byte) and mute (high byte) if success
//        -1 if fail
extern long sgCodecAux_get_aux1_level(void);

// Set Aux #2 Input Level
// parameters: auxiliary input #2 attenuate select (X2A)
//               J-grade: 0 - 15 (0 dB to 22.5 dB)
//               K-grade: 0 - 31 (12 dB to -34.5 dB)
//             auxiliary #2 mute (MX2)
//               0 - OFF
//               1 - ON
// return previous value:
//          upper word (left channel) and lower word (right channel)
//          X2A (low byte) and MX2 (high byte) if success
//        -1 if fail
extern long sgCodecAux_set_aux2_level(BYTE leftatten, BYTE leftmute_enable,
				      BYTE rightatten, BYTE rightmute_enable);

// Get Aux #2 Input Level
// parameters: none
// return current value:
//          upper word (left channel) and lower word (right channel)
//          attenuation (low byte) and mute (high byte) if success
//        -1 if fail
extern long sgCodecAux_get_aux2_level(void);

// Set DAC Output Level
// parameters: DAC output attenuate select (DA)
//               0 - 63 (0 dB to -94.5 dB)
//             DAC mute (DM)
//               0 - OFF
//               1 - ON
// return previous value:
//          upper word (left channel) and lower word (right channel)
//          DA (low byte) and DM (high byte) if success
//        -1 if fail
extern long sgCodecAux_set_DAC_level(BYTE leftatten, BYTE leftmute_enable,
				     BYTE rightatten, BYTE rightmute_enable);

// Get DAC Output Level
// parameters: none
// return current value:
//          upper word (left channel) and lower word (right channel)
//          attenuation (low byte) and mute (high byte) if success
//        -1 if fail
extern long sgCodecAux_get_DAC_level(void);

// Set Data Format
// (also used for CS4231 Playback Data Format Register under mode 2 operation)
// parameters: format tag (wTag)
//               1 - Linear PCM
//               6 - A-law Companded
//               7 - Mu-law Companded
//             sampling rate (in Hz) (wRate)
//               5.5125 kHz to 48 kHz
//             channel (bChannel)
//               0 - mono
//               1 - stereo
//             bits per sample (bBit)
//               0 - 8-bit
//               1 - 16-bit
// return previous data format if success
//        -1 if fail
extern int sgCodecAux_set_data_format(WORD wTag, WORD wRate,
				      BYTE bChannel, BYTE bBit);

// Set Interface Configuration
// parameters: play_enable (PEN)
//              0 - OFF
//              1 - ON
//             play_pio_enable (PPIO)
//              0 - OFF (DMA)
//              1 - ON  (PIO)
//             capture_enable (CEN)
//              0 - OFF
//              1 - ON
//             capture_pio_enable (CPIO)
//              0 - OFF (DMA)
//              1 - ON  (PIO)
//             dma_mode
//              0 - dual DMA channel mode
//              1 - single DMA channel mode
//             autocal
//              0 - no autocalibration
//              1 - autocalibration after power down/reset or mode change
// return previous interface configuration if success
//        -1 if fail
extern int sgCodecAux_set_interface_cfg(BYTE play_enable, BYTE play_pio_enable,
				  BYTE capture_enable, BYTE capture_pio_enable,
				  BYTE dma_mode, BYTE autocal);

//
// The following functions are meant for mode 2 operation of CS4231)
//

// Set CS4231 Codec Chip Enhanced Function
// parameters: bit 6: 0 - Mode 1
//                                               1 - Mode 2
// return: 0 if successful; -1 if fail
extern int sgCodecAuxCS_set_mode(int mode);

// Set DMA Base Count Register
// parameters:  mode: 0 - playback
//                    1 - capture
//              base count value: 0 - 65535
// return: 0 if successful; -1 if fail
extern int sgCodecAuxCS_DMABlock(int mode, unsigned int basecount);

// Set Timer Register
// parameters:  timer count value: 0 - 65535
// return: none
extern void sgCodecAuxCS_set_timer(unsigned int timercount);

// Set Line Input Level
// parameters: line attenuate select (LIA)
//               CS4231: 0 - 31 (12 dB to -34.5 dB)
//             line mute (MX)
//               0 - OFF
//               1 - ON
// return previous value:
//          upper word (left channel) and lower word (right channel)
//          LIA (low byte) and MX (high byte) if success
//        -1 if fail
extern long sgCodecAuxCS_set_line_level(BYTE leftatten, BYTE leftmute_enable,
				 BYTE rightatten, BYTE rightmute_enable);

// Get Line Input Level
// parameters: none
// return current value:
//          upper word (left channel) and lower word (right channel)
//          attenuation (low byte) and mute (high byte)
//        -1 if fail
extern long sgCodecAuxCS_get_line_level(void);

// Set Mono I/O Level
// parameters: mono input attenuate select (MIA)
//               CS4231: 0 - 15 (0 dB to -45.0 dB)
//             mono output mute (MOM): 0 - OFF, 1 - ON
//             mono input mute (MIM) : 0 - OFF, 1 - ON
// return previous value:
//             attenuate: MIA (bits 0 - 3)
//             mute     : MOM (bit 6)
//                        MIM (bit 7) if success
//        -1 if fail
extern int sgCodecAuxCS_set_mono_level(BYTE monoatten, BYTE monoinputmute_enable,
				BYTE monooutputmute_enable);

// Get Mono I/O Level
// parameters: none
// return current value:
//             attenuate: MIA (bits 0 - 3)
//             mute     : MOM (bit 6)
//                        MIM (bit 7) if success
extern int sgCodecAuxCS_get_mono_level(void);

// Set Data Format
// parameters: mode
//               0 - playback
//               1 - capture
//             format tag (wTag)
//               1 - Linear PCM
//               6 - A-law Companded
//               7 - Mu-law Companded
//              17 - ADPCM 4-bit IMA compatible
//             sampling rate (in Hz) (wRate)
//               5.5125 kHz to 48 kHz
//             channel (bChannel)
//               0 - mono
//               1 - stereo
//             bits per sample (bBit)
//               0 - 8-bit
//               1 - 16-bit
// return previous data format if success
//        -1 if fail
extern int sgCodecAuxCS_set_data_format(int mode, WORD wTag, WORD wRate,
				 BYTE bChannel, BYTE bBit);

// Set Alternate Feature Enable I Registers
// parameters: timer enable (TE): 0 - inactive, 1 - active
//             output level bit (OLB): 0 - full scale 2 Vpp
//                                     1 - full scale 2.8 Vpp
// return previous value:
//             Type I : TE  (low byte), OLB (high byte) if successful
//        -1 if fail
extern int sgCodecAuxCS_set_alt_feature1(BYTE timer_enable, BYTE output_level);

// Set Alternate Feature Enable II Registers
// parameters: high pass filter (HPF): 0 - disabled, 1 - enabled
//             DAC zero (DACZ): 0 - hold previous value during underrun error
//                              1 - go to zero on underrun error
// return previous value:
//             Type II: HPF (low byte), DACZ (high byte) if successful
//        -1 if fail
extern int sgCodecAuxCS_set_alt_feature2(BYTE hpf_enable, BYTE dac_zero);

// Get Alternate Feature Status Register
// parameters: None
// return current value (low byte)
extern int sgCodecAuxCS_get_feature_status(void);

// Reset Alternate Feature Status Register
// parameters: None
// return current value (low byte)
extern int sgCodecAuxCS_rst_feature_status(void);

// Play wave file through Codec
extern int sgCodecwd_output(WORD wMSSPort, WORD wDSPPort, BYTE bDMA,
			    FILE * fInput);

// Capture wave file through Codec
extern int sgCodecwd_input(WORD wMSSPort, WORD wDSPPort, BYTE bDMA,
		     FILE * fOutput, WORD wRate, WORD wChannel, WORD wBit);
extern SwitchModeToWSS(void);
extern SwitchModeToSB(void);

/* EOF: SGSDK.H */

