//***************************************************************************
//
// Copyright (c) 1991-93 Sierra Semiconductor Corp.
//
// FILE:    ariamidi.c
//
// LANGUAGES:
//          Microsoft C Versions 5.0, 6.0, and 7.0
//          Borland Turbo C Versions 1.5 and 2.0
//
// DESCRIPTION:
//
//          Aria MIDI Library
//
//          For Microsoft C 5.0, 6.0, and 7.0
//          =================================
//          compile:  cl /Zlp /Od /c ariamidi.c   (for small model)
//
//                    for medium  model include /AM option
//                    for compact model include /AC option
//                    for large   model include /AL option
//
//          For Borland Turbo C
//          ===================
//          compile:  bcc -DTURBOC -a- -O- -c ariamidi.c   (for small model)
//
//                    for medium  model include -mm option
//                    for compact model include -mc option
//                    for large   model include -ml option
//
// AdLib is a trademark of AdLib Inc.
// Aria, Aria Synthesizer and Aria Listener are trademarks of Sierra
//  Semiconductor Corp.
// QSound is a trademark of Archer Communications
// Sound Blaster is a trademark of Creative Labs Inc.
//
//***************************************************************************

// $Header:   F:\projects\ariai\dos\archives\ariamidi.c_v   2.2   03 Sep 1993 09:58:12   golds  $
// $Log:   F:\projects\ariai\dos\archives\ariamidi.c_v  $
// 
//    Rev 2.2   03 Sep 1993 09:58:12   golds
// Fixed recording from external input source
// 
//    Rev 2.1   13 Aug 1993 09:13:52   golds
// Reverb support, 16-channel MIDI, other optimizations
// 
//    Rev 2.0   24 Jun 1993 12:50:56   golds
// Initial revision.

#include <stdio.h>
#include <stdlib.h>

#include "aria.h"
#include "ariamidi.h"
#include "pitches.h"

// Macros to return low or high word of long value

#define LOWORD(l) ((WORD)(l))
#define HIWORD(l) ((WORD)(((DWORD)(l) >> 16) & 0xFFFF))

// Macros to return low or high byte of word value

#define LOBYTE(w) ((BYTE)(w))
#define HIBYTE(w) ((BYTE)(((WORD)(w) >> 8) & 0xFF))

#define ACTIVE       1        // Operator active flag
#define SUSTAINED    2        // Operator sustaining flag

//*** Local variables ***

// Patch bank definitions

static BANKDEF   HUGE *bankHdr = NULL;    // Patch bank header pointers
static BANKDEF2  HUGE *bankHdr2 = NULL;
static BYTE      HUGE *firstPatch = NULL; // First patch pointer
static OPDEF     HUGE *firstOp = NULL;    // First operator pointer
static PATCHDEF  HUGE *patchptr1;         // Patch pointers
static PATCHDEF2 HUGE *patchptr2;
static OPDEF     HUGE *operptr1;          // Operator pointers
static OPDEF2    HUGE *operptr2;
static OP_INFO   op[DSPMAXOPS];

static CH_INFO   ch[MAXCHANNEL];          // Channel structures
static CHANNELPTR cpt;                    // Channel pointer
static PATCH     HUGE *pptr;              // Patch pointer
static OP        HUGE *optr;              // Operator pointer
static BYTE      HUGE *percOp[128];       // Percusion operator pointer

static WPTINFO   WPT[MAXOPSPERPATCH*MAXCHANNEL]; // Active WPT table
static WPTCHAN   WPTchan[MAXCHANNEL]=            // WPT MIDI allocation table
         {{0, 5},{5, 5},{10,5},{15,5},
          {20,5},{25,5},{30,5},{35,5},
          {40,5},{45,MAXPERCUSSIONON},{53,5},{58,5},
          {63,5},{68,4},{72,4},{76,4}};

static BYTE bankVersion = 0;              // Patch bank format version
static WORD patchCount = 0;               // Number of patches in bank
static BYTE RPNflag[MAXCHANNEL];          // Registered Parameter Number flag
static WORD RPNword[MAXCHANNEL];          // RPN data word
static BOOL pbsens[MAXCHANNEL];           // Pitch Bend Sensitivity flag

static DWORD dwDATE = 0L;                 // Note on/off counter for dynamic
                                          // operator allocation
static PERCPATCH PercPatch[MAXPERCUSSIONON]; // Percussion patch information
static short gPercussionOn = 0;           // Number of perc operators on
static LFO_INFO lfo[MAXLFOS];             // LFO information structures
static WORD basenote;                     // Temporary variable
static WORD midipitch;                    // Temporary variable
static WORD wavetype;                     // Temporary variable
static LONG dsppitch;                     // Temporary variable

// Stereo pan settings for DSP operators

static WORD pantab[64] = {
   0x0000, 0x0008, 0x0020, 0x0048, 0x0080, 0x00C8, 0x0120, 0x0188,
   0x0200, 0x0288, 0x0320, 0x03C8, 0x0480, 0x0548, 0x0620, 0x0708,
   0x0800, 0x0908, 0x0A20, 0x0B48, 0x0C80, 0x0DC8, 0x0F20, 0x1088,
   0x1200, 0x1388, 0x1520, 0x16C8, 0x1880, 0x1A48, 0x1C20, 0x1E08,
   0x2000, 0x2208, 0x2420, 0x2648, 0x2880, 0x2AC8, 0x2D20, 0x2F88,
   0x3200, 0x3488, 0x3720, 0x39C8, 0x3C80, 0x3F48, 0x4220, 0x4508,
   0x4800, 0x4B08, 0x4E20, 0x5148, 0x5480, 0x57C8, 0x5B20, 0x5E88,
   0x6200, 0x6588, 0x6920, 0x6CC8, 0x7080, 0x7448, 0x7820, 0x7FFF };

//*** Local functions ***

static VOID  ChangeWPTpan (WORD, WORD);
static VOID  CheckRPN (WORD);
static WORD  WPTnumber (WORD, WORD);
static VOID  NoteOff (WORD, WORD);
static VOID  ProcessReverb (WORD, WORD);
static LONG  GetPitch (LONG);
static LONG  PitchBend (short, short);
static WORD  GetPercOperator (VOID);
static WORD  GetMeloOperator (VOID);
static WORD  GetLFO (VOID);
static VOID  CancelLFO (WORD);

static short InitPatchBank1 (VOID FAR *, UINT);
static short InitPatchBank2 (VOID FAR *, UINT);
static VOID  SynthInit (VOID);
static VOID  DoNoteOn (BYTE, BYTE, BYTE);
static VOID  DoPercOn (WORD, WORD);
static WORD  LoadPercussion (WORD);
static WORD  ValidWave (BYTE, WORD, BYTE);
static VOID  DoNoteOff (BYTE, BYTE);
static VOID  DoController (BYTE, BYTE, BYTE);
static VOID  DoProgramChange (BYTE, BYTE);
static VOID  DoPitchBend (BYTE, BYTE, BYTE);
static VOID  ProcessController (BYTE, BYTE, WORD, WORD);
static WORD  GetAmp (BYTE, short, BYTE, BYTE);

static PATCHDEF HUGE *findPatch1 (BYTE);
static PATCHDEF2 HUGE *findPatch2 (BYTE);
static OPDEF HUGE *findOp1 (PATCHDEF HUGE *, short);
static OPDEF2 HUGE *findOp2 (PATCHDEF2 HUGE *, short);
static VOID copyPatch1 (HPBYTE, HPBYTE);
static VOID copyPatch2 (HPBYTE, HPBYTE);
static VOID copyOp1 (HPBYTE, HPBYTE);
static VOID copyOp2 (HPBYTE, HPBYTE);

//*** External variable declarations ***

extern WORD  wMidiSynthAllocated;
extern WORD  wMaxSynthOp;
extern WORD  wMaxPercOp;
extern WORD  playMode;
extern WORD  reverbMode;

//*** Public function declarations ***

VOID ChangeOpCount (WORD, WORD);
VOID SetReverbControls (VOID);
VOID SetPanControls (VOID);


//***************************************************************************
//
//  FUNCTION:      short InitPatchBank (VOID FAR *bankptr, UINT mode)
//
//  DESCRIPTION:   Function to initialize an Aria patch bank.  Must be
//                 done prior to MIDImessage calls.
//
//  PARAMETERS:    bankptr = pointer to patch bank loaded in PC RAM.
//                           patch bank cannot be moved while synthesis
//                           in progress
//                 mode    = 0 - No synthesis
//                           1 - 20 operators
//                           2 - 32 operators
//                                 or
//                           High order byte can contain number of
//                           operators to allow (1 to 32)
//
//  RETURNS:       0 = playback mode set
//                 1 = illegal patch bank
//                 2 = mode out of range
//                -1 = DSP not responding
//                -2 = mode not available
//
//***************************************************************************

short InitPatchBank (VOID FAR *bankptr, UINT mode)
   {
   if ((bankHdr = (BANKDEF HUGE *) bankptr) == NULL)
      return 1;

   switch (bankHdr->version)
      {
      case 1:
         bankVersion = 1;
         return InitPatchBank1 (bankptr, mode);

      case 2:
         bankVersion = 2;
         return InitPatchBank2 (bankptr, mode);

      default:
         // Unsupported patch bank format
         patchCount = 0;
         firstPatch = NULL;
      }
   return 1;
   }


//***************************************************************************
//
//  FUNCTION:      short MIDImessage (BYTE cmd, BYTE dat1, BYTE dat2)
//
//  DESCRIPTION:   Function to process a MIDI message.  Acceptable events
//                 include note on/off, controllers, program changes and
//                 pitch bends.  InitPatchBank must be called before any
//                 calls to MIDImessage.  Channels 1 to 9, 11 to 16 (melodic)
//                 and channel 10 (percussion) will be processed.
//
//  PARAMETERS:    cmd  = MIDI command and channel
//                 dat1 = first data byte
//                 dat2 = second data byte (if not required, it will be
//                                          ignored)
//
//  RETURNS:       0 = message processed
//                -1 = channel out of range
//                -2 = illegal MIDI command
//                -3 = InitPatchBank was not called
//
//***************************************************************************

short MIDImessage (BYTE cmd, BYTE dat1, BYTE dat2)
   {
   BYTE chan, patchID;
   static BYTE lastcmd = 0;

   if (bankVersion == 0 || bankVersion > 2)
      return -3;

   // Check for running status

   if (cmd < NOTEOFF)
      {
      dat2 = dat1;
      dat1 = cmd;
      cmd  = lastcmd;
      }
   else
      lastcmd = cmd;

   // Validate patch bank

   if (firstPatch == NULL)
      return -3;

   // Check for patch on selected channel

   chan = cmd & CHMASK;
   cpt = (CHANNELPTR) &ch[chan];
   if (cpt->patch.totalOperators == 0)
      if ((cmd & COMMASK) != PCHANGE)
         {
         if (bankVersion == 1)
            patchID = ((PATCHDEF HUGE *) firstPatch)->patchID;
         else
            patchID = ((PATCHDEF2 HUGE *) firstPatch)->patchID;
         DoProgramChange (chan, patchID);
         }

   // Process command

   switch (cmd & COMMASK)
      {
      case NOTEON:
         if (dat2)
            {
            if (chan == PERCUSSIONCH)
               DoPercOn (dat1, dat2);
            else
               DoNoteOn (chan, dat1, dat2);
            break;
            }

      case NOTEOFF:
         DoNoteOff (chan, dat1);
         break;

      case CONTROLLER:
         DoController (chan, dat1, dat2);
         break;

      case PCHANGE:
         DoProgramChange (chan, dat1);
         break;

      case PBEND:
         DoPitchBend (chan, dat1, dat2);
         break;

      case SYSEX:
         if (cmd <= EOX)
            lastcmd = 0;
         break;

      default:
         return -2;
      }
   return 0;
   }


//**************************************************************************
// Initialize patch bank format 1
//**************************************************************************

static short InitPatchBank1 (VOID FAR *bankptr, UINT mode)
   {
   register short i, j;
   WORD k, m;
   BOOL first=TRUE;

   patchCount = bankHdr->patchCount;         // Get number of patches
   firstPatch = (BYTE HUGE *) (bankHdr + 1); // Set first patch pointer

   if ((i = SetPlaybackMode (mode)) != 0)    // Set playback mode
      return i;

   // Set first operator pointer

   firstOp = (OPDEF HUGE *) (((PATCHDEF HUGE *) firstPatch) + patchCount);

   if (patchCount && (bankHdr != NULL))
      {
      SynthInit ();  // Initialize internal synthesis variables

      // Initialize pointers to all percussion operators

      for (i = 127; i >= 0; --i)
         {
         percOp[i] = NULL;
         patchptr1 = (PATCHDEF HUGE *) firstPatch;

         // Search for each percussion patch

         for (j = 0; j < (short) patchCount; ++j, ++patchptr1)
            if (patchptr1->patchID == (BYTE) (i + 128))
               {
               if (first)
                  {
                  // Initialize percussion channel buffer

                  cpt = (CHANNELPTR) &ch[PERCUSSIONCH];
                  copyPatch1 ((HPBYTE) &cpt->patch, (HPBYTE) patchptr1);
                  cpt->patch.level      = 127;
                  cpt->patch.pitchWheel = 2;
                  first = FALSE;
                  }

               // Search for associated percussion operator

               operptr1 = firstOp;
               k = patchCount * MAXOPSPERPATCH;
               for (m = 0; m < k; ++m, ++operptr1)
                  if (patchptr1->operatorID[0] == operptr1->operatorID)
                     {
                     // Only one operator, so relative operator amplitude 
                     // must be patch level

                     operptr1->opAmp = patchptr1->level; 
                     percOp[i] = (BYTE HUGE *) operptr1;
                     }
               }
         }
      }
   wMidiSynthAllocated = (WORD) mode;

   return 0;
   }


//**************************************************************************
// Initialize patch bank format 2
//**************************************************************************

static short InitPatchBank2 (VOID FAR *bankptr, UINT mode)
   {
   register short i, j;
   BOOL first=TRUE;

   bankHdr2 = (BANKDEF2 HUGE *) bankptr;        // Set bank header pointer
   patchCount = bankHdr2->patchCount;           // Get number of patches
   firstPatch = (BYTE HUGE *) (bankHdr2 + 1);   // Set first patch pointer

   if ((i = SetPlaybackMode (mode)) != 0)       // Set playback mode
      return i;

   if (patchCount && (bankHdr2 != NULL))
      {
      SynthInit ();  // Initialize internal synthesis variables

      // Initialize pointers to all percussion operators

      for (i = 127; i >= 0; --i)
         {
         percOp[i] = NULL;
         patchptr2 = (PATCHDEF2 HUGE *) firstPatch;

         for (j = 0; j < (short) patchCount; ++j, ++patchptr2)
            if (patchptr2->patchID == (BYTE) (i + 128))
               {
               if (first)
                  {
                  // Initialize percussion channel buffer

                  cpt = (CHANNELPTR) &ch[PERCUSSIONCH];
                  copyPatch2 ((HPBYTE) &cpt->patch, (HPBYTE) patchptr2);
                  cpt->patch.level      = 127;
                  cpt->patch.pitchWheel = 2;
                  first = FALSE;
                  }

               // Locate associated percussion operator

               operptr2 = (OPDEF2 HUGE *) (((BYTE HUGE *) bankHdr2) + 
                                            patchptr2->opIndex[0]);

               // Only one operator, so relative operator amplitude 
               // must be patch level

               operptr2->opAmp = patchptr2->level;
               percOp[i] = (BYTE HUGE *) operptr2;
               }
         }
      }
   wMidiSynthAllocated = (WORD) mode;

   return 0;
   }


//**************************************************************************
//  Initialize internal synthesis variables
//**************************************************************************

static VOID SynthInit (VOID)
   {
   register short i, j;

   gPercussionOn = 0;
   for (i = 0; i < MAXPERCUSSIONON; ++i)
      {
      PercPatch[i].midipitch = UNUSEDW;
      PercPatch[i].date      = 0L;
      }
   for (i = 0; i < MAXOPSPERPATCH*MAXCHANNEL; ++i)
      {
      WPT[i].wptno   = UNUSEDB;
      WPT[i].opmask  = 0L;
      WPT[i].date    = 0L;
      }
   for (i = 0; i < MAXCHANNEL; ++i)
      {
      ch[i].patch.totalOperators = 0;
      for (j = 0; j < 3; ++j)
         {
         ch[i].wptLFO[j].LFOnum   = UNUSEDW;
         ch[i].wptLFO[j].LFOdepth = 0;
         }
      ch[i].vol    = VOLUME_DEFAULT;
      ch[i].pan    = CENTER_PAN;
      ch[i].reverb = REVERB_DEFAULT;
      ch[i].sus    = FALSE;
      ch[i].pbend  = CENTER_PITCH;
      RPNflag[i]   = 0;
      RPNword[i]   = 0;
      pbsens[i]    = FALSE;
      }
   for (i = 0; i < DSPMAXOPS; ++i)
      {
      op[i].channel    = UNUSEDB;
      op[i].note       = UNUSEDB;
      op[i].transposed = 0;
      op[i].velocity   = 0;
      op[i].opnum      = UNUSEDB;
      op[i].type       = 0;
      op[i].flags      = 0;
      op[i].priority   = 0;
      op[i].date       = 0L;
      }
   for (i = 0; i < MAXLFOS; ++i)
      lfo[i].channel = UNUSEDB;
   }


//***************************************************************************
//  Turn on operators associated with the MIDI note on message
//***************************************************************************

static VOID DoNoteOn (BYTE chan, BYTE midinote, BYTE velocity)
   {
   register WORD i, j=0;
   WORD  slot, wave, wpt;
   short ipitch, base, diff;
   LONG  freq;
   BYTE  scale;
   static WORD steps[4] = {1, 2, 4, 8};

   for (i = 0; i < ch[chan].patch.totalOperators; ++i)
      {
      optr = (OP HUGE *) &ch[chan].op[i];

      // Ensure note is in playable pitch and velocity range

      if (optr->minPitch    <= midinote && optr->maxPitch    >= midinote &&
          optr->minVelocity <= velocity && optr->maxVelocity >= velocity)
         {
         // Check to make sure a wave should play at this pitch

         if ((wave = ValidWave (chan, i, midinote)) != UNUSEDW &&
             (wpt = WPTnumber (chan, i)) != UNUSEDW)
            {
            base   = (short) optr->waveTable[wave].basePitch;
            ipitch = (short) midinote + optr->transpose;

            // Check for pitch range scaling

            scale = optr->pitchScale;
            if (scale)
               {
               // Adjust MIDI note according to scale factor

               diff = ipitch - base;
               ipitch = base + (diff >> scale);
               }

            // Adjust if note is out of playable range

            while (ipitch > HIMIDI)
               ipitch -= 12;
            while (ipitch < LOMIDI)
               ipitch += 12;

            // Find an available operator

            slot = GetMeloOperator ();

            // Record information about the operator to be turned on

            op[slot].priority   = (BYTE) (j++ < ch[chan].patch.minOperators);
            op[slot].basenote   = (BYTE) base;
            op[slot].note       = (BYTE) midinote;
            op[slot].transposed = (BYTE) ipitch;
            op[slot].velocity   = (BYTE) velocity;
            op[slot].opnum      = (BYTE) i;
            op[slot].channel    = (BYTE) chan;
            op[slot].type       = optr->waveTable[wave].waveType;
            op[slot].date       = dwDATE;
            op[slot].flags      = ACTIVE;
            WPTon (wpt, slot);

            // Store pitch scaling factor, if necessary

            basenote  = (WORD) op[slot].basenote;
            midipitch = (WORD) op[slot].transposed;
            wavetype  = (WORD) op[slot].type;
            dsppitch  = CALCPITCH (basenote, midipitch, wavetype);

            if (scale)
               {
               op[slot].pfactor = PitchBend ((CENTER_PITCH / steps[scale]) *
                                             (diff % steps[scale]), 1);
               freq = GetPitch ((LONG) optr->detune) +
                      PitchBend ((short) ch[chan].pbend,
                                 (short) ch[chan].patch.pitchWheel) +
                      op[slot].pfactor;
               }
            else
               {
               op[slot].pfactor = 0L;
               freq = GetPitch ((LONG) optr->detune) +
                      PitchBend ((short) ch[chan].pbend,
                                 (short) ch[chan].patch.pitchWheel);
               }

            // Send OperatorOn message to DSP

            OperatorOn (slot,
                        wpt,
                        optr->waveTable[wave].waveIndex,
                        GetAmp (chan, i, (BYTE) ipitch,
                                (BYTE) velocity),
                        LOWORD (freq >> 8),
                        (WORD) optr->waveTable[wave].blockCount,
                        optr->waveTable[wave].waveType
                       );
            }
         }
      }
   ++dwDATE;   // all NOTES will get the same date stamp
   }


//***************************************************************************
//  Turn on a percussion operator
//***************************************************************************

static VOID DoPercOn (WORD midinote, WORD velocity)
   {
   register WORD slot, opnum;
   WORD  wpt;
   short ipitch;
   LONG  freq;

   // Does percussion operator exist?

   if (percOp[midinote] == NULL)
      return;

   // Is percussion wave table cached?

   for (opnum = 0; opnum < MAXPERCUSSIONON; opnum++)
      if (PercPatch[opnum].midipitch == midinote)
         break;

   if (opnum >= MAXPERCUSSIONON) // Not cached, go load it (will take time!!)
      if ((opnum = LoadPercussion (midinote)) == UNUSEDW)
         return;                 // failed to load, do nothing

   // Check to make sure operator should play at this velocity

   optr = (OP HUGE *) &cpt->op[opnum];
   if ((optr->minVelocity <= (BYTE) velocity) &&
       (optr->maxVelocity >= (BYTE) velocity) &&
       (optr->waveTable[0].waveIndex != UNUSEDW) &&
        optr->waveTable[0].blockCount)
      {
      // Find an available operator

      if (gPercussionOn >= (short) wMaxPercOp)
         slot = GetPercOperator ();
      else
         {
         gPercussionOn++;
         slot = GetMeloOperator ();
         }

      PercPatch[opnum].date = dwDATE;
      wpt = WPT[calcOFF(PERCUSSIONCH,opnum)].wptno;

      // Ensure pitch is in playable range

      ipitch = optr->waveTable[0].basePitch + optr->transpose;
      while (ipitch > HIMIDI)
         ipitch -= 12;
      while (ipitch < LOMIDI)
         ipitch += 12;

      // Record information about the operator to be turned on

      op[slot].priority   = 2;
      op[slot].note       = (BYTE) midinote;
      op[slot].basenote   = (BYTE) optr->waveTable[0].basePitch;
      op[slot].transposed = (BYTE) ipitch;
      op[slot].velocity   = (BYTE) velocity;
      op[slot].opnum      = (BYTE) opnum;
      op[slot].channel    = PERCUSSIONCH;
      op[slot].type       = optr->waveTable[0].waveType;
      op[slot].pfactor    = 0L;
      op[slot].date       = dwDATE;
      op[slot].flags      = ACTIVE;
      WPTon (wpt, slot);

      // Send OperatorOn message to DSP

      basenote  = (WORD) op[slot].basenote;
      midipitch = (WORD) op[slot].transposed;
      wavetype  = (WORD) op[slot].type;
      dsppitch  = CALCPITCH (basenote, midipitch, wavetype);

      freq = GetPitch ((LONG) optr->detune) +
             PitchBend ((short) cpt->pbend, (short) cpt->patch.pitchWheel);
      OperatorOn (slot,
                  wpt,
                  optr->waveTable[0].waveIndex,
                  GetAmp ((BYTE) PERCUSSIONCH, opnum,
                          (BYTE) ipitch, (BYTE) velocity),
                  LOWORD (freq >> 8),
                  (WORD) optr->waveTable[0].blockCount,
                  optr->waveTable[0].waveType
                 );

      // Check for Pedal Hi-hat (key 44) or Open Hi-hat (key 46)
      // and cancel them if Closed Hi-hat (key 42) has been started

      if (midinote == CLOSEDHIHAT)
         for (slot = 0; slot <= wMaxSynthOp; slot++)
            if (op[slot].note == OPENHIHAT || op[slot].note == PEDALHIHAT)
               {
               OperatorCancel (slot);
               if (op[slot].flags & ACTIVE)
                  WPToff (calcOFF (PERCUSSIONCH, op[slot].opnum), slot);
               op[slot].flags = 0;
               op[slot].date  = dwDATE;
               }
      ++dwDATE;
      }
   }


//***************************************************************************
//  Load a percussion wave parameter table
//***************************************************************************

static WORD LoadPercussion (WORD midinote)
   {
   register WORD i, j;
   WORD k;
   DWORD olddate = MAXDATE;
   WPTLFO lfo[3];

   // Maximum percussion instrument WPTs already cached?

   for (j = 0; j < (WORD) WPTchan[PERCUSSIONCH].size; j++)
      {
      if (PercPatch[j].midipitch == UNUSEDW)
         {
         i = j;
         break;
         }
      if (PercPatch[j].date < olddate)
         {
         i = j;
         olddate = PercPatch[j].date;
         }
      else
         i = j;
      }

   // We mark the patch as loaded, although we at this point we don't know
   // if the patch will be found. This will prevent a repeated error message
   // each time we attempt to play the patch.

   PercPatch[i].midipitch = midinote;

   // Notice, we assume the patch definition is the same for all
   // percussion patches.  Otherwise we would have to keep track of each
   // individual patch in more sophisticated way than just PercPatch.

   optr = (OP HUGE *) &cpt->op[i];
   if (bankVersion == 1)
      copyOp1 ((HPBYTE) optr, (HPBYTE) percOp[midinote]);
   else
      copyOp2 ((HPBYTE) optr, (HPBYTE) percOp[midinote]);

   // Send new wave parameter table

   for (j = 0; j < 3; ++j)
      {
      lfo[j].LFOnum = cpt->wptLFO[j].LFOnum;
      if (lfo[j].LFOnum == UNUSEDW)
         lfo[j].LFOnum = 0;
      lfo[j].LFOdepth = cpt->wptLFO[j].LFOdepth;
      }
   j = calcWPT (PERCUSSIONCH) + i;
   SetWaveParameter (j,
                     (ENVDEF FAR *) optr->opEnvelope,
                     (WPTLFO FAR *) lfo,
                     (UINT) GetOpPan(64,optr->balance));
   k = calcOFF(PERCUSSIONCH,i);
   WPT[k].wptno  = (BYTE) j;
   WPT[k].opmask = 0L;
   WPT[k].date   = dwDATE;

   // Update reverb if necessary

   if (reverbMode)
      SetReverbChannel ((UINT) j, (UINT) cpt->reverb << 8);

   return i;
   }


//***************************************************************************
//  Find a valid operator wave within pitch range.  Return wave index
//  number for waveTable[n] or 0xFFFF if not found.
//***************************************************************************

static WORD ValidWave (BYTE chan, WORD opnum, BYTE midinote)
   {
   register WORD wave;

   optr = (OP HUGE *) &ch[chan].op[opnum];

   // Find correct wave

   for (wave = 0; wave < optr->waveCount; wave++)
      {
      if (optr->waveTable[wave].waveIndex == UNUSEDW ||
          !optr->waveTable[wave].blockCount)
         break;

      if (optr->waveTable[wave].lowPitch  <= midinote &&
          optr->waveTable[wave].highPitch >= midinote)
         return wave;
      }
   return UNUSEDW;
   }


//***************************************************************************
//  Turn off all operators associated with a MIDI note off message
//***************************************************************************

static VOID DoNoteOff (BYTE chan, BYTE midinote)
   {
   register WORD i;

   // If operator is active, matches channel and matches note -- turn it off

   for (i = 0; i < wMaxSynthOp; i++)
      if ((op[i].flags & ACTIVE) && op[i].channel == chan && 
           op[i].note == midinote)
         {
         // If sustain pedal is on, just flag it and continue

         if (ch[chan].sus)
            op[i].flags |= SUSTAINED;
         else
            NoteOff (chan, i);
         }
   ++dwDATE;
   }


//***************************************************************************
//  Process a MIDI control change message
//***************************************************************************

static VOID DoController (BYTE chan, BYTE dat1, BYTE dat2)
   {
   register WORD i, j;
   WORD k;

   // Check for custom controllers defined in the patch

   for (i = 0; i < 3; ++i)
      if (ch[chan].patch.ctrlID[i] && ch[chan].patch.ctrlDepth[i] && 
          (ch[chan].patch.ctrlID[i] == dat1))
         {
         ProcessController (chan, ch[chan].patch.ctrlUse[i],
                            (WORD) ch[chan].patch.ctrlDepth[i],
                            (WORD) dat2);
         return;
         }

   // Check for standard controller message

   switch (dat1)
      {
      case DATAENTRY:
         if (pbsens[chan])
            ch[chan].patch.pitchWheel = (BYTE) ((dat2 > 24) ? 24 : dat2);
         break;

      case VOL:
         ch[chan].vol = dat2;
         for (i = 0; i < wMaxSynthOp; ++i)
            {
            if (op[i].channel == chan && op[i].note != UNUSEDB)
               SetOperatorVolume (i, GetAmp (chan, op[i].opnum,
                                     op[i].transposed,
                                     op[i].velocity));
            }
         break;

      case PAN:
         ch[chan].pan = dat2;
         j = calcOFF(chan,0);

         // For each wave parameter associated with the patch, update pan

         for (i = 0; i < MAXOPSPERPATCH; ++i, ++j)
            if ((k = (WORD) WPT[j].wptno) != UNUSEDB)
               {
               optr = (OP HUGE *) &ch[chan].op[op[i].opnum];
               ChangeWPTpan (k, GetOpPan(dat2,optr->balance));
               }
         break;

      case SUS:
         if (dat2 < 64) // Sustain off
            {
            if (ch[chan].sus) // Is channel sustaining?
               {
               // Turn off operators tagged as SUSTAINED

               for (i = 0; i < wMaxSynthOp; ++i)
                  if (op[i].channel == chan && (op[i].flags & SUSTAINED))
                     {
                     OperatorOff (i);
                     if (op[i].flags & ACTIVE)
                        WPToff (calcOFF (chan, op[i].opnum), i);
                     op[i].flags = 0;
                     op[i].date  = dwDATE;
                     }
               ++dwDATE;
               ch[chan].sus = FALSE;
               }
            }
         else  // Sustain on
            ch[chan].sus = TRUE;
         break;

      case REVERB:
         if (reverbMode)
            ProcessReverb ((WORD) chan, (WORD) dat2);
         break;

      case DATADEC:
         if (pbsens[chan]) // Pitch bend sensitivity active?
            {
            if (ch[chan].patch.pitchWheel > (BYTE) dat2)
                ch[chan].patch.pitchWheel -= (BYTE) dat2;
            else
               ch[chan].patch.pitchWheel = 0;
            }
         break;

      case DATAINC:
         if (pbsens[chan]) // Pitch bend sensitivity active?
            {
            if ((ch[chan].patch.pitchWheel + (BYTE) dat2) < 24)
                 ch[chan].patch.pitchWheel += (BYTE) dat2;
            else
               ch[chan].patch.pitchWheel = 24;  // 2 octaves is maximum
            }
         break;

      case RPNLSB:
         RPNflag[chan] |= (BYTE) 1;
         RPNword[chan] &= 0xFF00;
         RPNword[chan] |= (WORD) dat2;
         CheckRPN (chan);
         break;

      case RPNMSB:
         RPNflag[chan] |= (BYTE) 2;
         RPNword[chan] &= 0x00FF;
         RPNword[chan] |= (((WORD) dat2) << 8);
         CheckRPN (chan);
         break;

      case ALLCTRLOFF:
         // Reset pitch bend and controllers

         DoPitchBend (chan, (BYTE) (CENTER_PITCH & 0x7F),
                            (BYTE) (CENTER_PITCH >> 7));
         for (i = 0; i < 3; ++i)
            if (ch[chan].patch.ctrlDepth[i])
               DoController (chan, ch[chan].patch.ctrlID[i], 0);
         DoController (chan, SUS, 0);
         RPNflag[chan] = 0;
         RPNword[chan] = 0;
         pbsens[chan] = FALSE;
         break;

      case ALLSOFF:
         // Cancel all active operators

         for (i = 0; i < wMaxSynthOp; ++i)
            if (op[i].channel == chan)
               {
               OperatorCancel (i);
               if (op[i].flags & ACTIVE)
                  WPToff (calcOFF (chan, op[i].opnum), i);
               op[i].flags = 0;
               op[i].date  = 0L;
               }
         break;

      case ALLNOFF:
      case 124:
      case 125:
      case 126:
      case 127:
         // Turn off all active operators

         for (i = 0; i < wMaxSynthOp; ++i)
            if (op[i].channel == chan && (op[i].flags & ACTIVE))
               {
               // If sustain pedal is on, flag it and continue
               if (ch[chan].sus)
                  op[i].flags |= SUSTAINED;
               else
                  NoteOff (chan, i);
               }
         break;
      }
   }


//***************************************************************************
//  Process a MIDI program change message
//***************************************************************************

static VOID DoProgramChange (BYTE chan, BYTE dat1)
   {
   register WORD j, m;
   WORD n, maxAmp, opnum, LFOnum;
   WPTLFO lfo[3];

   // Ignore for percussion patches

   if (chan == PERCUSSIONCH)
      return;

   // Turn off all notes on channel

   DoController (chan, ALLSOFF, 1);

   // Locate patch to load

   pptr = (PATCH HUGE *) &cpt->patch;
   if (bankVersion == 1)
      {
      if ((patchptr1 = findPatch1 (dat1)) == NULL)
         return;
      copyPatch1 ((HPBYTE) pptr, (HPBYTE) patchptr1);
      }
   else
      {
      if ((patchptr2 = findPatch2 (dat1)) == NULL)
         return;
      copyPatch2 ((HPBYTE) pptr, (HPBYTE) patchptr2);
      }

   // Clear Wave Parameter Table structures for this patch

   j = calcOFF(chan,0);
   for (n = 0; n < MAXOPSPERPATCH; ++n, ++j)
      {
      WPT[j].wptno  = UNUSEDB;
      WPT[j].opmask = 0L;
      WPT[j].date   = 0L;
      }

   // Turn off any currently active LFOs for the channel

   for (j = 0; j < 3; ++j)
      if (cpt->wptLFO[j].LFOdepth)
         {
         CancelLFO (cpt->wptLFO[j].LFOnum);
         cpt->wptLFO[j].LFOnum = UNUSEDW;
         cpt->wptLFO[j].LFOdepth = 0;
         }

   // Turn on any required LFOs for the new patch

   if (cpt->wptLFO[VIBLFO].LFOdepth = (WORD) pptr->vibmod.depth)
      {
      if ((LFOnum = GetLFO ()) == UNUSEDW)
         {
         cpt->wptLFO[VIBLFO].LFOnum = LFOnum;
         cpt->wptLFO[VIBLFO].LFOdepth = 0;
         }
      else
         {
         cpt->wptLFO[VIBLFO].LFOnum = LFOnum;
         LFOactivate (LFOnum, (WORD) pptr->vibmod.shape,
                     getLFOspeed(pptr->vibmod.speed,0,0));
         }
      }

   if (cpt->wptLFO[TREMLFO].LFOdepth = (WORD) pptr->tremmod.depth)
      {
      if ((LFOnum = GetLFO ()) == UNUSEDW)
         {
         cpt->wptLFO[TREMLFO].LFOnum = LFOnum;
         cpt->wptLFO[TREMLFO].LFOdepth = 0;
         }
      else
         {
         cpt->wptLFO[TREMLFO].LFOnum = LFOnum;
         LFOactivate (LFOnum, (WORD) pptr->tremmod.shape,
                     getLFOspeed(pptr->tremmod.speed,0,0));
         }
      }

   if (cpt->wptLFO[PANLFO].LFOdepth = (WORD) pptr->panmod.depth)
      {
      if ((LFOnum = GetLFO ()) == UNUSEDW)
         {
         cpt->wptLFO[PANLFO].LFOnum = LFOnum;
         cpt->wptLFO[PANLFO].LFOdepth = 0;
         }
      else
         {
         cpt->wptLFO[PANLFO].LFOnum = LFOnum;
         LFOactivate (LFOnum, (WORD) pptr->panmod.shape,
                     getLFOspeed(pptr->panmod.speed,0,0));
         }
      }

   // Locate operators and send Wave Parameter Tables for new patch

   maxAmp = 0;
   optr = (OP HUGE *) &cpt->op[0];
   for (opnum = 0, n = calcOFF (chan,0), m = calcWPT(chan); 
        opnum < pptr->totalOperators; ++opnum, ++n, ++m, ++optr)
      {
      if (bankVersion == 1)
         {
         if ((operptr1 = findOp1 (patchptr1, opnum)) == NULL)
            return;
         copyOp1 ((HPBYTE) optr, (HPBYTE) operptr1);
         }
      else
         {
         if ((operptr2 = findOp2 (patchptr2, opnum)) == NULL)
            return;
         copyOp2 ((HPBYTE) optr, (HPBYTE) operptr2);
         }

      for (j = 0; j < 3; ++j)
         {
         lfo[j].LFOnum = cpt->wptLFO[j].LFOnum;
         if (lfo[j].LFOnum == UNUSEDW)
            lfo[j].LFOnum = 0;
         lfo[j].LFOdepth = cpt->wptLFO[j].LFOdepth;
         }
      if ((WORD) optr->opAmp > maxAmp)
         maxAmp = (WORD) optr->opAmp;

      // Send new Wave Parameter Table to DSP

      if (opnum < (WORD) WPTchan[chan].size)
         {
         SetWaveParameter (m,
                           (ENVDEF FAR *) optr->opEnvelope,
                           (WPTLFO FAR *) lfo,
                           (UINT) GetOpPan(cpt->pan,optr->balance));
         WPT[n].wptno = (BYTE) m;
         WPT[n].date  = dwDATE;
         }
      }

   // Normalize operator amplitudes

   optr = (OP HUGE *) &cpt->op[0];
   for (opnum = 0; opnum < pptr->totalOperators; ++opnum, ++optr)
      optr->opAmp += (127 - maxAmp);

   // Zero any custom controllers

   for (j = 0; j < 3; j++)
      DoController (chan, pptr->ctrlID[j], 0);

   // Update reverb if necessary

   if (reverbMode)
      ProcessReverb ((WORD) chan, (WORD) cpt->reverb);
   }


//**************************************************************************
//  Locate patch bank format 1 patch information
//**************************************************************************

static PATCHDEF HUGE *findPatch1 (BYTE patchID)
   {
   register WORD i;
   PATCHDEF HUGE *pptr;

   pptr = (PATCHDEF HUGE *) firstPatch;
   for (i = 0; i < patchCount; ++i, ++pptr)
      if (pptr->patchID == patchID)
         return pptr;

   return NULL;
   }


//**************************************************************************
//  Locate patch bank format 2 patch information
//**************************************************************************

static PATCHDEF2 HUGE *findPatch2 (BYTE patchID)
   {
   register WORD i;
   PATCHDEF2 HUGE *pptr;

   pptr = (PATCHDEF2 HUGE *) firstPatch;
   for (i = 0; i < patchCount; ++i, ++pptr)
      if (pptr->patchID == patchID)
         return pptr;

   return NULL;
   }


//**************************************************************************
//  Locate patch bank format 1 operator information
//**************************************************************************

static OPDEF HUGE *findOp1 (PATCHDEF HUGE *ptr, short opnum)
   {
   register WORD i, j;
   OPDEF HUGE *optr;

   optr = firstOp;
   j = patchCount * MAXOPSPERPATCH;
   for (i = 0; i < j; ++i, ++optr)
      if (ptr->operatorID[opnum] == optr->operatorID)
         return optr;

   return NULL;
   }


//**************************************************************************
//  Locate patch bank format 2 operator information
//**************************************************************************

static OPDEF2 HUGE *findOp2 (PATCHDEF2 HUGE *ptr, short opnum)
   {
   return (OPDEF2 HUGE *) (((BYTE HUGE *) bankHdr2) + ptr->opIndex[opnum]);
   }


//**************************************************************************
//  Copy format 1 patch information to channel buffer
//**************************************************************************

static VOID copyPatch1 (HPBYTE dest, HPBYTE src)
   {
   register WORD i;

   for (i = 0; i < PATCHNAMELEN+1; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   src += (MAXOPSPERPATCH << 1);
   for (i = 0; i < 26; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   }


//**************************************************************************
//  Copy format 2 patch information to channel buffer
//**************************************************************************

static VOID copyPatch2 (HPBYTE dest, HPBYTE src)
   {
   register WORD i;
   PATCH HUGE *tmpdest;
   PATCHDEF2 HUGE *tmpsrc;

   tmpdest = (PATCH HUGE *) dest;
   tmpsrc  = (PATCHDEF2 HUGE *) src;
   for (i = 0; i < PATCHNAMELEN; ++i)
      {
      *dest = '\0';
      ++dest;
      }
   tmpdest->level          = tmpsrc->level;
   tmpdest->patchID        = tmpsrc->patchID;
   tmpdest->minOperators   = tmpsrc->minOperators;
   tmpdest->totalOperators = tmpsrc->totalOperators;
   src  += ((MAXOPSPERPATCH * 2) + 4);
   dest += 4;
   for (i = 0; i < 13; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   tmpdest->ctrlID[0]    = tmpsrc->ctrlID;
   tmpdest->ctrlDepth[0] = tmpsrc->ctrlDepth;
   tmpdest->ctrlDepth[1] = 0;
   tmpdest->ctrlDepth[2] = 0;
   tmpdest->ctrlUse[0]   = tmpsrc->ctrlUse;
   tmpdest->pan          = tmpsrc->pan;
   }


//**************************************************************************
//  Copy format 1 operator information to channel buffer
//**************************************************************************

static VOID copyOp1 (HPBYTE dest, HPBYTE src)
   {
   register WORD i;
   OP HUGE *tmpdest;
   HPBYTE tmp;

   src += 2;
   tmpdest = (OP HUGE *) dest;
   for (i = 0; i < 24; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   tmp = dest + 6;
   for (i = 0; i < 48; ++i)
      {
      *tmp = *src;
      ++src;
      ++tmp;
      }
   for (i = 0; i < 5; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   for (i = 0; i < MAXWAVESPEROP; ++i)
      if (!tmpdest->waveTable[i].blockCount)
         tmpdest->waveCount = (BYTE) i;
   if (i >= MAXWAVESPEROP)
      tmpdest->waveCount = (BYTE) MAXWAVESPEROP;
   }


//**************************************************************************
//  Copy format 2 operator information to channel buffer
//**************************************************************************

static VOID copyOp2 (HPBYTE dest, HPBYTE src)
   {
   register WORD i, j;
   OP HUGE *tmpdest;

   tmpdest = (OP HUGE *) dest;
   for (i = 0; i < 30; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   j = tmpdest->waveCount * 6;
   for (i = 0; i < j; ++i)
      {
      *dest = *src;
      ++src;
      ++dest;
      }
   }


//***************************************************************************
//  Process a MIDI pitch bend message
//***************************************************************************

static VOID DoPitchBend (BYTE chan, BYTE dat1, BYTE dat2)
   {
   register WORD i;
   DWORD freq;

   cpt->pbend = ((WORD) dat2 << 7) | dat1;

   // Locate and change all operators effected by pitch bend

   for (i = 0; i < wMaxSynthOp; ++i)
      if (op[i].channel == chan && op[i].note != UNUSEDB)
         {
         basenote  = (WORD) op[i].basenote;
         midipitch = (WORD) op[i].transposed;
         wavetype  = (WORD) op[i].type;
         dsppitch  = CALCPITCH (basenote, midipitch, wavetype);

         freq = GetPitch ((LONG) cpt->op[op[i].opnum].detune) +
                PitchBend ((short) cpt->pbend,
                           (short) cpt->patch.pitchWheel) +
                op[i].pfactor;
         SetOperatorPitch (i, LOWORD (freq >> 8));
         }
   }


//***************************************************************************
//  Process a custom controller message
//***************************************************************************

static VOID ProcessController (BYTE chan, BYTE type, WORD depth, WORD value)
   {
   register WORD i, m;

   pptr = (PATCH HUGE *) &cpt->patch;
   m = calcOFF(chan,0);
   switch (type)
      {
      case VIBRATOSPEED:
         if (cpt->wptLFO[VIBLFO].LFOnum != UNUSEDW)
            SetLFOrate (cpt->wptLFO[VIBLFO].LFOnum,
                        getLFOspeed(pptr->vibmod.speed,depth,value));
         break;

      case VIBRATODEPTH:
         cpt->wptLFO[VIBLFO].LFOdepth =
            getLFOdepth(pptr->vibmod.depth,depth,value);

         for (i = 0; i < pptr->totalOperators; ++i, ++m)
            if (WPT[m].wptno != UNUSEDB)
               SetLFOdepth ((chan << 3) + i, VIBLFO,
                            cpt->wptLFO[VIBLFO].LFOdepth);
         break;

      case TREMOLOSPEED:
         if (cpt->wptLFO[TREMLFO].LFOnum != UNUSEDW)
            SetLFOrate (cpt->wptLFO[TREMLFO].LFOnum,
                        getLFOspeed(pptr->tremmod.speed,depth,value));
         break;

      case TREMOLODEPTH:
         cpt->wptLFO[TREMLFO].LFOdepth =
            getLFOdepth(pptr->tremmod.depth,depth,value);

         for (i = 0; i < pptr->totalOperators; ++i, ++m)
            if (WPT[m].wptno != UNUSEDB)
               SetLFOdepth ((chan << 3) + i, TREMLFO,
                            cpt->wptLFO[TREMLFO].LFOdepth);
         break;

      case PANMODSPEED:
         if (cpt->wptLFO[PANLFO].LFOnum != UNUSEDW)
            SetLFOrate (cpt->wptLFO[PANLFO].LFOnum,
                        getLFOspeed(pptr->panmod.speed,depth,value));
         break;

      case PANMODDEPTH:
         cpt->wptLFO[PANLFO].LFOdepth =
            getLFOdepth(pptr->panmod.depth,depth,value);

         for (i = 0; i < pptr->totalOperators; ++i, ++m)
            if (WPT[m].wptno != UNUSEDB)
               SetLFOdepth ((chan << 3) + i, PANLFO,
                            cpt->wptLFO[PANLFO].LFOdepth);
         break;
      }
   }


//***************************************************************************
//  Determine the amplitude of an operator based on operator level,
//  key bias, note velocity, and channel volume.
//***************************************************************************

static WORD GetAmp (BYTE chan, short opnum, BYTE MIDIpitch, BYTE velocity)
   {
   register WORD amp, level;

   optr = (OP HUGE *) &cpt->op[opnum];

   // Calculate linear level from operator amplitude, patch level, and
   // channel volume -- level will range from 0 to 32512

   level = (((WORD) optr->opAmp *
             (WORD) cpt->patch.level) / 63) *
             (WORD) cpt->vol;

   switch (optr->keybias & 0x0F)
      {
      case 1:
         level -= (WORD) (((LONG) level * (LONG) (127 - MIDIpitch)) >> 8);
         break;

      case 2:
         level -= (WORD) (((LONG) level * (LONG) (127 - MIDIpitch)) >> 7);
         break;

      case 3:
         level -= (WORD) (((LONG) level * (LONG) MIDIpitch) >> 8);
         break;

      case 4:
         level -= (WORD) (((LONG) level * (LONG) MIDIpitch) >> 7);
         break;
      }

   switch (optr->velocityTableID)
      {
      case 1:
      case 4:
         amp = (WORD) velocity << 8;
         break;
      case 2:
      case 5:
         amp = ((WORD) velocity << 8) + ((WORD) velocity << 7);
         break;
      case 3:
      case 6:
         amp = (WORD) velocity << 9;
         break;
      case 0:
      default:
         amp = 0x3FFF;
         break;
      }
   if (amp > 0x7FFF)
      amp = 0x7FFF;
   if (optr->velocityTableID >= 4)
      amp = 0x7FFF - amp;

   // amp now ranges from 0 to 32767, scale by level and we're done

   return (WORD) (((LONG) level * (LONG) amp) / 32512L);
   }


//***************************************************************************
//  Locate a melodic operator to allocate
//***************************************************************************

static WORD GetMeloOperator (VOID)
   {
   register WORD i, j = DSPMAXOPS;
   short k;
   DWORD olddate = MAXDATE;

   // Find the oldest MELODIC note off

   for (i = 0; i < wMaxSynthOp; i++)
      {
      if (!(op[i].flags & ACTIVE) && op[i].channel != (BYTE) PERCUSSIONCH)
         {
         if (op[i].date < olddate)
            {
            j = i;
            olddate = op[i].date;
            }
         }
      }
   if (j < DSPMAXOPS)
      return j;

   // Find the oldest note off

   for (i = 0; i < wMaxSynthOp; i++)
      {
      if (!(op[i].flags & ACTIVE))
         {
         if (op[i].date < olddate)
            {
            j = i;
            olddate = op[i].date;
            }
         }
      }
   if (j < DSPMAXOPS)
      return j;

   // None found, so steal oldest melodic note with low channel and
   // low operator priority

   for (k = MAXCHANNEL-1; k >= 0; k--)
      {
      if (k != PERCUSSIONCH)
         {
         for (i = 0; i < wMaxSynthOp; i++)
            {
            if (op[i].channel == (BYTE) k && op[i].priority == 0 &&
                op[i].date < olddate)
               {
               j = i;
               olddate = op[i].date;
               }
            }
         if (j < DSPMAXOPS)
            return j;
         }
      }

   // Could not find any, so steal oldest melodic note with
   // low channel priority

   j = 0;
   for (k = MAXCHANNEL-1; k >= 0; k--)
      {
      if (k != PERCUSSIONCH)
         {
         for (i = 0; i < wMaxSynthOp; i++)
            {
            if (op[i].channel == (BYTE) k && op[i].date < olddate)
               {
               j = i;
               olddate = op[i].date;
               }
            }
         if (olddate != MAXDATE)
            break;
         }
      }

   // Turn off all operators with the same date

   for (i = 0; i < wMaxSynthOp; i++)
      {
      if (op[i].date == olddate)
         {
         OperatorCancel (i);
         if (op[i].flags & ACTIVE)
            WPToff (calcOFF (op[i].channel, op[i].opnum), i);
         op[i].flags = 0;  // not ACTIVE, SUSTAINED no more
         op[i].date  = dwDATE;
         }
      }

   return j;
   }


//***************************************************************************
//  Locates the next available operator for percussion
//  Called only if gPercussion >= MAXPERCUSSIONON.
//***************************************************************************

static WORD GetPercOperator (VOID)
   {
   register WORD i, j;
   DWORD olddate = MAXDATE;

   for (j = i = 0; i < wMaxSynthOp; i++)
      {
      if ((op[i].flags & ACTIVE) && op[i].channel == (BYTE) PERCUSSIONCH)
         {
         if (op[i].date < olddate)
            {
            j = i;
            olddate = op[i].date;
            }
         }
      }

   // Turn the note off

   OperatorCancel (j);
   if (op[j].flags & ACTIVE)
      WPToff (calcOFF (PERCUSSIONCH, op[j].opnum), j);
   return j;
   }


//**************************************************************************
//  Turn off operator and adjust operator and WPT information structures
//**************************************************************************

static VOID NoteOff (WORD chan, WORD slot)
   {
   if (chan == PERCUSSIONCH)
      {
      if (gPercussionOn)
         --gPercussionOn;
      }
   else
      {
      OperatorOff (slot);
      WPToff (calcOFF (chan, op[slot].opnum), slot);
      }
   op[slot].flags = 0;
   op[slot].date  = dwDATE;
   }


//**************************************************************************
//  Set new stereo pan for operator by modifying Wave Parameter Table
//**************************************************************************

static VOID ChangeWPTpan (WORD wptno, WORD pan)
   {
   if (qsoundMode & 2)
      TableModify (0, wptno, 14, pan);              // QSound pan
   else if (pan >= CENTER_PAN)
      {
      TableModify (0, wptno, 14, 0x7FFF);           // right pan
      TableModify (0, wptno, 15, pantab[127-pan]);  // left pan
      }
   else
      {
      TableModify (0, wptno, 14, pantab[pan]);      // right pan
      TableModify (0, wptno, 15, 0x7FFF);           // left pan
      }
   }


//**************************************************************************
//  Determine if MIDI Registered Parameter Number is one that we recognize
//**************************************************************************

static VOID CheckRPN (WORD chan)
   {
   pbsens[chan] = FALSE;
   if (RPNflag[chan] == 3)
      {
      switch (RPNword[chan])
         {
         case RPN_BENDSENS:
            pbsens[chan] = TRUE;
            break;
         case RPN_RESET:
            RPNflag[chan] = 0;
            pbsens[chan]  = FALSE;
            break;
         }
      }
   }


//**************************************************************************
//  Routine to dynamically allocate Wave Parameter Tables
//**************************************************************************

static WORD WPTnumber (WORD chan, WORD opnum)
   {
   register WORD i, j;
   WORD   k, wpt;
   DWORD  olddate = MAXDATE;
   WPTLFO lfo[3];

   // Look up WPT index

   i = calcOFF(chan,opnum);
   if ((j = (WORD) WPT[i].wptno) != UNUSEDB)
      {
      WPT[i].date = dwDATE;
      return j;
      }

   wpt = i;
   
   // Find oldest WPT

   i -= opnum;
   k = UNUSEDW;
   pptr = (PATCH HUGE *) &cpt->patch;
   for (j = 0; j < pptr->totalOperators; ++j, ++i)
      if (WPT[i].wptno != UNUSEDB && 
          WPT[i].date  <  olddate && 
          WPT[i].opmask == 0L)
         {
         k = i;
         olddate = WPT[i].date;
         }

   if (k == UNUSEDW)
      {
      i -= pptr->totalOperators;
      for (j = 0; j < pptr->totalOperators; ++j, ++i)
         if (WPT[i].wptno != UNUSEDB && WPT[i].date < olddate)
            {
            k = i;
            olddate = WPT[i].date;
            }
      if (k == UNUSEDW)
         return UNUSEDW;

      olddate = 1L;
      for (j = 0; j < wMaxSynthOp; ++j, olddate<<=1)
         if (WPT[k].opmask & olddate)
            {
            OperatorCancel (j);
            op[j].flags = 0;
            }
      WPT[k].opmask = 0L;
      }

   // Switch WPT

   i = (WORD) WPT[k].wptno;
   optr = (OP HUGE *) &cpt->op[opnum];
   for (j = 0; j < 3; ++j)
      {
      lfo[j].LFOnum = cpt->wptLFO[j].LFOnum;
      if (lfo[j].LFOnum == UNUSEDW)
         lfo[j].LFOnum = 0;
      lfo[j].LFOdepth = cpt->wptLFO[j].LFOdepth;
      }
   j = (WORD) GetOpPan(cpt->pan,optr->balance);
   SetWaveParameter (i,
                     (ENVDEF FAR *) optr->opEnvelope,
                     (WPTLFO FAR *) lfo,
                     (UINT) j);
   ChangeWPTpan (i, j);
   WPT[wpt].wptno  = (BYTE) i;
   WPT[wpt].opmask = 0L;
   WPT[wpt].date   = dwDATE;
   WPT[k].wptno = UNUSEDB;
   return i;
   }


//***************************************************************************
//  Determine operator pitch based on transposed MIDI note and detuning
//***************************************************************************

static LONG GetPitch (LONG detune)
   {
   LONG diff;

   // First check to see if detuning is necessary

   if (detune == 0L)
      return dsppitch;

   // Factor in any detuning

   if (detune > 0L)
      diff = (detune *
             (CALCPITCH (basenote, midipitch+1, wavetype) - dsppitch)) / 100L;
   else
      diff = (detune *
             (dsppitch - CALCPITCH (basenote, midipitch-1, wavetype))) / 100L;

   return (dsppitch + diff);
   }


//***************************************************************************
//  Calculate new pitch based on pitch bend information
//***************************************************************************

static LONG PitchBend (short pb, short range)
   {
   LONG x, rem, p1, p2;

   // First check to see if pitch bend is even necessary

   if (pb == CENTER_PITCH || range == 0)
      return 0L;

   if (pb > CENTER_PITCH)     // Bend note upward
      {
      // x   = closest MIDI note (truncated)
      // rem = remainder (in pitch bend units)

      x = (LONG) range * ((LONG) pb - (LONG) CENTER_PITCH);
      rem = (x & 8191L) / (LONG) range;
      x >>= 13;

      // p2 = closest MIDI note pitch value (for DSP)

      p2 = CALCPITCH (basenote, midipitch+x, wavetype);
      if (rem)
         {
         // Compute fractional amount between MIDI notes
         // Unfortunately, this is a linear calculation

         p1 = CALCPITCH (basenote, midipitch+x+1, wavetype);
         p2 += ((LONG) (p1 - p2) * (LONG) rem * (LONG) range) >> 13;
         }
      }
   else                       // Bend note downward
      {
      x = (LONG) range * ((LONG) CENTER_PITCH - (LONG) pb);
      rem = (x & 8191L) / (LONG) range;
      x >>= 13;
      p2 = CALCPITCH (basenote, midipitch-x, wavetype);
      if (rem)
         {
         p1 = CALCPITCH (basenote, midipitch-x-1, wavetype);
         p2 -= ((LONG) (p2 - p1) * (LONG) rem * (LONG) range) >> 13;
         }
      }

   return (p2 - dsppitch);
   }


//***************************************************************************
//  Locate an LFO generator to allocate
//***************************************************************************

static WORD GetLFO (VOID)
   {
   register WORD i;

   for (i = 0; i < MAXLFOS; ++i)
      if (lfo[i].channel == UNUSEDB)
         {
         lfo[i].channel = (BYTE) i;
         return i;
         }
   return UNUSEDW;
   }


//***************************************************************************
//  Turn off an active LFO generator
//***************************************************************************

static VOID CancelLFO (WORD num)
   {
   lfo[num].channel = UNUSEDB;
   LFOdeactivate (num);
   }


//***************************************************************************
//  Adjust number of available operators depending upon the playback mode
//***************************************************************************

VOID ChangeOpCount (WORD synthop, WORD percop)
   {
   register WORD i, j;
   WORD currentMaxOps;

   currentMaxOps = wMaxSynthOp;
   wMaxSynthOp   = synthop;
   wMaxPercOp    = percop;

   if (wMidiSynthAllocated)
      {
      if (synthop > DSPMINOPS)
         SetMode (MAXOPERATORS, (synthop == DSPMAXOPS)? 0: synthop);

      // Cancel excess operators

      for (i = synthop; i < currentMaxOps; ++i)
         {
         OperatorCancel (i);
         if (op[i].flags & ACTIVE)
            {
            j = op[i].channel;
            WPToff (calcOFF (j, op[i].opnum), i);
            if (j == PERCUSSIONCH && gPercussionOn)
               --gPercussionOn;
            }
         op[i].flags = 0;
         op[i].date  = dwDATE;
         }
      if (synthop <= DSPMINOPS)
         SetMode (MINOPERATORS, (synthop == DSPMINOPS)? 0: synthop);
      }
   else if (playMode != NOSYNTH)
      {
      if (synthop > DSPMINOPS)
         SetMode (MAXOPERATORS, (synthop == DSPMAXOPS)? 0: synthop);
      else
         SetMode (MINOPERATORS, (synthop == DSPMINOPS)? 0: synthop);
      }
   }


//**************************************************************************
//  Process MIDI reverb level message
//**************************************************************************

static VOID ProcessReverb (WORD chan, WORD dat)
   {
   WORD i, j;

   ch[chan].reverb = (BYTE) dat;
   i = calcOFF(chan,0);
   dat <<= 8;

   // For each active operator, adjust reverb level

   for (j = 0; j < MAXOPSPERPATCH; j++, i++)
      if (WPT[i].wptno != UNUSEDB)
         SetReverbChannel ((UINT) WPT[i].wptno, (UINT) dat);
   }


//***************************************************************************
//  Send current reverb settings for each MIDI channel
//***************************************************************************

VOID SetReverbControls (VOID)
   {
   BYTE chan;

   if (!wMidiSynthAllocated)
      return;

   for (chan = 0; chan < MAXCHANNEL; ++chan)
      ProcessReverb (chan, ch[chan].reverb);
   }


//***************************************************************************
//  Send current pan settings for each MIDI channel
//***************************************************************************

VOID SetPanControls (VOID)
   {
   BYTE chan;

   if (!wMidiSynthAllocated)
      return;

   for (chan = 0; chan < MAXCHANNEL; ++chan)
      DoController (chan, PAN, ch[chan].pan);
   }

