/*
    CGAEX: CGA video exerciser
    Copyright (C) 2003  John Elliott

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <dos.h>
#include <string.h>
#include <conio.h>
#include "vga.h"

#include <stdio.h>
#include <stdlib.h>

/* MDA attributes:
 * bits 0-3: Foreground. 0=black 1=underlined 7=white 9=bright underline 
 *           15=bright white
 * bits 4-7: Background. 0=black 7=white 15=bright white
 */

gfx_byte mda_attr=0x70;
gfx_byte cga_background=CGA_WHITE;
gfx_byte cga_foreground=CGA_BLACK;
gfx_byte vga_foreground=VGA_BLACK;
gfx_byte vga_background=VGA_GREY87;

gfx_word gfx_x = 0;
gfx_word gfx_y = 0;

#define VGA_WIDTH  (gfx_force_mda ? 80 : 40)
#define VGA_HEIGHT 25

gfx_byte gfx_force_cga = 0;
gfx_byte gfx_force_mda = 0;
gfx_byte gfx_cga_pal   = 1;

void gfx_setbackground(gfx_byte bg)
{
	vga_background = bg;
	switch(bg)
	{
		case VGA_GREY87:
		case VGA_WHITE: cga_background = CGA_WHITE; break;
		case VGA_BLACK: cga_background = CGA_BLACK; break;
		case VGA_ZCYAN: cga_background = CGA_CYAN;  break;
		default:	cga_background = CGA_MAGENTA; break;
	}
	mda_attr &= 0x0F; 
	switch(bg)
	{
		case VGA_GREY87: 
		case VGA_WHITE:  mda_attr |= 0x70; break;
	}
}


void gfx_setforeground(gfx_byte bg)
{
	vga_foreground = bg;
	switch(bg)
	{
		case VGA_GREY87:
		case VGA_WHITE: cga_foreground = CGA_WHITE; break;
		case VGA_BLACK: cga_foreground = CGA_BLACK; break;
		case VGA_ZCYAN: cga_foreground = CGA_CYAN;  break;
		default:	cga_foreground = CGA_MAGENTA; break;
	}
	mda_attr &= 0xF0; 
	switch(bg)
	{
		case VGA_GREY87: mda_attr |= 0x07; break;
		case VGA_WHITE:  mda_attr |= 0x0F; break;
		case VGA_BLACK:  break; 
		default:         mda_attr |= 0x01; break;
	}
}

void gfx_setcursor(gfx_word shape)
{
	union REGS r;
/* Work around AMIBIOS bug */
	r.x.ax = 0x0F00;
	int86(0x10, &r, &r);	/* AL is now mode */
	r.h.ah = 1;
	r.x.cx = shape;
	int86(0x10, &r, &r);
}


gfx_word gfx_getcursor()
{
	union REGS r;
	r.x.ax = 0x0F00;
	int86(0x10, &r, &r);	/* BH is current page number */
	r.h.ah = 3;
	int86(0x10, &r, &r);
	return r.x.cx;
}



/* In CGA 320x200 mode: Select palette */
void cga_setpalette(gfx_byte pal)
{
	union REGS r;

	r.x.ax = 0x0B00;
	r.x.bx = 0x100 | pal;
	int86(0x10, &r, &r);
}


/* Select video mode */
void gfx_setmode(gfx_byte mode)
{
	union REGS r;

	r.x.ax = mode;	/* AH = 0: Set video mode */
	int86(0x10, &r, &r);
}

gfx_byte gfx_getmode(void)
{
	union REGS r;

	r.h.ah = 0x0F;	/* Get video mode */
	int86(0x10, &r, &r);
	return r.h.al;
}


/* Probe video hardware to get adapter type */
gfx_byte gfx_probe()
{
	union REGS r;
	/* Check if DRDOS's MemoryMAX is using VGA memory */
	r.x.ax = 0x12FF;
	r.x.bx = 6;
	r.x.cx = 0;
	r.x.dx = 0;
	int86(0x2F, &r, &r);

	if (r.x.ax || (r.x.cx == 0))
	{
 		r.x.ax = 0x1200;	/* Test for VGA */
		r.h.bl = 0x32;
		int86(0x10, &r, &r);
		if (r.h.al == 0x12) return 3;	/* VGA */

		r.h.ah = 0x12;		/* Test for EGA */
		r.h.bl = 16;
		int86(0x10, &r, &r);
		if (r.h.bl <= 3) return 2;	/* EGA */
	}
	r.h.ah = 0x0F;		/* Test for mono mode */
	int86(0x10, &r, &r);
	if (r.h.al == 7) return 0;	/* MDA */
	return 1;			/* CGA */
}


/* Select an appropriate graphics mode */
gfx_byte gfx_gmode(void)
{
	gfx_byte om = gfx_getmode();

	switch(gfx_probe())
	{
		case 2: case 1: gfx_force_cga = 1; break;
		case 0:		gfx_force_mda = 1; break;
	}
	if (gfx_force_mda) gfx_setmode(7);
	else if (gfx_force_cga) 
	{
		gfx_setmode(0x05);
		cga_setpalette(gfx_cga_pal);
	}
	else	gfx_setmode(0x13);	/* VGA mode 0x13 */
	return om;
}



void gfx_cls(void)
{
	if (gfx_force_mda)
	{
#asm
	push	di
	push	es
	mov	bl,_mda_attr
	mov	cx,#2000
	mov	ax,#0xB000
	mov	es,ax
	mov	di,#0
	mov	al,#' '
	mov	ah,bl
	rep	stosw
	pop	es
	pop	di
#endasm

	}
	else if (gfx_force_cga)
	{
#asm
	push	di
	push	es
	mov	bl,_cga_background
	mov	cx,#4000
	mov	ax,#0xB800
	mov	es,ax
	mov	di,#0		;Odd-numbered lines
	mov	al,bl
	mov	ah,bl
	rep	stosw
	mov	cx,#4000	;Even-numbered lines
	mov	di,#0x2000
	rep	stosw
	pop	es
	pop	di
#endasm
	}
	else
	{
#asm
	push	di
	push	es
	mov	bl,_vga_background
	mov	cx,#32000
	mov	ax,#0xA000
	mov	es,ax
	mov	ax,#0
	mov	di,ax
	mov	al,bl
	mov	ah,bl
	rep	stosw
	pop	es
	pop	di
#endasm
	}
}

gfx_byte far *cga_pixaddr(gfx_word y, gfx_word x)
{
	gfx_byte far *mem = MK_FP(0xB800, x/4);

	if (y & 1) mem += 0x2000;
	mem += 80 * (y/2);
	return mem;
}


void mda_putchar(gfx_word y, gfx_word x, unsigned char c)
{
	gfx_byte far *mem = MK_FP(0xB000, 160 * y + 2 * x);
	mem[0] = c;
	mem[1] = mda_attr;
}



void cga_putchar(gfx_word y, gfx_word x, char c)
{
	gfx_byte *src     = gfx_font + 8 * (gfx_byte)c;
	gfx_word px, py;
	gfx_byte mask, dmask;

	for (py = 0; py < 8; py++)
	{
		gfx_byte far *mem = cga_pixaddr(y*8+py,x*8);
		mask = 0x80;
		dmask = 0xC0;
		for (px = 0; px < 8; px++)
		{
			mem[0] &= ~dmask;
			mem[0] |= (src[py] & mask) ? 
				(cga_foreground & dmask) : 
				(cga_background & dmask) ;
			mask = mask >> 1;
			dmask = dmask >> 2;
			if (!dmask) { dmask = 0xC0; ++mem; }
		}
	}
}



void vga_putchar(gfx_word y, gfx_word x, char c)
{
	gfx_byte far *mem = MK_FP(0xA000, 2560 * y + 8 * x);
	gfx_byte *src     = gfx_font + 8 * (gfx_byte)c;
	gfx_word px, py;
	gfx_byte mask;

	for (py = 0; py < 8; py++)
	{
		mask = 0x80;
		for (px = 0; px < 8; px++)
		{
			mem[px] = (src[py] & mask) ? vga_foreground : vga_background;
			mask = mask >> 1;
		}
		mem += 320;
	}
}


void gfx_putchar(gfx_word y, gfx_word x, char c)
{
	if (gfx_force_mda) mda_putchar(y,x,(unsigned char)c);
	if (gfx_force_cga) cga_putchar(y,x,c);
	else	       vga_putchar(y,x,c);
}



void gfx_putstr(char *s)
{
	while (*s)
	{
		gfx_putchar(gfx_y, gfx_x, *s);
		++s;
		++gfx_x;
		if (gfx_x == VGA_WIDTH) { gfx_x = 0; ++gfx_y;}
		if (gfx_y == VGA_HEIGHT) { gfx_y = 0; }
	}
}

char gfx_getch(void)
{
	char c = getch();

	if (c) return c;
	c = getch();	/* Extended keycode */
	switch(c)
	{
		case 59: /* F1 */
		case 60: /* F2 */
		case 61: /* F3 */
		case 62: /* F4 */
		case 63: /* F5 */
		case 64: /* F6 */
		case 65: /* F7 */
		case 66: /* F8 */
		case 67: /* F9 */
		case 68: /* F10 */
			 return (c - 75);
		case 71: return 'A' - '@'; /* Home */
		case 72: return 'E' - '@'; /* Up */
		case 73: return 'R' - '@'; /* PageUp */
		case 75: return 'S' - '@'; /* Left */
		case 77: return 'D' - '@'; /* Right */
		case 79: return 'F' - '@'; /* End */
		case 80: return 'X' - '@'; /* Down */
		case 81: return 'C' - '@'; /* PageDn */
		case 82: return 'V' - '@'; /* Insert */
		case 83: return 'G' - '@'; /* Delete */
		default: break;
	}
	return 0;
}


void mda_stripe(gfx_word y, gfx_word x)
{
	int n;
	for (n = 0; n < 4; n++) mda_putchar(y,x+n,'/');
}


void cga_stripe(gfx_word y, gfx_word x)
{
	gfx_word n, m;
	gfx_byte mask;
	
	for (n = 0; n < 8; n++)
	{
		gfx_byte far *mem = cga_pixaddr(y*8+n,x*8);
		switch(n%4)
		{
			case 0: mask = 0x03; break;
			case 1: mask = 0x0C; break;
			case 2: mask = 0x30; break;
			case 3: mask = 0xC0; break;
		}
		mem -= (n/4);
		for (m = 0; m < 32; m++)
		{
			mem[0] &= ~mask;
			mem[0] |= ((m & 8) ? (CGA_CYAN : CGA_MAGENTA)) & mask;
			mask = mask >> 2;
			if (mask == 0) { mask = 0xC0; ++mem; }	
		}
	}
}


void vga_stripe(gfx_word y, gfx_word x)
{
	gfx_word n, m;
	gfx_byte far *mem = MK_FP(0xA000, y * 2560 + x*8);

	for (n = 0; n < 8; n++)
	{
		for (m = 0; m < 8; m++)
		{
			mem[m]    = VGA_ZRED;
			mem[m+8]  = VGA_ZYELLOW;
			mem[m+16] = VGA_ZGREEN;
			mem[m+24] = VGA_ZCYAN;
		}
		mem += 319;
	}
}

void gfx_stripe(gfx_word y, gfx_word x)
{
	if (gfx_force_mda)      mda_stripe(y,x);
	else if (gfx_force_cga) cga_stripe(y,x);
	else	       	    vga_stripe(y,x);
}


void gfx_hbar(gfx_word  y, gfx_word x, gfx_word w, char *s)
{
	gfx_word n;

	gfx_setbackground(VGA_BLACK);
	gfx_setforeground(VGA_WHITE);
	for (n = 0; n < w; n++) gfx_putchar(y, x + n, ' ');
	gfx_x = x; gfx_y = y;
	gfx_putstr(s);
	gfx_stripe(y,x + w - 5);
}

void gfx_bbar(char *s)
{
	gfx_hbar(VGA_HEIGHT - 3, 0, VGA_WIDTH, s);
}

gfx_word gfx_menu_width(GFX_MENU *menu)
{
	int ml = 6 + strlen(menu->vm_title);
	int n, l;

	for (n = 0; n < menu->vm_count; n++)
	{
		l = 2 + strlen(menu->vm_entries[n].vml_text);
		if (l > ml) ml = l;
	}
	return ml;
}



void mda_menu_redrawline(GFX_MENU *menu, gfx_word opt)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word m, bkg;
	gfx_byte far *mem = MK_FP(0xB000, 0);

	if (menu->vm_active == opt) bkg = 0x07;
	else			    bkg = 0x70;

	mem += 160 * (y+opt+1) + 2 * (x+1);
	for (m = 0; m < (w-2); m++)
	{
		mem[m*2+1] = bkg;
	}
}



void cga_menu_redrawline(GFX_MENU *menu, gfx_word opt)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word n, m, d, mask, bkg;
	gfx_byte far *mem;

	if (menu->vm_active == opt) bkg = CGA_CYAN;
	else			    bkg = CGA_WHITE;

	for (n = 0; n < 8; n++)
	{
		mem = cga_pixaddr(n + (8 * (y+opt+1)), 8 * x);
		for (m = 0; m < w*2; m++)
		{
			mask = 0xC0;
			for (d = 0; d < 4; d++)
			{
				if ((mem[m] & mask) == (CGA_CYAN  & mask) ||
				    (mem[m] & mask) == (CGA_WHITE & mask))
				{
					mem[m] &= ~mask;
					mem[m] |= (bkg & mask);
				}
				mask = mask >> 2;
			}
		}
	}
}


void vga_menu_redrawline(GFX_MENU *menu, gfx_word opt)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word n, m, bkg;
	gfx_byte far *mem = MK_FP(0xA000, 0);

	if (menu->vm_active == opt) bkg = VGA_ZCYAN;
	else			    bkg = VGA_WHITE;

	mem += 2560 * (y+opt+1) + 8 * x;
	for (n = 0; n < 8; n++)
	{
		for (m = 0; m < w*8; m++)
		{
			if (mem[m] == VGA_ZCYAN || mem[m] == VGA_WHITE)
			{
				mem[m] = bkg;
			}
		}
		mem += 320;
	}
}


void gfx_menu_redrawline(GFX_MENU *menu, gfx_word opt)
{
	if (gfx_force_mda)      mda_menu_redrawline(menu, opt);
	else if (gfx_force_cga) cga_menu_redrawline(menu, opt);
	else	            vga_menu_redrawline(menu, opt);
}



void gfx_draw_menuline(gfx_word y, gfx_word x, gfx_word w, GFX_MENULINE *mline)
{
	gfx_word sl = strlen(mline->vml_text);
	gfx_word n;
	for (n = 0;  n < sl; n++) gfx_putchar(y, x+n+1, mline->vml_text[n]);
	for (n = sl; n <= w; n++) gfx_putchar(y, x+n+1, ' ');
	if (gfx_force_mda) 
	{
		gfx_byte a = mda_attr;
		mda_attr = 0x70;
		mda_putchar(y, x, 186);
		mda_putchar(y, x+n, 186);
		mda_attr = a;
	}
	else	gfx_putchar(y, x, ' ');
}


gfx_byte *cga_backmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_byte *backmem = NULL;
	gfx_byte far *mem;
	gfx_word n,m,pw;

	x *= 8;
	y *= 8;
	w *= 8;
	h *= 8;
	pw = (w+3)/4;
	backmem = malloc(pw * h);
	if (!backmem) return NULL;

	for (n = 0; n < h; n++)
	{
		mem = cga_pixaddr(y+n, x);
		for (m = 0; m < pw; m++)
		{
			backmem[n*pw+m] = mem[m];
		}
	}
	return backmem;
}


gfx_byte *mda_backmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_byte far *mem = MK_FP(0xB000, 0);
	gfx_byte *backmem = NULL;
	gfx_word n,m;

	x *= 2;
	w *= 2;
	backmem = malloc(w * h);
	if (!backmem) return NULL;

	for (n = 0; n < h; n++)
		for (m = 0; m < w; m++)
		{
			backmem[n*w+m] = mem[(n+y)*160 + (x+m)];
		}
	return backmem;
}




gfx_byte *vga_backmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_byte far *mem = MK_FP(0xA000, 0);
	gfx_byte *backmem = NULL;
	gfx_word n,m;

	x *= 8;
	y *= 8;
	w *= 8;
	h *= 8;
	backmem = malloc(w * h);
	if (!backmem) return NULL;

	for (n = 0; n < h; n++)
		for (m = 0; m < w; m++)
		{
			backmem[n*w+m] = mem[(n+y)*320 + (x+m)];
		}
	return backmem;
}


gfx_byte *gfx_backmenu(GFX_MENU *menu)
{
	if (gfx_force_mda) return mda_backmenu(menu);
	else if (gfx_force_cga) return cga_backmenu(menu);
	else return vga_backmenu(menu);
}




void mda_restrmenu(GFX_MENU *menu, gfx_byte *backmem)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_byte far *mem = MK_FP(0xB000, 0);
	gfx_word n,m;

	x *= 2;
	w *= 2;
	for (n = 0; n < h; n++)
		for (m = 0; m < w; m++)
		{
			mem[(n+y)*160 + (x+m)] = backmem[n*w+m];
		}
}


void cga_restrmenu(GFX_MENU *menu, gfx_byte *backmem)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_word n,m,pw;
	gfx_byte far *mem;
	
	x *= 8;
	y *= 8;
	w *= 8;
	h *= 8;

	pw = (w+3)/4;
	for (n = 0; n < h; n++)
	{
		mem = cga_pixaddr(y+n,x);
		for (m = 0; m < pw; m++)
		{
			if (backmem) mem[m] = backmem[n*pw+m];
			else         mem[m] = CGA_MAGENTA;
		}
	}
	if (backmem) free(backmem);
}




void vga_restrmenu(GFX_MENU *menu, gfx_byte *backmem)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word h = 2 + menu->vm_count;
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - h) / 2;
	gfx_byte far *mem = MK_FP(0xA000, 0);
	gfx_word n,m;

	x *= 8;
	y *= 8;
	w *= 8;
	h *= 8;

	for (n = 0; n < h; n++)
		for (m = 0; m < w; m++)
		{
			if (backmem) mem[(n+y)*320 + (x+m)] = backmem[n*w+m];
			else         mem[(n+y)*320 + (x+m)] = VGA_GREY87;
		}
	if (backmem) free(backmem);
}

void gfx_restrmenu(GFX_MENU *menu, gfx_byte *backmem)
{
	if (gfx_force_mda)      mda_restrmenu(menu, backmem);
	else if (gfx_force_cga) cga_restrmenu(menu, backmem);
	else		    vga_restrmenu(menu, backmem);
}



void mda_drawmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word n;
	
	gfx_hbar(y, x, w, menu->vm_title);
	for (n = 0; n < menu->vm_count; n++)
	{
		if (menu->vm_active == n) mda_attr = 0x07;
		else			  mda_attr = 0x70;
		gfx_draw_menuline(y+n+1, x, w-2, &menu->vm_entries[n]);
	}
	mda_attr = 0x70;
	mda_putchar(y+menu->vm_count+1,x,200);
	for (n = 1; n < w-1; n++) mda_putchar(y+menu->vm_count+1, x+n, 205);
	mda_putchar(y+menu->vm_count+1,x+n,188);
}


void cga_drawmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word n;
	gfx_byte far *mem;
	
	gfx_hbar(y, x, w, menu->vm_title);
	for (n = 0; n < menu->vm_count; n++)
	{
		if (menu->vm_active == n) gfx_setbackground(VGA_ZCYAN);
		else			  gfx_setbackground(VGA_WHITE);
		gfx_setforeground(VGA_BLACK);
		gfx_draw_menuline(y+n+1, x, w-2, &menu->vm_entries[n]);
	}
	gfx_setbackground(VGA_WHITE);
	for (n = 0; n < w; n++) gfx_putchar(y+menu->vm_count+1, x+n, ' ');
	for (n = 0; n < 7+8*menu->vm_count; n++) 
	{
		mem = cga_pixaddr(n + (y+1)*8, x*8);
		mem[0]      &= 0x3F;
		mem[2*w-1]  &= 0xFC;
	}
	mem = cga_pixaddr(n + (y+1)*8, x*8);
	for (n = 0; n < 2*w; n++) mem[n] = CGA_BLACK;
}



void vga_drawmenu(GFX_MENU *menu)
{
	gfx_word w = gfx_menu_width(menu);
	gfx_word x = (VGA_WIDTH - w) / 2;
	gfx_word y = (VGA_HEIGHT - (2 + menu->vm_count)) / 2;
	gfx_word n, m;
	gfx_byte far *mem = MK_FP(0xA000, 0);
	
	gfx_hbar(y, x, w, menu->vm_title);
	for (n = 0; n < menu->vm_count; n++)
	{
		if (menu->vm_active == n) gfx_setbackground(VGA_ZCYAN);
		else			  gfx_setbackground(VGA_WHITE);
		gfx_setforeground(VGA_BLACK);
		gfx_draw_menuline(y+n+1, x, w-2, &menu->vm_entries[n]);
	}
	gfx_setbackground(VGA_WHITE);
	for (n = 0; n < w; n++) gfx_putchar(y+menu->vm_count+1, x+n, ' ');
	m = (y + 1) * 2560 + (x*8);
	for (n = 0; n < 7+8*menu->vm_count; n++) 
	{
		mem[m]       = VGA_BLACK;
		mem[m+8*w-1] = VGA_BLACK;
		m += 320;
	}
	for (n = 0; n < 8*w; n++) mem[m+n] = VGA_BLACK;
}

void gfx_drawmenu(GFX_MENU *menu)
{
	if (gfx_force_mda)	mda_drawmenu(menu);
	else if (gfx_force_cga)	cga_drawmenu(menu);
	else			vga_drawmenu(menu);
}



gfx_word gfx_domenu(GFX_MENU *menu)
{
	gfx_word inmenu = 1, old_opt;
	char keyhit;
	gfx_byte *backmem;
	
	/* Back up screen behind the menu */
	backmem = gfx_backmenu(menu);
	gfx_drawmenu(menu);
	while (inmenu)
	{
		keyhit = gfx_getch();
		if (!keyhit) continue;
		/* TODO: Support mice? */

		old_opt = menu->vm_active;
		switch(keyhit)
		{
			case 'E'-'@':	if (menu->vm_active > 0) --menu->vm_active;
					else menu->vm_active = menu->vm_count - 1;
					break;
			case 'X'-'@':	++menu->vm_active;
					if (menu->vm_active >= menu->vm_count) menu->vm_active = 0;
					break;
			case '\n':
			case '\r':
			case 27:	inmenu = 0; break;
		}
		if (menu->vm_active != old_opt)
		{
			gfx_menu_redrawline(menu, menu->vm_active);
			gfx_menu_redrawline(menu, old_opt);
			old_opt = menu->vm_active;
		}

	}
	/* Restore screen behind the menu */
	gfx_restrmenu(menu, backmem);
	if (keyhit == 27) return -1;
	return menu->vm_active;
}







