/*
 * Get input string, with VMS-style input line editing
 * and previous-command scrolling.
 *
 * Written for Turbo C 2.0 / Borland C++ 2.0
 * Bob Bybee, 2/91
 *
 * Modified for usual msdos keys,
 * Eyal Lebedinsky, Feb 1992
 *
 * Use microsoft C. Original program compiled and run without any changes
 *   with this compiler.
 * Dont loose strings that look like part of an escape sequence [push_back].
 * Show insert/overtype mode as cursor size[sys_show].
 * Initial mode is overwrite!
 * Selectable system access routines [get_str_init()].
 * User can select length for shortest string kept [shortest_saved].
 * If an earlier line is selected (with up/down arrows) and entered then it
 *   is not saved again [last_recalled].
 * Provide history search with PAGE_UP key: Previous entry that matches up
 *   to the cursor is retrieved.
 * Added keys:
 *   HOME		move to start of line (CTRL-B)
 *   END		move to end   of line (CTRL-E)
 *   ESC		clear line (CTRL-X)
 *   INSERT		toggle mode (CTRL-A)
 * Added keys with new functions:
 *   C_RIGHT_ARROW	move to next word start
 *   C_LEFT_ARROW	move to prev word start
 *   DEL		delete char at right of cursor
 *   C_END		delete right of cursor to end of line
 *   C_HOME		delete left of cursor to start of line
 *   PAGE_UP		search line prefix
 * Notes:
 * <time.h> used to access clock(), using CLK_TCK as ticks-per-second.
 * <dos.h> used to access int86() for setting cursor shape.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include <dos.h>

/* ASCII key definitions */
#define ESC_KEY			0x1b
#define DELETE_KEY		0x7f
#define BACKSPACE_KEY	0x08
#define RETURN_KEY 		0x0d
#define CTRL(x) ((x) & 0x1f)

/* Arbitrary values for tracking cursor key entry.
 * These happen to match PC BIOS scan codes, but any
 * unique values would work.
 */
#define UP_ARROW	0x4800
#define DOWN_ARROW	0x5000
#define RIGHT_ARROW	0x4d00
#define LEFT_ARROW	0x4b00
#define HOME		0x4700
#define END		0x4f00
#define INSERT		0x5200
#define C_LEFT_ARROW	0x7300
#define C_RIGHT_ARROW	0x7400
#define C_END		0x7500
#define C_HOME		0x7700
#define DEL		0x5300
#define PAGE_UP		0x4900

/* MAX_RECALL is two greater than the number of lines
 * we want in the "lastlines" recall buffer.
 */
#define MAX_RECALL	22
#define	SHORTEST_SAVED	3
#define RECSUB(x) (((x) + MAX_RECALL) % MAX_RECALL)
#define RECALL_LEN	100
static char lastlines[MAX_RECALL][RECALL_LEN];
static int push_back[10], push_back_index = 0;

static int num_got;			/* # chars in input buffer */
static int cursor_pos;			/* cursor position on line */
static int lastptr = 0;			/* ptr to last line entered */
static char far *str_ptr;		/* ptr to current input string */


/* prototypes for this file */
static void clear_line( void );
static int cursor_right( void );
static int cursor_left( void );
static int get_char_esc( void );
static void put_str( char far *str );

/* external functions (see listing2.c) */
static void far	sys_put( int ch );
static int far	sys_get( void );
static void far	sys_show( int mode );
static void	(far *sys_putchar)( int ch ) = sys_put;
static int	(far *sys_getchar)( void ) = sys_get;
static int	(far *sys_showmode)( int mode ) = sys_show;
static int	shortest_saved = SHORTEST_SAVED;
static int	init_mode = 0;

/*
 *
 */
void far
get_str_init (int (far *get)( void ),
	void (far *put)( int ch ),
	void (far *show)( int mode ),
	int shortest, int mode)
{
	if (put)
		sys_putchar = put;
	else
		sys_putchar = sys_put;

	if (get)
		sys_getchar = get;
	else
		sys_getchar = sys_get;

	if (show)
		sys_showmode = show;
	else
		sys_showmode = sys_show;
	if (shortest)
		shortest_saved = shortest;
	init_mode = mode;
}

/*
 * get_str() is called by main() to get a line of input.
 * The input line is placed in "str" and will be no
 * more than "len" characters.
 */
void far
get_str( char far *str, int len )
{
	int i, c, curptr, insert_mode, last_recalled = -1;

	push_back_index = 0;
	sys_showmode (insert_mode = init_mode);
	num_got = 0;
	cursor_pos = 0;
	str_ptr = str;				/* copy the buffer pointer */
	curptr = RECSUB(lastptr + 1);
	lastlines[curptr][0] = '\0';
	lastlines[RECSUB(curptr + 1)][0] = '\0';
	if (len > RECALL_LEN - 1)	/* limit len to RECALL_LEN */
		len = RECALL_LEN - 1;

	while (1)
		{
		c = get_char_esc();
		if (c == RETURN_KEY)
			break;
		else if (c == DELETE_KEY || c == BACKSPACE_KEY || c == DEL) {
			if (c == DEL && cursor_pos < num_got ||
			    c != DEL && cursor_left()) {
				++cursor_pos;
				for (i = cursor_pos; i < num_got; ++i) {
					str[i - 1] = str[i];
					(*sys_putchar)(str[i]);
				}
				(*sys_putchar)(' ');
				for (i = cursor_pos; i <= num_got; ++i)
					(*sys_putchar)('\b');
				--num_got;
				--cursor_pos;
			}
		} else if (c == C_END) {
			if (cursor_pos < num_got) {
				for (i = cursor_pos; i < num_got; ++i) {
					str[i] = ' ';
					(*sys_putchar)(' ');
				}
				for (i = cursor_pos; i < num_got; ++i)
					(*sys_putchar)('\b');
				num_got = cursor_pos;
			}
		} else if (c == C_HOME) {
			if (cursor_pos > 0) {
				for (i = 0; i < cursor_pos; ++i)
					(*sys_putchar)('\b');
				num_got -= cursor_pos;
				for (i = 0; i < num_got; ++i) {
					str[i] = str[cursor_pos+i];
					(*sys_putchar)(str[i]);
				}
				for (i = 0; i < cursor_pos; ++i)
					(*sys_putchar)(' ');
				for (i = cursor_pos + num_got; i-- > 0;)
					(*sys_putchar)('\b');
				cursor_pos = 0;
			}
		} else if (c == CTRL('X') ||	/* erase line? */
			 c == ESC_KEY)
			clear_line();
		else if (c == CTRL('A') ||	/* insert/overtype? */
			 c == INSERT)
			sys_showmode (insert_mode ^= 1);
		else if (c == CTRL('B') ||	/* beginning-of-line? */
			 c == HOME)
			{
			while (cursor_left())
				;
			}
		else if (c == CTRL('E') ||	/* end-of-line? */
			 c == END)
			{
			while (cursor_right())
				;
			}
		else if (c == CTRL('R'))	/* recall last line? */
			{
			clear_line();
			strcpy(str, lastlines[lastptr]);
			if ((num_got = strlen(str)) > 0)
				{
				put_str(str);
				last_recalled = lastptr;
				break;
				}
			}
		else if (c == UP_ARROW)
			{
			clear_line();
			if (lastlines[curptr][0] != '\0' ||
				lastlines[RECSUB(curptr - 1)][0] != '\0')
				{
				curptr = RECSUB(curptr - 1);
				strcpy(str, lastlines[curptr]);
				last_recalled = curptr;
				put_str(str);
				cursor_pos = num_got = strlen(str);
				}
			}
		else if (c == DOWN_ARROW)
			{
			clear_line();
			if (lastlines[curptr][0] != '\0' ||
				lastlines[RECSUB(curptr + 1)][0] != '\0')
				{
				curptr = RECSUB(curptr + 1);
				strcpy(str, lastlines[curptr]);
				last_recalled = curptr;
				put_str(str);
				cursor_pos = num_got = strlen(str);
				}
			}
		else if (c == LEFT_ARROW)
			{
			if (cursor_pos > 0)
				{
				(*sys_putchar)('\b');
				--cursor_pos;
				}
			}
		else if (c == RIGHT_ARROW)
			cursor_right();
		else if (c == C_LEFT_ARROW) {
			while (cursor_pos > 0 && str[cursor_pos-1] == ' ') {
				(*sys_putchar)('\b');
				--cursor_pos;
			}
			while (cursor_pos > 0 && str[cursor_pos-1] != ' ') {
				(*sys_putchar)('\b');
				--cursor_pos;
			}
		} else if (c == C_RIGHT_ARROW) {
			while (cursor_pos < num_got &&
			       str[cursor_pos] != ' ')
				cursor_right();
			while (cursor_pos < num_got &&
			       str[cursor_pos] == ' ')
				cursor_right();
		} else if (c == PAGE_UP) {
			for (i = curptr; (i = RECSUB(i-1)) != curptr;) {
				if (memcmp (str, lastlines[i], cursor_pos))
					continue;
				c = cursor_pos;
				clear_line();
				strcpy(str, lastlines[i]);
				if ((num_got = strlen(str)) > 0) {
					curptr = last_recalled = i;
					put_str(str);
					cursor_pos = num_got;
					while (cursor_pos > c)
						cursor_left ();
				}
				c = ~PAGE_UP;
				break;
			}
			if (c == PAGE_UP)
				putch ('\a');
		} else if (' ' <= c && c < 0x7f && num_got < len - 1)
			{
			if (insert_mode)
				{
				/* Move right, all the characters
				 * to the right of cursor_pos.
				 */
				for (i = num_got; i > cursor_pos; --i)
					str[i] = str[i - 1];
				str[cursor_pos] = (char)c;
				for (i = cursor_pos; i <= num_got; ++i)
					(*sys_putchar)(str[i]);
				for (i = cursor_pos; i < num_got; ++i)
					(*sys_putchar)('\b');
				++num_got;
				++cursor_pos;
				}
			else	/* insert is off, use overtype mode */
				{
				str[cursor_pos] = (char)c;
				(*sys_putchar)(c);
				if (cursor_pos == num_got)
					++num_got;
				++cursor_pos;
				}
			}
		}

	str[num_got] = '\0';
	(*sys_putchar)('\n');

	/* If this line is non-empty, and different
	 * from the last one accepted, store it into
	 * the recall buffer.
	 */
	if (num_got >= shortest_saved && strcmp(str, lastlines[lastptr]) &&
	    (last_recalled < 0 || strcmp(str, lastlines[last_recalled]))) {
		lastptr = RECSUB(lastptr + 1);
		strcpy(lastlines[lastptr], str);
	}
}

/*
 * Move the cursor right one position, by echoing the
 * character it's currently over.
 * Return 1-OK, 0-can't move.
 */
static int cursor_right( void )
	{
	if (cursor_pos < num_got)
		{
		(*sys_putchar)(str_ptr[cursor_pos]);
		++cursor_pos;
		return (1);
		}
	return (0);
	}


/*
 * Move the cursor left one position, by echoing
 * a backspace.  Return 1-OK, 0-can't move.
 */
static int cursor_left( void )
	{
	if (cursor_pos > 0)
		{
		(*sys_putchar)('\b');
		--cursor_pos;
		return (1);
		}
	return (0);
	}


/*
 * Erase all characters on the current line.
 */
static void
clear_line( void )
{
	int	i;

	while (cursor_left())		/* move to begining of line */
		;

	for (i = 0; i < num_got; ++i)
		(*sys_putchar)(' ');
	for (i = 0; i < num_got; ++i)
		(*sys_putchar)('\b');
	cursor_pos = num_got = 0;
}

static int
get_push_back( void )
{
	if (push_back_index > 0)
		return (push_back[--push_back_index]);
	return (0);
}


/*
 * Get a character, with escape processing.
 * Handles special sequences like "ESC [ A" for up-arrow.
 * This function would need to be modified to handle
 * keyboards that are neither PC's nor VT-100's.
 */
static int
get_char_esc( void )
{
int	ch;
clock_t	t;

	if (push_back_index > 0)
		return (push_back[--push_back_index]);

	ch = (*sys_getchar)();
	if (ch != ESC_KEY)
		return (ch);

	t = clock ();			/* allow ESC as char */
	while (clock () - t < CLK_TCK/10)	/* wait 100ms */
		;
	if (!kbhit ())
		return (ch);

	ch = (*sys_getchar)();
	if (ch != '[') {
		push_back[push_back_index++] = ch;
		return (ESC_KEY);
	}

	ch = (*sys_getchar)();
	if (ch == 'A')
		return (UP_ARROW);	/* was ESC [ A */
	else if (ch == 'B')
		return (DOWN_ARROW);	/* was ESC [ B */
	else if (ch == 'C')
		return (RIGHT_ARROW);	/* was ESC [ C */
	else if (ch == 'D')
		return (LEFT_ARROW);	/* was ESC [ D */
	else {
		push_back[push_back_index++] = '[';
		push_back[push_back_index++] = ch;
		return (ESC_KEY);
	}
}


/*
 * Put a string to (*sys_putchar)().
 */
static void
put_str( char far *str )
{
	while (*str != '\0')
		(*sys_putchar)(*str++);
}


/*********************************************************
 * The following two routines will need to be changed,
 * in order to use get_str() in a different environment.
 *********************************************************/

/*
 * Put a character to the output device.
 * Expand \n to \r\n.
 */
static void far
sys_put( int ch )
{

	putchar(ch);
	if (ch == '\n')
		putchar('\r');
}


/*
 * Get a character from the input device.
 * Use the BIOS call so we can detect arrow keys.
 */
static int far
sys_get( void )
{
int	ch;

	ch = getch () & 0x00ff;		/* wait and get a key */
	if (ch == 0)
		ch = (getch () & 0x00ff) << 8;
	return (ch);
}


/*
 * Set cursor shape to indicate inser/overtype mode.
 */
static void far
sys_show( int insert_mode )
{
	union REGS	inregs, outregs;

	inregs.h.ah = 1;
	inregs.h.ch = (unsigned char)(insert_mode ? 10 : 12);
	inregs.h.cl = 13;
	(void) int86 (0x10, &inregs, &outregs);
}
