/* --------------------------------- omk82.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* object: MK82 bomb.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "plane.h"

extern int FAR
BombSpeed (OBJECT *p, VECT V)
{
	Vcopy (V, p->V);
	return (iabs(p->speed));
}

extern int FAR
BombIP (LVECT R, VECT V, long tz, LVECT IP)
{
	int	v, dt;
	long	z;

	z = R[Z]-tz;
	v = V[Z];
	dt = v + SQRT (v*(long)v + 2L*GACC*z);

	IP[X] = R[X] + dt*(long)V[X]/GACC;
	IP[Y] = R[Y] + dt*(long)V[Y]/GACC;
	IP[Z] = tz;
	return (muldiv (dt, 100, GACC));
}

extern void FAR
BombToGnd (OBJECT *p)
{
	int	v, a;

	v = TADJ (p->V[Z]);
	if (v) {
		a = fdiv ((int)p->R[Z], v);
		p->R[X] -= fmul (a, TADJ (p->V[X]));
		p->R[Y] -= fmul (a, TADJ (p->V[Y]));
	}
	p->R[Z] = 0;
}

#define	EX_V	(300*VONE)
#define	EX_A	(VD360*4)

static int FAR
BombExplode (OBJECT *p)
{
	int	i;
	OBJECT	*b;

	st.owner = p;
	for (i = 0; i < 16; ++i) {
		b = create_object (O_BROKEN, 1);
		if (!b)
			break;
		b->flags |= F_HIT;
		b->V[X] = rand()%(EX_V+1) - EX_V/2;
		b->V[Y] = rand()%(EX_V+1) - EX_V/2;
		b->V[Z] = rand()%(EX_V+1);
		b->speed = ihypot3d (b->V);
		b->da[X] = rand()%(EX_A+1) - EX_A/2;
		b->da[Y] = rand()%(EX_A+1) - EX_A/2;
		b->da[Z] = rand()%(EX_A+1) - EX_A/2;
	}
	create_object (O_CRATER, 1);
	if (st.quiet)
		Snd->Beep (1414, 50);
	return (i);
}

static SHAPE shape_mk82 = {
	0,
	0,
	SH_G|SH_HIT|SH_LOCALSIM,
	227*1000L,	/* weight */
	FONE		/* drag */
};

static int FAR
init_mk82 (BODY *b)
{
	if (shape_read (&shape_mk82, "mk82.vxx"))
		return (1);
	return (0);
}

static void FAR
term_mk82 (BODY *b)
{
	shape_free (shape_mk82.v);
}

static int FAR
create_mk82 (OBJECT *p)
{
	OBJECT	*owner;
	E_BOMB	*b;
	long	tz;

	if (!(owner = st.owner))
		return (1);

	p->color = st.white;
	p->time = FOREVER;
	p->damage = 1;
	p->damaging = 10;
	p->owner = owner;
	p->ownerid = owner->id;
	if (ET_PLANE == owner->e_type)
		++EE(owner)->misc[11];			/* count bomb */

	LVcopy (p->R, owner->R);
	Vcopy (p->V, owner->V);
	Vcopy (p->a, owner->a);
	Mcopy (p->T, owner->T);
	p->speed = owner->speed;

	if (owner->flags & F_IMPORTED) {
		p->flags |= F_IMPORTED;
		p->rplayer = owner->rplayer;
	} else if (NEW (b)) {
		p->e_type = ET_BOMB;
		EBM(p) = b;

		if (ET_PLANE == owner->e_type &&
		    EE(owner)->target &&
		    EE(owner)->target->id == EE(owner)->tid)
			tz = EE(owner)->target->R[Z];
		else
			tz = 0L;
		BombIP (p->R, p->V, tz, b->IP);
	}
	p->flags |= F_VISIBLE|F_DONE;

	if (st.quiet)
		Snd->Beep (1000, 10);

	if (owner == CC)
		++st.nbullets;
	return (0);
}

static void FAR
delete_mk82 (OBJECT *p)
{
	OBJECT	*owner;

	if ((owner = p->owner) && owner->id == p->ownerid)
		if (ET_PLANE == owner->e_type)
			--EE(owner)->misc[11];	/* count bomb */
}

static void FAR
dynamics_mk82 (OBJECT *p, int interval)
{
	VECT	AA;

	if (p->flags & F_HIT) {
		body_dynamics (p, interval);
		return;
	}
	if (p->R[Z] <= 0) {
		BombToGnd (p);
		BombExplode (p);
		p->flags |= F_DEL|F_MOD;
		return;
	}
	AA[X] = AA[Y] = 0;
	AA[Z] = -GACC;
#if 0						/* testing */
{
	int	vmag, q, drag;

	vmag = iabs(p->speed/VONE);
	q = muldiv (vmag, vmag, 128);
	drag = fmul (q, FCON(0.01));
	acc = muldiv (drag, 128, 250);		/* weight = 250 Kg */

	AA[X] -= muldiv (p->V[X], p->speed, acc);
	AA[Y] -= muldiv (p->V[Y], p->speed, acc);
	AA[Z] -= muldiv (p->V[Z], p->speed, acc);
}
#else
#endif

	p->V[X] += TADJ(AA[X]);
	p->V[Y] += TADJ(AA[Y]);
	p->V[Z] += TADJ(AA[Z]);

#if 0
{
	ANGLE	daX;

	p->da[X] = v[Z]/64;

	daX = TADJ(p->da[X])*VONE;

	Mident (p->T);
	Mrotx (p->T, daX);
	Mroty (p->T, p->a[Y]);
	Mrotx (p->T, p->a[X]);
	Mrotz (p->T, p->a[Z]);
	Mangles1 (p->T, p->a);
}
#endif
	p->speed = ihypot3d (p->V);
}

BODY BoMK82 = {
	0,
	0,
	"MK82",
	&shape_mk82,
	init_mk82,
	term_mk82,
	create_mk82,
	delete_mk82,
	dynamics_mk82,
	body_hit
};
