/* --------------------------------- packet.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for packet level exchanges (low level). It uses a packet driver
 * as the communications medium. At the moment it expects etherslip but it
 * is rather general.
 * Here we assume only one handle is used and ANYTYPE was selected. Up to 4
 * drivers can be handled, all call the same fuction with the first argument
 * 'dev' indicating which one it is.
 * Options:
 *  0 interrupt number (usualy 0x65)
 *  1 packet type (default is TYPE_FLY8 which is usualy 0xf8f8)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>

#include "fly.h"
#include "pktdrvr.h"

#define	TYPE_FLY8	0xf8f8

#define	PKSSIZE		1024		/* packet-driver's stack size */

#define	LDATA		(PACKHEADLEN+ONEPACKLEN)
#define	LDATAPAK	(PACKHEADLEN+PAKPACKLEN)
#define PHLEN		(2*LADDRESS+2+2)
#define PHEAD		(pack->data+PACKHEADLEN-PHLEN)

typedef struct port PORT;
struct port {
	int	flags;
#define POF_ON		0x0001
	void (INTERRUPT FAR *pkint) (void);
	int	intno;			/* packet-driver interrupt */
	int	netport;		/* back-pointer */
	Uchar	address[LADDRESS];	/* my MAC address */
	int	handle;			/* packet-driver handle */
	char	packet_type[2];		/* eth_type for fly8 */
	PACKET	*pack;			/* packet being received */
	int	*stack;			/* packet-driver stack */
	int	version;
	int	class;
	int	type;
	int	number;
	char	*name;
	int	basic;
};

static PORT	ports[] = {
	{0, pkint0},
	{0, pkint1},
	{0, pkint2},
	{0, pkint3}
};
#define	NDEV	(sizeof(ports)/sizeof(ports[0]))

static int	nports = 0;		/* number of active ports */

static void FAR
receiver (int dev, Ushort di, Ushort si, Ushort bp, Ushort dx, Ushort cx,
	Ushort bx, Ushort ax, Ushort ds, Ushort es)
{
	PORT	*port;
	PACKET	*pack;
	char	*buff;
	int	len;

	switch (ax) {
	case 0:				/* Space allocate call */
		es = di = 0;
		if (dev < 0 || dev >= NDEV || cx > LDATAPAK)
			goto badret;
		port = &ports[dev];
		if (!(port->flags & POF_ON))
			goto badret;
		if (port->pack) {		/* stray packet? */
			/* stats... */
			packet_del (port->pack);
			port->pack = 0;
		}
		len = (cx > LDATA) ? ((cx+PAKGRAIN-1)&~(PAKGRAIN-1)) : 0;
		if (!(pack = packet_new (len)))
			goto badret;
		port->pack = pack;
		pack->length = cx;
		buff = PHEAD;
		es = FP_SEG (buff);
		di = FP_OFF (buff);
		break;
	case 1:				/* Packet complete call */
		if (dev < 0 || dev >= NDEV)
			goto badret;
		port = &ports[dev];
		if (!(port->flags & POF_ON))
			goto badret;
		if (!(pack = port->pack))
			goto badret;
		pack->netport = port->netport;
		len  = (PHEAD[2*LADDRESS+2] << 8) + PHEAD[2*LADDRESS+3];
		if (len < 3 || (Uint)len > pack->size) {
			packet_del (pack);
			goto badret;
		}
		pack->length = len;
		pack->address = PHEAD+LADDRESS;		/* from */

		if (packet_deliver (pack))
			packet_del (pack);
		port->pack = 0;
		break;
	default:
badret:
		++st.stats[5];
		break;
	}
}

static int FAR
find_driver (void)
{
	int	i;

	for (i = 0x0060; i < 0x0080; ++i)
		if (test_for_pd (i))
			return (i);
	return (-1);
}

static int FAR
get_options (PORT *port, char *options)
{
	long	l;

	port->intno = (int) get_inarg (options, 0);
	if (-1L == (l = get_narg (options, "type=")))
		l = TYPE_FLY8;
	port->packet_type[0] = (char)(l>>8);
	port->packet_type[1] = (char)(l   );

	return (0);
}

static int FAR
com_init (NETPORT *np, char *options)
{
	int	portno, rc;
	PORT	*port;
	char	msg[80], hex[10];

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV) {
		MsgEPrintf (-100, "%s.%c: bad port",
			np->NetDriver->name, np->unit);
		return (1);
	}
	port = &ports[portno];
	if (port->flags & POF_ON) {
		MsgEPrintf (-100, "%s.%c: already on",
			np->NetDriver->name, np->unit);
		return (1);
	}

	if (get_options (port, options))
		return (1);

	if (-1 == port->intno)
		port->intno = find_driver ();
	else if (!test_for_pd (port->intno))
		port->intno = -1;
	if (-1 == port->intno) {
		MsgEPrintf (-100, "%s.%c: no driver",
			np->NetDriver->name, np->unit);
		return (1);
	}
	MsgPrintf (-100, "Intno 0x%x", port->intno);

	if (!(port->stack = (int *)xcalloc (PKSSIZE, sizeof (int)))) {
		MsgEPrintf (-100, "%s.%c: no mem",
			np->NetDriver->name, np->unit);
		return (1);
	}
	pkset (portno, receiver, &port->stack[PKSSIZE]);

	port->handle = access_type (port->intno, CL_ETHERNET, ANYTYPE, 0,
		(char FAR *)port->packet_type, 2, port->pkint);
	if (-1 == port->handle) {
		MsgEPrintf (-100, "%s.%c: no handle",
			np->NetDriver->name, np->unit);
		port->stack = xfree (port->stack);
		return (1);
	}
	if (driver_info (port->intno, port->handle, &port->version,
			&port->class, &port->type, &port->number, &port->name,
			&port->basic)) {
		port->basic = 1;	/* what else ? */
	}
	MsgPrintf (-100, "Basic 0x%x", port->basic);

	port->flags |= POF_ON;
	port->netport = np->netport;
	port->pack = 0;
	++nports;

	rc = get_address (port->intno, port->handle, port->address,
						sizeof (port->address));
	strcpy (msg, "MAC ");
	if (rc) {
		strcat (msg, "failed ");
		sprintf (hex, "%0x", Derr);
		strcat (msg, hex);
		memset (port->address, 0, sizeof (port->address));
	} else {
		for (rc = 0; rc < LADDRESS; ++rc) {
			sprintf (hex, "%02x", port->address[rc]);
			strcat (msg, hex);
		}
	}
	MsgPrintf (-100, msg);

	return (0);
}

static void FAR
com_term (NETPORT *np)
{
	int	portno;
	PORT	*port;

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return;
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return;
	release_type (port->intno, port->handle);
	if (port->pack) {
		packet_del (port->pack);
		port->pack = 0;
	}
	port->flags = 0;
	port->stack = xfree (port->stack);
}

static int FAR
com_send (NETPORT *np, PACKET *pack)
{
	int	portno;
	PORT	*port;

	if (0 == pack)
		return (0);

	portno = np->unit-'1';
	if (portno < 0 || portno >= NDEV)
		return (1);
	port = &ports[portno];
	if (!(port->flags & POF_ON))
		return (1);

	if (pack->address) {
		memcpy (PHEAD, pack->address, LADDRESS);	/* to */
		pack->address = 0;
	} else
		memset (PHEAD, 0xff, LADDRESS);
	memcpy (PHEAD+LADDRESS, port->address, LADDRESS);	/* from */
	memcpy (PHEAD+2*LADDRESS, port->packet_type, 2);	/* type */
	PHEAD[2*LADDRESS+2] = (Uchar)(0x0ff&(pack->length >> 8));
	PHEAD[2*LADDRESS+3] = (Uchar)(0x0ff&(pack->length));
	send_pkt (port->intno, PHEAD, PHLEN + pack->length);

	return (0);
}

static void FAR
com_receive (NETPORT *np)
{
	return;
}

extern struct NetDriver NetPack = {
	"PKT",
	0,
	com_init,
	com_term,
	com_send,
	com_receive
};
