/* --------------------------------- macros.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handle keyboard macros.
 *
 * This module manages the input data stream. KeysBuffer hold keystrokes that
 * are processed before the keyboard device is interrogated for more. It is
 * filled with data by calling mac_key().
 * Another subject is key macros. Some keys are bound to a definition which
 * is a list of keystokes. When such a character is encountered it is this
 * list that gets placed in the KeysBuffer.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

#define	KEYSBUFFERSIZE	1024
#define	KEYSBUFFEREND	(KeysBuffer+KEYSBUFFERSIZE)
#define	NMACROS		256
#define	RECBUFFERSIZE	1024

#define	MACRO	struct Macro
MACRO {
	int	name;
	int	len;
	int	*def;
};

static int	*KeysBuffer = 0;	/* circular buffer control */
static int	*KeysHead = 0;
static int	*KeysTail = 0;
static int	KeysCount = 0;
static int	*RecordBuffer = 0;	/* macro recording buffer */
static int	*RecordPtr = 0;		/* position in RecordBuffer */
static int	Quoting = 0;		/* mread ignores special keys */
static MACRO	*Macros = 0;		/* macro definitions */
static MACRO	*Recording = 0;		/* Macro being recorded */
static HMSG	*RecordingMsg = 0;

static int	FAR record_char (int ch);
static int	FAR mac_expand (int name);
static int	FAR mac_hot (int name);
static int	FAR mac_key (int key);
static int	FAR mac_record (void);
static int	FAR mac_play (void);
static int	FAR mac_write (char *mname);
static int	FAR mac_read (char *mname);

extern int FAR
mread (void)				/* non-blocking read */
{
	int	ch;

ReRead:
	if (KeysCount > 0) {
		--KeysCount;
		ch = *KeysHead++;
		if (KeysHead == KEYSBUFFEREND)
			KeysHead = KeysBuffer;
	} else {
		for (;;) {
			ch = Kbd->Read ();
			if (-1 != ch && Recording) {
				ch = record_char (ch);
				if (ch == -1)
					continue;
			}
			break;
		}
	}

	if (ch != -1) {
		if (mac_hot (ch))
			goto ReRead;

	}

	if (!Quoting) {
		if (KF_MACRECORD == ch) {
			mac_record ();
			goto ReRead;
		}

		if (KF_MACPLAY == ch) {
			mac_play ();
			goto ReRead;
		}
	}

	return (ch);
}

extern int FAR
mgetch (void)				/* blocking read */
{
	int	ch;

	while (-1 == (ch = mread ()))
		sys_poll ();
	return (ch);
}

extern int FAR
mac_interpret (int *keys, int len)
{
	int	key;

	while (len-- > 0) {
		key = *keys++;
		if (mac_hot (key))
			continue;
		if (KF_MACPLAY == key) {
			if (len-- <= 0)
				return (1);
			if (mac_expand (*keys++))
				continue;
			MsgPrintf (10, "Macro undefined");
		}
		if (mac_key (key))
			return (1);
	}
	return (0);
}

extern void FAR
mac_flush (void)
{
	KeysHead = KeysTail = KeysBuffer;
	KeysCount = 0;
	Recording = 0;
}

extern int FAR
mac_init (void)
{
	int	i;
	MACRO	*m;

	KeysBuffer = (int *)xcalloc (KEYSBUFFERSIZE, sizeof (int));
	if (!KeysBuffer) {
		MsgEPrintf (-50, "macros: no mem (1)");
		return (1);
	}
	RecordBuffer = (int *)xcalloc (RECBUFFERSIZE, sizeof (int));
	if (!RecordBuffer) {
		MsgEPrintf (-50, "macros: no mem (2)");
		return (1);
	}

	mac_flush ();

	Macros = (MACRO *)xcalloc (NMACROS, sizeof (MACRO));
	if (!Macros) {
		MsgEPrintf (-50, "macros: no mem (3)");
		return (1);
	}

	for (m = Macros, i = 0; i < NMACROS; ++i, ++m) {
		m->name = -1;
		m->len  = 0;
		m->def  = 0;
	}

	mac_read (st.mname);

	return (0);
}

extern void FAR
mac_term (void)
{
	int	i;
	MACRO	*m;

	mac_flush ();
	KeysHead = KeysTail = 0;

	KeysBuffer = xfree (KeysBuffer);
	RecordBuffer = xfree (RecordBuffer);

	if (Macros) {
		mac_write (st.mname);

		for (m = Macros, i = 0; i < NMACROS; ++i, ++m) {
			m->name = -1;
			m->len  = 0;
			m->def  = xfree (m->def);
		}

		Macros = xfree (Macros);
	}

	return;
}

static int FAR
mac_key (int key)
{
	if (KeysBuffer && KeysCount < KEYSBUFFERSIZE) {
		++KeysCount;
		*KeysTail++ = key;
		if (KeysTail == KEYSBUFFEREND)
			KeysTail = KeysBuffer;
		return (0);
	} else
		return (1);
}

static MACRO * FAR
mac_find (int name)
{
	int	i;
	MACRO	*m;

	if (!Macros)
		return (0);

	for (m = Macros, i = 0; i < NMACROS; ++i, ++m)
		if (m->name == name)
			return (m);
	return (0);
}

static void FAR
record_add (int ch)
{
	if (!RecordBuffer)
		MsgEPrintf (50, "No Recording");
	else if (RecordPtr == RecordBuffer+RECBUFFERSIZE)
		MsgEPrintf (50, "Recording overflow");
	else
		*RecordPtr++ = ch;
}

static int FAR
record_char (int ch)
/*
 * return (-1): control char used up, re-read.
 * else		use this char.
*/
{
	MACRO	*m;
	int	new, l;
	HMSG	*p;

	if (ch == KF_MACPLAY) {
		m = Recording;
		Recording = 0;

		p = MsgWPrintf (0, "Abort/Cont/Quote?");
		ch = mgetch ();
		msg_del (p);
		if (ch == KF_MACPLAY) {			/* abort */
			if (m->def == 0)
				m->name = -1;
			MsgPrintf (50, "Macro aborted");
			msg_del (RecordingMsg);
			--Quoting;
			return (-1);
		} else if (ch == KF_MACRECORD) {	/* cont */
			Recording = m;
			return (-1);
		}
		Recording = m;
		record_add (KF_MACPLAY);
		record_add (ch);
		if (!mac_expand (ch)) {
			MsgPrintf (10, "Macro undefined");
			return (ch);
		}
		return (-1);
	} else if (ch == KF_MACRECORD) {
		msg_del (RecordingMsg);
		RecordingMsg = 0;
		--Quoting;

		m = Recording;
		Recording = 0;

		new = (m->def == 0);
		l = RecordPtr - RecordBuffer;

		if (new && !l) {			/* aborted */
			MsgPrintf (50, "Macro aborted");
			m->name = -1;
			return (-1);
		}
		m->len = 0;
		if (!new)				/* delete old */
			m->def = xfree (m->def);

		if (!l) {				/* just deleting */
			MsgPrintf (50, "Macro deleted");
			m->name = -1;
			return (-1);
		}

		m->def = (int *)xcalloc (l, sizeof (int));
		if (!m->def) {
			MsgPrintf (100, "macros: no mem (4)");
			m->name = -1;
			return (-1);
		}
		memcpy (m->def, RecordBuffer, l * sizeof (int));
		m->len = l;

		MsgPrintf (50, "Macro %sdefined", new ? "" : "re");
		return (-1);
	}

	record_add (ch);
	return (ch);
}

static int FAR
mac_record (void)
{
	int	name;
	MACRO	*m;
	HMSG	*p;

	if (!Macros || !RecordBuffer) {
		MsgPrintf (50, "No Macros");
		return (-1);
	}

	p = MsgWPrintf (0, "Macro Key ?");
	name = Kbd->Getch ();			/* get macro name */
	msg_del (p);
	if (name == KF_MACRECORD || name == KF_MACPLAY) {
		MsgPrintf (50, "Macro aborted");
		return (0);
	}

	m = mac_find (name);
	if (!m) {
		m = mac_find (-1);		/* new macro */
		if (!m) {
			MsgWPrintf (100, "All macros used");
			return (1);		/* all macros defined */
		}
		m->name = name;
		m->len = 0;
		m->def = 0;
	} else
		MsgWPrintf (100, "Warn ReDefining");

	Recording = m;
	RecordPtr = RecordBuffer;
	RecordingMsg = MsgWPrintf (0, "Recording...");
	++Quoting;

	return (0);
}

static int FAR
mac_hot (int name)
/*
 * return 1 if hot (and then expand it too).
*/
{
	if (name >= 0 && name <= 26) {
		if (mac_expand (name))
			return (1);
	}
	return (0);
}

static int FAR
mac_expand (int name)
/*
 * return 1 if a macro (and then expand it too).
*/
{
	MACRO		*m;
	static int	depth = 0;

	m = mac_find (name);
	if (!m || !m->def)
		return (0);

	if (depth > 16) {
		MsgEPrintf (10, "Macro nesting > 16");
		return (1);
	}
	++depth;
	mac_interpret (m->def, m->len);
#if 0
	for (p = m->def, i = m->len; i-- > 0;) {
		key = *p++;
		if (mac_hot (key))
			continue;
		if (KF_MACPLAY == key) {
			--i;
			if (mac_expand (*p++))
				continue;
			MsgPrintf (10, "Macro undefined");
		}
		if (mac_key (key))
			return (1);
	}
#endif
	--depth;

	return (1);
}

static int FAR
mac_play (void)
{
	int	name;
	HMSG	*p;

	if (!Macros || !RecordBuffer) {
		MsgPrintf (100, "No Macros");
		return (-1);
	}

	p = MsgWPrintf (0, "Play/Abort?");
	++Quoting;
	name = mgetch ();			/* get macro name */
	--Quoting;
	msg_del (p);
	if (name == KF_MACRECORD || name == KF_MACPLAY) {
		MsgPrintf (100, "Macro aborted");
		return (-1);
	}

	if (mac_expand (name))			/* play */
		return (0);

	MsgPrintf (10, "Macro undefined");
	return (-1);
}

static int
write_short (int *i, FILE *f)
{
	int	n;

	n = *i;
	fputc (0x0ff&(n>>8), f);
	if (ferror (f))
		return (0);
	fputc (0x0ff&n, f);
	if (ferror (f))
		return (1);
	return (2);
}

static int
read_short (int *i, FILE *f)
{
	int	n;

	n = fgetc (f);
	if (ferror (f) || feof (f))
		return (0);
	n = (n<<8) + fgetc (f);
	if (ferror (f) || feof (f))
		return (1);
	*i = n;
	return (2);
}

static int FAR
mac_write (char *mname)
{
	int	i, j;
	MACRO	*m;
	FILE	*mac;

	if (!Macros || !mname)
		return (1);

	mac = fopen (mname, WBMODE);
	if (!mac) {
		MsgEPrintf (-50, "macros: open %s failed (1)", mname);
		return (1);
	}

	for (m = Macros, i = 0; i < NMACROS; ++i, ++m) {
		if (!m->def)
			continue;
		if (2 != write_short (&m->name, mac)) {
			MsgEPrintf (-50, "macros: write %s failed (1)", mname);
			goto ret;
		}
		if (2 != write_short (&m->len, mac)) {
			MsgEPrintf (-50, "macros: write %s failed (2)", mname);
			goto ret;
		}
		for (j = 0; j < m->len; ++j) {
			if (2 != write_short (m->def+j, mac)) {
				MsgEPrintf (-50, "macros: write %s failed (3)",
					mname);
				goto ret;
			}
		}
	}
ret:
	fclose (mac);
	return (0);
}

static int FAR
mac_read (char *mname)
{
	int	i, j, t;
	MACRO	*m;
	FILE	*mac;

	if (!Macros || !mname) {
		MsgPrintf (50, "No Macros", mname);
		return (1);
	}

	mac = fopen (mname, RBMODE);
	if (!mac) {
		MsgEPrintf (-50, "macros: open %s failed (2)", mname);
		return (1);
	}

	for (i = 0, m = Macros;; ++i, ++m) {
		if (2 != read_short (&t, mac))
			break;
		if (i >= NMACROS) {
			MsgEPrintf (50, "too many macros");
			goto ret;
		}
		m->name = t;
		if (2 != read_short (&m->len, mac)) {
			MsgEPrintf (-50, "macros: read %s failed (1)", mname);
			m->name = -1;
			goto ret;
		}
		m->def = (int *)xcalloc (m->len, sizeof (int));
		if (!m->def) {
			MsgPrintf (-50, "macros: no mem (5)");
			m->name = -1;
			goto ret;
		}
		for (j = 0; j < m->len; ++j) {
			if (2 != read_short (m->def+j, mac)) {
				MsgEPrintf (-50, "macros: read %s failed (2)",
					mname);
				m->name = -1;
				m->def = xfree (m->def);
				goto ret;
			}
		}
	}
	if (!feof (mac))
		MsgEPrintf (-50, "read %s failed (3)", mname);
ret:
	fclose (mac);
	return (0);
}
