/* --------------------------------- fly.h ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* structures used by the system.
*/

#include "config.h"
#include "shape.h"

#define rangeof(array)	(sizeof(array)/sizeof(array[0]))

#define	BUFLEN	100		/* number of points in a display list block */
 
typedef enum oname ONAME;
enum oname {O_GROUND, O_PLANE, O_BOX, O_RUNWAY, O_M61, O_TARGET, O_BROKEN,
	O_VIEWER, O_CHUTE, O_HOUSE, O_TOWER, O_LOW, O_GTARGET, O_MK82,
	O_CRATER, O_SMOKE,
O_INT};
#define	O_EXT	10

typedef unsigned char	Uchar;
typedef unsigned int	Uint;
typedef unsigned short	Ushort;
typedef unsigned long	Ulong;

typedef short	ANGLE;
typedef short	VECT[3];
typedef long	LVECT[3];
typedef ANGLE	AVECT[3];
typedef short	MAT[3][3];

#define X	0
#define Y	1
#define Z	2

#define	HEAD	0
#define	TAIL	1

#define QUEUE(q,m) \
	if ((q)[HEAD])			\
		(q)[TAIL]->next = (m);	\
	else				\
		(q)[HEAD] = (m);	\
	(q)[TAIL] = (m)

typedef struct vertex VERTEX;
typedef struct shape SHAPE;
typedef struct object OBJECT;
typedef struct bufline BUFLINE;
typedef struct buffer BUFFER;
typedef struct device DEVICE;
typedef struct pointer POINTER;
typedef struct screen SCREEN;
typedef struct view VIEW;
typedef struct window WINDOW;
typedef struct viewport VIEWPORT;
typedef struct player PLAYER;
typedef struct packet PACKET;
typedef struct netport NETPORT;
typedef struct msg HMSG;
typedef struct hdd HDD;
typedef struct pid PID;

struct vertex {
	VECT	V;
	short	flags;
};

struct shape {
	VERTEX	*v;
	Ushort	extent;
	short	flags;
#define SH_G		0x0001
#define SH_HIT		0x0002
#define SH_BEHIT	0x0004
#define SH_LOCALSIM	0x0008
#define SH_FINE		0x0010
	long	weight;		/* grams */
	short	drag;		/* drag factor */
};

#define	SH(p)		(st.bodies[p->name]->shape)
#define TITLE(p)	(st.bodies[p->name]->title)

struct pid {
	long	Kp;
	long	Iband;
	long	Ki;
	long	Dband;
	long	Kd;
	long	factor;
	long	range;
	long	Pprev;
	long	I;
};

typedef struct e_imported E_IMPORT;
struct e_imported {
	Ulong	timeout;
	short	misc[5];
};

#define EIM(p)	(p->extra.e_imported)


typedef struct e_plane E_PLANE;
struct e_plane {
	struct e_parms	*parms;		/* plane parameters */
	short	flags;
#define	PF_AUTO		0x0001
#define	PF_ONGROUND	0x0002
#define PF_CHASE	0x0004
#define PF_KILL		0x0008
#define PF_INSTRUMENTS	0x0010
	short	hud;
#define	HUD_ON		0x0001
#define	HUD_LADDER	0x0002
#define	HUD_BIG		0x0004
#define	HUD_FINE	0x0008
#define	HUD_XFINE	0x0010
#define	HUD_PLUS	0x0020
#define	HUD_CURSOR	0x0040
#define	HUD_VV		0x0080
#define	HUD_TARGET	0x0100
#define	HUD_DATA	0x0200
#define	HUD_RETICLE	0x0400
#define	HUD_ROSS	0x0800
#define	HUD_INFO	0x1000
#define	HUD_INFOM	0x7000
#define	HUD_DEFAULT	(HUD_TARGET|HUD_RETICLE)
#define	HUD_FULLHEADING	0x8000
	short	hud1;
#define	HUD_TOP		0x0001
#define	HUD_LIMIT	0x0002
#define	HUD_CORNER	0x0004
#define	HUD_AALARM	0x0008
#define	HUD_VALARM	0x0010
#define	HUD_KNOTS	0x0020
#define	HUD_MISC	0x0040
#define	HUD_ACCVECT	0x0080
#define	HUD_TYPE	0x0100
#define	HUD_TYPES	0x0700
#define	HUD_CLASSIC	(0*HUD_TYPE)
#define	HUD_FA18	(1*HUD_TYPE)
#define	HUD_F16		(2*HUD_TYPE)
#define	HUD_F15		(3*HUD_TYPE)
#define	HUD_FLIR	(4*HUD_TYPE)
#define	HUD_PENDULUM	0x0800
#define	HUD_INAME	0x1000
#define	HUD_IDIST	0x2000
#define	HUD_THICK	0x4000
#define	HUD_BORDER	0x8000
	short	hud2;
#define	HUD_ILS		0x0001
#define	HUD_VW 		0x0002
#define	HUD_ALTITUDE	0x0004
#define	HUD_SPEED	0x0008
#define	HUD_HEADING	0x0010
#define	HUD_HIDETGT	0x0020
#define	HUD_NAV		0x0040
	short	radar;
#define	R_ON		0x0001
#define	R_LOCK		0x0002
#define	R_INTEL		0x0004
#define	R_INTELCC	0x0008
#define	R_MODE		0x0010
#define	R_MODES		0x0070
#define	R_SHOOT		0x0080
#define	R_SELECT3	0x0100
#define	R_SELECT20	0x0200
#define	R_SELECT5	0x0400
	short	equip;
#define	EQ_AIRBRAKE	0x0001
#define	EQ_GNDBRAKE	0x0002
#define	EQ_GEAR		0x0004
	short	hudarea;		/* degrees from center to edge */
	short	ils;			/* ils beacon id */
	short	tapelen;		/* speed/alt scale length */
	short	weapon;			/* weapon type */
#define	WE_M61		0x0001
#define	WE_MK82		0x0002
#define	N_WE		5
	long	fuel;			/* fuel left [*100] */
	short	stores[N_WE];		/* weapons available */
	short	throttle;		/* position [-100..100]*/
	short	elevators;		/* position [0..100] */
	short	ailerons;		/* position [-100..100] */
	short	rudder;			/* position [-70..70] */
	short	flaps;			/* position [0..max] */
	short	spoilers;		/* position [0..max] */
	short	airbrake;		/* position [0..100] */
	short	afterburner;		/* position [0..100] */
	short	thrust;			/* lb_thrust/10 */
	VECT	v;			/* velocity in plane coordinates */
	short	wing_stall;
	short	StFont;
	short	StFontSize;
	OBJECT	*target;		/* aqcuired target */
	long	tid;			/* target id */
	PID	*PIDthrottle;
	PID	*PIDpitch;
	PID	*PIDroll;
	VECT	tvold;			/* old average velocity */
	VECT	tvnew;			/* new average velocity */
	short	itvhist;		/* ring buffer index */
#define	TVHIST	8
	VECT	tvhist[TVHIST];		/* target velocity history */
	short	misc[20];		/* autopilot, history etc. */
#define EE(p)	(p->extra.e_plane)
#define	LIFETIME(p)	EE(p)->misc[0]	/* millisecs to next randomization */
#define	SPEED(p)	EE(p)->misc[1]	/* desired speed */
#define	HEADING(p)	EE(p)->misc[2]	/* desired heading */
#define	ALTITUDE(p)	EE(p)->misc[3]	/* desired altitude */
#define	NEWTGT(p)	EE(p)->misc[4]
};


typedef struct e_bomb E_BOMB;
struct e_bomb {
	LVECT	IP;
};

#define EBM(p)	(p->extra.e_bomb)


struct object {
	OBJECT		*next;
	OBJECT		*prev;
	union {
			E_IMPORT	*e_imported;
			E_PLANE		*e_plane;
			E_BOMB		*e_bomb;
	}		extra;
	OBJECT		*owner;
	PLAYER		*rplayer;
	POINTER		*pointer;
	VIEWPORT	*viewport;

	long		id;		/* local object id */
	long		rid;		/* remote object id */
	long		ownerid;	/* local object id of owner-object */
	Ulong		rtime;		/* remote time */
	LVECT		R;		/* position */
	VECT		V;		/* velocity */
	AVECT		a;		/* orientation */
	AVECT		da;		/* rotation rate */
	MAT		T;		/* obj->world xform matrix */
	short		name;
	short		color;
#define	FOREVER		0x7fff
	short		time;
#define	TIMEPSEC	100
	short		flags;
#define F_VISIBLE	0x0001		/* permanent: */
#define F_IMPORTED	0x0002
#define F_EXPORTED	0x0004
#define F_MAINT		0x0008
#define F_STEALTH	0x0010
#define F_CC		0x0020
#define F_LAND		0x0040		/* part of landscape */
#define F_ALIVE		0x0200		/* transitional: */
#define F_NEW		0x0400
#define F_DEL		0x0800
#define F_HIT		0x1000
#define F_KEEPNAV	0x2000
#define F_DONE		0x4000
#define F_MOD		0x8000
	short		gpflags;
#define GPF_PILOT	0x0001
	short		shflags;	/* copy of SH flags */
	short		speed;
	short		e_type;
#define ET_IMPORTED	0x0001
#define ET_PLANE	0x0002
#define ET_BOMB		0x0003
	short		score;
	short		damage;
	short		damaging;
	short		misc[5];
};


struct bufline {
	short	x;
	short	y;
	short	t;	/* color or (if -ve) type */
#define	T_MOVE		-1
#define	T_MOR		-2
#define	T_MXOR		-3
#define	T_MSET		-4
#define	T_ELLIPSE	-5
#define	T_NOP		-99
};

struct buffer {
	BUFFER		*next;
	BUFLINE		*p;		/* first free */
	BUFLINE		*end;		/* &first[size] */
	int		size;		/* allocated size */
	BUFLINE		first[1];	/* start of data array */
};

struct device {
	DEVICE	*next;
	char	*name;
	void	*pdevice;
	int	mode;		/* vga bios mode etc. */
	int	colors;		/* number of colors */
	int	minx;		/* physical */
	int	miny;		/* physical */
	int	sizex;		/* physical */
	int	sizey; 		/* physical */
	int	npages;		/* how many display pages */
	int	lengx;		/* millimeter */
	int	lengy;		/* millimeter */
	int	FontWidth;	/* pixels */	/* belongs in SCREEN! */
	int	FontHeight;	/* pixels */	/* ditto */
	int	sync;		/* 1 if video sync needed */
};

struct pointer {
	char	*name;
	short	flags;
#define PF_PRESENT	0x01
#define PF_INITED	0x02
	struct PtrDriver FAR *control;
	int	a[4];		/* analog -100...+100 */
	int	l[4];		/* last value of a[] */
	int	b[20];		/* button 0 and 1 and other virtual ones */
	int	c[4];		/* cal    -100...+100 */
#define	NOPTS	10
	int	opt[NOPTS];
};

struct screen {
	DEVICE	*device;
	void	*pscreen;
	int	minx;		/* physical */
	int	miny;		/* physical */
	int	sizex;		/* physical */
	int	sizey;		/* physical */
	Uint	FgColor;
	Uint	BgColor;
	Uint	BoColor;	/* border */
	int	FontWidth;	/* pixels */
	int	FontHeight;	/* pixels */
};

struct window {
	int	orgx;	/* all in normalized coordinates [0...1] */
	int	orgy;
	int	maxx;	/* left=orgx-maxx right =orgx+maxx */
	int	maxy;	/* top =orgy-maxy bottom=orgy+maxy */
};

struct viewport {
	int	x;	/* NDC normally 0.0 */
	int	y;	/* NDC normally 0.0 */
	int	z;	/* NDC zoom, normally 0.5, must >= maxx and maxy! */
	int	maxx;	/* NDC vp width normally 0.5 */
	int	maxy;	/* NDC vp height normally 0.5 */
			/* distz and shift in units of VONE*VONE */
	int	distz;	/* eye to vp in viewers coords */
	int	shift;	/* eye x shift (v coords) in stereo */
	int	eyex;	/* viewer coords: */
	int	eyey;	/* eye position relative to viewer object origin */
	int	eyez;
	ANGLE	rotx;	/* eye direction relative to viewer origin */
	ANGLE	roty;
	ANGLE	rotz;
	int	zoom;	/* zoom count */
};

struct	view {
	VIEWPORT	*viewport;	/* camera: world -> 2D */
	WINDOW		*window;	/* camera -> NDC window */
	SCREEN		*screen;	/* window -> physical screen */
};

typedef struct body BODY;
struct body {
	int	name;
	short	flags;
	char	*title;
	SHAPE	*shape;
	int	(FAR *init) (BODY *b);
	void	(FAR *term) (BODY *b);
	int	(FAR *create) (OBJECT *object);
	void	(FAR *delete) (OBJECT *object);
	void	(FAR *dynamics) (OBJECT *object, int interval);
	void	(FAR *hit) (OBJECT *object, int speed, int extent,
			int damaging);
};

#define	LADDRESS	6
#define	LNAME		20

struct player {
	PLAYER	*next;
	short	flags;
#define	RMT_ACTIVE	0x0001
#define	RMT_PENDREPLY	0x0002
#define	RMT_PENDBOSS	0x0004
#define	RMT_PENDCONFIRM	0x0008
#define	RMT_PEND	(RMT_PENDREPLY|RMT_PENDCONFIRM|RMT_PENDBOSS)
#define	RMT_PLAYING	0x0010
#define	RMT_RECEIVE	(RMT_PENDCONFIRM|RMT_PLAYING)
#define	RMT_SEND	(RMT_PLAYING)
#define	RMT_NOTIDLE	(RMT_ACTIVE|RMT_PEND|RMT_PLAYING)
	int	netport;
	Uchar	address[LADDRESS];
	char	name[LNAME];
	PACKET	*incoming;
	PACKET	*tail;
	Ulong	timeout;
	long	rtime;				/* RemoteTime-LocalTime */
	short	rtimeErr;			/* intergrated error */
};

struct netport {
	short			flags;
#define	NP_ON		0x0001
#define	NP_BCAST	0x0002
#define	NP_PACKING	0x0004
	int			netport;
	int			unit;
	int			nplayers;	/* playing users */
	struct NetDriver	*NetDriver;
	PACKET			*incoming[2];
	PACKET			*outgoing;	/* async send */
	PACKET			*outpak;	/* packed packet */
};

#define	PACKHEADLEN	(2*LADDRESS+2+2)	/* to, from, type, len */
#define	ONEPACKLEN	64
#define	PAKPACKLEN	512
#define	PAKGRAIN	64			/* must be power of 2 */

struct packet {
	PACKET	*next;		/* must be first! */
	short	flags;		/* running length if incomming */
	short	netport;
	short	length;		/* message length */
	Ushort	size;		/* data size */
	Uchar	*address;
	Uchar	*data;
	Ulong	arrived;	/* arrival time */
};

struct PtrDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (POINTER *p);
	void	(FAR *Term) (POINTER *p);
	int	(FAR *Cal) (POINTER *p);
	int	(FAR *Center) (POINTER *p);
	int	(FAR *Read) (POINTER *p);
	void	(FAR *Key) (POINTER *p, int key);
};

struct GrDriver {
	char	*name;
	short	flags;
	DEVICE	*devices;
	int	(FAR *Init) (DEVICE *dev);
	void	(FAR *Term) (DEVICE *dev);
	void	(FAR *OpenTextWindow) (SCREEN *scr);
	void	(FAR *CloseTextWindow) (SCREEN *scr);
	void 	(FAR *FontSet) (DEVICE *dev, char *text);
	void	(FAR *TextPut) (int c);
	void 	(FAR *TextChar) (int c);
	void	(FAR *TextColor) (Uint fg, Uint bg);
	int	(FAR *CursorMode) (int mode);
	void	(FAR *MoveTo) (int x1, int y1);
	void	(FAR *DrawTo) (int x2, int y2, Uint c);
	void	(FAR *SetVisual) (int page);
	void	(FAR *SetActive) (int page);
	void	(FAR *Clear) (SCREEN *scr);
	void	(FAR *SetTextPos) (int row, int col);
	void	(FAR *PushTextPos) (void);
	void	(FAR *PopTextPos) (void);
	void	(FAR *TextClear) (void);
	void	(FAR *WriteMode) (int mode);
	void	(FAR *SetPalette) (int index, long color);
	void	(FAR *Ellipse) (int x, int y, int rx, int ry, Uint c);
	void	(FAR *Sync) (void);
	void	(FAR *Flush) (void);
	void	(FAR *Shutters) (int eye);
};

struct SndDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (void);
	void	(FAR *Term) (void);
	void	(FAR *Poll) (void);
	int 	(FAR *Beep) (int f, int milli);
	int 	(FAR *Effect) (int eff);
	int 	(FAR *List) (int *list, int id);
};

struct TmDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (void);
	void	(FAR *Term) (void);
	Ulong	(FAR *Milli) (void);
	int	(FAR *Hires) (void);
	char *	(FAR *Ctime) (void);
};

struct KbdDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (void);
	void	(FAR *Term) (void);
	int	(FAR *Read) (void);
	int	(FAR *Getch) (void);
	int	(FAR *Wait) (void);
};

struct SysDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (void);
	void	(FAR *Term) (void);
	void	(FAR *Poll) (void);
	short	(FAR *Disable) (void);
	void	(FAR *Enable) (short i);
	void	(FAR *Shell) (void);
};

struct NetDriver {
	char	*name;
	short	flags;
	int	(FAR *Init) (NETPORT *port, char *options);
	void	(FAR *Term) (NETPORT *port);
	int	(FAR *Send) (NETPORT *port, PACKET *pack);
	int	(FAR *Receive) (NETPORT *port);
};

struct netname {
	struct netname	*next;
	char		*name;
};

struct msg {
	HMSG		*next;
	char		*text;
	Ulong		timeout;	/* time to delete */
	int		flags;
#define MSG_WARN	0x0001
#define MSG_ERR		0x0002
};

#define NBUFS		2		/* must be 2 for now */

struct hdd {
	short		type;
#define	HDD_FRONT	 0
#define	HDD_NONE	 1
#define	HDD_REAR	 2
#define	HDD_MAP		 3
#define	HDD_RADAR	 4
#define	HDD_TARGET	 5
#define	HDD_PAN		 6
#define	HDD_GAZE	 7
#define	HDD_CHASE	 8
#define	HDD_FOLLOW	 9
#define	HDD_HUD		10
#define	HDD_INFO	11
#define	HDD_MISC	12
#define	HDD_RIGHT	13
#define	HDD_LEFT	14
	Ushort		flags;
#define HDD_ON	0x0001
	VIEW		view;
	BUFFER		*bufs[NBUFS];	/* HDD window */
};
#define NVIEWERS	10

struct status {
	struct SysDriver	*system;
	struct TmDriver		*time;
	struct GrDriver		*graphics;
	struct SndDriver	*sound;
	struct KbdDriver	*keyboard;
	BODY		**bodies;
	char		*iname;			/* init file */
	char		*mname;			/* macros file */
	char		*fname;			/* font file */
	char		*vmdname;		/* video modes file */
	char		*lname;			/* log file */
	char		*fdir;			/* support files directory */
	char		*pname;			/* pointer name */
	char		*dname;			/* video device mode */
	char		*vname;			/* video device name */
	char		*ptype;			/* plane type */
	char		*dtype;			/* drone type */
	char		*options;		/* options for 'create_obj' */
	char		*nikname;
	struct netname	*netnames;		/* net drivers */
	Ulong		big_bang;		/* keep relative time */
	Ulong		present;
	Ulong		lasttime;
#define	TO_OBJECT	(4*1000L)
	Ulong		ObjectTimeout;
#define	TO_PLAYER	(8*1000L)
	Ulong		PlayerTimeout;
#define	TO_DRONE	(8*1000L)
	Ulong		DroneTime;		/* drone takeoff delay */
#define	TO_REFRESH	(TO_OBJECT/4)
	Ulong		RefreshTime;
	Ulong		ShutdownTime;
	Ulong		test_start;
	long		hud_color;
	long		mscore;
	Ulong		nbullets;
	Uint		ComVersion;
	Uint		nbuffers;
	Uint		maxbuffers;
	short		flags;
#define	SF_BLANKER	0x0001
#define	SF_LANDSCAPE	0x0002
#define	SF_SKY		0x0004
#define	SF_VERBOSE	0x0008
#define	SF_INTERACTIVE	0x0010
#define	SF_SIMULATING	0x0020
#define	SF_INITED	0x0040
#define	SF_PAUSED	0x0080
#define	SF_MODES	0x0100
#define	SF_HUD		0x0200
#define	SF_HELP		0x0400
#define	SF_NET 		0x0800
#define	SF_LISTS	(SF_MODES|SF_HUD|SF_HELP|SF_NET)
#define SF_DEBUG	0x1000
#define SF_FONT		0x2000
#define SF_CLEARED	0x4000
#define	SF_MAIN		0x8000
	short		flags1;
#define	SF_USEG		0x0001
/*#define SF_		0x0002*/
#define	SF_DBUFFERING	0x0004
#define	SF_TESTING	0x0008
#define	SF_STEREOREV	0x0010
#define	SF_EXTVIEW	0x0020
#define	SF_IDENT	0x0040
#define	SF_INFO		0x0080
#define	SF_SMOKE	0x0100
#define SF_TERM		0x0200
	int		ntargets;
	short		extview;
	short		info;
	int		stereo;		/* 0=mono, 1=stereoscopic 2=red/blue */
	int		big_screen;	/* window configuration */
	int		paralax;	/* stereo eye shift. MAX 100*VONE! */
	int		focus;		/* stereo eye focus distance */
	int		gap;		/* gap between stereo frames: 1/gap */
	int		quiet;
	short		network;
#define	NET_ON		0x0001
#define	NET_NOBCAST	0x0002
#define	NET_INITED	0x0004
#define	NET_AUTOACCEPT	0x0010
#define	NET_AUTODECLINE	0x0020
#define	NET_AUTOREPLY	(NET_AUTOACCEPT|NET_AUTODECLINE)
	int		elapsed;
	int		tfg;		/* text fg */
	int		tbg;		/* text bg */
	int		wfg;		/* watch fg */
	int		cfg;		/* cursor color */
	int		hfg;		/* hud color */
	int		hfgi;		/* hud color (intensified)*/
	OBJECT		*owner;
	int		video_time;
	BUFFER		*bufs[NBUFS];	/* main window */
	BUFFER		*buf[2];	/* current writable buffer */
	int		which_buffer;
	int		nobjects;
	long		object_id;
	OBJECT		*world[2];	/* all objects in world */
#define CO	st.world[0]
#define COT	st.world[1]
	OBJECT		*land[2];	/* all objects in landscape */
#define CL	st.land[0]
#define CLT	st.land[1]
	OBJECT		*viewer;	/* origin of view */
#define CV	st.viewer
	OBJECT		*control;	/* object under user control */
#define CC	st.control
	VIEW		view[1];	/* active view */
#define CVIEW	st.view
#define CP	st.view->viewport
#define CW	st.view->window
#define CS	st.view->screen
	SCREEN		*textscreen;	/* text screen info */
#define CT	st.textscreen
#define NHDD	10
	HDD		hdd[NHDD];	/* HDD windows */
	int		interval;
	int		dither;		/* rand () % 1000 */
	POINTER		*pointer;	/* kludge for oplane.c */
	int		StFont;
	int		StFontSize;
	int		landx;		/* land square: x and y */
	int		landy;
	int		gravity;	/* meters/sec/sec */
	HMSG		*msg;		/* message queue */
	int		drones;
	int		killers;
	int		SkyLines;
	int		black;
	int		white;
	int		red;
	int		blue;
	int		green;
	int		magenta;
	int		brown;
	int		gray;
	int		lblue;
	int		lred;
	int		hudlow;
	int		hudhigh;
	int		lgray;
	int		ground;
	char		filename[1024];	/* file name work area */
#define	NIND	20
	int		indicators[NIND];
	int		misc[20];
	long		stats[100];
};

#define Sys	st.system
#define Tm	st.time
#define Gr	st.graphics
#define Snd	st.sound
#define Kbd	st.keyboard

#define GACC	st.gravity

#include "extern.h"
