/*
    Play3voi.c

    This is a program to play uncompressed .wav files on the Tandy 
    1000/PCjr 3-voice chip from the DOS command line.  Syntax:

        PLAY3VOI [/An] [/Tm] <filename>

    If /An is given, n is the amplification factor to apply to the samples 
    in the file.  0 is the default and means no amplification.

    If /Tm is given, m is the translation table to use to convert unsigned 
    sound samples to attenuation register values for the 3-voice chip.  0 
    is the default and uses the nearest available approximation to the 
    sample.  1 is an alternate table that is "smoothed" to reduce static; 
    it is slightly less accurate than table 0.
*/

#include <stdio.h>      /* Standard I/O functions header file */
#include <string.h>     /* Standard string functions header file */
#include <dos.h>        /* Standard DOS functions header file */
#include <stdlib.h>     /* Standard library functions header file */
#include "play3voi.h"   /* Global constants and variables */
#include "chip3voi.h"   /* Routines for accessing the 3-voice chip */
#include "wav3voi.h"    /* Routines for reading in a .wav file */


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* Amplification factor to apply. */
int Amplification = 0;

    /* I/O port value translation table to use. */
int Translation = 0;

    /* File handle of sound file. */
FILE *FileHandle = NULL;

    /* Buffer for sound, and size of that buffer. */
char huge *SoundBuf = (char huge *) NULL;
unsigned long SoundBufSize = 0L;

    /* Number of bytes of sound in the sound buffer. */
unsigned long SoundLen = 0;

    /* Overflow flag.  Set to 1 if the file was too long to fit entirely in 
       the buffer. */
int TooLong = 0;

    /* Sampling rate of the file. */
unsigned SampRate = 11000;


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

static void Show_Syntax_Message( void );
static int Get_Amplification( char *cmdswitch );
static int Get_Translation( char *cmdswitch );
static int Allocate_Sound_Buffer( void );


/*************************************************************************/
/*                                                                       */
/* Show_Syntax_Message() function.  Displays the program's command-line  */
/* syntax.                                                               */
/*                                                                       */
/*************************************************************************/

static void Show_Syntax_Message( void )
{
    fputs( "Syntax:\n\n\tPLAY3VOI [/An] [/Tm] <filename>\n\n", stderr );
    fprintf( stderr,
"where n is the amplification factor (default 0, max %d), and m is the\n\
translation table to use (default 0, max %d).  See PLAY3VOI.DOC.\n",
        MAX_AMPLIFICATION, MAX_TRANSLATE );
}


/*************************************************************************/
/*                                                                       */
/* Get_Amplification() function.  This function takes a command-line     */
/* argument, i.e., "/An", where n is some number, and sets the global    */
/* Amplification variable according to n.  Returns 0 if successful, -1   */
/* if the command-line switch is invalid, 1 if cmdswitch is another      */
/* command-line switch.                                                  */
/*                                                                       */
/*************************************************************************/

int Get_Amplification( char *cmdswitch )
{
    long lamp;      /* amplification specified, as a long */
    char *endptr;   /* character at end of scan for strtol() */

    /* If the cmdswitch is less than 3 characters long, it's invalid. */
    if ( strlen( cmdswitch ) < 3 )
        return( -1 );

    /* If the first character of the cmdswitch is not "/" or "-", the 
       switch is invalid. */
    if ( cmdswitch[0] != '/' && cmdswitch[0] != '-' )
        return( -1 );

    /* If the second character of the cmdswitch is not "A" or "a", it's not 
       an amplification switch. */
    if ( cmdswitch[1] != 'A' && cmdswitch[1] != 'a' )
        return( 1 );

    /* Convert the amplification specified to a long.  Return error if 
       invalid. */
    lamp = strtol( &cmdswitch[2], &endptr, 10 );
    if ( *endptr != '\0' || lamp < 0L || lamp > (long) MAX_AMPLIFICATION )
        return( -1 );

    /* Set the amplification and return success. */
    Amplification = (int) lamp;
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* Get_Translation() function.  This function takes a command-line       */
/* argument, i.e., "/Tn", where n is some number, and sets the global    */
/* Translation variable according to n.  Returns 0 if successful, -1 if  */
/* the command-line switch is invalid, 1 if cmdswitch is another         */
/* command-line switch.                                                  */
/*                                                                       */
/*************************************************************************/

int Get_Translation( char *cmdswitch )
{
    long lamp;      /* translation specified, as a long */
    char *endptr;   /* character at end of scan for strtol() */

    /* If the cmdswitch is less than 3 characters long, it's invalid. */
    if ( strlen( cmdswitch ) < 3 )
        return( -1 );

    /* If the first character of the cmdswitch is not "/" or "-", the 
       switch is invalid. */
    if ( cmdswitch[0] != '/' && cmdswitch[0] != '-' )
        return( -1 );

    /* If the second character of the cmdswitch is not "T" or "t", it's not 
       a translation table switch. */
    if ( cmdswitch[1] != 'T' && cmdswitch[1] != 't' )
        return( 1 );

    /* Convert the translation table specified to a long.  Return error if 
       invalid. */
    lamp = strtol( &cmdswitch[2], &endptr, 10 );
    if ( *endptr != '\0' || lamp < 0L || lamp > (long) MAX_TRANSLATE )
        return( -1 );

    /* Set the translation table and return success. */
    Translation = (int) lamp;
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* Allocate_Sound_Buffer() function.  This function allocates a sound    */
/* buffer from DOS, saving its address and size in SoundBuf and          */
/* SoundBufSize, respectively.  Returns 0 if successful, -1 if not.      */
/*                                                                       */
/*************************************************************************/

static int Allocate_Sound_Buffer( void )
{
    unsigned segment;       /* segment address of buffer */
        /* Size of largest contiguous block of memory available from DOS, 
           in paragraphs. */
    unsigned maxsize;

    /* Determine the size of the largest available block of memory.  Under 
       DOS, this is done by requesting an impossibly large block.  DOS 
       returns the size of the largest block available. */
    _dos_allocmem( 0xffff, &maxsize );

    /* If there is no memory available, return -1. */
    if ( maxsize == 0 )
        return( -1 );

    /* Allocate a buffer of the maximum size.  Return -1 if unsuccessful. 
       */
    if ( _dos_allocmem( maxsize, &segment ) != 0 )
        return( -1 );

    /* We have a buffer allocated.  Set its address and size. */
    FP_SEG( SoundBuf ) = segment;
    FP_OFF( SoundBuf ) = 0;
    SoundBufSize = (unsigned long) maxsize << 4;

    /* Return success. */
    return( 0 );
}


/*************************************************************************/
/*                                                                       */
/* main() program.                                                       */
/*                                                                       */
/*************************************************************************/

int main( int argc, char *argv[] )
{
    int i;                  /* for looping over command-line switches */
    int code;               /* return code from switch parsers */
    char *filename;         /* sound file from command line */

    /* Detect the 3-voice chip.  Halt program with a message if not 
       present. */
    if ( Detect_3_Voice() == -1 )
    {
        fputs( "This program requires a Tandy 1000/PCjr 3-voice sound chip.\n",
            stderr );
        exit( 1 );
    }

    /* Detect the presence of an XT.  Halt program with a message if an AT. 
       */
    if ( Detect_XT() == -1 )
    {
        fputs( "This program only runs on XT-class systems.\n", stderr );
        exit( 1 );
    }

    /* If there is an invalid number if command-line arguments, display 
       syntax and exit. */
    if ( argc < 2 || argc > 4 )
    {
        Show_Syntax_Message();
        exit( 1 );
    }

    /* If an amplification and/or a translation table were specified on the 
       command line, get it/them.  If invalid, display syntax and exit. */
    for ( i = 1; i < argc-1; i++ )
    {
        /* Check if it's an amplification switch. */
        if ( (code = Get_Amplification( argv[i] )) == -1 )
        {
            Show_Syntax_Message();
            exit( 1 );
        }
        else if ( code == 0 )
            continue;

        /* Check if it's a translation table switch. */
        if ( (code = Get_Translation( argv[i] )) == -1 )
        {
            Show_Syntax_Message();
            exit( 1 );
        }
        else if ( code == 0 )
            continue;

        /* It's neither.  Display syntax and exit. */
        Show_Syntax_Message();
        exit( 1 );
    }

    /* Open the specified file.  Halt if unsuccessful. */
    filename = argv[argc-1];
    if ( (FileHandle = fopen( filename, "rb" )) == NULL )
    {
        fprintf( stderr, "Unable to open file \"%s\".\n", filename );
        exit( 1 );
    }

    /* Allocate a buffer for sounds.  Get the largest contiguous block 
       available from DOS. */
    if ( Allocate_Sound_Buffer() == -1 )
    {
        fputs( "Unable to allocate buffer for sound.\n", stderr );
        goto close_file;
    }

    /* Load the sound file into the buffer. */
    fputs( "Loading file ...", stderr );
    if ( Read_Wave_File() == -1 )
        goto free_buffer;

    /* If the file was too long to fit, tell the user. */
    if ( TooLong )
        fputs( "File too large - playing partial file.\n", stderr );

    /* Play the sound. */
    Play_Sound();

    /* Free the sound buffer. */
free_buffer:
    _dos_freemem( FP_SEG( SoundBuf ) );

    /* Close the input file. */
close_file:
    fclose( FileHandle );
}
