// #########################################################
// === File #7 of 14 : pfmeter2.h ==========================
// ------------------- C++ code : BEGIN --------------------

// ==============================================================
//
//  Copyright (c) 2002 by Alex Vinokur.  This work and all works
//  derived from it may be copied and modified without any
//  restrictions other than that a copy of this copyright notice
//  must be included in any copy of this work or any derived work.
//
// ==============================================================
// #########################################################
// ## mailto:alexvn@go.to, mailto:alexv@hitechclub.com
// ## http://up.to/alexv, http://go.to/alexv_math
// #########################################################
// ==============================================================


// ##############################################################
//
//  SOFTWARE : C/C++ Program Perfometer
//  FILE     : pfmeter2.h
//
//  DESCRIPTION :
//         Implementation of perfometer's common template classes
//
// ##############################################################



/////////////////////
#ifndef _PRFMETER2_H
#define _PRFMETER2_H
/////////////////////

//====================
#include "pfmeter.h"
//====================




//#########################################
//#
//#  Class-2 : ClassResultsTimer	
//#  	METHODS
//#
//#########################################

//=============================
// Constructor-0
template <typename T2>
ClassResultsTimer<T2>::ClassResultsTimer()
//=============================
{
  assert (0);
} // ClassResultsTimer


//=============================
// Constructor-1
template <typename T2>
ClassResultsTimer<T2>::ClassResultsTimer(
	const T2&	total_run_cost_i
	) :
      		total_run_cost_ (total_run_cost_i)
//=============================
{
} // ClassResultsTimer


//=============================
// Default Copy Constructor
// template <typename T2>
// ClassResultsTimer<T2>::ClassResultsTimer(const ClassResultsTimer& inst_i)
//=============================


//=============================
// Destructor
template <typename T2>
ClassResultsTimer<T2>::~ClassResultsTimer()
//=============================
{
} // ~ClassResultsTimer


//=============================
template <typename T2>
T2 ClassResultsTimer<T2>::get_total_run_cost() const
//=============================
{
 return total_run_cost_;

} // ~ClassResultsTimer


//#########################################
//#
//#  Class-3 : ClassResultsMeter	
//#  	METHODS
//#
//#########################################

//=============================
template <typename T1, typename T2, ResourceType T3>
class less_cost
{
public:
  bool operator()(
	const ClassResultsMeter<T1, T2, T3>& inst1_i,
	const ClassResultsMeter<T1, T2, T3>& inst2_i
	)
  {
    if (inst1_i.scaled_run_cost_ < inst2_i.scaled_run_cost_) return true;
    if (inst2_i.scaled_run_cost_ < inst1_i.scaled_run_cost_) return false;

    if (inst1_i.testno_ < inst2_i.testno_) return true;
    if (inst2_i.testno_ < inst1_i.testno_) return false;

    assert (0);
    return true;
  }
};


//=============================
// Constructor-0
template <typename T1, typename T2, ResourceType T3>
ClassResultsMeter<T1, T2, T3>::ClassResultsMeter()
//=============================
{
  assert (0);
} // ClassResultsMeter						 


//=============================
// Constructor-1
template <typename T1, typename T2, ResourceType T3>
ClassResultsMeter<T1, T2, T3>::ClassResultsMeter(
	size_t		testno_i,
	const T2&	total_run_resource_i,
    	ulong		total_iterations_i
	) :
      		ClassResultsTimer<T2> (
			total_run_resource_i 
			),		
      		testno_ (testno_i),		
      		total_iterations_ (total_iterations_i),		
		test_confidence_filter_ (UNDEF_TestConfidenceFilterType),
                run_cost_value_filter_ (UNDEF_RunCostValueFilterType)
//=============================
{
 assert ((total_iterations_ % perf_ins.get_scale_per_calls()) == 0);

 assert (run_cost_value_filter_ == UNDEF_RunCostValueFilterType);

 if (ResourceValueIsSuspiciouslySmall<T1, T2, T3> (get_total_run_cost()))
 { 
   run_cost_value_filter_ = RUN_COST_VALUE_too_small;
 }

 if (ResourceValueIsSuspiciouslyBig<T1, T2, T3> (get_total_run_cost()))
 { 
   run_cost_value_filter_ = RUN_COST_VALUE_too_big;
 }

 if  (run_cost_value_filter_ == UNDEF_RunCostValueFilterType)
 {
   run_cost_value_filter_ = RUN_COST_VALUE_valid;
 }

 //---------------------------------------------
 scaled_run_cost_ =
        get_total_run_cost() / 
        (total_iterations_ / perf_ins.get_scale_per_calls()
       );
 //---------------------------------------------

 if (ResourceValueIsSuspiciouslySmall<T1, T2, T3> (scaled_run_cost_))
 { 
   run_cost_value_filter_ = RUN_COST_VALUE_too_small;
 }

 if (ResourceValueIsSuspiciouslyBig<T1, T2, T3> (scaled_run_cost_))
 { 
   run_cost_value_filter_ = RUN_COST_VALUE_too_big;
 }


} // ClassResultsMeter						 


//=============================
// Default Copy Constructor
// template <typename T1, typename T2, ResourceType T3>
// ClassResultsMeter<T1, T2, T3>::ClassResultsMeter(const ClassResultsMeter& inst_i)


//=============================
// Destructor
template <typename T1, typename T2, ResourceType T3>
ClassResultsMeter<T1, T2, T3>::~ClassResultsMeter()
//=============================
{
} // ~ClassResultsMeter						 


//=============================
template <typename T1, typename T2, ResourceType T3>
string ClassResultsMeter<T1, T2, T3>::getstr_show_results_meter () const
//=============================
{
string str_scaled_run_cost;

ostringstream osstr;

  osstr << scaled_run_cost_;
  str_scaled_run_cost = osstr.str();

  assert (test_confidence_filter_ >= UNDEF_TestConfidenceFilterType);
  assert (test_confidence_filter_ < LAST_TestConfidenceFilterType);

string filter_sign;
  switch (test_confidence_filter_)
  { 
    case UNDEF_TestConfidenceFilterType :
      filter_sign = "";
      break;

    case TEST_RESULTS_confidence :
      filter_sign = "+";
      break;

    case TEST_RESULTS_non_confidence :
      filter_sign = "-";
      break;

    default :
      // Do nothing
      break;
  }

  osstr.str(string());

  osstr << ""
         << filter_sign
         << "Test["
         << setw(5)
         << testno_
         << "]"
         << " : "
         << setw(12)
         << str_scaled_run_cost.c_str()
         << "  ["
         << setw(7)
         << total_iterations_
         << " iterations]"
         ; 


  switch (run_cost_value_filter_)
  {
    case RUN_COST_VALUE_valid :
      // Do nothing
      break;

    case RUN_COST_VALUE_too_small :
      osstr << " ---> Too small ";
      break;

    case RUN_COST_VALUE_too_big :
      osstr << " ---> Too big";
      break;

    default :
      assert (0);
      break;
  }

  return osstr.str();
} 						 


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassResultsMeter<T1, T2, T3>::show_results_meter () const
//=============================
{
  cout << "\t" << getstr_show_results_meter() << endl;

}

//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassResultsMeter<T1, T2, T3>::set_test_confidence_filter (TestConfidenceFilterType flag_i)
//=============================
{
  test_confidence_filter_ = flag_i;
}



//=============================
template <typename T1, typename T2, ResourceType T3>
T2 ClassResultsMeter<T1, T2, T3>::get_scaled_run_cost () const
//=============================
{
 return scaled_run_cost_;

}


//=============================
template <typename T1, typename T2, ResourceType T3>
size_t ClassResultsMeter<T1, T2, T3>::get_testno () const
//=============================
{
 return testno_;

}

//=============================
template <typename T1, typename T2, ResourceType T3>
RunCostValueFilterType ClassResultsMeter<T1, T2, T3>::get_cost_type () const
//=============================
{
 return run_cost_value_filter_;

}


//#########################################
//#
//#  Class-4 : ClassRunCost
//#  	METHODS
//#
//#########################################

//=============================
// Constructor-0
template <typename T1, typename T2, ResourceType T3>
ClassRunCost<T1, T2, T3>::ClassRunCost()
//=============================
{
  assert(0);
} 						 


//=============================
// Constructor-1
template <typename T1, typename T2, ResourceType T3>
ClassRunCost<T1, T2, T3>::ClassRunCost(TestConfidenceFilterType test_confidence_filter_to_be_set_i)
		:
		test_confidence_filter_to_be_set_ (test_confidence_filter_to_be_set_i),   
		total_confidence_tests_ (0),   
		total_run_cost_ (T2()),		 
		list_of_tests_with_too_small_costs_(vector<count_t>()),		
		list_of_tests_with_too_big_costs_(vector<count_t>())
//=============================
{
} 						 


//=============================
// Destructor
template <typename T1, typename T2, ResourceType T3>
ClassRunCost<T1, T2, T3>::~ClassRunCost()
//=============================
{
} 						 

//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassRunCost<T1, T2, T3>::operator() (
	ClassResultsMeter<T1, T2, T3>& elem_from_sorted_vector_i
	)
//=============================
{
static T2	prev_value;	// for assert only;

  switch (test_confidence_filter_to_be_set_)
  {
    case TEST_RESULTS_confidence : 
      // Do nothing. See below
      break;

    case TEST_RESULTS_non_confidence : 
      elem_from_sorted_vector_i.set_test_confidence_filter (test_confidence_filter_to_be_set_);
      return;
      break;  // unused

    default :
      assert (0);
  }



  //---------------------------------------------------
  assert (test_confidence_filter_to_be_set_ == TEST_RESULTS_confidence);

  if (total_confidence_tests_ > 0)
  {
    assert (prev_value <= elem_from_sorted_vector_i.get_scaled_run_cost());
  }

  total_confidence_tests_++; 
  
  total_run_cost_ += elem_from_sorted_vector_i.get_scaled_run_cost();
  elem_from_sorted_vector_i.set_test_confidence_filter (test_confidence_filter_to_be_set_);

  prev_value = elem_from_sorted_vector_i.get_scaled_run_cost();

  //-----------------------------------
  switch (elem_from_sorted_vector_i.get_cost_type())
  {
    case RUN_COST_VALUE_valid :
      // Do nothing
      break;

    case RUN_COST_VALUE_too_small :
      list_of_tests_with_too_small_costs_.push_back(elem_from_sorted_vector_i.get_testno());
      break;

    case RUN_COST_VALUE_too_big :
      list_of_tests_with_too_big_costs_.push_back(elem_from_sorted_vector_i.get_testno());
      break;

    default :
      assert (0);
      break;
  } 

} 						 

//=============================
template <typename T1, typename T2, ResourceType T3>
T2 ClassRunCost<T1, T2, T3>::get_average_value () const
//=============================
{
  return (total_run_cost_/total_confidence_tests_);
} 						 

//=============================
template <typename T1, typename T2, ResourceType T3>
bool ClassRunCost<T1, T2, T3>::unable_to_get_average_value () const
//=============================
{
  return (
	   !(
	     list_of_tests_with_too_small_costs_.empty()
	     &&
	     list_of_tests_with_too_big_costs_.empty()
	    )
	 );
} 						 



//=============================
template <typename T1, typename T2, ResourceType T3>
T2 ClassRunCost<T1, T2, T3>::get_total_run_cost () const
//=============================
{
  return total_run_cost_;
} 						 


//=============================
template <typename T1, typename T2, ResourceType T3>
count_t ClassRunCost<T1, T2, T3>::get_total_confidence_tests () const
//=============================
{
  return total_confidence_tests_;
} 						 


//=============================
template <typename T1, typename T2, ResourceType T3>
vector<count_t> ClassRunCost<T1, T2, T3>::get_too_small_test_costs () const
//=============================
{
  return list_of_tests_with_too_small_costs_;
} 						 


//=============================
template <typename T1, typename T2, ResourceType T3>
vector<count_t> ClassRunCost<T1, T2, T3>::get_too_big_test_costs () const
//=============================
{
  return list_of_tests_with_too_big_costs_;
} 						 


//#########################################
//#
//#  Class-5 : ClassPerfoTimer	
//#  	METHODS
//#
//#########################################
//=============================
// Constructor-0
template <typename T1, typename T2, ResourceType T3>
ClassPerfoTimer<T1, T2, T3>::ClassPerfoTimer() : end_resource_value_()
//=============================
{
  assert (0);
} // ClassPerfoTimer						 


//=============================
// Copy Constructor
template <typename T1, typename T2, ResourceType T3>
ClassPerfoTimer<T1, T2, T3>::ClassPerfoTimer(const ClassPerfoTimer& inst_i)
//=============================
{
  assert (0);
} // ClassPerfoTimer						 


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoTimer<T1, T2, T3>::ctor_action ()
//=============================
{
} // ClassPerfoTimer::ctor_action


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoTimer<T1, T2, T3>::ctor_trace ()
//=============================
{

} // ClassPerfoTimer::ctor_trace


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoTimer<T1, T2, T3>::dtor_action () const
//=============================
{
  assert (start_resource_value_ <= end_resource_value_);
} // ClassPerfoTimer::dtor_action


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoTimer<T1, T2, T3>::dtor_trace () const
//=============================
{

  if (typeid_name_ == typeid(*this).name())
  {
    ostringstream osstr;
    osstr << "TRACING-T -> "
          << setw (ClassResultsFunc::get_max_funcname_len_S())
          << func_name_.c_str()
          << " ("
          << setw(ClassResultsFunc::get_max_filename_len_S())
          << file_name_.c_str()
          << ", #"
          << setw(ClassResultsFunc::get_max_lineno_len_S())
          << line_no_
          << ") "
          << "<"
          << ::GetResourceName<T1, T2, T3> ()
          << ", "
          << ::GetResourceCostUnitName<T1, T2, T3> ()
          << ", "
          << ::GetResourceStateUnitName<T1, T2, T3> ()
          << "> : start-state = "
          << start_resource_value_
          << ", end-state = "
          << end_resource_value_
          << "; expended = "
          << get_diff_resource_value ()
          ;

    cout << osstr.str() << endl;
  }

} // ClassPerfoTimer::dtor_trace


//=============================
template <typename T1, typename T2, ResourceType T3>
T2 ClassPerfoTimer<T1, T2, T3>::get_diff_resource_value () const
//=============================
{
  return GetDiffResourceValue<T1, T2, T3> (start_resource_value_, end_resource_value_);

} // ClassPerfoTimer::get_diff_resource_value



//#########################################
//#
//#  Class-6 : ClassPerfoMeter	
//#  	METHODS
//#
//#########################################
//-----------------------------
template <typename T1, typename T2, ResourceType T3>
ClassMeasurementCollection<T1, T2, T3>*	ClassPerfoMeter<T1, T2, T3>::ptr_local_measurement_collection_s(new ClassMeasurementCollection<T1, T2, T3>());
//-----------------------------


//=============================
// Constructor-0
template <typename T1, typename T2, ResourceType T3>
ClassPerfoMeter<T1, T2, T3>::ClassPerfoMeter() : ClassPerfoTimer<T1, T2, T3> ()
//=============================
{
  assert (0);
} // ClassPerfoMeter						 


//=============================
// Copy Constructor
template <typename T1, typename T2, ResourceType T3>
ClassPerfoMeter<T1, T2, T3>::ClassPerfoMeter(const ClassPerfoMeter& inst_i)
//=============================
{
  assert (0);
} // ClassPerfoMeter						 


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoMeter<T1, T2, T3>::ctor_action ()
//=============================
{
} // ClassPerfoMeter::ctor_action


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoMeter<T1, T2, T3>::ctor_trace ()
//=============================
{

} // ClassPerfoMeter::ctor_trace


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoMeter<T1, T2, T3>::dtor_action () const
//=============================
{

  assert (start_resource_value_ <= end_resource_value_);

  ptr_local_measurement_collection_s->add_results_meter (
		ClassResultsFunc (
			func_ptr_,
			pretty_func_name_, 
			func_name_, 
			file_name_, 
			line_no_, 
			descr_msg_
			),
		*this
		);


} // ClassPerfoMeter::dtor_action




//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassPerfoMeter<T1, T2, T3>::dtor_trace () const
//=============================
{

  if (!ClassManager::get_tracing_report_flag_S ()) return;

const string	trace_info_msg_CNS = "TRACE-INFO";
static count_t	global_count = 0; 
count_t		local_count = 0;
const size_t	setw11 = 11;

  global_count++;

ostringstream osstr1;
ostringstream osstr2;


  osstr2 << ""
         << setw (ClassResultsFunc::get_max_funcname_len_S())
         << func_name_.c_str()
         << " ("
         << setw(ClassResultsFunc::get_max_filename_len_S())
         << file_name_.c_str()
         << ", #"
         << setw(ClassResultsFunc::get_max_lineno_len_S())
         << line_no_
         << ") : ";


  osstr1.setf (ios::left, ios::adjustfield);

  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << " "
	 << " "
	 << "] -------------------"
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "metrics"
         << " - "
         << ::GetResourceName<T1, T2, T3> ()
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "cost unit"
         << " - "
         << ::GetResourceCostUnitName<T1, T2, T3> ()
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "state unit"
         << " - "
         << ::GetResourceStateUnitName<T1, T2, T3> ()
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "start-state"
         << " = "
         << start_resource_value_
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "end-state"
         << " = "
         << end_resource_value_
         << endl;


  local_count++;
  osstr1 << "" 
	 << "["
	 << trace_info_msg_CNS 
	 << "#"
	 << to_string (global_count, 3, '0')
	 << "-"
	 << local_count
	 << "] "
         << osstr2.str()

         << setw(setw11)
         << "expended"
         << " = "
         << get_diff_resource_value ()
         << endl;

  cout << osstr1.str();

} // ClassPerfoMeter::dtor_trace



//=============================
template <typename T1, typename T2, ResourceType T3>
ulong ClassPerfoMeter<T1, T2, T3>::get_total_iterations () const
//=============================
{
  return total_iterations_;
} 						 




//#########################################
//#
//#  Class-8 : ClassMeasurementCollection
//#  	METHODS
//#
//#########################################
//-----------------------------
template <typename T1, typename T2, ResourceType T3>
count_t	ClassMeasurementCollection<T1, T2, T3>::inst_counter_s(0);
//-----------------------------


//=============================
// Constructor-0
template <typename T1, typename T2, ResourceType T3>
ClassMeasurementCollection<T1, T2, T3>::ClassMeasurementCollection() 
		: BasicClassMeasurementCollection ()
//=============================
{
  //-----------------------
  inst_counter_s++;
  assert (inst_counter_s == 1);
  typeid_name_ = typeid (this).name();
  resource_description_ = get_resource_description();
  //-----------------------
} // ClassMeasurementCollection




//=============================
// Copy Constructor
template <typename T1, typename T2, ResourceType T3>
ClassMeasurementCollection<T1, T2, T3>::ClassMeasurementCollection(const ClassMeasurementCollection& inst_i) 
//=============================
{
  assert (0);
} // ClassMeasurementCollection


//=============================
// Destructor
template <typename T1, typename T2, ResourceType T3>
ClassMeasurementCollection<T1, T2, T3>::~ClassMeasurementCollection() 
//=============================
{
} // ClassMeasurementCollection





//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::add_results_func_if_need (
		const ClassResultsFunc& res_func_i
		) 
//=============================
{

  assert (!res_func_i.get_prettyfuncname().empty());
  //--------------------------
  add_resource_type_if_need ();
  //--------------------------

typedef map<
		ClassResultsFunc, 
		vector<ClassResultsMeter<T1, T2, T3> >, 
		less<ClassResultsFunc> 
		>
		map_typename;

typedef typename map_typename::value_type	map_value_type; 
pair<typename map_typename::iterator, bool>	pair_insert;
typename map_typename::iterator			map_pos;

  map_pos = resource_measurement_collection_.find(res_func_i);
  //--------------------------
  if (map_pos == resource_measurement_collection_.end())
  {
    map_value_type new_pair ( 
		res_func_i,
		vector<ClassResultsMeter<T1, T2, T3> > ()
		);

    pair_insert = resource_measurement_collection_.insert (new_pair);

    assert (pair_insert.second);

  }
  else
  {
    if (map_pos->first.get_funcptr () == NULL)
    {
      if (res_func_i.get_funcptr () != NULL)
      {
        assert (resource_measurement_collection_.count(res_func_i) == 1);
        map_value_type new_pair ( 
		res_func_i,
		map_pos->second
		);


        resource_measurement_collection_.erase(map_pos);
        assert (resource_measurement_collection_.count(res_func_i) == 0);

        pair_insert = resource_measurement_collection_.insert (new_pair);
        assert (pair_insert.second);
        assert (resource_measurement_collection_.count(res_func_i) == 1);

      }
    }
  }

  assert (!(resource_measurement_collection_.empty())); 

  total_measured_funcs_ = resource_measurement_collection_.size();

} // add_results_func_if_need


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::add_results_meter (
		const ClassResultsFunc&			res_func_i,
		const ClassPerfoMeter<T1, T2, T3>&	perfo_meter_i
		)	
//=============================
{

  assert (!res_func_i.get_prettyfuncname().empty());

  //------------------------------------
  add_results_func_if_need (res_func_i);
  //------------------------------------
  assert (!(resource_measurement_collection_.empty())); 

typedef map<
		ClassResultsFunc, 
		vector<ClassResultsMeter<T1, T2, T3> >, 
		less<ClassResultsFunc> 
		>
		map_typename;


typename map_typename::iterator map_pos;
  
  //------------------------------------
  map_pos = resource_measurement_collection_.find (res_func_i);
  assert (map_pos != resource_measurement_collection_.end());
  
  map_pos->second.push_back(
		ClassResultsMeter<T1, T2, T3> (
			map_pos->second.size() + 1,
			perfo_meter_i.get_diff_resource_value (),
			perfo_meter_i.get_total_iterations()
			)
		);


} // add_results_meter


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::show_resource_detailed_descr (
			count_t		resource_no_i,
			const string&	msg_i,
			bool		comparison_show_i  // or measurement
			)
//=============================
{
ostringstream osstr1;
ostringstream osstr2;
  
  osstr1 << "Resource#"
         << resource_no_i
         << " of "
         << get_total_resource_types_s ();

  cout << endl;
  cout << endl;
  cout << endl;
  cout << "\t" << string (40, '|') << endl;
  cout << "\t" << string (2, '|') << endl;

  cout << "\t" 
       << string (2, '|') 
       << "\t" 
       << string(osstr1.str().size(), '-') 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << "\t" 
       << osstr1.str() 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << "\t" 
       << string(osstr1.str().size(), '-') 
       << endl;


  for (count_t i = 0; i < resource_description_.size(); i++)
  {
    cout << "\t" 
         << string (2, '|') 
         << string (1, ' ') 
         << resource_description_[i] 
         << endl;
  }

  cout << "\t" 
       << string (2, '|') 
       << "\t" 
       << string(osstr1.str().size(), '-') 
       << endl;


  osstr2 << "" 
         << string (3, '*') 
         << string (1, ' ')  
         << msg_i 
         << string (1, ' ')  
         << string (3, '*');

  cout << "\t" 
       << string (2, '|') 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << string (2, ' ') 
       << string (osstr2.str().size(), '*') 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << string (2, ' ') 
       << osstr2.str() 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << string (2, ' ') 
       << string (osstr2.str().size(), '*') 
       << endl;

  cout << "\t" 
       << string (2, '|') 
       << endl;


  if (!comparison_show_i)
  {

    cout << "\t" 
         << string (2, '|') 
         << "\t" 
         << string(osstr1.str().size(), '-') 
         << endl;

    cout << "\t" 
         << string (2, '|') 
         << string (2, ' ') 
         << "Total measured functions : " 
         << resource_measurement_collection_.size() 
         << endl;

  //---------------------------------
typedef map<
		ClassResultsFunc, 
		vector<ClassResultsMeter<T1, T2, T3> >, 
		less<ClassResultsFunc> 
		>
		map_typename;
typename map_typename::iterator	map_iter;

    for (
	  map_iter = resource_measurement_collection_.begin();
	  map_iter != resource_measurement_collection_.end();
	  map_iter++
	  )
    {
      cout << "\t" << string (2, '|') << string (5, ' ') 
           << map_iter->first.get_prettyfuncname ()
           << " ---> See : "
           << map_iter->first.get_filename ()
           << ", Line#"
           << map_iter->first.get_lineno ()
           << endl;
    }
  //---------------------------------
  } // if (!comparison_show_i)

  cout << "\t" << string (2, '|') << endl;
  cout << "\t" << string (40, '|') << endl;
  cout << endl;

} // ClassMeasurementCollection<T1, T2, T3>::show_resource_detailed_descr


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::show_measurement_collection ()
//=============================
{

typedef map<
		ClassResultsFunc, 
		vector<ClassResultsMeter<T1, T2, T3> >, 
		less<ClassResultsFunc> 
		>
		map_typename;
typename map_typename::iterator	map_iter;

typedef typename map_typename::value_type map_value_type; 

count_t	func_serial_no;


  //---------------------------

  for (
	map_iter = resource_measurement_collection_.begin(), func_serial_no = 1;
	map_iter != resource_measurement_collection_.end();
	map_iter++, func_serial_no++
	)
  {
    report_a_func_measurement_S (
		map_iter->first,
		func_serial_no, 
		resource_measurement_collection_.size(),
		map_iter->second,
		ClassManager::get_discarding_threshold_S ()
		);
  }


} // ClassMeasurementCollection<T1, T2, T3>::show_measurement_collection						 


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::show_comparison_collection (const vector<TypeMeasuredFunc*>& func_ptrs_i)
//=============================
{


typedef map<
		ClassResultsFunc, 
		vector<ClassResultsMeter<T1, T2, T3> >, 
		less<ClassResultsFunc> 
		>
		map_typename;
typename map_typename::iterator	map_iter;

typedef typename map_typename::value_type map_value_type; 



  //---------------------------

count_t	total_compared_funcs = 0;
  for (
	map_iter = resource_measurement_collection_.begin();
	map_iter != resource_measurement_collection_.end();
	map_iter++
	)
  {
    if (find (
		func_ptrs_i.begin(),    
		func_ptrs_i.end(),
		map_iter->first.get_funcptr ()
		) == func_ptrs_i.end()
	)
    {
      continue;
    }
    total_compared_funcs++;
    report_a_func_comparison_S (
		map_iter->first,
		map_iter->second,
		ClassManager::get_discarding_threshold_S ()
		);
  }

  if (total_compared_funcs == 0)
  {
      cout << string (3, ' ') << ": " << "No functions to be compared" << endl;
  }


} // ClassMeasurementCollection<T1, T2, T3>::show_comparison_collection						 


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::report_a_func_measurement_S (
		const ClassResultsFunc&	res_func_i,
		count_t 		func_serial_no_i,
		count_t 		total_measured_funcs_i,
		const vector<ClassResultsMeter<T1, T2, T3> >&	test_results_i, 
		float		discarding_threshold_i
		)
//=============================
{

vector<vector<string> >	average_report_info(LAST_AverageReportKing);
  report_preparation1_S (average_report_info);


  //---------------------------------------
  //--- string constants ------------------
  //---------------------------------------
const string sub_header1 ("--- Results sorted according to a test number");
const string sub_header2 ("--- Results sorted according to run cost");
const string sub_header3 ("--- Summary report : Average run cost");

  //---------------------------------------


  //---------------------------------------
const size_t	&total_tests = test_results_i.size();
const size_t	half_total_discarded_tests = static_cast<size_t> (static_cast<float>(total_tests) * discarding_threshold_i);   
const size_t	total_confidence_tests	= total_tests - (half_total_discarded_tests * 2);

const size_t	serial_no_of_min_confidence_cost_value	= 1 + half_total_discarded_tests;   
const size_t	serial_no_of_max_confidence_cost_value	= total_tests - half_total_discarded_tests;   
  //---------------------------------------

  assert (
	  (serial_no_of_max_confidence_cost_value - serial_no_of_min_confidence_cost_value + 1)
          ==
	  total_confidence_tests
         );
  assert (serial_no_of_min_confidence_cost_value <= serial_no_of_max_confidence_cost_value);



  //-------------------------------------------------
  //-------------- sorting vector etc ---------------
  //-------------------------------------------------

vector<ClassResultsMeter<T1, T2, T3> >	run_cost_sorted (test_results_i); 

  stable_sort (
	run_cost_sorted.begin (), 
	run_cost_sorted.end (), 
	less_cost<T1, T2, T3> ()
	);

const T2	min_abs_cost_value	= run_cost_sorted.front().get_scaled_run_cost();
const T2	max_abs_cost_value	= run_cost_sorted.back().get_scaled_run_cost();		 
count_t		testno_of_min_abs_cost_value	= run_cost_sorted.front().get_testno();   
count_t		testno_of_max_abs_cost_value	= run_cost_sorted.back().get_testno();   

const T2	min_confidence_cost_value	= run_cost_sorted[serial_no_of_min_confidence_cost_value - 1].get_scaled_run_cost();
const T2	max_confidence_cost_value	= run_cost_sorted[serial_no_of_max_confidence_cost_value - 1].get_scaled_run_cost();		 
count_t		testno_of_min_confidence_cost_value	= run_cost_sorted[serial_no_of_min_confidence_cost_value - 1].get_testno();   
count_t		testno_of_max_confidence_cost_value	= run_cost_sorted[serial_no_of_max_confidence_cost_value - 1].get_testno();   

  //-------------------------------------------------

  average_report_info[DISCARDING_THRESHOLD_info].push_back(to_string(discarding_threshold_i));

  average_report_info[DISCARDED_TESTS_info].push_back(to_string(half_total_discarded_tests * 2));
  average_report_info[DISCARDED_TESTS_info].push_back(" (");
  average_report_info[DISCARDED_TESTS_info].push_back(to_string(half_total_discarded_tests) + " minimal");
  average_report_info[DISCARDED_TESTS_info].push_back(" and ");
  average_report_info[DISCARDED_TESTS_info].push_back(to_string(half_total_discarded_tests) + " maximal");
  average_report_info[DISCARDED_TESTS_info].push_back(")");

  average_report_info[TOTAL_TESTS_info].push_back(to_string(total_tests));

  average_report_info[MIN_ABS_VALUE_info].push_back(to_string(min_abs_cost_value));

  average_report_info[MAX_ABS_VALUE_info].push_back(to_string(max_abs_cost_value));

  average_report_info[TOTAL_SELECTED_TESTS_info].push_back(to_string(total_confidence_tests));

  average_report_info[MIN_SELECTED_VALUE_info].push_back(to_string(min_confidence_cost_value));

  average_report_info[MAX_SELECTED_VALUE_info].push_back(to_string(max_confidence_cost_value));

size_t		max_text2_size = 0;
  max_text2_size = MAX_VALUE (max_text2_size, average_report_info[MIN_ABS_VALUE_info].back().size());
  max_text2_size = MAX_VALUE (max_text2_size, average_report_info[MAX_ABS_VALUE_info].back().size());
  max_text2_size = MAX_VALUE (max_text2_size, average_report_info[MIN_SELECTED_VALUE_info].back().size());
  max_text2_size = MAX_VALUE (max_text2_size, average_report_info[MAX_SELECTED_VALUE_info].back().size());
  max_text2_size += 5; 

  average_report_info[MIN_ABS_VALUE_info].push_back(string (max_text2_size - average_report_info[MIN_ABS_VALUE_info].back().size(), ' '));
  average_report_info[MAX_ABS_VALUE_info].push_back(string (max_text2_size - average_report_info[MAX_ABS_VALUE_info].back().size(), ' '));
  average_report_info[MIN_SELECTED_VALUE_info].push_back(string (max_text2_size - average_report_info[MIN_SELECTED_VALUE_info].back().size(), ' '));
  average_report_info[MAX_SELECTED_VALUE_info].push_back(string (max_text2_size - average_report_info[MAX_SELECTED_VALUE_info].back().size(), ' '));

  average_report_info[MIN_ABS_VALUE_info].push_back(" --> in Test#");
  average_report_info[MAX_ABS_VALUE_info].push_back(" --> in Test#");
  average_report_info[MIN_SELECTED_VALUE_info].push_back(" --> in Test#");
  average_report_info[MAX_SELECTED_VALUE_info].push_back(" --> in Test#");


  average_report_info[MIN_ABS_VALUE_info].push_back(to_string(testno_of_min_abs_cost_value));
  average_report_info[MAX_ABS_VALUE_info].push_back(to_string(testno_of_max_abs_cost_value));
  average_report_info[MIN_SELECTED_VALUE_info].push_back(to_string(testno_of_min_confidence_cost_value));
  average_report_info[MAX_SELECTED_VALUE_info].push_back(to_string(testno_of_max_confidence_cost_value));

  //-------------------------------------------------
ClassRunCost<T1, T2, T3>	average_cost = 
  for_each (
	run_cost_sorted.begin() + half_total_discarded_tests,
	run_cost_sorted.end() - half_total_discarded_tests,
	ClassRunCost<T1, T2, T3> (TEST_RESULTS_confidence)
	);

  average_report_info[AVERAGE_VALUE_info].push_back(to_string (average_cost.get_average_value ()));
  average_report_info[AVERAGE_VALUE_info].push_back(" per ");
  average_report_info[AVERAGE_VALUE_info].push_back(to_string (perf_ins.get_scale_per_calls()));
  average_report_info[AVERAGE_VALUE_info].push_back(" iterations");
  average_report_info[AVERAGE_VALUE_info].push_back(" (");
  average_report_info[AVERAGE_VALUE_info].push_back(to_string(average_cost.get_total_run_cost()));
  average_report_info[AVERAGE_VALUE_info].push_back("/");
  average_report_info[AVERAGE_VALUE_info].push_back(to_string(average_cost.get_total_confidence_tests()));
  average_report_info[AVERAGE_VALUE_info].push_back(" )");

  for_each (
	run_cost_sorted.begin(),
	run_cost_sorted.begin() + half_total_discarded_tests,
	ClassRunCost<T1, T2, T3> (TEST_RESULTS_non_confidence)
	);

  for_each (
	run_cost_sorted.end() - half_total_discarded_tests,
	run_cost_sorted.end(),
	ClassRunCost<T1, T2, T3> (TEST_RESULTS_non_confidence)
	);

  //-----------------------------------------------
  //------------- output --------------------------
  //-----------------------------------------------
bool	need_to_show_detailed_report = 
	perf_ins.get_detailed_measurement_report_flag ()
	||
	average_cost.unable_to_get_average_value ();

  cout << "" 
       << res_func_i.getstr_func_detailed_descr(
		resource_description_,
		func_serial_no_i, 
		total_measured_funcs_i,
		test_results_i.size(),
		need_to_show_detailed_report 
		) 
      << endl;


  //-------------------
  if (need_to_show_detailed_report)
  {
    cout << "\t" << string (sub_header1.size(), '-') << endl;
    cout << "\t" << sub_header1 << endl;
    cout << "\t" << string (sub_header1.size(), '-') << endl;

    for_each (
	test_results_i.begin(),
	test_results_i.end(),
	mem_fun_ref (&ClassResultsMeter<T1, T2, T3>::show_results_meter)
	);

    cout << endl;

    //-------------------
    cout << "\t" << string (sub_header2.size(), '-') << endl;
    cout << "\t" << sub_header2 << endl;
    cout << "\t" << string (sub_header2.size(), '-') << endl;


    for_each (
	run_cost_sorted.begin(),
	run_cost_sorted.end(),
	mem_fun_ref (&ClassResultsMeter<T1, T2, T3>::show_results_meter)
	);

    cout << endl;
    cout << endl;
  }
  //-----------------------------------------------
  //-----------------------------------------------

#ifdef OPTIMIZATION_LEVEL
  cout << "\t" << OPTIMIZATION_LEVEL << endl;
#endif

  cout << "\t" << string (sub_header3.size(), '-') << endl;
  cout << "\t" << sub_header3 << endl;
  cout << "\t" << string (sub_header3.size(), '-') << endl;


  //--------------------------------------------------------
  if (average_cost.unable_to_get_average_value ())
  {
     cout << endl;
     cout << "\t=== Unable to get average value ===" << endl;
     vector<count_t> too_small_vect = average_cost.get_too_small_test_costs();
     vector<count_t> too_big_vect   = average_cost.get_too_big_test_costs();

     if (!too_small_vect.empty())
     {
        cout << "\t  Confidence tests " ;
        copy (
		too_small_vect.begin(),
		too_small_vect.end(),
		ostream_iterator<count_t> (cout, " ")
		);		
        cout << " holds too SMALL cost values" ;
        cout << endl;
     }
     

     if (!too_big_vect.empty())
     {
        cout << "\t  Confidence tests " ;
        copy (
		too_big_vect.begin(),
		too_big_vect.end(),
		ostream_iterator<count_t> (cout, " ")
		);		
        cout << " holds too BIG cost values" ;
        cout << endl;
     }
  }
  else
  {
    cout.setf (ios::left, ios::adjustfield);
    for (count_t i = 0; i < average_report_info.size(); i++)
    {
      cout << "\t";
      copy (
	average_report_info[i].begin(), 
	average_report_info[i].end(), 
	ostream_iterator<string> (cout, "")
	);
      cout << endl;
    }

  }
  //--------------------------------------------------------

  cout << endl;
  cout << endl;

} // ClassMeasurementCollection<T1, T2, T3>::report_a_func_measurement_S

//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::report_preparation1_S (
		vector<vector<string> >& vect_io
		)
//=============================
{
  vect_io[AVERAGE_VALUE_info].push_back(string(0, ' '));
  vect_io[AVERAGE_VALUE_info].back() += "Average value";

  vect_io[ALL_TESTS_info].push_back(string(2, ' '));
  vect_io[ALL_TESTS_info].back() += "*** All tests ***";

  vect_io[SELECTED_TESTS_info].push_back(string(2, ' '));
  vect_io[SELECTED_TESTS_info].back() += "*** Selected tests ***";

  vect_io[DISCARDING_THRESHOLD_info].push_back(string(5, ' '));
  vect_io[DISCARDING_THRESHOLD_info].back() += "Discarding threshold";

  vect_io[DISCARDED_TESTS_info].push_back(string(5, ' '));
  vect_io[DISCARDED_TESTS_info].back() += "Discarded tests";

  vect_io[TOTAL_TESTS_info].push_back(string(5, ' '));
  vect_io[TOTAL_TESTS_info].back() += "Total tests";

  vect_io[MIN_ABS_VALUE_info].push_back(string(5, ' '));
  vect_io[MIN_ABS_VALUE_info].back() += "Min abs value";

  vect_io[MAX_ABS_VALUE_info].push_back(string(5, ' '));
  vect_io[MAX_ABS_VALUE_info].back() += "Max abs value";

  vect_io[TOTAL_SELECTED_TESTS_info].push_back(string(5, ' '));
  vect_io[TOTAL_SELECTED_TESTS_info].back() += "Total selected tests";

  vect_io[MIN_SELECTED_VALUE_info].push_back(string(5, ' '));
  vect_io[MIN_SELECTED_VALUE_info].back() += "Min selected value";

  vect_io[MAX_SELECTED_VALUE_info].push_back(string(5, ' '));
  vect_io[MAX_SELECTED_VALUE_info].back() += "Max selected value";
  
size_t		max_text1_size = 0;
  for (count_t i = 0; i < vect_io.size(); i++)
  {
    max_text1_size = MAX_VALUE (max_text1_size, vect_io[i].back().size());
  }
  max_text1_size++;

  for (count_t i = 0; i < vect_io.size(); i++)
  {
    vect_io[i].push_back(
		string (max_text1_size - vect_io[i].back().size(), ' ')
		);
  }

  vect_io[AVERAGE_VALUE_info].push_back("=");
  vect_io[DISCARDING_THRESHOLD_info].push_back("=");
  vect_io[DISCARDED_TESTS_info].push_back("=");
  vect_io[TOTAL_TESTS_info].push_back("=");
  vect_io[MIN_ABS_VALUE_info].push_back("=");
  vect_io[MAX_ABS_VALUE_info].push_back("=");
  vect_io[TOTAL_SELECTED_TESTS_info].push_back("=");
  vect_io[MIN_SELECTED_VALUE_info].push_back("=");
  vect_io[MAX_SELECTED_VALUE_info].push_back("=");

  for (count_t i = 0; i < vect_io.size(); i++)
  {
    vect_io[i].push_back(string (1, ' '));
  }

} // ClassMeasurementCollection<T1, T2, T3>::report_preparation1_S


//=============================
template <typename T1, typename T2, ResourceType T3>
void ClassMeasurementCollection<T1, T2, T3>::report_a_func_comparison_S (
		const ClassResultsFunc&	res_func_i,
		const vector<ClassResultsMeter<T1, T2, T3> >&	test_results_i, 
		float		discarding_threshold_i
		)
//=============================
{
  //---------------------------------------
const size_t	&total_tests = test_results_i.size();
const size_t	half_total_discarded_tests = static_cast<size_t> (static_cast<float>(total_tests) * discarding_threshold_i);   
  //---------------------------------------

  //-------------------------------------------------
  //-------------- sorting vector etc ---------------
  //-------------------------------------------------

vector<ClassResultsMeter<T1, T2, T3> >	run_cost_sorted (test_results_i); 

  stable_sort (
	run_cost_sorted.begin (), 
	run_cost_sorted.end (), 
	less_cost<T1, T2, T3> ()
	);

  //-------------------------------------------------

ClassRunCost<T1, T2, T3>	average_cost = 
  for_each (
	run_cost_sorted.begin() + half_total_discarded_tests,
	run_cost_sorted.end() - half_total_discarded_tests,
	ClassRunCost<T1, T2, T3> (TEST_RESULTS_confidence)
	);

  //--------------------------------------------------------
ostringstream osstr;


  osstr << string (3, ' ') << ": ";
  osstr << setw(ClassResultsFunc::get_max_funcname_len_S()) << res_func_i.get_funcname().c_str();
  osstr << " : ";
  osstr << setw(ClassResultsFunc::get_max_filename_len_S()) << res_func_i.get_filename().c_str();
  osstr << "  ";
  osstr << setw(ClassResultsFunc::get_max_lineno_len_S() + 1) << string ("#" + to_string(res_func_i.get_lineno ())).c_str();
  osstr << " : ";

  osstr.setf (ios::left, ios::adjustfield);
  osstr << setw(ClassResultsFunc::get_max_descrmsg_len_S()) << res_func_i.get_descr().c_str();
  osstr << " -> ";

  osstr.setf (ios::right, ios::adjustfield);
  if (average_cost.unable_to_get_average_value ())
  {
     osstr << "Unable to get average cost";
  }
  else
  {
    ostringstream osstr2;
    osstr2 << average_cost.get_average_value ();
    osstr << setw(12) << osstr2.str().c_str();
  }
  cout << osstr.str() << endl;

} // ClassMeasurementCollection<T1, T2, T3>::report_a_func_comparison_S


//=============================
template <typename T1, typename T2, ResourceType T3>
vector<string> ClassMeasurementCollection<T1, T2, T3>::get_resource_description ()
//=============================
{
vector<string>	ret_vect;
  ret_vect.push_back ("Resource Name       : " + GetResourceName<T1, T2, T3> ());
  ret_vect.push_back ("Resource Cost Unit  : " + GetResourceCostUnitName<T1, T2, T3> ());
  ret_vect.push_back ("Resource State Unit : " + GetResourceStateUnitName<T1, T2, T3> ());

  return ret_vect;
} // ClassMeasurementCollection<T1, T2, T3>::get_resource_description

//###########################################
#endif
//###########################################

// ------------------- C++ code : END ----------------------
// === File #7 of 14 : pfmeter2.h ==========================
