// #########################################################
// === File #6 of 14 : pfmeter.h ===========================
// ------------------- C++ code : BEGIN --------------------

// ==============================================================
//
//  Copyright (c) 2002 by Alex Vinokur.  This work and all works
//  derived from it may be copied and modified without any
//  restrictions other than that a copy of this copyright notice
//  must be included in any copy of this work or any derived work.
//
// ==============================================================
// #########################################################
// ## mailto:alexvn@go.to, mailto:alexv@hitechclub.com
// ## http://up.to/alexv, http://go.to/alexv_math
// #########################################################
// ==============================================================


// ##############################################################
//
//  SOFTWARE : C/C++ Program Perfometer
//  FILE     : pfmeter.h
//
//  DESCRIPTION :
//         Definition of performeter's common classes
//
// ##############################################################



///////////////////
#ifndef _PFMETER_H
#define _PFMETER_H
///////////////////


//==================
#include "macro.h"
#include "user.h"


//#########################################
//#
//#  enum
//#
//#########################################
//=============================
enum TestConfidenceFilterType
{
  UNDEF_TestConfidenceFilterType,
  TEST_RESULTS_confidence,
  TEST_RESULTS_non_confidence,
  LAST_TestConfidenceFilterType
};


//=============================
enum RunCostValueFilterType
{
  UNDEF_RunCostValueFilterType,
  RUN_COST_VALUE_valid,
  RUN_COST_VALUE_too_small,
  RUN_COST_VALUE_too_big,
  LAST_RunCostValueFilterType
};


//#########################################
//#
//#  Classes
//#  	DECLARATION
//#
//#########################################
//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassMeasurementCollection;


//#########################################
//#
//#  Constants, Defines
//#
//#########################################
//=============================
const unsigned int TOTAL_ITERATIONS_PER_TEST_default	= 100000;
const unsigned int TOTAL_TESTS_default			= 7;
const unsigned int PER_CALLS_default			= 1000;

#define	DISCARDING_THRESHOLD_default			0.15

//#########################################
//#
//#  Global Template Functions 
//#  	DECLARATION
//#
//#########################################


//#########################################
//#
//#  Global (Non-Template) Functions 
//#  	DECLARATION
//#
//#########################################
//=============================
void MeasureFunc (TypeMeasuredFunc * const func_i);
void CreateComparisonGroup (const string& group_name_i);
void CompareFunc(const string& group_name_i, TypeMeasuredFunc* const func_i);

void SetDiscardingThreshold(float discarding_threshold_i);
void SetTotalTests (ulong tests_i);
void SetScaleAndTotalIterations (ulong per_calls_i, ulong iterations_i);
void SetDetailedMeasurementReportFlag(bool flag_i);
void SetMeasurementReportFlag(bool flag_i);
void SetTracingReportFlag(bool flag_i);

bool ParseCommandLine(int argc, char** argv);


//#########################################
//#
//#  Class-1 : ClassResultsFunc
//#  	DEFINITION
//#
//#########################################
//=============================
class ClassResultsFunc;

//=============================
bool operator< (
		const ClassResultsFunc& ins1, 
		const ClassResultsFunc& ins2
		);

//=============================
bool operator== (
		const ClassResultsFunc& ins1, 
		const ClassResultsFunc& ins2
		);



//=============================
class ClassResultsFunc 
//=============================
{
friend bool operator< (
		const ClassResultsFunc& ins1, 
		const ClassResultsFunc& ins2
		);

friend bool operator== (
		const ClassResultsFunc& ins1, 
		const ClassResultsFunc& ins2
		);

  private:
    //---------------------------
    TypeMeasuredFunc*	funcptr_;
    string		prettyfuncname_;
    string		funcname_;
    string		filename_;
    size_t		lineno_;
    string		descrmsg_;

    static size_t	max_prettyfuncname_len_s;
    static size_t	max_funcname_len_s;
    static size_t	max_filename_len_s;
    static size_t	max_descrmsg_len_s;


  public:
    //---------------------------
    // Constructor-0
    ClassResultsFunc();

    // Constructor-1
    ClassResultsFunc(
	TypeMeasuredFunc* 		const funcptr_i,
	const string&			prettyfuncname_i,
	const string&			funcname_i,
	const string&			filename_i,
	size_t				lineno_i,
	const string&			descrmsg_i
	);


    // Default Copy Constructor
    // ClassResultsFunc(const ClassResultsFunc& inst_i);


    // Destructor
    ~ClassResultsFunc();
 
    string		get_prettyfuncname () const;
    string		get_funcname () const;
    string		get_filename () const;
    size_t		get_lineno () const;
    string		get_descr () const;
    TypeMeasuredFunc*	get_funcptr () const;

    string 	getstr_func_detailed_descr (
			vector<string>	resource_descr_i,
			count_t		func_serial_no_i, 
			count_t		total_funcs_i, 
			count_t		total_tests_i, 
			bool		detailed_report_flag_i, 
			const string&	msg_i = string()
			) const;


    static size_t	get_max_prettyfuncname_len_S();
    static size_t	get_max_funcname_len_S();
    static size_t	get_max_filename_len_S();
    static size_t	get_max_lineno_len_S();
    static size_t	get_max_descrmsg_len_S();

}; // class ClassResultsFunc



//#########################################
//#
//#  Class-2 : ClassResultsTimer
//#  	DEFINITION
//#
//#########################################
//=============================
template <typename T2>
class ClassResultsTimer 
//=============================
{
  private:
    T2	total_run_cost_;

  protected:
    T2	get_total_run_cost() const;

  public:
    //---------------------------
    // Constructor-0
    ClassResultsTimer();

    // Constructor-1
    ClassResultsTimer(
	const T2&	total_run_cost_i
	);

    // Default Copy Constructor
    // ClassResultsTimer(const ClassResultsTimer& inst_i);

    // Destructor
    ~ClassResultsTimer();
 

}; // class ClassResultsTimer



//#########################################
//#
//#  Class-3 : ClassResultsMeter
//#  	DEFINITION
//#
//#########################################

//----------- template class less_cost -----------
template <typename T1, typename T2, ResourceType T3>
class less_cost;


//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassResultsMeter : public ClassResultsTimer<T2> 
//=============================
{
friend class less_cost<T1, T2, T3>;
  private:
    //---------------------------
    T2				scaled_run_cost_;
    size_t			testno_;
    ulong			total_iterations_;
    TestConfidenceFilterType	test_confidence_filter_;
    RunCostValueFilterType	run_cost_value_filter_;


  public:
    //---------------------------
    // Constructor-0
    ClassResultsMeter();

    // Constructor-1
    ClassResultsMeter(
	size_t		testno_i,
	const T2&	total_run_cost_i,
    	ulong		total_iterations_i
	);

    // Default Copy Constructor
    // ClassResultsMeter(const ClassResultsMeter& inst);

    // Destructor
    ~ClassResultsMeter();
 
    string		getstr_show_results_meter () const;
    void		show_results_meter () const;
    T2			get_scaled_run_cost () const;
    size_t		get_testno () const;
    RunCostValueFilterType	get_cost_type() const;

    void	set_test_confidence_filter (TestConfidenceFilterType flag_i);


}; // class ClassResultsMeter



//#########################################
//#
//#  Class-4 : ClassRunCost
//#  	DEFINITION
//#
//#########################################

//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassRunCost 
//=============================
{
  private:
    TestConfidenceFilterType	test_confidence_filter_to_be_set_;		 
    count_t			total_confidence_tests_;   
    T2				total_run_cost_;		 
    vector<count_t>		list_of_tests_with_too_small_costs_;		
    vector<count_t>		list_of_tests_with_too_big_costs_;		

  public:
    // Constructor-0
    ClassRunCost ();

    // Constructor-1
    ClassRunCost (TestConfidenceFilterType test_confidence_filter_to_be_set_i);

    // Default Copy Constructor
    // ClassRunCost(const ClassRunCost& inst);

    // Destructor
    ~ClassRunCost ();

    void operator() (ClassResultsMeter<T1, T2, T3>& elem_from_sorted_vector_i);	// not const&

    T2			get_average_value () const;
    T2			get_total_run_cost () const;
    count_t		get_total_confidence_tests() const;   
    bool		unable_to_get_average_value () const;
    vector<count_t>	get_too_small_test_costs() const;		
    vector<count_t>	get_too_big_test_costs() const;		
   
};



//#########################################
//#
//#  Class-5 : ClassPerfoTimer
//#  	DEFINITION
//#
//#########################################
//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassPerfoTimer 
//=============================
{
  private:
  protected:
    string		typeid_name_;
    //---------------------------
    TypeMeasuredFunc*	func_ptr_;
    string		pretty_func_name_;
    string		func_name_;
    string		file_name_;
    size_t		line_no_;
    string		descr_msg_;

    T1			start_resource_value_;
    T1			end_resource_value_;

    virtual void	ctor_action ();
    virtual void	ctor_trace ();

    virtual void	dtor_action () const;
    virtual void	dtor_trace () const;


  public:
    //---------------------------
    // Constructor-0
    ClassPerfoTimer();

    // Constructor-1
    ClassPerfoTimer(
    		TypeMeasuredFunc*		const func_ptr_i,
		string				pretty_func_name_i,
		string				func_name_i,
		string				file_name_i,
		size_t				line_no_i,
		string				descr_msg_i
		) : 
			func_ptr_ (func_ptr_i), 
			pretty_func_name_ (pretty_func_name_i), 
			func_name_ (func_name_i), 
			file_name_ (file_name_i), 
			line_no_ (line_no_i), 
			descr_msg_ (descr_msg_i), 
			end_resource_value_()
    {
      typeid_name_ = typeid(*this).name();
      ctor_action();
      ctor_trace();
      start_resource_value_ = GetCurrentResource<T1, T2, T3>();
    }

    // Copy Constructor
    ClassPerfoTimer(const ClassPerfoTimer& inst_i);

    // Destructor
    virtual ~ClassPerfoTimer()
    {
      T1 tmp_end_resource_value = GetCurrentResource<T1, T2, T3>();
      if (typeid_name_ == typeid(*this).name())
      {
        end_resource_value_ = tmp_end_resource_value;
      }
      dtor_action();
      dtor_trace();
    }

    T2		get_diff_resource_value () const;
    

}; // class ClassPerfoTimer




//#########################################
//#
//#  Class-6 : ClassPerfoMeter	
//#  	DEFINITION
//#
//#########################################
//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassPerfoMeter : public ClassPerfoTimer<T1, T2, T3>
//=============================
{
  private:
    //---------------------------
    ulong		total_iterations_;

    void		ctor_action ();
    void		ctor_trace ();

    void		dtor_action () const;
    void		dtor_trace () const;

    static ClassMeasurementCollection<T1, T2, T3>*	ptr_local_measurement_collection_s;

  public:
    // Constructor-0
    ClassPerfoMeter();

    // Constructor-1
    ClassPerfoMeter(
    		TypeMeasuredFunc*	func_ptr_i,
		string			pretty_func_name_i,
		string			func_name_i,
		string			file_name_i,
		size_t			line_no_i,
		string			descr_msg_i,
		ulong			total_iterations_i
		) : 
			ClassPerfoTimer<T1, T2, T3> (
				func_ptr_i, 
				pretty_func_name_i, 
				func_name_i, 
				file_name_i, 
				line_no_i, 
				descr_msg_i 
				),
			total_iterations_ (total_iterations_i)
			
    {
      typeid_name_ = typeid(*this).name();
      ctor_action();      
      ctor_trace();      
      start_resource_value_ = GetCurrentResource<T1, T2, T3>();
    }

    // Copy Constructor
    ClassPerfoMeter(const ClassPerfoMeter& inst_i);

    // Destructor
    ~ClassPerfoMeter()
    {
      end_resource_value_ = GetCurrentResource<T1, T2, T3>();
      dtor_action();      
      dtor_trace();      
    }

    ulong	get_total_iterations () const;

}; // class ClassPerfoMeter



//#########################################
//#
//#  Class-7 : BasicClassMeasurementCollection
//#  	DEFINITION
//#
//#########################################
//=============================
class BasicClassMeasurementCollection
//=============================
{
friend class ClassManager;
  private:
    static vector<BasicClassMeasurementCollection*>	all_measurement_collections_s;

  protected:

    string		typeid_name_;
    count_t		total_measured_funcs_;
    vector<string>	resource_description_;

    void		add_resource_type_if_need ();	
    bool		typeid_predicat (const BasicClassMeasurementCollection* const inst_i) const;	
    static size_t	get_total_resource_types_s ();

    virtual void	show_measurement_collection () = 0;
    virtual void	show_comparison_collection (
				const vector<TypeMeasuredFunc*>& func_ptrs_i
				) = 0;

    virtual void	show_resource_detailed_descr (
				count_t		resource_no_i,
				const string&	msg_i,
				bool		comparison_show_i  // or measurement
				) = 0;

  public:
    //---------------------------
    // Constructor-0
    BasicClassMeasurementCollection();

    // Copy Constructor
    BasicClassMeasurementCollection(const BasicClassMeasurementCollection& inst_i);

    // Destructor
    virtual ~BasicClassMeasurementCollection();

    static void	show_list_of_resources_s ();

}; // BasicClassMeasurementCollection



//#########################################
//#
//#  Class-8 : ClassMeasurementCollection
//#  	DEFINITION
//#
//#########################################
//=============================
template <typename T1, typename T2, ResourceType T3>
class ClassMeasurementCollection : public BasicClassMeasurementCollection
//=============================
{

  private:
    enum AverageReportKind
    {
      AVERAGE_VALUE_info,
      ALL_TESTS_info,
      TOTAL_TESTS_info,
      MIN_ABS_VALUE_info,
      MAX_ABS_VALUE_info,
      SELECTED_TESTS_info,
      DISCARDING_THRESHOLD_info,
      DISCARDED_TESTS_info,
      TOTAL_SELECTED_TESTS_info,
      MIN_SELECTED_VALUE_info,
      MAX_SELECTED_VALUE_info,
      LAST_AverageReportKing
    };

    static count_t	inst_counter_s;
    map<
	ClassResultsFunc, 
	vector<ClassResultsMeter<T1, T2, T3> >, 
	less<ClassResultsFunc> 
	> 
	resource_measurement_collection_;	

    void	add_results_func_if_need (const ClassResultsFunc& res_func_i);	

    void	show_measurement_collection ();
    void	show_comparison_collection (
			const vector<TypeMeasuredFunc*>& func_ptrs_i
			);

    void	show_resource_detailed_descr (
			count_t		resource_no_i,
			const string&	msg_i,
			bool		comparison_show_i  // or measurement
			);

    void	report_a_func_measurement_S (
			const ClassResultsFunc&	res_func_i,
			count_t 	func_serial_no_i,
			count_t 	total_measured_funcs_i,
			const vector<ClassResultsMeter<T1, T2, T3> >&	test_results_i, 
			float		discarding_threshold_i
			);

    void	report_a_func_comparison_S (
			const ClassResultsFunc&	res_func_i,
			const vector<ClassResultsMeter<T1, T2, T3> >&	test_results_i, 
			float		discarding_threshold_i
			);

    void	report_preparation1_S (vector<vector<string> >& vect_io);


    vector<string> get_resource_description();

  public:
    //---------------------------
    // Constructor-0
    ClassMeasurementCollection();

    // Copy Constructor
    ClassMeasurementCollection(const ClassMeasurementCollection& inst_i);

    // Destructor
    ~ClassMeasurementCollection();

    //---------------------------
    void	add_results_meter (
			const ClassResultsFunc& res_func_i,
			const ClassPerfoMeter<T1, T2, T3>& perfo_meter_i
			);	

}; // ClassMeasurementCollection



//#########################################
//#
//#  Class-9 : ClassManager
//#  	DEFINITION
//#
//#########################################
//=============================
class ClassManager
//=============================
{
  private:
    static count_t		inst_counter_s;
    static TypeMeasuredFunc*	cur_func_ptr_s;

    map<string, vector<TypeMeasuredFunc*>, less<string> >	comparison_groups_;
    vector<TypeMeasuredFunc*>	measured_funcs_;
    vector<string>		command_line_;

    bool	detailed_measurement_report_flag_;
    bool	measurement_report_flag_;
    bool	tracing_report_flag_;

    ulong	total_iterations_;
    ulong	total_tests_;
    ulong	scale_per_calls_;
    float	discarding_threshold_;

    vector<string>	notes_;

    //---------------------------
    void	perform_measurement () const;
    void	perform_a_measurement () const;
    void	report_measurement () const; 

    ulong	get_total_tests () const;

    void	show_help1 (const string& exe_name_i = "<executable-file-name>") const;
    void	show_help2 () const;
    void	show_settings (string msg_i) const;
    void	show_notes (string msg_i = string()) const;
    void	show_command_line (); 
    void	show_logo (string msg_i); 
    void	show_version (string msg_i = string()) const;

    void	delete_it (); 

  public:
    //---------------------------
    // Constructor-0
    ClassManager();

    // Copy Constructor
    ClassManager(const ClassManager& inst_i);

    // Destructor
    ~ClassManager();
    //---------------------------
    void	read_command_line (int argc, char** argv); 
    bool	parse_command_line (int argc, char** argv); 
    bool	parse1_command_line (); 
    bool	parse2_command_line (); 
    bool	parse3_command_line (); 
    bool	parse4_command_line () const; 
    bool	parse5_command_line () const; 

    void	set_discarding_threshold (float discarding_threshold_i = DISCARDING_THRESHOLD_default);
    void	set_tests	(ulong tests_i = TOTAL_TESTS_default);
    void	set_iterations	(ulong iterations_i = TOTAL_ITERATIONS_PER_TEST_default);
    void	set_per_calls	(ulong per_calls_i = PER_CALLS_default);
    void	set_detailed_measurement_report_flag(bool flag_i = false);
    void	set_measurement_report_flag(bool flag_i = true);
    void	set_tracing_report_flag	(bool format_i = false);

    void 	add_measured_func (TypeMeasuredFunc * const func_i);
    void 	add_comparison_group (const string& group_name_i);
    void 	add_compared_func (
			const string&		group_name_i, 
			TypeMeasuredFunc*	const func_i
			);

    ulong	get_current_total_iterations () const;
    ulong	get_scale_per_calls () const;
    bool	get_measurement_report_flag() const;
    bool	get_detailed_measurement_report_flag() const;
    bool	get_tracing_report_flag () const;
    float	get_discarding_threshold () const;

    static bool		get_tracing_report_flag_S ();
    static float	get_discarding_threshold_S();

    TypeMeasuredFunc*	get_cur_func_ptr () const;

}; // class ClassManager
//=============================
extern ClassManager	perf_ins;
//=============================


//###################
#endif	// _PFMETER_H
//###################

//#######################################
//############# END OF FILE #############
//#######################################

// ------------------- C++ code : END ----------------------
// === File #6 of 14 : pfmeter.h ===========================
