// ReportTask.cpp: implementation of the ReportTask class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "resource.h"
#include "ReportTask.h"
#include "Report.h"
#include "AboutTask.h"

//////////////////////////////////////////////////////////////////////
// WindowClass Definitions
//////////////////////////////////////////////////////////////////////

class ReportTask::WindowClass
{
public:
	BOOL registered;
	HMODULE instance;
	TCHAR name[128];
	WindowClass(WNDPROC lpfnWndProc);
};

ReportTask::WindowClass::WindowClass(WNDPROC lpfnWndProc)
{
	WNDCLASSEX wcex;

	this->instance = GetModuleHandle(NULL);
	LoadString(instance, IDC_DININGPHILOSOPHERS, this->name, 128);

	wcex.cbSize = sizeof(WNDCLASSEX); 
	wcex.style = CS_HREDRAW | CS_VREDRAW;
	wcex.lpfnWndProc = lpfnWndProc;
	wcex.cbClsExtra = 0;
	wcex.cbWndExtra = 0;
	wcex.hInstance = this->instance;
	wcex.hIcon = LoadIcon(this->instance, 
							MAKEINTRESOURCE(IDI_DININGPHILOSOPHERS));
	wcex.hCursor = LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
	wcex.lpszMenuName = MAKEINTRESOURCE(IDC_DININGPHILOSOPHERS);
	wcex.lpszClassName = this->name;
	wcex.hIconSm = LoadIcon(this->instance, MAKEINTRESOURCE(IDI_SMALL));

	this->registered = RegisterClassEx(&wcex);
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

ReportTask::ReportTask()
{
	end_flag = false;
	this->activate();
}

ReportTask::~ReportTask()
{
	this->terminate();
	delete this->log_ptr;
	delete this->about_task_ptr;
}

//////////////////////////////////////////////////////////////////////
// Member Functions
//////////////////////////////////////////////////////////////////////

void ReportTask::body()
{
	char msg_buf[256];

	if (!initialize_window(SW_SHOWNORMAL)) {
		MessageBox(NULL, "Error creating main window", 
					"Error", MB_ICONERROR | MB_OK);
		return;
	}
	Report::make_versioned_string
		(msg_buf, 256, "Welcome to DiningPhilosophers %s", this->hInstance);
	log_ptr->print(msg_buf);

	this->suspended = false;
	message_loop();
}

void ReportTask::message_loop()
{
	ExitCode exit_code;
	HACCEL hAccelTable 
		= LoadAccelerators(hInstance, MAKEINTRESOURCE(IDC_DININGPHILOSOPHERS));
	bool done = false;
	
	while (!done) {
		// Start rendezvous
		Rendezvous accept(guarded(&print, !suspended), 
							&msg_entry, guarded(&quit, end_flag));
		switch (accept.entry_selector()) {
		case 0:
			// Accepted print
			// Ignore if the window has already been closed
			if (!end_flag) {
				SenderName sender = print.actual_parameter1(accept);
				MessageStr message = print.actual_parameter2(accept);
				char str_buf[256];
			
				_snprintf(str_buf, 256, "[From %s] %s", sender, message);
				log_ptr->print(str_buf);
			}
			break;
		case 1: {
			// Accepted msg_entry
			MSG &msg = msg_entry.actual_msg(accept);
			
			if (msg.message == WM_QUIT) {
				exit_code = (int)msg.wParam;
				done = true;
			}
			else if (!TranslateAccelerator(msg.hwnd, hAccelTable, &msg)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			break;
				}
		case 2: {
			// Accepted quit
			ExitCode &act_param = quit.actual_parameter1(accept);

			act_param = 0;
			DestroyWindow(hWnd);
			break;
				}
		}
		// End Rendezvous
	}
}

bool ReportTask::initialize_window(int cmdShow)
{
	// There is only one window_class object for the task type, shared by all
	// task objects
	static WindowClass volatile window_class(window_proc);

	if (!window_class.registered) {
		return false;
	}
	this->hInstance = window_class.instance;

	// Create the window
	TCHAR window_title[128];
	LoadString(window_class.instance, IDS_APP_TITLE, window_title, 128);
	this->hWnd 
		= CreateWindow
			((TCHAR*)window_class.name, window_title, 
			 WS_OVERLAPPEDWINDOW | WS_VSCROLL, CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, 
			 NULL, NULL, this->hInstance, NULL);
	if (this->hWnd == NULL) {
		return false;
	}
	this->log_ptr = new WindowLog(hWnd, 10000);
	this->about_task_ptr = NULL;
	this->suspended = false;
	this->control_menu = GetSubMenu(GetMenu(hWnd), 1);

	ShowWindow(this->hWnd, cmdShow);
	UpdateWindow(this->hWnd);
	return true;
}

Priority ReportTask::priority()
{
	return highest_Priority;	// Because it services the user interface
}

size_t ReportTask::virtual_sizeof()
{
	return sizeof(ReportTask);
}

// Window message handler
LRESULT CALLBACK ReportTask::window_proc
		(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	ReportTask *this_task = (ReportTask*)current_task();
	
	switch (message) {
	case WM_COMMAND: {
		int wmId = LOWORD(wParam); 
		// Parse the menu selections:
		switch (wmId)
		{
		case IDM_ABOUT: {
			AboutTask *about_task_ptr = this_task->about_task_ptr;

			if (about_task_ptr != NULL && about_task_ptr->is_active()) {
				// The dialog box is already on the screen
			}
			else {
				delete about_task_ptr;
				about_task_ptr = 
					new AboutTask(this_task->hInstance, this_task->hWnd);
			}
			break;
						}
		case IDM_EXIT:
			this_task->terminate_application();
			break;
		case IDM_SUSPEND:
			this_task->suspended = true;
			EnableMenuItem(this_task->control_menu, IDM_SUSPEND, MF_GRAYED);
			EnableMenuItem(this_task->control_menu, IDM_RESUME, MF_ENABLED);
			this_task->log_ptr->print("Suspended ...");
			break;
		case IDM_RESUME:
			this_task->suspended = false;
			EnableMenuItem(this_task->control_menu, IDM_SUSPEND, MF_ENABLED);
			EnableMenuItem(this_task->control_menu, IDM_RESUME, MF_GRAYED);
			break;
		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
		}
		break;
					 }
	case WM_CLOSE:
		this_task->terminate_application();
		break;
	case WM_SYSCOMMAND: {
		unsigned int uCmdType = wParam & 0xFFF0;
		switch (uCmdType) {
		case SC_CLOSE:
			this_task->terminate_application();
			break;
		default:
			return DefWindowProc(hWnd, message, wParam, lParam);
		}
		break;
						}
	case WM_PAINT: {
		if (!this_task->log_ptr->do_paint(hWnd)) {
			return DefWindowProc(hWnd, message, wParam, lParam);
		}
		break;
				   }
	case WM_VSCROLL: {
		int code = (int) LOWORD(wParam); 
		short int pos = (short int) HIWORD(wParam);
		
		if (!this_task->log_ptr->do_vscroll(hWnd, code, pos)) {
			return DefWindowProc(hWnd, message, wParam, lParam);
		}
		break;
					 }
	case WM_DESTROY:
		PostQuitMessage(0);
		break;
	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}
	return 0;
}

void ReportTask::terminate_application()
{
	this->end_flag = true;
	this->suspended = false;
	// Don't destroy the window yet
	log_ptr->print("Terminating application, please wait ...");
}


