/*
 *   @(#) MPEG-I Video Decoder 1.0 Demicron (demicron@demicron.com)
 *
 *   BufferedBitStream.java   2002-08-20
 *
 *   Copyright (C) 2002  Demicron
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/**
 *   BufferedBitStream reads from a BitBuffer and provide data to audio and video decoder
 */
class BufferedBitStream
{
	/**
	 *   Synchronizing start code
	 */
	public static final int SYNC_START_CODE = 0x000001;

	/**
	 *   The data queue to get data buffer
	 */
	private BufferQueue dataQueue;

	/**
	 *   The system queue to enqueue used buffer
	 */
	private BufferQueue systemQueue;

	/**
	 *   The buffer that currently in use
	 */
	private ByteBuffer bytebuffer = null;

	/**
	 *   The buffer for bit processing
	 */
	private int bitbuffer;

	/**
	 *   The remaining bit length in bitbuffer
	 */
	private int bitlength;

	/**
	 *   The next int data from the bytebuffer
	 */
	private int nextdata;

	/**
	 *   If the nextdata is valid
	 */
	private boolean available;

        private SystemDecoder systemDecoder;
        private boolean video;  // Null if not stream of videodecoder

	/**
	 *   Initializes the input buffered stream.
	 * Note: the BufferedBitStream should hold one empty buffer while constructing
	 */
	public BufferedBitStream (BufferQueue dataQueue, BufferQueue systemQueue, ByteBuffer bytebuffer, SystemDecoder systemDecoder, boolean video)
	{
		this.dataQueue = dataQueue;
		this.systemQueue = systemQueue;
                this.systemDecoder = systemDecoder;
                this.video = video;


		// The first bytebuffer should have len = 0
		// This is used to hack for speed and convenience in read8Bits()
		this.bytebuffer = bytebuffer;
		bitbuffer = bitlength = 0;
		nextdata = 0;
		available = false;
	}

        public boolean isStopped() {
          return dataQueue.isStopped;
        }

	/**
	 *   Reset the BufferedBitStream and its dataQueue
	 */
	void reset()
	{
		bitbuffer = bitlength = 0;
		nextdata = 0;
		available = false;
		if (bytebuffer != null)
                  bytebuffer.reset();
		// clear all the data in the dataQueue
		dataQueue.reset();
	}

	/**
	 * Shows the next MPEG-1 layer start code.
	 * @see	getCode()
	 * @return	the 32-bits start code
	 */
	public int showCode() throws InterruptedException
	{
		alignBits(8);
		while (showBits(24)!= SYNC_START_CODE)
			flushBits(8);
		return showBits(32);
	}

	/**
	 * Gets the next MPEG-1 layer start code.
	 * @see	showCode()
	 * @return	the 32-bits start code
	 */
	public int getCode() throws InterruptedException
	{
		alignBits(8);
		while (showBits(24)!= SYNC_START_CODE)
			flushBits(8);
		return getBits(32);
	}


	// The method "next_bits" checks whether the next "n" bits match the
	// "pattern". If so it returns "true"; otherwise "false".
	// Note: This method changes the bit "pointer" physically BUT NOT
	// logically !!! (Comment taken from io_tool....is it ok to use the last lines here?)
	//public final boolean next_bits(int pattern, int n) throws InterruptedException {
        //  return (showBits(n)==pattern);
		//if (bit_pos < n) get_long();
		//return (int)((longword >>> (bit_pos - n)) & ((1L << n) - 1L)) == pattern;
	//}

	/**
	 * Shows next nbit bits from the MPEG-1 system stream.
	 * @see	getBits(int)
	 * @param	nbit		the number of bits to show
	 * @return	the wanted bits in the Least Significant Bits of the int
	 */
	public int showBits(int nbit) throws InterruptedException
	{
		int bits = bitbuffer >>> (32 - nbit);
		if (nbit > bitlength) {
			bits |= show32Bits() >>> (32 + bitlength - nbit);
		}
		return bits;
	}

	/**
	 * Gets next nbit bits from the MPEG-1 system stream.
	 * @see showBits(int)
	 * @param	nbit		the number of bits to get
	 * @return	the wanted bits in the Least Significant Bits of the int
	 */
	public int getBits(int nbit) throws InterruptedException
	{
		int bits = bitbuffer >>> (32 - nbit);
		if (nbit <= bitlength) {
			bitlength -= nbit;
			bitbuffer <<= nbit;
		} else { // Additional bits needed
			bitbuffer = get32Bits();
			nbit -= bitlength;
			bitlength = 32 - nbit;
			bits |= bitbuffer >>> (bitlength);
			bitbuffer <<= nbit;
		}
		return bits;
	}

	/**
	 * Flushes nbit bits from the MPEG-1 system stream.
	 * @param	nbit		the number of bits to be flushed
	 */
	public void flushBits(int nbit) throws InterruptedException
	{
		if (nbit <= bitlength) {
			bitlength -= nbit;
			bitbuffer <<= nbit;
		} else {
			nbit -= bitlength;
			bitlength = 32 - nbit;
			bitbuffer = get32Bits() << nbit;
		}
	}

	/**
	 * Aligns the MPEG-1 system stream to the given boundary .
	 * @param	nbit		the number of bits to align the stream
	 */
	public void alignBits(int nbit) throws InterruptedException
	{
		flushBits(bitlength % nbit);
	}

	/**
	 *   Change to the next buffer ( for audio decoder )
	 */
	void nextBuffer() throws InterruptedException
	{
		systemQueue.enQueue(bytebuffer);

		bitlength = bitbuffer = 0;
		available = false;

		bytebuffer = dataQueue.deQueue();
                //syncAudio();
	}

	/**
	 *   Show the next 32 bit data from the bytebuffer
	 */
	private int show32Bits() throws InterruptedException
	{
		if (!available) {
			int b0 = read8Bits();
			int b1 = read8Bits();
			int b2 = read8Bits();
			int b3 = read8Bits();
			nextdata = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
			available = true;
		}
		return nextdata;
	}

	/**
	 *   Get the next 32 bit data from the bytebuffer
	 */
	private int get32Bits() throws InterruptedException
	{

		if (!available) {
			int b0 = read8Bits();
			int b1 = read8Bits();
			int b2 = read8Bits();
			int b3 = read8Bits();
			nextdata = (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
		}
		available = false;
		return nextdata;
	}

	/**
	 *   Read next byte data from the bytebuffer
	 */
	private int read8Bits() throws InterruptedException
	{
		while (bytebuffer.position >= bytebuffer.len) {
			// Get the data from queue
			systemQueue.enQueue(bytebuffer);

			bytebuffer = dataQueue.deQueue();
                        //syncAudio();
		}
		return (bytebuffer.buffer[bytebuffer.position++] & 0xff);
	}


        // Let the video sleepor speed up this time interval
        /*long audioDelay;
        public long popAudioDelay() {
          long tmp = audioDelay;
          audioDelay = 0;
          return tmp;
        }*/


        public long getPTS() {
          return bytebuffer.getPTS();
        }


        public void slowdownSTC(long diff) {
          systemDecoder.slowdownSTC(diff);
        }

        public SystemDecoder getSysDecoder() {
          return systemDecoder;
        }

        // Sync video on dts and audio on pts
    //    private void syncAudio() {
       /*   if (video && bytebuffer.getDTS()!=0 && systemDecoder.getAudioDelta()!=0) {
            long videoDelta = bytebuffer.getDTS()-System.currentTimeMillis();
            videoAudioDelta = systemDecoder.getAudioDelta()-videoDelta; // How much audio is ahead of video
            //System.out.println(":"+videoSync);
          } else if (!video && bytebuffer.getPTS()!=0) {
            systemDecoder.setAudioDelta(bytebuffer.getPTS()-System.currentTimeMillis());
          }*/


          // If audio is behind
          /*if (!video && bytebuffer.getPTS()!=0) {
            long diff = systemDecoder.getSTC()-bytebuffer.getPTS();
            if (diff>50)
              systemDecoder.slowdownSTC(diff);
          }*/













         /* if  (!video && bytebuffer.getPTS()!=0) {
            long ts = bytebuffer.getPTS()-800;
            if (ts>systemDecoder.getSTC())
              try {
                Thread.currentThread().sleep(400);//ts-systemDecoder.getSTC());
                System.out.println(1);
              } catch (Exception e) {}
            else
              systemDecoder.slowdownSTC(-(ts-systemDecoder.getSTC()));
          }*/















         /* if (video && bytebuffer.getPTS()!=0) {
            long ts = bytebuffer.getPTS()+500;
            if (ts>systemDecoder.getSTC())
              videoSync+=ts-systemDecoder.getSTC();
            else
              systemDecoder.slowdownSTC(-(ts-systemDecoder.getSTC()));
          } else if (!video && bytebuffer.getPTS()!=0) {
            long ts = bytebuffer.getPTS();
            if (ts>systemDecoder.getSTC())
              try {
                Thread.currentThread().sleep(ts-systemDecoder.getSTC());
              } catch (Exception e) {}
            else
              systemDecoder.slowdownSTC(-(ts-systemDecoder.getSTC()));
          }*/

          // Audio ahead => forward time (so video catches up)
          // Audio behind: slow down video

      /*    if (video && bytebuffer.getDTS()!=0) {
            long diff = bytebuffer.getDTS()-systemDecoder.getSTC()+700; // Sound seems to be buffered(and hence slower than video) -> speed up video reading
//            System.out.println(diff+":"+bytebuffer.getDTS()+":"+systemDecoder.getSTC());
            // diff>0 => video fr snabb
            //if (diff>0)
//            if (Math.abs(diff)<500)
              extraVideoSleep+=diff; // Only set global diff is ahead of time
  //          else
            //else

          } else if (!video && bytebuffer.getPTS()!=0) {
            long diff = bytebuffer.getPTS()-systemDecoder.getSTC(); // Sound seems to be buffered(and hence slower than video) -> speed up video reading
            if (diff>100) {
              try {
                Thread.currentThread().sleep(diff);
              } catch (Exception e) {}
            } else {
              systemDecoder.slowdownSTC(-diff);
           // }
//            System.out.println();
          }*/


      //  }


     /*     long ts = (video) ? bytebuffer.getDTS() : bytebuffer.getPTS();
          if (ts!=0) {
            if (video) pts+=500; // Hmmmm
            long diff = ts-systemDecoder.getSTC();
//System.out.println(video+":"+diff);
            if (diff>200) { // Slow down if ahead of system time clock
                if (video) extraVideoSleep+=diff; // Only set global diff is ahead of time
                try {
                       Thread.currentThread().sleep(diff);
                } catch (Exception e) {e.printStackTrace();}
            } else if (diff<-200) { // Lagging STC...decrease system time clock, which makes the other decoder sleep for a while...
              systemDecoder.slowdownSTC(-diff);
            }
          }
        }*/
}