/**	
	Company:		Shout Interactive
	Project:		Shout3D Core Implementation
	Class:			BindingTestPanel
	Date:			April 26, 1999
	Description:	Class for panel that shows a test for binding viewpoints & backgrounds
	(C) Copyright Shout Interactive, Inc. - 1997/1998/1999 - All rights reserved
 */
package applets;

import java.applet.*;
import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.Date;
import java.net.URL;
import shout3d.*;
import shout3d.core.*;

/**
 * Panel that tests the changing of viewpoint and background bindings.
 * 
 * Press the 'v' key to cycle through three viewpoints.
 * The change is made just by setting the isBound field of
 * the viewpoints and letting the library's mechanism respond
 * 
 * Press the 'b' key to change between 4 backgrounds, also through 
 * the binding mechanism
 * 
 * @author Paul Isaacs
 * @author Jim Stewartson
 * @author Dave Westwood
 */

import shout3d.*;
public class BindingTestPanel extends Shout3DPanel implements DeviceObserver {
	
	// The viewpoint and background nodes that will be bound
	Viewpoint vp0, vp1, vp2;
	Background bk0, bk1, bk2, bk3;

	//{{ Shout3DApplet methods
	/**
	 *  Constructor
	 */
	public BindingTestPanel(Shout3DApplet applet){
		super(applet);
	}		
	    
	/**
	 *
	 * This method is automatically called by the parent class Shout3DPanel
	 * at the correct time during initialize().
	 * 
	 * Subclasses should implement this to perform any custom initialization tasks.
	 */
    public void customInitialize() {
        
		// Register to observe the keyboard
		addDeviceObserver(this,"KeyboardInput", null);
		
		// Create 3 viewpoints and bind to the first
		createViewpointNodes();
		vp0.isBound.setValue(true);
		
		// Create 4 backgrounds and bind to the first
		createBackgroundNodes();
		bk0.isBound.setValue(true);
	}

	/**
	 *  Finalize
	 */
	protected void finalize() throws Throwable { 
		// Clean up by unregistering observers
		removeDeviceObserver(this,"KeyboardInput");
		super.finalize();
	}
	//}} Shout3DApplet methods
	
	
	//{{ DeviceObserver methods
	/**
	 * 
	 * When keyboard input is received:
	 * 
	 * If 'v' or 'V' cycles through viewpoints
	 * If 'b' or 'B' cycles through backgrounds
	 */
	public boolean onDeviceInput(DeviceInput di, Object userData) {
		if (!(di instanceof KeyboardInput))
			return false;
		
		KeyboardInput ki = (KeyboardInput) di;
		if (ki.which == KeyboardInput.PRESS) {
			if (ki.key == 'v' || ki.key == 'V') {
				// V key pressed. Cycle to next viewpoint.
				Viewpoint curVP = (Viewpoint)getCurrentBindableNode("Viewpoint");

				if (curVP == vp0)
					vp1.isBound.setValue(true);
				else if (curVP == vp1)
					vp2.isBound.setValue(true);
				else if (curVP == vp2)
					vp0.isBound.setValue(true);
				return true;
			}
			if (ki.key == 'b' || ki.key == 'B') {
				// B key pressed. Toggle between backgrounds
				Background curBG = (Background) getCurrentBindableNode("Background");

				if (curBG == bk0)
					bk1.isBound.setValue(true);
				else if (curBG == bk1)
					bk2.isBound.setValue(true);
				else if (curBG == bk2)
					bk3.isBound.setValue(true);
				else
					bk0.isBound.setValue(true);
				return true;
			}
		}
		return false;
	}
	//}} DeviceObserver methods

	// Field values for the viewpoints 
	float[] pos0 = { 0, 10, -75 };
	float[] pos1 = { 40, 10, -40 };
	float[] pos2 = { 0, 50, -10 };
	float[] rot0 = { 0, 1, 0, 3.14159f };
	float[] rot1 = { 0, 1, 0, 2.36f };
	float[] rot2 = { 1, 0, 0, -1.57079f };
	
	void createViewpointNodes() {
		// Create three viewpoints, add to the scene, 
		// and give them different POVs.
		vp0 = new Viewpoint(this);
		vp1 = new Viewpoint(this);
		vp2 = new Viewpoint(this);
		vp0.isBound.setValue(false);
		vp1.isBound.setValue(false);
		vp2.isBound.setValue(false);
		Node[] newKids = new Node[3];
		newKids[0] = vp0;
		newKids[1] = vp1;
		newKids[2] = vp2;
		if ((getScene() != null) && (getScene() instanceof Group)) {
			((Group)getScene()).addChildren(newKids);
		}
		else {
			throw new Shout3DException("BindingTestPanel got null scene");
		}
	
		// Set positions and orientations
		vp0.position.setValue(pos0);
		vp1.position.setValue(pos1);
		vp2.position.setValue(pos2);
		vp0.orientation.setValue(rot0);
		vp1.orientation.setValue(rot1);
		vp2.orientation.setValue(rot2);
	}

	void createBackgroundNodes(){
		// bk0 is the starting background, bk1 is one we make right here
		bk0 = (Background) getCurrentBindableNode("Background");
		
		ImageTexture bkTexture = new ImageTexture(applet);
		String[] urlVals1 = {"images/shared/sky_starry_bg.gif" };
		bkTexture.url.setValue(urlVals1 );
		bk1 = new Background(this);
		bk1.texture.setValue(bkTexture);
		
		ImageTexture bkTexture2 = new ImageTexture(applet);
		String[] urlVals2 = {"images/shared/partly_transparent_bg.gif"};
		bkTexture2.url.setValue(urlVals2);
		bk2 = new Background(this);
		bk2.texture.setValue(bkTexture2);
		
		bk3 = new Background(this);
		bk3.texture.setValue(bkTexture2);
		bk3.stretchToFit.setValue(true);
	}
}