package cern.lhcias.csgui.Services;

import cern.lhcias.csgui.interfaces.*;
import cern.lhcias.csgui.Drivers.*;
import cern.lhcias.csgui.Utils.*;
import java.util.*;
import MD2SDriver;

/**
*   This class is a DataServer class loader. Given the name of the driver,
*   it will automatically create an instance of the DataServer and do the
*   necessary configuration. So all the rest of
*   the interface doesn't need to know about the existing Drivers.
*/
public class DataServerManager {
	static int nb_drivers = 3;
	static DataServer drivers[] = new DataServer[nb_drivers];
    static Vector aliases;
	
	Vector     myDataServers;
	DataServer currentDataServer;
	static boolean    globalRunningStatus = false;

    public DataServerManager() {
        myDataServers = new Vector();
    }

    /** Load the list of aliases for Drivers names
    */
    static public void loadAlias(String aliasAddress) {
        try {
            aliases = MyUtils.ReadRemoteValueFile(aliasAddress);
        } catch (Exception  e) {}
    }

    /** 
    *   Given the configuration String of a driver, this method will check wether the
    *   DataServer named ProcessName already exists or not. If it exists, the DataServer is
    *   returned. If not, a new DataServer corresponding to the config is instanciated.
    *   @param config the configuration string of the DataServer
    *   @see cern.lhcias.csgui.interfaces.DataServer#setConfig
    */
    public DataServer setNewDataServer(String config) {
        String ProcessName = config.substring(0,config.indexOf(';'));
		String config2 = config.substring(config.indexOf(';') + 1);
		String Protocol = config2.substring(0,config2.indexOf(';'));

        DataServer ds = getDataServer(ProcessName);
        if (ds != null) {
            currentDataServer = ds;
            return ds;
        }
        ds = createDataServer(Protocol);
        if (ds != null) {
            ds.setConfig(config);
            myDataServers.addElement(ds);
            currentDataServer = ds;
        }
        return ds;
    }
    
    /** 
    *   Given the name and the protocol of a driver, this method will check wether the
    *   DataServer named ProcessName already exists or not. If it exists, the DataServer is
    *   returned. If not, a new DataServer corresponding to the config is instanciated.
    *   @see cern.lhcias.csgui.interfaces.DataServer#setConfig
    */
    public DataServer setNewDataServer(String ProcessName, String Protocol, String Host, int Port) {
        DataServer ds = getDataServer(ProcessName);
        if (ds != null) {
            currentDataServer = ds;
            return ds;
        }
        ds = createDataServer(Protocol);
        if (ds != null) {
            ds.setProcessName(ProcessName);
            ds.setReadHost(Host);
            ds.setReadPort(Port);
            myDataServers.addElement(ds);
            currentDataServer = ds;
        }
        return ds;
    }
    
    /** 
    *   Given the name and the protocol of a driver, this method will check wether the
    *   DataServer named ProcessName already exists or not. If it exists, the DataServer is
    *   returned. If not, a new DataServer corresponding to the config is instanciated.
    *   Some properties of the DataServer are set:
    *   
    *   @see cern.lhcias.csgui.interfaces.DataServer#setConfig
    */
    public DataServer setNewDataServer(String ProcessName, String Protocol, String Host, 
                int Port, TagArrays tc, boolean TagsInURL, long frequency) {
        DataServer ds = getDataServer(ProcessName);
        if (ds != null) {
            currentDataServer = ds;
        } else {
            ds = createDataServer(Protocol);
            if (ds != null) {
                ds.setProcessName(ProcessName);
                ds.setReadHost(Host);
                ds.setReadPort(Port);
                myDataServers.addElement(ds);
                currentDataServer = ds;
            }
        }
        if (ds != null) {
	        ds.setTagArrays(tc);
            ds.setTagsSubscription(TagsInURL);
			ds.setPoolingInterval(frequency);
        }
        return ds;
    }
    
    /** set the current DataServer, which is usually the last driver created.
    *   Normally, this is done when setting a new driver by the setNewDriver method.
    */
    public void setCurrentDataServer(DataServer ds) {
        if (ds != null) currentDataServer = ds;
    }
    
    /** returns a vector of DataServers containing all the
    *   managed DataServers.
    */
    public Vector getDataServers() {
        return myDataServers;
    }
    
    /** Adds a DataServer to the list. This methods is used for particular DataServers which
    *   can't be configured with the setNewDriver method. It is assumed that the DataServer
    * has been properly configured before calling this method.
    */
    public void addDataServer(DataServer ds) {
        if (ds != null) {
            myDataServers.addElement(ds);
            currentDataServer = ds;
        }
    }
    
    /** returns the DataServer names ProcessName.
    *   If no such DataServer exists, null is returned.
    */
    public DataServer getDataServer(String ProcessName) {
        for (int i=0; i<myDataServers.size(); i++) {
            if (((DataServer) myDataServers.elementAt(i)).getProcessName().equals(ProcessName))
                return (DataServer) myDataServers.elementAt(i);
        }
        return null;
    }
    
    public DataServer getCurrentDataServer() {
        return currentDataServer;
    }
    
	public void startAllDataServers() {
	    for (int i=0; i< myDataServers.size(); i++) {
	        ((DataServer) myDataServers.elementAt(i)).enable();
		}
		globalRunningStatus = true;
	}
	
	public void stopAllDataServers() {
	    for (int i=0; i< myDataServers.size(); i++) {
	        ((DataServer) myDataServers.elementAt(i)).disable();
		}
		globalRunningStatus = false;
	}
	
	public static boolean getGlobalRunningStatus() {
	    return globalRunningStatus;
	}
		
	/**
	* 	need to be called once at the setup of the system
	*/
	static public void setup() {
		drivers[0] = new HTTPDriver();
		drivers[1] = new MD2SDriver();
		drivers[2] = new SocketDriver();
	}

    /**
        Returns all the possible DataServer types as a Vector of Strings
    */
    static public Vector getPossibleProtocols() {
        Vector types = new Vector();
        for (int i=0; i < nb_drivers; i++) {
            Vector stypes = drivers[i].getPossibleProtocols();
            for (int j=0; j<stypes.size(); j++)
                types.addElement(stypes.elementAt(j));
        }
        return(types);
    }

    /** Checks wether a given protocol is implemented by a DataServer.
    * Returns true if type exists.
    * @param type the type to check
    */
    static public boolean isProtocolImplemented(String protocol) {
        for (int i=0; i < nb_drivers; i++) {
			if (drivers[i].doYouImplement(protocol))
			    return true;
		}
	    return false;
	}

	/** Load a class named "name" and creates a new instance of it.
	*   The resulting object is returned. In case of error, a null is returned.
	*   @param name The name of the class to load
	*/
	static Object loadClass(String name) {

        Class c = null;
        try {
            c = Class.forName(name);
        }
        catch(ClassNotFoundException e) {
       	    System.out.println("Class not found: " + e);
        }

        if (c != null) {
          try {
              return c.newInstance();
          } catch (Exception e) {
       	      System.out.println("FrameMain: error: " + e);
          }
    	}
    	return null;
    }

    /** Find the correct DataServer for a given type, create an instance
    * of this DataServer and returns it. In case of error, a null is returned.
    * If an alias list has been loaded it is used to access the correct driver.
    * In case the type of the DataServer is unknown, we try to
	* locate a class which name is the type of the DataServer.
    * @param type the type of the DataServer
    */
	static public DataServer createDataServer(String protocol) {

        if (aliases != null) protocol = MyUtils.getAlias(protocol, aliases);

		for (int i=0; i<nb_drivers; i++)
			if (drivers[i].doYouImplement(protocol)) {
				switch (i) {
				case 0:
					HTTPDriver httpdriver = new HTTPDriver();
					return(httpdriver);
                case 1:
                    MD2SDriver md2d = new MD2SDriver();
                    return(md2d);
                case 2:
                    SocketDriver sd = new SocketDriver();
                    return(sd);
				default:
				}
		}

		// We didn't find a predefined class which implements type.
		// So we are looking for a class named type.
		return((DataServer) loadClass(protocol));
	}
}
