/*Data Access Library. v. 1.1.
 *Visual interface to code generator.
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.wizzard;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.sql.*;
import java.util.*;
import JProjects.eab.data.wizzard.*;


import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.border.*;

/**Visual interface of Data Access Builder.
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
*/
public class Database extends JFrame implements ActionListener,ListSelectionListener,ItemListener {
    private  JMenuItem _smi,_nmi;
    private  Connection _con = null;
    private DBMap _map;
    private  String _mapFile = null;

  //Dynamic GUI
    private JTabbedPane _center = null;
    private  GridBagLayout _lm;
    private  int _curPan;
    private  JList _tbls,_cols,_selCols;
    private  JTextField _tblAlias,_linkString;
    private  JTextField _colType,_fldName;
    private  JCheckBox _isPK,_isNull;
    private  JLabel _tblName;
    private  JTextField _pckgName,_clBaseName,_baseDir;
    private  JCheckBox _isRO;
  
    private static final String _fs = System.getProperty("file.separator");
    private static Properties prop;
    private static final String PROPERTIES = "Database.properties";

    //DBMap.dirty flag's values
    private static final int CLEAN = 0;
    private static final int NEW = 1;
    private static final int DIRTY = 2;

  public Database() {
    super("Database to Java Classes Mapping");
    init();
    setSize(600,400);
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation((d.width-600)/2,(d.height-400)/2);
  }

  public void actionPerformed(ActionEvent ev) {
    String com = ev.getActionCommand();
    if (com.equals("NEW"))
      doNewMap();
    else if (com.equals("OPEN"))
      doOpenMap();
    else if (com.equals("SAVE"))
      saveMap();
    else if (com.equals("CONNECT"))
      doConnect();
    else if (com.equals("TOSEL")) {
      Object[] selCols = _cols.getSelectedValues();
      String tN = (String)_tbls.getSelectedValue();
      for(int i=0;i<selCols.length;i++)
	_map.selCol(tN,(String)selCols[i]);
      _cols.setListData(_map.getUnsTblCols(tN));
      _selCols.setListData(_map.getSelTblCols(tN));
      _colType.setText("");
      _fldName.setText("");
    } else if (com.equals("FROMSEL")) {
      int sel = _selCols.getSelectedIndex();
      if (sel>=0) {
	String tN = (String)_tbls.getSelectedValue();
	String selCol = (String)_selCols.getModel().getElementAt(sel);
	_map.deselCol(tN,selCol);
	_cols.setListData(_map.getUnsTblCols(tN));
	_selCols.setListData(_map.getSelTblCols(tN));
	_colType.setText("");
	_fldName.setText("");
      }
    } else
      doExit();
  }

    private class MyDocListener implements DocumentListener {
	JTextComponent src;

	MyDocListener(JTextComponent source) {
	    src=source;
	}

	public void changedUpdate(DocumentEvent ev) {
	    updateMap();
	}

	public void insertUpdate(DocumentEvent ev) {	    
	    updateMap();
	}

	public void removeUpdate(DocumentEvent ev) {
	    updateMap();
	}

	void updateMap() {
	    if (src==_tblAlias && _tbls.getSelectedValue()!=null) {
		String tN = (String)_tbls.getSelectedValue();
		if (!_map.getTblAlias(tN).equals(_tblAlias.getText()))
		    _map.setTblAlias(tN,_tblAlias.getText());
	    } else if (src==_fldName && _tbls.getSelectedValue()!=null && _selCols.getSelectedValue()!=null) {
		String tN = (String)_tbls.getSelectedValue();
		String fN = (String)_selCols.getSelectedValue();
		if (!_map.getFieldName(tN,fN).equals(_fldName.getText())) 
		    _map.setFieldName(tN,fN,_fldName.getText());
	    }
	}
    }

  public void itemStateChanged(ItemEvent ev) {
    boolean isSel = ev.getStateChange()==ev.SELECTED;
    if (ev.getItemSelectable()==_isPK && _selCols.getSelectedValue()!=null) {
      _map.setPK((String)_tbls.getSelectedValue(),(String)_selCols.getSelectedValue(),isSel);
    }
    else if (ev.getItemSelectable()==_isRO)
      _map.setRO(isSel);
  }

    public void valueChanged(ListSelectionEvent e) {
	String tN = (String)_tbls.getSelectedValue();
	boolean adj = e.getValueIsAdjusting();

	if (e.getSource()==_tbls && tN!=null && !adj) {
	    String ta = _map.getTblAlias(tN);
	    _tblAlias.setText((ta==null?"":ta));
	    _cols.setListData(_map.getUnsTblCols(tN));
	    _selCols.setListData(_map.getSelTblCols(tN));
	    _colType.setText("");
	    _fldName.setText("");
	    _tblName.setText(tN);
	} else if (e.getSource()==_selCols && _selCols.getSelectedValue()!=null &&!adj) {
	    String selItem = (String)_selCols.getSelectedValue();
	    _colType.setText(_map.getDTString(tN,selItem));
	    _fldName.setText(_map.getFieldName(tN,selItem));
	    _isPK.removeItemListener(this);
	    _isPK.setSelected(_map.isPK(tN,selItem));
	    _isPK.addItemListener(this);
	    _isNull.setSelected(_map.isNull(tN,selItem));
	}
    }

    public static void main(String[] args) {
	new Database().show();
    }

  private void doNewMap() {
      if (saveMap() ) {
	  new MapWizzard(this).show();
	  if (_map.dirty==NEW)
	      initGUI();
      }
  }
  
  private void doConnect() {
      if (  saveMap() ) {
	  new ConnectionPanel(this).show();
	  if (_con != null)
	      _nmi.setEnabled(true);
      }
  }


  private void doExit() {
    if (!saveMap() )
	return;
    System.exit(0);
  }

  private void doOpenMap() {
      if (saveMap()) {
	  FileDialog fd = new FileDialog(this,"Open Map",FileDialog.LOAD);
	  fd.show();
	  if (fd.getFile()!=null) {
	      String mapFile = fd.getDirectory()+_fs+fd.getFile();
	      try {
		  FileInputStream fi = new FileInputStream(mapFile);
		  ObjectInputStream ois = new ObjectInputStream(fi);
		  DBMap map=new DBMap();
		  map.readExternal(ois);
		  ois.close();
		  fi.close();
		  if (!verifyMap(map))
		      map.dirty = DIRTY;
		  else
		      map.dirty = CLEAN;
		  _mapFile = mapFile;
		  _map = map;
		  initGUI();
	      } catch (Exception ex) {
		  handleException(ex);
	      }
	  }
      }
  }

    private void handleException(Exception ex) {
	JOptionPane.showMessageDialog(this,ex.getMessage(),"Exception",JOptionPane.ERROR_MESSAGE,null);
	System.err.println(ex);
	ex.printStackTrace(System.err);
    }

  private boolean saveMap() {
      if (_map == null )
	  return true;
      if (!_map.getSQLLink().equals(_linkString.getText()))
	  _map.setSQLLink(_linkString.getText());
      if (!_map.getPckg().equals(_pckgName.getText()))
	  _map.setPckg(_pckgName.getText());
      if (!_map.getBaseName().equals(_clBaseName.getText())) 
	  _map.setBaseName(_clBaseName.getText());
      if (!_map.getTargetDir().equals(_baseDir.getText())) 
	  _map.setTargetDir(_baseDir.getText());
      if (_isRO.isSelected() && !_map.isRO() || _map.isRO() && !_isRO.isSelected())
	  _map.setRO(_isRO.isSelected());
      if (_map.dirty == CLEAN)
	  return true;

      int dlgRes = JOptionPane.showConfirmDialog(this,"Save this mapping?","Map needs to be saved",JOptionPane.YES_NO_CANCEL_OPTION);
      if (dlgRes == JOptionPane.CANCEL_OPTION)
	  return false;
      if (dlgRes == JOptionPane.YES_OPTION) {
    String mapFile = null;
    int chck = _map.checkConsistency(),fl;
    if (chck != OK) {
      String msg="Incomplete map. Missed:\n";
      if ((fl=chck&LINK_MISS)>0)
	msg+="SQL Link String\n";
      if ((fl=chck&PACK_MISS)>0)
	msg+="Package name\n";
      if ((fl=chck&BASE_MISS)>0)
	msg+="Base name";
      if ((fl=chck&SELCOL_MISS)>0)
	  msg+="No Column selected\n";
      JOptionPane.showMessageDialog(this,msg,"Error",JOptionPane.ERROR_MESSAGE);
      return false;
    }
    if ((fl=_map.dirty&NEW)>0) {
      FileDialog fd = new FileDialog(this,"Save New Map",FileDialog.SAVE);
      fd.show();
      if (fd.getFile()!=null) {
	mapFile = fd.getDirectory()+_fs+fd.getFile();
      }
    } else
      mapFile = _mapFile;
    if (mapFile!=null)
      try {
	  _map.arrangePK();
	DCGenerator gen = new DCGenerator(_map.getAllTbls(),_map.getAliases(),_map.getFullBaseName(),_map.getAllCols(),_map.getAllFlds(),_map.getDTypes(),_map.getAllNull(),_map.getAllPK(),_map.getSQLLink());
	gen.setReadOnly(_map.isRO());
	String tgtDir = (_map.getTargetDir().length()==0?".":_map.getTargetDir());
	gen.generate(tgtDir);
	FileOutputStream fo = new FileOutputStream(mapFile);
	ObjectOutputStream oos = new ObjectOutputStream(fo);
	_map.dirty = CLEAN;
	_map.writeExternal(oos);
	oos.flush();
	oos.close();
	fo.close();
	_mapFile = mapFile;
	JOptionPane.showMessageDialog(this,"Classes has been generated.\nMap saved.","Information",JOptionPane.INFORMATION_MESSAGE);
      } catch (IOException ex) {
	handleException(ex);
	return false;
      }
      }
      return true;
  }

  private boolean verifyMap(DBMap map) throws SQLException {
    boolean res = true;
    if (_con != null) {
      DatabaseMetaData md = _con.getMetaData();
      for (Enumeration tblEls=map.getTables();tblEls.hasMoreElements();) {
	String tN = (String)tblEls.nextElement();
	String tblN = (tN.indexOf('.')>0?tN.substring(tN.indexOf('.')+1):tN);
	String schN = (tN.indexOf('.')>0?tN.substring(0,tN.indexOf('.')):"");
	ResultSet rs = md.getTables(null,schN,tblN,new String[] {"TABLE"});
	if (!rs.next()) {
	    int dlgRes = JOptionPane.showConfirmDialog(this,"Table "+tN+" is not present in DB: delete it from the map?","Error",JOptionPane.YES_NO_OPTION);
	  if (dlgRes==JOptionPane.YES_OPTION) {
	    map.delTbl(tN);
	    res = false;
	  }
	} else {
	  // see for columns
	  rs = md.getColumns(null,schN,tblN,"");
	  Hashtable dbCols = new Hashtable();
	  while(rs.next()) 
	    dbCols.put(rs.getString(4),new int[] {rs.getInt(5),rs.getInt(9),rs.getInt(11)});
	  rs.close();
	  Vector vec = map.getCols(tN);
	  for (int i=0;i<vec.size();i++) {
	    String colN = (String)vec.elementAt(i);
	    if (!dbCols.containsKey(colN)) {
	      res = false;
	      JOptionPane.showMessageDialog(this,"There is no column "+colN+" in the "+tN+" ! Deleting ..","Information",JOptionPane.INFORMATION_MESSAGE);
	      map.delCol(tN,colN);
	    }
	    else {
	      int[] dt = (int[])dbCols.get(colN);
	      int[] cDt = map.getColTypeNull(tN,colN);
	      if (dt[0]!=cDt[0] || dt[1]!=cDt[1] || dt[2]!=cDt[2]) {
		res = false;
		int dlgRes=JOptionPane.showConfirmDialog(this,"Type or nullable property of "+colN+" column of the "+tN+" table had been changed! Update mapping?","Error",JOptionPane.YES_NO_OPTION);
		if (dlgRes==JOptionPane.YES_OPTION)
		    map.setCol(tN,colN,dt[0],dt[1],dt[2]);
	      }
	    }
	  }
	  //check for added cols
	  if (map.getColCount(tN)<dbCols.size()) {
	    res = false;
	    JOptionPane.showMessageDialog(this,"New columns had been added to the "+tN+" ! Updating ..","Information",JOptionPane.INFORMATION_MESSAGE);
	    for (Enumeration addCols=dbCols.keys();addCols.hasMoreElements();){
	      String acN = (String)addCols.nextElement();
	      if (!map.hasCol(tN,acN)) {
		int[] dt= (int[])dbCols.get(acN);
		map.addCol(tN,acN,dt[0],dt[1],dt[2]);
	      }
	    }
	  }
	  //check for PK 
	  rs = md.getPrimaryKeys(null,schN,tblN);
	  while(rs.next()) {
	    String pkN = rs.getString(4);
	    if (!map.isPK(tN,pkN)) {
	      res = false;
	      JOptionPane.showMessageDialog(this,"Primary key of the "+tN+" table had been changed! Updating ..","Information",JOptionPane.INFORMATION_MESSAGE);
	      map.setPK(tN,pkN);
	    }
	  }
	}  
      }
    }
    return res;
  }

  private void init() {
    JMenuBar mb = new JMenuBar();
    JMenu m = new JMenu("File");
    JMenuItem mi = new JMenuItem("Connect to DB..");
    mi.setActionCommand("CONNECT");
    mi.addActionListener(this);
    m.add(mi);
    _nmi = new JMenuItem("New map..");
    _nmi.setActionCommand("NEW");
    _nmi.addActionListener(this);
    m.add(_nmi);
    _nmi.setEnabled(false);
    mi = new JMenuItem("Open map..");
    mi.setActionCommand("OPEN");
    mi.addActionListener(this);
    m.add(mi);
    _smi = new JMenuItem("Save map");
    _smi.setActionCommand("SAVE");
    _smi.addActionListener(this);
    m.add(_smi);
    _smi.setEnabled(false);
    mi = new JMenuItem("Exit");
    mi.setActionCommand("EXIT");
    mi.addActionListener(this);
    m.add(mi);
    mb.add(m);
    setJMenuBar(mb);
    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
	doExit();
      }
    });
  }

  private void initGUI() {
      boolean firstCall = _center==null;
    buildGUI();
    //display data from _map
    Vector tblData = new Vector();
    for(Enumeration tbls = _map.getTables();tbls.hasMoreElements();)
      tblData.addElement(tbls.nextElement());
    _tbls.setListData(tblData);
    _linkString.setText(_map.getSQLLink());
    if (_map.getTblCount()==1) {
      _linkString.setEnabled(false);
      _isRO.setSelected(_map.isRO());
      _isRO.setEnabled(true);
    }
    else {
      _linkString.setEnabled(true);
      _isRO.setSelected(true);
      _isRO.setEnabled(false);
      if (!_map.isRO())
	  _map.setRO(true);
    }
    _colType.setText("");
    _fldName.setText("");
    _pckgName.setText(_map.getPckg());
    _clBaseName.setText(_map.getBaseName());
    _baseDir.setText(_map.getTargetDir());
    String tN = (String)tblData.elementAt(0);
    _tblName.setText(tN);
    _tblAlias.setText(_map.getTblAlias(tN));
    Vector cols = _map.getUnsTblCols(tN);
    _cols.setListData(cols);
    cols = _map.getSelTblCols(tN);
    _selCols.setListData(cols);
    _tbls.removeListSelectionListener(this);
    _tbls.setSelectedIndex(0);
    _tbls.addListSelectionListener(this);
    _center.setSelectedIndex(0);
    //    if (firstCall) {
	getContentPane().validate();
	//    } else 
	//	getContentPane().repaint();
  }

  private void buildGUI() {
    if (_center==null) {
      _center = new JTabbedPane();
      
      _lm=new GridBagLayout();
      _center.addTab("Tables",crTblPan());
      _center.addTab("Columns",crColPan());
      _center.addTab("Classes",crClsPan());
      getContentPane().add(_center,BorderLayout.CENTER);
      _smi.setEnabled(true);
    }
  }

  private JPanel crTblPan() {
    JPanel res = new JPanel(_lm);
    JLabel lab = new JLabel("Tables");
    addCell(lab,res,1,0);
    lab = new JLabel("Table's alias");
    addCell(lab,res,2,0);
    _tbls = new JList();
    _tbls.addListSelectionListener(this);
    addCell(new JScrollPane(_tbls),res,3,2);
    _tblAlias=new JTextField();
    _tblAlias.getDocument().addDocumentListener(new MyDocListener(_tblAlias));
    addCell(_tblAlias,res,2,0);
    addCell(new JLabel(),res,2,0);
    lab = new JLabel("SQL Link String");
    addCell(lab,res,2,0);
    _linkString = new JTextField();
    addCell(_linkString,res,2,0);
    return res;
  }

  private JPanel crColPan() {
    JPanel res = new JPanel(_lm);
    _tblName = new JLabel("",SwingConstants.CENTER);
    addCell(_tblName,res,2,0);
    JLabel lab = new JLabel("Available");
    addCell(lab,res,5,2);
    lab = new JLabel("Selected");
    addCell(lab,res,2,0);
    _cols = new JList();
    _cols.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    addComponent(new JScrollPane(_cols),res,2,3);
    _selCols = new JList();
    _selCols.addListSelectionListener(this);
    addComponent(new JScrollPane(_selCols),res,GridBagConstraints.REMAINDER,3);
    JButton b = new JButton(">");
    b.setActionCommand("TOSEL");
    b.addActionListener(this);
    addCell(b,res,5,2);
    b = new JButton("<");
    b.setActionCommand("FROMSEL");
    b.addActionListener(this);
    addCell(b,res,2,0);
    lab = new JLabel("Column's information",SwingConstants.CENTER);
    addCell(lab,res,2,0);
    lab = new JLabel("Type");
    addCell(lab,res,1,0);
    _colType = new JTextField();
    _colType.setEditable(false);
    addCell(_colType,res,1,0);
    lab = new JLabel("Field Name");
    addCell(lab,res,1,0);
    _fldName = new JTextField();
    _fldName.getDocument().addDocumentListener(new MyDocListener(_fldName));
    addCell(_fldName,res,2,0);
    _isPK = new JCheckBox("Is PK");
    _isPK.addItemListener(this);
    addCell(_isPK,res,5,2);
    _isNull = new JCheckBox("Is Nullable");
    _isNull.setEnabled(false);
    addCell(_isNull,res,2,0);
    return res;
  }

  private JPanel crClsPan() {
    JPanel res = new JPanel(_lm);
    JLabel lab = new JLabel("Package");
    addCell(lab,res,1,0);
    _pckgName = new JTextField();
    addCell(_pckgName,res,2,0);
    lab = new JLabel("Base Name");
    addCell(lab,res,4,0);
    _clBaseName = new JTextField();
    addCell(_clBaseName,res,2,0);
    lab = new JLabel("Target Directory");
    addCell(lab,res,4,0);
    _baseDir = new JTextField();
    addCell(_baseDir,res,2,0);
    _isRO = new JCheckBox("Is read only");
    addCell(_isRO,res,2,0);
    return res;
  }

  private void addCell(Component com,Container con,int var,int subv) {
    switch(var) {
    case 2: //last col
	addComponent(com,con,GridBagConstraints.REMAINDER,-1);
      break;
    case 3: // cell has subv lines in height
	addComponent(com,con,-1,subv);
      break;
    case 4: // cell occupies space next to previous
	addComponent(com,con,GridBagConstraints.RELATIVE,-1);
      break;
    case 5: //cell has subv cols in width
	addComponent(com,con,subv,-1);
	break;
    default:
	addComponent(com,con,1,-1);
    }
  }

    private void addComponent(Component com,Container con,int gw,int gh) {
	GridBagConstraints cs = new GridBagConstraints();
	cs.insets = new Insets(2,2,2,2);
	if (gw>=0) {
	    cs.weightx = 1;
	    cs.gridwidth = gw;
	}
	if (gh>=0) {
	    cs.weighty=1;
	    cs.gridheight = gh;
	}
	cs.fill = cs.BOTH;
	con.add(com,cs);
    }

  //Connect to DB dialog
  private class ConnectionPanel extends JDialog implements ActionListener {
    private JTextField _drvTF;
    private JTextField _urlTF;
    private JTextField _usrTF;
    private JPasswordField _pwdTF;

    public ConnectionPanel(Frame par) {
      super(par,"Connecting to Database",true);
      init();
      setSize(400,300);
      Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
      setLocation((d.width-400)/2,(d.height-300)/2);
    }
    
    public void actionPerformed(ActionEvent ev) {
      if (ev.getActionCommand().equals("OK")) {
	if (_urlTF.getText().length()==0) {
	    JOptionPane.showMessageDialog((Frame)getParent(),"Supply URL to connect to","Error",JOptionPane.ERROR_MESSAGE);
	  return;
	}
	try {
	  if (_drvTF.getText().length()>0)
	    Class.forName(_drvTF.getText());
	  if (_usrTF.getText().length()>0 && _pwdTF.getPassword()!=null)
	    _con = DriverManager.getConnection(_urlTF.getText(),_usrTF.getText(),new String(_pwdTF.getPassword()));
	  else
	    _con = DriverManager.getConnection(_urlTF.getText());
	} catch (Exception ex) {
	  System.out.println(ex.getMessage());
	  ex.printStackTrace();
	  JOptionPane.showMessageDialog((Frame)getParent(),"Couldn't connect.","Exception",JOptionPane.ERROR_MESSAGE);
	  return;
	}
      }
      this.dispose();
    }

    private void init() {
      GridBagConstraints cs = new GridBagConstraints();
      GridBagLayout lm = new GridBagLayout();
      Insets ins = new Insets(2,2,2,2);
      JPanel center = new JPanel(lm);
      cs.fill = cs.HORIZONTAL;
      cs.insets = ins;
      cs.weightx=1;
      JLabel la = new JLabel("Driver");
      lm.setConstraints(la,cs);
      center.add(la);
      cs.gridwidth=cs.REMAINDER;
      _drvTF = new JTextField();
      lm.setConstraints(_drvTF,cs);
      center.add(_drvTF);
      cs.gridwidth=cs.RELATIVE;
      la = new JLabel("URL");
      lm.setConstraints(la,cs);
      center.add(la);
      cs.gridwidth=cs.REMAINDER;
      _urlTF = new JTextField();
      lm.setConstraints(_urlTF,cs);
      center.add(_urlTF);
      cs.gridwidth=cs.RELATIVE;
      la = new JLabel("User");
      lm.setConstraints(la,cs);
      center.add(la);
      cs.gridwidth=cs.REMAINDER;
      _usrTF = new JTextField();
      lm.setConstraints(_usrTF,cs);
      center.add(_usrTF);
      cs.gridwidth=cs.RELATIVE;
      la = new JLabel("Password");
      lm.setConstraints(la,cs);
      center.add(la);
      cs.gridwidth=cs.REMAINDER;
      _pwdTF=new JPasswordField();
      _pwdTF.setEchoChar('*');
      lm.setConstraints(_pwdTF,cs);
      center.add(_pwdTF);
      this.getContentPane().add(center,BorderLayout.CENTER);
      JPanel south = new JPanel();
      JButton b = new JButton("Connect");
      b.addActionListener(this);
      b.setActionCommand("OK");
      south.add(b);
      b = new JButton("Cancel");
      b.addActionListener(this);
      south.add(b);
      this.getContentPane().add(south,BorderLayout.SOUTH);
      south.setBorder(new EtchedBorder());
      if (prop==null) try {
	  prop = new Properties();
	  File propFile = new File(Database.PROPERTIES);
	  InputStream cfg=null;
	  if (propFile.exists()) {
	      cfg = new FileInputStream(propFile);
	  } else { //try from package resource
	      cfg = Database.class.getResourceAsStream(Database.PROPERTIES);
	  }
	  if (cfg!=null) {
		  prop.load(cfg);
		  cfg.close();
	  }
	  
      } catch (IOException ex) {
	  ex.printStackTrace();
      }
      String val = prop.getProperty("DRIVER");
      if (val!=null)
	  _drvTF.setText(val);
      val = prop.getProperty("URL");
      if (val!=null)
	  _urlTF.setText(val);
      val = prop.getProperty("USER");
      if (val!=null)
	  _usrTF.setText(val);
      val = prop.getProperty("PASSWORD");
      if (val!=null)
	  _pwdTF.setText(val);
    }

  }

  //Select table(s) dialog.
  private class MapWizzard extends JDialog implements ActionListener {
    private JTextField _schemaMsk,_tblMsk;
    private JList _tbls;
    private Vector _schemas;

    public MapWizzard(Frame par) {
      super(par,"Database Map Wizzard",true);
      init();
      setSize(400,400);
      Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
      setLocation((d.width-400)/2,(d.height-400)/2);
    }

    public void actionPerformed(ActionEvent ev) {
      String com = ev.getActionCommand();
      if (com.equals("RETRIEVE"))
	doRetrieve();
      else if (com.equals("OK")) {
	int[] si = _tbls.getSelectedIndices();
	if (si.length>0)
	  doMap();
      }
      else
	this.dispose();
    }
    
    private void doRetrieve() {
      //retrieve table names
      try {
	DatabaseMetaData md = _con.getMetaData();
	boolean isSchema = _schemaMsk.getText().length()>0;
	ResultSet rs = md.getTables(null,_schemaMsk.getText(),_tblMsk.getText(),new String[] {"TABLE"});
	//	_tbls.removeAll();
	Vector tbls = new Vector();
	_schemas.removeAllElements();
	while(rs.next()) {
	  String sch="";
	  if (isSchema) 
	      sch = rs.getString(2);
	  tbls.addElement(rs.getString(3));
	  _schemas.addElement(sch);
	}
	_tbls.setListData(tbls);
      } catch (SQLException ex) {
	handleException(ex);
      }
    }

    private void doMap() {
      //make _map
      try {
	DatabaseMetaData md = _con.getMetaData();
	int selTbls[] = _tbls.getSelectedIndices();
	_map = new DBMap();
	for (int i=0;i<selTbls.length;i++) {
	  String schN = (String)_schemas.elementAt(selTbls[i]);
	  String tN = (String)_tbls.getModel().getElementAt(selTbls[i]);
	  String tablN = (schN.length()>0?schN+"."+tN:tN);
	  _map.addTbl(tablN);
	  ResultSet rs = md.getColumns(null,schN,tN,"");
	  while(rs.next()) 
	    _map.addCol(tablN,rs.getString(4).trim(),rs.getInt(5),rs.getInt(9),rs.getInt(11));
	  rs = md.getPrimaryKeys(null,schN,tN);
	  while(rs.next()) 
	    _map.setPK(tablN,rs.getString(4).trim());
	}
	this.dispose();
      } catch (SQLException ex) {
	handleException(ex);
      }
    }

    private void init() {
      _schemas = new Vector();
      JPanel center = new JPanel(new GridBagLayout());
      JLabel lab = new JLabel("Schema mask");
      addCell(lab,center,1,0);
      _schemaMsk = new JTextField();
      addCell(_schemaMsk,center,2,0);
      lab = new JLabel("Table mask");
      addCell(lab,center,4,0);
      _tblMsk = new JTextField();
      addCell(_tblMsk,center,2,0);
      _tbls = new JList();
      _tbls.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      addCell(new JScrollPane(_tbls),center,2,0);
      this.getContentPane().add(center,BorderLayout.CENTER);
      JPanel south = new JPanel();
      JButton b = new JButton("Show Tables");
      b.setActionCommand("RETRIEVE");
      b.addActionListener(this);
      south.add(b);
      b = new JButton("Make Map");
      b.setActionCommand("OK");
      b.addActionListener(this);
      south.add(b);
      b = new JButton("Cancel");
      b.setActionCommand("CANCEL");
      b.addActionListener(this);
      south.add(b);
      south.setBorder(new EtchedBorder());
      this.getContentPane().add(south,BorderLayout.SOUTH);
    }

  }

    //DBMap.checkConsistency() return codes
    private static final int OK=0;
      //    public static final int ALIAS_MISS=1;
    private static final int LINK_MISS=2;
    private static final int PACK_MISS=4;
    private static final int BASE_MISS=8;
    private static final int SELCOL_MISS=16;

  //DB to Java classes map data container
  private class DBMap implements Externalizable {
    private Hashtable _tblDsc;
    private boolean _isRO = false;
    private String _pckg = "";
    private String _baseN = "";
    private String _targetDir = "";
    private String _SQLLink = "";
      public int dirty = NEW;
    
    public DBMap() {
      _tblDsc = new Hashtable();
    }

    //return error code: OK or combination of other error codes.
      public int checkConsistency() {
	  int res = OK;
	  if (_tblDsc.size()>1 && _SQLLink.length()==0)
	      res|=LINK_MISS;
	  if (_pckg.length()==0)
	      res|=PACK_MISS;
	  if (_baseN.length()==0)
	      res|=BASE_MISS;
	  if (getAllCols().length==0)
	      res|=SELCOL_MISS;
	  return res;
      }

      //PK is constituted by 1-st table's PK cols. And it must be at head of the list
      public void arrangePK() {
	  if (_tblDsc.size()>0) {
	      Enumeration te = _tblDsc.keys();
	      TblDescr t1 = (TblDescr)_tblDsc.get(te.nextElement());
	      t1.arrangePK();
	  }
      }

    //table level

    public void addTbl(String tN) {
      if (!_tblDsc.containsKey(tN))
	_tblDsc.put(tN,new TblDescr(tN,"t"+(_tblDsc.size()+1)));
    }

    public int getTblCount() {
      return _tblDsc.size();
    }

    public Enumeration getTables() {
      return _tblDsc.keys();
    }

    public void delTbl(String tN) {
      _tblDsc.remove(tN);
    }

    public String getTblAlias(String tN) {
      if (!_tblDsc.containsKey(tN))
	return null;
      return ((TblDescr)_tblDsc.get(tN)).aN;
    }

    public void setTblAlias(String tN,String aN) {
	if (_tblDsc.containsKey(tN)) {
	    ((TblDescr)_tblDsc.get(tN)).aN = aN;
	    dirty = (dirty!=NEW?DIRTY:dirty);
	}
    }

    //column - level

    public void addCol(String tN,String cN,int typ,int sc,int nlFlg) {
	if (_tblDsc.containsKey(tN)) {
	    TblDescr t = (TblDescr)_tblDsc.get(tN);
	    t.cols.addElement(new ColDescr(t,cN,typ,sc,nlFlg));
	}
    }

    public void delCol(String tN,String cN) {
	if (_tblDsc.containsKey(tN)) {
	    TblDescr t = (TblDescr)_tblDsc.get(tN);
	    t.cols.removeElement(new ColDescr(t,cN));
	}
    }

    public void setCol(String tN,String cN,int typ,int sc,int nlFlg) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null) {
	cd.dt = typ;
	cd.scale = sc;
	cd.nlFlg = nlFlg;
      }
    }

    public void setPK(String tN,String cN) {
      setPK(tN,cN,true);
    }

    public void setPK(String tN,String cN,boolean flg) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null) 
	cd.isPK = flg;
    }

    public boolean isPK(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      return (cd!=null && cd.isPK);
    }

      public boolean isColModi(String tN,String cN) {
	  ColDescr cd = findCol(tN,cN);
	  return cd!=null && cd.isModi;
      }

      public void setColModi(String tN,String cN,boolean flag) {
	  ColDescr cd = findCol(tN,cN);
	  if (cd!=null)
	      cd.isModi = flag;
      }

    public boolean hasCol(String tN,String cN) {
      return findCol(tN,cN)!=null;
    }

    public Vector getCols(String tN) {
      Vector res = new Vector();
      if (_tblDsc.containsKey(tN)) {
	Enumeration els = ((TblDescr)_tblDsc.get(tN)).cols.elements();
	while(els.hasMoreElements())
	  res.addElement(((ColDescr)els.nextElement()).colN);
      }
      return res;
    }

    public int getColCount(String tN) {
      int res=0;
      if (_tblDsc.containsKey(tN)) 
	res = ((TblDescr)_tblDsc.get(tN)).cols.size();
      return res;
    }

    public int[] getColTypeNull(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null)
	return new int[] {cd.dt,cd.scale,cd.nlFlg};
      return null;
    }

    public boolean isNull(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null)
	return cd.isNullable();
      return false;
    }


    //data for generator

    public String getFieldName(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null)
	return cd.fldN;
      return null;
    }

    public void setFieldName(String tN,String cN,String fN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null) {
	cd.fldN = fN;
	dirty= (dirty!=NEW?DIRTY:dirty);
      }
    }

    public String getDTString(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null)
	return cd.getDTString();
      return null;
    }

    public String getTargetDir() {
      return _targetDir;
    }

    public void setTargetDir(String dN) {
      _targetDir = dN;
      dirty= (dirty!=NEW?DIRTY:dirty);
    }

    public String getBaseName() {
      return _baseN;
    }

    public void setBaseName(String bN) {
      _baseN = bN;
      dirty= (dirty!=NEW?DIRTY:dirty);
    }

    public String getPckg() {
      return _pckg;
    }

    public void setPckg(String pN) {
      _pckg = pN;
      dirty= (dirty!=NEW?DIRTY:dirty);
    }

    public boolean isRO() {
      return _isRO;
    }

    public void setRO(boolean flg) {
      _isRO=flg;
      dirty= (dirty!=NEW?DIRTY:dirty);
    }

    public void selCol(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null) {
	dirty= (dirty!=NEW?DIRTY:dirty);
	cd.isSel = true;
      }
    }

    public void deselCol(String tN,String cN) {
      ColDescr cd = findCol(tN,cN);
      if (cd!=null) {
	dirty= (dirty!=NEW?DIRTY:dirty);
	cd.isSel = false;
      }
    }

    public Vector getSelTblCols(String tN) {
      return getTblCols(tN,true);
    }

    public Vector getUnsTblCols(String tN) {
      return getTblCols(tN,false);
    }

      public Vector getTblCols(String tN,boolean isSel) {
	  Vector res = new Vector();
	  if (_tblDsc.containsKey(tN)) {
	      Vector cds = ((TblDescr)_tblDsc.get(tN)).cols;
	      for (int i=0;i<cds.size();i++)
		  if (((ColDescr)cds.elementAt(i)).isSel&&isSel|| !isSel&&!((ColDescr)cds.elementAt(i)).isSel)
		      res.addElement(((ColDescr)cds.elementAt(i)).colN);
	  }
	  return res;
      }

    //methods used to invoke DCGenerator's constructor

    public String getSQLLink() {
      return _SQLLink;
    }

    public void setSQLLink(String sqlL) {
      _SQLLink = sqlL;
      dirty= (dirty!=NEW?DIRTY:dirty);
    }

      public String[] getAllTbls() {
	  String[] res = new String[_tblDsc.size()];
	  int i=0;
	  for (Enumeration tbls = _tblDsc.keys();tbls.hasMoreElements();i++)
	      res[i]=(String)tbls.nextElement();
	  return res;
      }

      public String[] getAliases() {
	  String[] res = new String[_tblDsc.size()];
	  int i=0;
	  for (Enumeration tbls = _tblDsc.keys();tbls.hasMoreElements();i++)
	      res[i]=((TblDescr)_tblDsc.get(tbls.nextElement())).aN;
	  return res;
      }

      public String getFullBaseName() {
	  return (_pckg.length()>0?_pckg+"."+_baseN:_baseN);
      }

      public String[] getAllCols() {
	  Vector cols = getAllColDescr();
	  String[] res = new String[cols.size()];

	  for (int j=0;j<res.length;j++) {
	      ColDescr col = (ColDescr)cols.elementAt(j);
	      res[j]=col.tbl.aN+"."+col.colN;
	  }
	  return res;
      }

      public String[] getAllFlds() {
	  Vector cols = getAllColDescr();
	  String[] res = new String[cols.size()];
	  
	  for (int i=0;i<res.length;i++) {
	      ColDescr col = (ColDescr)cols.elementAt(i);
	      res[i] = (col.colN.equals(col.fldN)?col.tbl.aN+col.fldN:col.fldN);
	  }
	  return res;
      }

      public int[][] getDTypes() {
	  Vector cols = getAllColDescr();
	  int[][] res = new int[cols.size()][2];
	  
	  for (int i=0;i<res.length;i++) {
	      ColDescr col = (ColDescr)cols.elementAt(i);
	      res[i][0] = col.dt;
	      res[i][1] = col.scale;
	  }
	  return res;
      }

    public boolean[] getAllNull() {
	Vector cols = getAllColDescr();
	boolean[] res = new boolean[cols.size()];

	for(int i=0; i<res.length; i++)
	    res[i]= ((ColDescr)cols.elementAt(i)).isNullable();
	return res;
    }

    public boolean[] getAllPK() {
	Enumeration tbls = _tblDsc.keys();
	TblDescr t1 = (TblDescr)_tblDsc.get(tbls.nextElement());
	Vector cols = t1.cols;
	int pkCnt=0,i;
	for(pkCnt=0;pkCnt<cols.size()&&((ColDescr)cols.elementAt(pkCnt)).isPK;pkCnt++) ;
	boolean[] res = new boolean[pkCnt];
	for (i=0;i<pkCnt;i++)
	    res[i]=((ColDescr)cols.elementAt(i)).isModi;
	return res;
    }

      //externalization support
      public void writeExternal(ObjectOutput out) throws IOException {
	  out.writeInt(_tblDsc.size());
	  for (Enumeration k=_tblDsc.keys();k.hasMoreElements();) {
	      String key = (String)k.nextElement();
	      TblDescr td = (TblDescr)_tblDsc.get(key);
	      out.writeObject(key);
	      td.writeExternal(out);
	  }
	  out.writeBoolean(_isRO);
	  out.writeObject(_pckg);
	  out.writeObject(_baseN);
	  out.writeObject(_targetDir);
	  out.writeObject(_SQLLink);
	  out.writeInt(dirty);
      }

      public void readExternal(ObjectInput in) throws IOException,ClassNotFoundException {
	  int tblCnt = in.readInt();
	  for (int i=0;i<tblCnt;i++) {
	      TblDescr td = new TblDescr();
	      String key = (String)in.readObject();
	      td.readExternal(in);
	      _tblDsc.put(key,td);
	  }
	  _isRO = in.readBoolean();
	  _pckg = (String)in.readObject();
	  _baseN = (String)in.readObject();
	  _targetDir = (String)in.readObject();
	  _SQLLink = (String)in.readObject();
	  dirty = in.readInt();
      }


    //gets all selected cols (ColDescr)
      private Vector getAllColDescr() {
	  int colCnt=0;
	  int i=0;
	  String[] tbls = getAllTbls();
	  Vector rv = new Vector();

	  for (i=0,colCnt=0;i<tbls.length;i++) {
	      TblDescr t = (TblDescr)_tblDsc.get(tbls[i]);
	      for (int j=0;j<t.cols.size();j++) {
		  ColDescr col = (ColDescr)t.cols.elementAt(j);
		  if (col.isSel)
		      rv.addElement(col);
	      }
	  }
	  return rv;
      }

    private ColDescr findCol(String tN,String cN) {
      if (_tblDsc.containsKey(tN)) {
	TblDescr td = (TblDescr)_tblDsc.get(tN);
	int pos = td.cols.indexOf(new ColDescr(td,cN));
	if (pos>=0)
	  return (ColDescr)td.cols.elementAt(pos);
      }
      return null;
    }

    //table descriptor
    private class TblDescr implements Externalizable {
      public String tN;
      public String aN;
      public Vector cols;
      
	public TblDescr() {
	    this("","");
	}

      public TblDescr(String tN,String aN) {
	this.tN = tN;
	this.aN = aN;
	cols = new Vector();
      }

	//realization of Externalizable
	public void writeExternal(ObjectOutput out) throws IOException {
	    out.writeObject(tN);
	    out.writeObject(aN);
	    out.writeInt(cols.size());
	    for (int i=0;i<cols.size();i++) {
		ColDescr cd = (ColDescr)cols.elementAt(i);
		out.writeObject(cd.colN);
		out.writeObject(cd.fldN);
		out.writeInt(cd.dt);
		out.writeInt(cd.scale);
		out.writeInt(cd.nlFlg);
		out.writeBoolean(cd.isPK);
		out.writeBoolean(cd.isSel);
		out.writeBoolean(cd.isModi);
	    }
	}

	public void readExternal(ObjectInput in) throws IOException,ClassNotFoundException {
	    tN = (String)in.readObject();
	    aN = (String)in.readObject();
	    int cCnt = in.readInt();
	    for (int i=0;i<cCnt;i++) {
		String colN = (String)in.readObject();
		String fldN = (String)in.readObject();
		int dt = in.readInt();
		int sc = in.readInt();
		int nlFlg = in.readInt();
		ColDescr cd = new ColDescr(this,colN,dt,sc,nlFlg);
		cd.fldN = fldN;
		cd.isPK = in.readBoolean();
		cd.isSel = in.readBoolean();
		cd.isModi = in.readBoolean();
		cols.addElement(cd);
	    }
	}

	public void arrangePK() {
	    Vector ncols = new Vector();
	    int i=0;
	    for (;i<cols.size();i++)
		if (((ColDescr)cols.elementAt(i)).isPK)
		    ncols.addElement(cols.elementAt(i));
	    for (i=0;i<cols.size();i++)
		if (!((ColDescr)cols.elementAt(i)).isPK)
		    ncols.addElement(cols.elementAt(i));
	    cols = ncols;
	}

    }

    //column descriptor
    private class ColDescr {
      public String colN;
      public String fldN;
      public int dt;
      public int scale;
      public int nlFlg;
      public boolean isPK;
      public boolean isSel = false;
	public TblDescr tbl;
	public boolean isModi = true;



      public ColDescr(TblDescr t,String cN) {
	this(t,cN,Types.CHAR,0,DatabaseMetaData.columnNullable);
      }

      public ColDescr(TblDescr t,String cN,int dataT,int sc,int nF) {
	  tbl = t;
	fldN=colN=cN;
	dt=dataT;
	scale = sc;
	nlFlg = nF;
	isPK=false;
      }

      public boolean isNullable() {
	return nlFlg!=DatabaseMetaData.columnNoNulls;
      }

      public boolean equals(Object o) {
	if (!(o instanceof ColDescr))
	  return false;
	return colN.equals(((ColDescr)o).colN);
      }

      public String getDTString() {
	return JDBCTypes.getSQLType(dt);
      }

    }

  }

}
