/*Data Access Library. v. 1.1.
 *Data Access Manager. 
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data;
import java.sql.*;
import java.util.*;

/**This abstract class defines the standard interface for a data access class.
 * It provides access to a collection of rows from a table, and facilitates 
 *manipulation of rows by using a database cursor.
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
*/
public abstract class DAManager implements OnThreadActionExecutor {
    private DatastoreJDBC _objDatastore;
    private OnThreadActionDispatcher _disp;
    private boolean _isBusy;
    private boolean _isAsync;
    private boolean _isOpen, _isFetched, _isAfterLast;
    private int _maxRows; // _stmt option
    private int _qryTime; // _stmt option
    protected transient PreparedStatement _stmt;
    protected transient ResultSet _rs;
    protected transient DAMap _map;
    protected transient DataAccessObject _cachedObject;
    protected transient Vector _internalSequence;
    protected transient DAPropertySupport _changes;
    protected transient ManagerCompleteSupport _managerSupport;

    public void _executeAction(String acnN,Object[] params) throws Exception {
	if (acnN.equals("append(Vector,int)")) {
	    append((Vector)params[0],((Integer)params[1]).intValue());
	} else if (acnN.equals("close()")) {
	    close();
	} else if (acnN.equals("fetchNextInto(DataAccessObject)")) {
	    fetchNextInto((DataAccessObject)params[0]);
	} else if (acnN.equals("fill(Vector,int)")) {
	    fill((Vector)params[0],((Integer)params[1]).intValue());
	} else if (acnN.equals("select(String)")) {
	    select((String)params[0]);
	}
    }

    public void _handleException(Exception ex) {
	ex.printStackTrace(System.err);
    }

    /**Use this method to register a listener for method complete events.*/
    public void addManagerCompleteListener(ManagerCompleteListener l) {
	_managerSupport.addManagerCompleteListener(l);
    }
 
    /**Use this method to register a listener for property changes.*/
    public void addPropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.addPropertyChangeListener(l);
    }

    /**Use this method to append the specified number of rows from the datastore 
     *to the current internal sequence that you have created by
     *calling the fill method.
     *@param rowC - number of rows to retrieve from DB.
     *@return - number of rows retrieved.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public int append(int rowC) throws DAException {
	return append(_internalSequence,rowC);
    }

    /**Use this method to append the specified number of rows from the datastore 
     *to the passed-in Vector object.
     *@param v - vector for storing of retrieved rows.
     *@param rowC - number of rows to retrieve from DB.
     *@return - number of rows retrieved.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public int append(Vector v, int rowC) throws DAException {
	int resC = 0;
	if ( !_onBackground() && isAsynchronous() ) {
	  Object[] params = { v,new Integer(rowC) };
	  _putOnBackgroundThread( "append(Vector,int)", params );
	  return resC;
	}
	try {
	    if (getCurrentDatastore() == null) 
		throw new DAException(DAResource.NO_CONNECT_EXIST);
	    if ( !isOpen() ) 
		throw new DAException(DAResource.CURSOR_NOT_OPEN);
	    _cachedObject = newElement();
	    while (_rs.next() && resC < rowC) {
		_map.putResultSetInto(_cachedObject,_rs,false);
		resC++;
		v.addElement(_cachedObject);
	    }
	} catch (SQLException ex) {
	    throw new DAException(DAResource.ERROR_IN_METHOD,"append(Vector,int)",ex);
	}finally {
	    _setBusy(false);
	}
	_managerSupport.fireItemsModified();
	return resC;
    }

    /**This method cancels the SQL statement that is currently running.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public void cancel() throws DAException {
	if ( !_onBackground() && isAsynchronous() && isBusy()) {
	    try {
		_stmt.cancel();
		_managerSupport.fireCancelComplete();
	    } catch (SQLException ex) {
		throw new DAException(DAResource.CANCEL_FAILED,ex);
	    } finally {
		_setBusy(false);
	    }
	}
    }

    /**This method closes the cursor if one is open.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public void close() throws DAException {
	if ( !_onBackground() && isAsynchronous() ) {
	  Object[] params = {};
	  _putOnBackgroundThread( "close()", params );
	  return;
	}
	try {
	    _dbclose();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.CLOSE_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
	_managerSupport.fireCloseComplete();
    }

    /**This method, when implemented in a subclass (such as <var>&lt;class&gt;
     *</var>Manager, which is generated by Data Access Builder), deletes the
     *row corresponding to the fetched object, if there is one.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public void deleteFetched() throws DAException {
	throw new DAException(DAResource.DEL_FETCHED_NOT_IMPL);
    }

    /**Use this method when You want prepare and execute parametrized statement.
     *@param sqlSuffix - where clause of the statement.
     *@param vals - array of values of the parameters. Length of the array must
     *correspond to the number of '?' placeholders in the statement.
     *@param types - array of type codes for the parameters of the statement. 
     *Length of the array must be equal to the <code>vals</code>. Code values are
     *defined in the <code>java.sql.Types</code>.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     *@see java.sql.Types
     */
    public void prepareOpen(String sqlSuffix,Object[] vals,int[] types) throws DAException {
	throw new DAException(DAResource.PREP_OPEN_NOT_IMPL);
    }

    /**@return - the current cached object (that is, the
     * currently-fetched row).*/
    public DataAccessObject elementAsDataAccessObject() {
	return _cachedObject;
    }

    /**This method fetches the attributes from the next row into the cached
     *object (a default object is provided).
     *@return - true if fetch was OK, false - if the last row had been fetched.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public boolean fetchNext() throws DAException {
	_cachedObject = (_cachedObject==null?newElement():_cachedObject);
	return fetchNextInto(_cachedObject);
    }

    /**This method fetches the attributes from the next row into a
     *PersistentObject object.
     *@param o - PersistentObject where to fetch current row.
     *@return - true if fetch was OK, false - if the last row had been fetched.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public boolean fetchNextInto(DataAccessObject o) throws DAException {
	boolean res = false;
	if ( !_onBackground() && isAsynchronous() ) {
	  Object[] params = {o};
	  _putOnBackgroundThread( "fetchNextInto(DataAccessObject)", params );
	  return res;
	}
	try {
	    if (getCurrentDatastore() == null) 
		throw new DAException(DAResource.NO_CONNECT_EXIST);
	    if ( !isOpen() ) 
		throw new DAException(DAResource.CURSOR_NOT_OPEN);
	    res = _rs.next();
	    if (res)
		_map.putResultSetInto(o,_rs,false);
	    _setFetched(true);
	} catch (SQLException ex) {
	    throw new DAException(DAResource.FETCH_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
	_managerSupport.fireFetchComplete("fetchNextInto(DataAccessObject)",new Boolean(res));
	return res;
    }

    /**This method fetches the specified number of rows into the cached
     * vector, starting at the row after the last-fetched row.
     *@param rowC - number of rows to fetch.
     *@return - number of rows fetched.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public int fill(int rowC) throws DAException {
	return fill(_internalSequence,rowC);
    }


    /**This method fetches the specified number of rows into the passed-in
     *Vector object, starting at the row after the last-fetched row.
     *@param v - vector for storing fetched rows.
     *@param rowC - number of rows to fetch.
     *@return - number of rows fetched.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public int fill(Vector v, int rowC) throws DAException {
	int res = 0;
	if ( !_onBackground() && isAsynchronous() ) {
	  Object[] params = { v,new Integer(rowC) };
	  _putOnBackgroundThread( "fill(Vector,int)", params );
	  return 0;
	}
	try {
	    if (getCurrentDatastore() == null) 
		throw new DAException(DAResource.NO_CONNECT_EXIST);
	    if ( !isOpen() ) 
		throw new DAException(DAResource.CURSOR_NOT_OPEN);
	    res= _dbfill(v,rowC);
	} catch (SQLException ex) {
	    throw new DAException(DAResource.FILL_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
	_managerSupport.fireFillComplete("fill(Vector,int)",new Integer(res));
	return res;
    }

    /**This method is to be implemented in a subclass. 
     *@return - the object's current datastore.    
     */
     public abstract DatastoreJDBC getCurrentDatastore();

    /**@return - the datastore object associated with the current object.*/
    public DatastoreJDBC getObjectsDatastore() {
	return _objDatastore;
    }

    /**@return - the setting for the maximum length of time (in
     *seconds) that the database driver will wait for a query to run.*/
    public int getQueryTimeout()  {
	return _qryTime;
    }

    /**@return - the setting for the maximum number of rows that
     * the result set of a select or open query can contain.*/
    public int getMaxRows() {
	return _maxRows;
    }

    /**@return true if the current object is in asynchronous mode.*/
    public boolean isAsynchronous() {
	return _isAsync;
    }

    /**@return true if the current object is in asynchronous mode and interacting
     *with DB server.
     */
    public boolean isBusy() {
	return _isBusy;
    }

    /**This method tells you whether the current row has been fetched into the
     * cached object.*/
    public boolean isFetched() {
	return _isFetched;
    }

    /**This method tells you whether a cursor is open and usable.*/
    public boolean isOpen() {
	return _isOpen;
    }

    /**This method tells you whether last row was already fetched.*/
    public boolean isAfterLastRow() {
	return _isAfterLast;
    }

    /**@return - a vector containing references to all the row
     * objects in the current cached sequence, thus providing
     *  you with access to the sequence.*/
    public Vector items() {
	return _internalSequence;
    }


    /**This abstract method can be implemented in a extending class, as it is
     * in the classes generated by the Data Access Builder, to
     *  open a cursor for the rows specified in the passed-in SQL suffix.
     *@param whereSfx - where clause
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public abstract void open(String whereSfx)  throws DAException;

    //    public void reopen();

    /**Use this method to remove the method complete listener on the manager
       object.*/
    public void removeManagerCompleteListener(ManagerCompleteListener l) {
	_managerSupport.removeManagerCompleteListener(l);
    }

    /**Use this method to remove the property change listener on the manager
       object.*/
    public void removePropertyChangeListener(java.beans.PropertyChangeListener l) {
	_changes.removePropertyChangeListener(l);
    }

    /**This abstract method can be implemented in an extending class so that
     * it opens a cursor, retrieves rows as specified in the
     * passed-in SQL predicate into a collection, and closes the cursor.
     *@param  whereSfx - where clause
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public abstract void select(String whereSfx) throws DAException;

    /**This method lets you set whether the database methods are run on a
       background thread.*/
    public void setAsynchronous(boolean isAs) {
	_isAsync = isAs;
    }

    /**This method sets the datastore associated with the current manager object
       to the DatastoreJDBC object passed in. */
    public void setObjectsDatastore(DatastoreJDBC oDs) {
	_objDatastore = oDs;
	if (_cachedObject != null )
	    _cachedObject.setObjectsDatastore(oDs);
    }

    /**Use this method to set the maximum time (in seconds) that the database
       driver will wait for a query to be run.*/
    public void setQueryTimeout(int ti) {
	_qryTime = ti;
    }

    /**Use this method to set the maximum number of rows that a result set can
       contain.*/
    public void setMaxRows(int rc) {
	_maxRows = rc;
    }

    /**This method, when implemented in a subclass (such as <var>&lt;class&gt;
     *</var>Manager, which is generated by Data Access Builder), updates the
     * row corresponding to the fetched object, if there is one,
     * with the cached object's attributes.
     *@exception JProjects.eab.data.DAException - when data access error occurs.
     */
    public void updateFetched() throws DAException {
	throw new DAException(DAResource.UPD_FETCHED_NOT_IMPL);
    }

    protected DAManager() {
	this(null);
    }

    protected DAManager(int iCap,int iIncr) {
	_internalSequence = new Vector(iCap,iIncr);
	_changes = new DAPropertySupport(this);
	_managerSupport = new ManagerCompleteSupport(this);
	_isAsync = _isBusy = _isOpen = _isFetched = _isAfterLast = false;
    }

    protected DAManager(DatastoreJDBC objD) {
	_internalSequence = new Vector();
	_changes = new DAPropertySupport(this);
	_managerSupport = new ManagerCompleteSupport(this);
	_objDatastore = objD;
	_isAsync = _isBusy = _isOpen = _isFetched = _isAfterLast = false;
    }

    protected void _dbclose() throws SQLException {
	if (_isOpen ) {
	    _isOpen = false;
	    _setFetched(false);
	    _rs.close();
	    _stmt.close();
	}
    }

    protected int _dbfill(Vector res, int rowCnt) throws SQLException, DAException {
	int resC =0;
	res.removeAllElements();
	while(_rs.next()) {
	    DataAccessObject o = newElement();
	    _map.putResultSetInto(o,_rs,false);
	    res.addElement(o);
	    if (++resC == rowCnt)
		break;
	}
	return resC;
    }

    protected void _fillInternalSequence() throws SQLException, DAException {
	_dbfill(_internalSequence,0);
    }

    protected abstract DataAccessObject newElement();

    protected DataAccessObject _newElement() {
	DataAccessObject el = newElement();
	el.setObjectsDatastore(getObjectsDatastore());
	return el;
    }

    protected boolean _onBackground() {
	if ( _disp != null)
	    return _disp.onBackground();
	return false;
    }

    protected void _putOnBackgroundThread(String acnN,Object[] params) {
	_setBusy(true);
	_disp = new OnThreadActionDispatcher(this,acnN,params);
    }

    protected void _setBusy(boolean isBusy) {
	_isBusy = isBusy;
    }

    protected void _setFetched(boolean isFetched) {
	_isFetched = isFetched;
    }

    protected void _setOpen(boolean isOpen) {
	_isOpen = isOpen;
    }

    protected void _setAfterLastRow(boolean isAfterLast) {
	_isAfterLast = isAfterLast;
    }

    protected void _setStmtOptions() throws SQLException {
	if (_qryTime>0)
	    _stmt.setQueryTimeout(_qryTime);
	if (_maxRows > 0)
	    _stmt.setMaxRows(_maxRows);
    }

    protected void firePropertyChange(String prN,Object oldV,Object newV) {
	_changes.firePropertyChange(prN,oldV,newV);
    }


}
