/* PhotoOrganizer - PhotoCollectionPanel
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer.renderer;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.MouseInputAdapter;
import javax.swing.event.TableModelEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListDataListener;
import javax.swing.table.*;
import javax.swing.tree.*;

import javax.mail.MessagingException;

import rogatkin.*;
import photoorganizer.*;
import photoorganizer.formats.*;
import photoorganizer.courier.*;

public class PhotoCollectionPanel extends JTable implements ActionListener, Persistentable {
    final static String SECNAME = "PhotoCollectionPanel";
    final static String COLWIDTH = "ColumnWidthes";

    final static int TOO_MANY_WINDOWS = 20;
    
    private Controller controller;
    private Point lastmouse;
    private CollectionThumbnailsPanel thumbnailspanel;
    private AlbumPane albumpane;
    private StatusBar statusbar;
    private Class fclass;
    private boolean keeporigmarkers, transformmove, transformaddsel;
    public final static String DEFTNMASK = "tumbnail%00c.jpg";
    public static String LastHtmlName = "??";

    public PhotoCollectionPanel(Controller controller) {
	setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        this.controller = controller;
        statusbar = (StatusBar)controller.component(Controller.COMP_STATUSBAR);
        thumbnailspanel = (CollectionThumbnailsPanel)controller.component(Controller.COMP_THUMBCOLLCTPANEL);
        thumbnailspanel.setCollection(this);
        albumpane = (AlbumPane)controller.component(Controller.COMP_ALBUMPANEL);
	albumpane.setCollectionPanel(this);
	addMouseListener(new MouseInputAdapter() {
	    public void mouseClicked(MouseEvent e) {
		int m = e.getModifiers();
		lastmouse = new Point(e.getX(), e.getY());
		if ((m & InputEvent.BUTTON3_MASK) > 0)
		    getRMouseMenu().show(PhotoCollectionPanel.this, e.getX(), e.getY());
	    }
	});
	setModel(new CollectionModel());
        setMinimumSize(Resources.MIN_PANEL_DIMENSION);
	getSelectionModel().addListSelectionListener(new ListSelectionListener() {
            public void valueChanged(ListSelectionEvent e) {
		int f = e.getFirstIndex();
		int l = e.getLastIndex();
		if (f > l) {
		    l = e.getFirstIndex();
		    f = e.getLastIndex();
		}
		CollectionModel model = (CollectionModel) getModel();
                Thumbnail tn;
		boolean selected, were_sel = false;
		for (int i = f; i <= l; i++) {
		    if ((tn = (Thumbnail)model.getElementAt(i)) != null) {
                        tn.select(selected = isRowSelected(i));
			were_sel |= selected;
		    }
		}
                if (!were_sel)  // we can still have some selections
                    were_sel = getSelectedRowCount() > 0;
		PhotoCollectionPanel.this.controller.getUiUpdater().notify(were_sel, UiUpdater.SELECTION_SELECTED);
            }
        });
	addKeyListener(new KeyAdapter() {
	}); // TODO: implement performing delete key
    }
    
    public void actionPerformed(ActionEvent a) {
        String cmd = a.getActionCommand();
	CollectionModel model = (CollectionModel) getModel();
	int [] selections = getSelectedRows();
        //TODO: create own copy of selection array
        selections = (int[])selections.clone();

        if (cmd.equals(Resources.MENU_REVERSE_SELECT)) {
	    int sel = rowAtPoint(lastmouse);
	    if (sel >= 0) {
		if (isRowSelected(sel))
		    removeRowSelectionInterval(sel, sel);
		else
		    addRowSelectionInterval(sel, sel);
		controller.getUiUpdater().notify(getSelectedRowCount()>0, UiUpdater.SELECTION_SELECTED);
	    }
        } else if (cmd.equals(Resources.MENU_SELECTALL)) {
	    selectAll();
        } else if (cmd.equals(Resources.MENU_CF_TOCOLLECT) || // from tool bar
            (a.getSource() instanceof JButton && ((JButton)a.getSource()).getToolTipText().equals(Resources.MENU_CF_TOCOLLECT))) {
            String path = (String)controller.getSerializer().getProperty(MiscellaneousOptionsTab.SECNAME, MiscellaneousOptionsTab.FC_FOLDER);
            if (path != null && path.length() > 0)
                add(new File(path));
        } else if (cmd.equals(Resources.MENU_EXTRACTMARKERS)) {
            extractMarkers(selections);
        } else if (cmd.equals(Resources.MENU_ADDTOALBUM)) {
            AlbumSelectionDialog asd = albumpane.getSelectionDialog();
            asd.setTitle(Resources.TITLE_SELECT_ALBUM+":"+selections.length);
            asd.setVisible(true);
            TreePath[] tps = asd.getSelectedAlbums();
            if (tps != null) {
		Serializer s = controller.getSerializer();
		final boolean removeafter = Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.MOVETOFOLDER), 0) == 1 &&
                    Serializer.getInt(s.getProperty(AlbumOptionsTab.SECNAME, AlbumOptionsTab.USEALBUMFOLDER), 0) == 1;
                BasicJpeg[] images = new BasicJpeg[selections.length];
		int minr=0, maxr=0;
                for (int i = 0; i < selections.length; i++) {
                    images[i] = ((Thumbnail)model.getElementAt(selections[i])).getImageInfo();
		    if (removeafter) {
			((Thumbnail)model.getElementAt(selections[i])).select(false);
			model.markDelete(selections[i]);
                        if (selections[i] > maxr)
                            maxr = selections[i];
                        if (selections[i] < minr)
                            minr = selections[i];
		    }
		}
		albumpane.addToAlbum(images, tps);
		if (removeafter) {
		    model.removeAllMarked();
		    model.fireTableRowsDeleted(minr, maxr);
		    controller.getUiUpdater().notify(false, UiUpdater.SELECTION_SELECTED);
		} else
		    model.fireTableRowsUpdated(minr, maxr);
            }
        } else if (cmd.equals(Resources.MENU_RENAME)) {
            rename(selections);
        } else if (cmd.equals(Resources.MENU_DELETE)) {
            delete(selections);
	} else if (cmd.equals(Resources.MENU_SHOW)) {
	    int sel = rowAtPoint(lastmouse);
	    if (sel >= 0) {
		BasicJpeg format = ((Thumbnail)model.getElementAt(sel)).getImageInfo();
		if (format != null)
		    ((PhotoImagePanel)controller.component(Controller.COMP_IMAGECOLLCTPANEL)).updateView(format);
	    }
	} else if (cmd.equals(Resources.MENU_PROPERTIES)) {
            for(int i=0; i<selections.length && i<TOO_MANY_WINDOWS; i++) {
		PropertiesPanel.showProperties(((Thumbnail)model.getElementAt(selections[i])).getImageInfo(), controller);
            }
        } else if (cmd.equals(Resources.MENU_EXTRACTTUMBNAILS)) {
	    BasicJpeg format;
	    Serializer s = controller.getSerializer();
	    String destpath = (String)s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.DESTFOLDER);
	    if (destpath == null || destpath.length() == 0)
		destpath = controller.getSerializer().getHomeDirectory();
	    String m = (String)s.getProperty(ThumbnailsOptionsTab.SECNAME, ThumbnailsOptionsTab.FILEMASK);
	    if (m == null || m.length() == 0)
		m = DEFTNMASK;
	    FileNameFormat fnf = new FileNameFormat(m, true);
            for(int i=0; i<selections.length; i++) {
		format = ((Thumbnail)model.getElementAt(selections[i])).getImageInfo();
		AbstractImageInfo ii = format.getImageInfo();
		try {
		    if (ii != null)
			ii.saveThumbnailImage(format,
			new FileOutputStream(new File(destpath, FileNameFormat.makeValidPathName(fnf.format(format), ii.getThumbnailExtension()))));
		} catch(IOException ioe) {
		    System.err.println(""+ioe);
		}
            }
        } else if (cmd.equals(Resources.MENU_GENERATEHTML)) {
            if (selections.length == 0)
                return;
            final Serializer s = controller.getSerializer();
            final boolean copyImagesOnly = Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.CPYPICSONLY), 1) == 0;
            String htmlname = "";
            if (!copyImagesOnly) {
                htmlname = (String)JOptionPane.showInputDialog(this,
                                                               Resources.LABEL_HTML_NAME, Resources.TITLE_CREATEHTML,
                                                               JOptionPane.QUESTION_MESSAGE, null, null, "");
                if (htmlname != null)
                    LastHtmlName = htmlname;
            }
            if (htmlname != null) {
                if (htmlname.indexOf('.') < 0)
                    htmlname += Resources.EXT_HTML;
                
                final String fhtmlname = htmlname;
                // prepare array of files
                final File [] files = selectionsToFiles(selections);
                new Thread(new Runnable() {
                    public void run() {
                        try {
                            Courier courier=null;
                            switch(Serializer.getInt(s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.PUBMODE), WebPublishOptionsTab.LOCAL)) {
                            case WebPublishOptionsTab.FTP:
                                courier = new FTPCourier(controller);
                                break;
                            case WebPublishOptionsTab.HTTP: // http
                                courier = new HTTPCourier(controller);
                                break;
                            case WebPublishOptionsTab.EMAIL:
                                // TODO: add send mail like feature
                                //courier = new MailCourier(controller);
                                new SendEmailFrame(controller, files);
                                return;
                            case WebPublishOptionsTab.XML_SVG: // SVG
                                break;
                            default:
                                courier = new FileCourier(controller);
                            }
                            if (copyImagesOnly) {
                                try {
                                    statusbar.displayInfo(Resources.INFO_CONNECTING);
                                    courier.init();
                                    String imagePath = (String)s.getProperty(WebPublishOptionsTab.SECNAME, WebPublishOptionsTab.IMAGEPATH);
                                    if (imagePath == null)
                                        imagePath = "";
                                    else
                                        courier.checkForDestPath(imagePath);
                                    statusbar.clearProgress();
                                    statusbar.displayInfo(Resources.INFO_COPYING);
                                    statusbar.setProgress(files.length);				   
                                    for (int i = 0; i < files.length; i++) {
                                        courier.deliver(files[i].toString(), imagePath);
                                        statusbar.tickProgress();
                                    }
                                } finally {
                                    statusbar.clearInfo();
                                    statusbar.clearProgress();
                                    courier.done();				
                                }
                            } else {
                                new HtmlProducer(controller).produce(fhtmlname, files, courier);
                            }
                        } catch (IOException e) {
                            statusbar.flashInfo(Resources.INFO_ERR_WEBPUBLISHING);
                            System.err.println("Exception in Web publishing "+e);
                            e.printStackTrace();
                        }
                    }
                }, "WebPublishing").start();
            }
        } else if (cmd.equals(Resources.MENU_PRINT)) {
	    controller.print(selectionsToFiles(selections));
        } else if (cmd.equals(Resources.MENU_SENDTO)) {
            new SendEmailFrame(controller, selectionsToFiles(selections));
        }
        int op;
	if ((op = Controller.convertCmdToTrnasformOp(cmd)) != -1) {
            transform(selections, op);
        }
    }
    
    public int add(File f) {
	CollectionModel model = (CollectionModel) getModel();
	int end, start = model.getRowCount();
	end = start;
	if (f.isFile()) {
	    thumbnailspanel.addImage(f);
	    end++;
	} else if (f.isDirectory()) {
	    String[] files = f.list();
	    String path = f.getPath();
	    for (int i = 0; i<files.length; i++) {
		f = new File(path, files[i]);
		if (f.isFile()) {
		    thumbnailspanel.addImage(f);
		    end++;
		}
	    }
	}
	model.fireTableRowsInserted(start, end);
	return end;
    }

    public int add(File[] fs) {
        controller.setWaitCursor(this, true);
	CollectionModel model = (CollectionModel) getModel();
	int end, start = model.getRowCount();
	end = start;
	for (int i=0; i<fs.length; i++) {
	    if (fs[i].isFile()) {
		thumbnailspanel.addImage(fs[i]);
		end++;
	    } else if (fs[i].isDirectory()) {
		String[] files = fs[i].list();
		String path = fs[i].getPath();
		for (int k = 0; k<files.length; k++) {
		    fs[i] = new File(path, files[k]);
		    if (fs[i].isFile()) {
			thumbnailspanel.addImage(fs[i]);
			end++;
		    }
		}
	    }
	}
	model.fireTableRowsInserted(start, end);
        controller.setWaitCursor(this, false);
	return end;
    }

    String transformName(File file, int op) {
        Serializer s = controller.getSerializer();
        String destpath = (String)s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.DESTFOLDER);
        if (destpath == null || destpath.length() == 0)
            destpath = controller.getSerializer().getHomeDirectory();
	String m = (String)s.getProperty(TransformOptionsTab.SECNAME, TransformOptionsTab.MASK);
        if (m != null) {
            return destpath+File.separatorChar+
		FileNameFormat.makeValidPathName(new FileNameFormat(m, op, true).format(new BasicJpeg(file)));
        } else
            return destpath+File.separatorChar+file.getName();
    }
    
    int findIndexOf(Thumbnail im) {
        CollectionModel model = (CollectionModel) getModel();
        for (int i=0; i<model.getRowCount(); i++) {
            if (model.getElementAt(i) == im)
                return i;
        }
        return -1;
    }

    File[] selectionsToFiles(int [] selections) {
        File []files = new File[selections.length];
        CollectionModel model = (CollectionModel) getModel();
        for (int i=0; i<selections.length; i++) {
	    Thumbnail tn = (Thumbnail)model.getElementAt(selections[i]);
	    if (tn != null)
		files[i] = new File(tn.getImageInfo().getLocationName());
	}
	return files;
    }

    void extractMarkers(int [] selections) {
        Serializer s = controller.getSerializer();
        CollectionModel model = (CollectionModel) getModel();
        String  name;
        BasicJpeg im;
        AbstractImageInfo ii;
        int dp;
        for(int i=0; i<selections.length; i++) {
            im = ((Thumbnail)model.getElementAt(selections[i])).getImageInfo();
	    ii = im.getImageInfo();
            if (ii != null) {
                name = ii.getName();
                dp = name.lastIndexOf('.');
                if (dp > 0)
                    name = name.substring(0, dp+1);
                else
                    name += '.';
                name += ii.getFormat();
                try {
                    im.saveMarkers(new FileOutputStream(new File(im.getParentPath(), name)));
                } catch(IOException e) {
                    System.err.println("Exception in markers extraction "+e);
                }
            }
        }
    }

    void rename(final int [] selections) {
        Serializer s = controller.getSerializer();
        String destpath = (String)s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.DESTFOLDER);
        if (destpath == null || destpath.length() == 0)
            destpath = controller.getSerializer().getHomeDirectory();
        final String value = getRenameMask(this, s, " "+destpath);
	if (value == null)
	    return;
        final String dp = destpath;
        statusbar.displayInfo(Resources.INFO_RENAMING);
        statusbar.setProgress(selections.length);
	final boolean removeafter = s.getInt(s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.REMOVEAFTER), 0) == 1;
        
        new Thread(new Runnable () {
            public void run() {
                BasicJpeg format;
                int minr=0, maxr=0;
                CollectionModel model = (CollectionModel) getModel();
		FileNameFormat fnf = new FileNameFormat(value, true);
                for(int i=0; i<selections.length; i++) {
                    format = ((Thumbnail)model.getElementAt(selections[i])).getImageInfo();
                    File dest = new File(dp,
			FileNameFormat.makeValidPathName(fnf.format(format)));
                    if (format.renameTo(dest)) {
                        if (selections[i] > maxr)
                            maxr = selections[i];
                        if (selections[i] < minr)
                            minr = selections[i];
                    }
		    if (removeafter) {
			((Thumbnail)model.getElementAt(selections[i])).select(false);
			model.markDelete(selections[i]);
		    }
                    statusbar.tickProgress();
                }
		if (removeafter) {
		    model.removeAllMarked();
		    model.fireTableRowsDeleted(minr, maxr);
		    controller.getUiUpdater().notify(false, UiUpdater.SELECTION_SELECTED);
		} else
		    model.fireTableRowsUpdated(minr, maxr);
                statusbar.clearInfo();
                statusbar.clearProgress();
            }
        }, Resources.INFO_RENAMING).start();
    }
    
    static String getRenameMask(Component p, Serializer s, String to) {
        Object value;
        if (Serializer.getInt(s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.ASKEDIT), 0) == 1) {
            value = JOptionPane.showInputDialog(p,
                Resources.LABEL_NEW_NAME, Resources.TITLE_RENAME+to,
                JOptionPane.QUESTION_MESSAGE, null, null, 
                s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.MASK));
        } else
            value = s.getProperty(RenameOptionsTab.SECNAME, RenameOptionsTab.MASK);
        return (String)value;
    }

    void delete(int [] selections) {
        CollectionModel model = (CollectionModel) getModel();
        statusbar.displayInfo(Resources.INFO_REMOVING);
        int minr=0, maxr=0;
        for(int i=0; i<selections.length; i++) {
            ((Thumbnail)model.getElementAt(selections[i])).select(false);
	    model.markDelete(selections[i]);
            if (selections[i] > maxr)
                maxr = selections[i];
            if (selections[i] < minr)
                minr = selections[i];
        }
	model.removeAllMarked();
        model.fireTableRowsDeleted(minr, maxr);
        tableChanged(new TableModelEvent(model, minr, maxr, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
	controller.getUiUpdater().notify(false, UiUpdater.SELECTION_SELECTED);
        statusbar.clearInfo();
    }

    void transform(final int [] selections, final int op) {
        statusbar.displayInfo(Resources.INFO_TRANSFORMING);
        statusbar.setProgress(selections.length);
	loadTransformOptions();
        
        new Thread(new Runnable () {
            public void run() {
		CollectionModel model = (CollectionModel) getModel();
		int minr=0, maxr=0, bound = model.getRowCount();
                String transformname;
		File file;
		for(int i=0; i<selections.length; i++) {
		    file = new File(((Thumbnail)model.getElementAt(selections[i])).getImageInfo().getLocationName());
                    transformname = transformName(file, op);
		    if (op == BasicJpeg.COMMENT) {
			BasicJpeg im = ((Thumbnail)model.getElementAt(selections[i])).getImageInfo();
			String value = im.getComment();
			value = (String)JOptionPane.showInputDialog(PhotoCollectionPanel.this,
			    Resources.LABEL_COMMENT, Resources.TITLE_COMMENT,
			    JOptionPane.QUESTION_MESSAGE, null, null, value);
			if (value != null)
			    im.setComment(value);
                        else
                            continue;
		    }
		    if (!((Thumbnail)model.getElementAt(selections[i])).getImageInfo().
		        transform(transformname, op, keeporigmarkers, fclass))
                        continue;
		    if (transformmove) {
			if (file.delete()) {
			    model.removeElementAt(selections[i]);
			    if (selections[i] > maxr)
				maxr = selections[i];
			    if (selections[i] < minr)
				minr = selections[i];
			}
		    }
		    if (transformaddsel) {
			add(new File(transformname));
		    }
                    statusbar.tickProgress();
		}
		if (transformaddsel) {
                    model.fireTableRowsInserted(bound-1, bound+selections.length-1);
		    tableChanged(new TableModelEvent(model, bound-1, bound+selections.length-1, TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT));
                } 
                if (transformmove) {
		    model.fireTableRowsDeleted(minr, maxr);
		    tableChanged(new TableModelEvent(model, minr, maxr, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
		    controller.getUiUpdater().notify(false, UiUpdater.SELECTION_SELECTED);
		}
                statusbar.clearInfo();
                statusbar.clearProgress();
	    }
	}, Resources.INFO_TRANSFORMING).start();
    }

    void loadTransformOptions() {
	Serializer s = controller.getSerializer(); 
	keeporigmarkers = Serializer.getInt(s.getProperty(TransformOptionsTab.SECNAME, TransformOptionsTab.KEEP), 0) == 1;
	transformmove = Serializer.getInt(s.getProperty(TransformOptionsTab.SECNAME, TransformOptionsTab.MOVE), 0) == 0;
        transformaddsel = Serializer.getInt(s.getProperty(TransformOptionsTab.SECNAME, TransformOptionsTab.TRANSFTOSEL), 1) == 1;
	String format = (String)s.getProperty(TransformOptionsTab.SECNAME, TransformOptionsTab.FORMAT);
	fclass = null;
	if (format != null && !keeporigmarkers) {
	    if (Resources.LIST_EXIF.equals(format))
		fclass = Exif.class;
	    else if (Resources.LIST_CIFF.equals(format))
		fclass = CIFF.class;
	    else if (Resources.LIST_JFIF.equals(format))
		fclass = JFXX.class;
	    else if (Resources.LIST_EXTRN.equals(format))
		fclass = AbstractImageInfo.class;
	}
    }
    public void save() {
       	Serializer s = controller.getSerializer();
	Integer[] w = new Integer[getColumnCount()];
        for (int i = 0; i < w.length; i++) {
            w[i] = new Integer(getColumn(getColumnName(i)).getWidth());
        }
	s.setProperty(SECNAME, COLWIDTH, w);
    }
    
    public void load() {
       	Serializer s = controller.getSerializer();
	Object[] w = (Object[])s.getProperty(SECNAME, COLWIDTH);
        if (w == null)
            return;
        for (int i = 0; i < w.length; i++) {
	    getColumn(getColumnName(i)).setWidth(((Integer)w[i]).intValue());
            getColumn(getColumnName(i)).setPreferredWidth(((Integer)w[i]).intValue());
        }
    }

    JPopupMenu getRMouseMenu() {
	return new RButtonMenu();
    }

    class RButtonMenu extends JPopupMenu {
	RButtonMenu() {
	    JMenuItem item;
	    this.add(item = new JMenuItem(Resources.MENU_REVERSE_SELECT));
	    item.addActionListener(PhotoCollectionPanel.this);
	    this.add(item = new JMenuItem(Resources.MENU_SELECTALL));
	    item.addActionListener(PhotoCollectionPanel.this);
	    this.add(item = new JMenuItem(Resources.MENU_SHOW));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
            this.add(item = new JMenuItem(Resources.MENU_ADDTOALBUM));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	    this.add(item = new JMenuItem(Resources.MENU_EXTRACTMARKERS));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	    addSeparator();
	    this.add(item = new JMenuItem(Resources.MENU_RENAME));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	    this.add(item = new JMenuItem(Resources.MENU_DELETE));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	    addSeparator();
            this.add(item = new JMenuItem(Resources.MENU_PRINT));
            item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
            addSeparator();
	    JMenu menu;
	    this.add(menu = Controller.createTransformMenu(PhotoCollectionPanel.this));
	    menu.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	    addSeparator();
	    this.add(item = new JMenuItem(Resources.MENU_PROPERTIES));
	    item.addActionListener(PhotoCollectionPanel.this);
	    item.setEnabled(controller.getUiUpdater().isEnabled(UiUpdater.SELECTION_SELECTED));
	}
    }

    class CollectionModel extends AbstractTableModel implements ListModel {
	Vector collection;
	String [] colnames = {Resources.HDR_NAME, Resources.HDR_TAKEN,
	    Resources.HDR_SHUTTER, Resources.HDR_APERTURE, Resources.HDR_FLASH};
	
	public int getRowCount() {
	    return thumbnailspanel.getComponentCount();
	}
	
	public int getColumnCount() {
	    return colnames.length; 
	}
	
	public String getColumnName(int column) {
	    return colnames[column];
	}
	
	public Object getValueAt(int row, int column) {
	    BasicJpeg im = ((Thumbnail)thumbnailspanel.getComponent(row)).getImageInfo();
	    if (im == null)
		return null;
	    AbstractImageInfo ii = im.getImageInfo();
	    try {
		switch(column) {
		case 0:
		    if (ii == null)
			return im.getLocationName();
		    return ii.getName();
		case 1:
		    if (ii != null) {
			Date date = ii.getDateTimeOriginal();
			if (date != null)
			    return date;
		    }
		    break;
		case 2:
		    if (ii != null)
			return ii.getShutter();
		    break;
		case 3:
		    if (ii != null)
			return new Float(ii.getFNumber());
		    break;
		case 4:
		    if (ii != null)
			return ii.isFlash()?Boolean.TRUE:Boolean.FALSE;
		    break;
		}
	    }
	    catch  (NullPointerException ne) { }
	    
	    return null; 
	}
	
	public Object /*Thumbnail*/ getElementAt(int index) {
            try {
                return /*(Thumbnail)*/thumbnailspanel.getComponent(index);
            } catch (ArrayIndexOutOfBoundsException e) {
                return null;
            }
	}
	
	void removeElementAt(int row) {
	    thumbnailspanel.remove(row);
	}

	void markDelete(int row) {
	    marked.addElement(thumbnailspanel.getComponent(row));
	}

	void removeAllMarked() {
	    for (int i=0; i<marked.size(); i++)
		thumbnailspanel.remove((Component)marked.elementAt(i));
	    marked.removeAllElements();
	    thumbnailspanel.adjustDimension();
	}
	
	public void fireTableRowsUpdated(int firstRow, int lastRow) {
	    super.fireTableRowsUpdated(firstRow, lastRow);
	    for (int i=firstRow; i<=lastRow; i++) {
		((Thumbnail)thumbnailspanel.getComponent(i)).update();
	    }
	}
	
	// list model
	public int getSize() {
	    return getRowCount();
	}

	public void addListDataListener(ListDataListener l) {
	}

	public void removeListDataListener(ListDataListener l) {
	}

	Vector marked = new Vector(10);
    }
}