/********************************************************************************************
** Vrml2OGL.cpp	- Main Source Code Module for Vrml2OGL.exe V1.0
** Copyright 1999 by Alexander Rohra. All rights reserved.
**
** Program Name:	Vrml2OGL.exe V1.0
** Author:			Alexander Rohra
** Date:			May 1999
** Compiler:		MSVC++ V5.0	(MS Visual Studio'97)
** App Type:		Command Line Utility (Win32 Console Application)
** Revisions:
** Code Desc.:		The code contained is this module performs all startup and
**					initialization tasks such as scanning the command line,
**					opening all files, etc. and finally	calling the program's
**					core function findNodes().
**					Each function that returns a value (other than void)
**					returns either NOERR or one of the error stati enumerated
**					in sharedDefs.h unless where denoted otherwise in
**					function descriptions.
**
** General Program Info
** --------------------
** Usage:			Vrml2OGL [-?] | [[-it] [-c | [-cxVAL -cyVAL -czVAL]] [-rf] [-rn]
**                          [-ft | -f0<=VAL<8] [-bt | -b0<=VAL<8]
**                          vrmlFile [oglFile hdrFile datFile]]
**
** Prg Description:	Vrml2OGL converts a VRML V1.0 file to OpenGL C source code.
**					It generates 3 output files:
**					1) an OpenGL file which contains the object drawing
**					   code contained in a C-callable function
**					2) a header file containing the function prototype for the
**					   object drawing function contained in the OpenGL C file,
**					   a few #define statements and extern variable definitions
**					3) a data file containing matrix as well material color
**					   arrays (which are made public by the header file)
**					These output files correspond to the input .WRL file as
**					closely as possible.
**
** For program limitations, notes, disclaimer and distribution info please
** see ReadMe.txt (supplied with the archive containing this file).
********************************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#include "SharedDefs.h"		// needed for typedefs
#include "Vrml2OGL.h"		// needed for NODETYPES definition
#include "findNodes.h"		// needed for prototypes
#include "Vrml2OGLUtils.h"	// needed for prototypes
#include "GeometryUtils.h"	// needed for prototypes

//*** global/puplic variables ***
#ifdef _V_DEBUGMODE_
	BYTE debugMode=V_FALSE;
#endif _V_DEBUGMODE_
BYTE centerMode=OFF;
VERTEX objCenter;
BYTE reverseNormals=V_FALSE;
BYTE reverseFaces=V_FALSE;
char *baseName;
char *filler[MAXINDENTS];

// function prototypes for internal/private functions
int scanCommandLine(int argc, char *argv[], VERTEX *center, int *baseFillerSize,
					int *fillerSize, char *cmdLSwitches, BYTE *inclXforms, BYTE *revNormals,
					BYTE *revFaces, BYTE *showHelp);
BYTE openVrmlFile(FILE *vrmlFile[], char *fileName);
void closeVrmlFile(FILE *vrmlFile[]);
BYTE openOutFiles(FILE *outFiles[], char *outFNames[], int count);
void closeOutFiles(FILE *outFiles[], int count);
void outputGenericErrorMsgs(BYTE errorCode);
char *getBaseName(char *filename);
char **createFNamesFromInputFName(char *baseName);
char *createDefineFromFName(char *fName);
void stripToFileName(char *fileNames[], int count);
char *getCurrentDateString(void);
int sum(int n);
BYTE generateFillers(char *filler[], int baseFillerSize, int fillerSize);
void deleteFillers(char *filler[]);

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: main()
// 
// Description:
//	This function scans in the command line, opens and closes all files, performs some
//	initialization tasks and calls findNodes() which performs the actual conversion.
//	If any error occurs its code will be returned to the program's invokation environment;
//	otherwise NOERR is returned.
/////////////////////////////////////////////////////////////////////////////////////////////
int main(int argc, char *argv[])
{	
	int ret=NOERR;
	char functionName[MAXBASENAME+4];
	char center[MAXCENTERSTR];
	char cmdLSwitches[MAXCMDLSWSTR];
	FILE *vrmlFile[MAXVRMLFPOINTER];
	FILE *outFiles[MAXOUTFILES];
	NODE nodeTypes[]=NODETYPES;
	VERTEX boundingBox[8];
	int i, nextIdx;
	int fillerSize=DEFLTFILLERSIZE;
	int baseFillerSize=DEFLTFILLERSIZE;
	BYTE includeTransforms=V_FALSE, showHelp=V_FALSE;
	HEADERFOOTERINFO hfInfo;

	for(i=0; i<MAXVRMLFPOINTER; i++)
		vrmlFile[i]=NULL;
	oglFile=hdrFile=datFile=NULL;

	if(argc<MINARGS || (nextIdx=scanCommandLine(argc, argv, &objCenter, &baseFillerSize,
		&fillerSize, cmdLSwitches, &includeTransforms, &reverseNormals, &reverseFaces,
		&showHelp))==0)
	{
		if(!showHelp)
			fprintf(stderr, USAGE);
		else
			printf(HELPSCREEN);
	}

	// open the input file and check the version string in it
	else if((ret=openVrmlFile(vrmlFile, argv[nextIdx]))==NOERR)
	{
		if((ret=checkVersion(vrmlFile[0]))==NOERR)
		{
			if(!includeTransforms)
				for(i=0; nodeTypes[i].name!=NULL; i++)
					if(nodeTypes[i].type==GENXFRM)
						nodeTypes[i].handlingFunc=handleGenericNode;

			ret=generateFillers(filler, baseFillerSize, fillerSize);

			// if the auto centering was specified (-c) find the bounding box in the input
			// filen, calculate the box center, assign it to objCenter and print it to the
			// screen
			if(ret==NOERR && centerMode==AUTOCENTER &&
				(ret=findBoundingBox(vrmlFile[0], boundingBox))==NOERR)
			{
				rewind(vrmlFile[0]);
				objCenter=getBoxCenter(boundingBox);
				printf("Computed object center: (%5.4f, %5.4f, %5.4f)\n",
					objCenter.x, objCenter.y, objCenter.z);
			}
		
			if(ret==NOERR)
			{
				// get the base name of the input file
				baseName=getBaseName(argv[nextIdx]);

				// check if there are enough arguments left as the output file names
				if(nextIdx+4<=argc)			// if so, use them
					hfInfo.outFNames=argv+nextIdx+1;
				else						// otherwise create them from the input file name
					hfInfo.outFNames=createFNamesFromInputFName(baseName);

				// open all output files
				if((ret=openOutFiles(outFiles, hfInfo.outFNames, MAXOUTFILES))==NOERR)
				{
					// collect info for file headers and footers
					// NOTE: from this point on the paths to the input and output files are
					// lost
					stripToFileName(hfInfo.outFNames, MAXOUTFILES);
					stripToFileName(argv+nextIdx, 1);

					// initialize header information structure
					sprintf(functionName, "Draw%s", baseName);
					hfInfo.funcName=functionName;
					hfInfo.inFName=argv[nextIdx];
					hfInfo.hdrDefine=createDefineFromFName(hfInfo.hdrFName);
					hfInfo.currDate=getCurrentDateString();
					hfInfo.prgName=PRGNAME;
					hfInfo.cmdLSwitches=cmdLSwitches[0]!='\0'? cmdLSwitches : NULL;
					if(centerMode)
					{
						sprintf(center, "(%5.4f, %5.4f, %5.4f)",
							objCenter.x, objCenter.y, objCenter.z);
						hfInfo.center=center;
					}
					else
						hfInfo.center=NULL;

					// store length of the node type strings
					for(i=0; nodeTypes[i].name!=NULL; i++)
						nodeTypes[i].len=strlen(nodeTypes[i].name);

					// write header info, perform the actual conversion and write footer info
					if((ret=writeFileHeaders(outFiles, &hfInfo))==NOERR)
					if((ret=findNodes(vrmlFile, outFiles, nodeTypes))==NOERR)
						ret=writeFileFooters(outFiles, &hfInfo);

					closeOutFiles(outFiles, MAXOUTFILES);
				}
			}

			deleteFillers(filler);
		}

		closeVrmlFile(vrmlFile);
	}

	if(ret!=NOERR)
	{
		// output generic error messages (if there is any)
		outputGenericErrorMsgs(ret);
		fprintf(stderr, "Error code %d detected - aborted\n", ret);
	}

	return(ret);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: stripToFileName()
// 
// Description:
//	This function strips of the path of each file name contained in filenames[].
/////////////////////////////////////////////////////////////////////////////////////////////
void stripToFileName(char *fileNames[], int count)
{
	int i, j, len;

	for(i=0; i<count; i++)
	{
		len=strlen(fileNames[i]);
		for(j=len; j>=0; j--)
		{
			if(fileNames[i][j]=='\\')
				fileNames[i]=fileNames[i]+j+1;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: getCurrentDateString()
// 
// Description:
//	This function retrieves the system's current date and time, converts it to a string
//	similar to "Sat Apr 18 16:17:12 1999" and returns a pointer to this string.
// Note: If this function is called more than once, each returned date string should
//		 be copied to a different buffer prior to calling this function again. Otherwise
//		 the string previously created will be overwritten.
/////////////////////////////////////////////////////////////////////////////////////////////
char *getCurrentDateString(void)
{
	char *retPtr;
	time_t ltime;

	time(&ltime);
	retPtr=ctime(&ltime);
	retPtr[strlen(retPtr)-1]='\0';	// replace newline character

	return(retPtr);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: createDefineFromFName()
// 
// Description:
//	This function creates a define tag from the passed in string by preceding it with one
//	underscrore character ('_'), converting any non alpha numeric characters to underscores
//	and converting it to uppercase. It returns a pointer to the buffer containing the created
//	define string.
// Example: Flower -> _FLOWER
// Note: If this function is called more than once, each returned define tag should
//		 be copied to a different buffer prior to calling this function again. Otherwise
//		 the define previously created will be overwritten.
/////////////////////////////////////////////////////////////////////////////////////////////
char *createDefineFromFName(char *fName)
{
	static char define[MAXFNAME];
	int i, j, len;

	len=strlen(fName);
	define[0]='_';
	for(i=0, j=1; i<len; i++)
		if(isalnum(fName[i]))
			define[j++]=toupper(fName[i]);
		else
			define[j++]='_';

	return(define);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: createFNamesFromInputFName()
// 
// Description:
//	This function creates the file names for the three output files from the base of
//	the input file name. These file names are then returned via a double pointer to the
//	array in which they are stored.
/////////////////////////////////////////////////////////////////////////////////////////////
char **createFNamesFromInputFName(char *baseName)
{
	static char outFNames[MAXOUTFILES][MAXFNAME];
	static char *outFNamePtr[]={outFNames[0], outFNames[1], outFNames[2]};

	sprintf(outFNames[0], "%sOGL.c", baseName);
	sprintf(outFNames[1], "%sHdr.h", baseName);
	sprintf(outFNames[2], "%sDat.c", baseName);

	return(outFNamePtr);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: openOutFiles()
// 
// Description:
//	This function opens the count files whose names are contained in outFNames[] and assigns
//	the file pointers to the opened files to outFiles[].
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE openOutFiles(FILE *outFiles[], char **outFNames, int count)
{
	int i;

	for(i=0; i<count; i++)
	{
		if((outFiles[i]=fopen(outFNames[i], WRITEMODE))==NULL)
		{
			fprintf(stderr, "Error opening %s for writing\n", outFNames[i]);
			closeOutFiles(outFiles, i-1);
			return(OPENERR);
		}
	}

	return(NOERR);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: closeOutFiles()
// 
// Description:
//	This function closes the count files referenced by the file pointers in outFiles[] unless
//	they point no NULL.
/////////////////////////////////////////////////////////////////////////////////////////////
void closeOutFiles(FILE *outFiles[], int count)
{
	int i;

	for(i=0; i<count; i++)
		if(outFiles[i]!=NULL)
			fclose(outFiles[i]);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: openVrmlFile()
// 
// Description:
//	This function opens MAXVRMLFPOINTER times the file refered to by filename. Each file
//	pointer is assigned to the pointers contained in vrmlFIle. If an error occurs, an error
//	message is output, the files opened prior to the error are closed (if any) and an error
//	status is returned.
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE openVrmlFile(FILE *vrmlFile[], char *fileName)
{
	int i;
	
	for(i=0; i<MAXVRMLFPOINTER; i++)
		if((vrmlFile[i]=fopen(fileName, READMODE))==NULL)
		{
			fprintf(stderr, "Error opening input file %s\n", fileName);
			closeVrmlFile(vrmlFile);
			return(OPENERR);
		}

	return(NOERR);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: closeVrmlFile()
// 
// Description:
//	This function closes MAXVRMLFPOINTER files refered to by the file pointers in vrmlFile[]
//	unless they point to NULL.
/////////////////////////////////////////////////////////////////////////////////////////////
void closeVrmlFile(FILE *vrmlFile[])
{
	int i;

	for(i=0; i<MAXVRMLFPOINTER; i++)	
		if(vrmlFile[i]!=NULL)
			fclose(vrmlFile[i]);	
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: scanCommandLine()
// 
// Description:
//	This function scans the passed in command line for recognized command line options.
//	Unless the first argument is "-?" (for help) the command line options are only scanned
//	up to argv[argc-2] since one parameter is considered the file name of the input file.
//	The scan might terminate before this if a parameter is found that is not recognized
//	as one of the options in which case this parameter is assumed to be the name of the
//	input file. Each option recognized is added to cmdLSwitches (except for "-?" and "-d"
//	(debug)).
//	The return value contains the index to the parameter assumed to contain the name of the
//	input file or 0 if one of the options was specified incorrectly.
/////////////////////////////////////////////////////////////////////////////////////////////
int scanCommandLine(int argc, char *argv[], VERTEX *center, int *baseFillerSize,
					int *fillerSize, char *cmdLSwitches, BYTE *inclXforms, BYTE *revNormals,
					BYTE *revFaces, BYTE *showHelp)
{
	int i;

	cmdLSwitches[0]='\0';

	if(argc==MINARGS)
	{
		if(!strncmp(argv[1], "-?", 2))
			{
				*showHelp=V_TRUE;
				return(0);
			}

		return(1);
	}
	else
	{
		center->x=center->y=center->z=0;

		for(i=1; i<=argc-MINARGS; i++)
		{
			if(!strncmp(argv[i], "-cx", 3))
			{
				centerMode=ON;
				center->x=atof(argv[i]+3);
				
				strcat(cmdLSwitches, argv[i]);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-cy", 3))
			{
				centerMode=ON;
				center->y=atof(argv[i]+3);

				strcat(cmdLSwitches, argv[i]);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-cz", 3))
			{
				centerMode=ON;
				center->z=atof(argv[i]+3);

				strcat(cmdLSwitches, argv[i]);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-c", 2))
			{
				centerMode=AUTOCENTER;

				strncat(cmdLSwitches, argv[i], 2);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-f", 2))
			{
				if(argv[i][2]=='t')
					*fillerSize=USETAB;
				else
				{
					*fillerSize=atoi(argv[i]+2);
					if(*fillerSize<MINFILLERSIZE || *fillerSize>MAXFILLERSIZE)
						return(0);
				}

				strncat(cmdLSwitches, argv[i], 3);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-b", 2))
			{
				if(argv[i][2]=='t')
					*baseFillerSize=USETAB;
				else
				{
					*baseFillerSize=atoi(argv[i]+2);
					if(*baseFillerSize<MINFILLERSIZE || *baseFillerSize>MAXFILLERSIZE)
						return(0);
				}

				strncat(cmdLSwitches, argv[i], 3);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-it",3 ))
			{
				*inclXforms=V_TRUE;

				strncat(cmdLSwitches, argv[i], 3);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-rf",3 ))
			{
				*revFaces=V_TRUE;

				strncat(cmdLSwitches, argv[i], 3);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-rn",3 ))
			{
				*revNormals=V_TRUE;

				strncat(cmdLSwitches, argv[i], 3);
				strcat(cmdLSwitches, " ");
			}
			else if(!strncmp(argv[i], "-?", 2))
			{
				*showHelp=V_TRUE;
				return(0);
			}
#ifdef _V_DEBUGMODE_
			else if(!strncmp(argv[i], "-d", 2))			
				debugMode=V_TRUE;
#endif // _V_DEBUGMODE_
			else
				break;
		}

		if(cmdLSwitches[0]!='\0')
			cmdLSwitches[strlen(cmdLSwitches)-1]='\0';

		return(i);
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: getBaseName()
// 
// Description:
//	This function strips of everything (including the file extension) from the passed in
//	file name and only retains its base name. It then converts all characters of the base name
//	to lower case except for the first character which is converted to upper case. Non alpha
//	numeric characters are excluded from the resulting string.
// Example: C:\graphics\flower-1.wrl -> Flower1
// Note: If this function is called more than once, each returned base name should
//		 be copied to a different buffer prior to calling this function again. Otherwise
//		 the base name previously created will be overwritten.
/////////////////////////////////////////////////////////////////////////////////////////////
char *getBaseName(char *filename)
{	
	static char baseName[MAXBASENAME];
	char tmpStr[MAXBASENAME];

	int i, j;

	_splitpath(filename, NULL, NULL, tmpStr, NULL);

	for(i=j=0; tmpStr[i]!='\0'; i++)
		if(isalnum(tmpStr[i]))
			baseName[j++]=tolower(tmpStr[i]);

	baseName[j]='\0';
	baseName[0]=toupper(baseName[0]);

	return(baseName);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: sum()
// 
// Description:
//	This function recursivly calculates the sum of a sequence of numbers as x=n-1+n
//	with x=1 if n=1. The sum is returned to the calling function.
// Example:	input value: 5 -> 1+2+3+4+5 = 15
/////////////////////////////////////////////////////////////////////////////////////////////
int sum(int n)
{
	if(n==1)
		return(1);
	else
		return(sum(n-1)+n);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: generateFillers()
// 
// Description:
//	This function generates an array of fillers for each text indentation/nesting level.
//	The filler at each level is preceded by the base filler and level amount of
//	indentation fillers so that filler[level of text indentation/nesting] will yield the
//	correct amount of filler characters needed at this text indentation/nesting level.
//	filler[] is expected to be an array of character pointers.
//	This function calculates the amount of characters needed for all fillers together and
//	allocates memory of the resulting total size. It then populates each filler	within the
//	allocated array and assigns each filler's beginning address according to its level to
//	the appropriate pointer in the passed in array of character pointers (filler[]).
// Note: When these fillers are not needed anymore their memory space has to be deallocated.
//		 Call deleteFillers() for this purpose.
// Example: filler[x] contains: baseFiller characters + x*filler characters + '\0'
//			If baseFillerSize==3, fillerSize==TAB and x==5
//				Then
//				filler[x]  ="   \t\t\t\t\t\0" (3 spaces=base filler, 5 TABs=filler at
//											   the current level, string terminator)
//				filler[x+1]="   \t\t\t\t\t\t\0" (3 spaces=base filler, 6 TABs=filler at
//											     the current level, string terminator)
/////////////////////////////////////////////////////////////////////////////////////////////
BYTE generateFillers(char *filler[], int baseFillerSize, int fillerSize)
{
	int i, j;
	char *tmpPtr;
	char tmpFiller[MAXFILLERSIZE+1];
	char baseFiller[MAXFILLERSIZE+1];

#ifdef _V_DEBUGMODE_
	if(debugMode)
	{
		if(baseFillerSize!=USETAB)
		{
			if(fillerSize!=USETAB)
				printf("generating fillers of size: %d with a base filler of: %d\n",
					fillerSize, baseFillerSize);
			else
				printf("generating fillers of TABS with a base filler of: %d\n",
					baseFillerSize);
		}
		else
		{
			if(fillerSize!=USETAB)
				printf("generating fillers of size: %d with a base filler of TABS\n",
					fillerSize);
			else
				printf("generating fillers of TABS with a base filler of TABS\n");
		}
	}
#endif // _V_DEBUGMODE_

	// depending on baseFillerSize use a TAB or up to 7 spaces as the base filler
	if(baseFillerSize==USETAB)
	{
		baseFiller[0]='\t';
		baseFiller[1]='\0';
		baseFillerSize=1;
	}
	else
	{
		strncpy(baseFiller, BLANKS, baseFillerSize);
		baseFiller[baseFillerSize]='\0';
	}

	// depending on fillerSize use a TAB or up to 7 spaces as the filler
	if(fillerSize==USETAB)
	{
		tmpFiller[0]='\t';
		tmpFiller[1]='\0';
		fillerSize=1;
	}
	else
	{
		strncpy(tmpFiller, BLANKS, fillerSize);
		tmpFiller[fillerSize]='\0';
	}

	// allocate enough space for MAXINDENTS complete fillers
	if((tmpPtr=new char[baseFillerSize*MAXINDENTS+sum(MAXINDENTS-1)*fillerSize+MAXINDENTS])
		==NULL)
	{
		filler[0]=NULL;
		return(GENALLOCERR);
	}

	// generate the actual fillers for each indentation level
	for(i=0; i<MAXINDENTS; i++)
	{
		// assign the address of the new filler level to be generated
		filler[i]=tmpPtr;

		// insert the baseFiller 
		strcpy(filler[i], baseFiller);

		// now depending on the indentation level (i) insert however many fillers and
		// terminate this new complete filler level
		for(j=0; j<i; j++)
			strcpy(filler[i]+baseFillerSize+j*fillerSize, tmpFiller);
		filler[i][baseFillerSize+j*fillerSize]='\0';

		// advance the pointer to the beginning of the next filler level to generate
		tmpPtr+=baseFillerSize+j*fillerSize+1;
	}

	return(NOERR);
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: deleteFillers()
// 
// Description:
//	This function deallocates the memory allocated by generateFillers() provided it did not
//	fail (in which case it is expected that filler[0]==NULL).
/////////////////////////////////////////////////////////////////////////////////////////////
void deleteFillers(char *filler[])
{
	if(filler[0]!=NULL)
	{
		delete [] filler[0];
#ifdef _V_DEBUGMODE_
		if(debugMode)
			printf("deleted fillers\n");
#endif // _V_DEBUGMODE_
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Function Name: outputGenericErrorMsgs()
// 
// Description:
//	This function ouputs a generic error message for all the enumerators in sharedDefs.h
//	named GEN*ERR.
/////////////////////////////////////////////////////////////////////////////////////////////
void outputGenericErrorMsgs(BYTE errorCode)
{
	switch(errorCode)
	{	
		case GENFTELLERR:
		{
			fprintf(stderr, "Error getting file position\n");
			break;
		}
		case GENFSEEKERR:
		{
			fprintf(stderr, "Error setting file position\n");
			break;
		}
		case GENUNEXPEOFERR:
		{
			fprintf(stderr, "End of file unexpected\n");
			break;
		}
		case GENINPREADERR:
		{
			fprintf(stderr, "Error reading input file\n");
			break;
		}
		case GENHDRWRITEERR:
		{
			fprintf(stderr, "Error writing to header file\n");
			break;
		}
		case GENDATWRITEERR:
		{
			fprintf(stderr, "Error writing to data file\n");
			break;
		}
		case GENOUTPWRITEERR:
		{
			fprintf(stderr, "Error writing to OpenGL file\n");
			break;
		}
		case GENALLOCERR:
		{
			fprintf(stderr, "Error allocating memory\n");
			break;
		}
	}
}