/* Quat - A 3D fractal generation program */ 
/* Copyright (C) 1997,98 Dirk Meyer */ 
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */ 
/* mail:  Dirk Meyer */ 
/*        Marbacher Weg 29 */ 
/*        D-71334 Waiblingen */ 
/*        Germany */ 
/* */ 
/* This program is free software; you can redistribute it and/or */ 
/* modify it under the terms of the GNU General Public License */ 
/* as published by the Free Software Foundation; either version 2 */ 
/* of the License, or (at your option) any later version. */ 
/* */ 
/* This program is distributed in the hope that it will be useful, */ 
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */ 
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */ 
/* GNU General Public License for more details. */ 
/* */ 
/* You should have received a copy of the GNU General Public License */ 
/* along with this program; if not, write to the Free Software */ 
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */ 
 
#include <stdlib.h>
#include <time.h>
#include <math.h> /* floor */
#include <qapp.h>
#include <qwindow.h>
#include <qfont.h>
#include <qmsgbox.h>
#include <qfiledlg.h>
#include <qstrlist.h>
#include <qcolor.h>
#include <qscrbar.h>
#include "quat.h"
#include "menu.h"   // includes also main.h
#include "main.moc"
#include "ObjectEditor.h"
#include "ViewEditor.h"
#include "ColorEditor.h"
#include "IntersecEditor.h"
#include "Other.h"

#include <stdio.h>

extern VER_ReturnVideoInfo ReturnVideoInfo;
extern VER_SetColors SetColors;
extern VER_Initialize Initialize;
extern VER_Done Done;
extern VER_update_bitmap update_bitmap;
extern VER_getline getline;
extern VER_putline putline;
extern VER_check_event check_event;
extern VER_Change_Name Change_Name;
extern VER_Debug Debug;
extern VER_eol eol;

extern time_t calc_time;
time_t old_time = 0;

#define minsize 320

QuatMenu *menu;
QPixmap *pixmap;
QImage *RGBmap;
QCmds *cmds;
unsigned char *ZBuf;
int picx, picy, pixx, pixy, bmx, bmy;
int xadd, yadd;
bool auto_resize = TRUE, stop = FALSE;
int maxsizeX, maxsizeY;
QString act_file, act_name;

QCmds::QCmds(QObject *parent = 0, const char* name = 0) : QObject(parent, name)
{
   QWidget *d;
   char f1[stringlength], f2[stringlength], Error[stringlength];

   SetDefaults(&frac, &view, &realpal, colscheme, cutbuf);
   if (BuildName(f1, f2, ".png", "Noname", Error))
   {
      QMessageBox::critical(NULL, "Quat: Error", Error, "OK", NULL, NULL, 0, 0);
      qApp->quit();
   }
   act_file.setStr(f1);
   act_name = act_file;
   ImgInMem = FALSE; ZBufInMem = FALSE; ImgChanged = FALSE; ZBufChanged = FALSE;
   ImgReady = FALSE; ZBufReady = FALSE; InCalc = FALSE;
   RGBmap = new QImage();
   d = QApplication::desktop();
   maxsizeX = d->width(); maxsizeY = d->height();
   cmds = this;
}

void QCmds::SetMenuState()
{
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
}

void QCmds::DoImgOpen(const char *givenfile, int zflag)
{
   char Error[stringlength];
   QString filename;
   int xres, yres, xs, ys;
   unsigned char ready;

   if (!givenfile)
   {
      if (zflag==0) filename = QFileDialog::getOpenFileName(0, "*.png", 0, "Open Image");
      else filename = QFileDialog::getOpenFileName(0, "*.zpn", 0, "Open ZBuffer");
   }
   else filename.setStr(givenfile);
   if (!filename.isEmpty())
   {
      ReturnVideoInfo = (VER_ReturnVideoInfo)QT_ReturnVideoInfo;
      SetColors = (VER_SetColors)QT_SetColors;
      Initialize = (VER_Initialize)QT_Initialize;
      Done = (VER_Done)QT_Done;
      update_bitmap = (VER_update_bitmap)QT_update_bitmap;
      getline = (VER_getline)QT_getline;
      putline = (VER_putline)QT_putline;
      check_event = (VER_check_event)QT_check_event;
      Change_Name = (VER_Change_Name)QT_Change_Name;
      Debug = (VER_Debug)QT_Debug;
      eol = (VER_eol)QT_eol;
      if (ReadParametersPNG((const char *)filename, Error, &_f, &_v, &_r, _col, _cut))
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         if (zflag==0)
         { 
            ImgChanged = FALSE; DoImgClose();
         }
         else
         {
            ZBufChanged = FALSE; DoZBufClose();
         }
         return;
      }
      Restore();
      if (zflag == 0) xres = view.xres;
      else xres = view.xres*view.antialiasing;
      if (view.interocular!=0) xres *= 2;
      yres = view.yres;
      if (zflag != 0) yres *= view.antialiasing;
      if (DoInitMem(xres, yres, Error, zflag))
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         return;
      }
      if (ReadParametersAndImage(Error, (const char *)filename, &ready, &xs, &ys,
         &_f, &_v, &_r, _col, _cut, zflag))
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         if (zflag==0)
         {
            ImgChanged = 0; DoImgClose();
         }
         else
         {
            ZBufChanged = 0; DoZBufClose();
         }
         return;
      }
      if (zflag==0)
      {
         act_file = filename.copy(); 
         act_name = ((const char*)filename+1+filename.findRev('/', -1, TRUE));
         ImgReady = (ready != 0);
         ImgChanged = FALSE;
         ImgInMem = TRUE;
         imgxstart = xs;
         imgystart = ys;
         filename.sprintf("%s", (const char*)act_name);
         QT_Change_Name((const char*)filename);
         filename.sprintf("%s", "Image ");
      }
      else
      {
          ZBufReady = (ready != 0);
          ZBufChanged = FALSE;
          ZBufInMem = TRUE;
          zbufxstart = xs;
          zbufystart = ys;
          filename.sprintf("%s", "ZBuffer ");
      }
      if ((zflag==0 && ImgReady) || (zflag!=0 && ZBufReady)) 
         filename.sprintf("%s%s", (const char*)filename, "read successfully. Calculation has been completed.");
      else filename.sprintf("%s%s %i.", (const char*)filename, "read successfully. Calculation is in line ", ys);
      QMessageBox::information(0, "Quat: Message", (const char*)filename, "OK", 0, 0, 0, 0);
   }
   return;
}

void QCmds::ImageOpen()
{
   menu->setAllDisabled();
   menu->setCursor(waitCursor);
   DoImgOpen(NULL, 0);
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   menu->setCursor(arrowCursor);
}

int QCmds::ImageClose()
{
   char Error[stringlength], f1[stringlength], f2[stringlength];
   QPainter painter;
   long l;
   unsigned char R;
   int i, j;

   if (!ImgInMem) return(1);
   if (ImgChanged)
   {
      QMessageBox mb("Quat: Confirmation", "Image: There are unsaved modifications. Do you want to close?", 
         QMessageBox::Information, QMessageBox::Yes | QMessageBox::Escape,
         QMessageBox::No | QMessageBox::Default, 0, 0, 0);
      if (mb.exec() == QMessageBox::No) return(0);
   }
   delete pixmap;
   pixmap = 0;
   picx = 0; picy = 0; xadd = 0; yadd = 0; bmx = 0; bmy = 0;
   if (ZBufInMem)
   {
      menu->setCursor(waitCursor);
      picx = view.xres*view.antialiasing; 
      if (view.interocular!=0) picx *= 2;
      picy = view.yres*view.antialiasing;
      picx += 2; picy += 2;
      bmx = picx; bmy = picy;
      pixmap = new QPixmap(bmx, bmy, -1);
      if (!pixmap)
      {
         sprintf(Error, "Couldn't create ZBuffer pixmap (probably not enough memory.)");
         return(1);
      }
      painter.begin(pixmap);
      painter.drawRect(0, 0, bmx, bmy);   
      for (j=0; j<picy-2; j++)
         for (i=0; i<picx-2; i++)
         {
            l = (long)ZBuf[3*(i+j*(picx-2))]<<16 | (long)ZBuf[3*(i+j*(picx-2))+1]<<8 
               | (long)ZBuf[3*(i+j*(picx-2))+2];
            R = 255-(unsigned char)(2.55*(float)l/(float)view.zres);
            painter.setPen(QColor(R, R, R));
            painter.drawPoint(i+1,j+1);
         }
      painter.end();
      if (auto_resize) ImageAdjustWindow();
      xadd = (pixx-picx)/2; yadd = (pixy-picy)/2;
      if (xadd<0) xadd = 0; if (yadd<0) yadd = 0;
      yadd += menu->menu->heightForWidth(menu->width());
      menu->setCursor(arrowCursor);
      menu->repaint();
   }
   RGBmap->reset();
   ImgInMem = FALSE; ImgReady = FALSE; ImgChanged = FALSE;
   calc_time = old_time;
   imgxstart = 0; imgystart = 0;
   if (BuildName(f1, f2, ".png", "Noname", Error))
   {
      QMessageBox::critical(NULL, "Quat: Error", Error, "OK", NULL, NULL, 0, 0);
      qApp->quit();
   }
   if (!ZBufInMem) menu->TitleImage();
   act_file.setStr(f1);
   act_name = act_file;
   sprintf(Error, "%s", (const char *)act_name);
   QT_Change_Name(Error);
   if (!ZBufInMem && auto_resize) ImageAdjustWindow();
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   return(1);
}

void QCmds::ImageSave()
{
   char Error[stringlength];

   if (SavePNG(Error, (const char *)act_file, 0, imgystart, NULL, &frac, &view,
      &realpal, colscheme, cutbuf, 0)!=0)
   {
      QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      return;
   }
   ImgChanged = FALSE;
   QMessageBox::information(0, "Quat: Message", "Image written successfully.", 
      "OK", 0, 0, 0, 0);
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   return;
}

void QCmds::ImageSaveAs()
{
   FILE *f;
   char Error[stringlength];
   QString filename;
   filename = QFileDialog::getSaveFileName
      (0, "*.png", 0, "Save image as");
   if (!filename.isEmpty())
   {
      if (filename.find('.', 0, FALSE)==-1) 
         filename += ".png";
      if ((f = fopen((const char *)filename, "r")) != NULL)
      {
         QMessageBox mb("Quat: Confirmation", "Selected file already exists. Overwrite?", 
            QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Escape,
            QMessageBox::No | QMessageBox::Default, 0, 0, 0);
         fclose(f);
         if (mb.exec() == QMessageBox::No) return;
      }
      if (SavePNG(Error, (const char*)filename, 0, imgystart, NULL, &frac, &view,
         &realpal, colscheme, cutbuf, 0)!=0)
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         return;
      }
      ImgChanged = FALSE;
      QMessageBox::information(0, "Quat: Message", "Image written successfully.", 
         "OK", 0, 0, 0, 0);
      act_file = filename.copy(); 
      act_name = ((const char*)filename+1+filename.findRev('/', -1, TRUE));
      sprintf(Error, "%s", (const char*)act_name);
      QT_Change_Name(Error);
   }
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   return;
}

void QCmds::ImageAdjustWindow()
{
   pixx = picx; 
   if (pixx<minsize) pixx = minsize;
   if (pixx>maxsizeX) pixx = maxsizeX;
   pixy = picy;
   if (pixy>maxsizeY) pixy = maxsizeY;
   menu->resize(pixx, pixy + menu->menu->heightForWidth(pixx));
  // SetScrollBars();
}

void QCmds::ImageAbout()
{
   QMessageBox::about(0, "About Quat",
      PROGNAME " " PROGVERSION PROGSTATE "\n\n"
      "A 3D Fractal Generation Program\n"
      "Copyright (C) 1997,98 Dirk Meyer\n"
      "dirk.meyer@studbox.uni-stuttgart.de\n"
      "http://wwwcip.rus.uni-stuttgart.de/~phy11733/index_e.html\n"
      "Local manual: quateng.html (English) or quatger.html (German)\n\n"
      "This program is distributed under the terms of the\n"
      "GNU General Public License Version 2 (See COPYING.TXT for details)\n\n"
      "This program uses the zlib library to write PNG files.\n"
      "More info on zlib: http://www.cdrom.com/pub/infozip/zlib\n\n"
      "The Color Selection Tool was taken from the KDE project (GNU licensed)\n"
      "and slightly modified.\nMore info on K Desktop Environment (KDE): http://www.kde.org\n\n"
      COMMENT);
}

void QCmds::Exit()
{
   if (!DoImgClose()) return;
   if (!DoZBufClose()) return;
   qApp->quit();
}

void QCmds::DoStartCalc(char zflag)
{
   ReturnVideoInfo = QT_ReturnVideoInfo;
   SetColors = (VER_SetColors)QT_SetColors;
   Initialize = (VER_Initialize)QT_Initialize;
   Done = (VER_Done)QT_Done;
   update_bitmap = (VER_update_bitmap)QT_update_bitmap;
   getline = (VER_getline)QT_getline;
   putline = (VER_putline)QT_putline;
   check_event = (VER_check_event)QT_check_event;
   Change_Name = (VER_Change_Name)QT_Change_Name;
   Debug = (VER_Debug)QT_Debug;
   if (zflag != 2) eol = (VER_eol)QT_eol;
   else eol = (VER_eol)QT_eol_4;
   InCalc = TRUE;
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
}

void QCmds::CalculationStartImage()
{
   char Error[stringlength];
   int zflag, xres, i;

   if (ZBufInMem)
   {
      zflag = 2; old_time = calc_time;
   }
   else
   {
      zflag = 0; old_time = 0;
   }
   DoStartCalc(zflag);
   xres = view.xres; if (view.interocular != 0) xres *= 2;
   if (!RGBmap || RGBmap->isNull()) if (DoInitMem(xres, view.yres, Error, 0))
   {
      QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      InCalc = FALSE;
      menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
         InCalc);
      return;
   }
   Error[0] = 0;
   i = CreateImage(Error, &imgxstart, &imgystart, &frac, &view, &realpal, 
      colscheme, cutbuf, 4, zflag);
   InCalc = FALSE;
   if (i==0 || i==-2 || i==-128)
   {
      ImgInMem = TRUE; ImgReady = FALSE;
      ImgChanged = TRUE;
   }
   if (imgystart==view.yres) ImgReady = TRUE;
   if (i!=-2)
   {
      if (Error[0]!=0) QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
         InCalc);
   }
   return;
}

void QCmds::CalculationStartZBuf()
{
   char Error[stringlength];
   long xres;
   int i;

   Error[0] = 0;
   xres = view.xres*view.antialiasing;
   if (view.interocular!=0) xres *= 2;
   if (ZBuf==NULL) if (DoInitMem(xres, view.yres*view.antialiasing, Error, 1))
   {
      QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      return;
   }
   DoStartCalc(1);
   ZBufChanged = TRUE;
   i = CreateZBuf(Error, &zbufxstart, &zbufystart, &frac, &view, cutbuf, 4);
   if (i==0 || i==-2 || i==-128) ZBufInMem = TRUE;
   if (zbufystart==view.yres) ZBufReady = TRUE;
   if (i != -2)
   {
      if (Error[0] != 0)
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      InCalc = FALSE;
      menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
         InCalc);
   }
   return;
}

void QCmds::CalculationStop()
{
   stop = TRUE;
}

void QCmds::ParametersReset()
{
   if (!QMessageBox::information(0, "Quat: Confirmation", 
      "This will reset all parameters to their default values.",
      "OK", "Cancel", 0, 0, 1))
      SetDefaults(&frac, &view, &realpal, colscheme, cutbuf);
}

void QCmds::ParametersReadINI()
{
   int i = 0;
   char Error[stringlength];
   QString filename = QFileDialog::getOpenFileName
      (0, "*.ini", 0, "Read parameters from INI");
   if (!filename.isEmpty())
   {
      Backup();
      i = ParseINI((const char*)filename, Error, &_f, &_v, &_r, _col, _cut);
      if (i!=0)
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         return;
      }
      else
      {
         if (ZBufInMem)
         {
            for (i=0; i<3; i++) view.light[i] = _v.light[i];
            view.ambient = _v.ambient;
            view.phongmax = _v.phongmax;
            view.phongsharp = _v.phongsharp;
            realpal = _r;
            strncpy(colscheme, _col, 251);
            QMessageBox::information(0, "Quat: Information", 
               "Those parameters that can be modified for a Z Buffer have been read.",
               "OK", 0, 0, 0, 0);
         }
         else 
         {
//            if (stricmp((const char *)filename+filename.length()-4, ".col") == 0) 
//               memcpy(_cut, cutbuf, sizeof(double)*140);
            Restore();
            QMessageBox::information(0, "Quat: Information", 
               "INI file read successfully. Parameters added to current ones.\nMaybe you wanted to reset them before"
               " reading.", "OK", 0, 0, 0, 0);
         }
      }
   }
}

void QCmds::ParametersReadPNG()
{
   int i = 0;
   char Error[stringlength];
   QString filename = QFileDialog::getOpenFileName
      (0, "*.png", 0, "Read parameters from PNG");
   if (!filename.isEmpty())
   {
      Backup();
      i = ReadParametersPNG((const char*)filename, Error, &_f, &_v, &_r, _col, _cut);
      if (i!=0)
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         return;
      }
      else 
      {
         if (ZBufInMem)
         {
            for (i=0; i<3; i++) view.light[i] = _v.light[i];
            view.ambient = _v.ambient;
            view.phongmax = _v.phongmax;
            view.phongsharp = _v.phongsharp;
            realpal = _r;
            strncpy(colscheme, _col, 251);
            QMessageBox::information(0, "Quat: Information", 
               "Those parameters that can be modified for a Z Buffer have been read.",
               "OK", 0, 0, 0, 0);
         }
         else 
         {
            Restore();
            QMessageBox::information(0, "Quat: Information", 
               "Parameters read successfully.", "OK", 0, 0, 0, 0);
         }
      }
   }
}

void QCmds::DoSaveParamsAs(const char *def_suffix, unsigned char mode)
{
   FILE *f;
   char Error[stringlength];
   QString filter(def_suffix);
   QString filename;
   filter.insert(0, '*');
   filename = QFileDialog::getSaveFileName
      (0, (const char*)filter, 0, "Save parameters");
   if (!filename.isEmpty())
   {
      if (filename.find('.', 0, FALSE)==-1) 
         filename += def_suffix;
      if ((f = fopen((const char *)filename, "r")) != NULL)
      {
         QMessageBox mb("Quat: Confirmation", "Selected file already exists. Overwrite?", 
            QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Escape,
            QMessageBox::No | QMessageBox::Default, 0, 0, 0);
         fclose(f);
         if (mb.exec() == QMessageBox::No) return;
      }
      if (!WriteINI(Error, (const char *)filename, mode, &frac, &view, 
         &realpal, colscheme, cutbuf)) QMessageBox::information(
         0, "Quat: Message", "File written.", "OK", 0, 0, 0, 0);
      else QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 
         0, 0, 0);
   }
}

void QCmds::ParametersSaveAs()
{   
   DoSaveParamsAs(".ini", PS_OBJ | PS_VIEW | PS_COL | PS_OTHER);
}

void QCmds::ParametersObjectEditor()
{
   ObjectEditor *D = new ObjectEditor(0, "ObjectEditor", ImgInMem || ZBufInMem);
   D->SetData(frac.maxiter, frac.lvalue, frac.bailout, frac.formula,
      frac.c[0], frac.c[1], frac.c[2], frac.c[3]);
   if (D->exec())
   {
      D->GetData(&frac.maxiter, &frac.lvalue, &frac.bailout, &frac.formula,
         &frac.c[0], &frac.c[1], &frac.c[2], &frac.c[3]);
   }
}

void QCmds::ParametersViewEditor()
{
   ViewEditor *D = new ViewEditor(0, "ViewEditor", ImgInMem, ZBufInMem);
   D->SetData(&view.s[0], &view.up[0], &view.light[0], &view.Mov[0], view.LXR, view.interocular);
   if (D->exec())
   {
      D->GetData(view.s, view.up, view.light, view.Mov, &view.LXR, &view.interocular);
   }
}

void QCmds::ParametersColorEditor()
{
   ColorEditor *D = new ColorEditor(0, "ColorEditor", ImgInMem);
   D->SetData(&realpal, colscheme);
   if (D->exec())
   {
      D->GetData(&realpal, colscheme);
   }
}

void QCmds::ParametersIntersectionEditor()
{
   IntersecEditor *D = new IntersecEditor(0, "IntersecEditor", ImgInMem || ZBufInMem);
   D->SetData(cutbuf);
   if (D->exec())
   {
      D->GetData(cutbuf);
   }
}

void QCmds::ParametersOther()
{
   Other *D = new Other(0, "Other", ImgInMem, ZBufInMem);
   D->SetData(view.xres, view.yres, view.zres, view.phongmax,
      view.phongsharp, view.ambient, view.antialiasing, calc_time);
   if (D->exec())
   {
     D->GetData(&view.xres, &view.yres, &view.zres, &view.phongmax,
        &view.phongsharp, &view.ambient, &view.antialiasing);
   }
}

void QCmds::ZBufferOpen()
{
   menu->setAllDisabled();
   menu->setCursor(waitCursor);
   DoImgOpen(NULL, 1);
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   menu->setCursor(arrowCursor);
}

int QCmds::DoZBufClose()
{
   if (!ZBufInMem) return(1);
   if (ZBufChanged)
   {
      QMessageBox mb("Quat: Confirmation", "ZBuffer: There are unsaved modifications. Do you want to close?", 
         QMessageBox::Information, QMessageBox::Yes | QMessageBox::Escape,
         QMessageBox::No | QMessageBox::Default, 0, 0, 0);
      if (mb.exec() == QMessageBox::No) return(0);
   }
   free(ZBuf);
   ZBuf = NULL;
   ZBufInMem = FALSE; ZBufChanged = FALSE; ZBufReady = FALSE;
   delete pixmap;
   pixmap = NULL;
   bmx = 0; bmy = 0; picx = 0; picy = 0; xadd = 0; yadd = 0;
   calc_time = 0; old_time = 0;
   zbufxstart = 0; zbufystart = 0;
   menu->TitleImage();
   if (auto_resize) ImageAdjustWindow();
   QT_Change_Name((const char*)act_name);
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   menu->repaint();
   return(1);
}

int QCmds::ZBufferClose()
{
   return DoZBufClose();
}

void QCmds::ZBufferSave()
{
   char file[stringlength], Error[stringlength];

   if (BuildName(file, NULL, ".zpn", (const char*)act_file, Error))
   {
      QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      return;
   }
   if (SavePNG(Error, file, 0, zbufystart, NULL, &frac, &view, &realpal,
      colscheme, cutbuf, 1)!=0)
   {
      QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      return;
   }
   ZBufChanged = FALSE;
   sprintf(Error, "ZBuffer '%s' was written successfully.", file);
   QMessageBox::information(0, "Quat: Message", Error, "OK", 0, 0, 0, 0);
   return;
}

void QCmds::ZBufferSaveAs()
{
   FILE *f;
   char Error[stringlength];
   QString filename;
   filename = QFileDialog::getSaveFileName
      (0, "*.zpn", 0, "Save ZBuffer as");
   if (!filename.isEmpty())
   {
      if (filename.find('.', 0, FALSE)==-1) 
         filename += ".zpn";
      if ((f = fopen((const char *)filename, "r")) != NULL)
      {
         QMessageBox mb("Quat: Confirmation", "Selected file already exists. Overwrite?", 
            QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Escape,
            QMessageBox::No | QMessageBox::Default, 0, 0, 0);
         fclose(f);
         if (mb.exec() == QMessageBox::No) return;
      }
      if (SavePNG(Error, (const char*)filename, 0, zbufystart, NULL, &frac, &view,
         &realpal, colscheme, cutbuf, 1)!=0)
      {
         QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
         return;
      }
      ZBufChanged = FALSE;
      QMessageBox::information(0, "Quat: Message", "ZBuffer written successfully.", 
         "OK", 0, 0, 0, 0);
   }
   menu->ChangeState(ImgInMem, ZBufInMem, ImgChanged, ZBufChanged, ImgReady, ZBufReady,
      InCalc);
   return;
}

int QCmds::DoImgClose()
{
   return ImageClose();
}

void QCmds::SetScrollBars()
{
   bool hs = FALSE, vs = FALSE;
   int size = 16;

   pixx = menu->width();
   pixy = menu->height() - menu->menu->heightForWidth(pixx);
   if (picx>pixx) { hs = TRUE; pixy -= size; }
   if (picy>pixy) { vs = TRUE; pixx -= size; }
   if (picx>pixx && vs && !hs) { hs = TRUE; pixy -= size; }
   if (menu->HScroll)
   {
      menu->HScroll->hide();
      menu->removeChild(menu->HScroll);
      delete menu->HScroll;
      menu->HScroll = 0;
   }
   if (menu->VScroll)
   { 
      menu->VScroll->hide();
      menu->removeChild(menu->VScroll);
      delete menu->VScroll;
      menu->VScroll = 0;
   }
   if (hs)
   {
      menu->HScroll = new QScrollBar(QScrollBar::Horizontal, menu, "HSCROLL");
      if (!vs) menu->HScroll->setGeometry(0, menu->height()-size, menu->width(), size);
      else menu->HScroll->setGeometry(0, menu->height()-size, menu->width()-size, size);
      menu->HScroll->setRange(0, picx-pixx);
      connect(menu->HScroll, SIGNAL(valueChanged(int)), menu, SLOT(SetHScrollPos(int)) );
      menu->HScroll->show();
      xadd = 0; yadd -= size/2;
   }
   if (vs)
   {
      menu->VScroll = new QScrollBar(QScrollBar::Vertical, menu, "VSCROLL");
      if (!hs) menu->VScroll->setGeometry(menu->width()-size, menu->menu->heightForWidth(menu->width()),
         size, menu->height()-menu->menu->heightForWidth(menu->width()));
      else menu->VScroll->setGeometry(menu->width()-size, menu->menu->heightForWidth(menu->width()),
         size, menu->height()-menu->menu->heightForWidth(menu->width())-size);
      menu->VScroll->setRange(0, picy-pixy);
      connect(menu->VScroll, SIGNAL(valueChanged(int)), menu, SLOT(SetVScrollPos(int)) );
      menu->VScroll->show();
      yadd = menu->menu->heightForWidth(menu->width());
      xadd -= size/2;
   }
}

int QCmds::DoInitMem(int x, int y, char *Error, char zbuf)
{
   if (!zbuf)
   {
      RGBmap->reset();
      if (!RGBmap->create(x, y, 32, 0, QImage::IgnoreEndian))
      {
         if (ZBufInMem)
            sprintf(Error, "Couldn't create image (probably not enough memory.) Try to close the ZBuffer and calculate "
                           "directly.");
         else sprintf(Error, "Couldn't create image (probably not enough memory.)");   
         return(-1);
      }
   }
   else
   {
      ZBuf = (unsigned char*)malloc((size_t)x*(size_t)y*3L);
      if (!ZBuf)
      {
         sprintf(Error, "Couldn't allocate memory for ZBuffer.");
         return(-1);
      }  
   }
   return(0);
}

void QCmds::Backup()
{
    _f = frac; _v = view; _r = realpal; strncpy(_col, colscheme, 251);
   memcpy(_cut, cutbuf, 140*sizeof(double));
   return;
}

void QCmds::Restore()
{
   frac = _f; view = _v; realpal = _r; strncpy(colscheme, _col, 251);
   memcpy(cutbuf, _cut, 140*sizeof(double));
   return;
}

int QCmds::QT_ReturnVideoInfo(struct vidinfo_struct *vidinfo)
{
   vidinfo->maxcol = -1;   /* Qt simulates TrueColor through color cube */
   vidinfo->rdepth = QColor::numBitPlanes();
   vidinfo->gdepth = QColor::numBitPlanes();
   vidinfo->bdepth = QColor::numBitPlanes();
   vidinfo->rgam = 0.6; vidinfo->ggam = 0.8; vidinfo->bgam = 0.6;
   return(0);
}

int QCmds::QT_SetColors(struct disppal_struct *disppal)
{
// Color handling is done by Qt. Implementation not neccessary.
   return(0);
}

int QCmds::QT_Initialize(int x, int y, char *Error)
{
   QPainter painter;

   picx = x + 2; picy = y + 2; 
   /* Check whether image has right dimensions. It doesn't have if you switch from an
      AA-ZBuffer to an image */
   if (pixmap && (bmx!=picx || bmy!=picy))
   {
      delete pixmap;
      bmx = picx; bmy = picy;
      pixmap = new QPixmap(bmx, bmy, -1);
      if (!pixmap)
      {
         sprintf(Error, "Couldn't create pixmap (probably not enough memory.)");
         return(1);
      }
   }
   if (!pixmap)
   {
      bmx = picx; bmy = picy;
      pixmap = new QPixmap(bmx, bmy, -1);
      if (!pixmap)
      {
         sprintf(Error, "Couldn't create pixmap (probably not enough memory.)");
         return(1);
      }
   }
   if (auto_resize) cmds->ImageAdjustWindow();
   xadd = (pixx-picx)/2; yadd = (pixy-picy)/2;
   if (xadd<0) xadd = 0; if (yadd<0) yadd = 0;
   yadd += menu->menu->heightForWidth(menu->menu->width());
   painter.begin(pixmap);
   painter.drawRect(0, 0, bmx, bmy);   
   painter.end();
   return(0);
}

int QCmds::QT_Done()
{
   return(0);
}

int QCmds::QT_update_bitmap(long x1, long x2, long xres, int y, unsigned char *Buf, int which)
{
   static QImage img;
   static QPixmap pix(0, 0, -1);
   QPainter painter, painter2;
   int i;
   uint *p1, *p2;
   int chunk, start; 
   unsigned char R;
   long l;  

   switch (which)
   {
      case 0:
         chunk = 4;
         start = (int)(y/chunk)*chunk; 
         if (start == y && x1 == 0)
         {
            img.reset();
            img.create(xres, chunk, 32, 0, QImage::IgnoreEndian);
            img.fill(qRgb(0, 0, 0));
            pix.resize((int)xres, chunk);
            pix.fill(qRgb(0, 0, 0));
         }
         p1 = (uint *)RGBmap->scanLine(y);
         p2 = (uint *)img.scanLine(y % chunk);
         memcpy(p2+x1, p1+x1, 4*(x2-x1+1));
         painter.begin(pixmap);
         if ((y == start+chunk-1 || y == cmds->view.yres-1) 
            && x2 == xres-1)
         {
            pix.convertFromImage(img, 0);
            painter.drawPixmap(1, start+1, pix, 0, 0, -1, -1);
            painter.end();
            menu->repaint(1+xadd, start+1+yadd, xres, chunk, FALSE);
         }
         else
         {
            painter2.begin(&pix);
            for (i=x1; i<=x2; i++) 
            {
               painter2.setPen(QColor(Buf[3*i], Buf[3*i+1], Buf[3*i+2]));
               painter2.drawPoint(i, y % chunk);
            }
            painter2.end();
            painter.drawPixmap(1+x1, y+1, pix, x1, y % chunk, x2-x1+1, 1);
            painter.end();
            menu->repaint(x1+xadd+1, y+yadd+1, x2-x1+1, 1, FALSE);
         }
//         painter.drawPixmap(1, start+1, pix, 0, 0, -1, -1);
//         painter.end(); painter.flush();
//         menu->repaint(x1+xadd+1, start+yadd+1,
//            x2-x1+1, chunk, FALSE);
         break;
      default:
         painter.begin(pixmap);
         for (i=x1; i<=x2; i++)
         {
            l = (long)Buf[3*i]<<16 | (long)Buf[3*i+1]<<8 | (long)Buf[3*i+2];
            R = 255-(unsigned char)(2.55*(float)l/(float)cmds->view.zres);
            painter.setPen(QColor(R, R, R));
            painter.drawPoint(i+1,y+1);
         }
         painter.end();
         menu->repaint(x1+xadd, y+yadd, x2-x1+1, 2, FALSE);
   }
   return(0);
}

int QCmds::QT_getline(unsigned char *line, int y, long xres, int whichbuf)
{
   uint *p, i;

   switch (whichbuf)
   {
      case 0: 
         p = (uint *)RGBmap->scanLine(y);
         for (i=0; i<(uint)xres; i++) 
         {
            *(line+i*3)   = qRed(*(p+i));
            *(line+i*3+1) = qGreen(*(p+i));
            *(line+i*3+2) = qBlue(*(p+i));
         }
         break;
      case 1:
         memcpy(line, ZBuf+y*xres*3L, xres*3L);
         break;
      default: return(1);
   }
   return(0);
}

int QCmds::QT_putline(long x1, long x2, long xres, int y, unsigned char *Buf, int whichbuf)
{
   uint *p, i;

   switch (whichbuf)
   {
      case 0: 
         p = (uint *)RGBmap->scanLine(y);
         for (i=x1; i<=(uint)x2; i++) *(p+i) = qRgb(*(Buf+3*i), *(Buf+3*i+1), *(Buf+3*i+2));
         break;
      case 1:
         memcpy(ZBuf+3L*(y*xres+x1), Buf+3L*x1, (x2-x1+1)*3L);
         break;
      default: return(1);
   }
   return(0);
}

int QCmds::QT_check_event()
{
   qApp->processEvents(2);
   if (stop) { stop = FALSE; return(-128); }
   else return(0);
}

int QCmds::QT_Change_Name(const char *s)
{
   char s2[stringlength];
   
   strcpy(s2, "XQuat - ");
   strcat(s2, s);
   menu->setCaption(s2);
   return(0);
}

void QCmds::QT_Debug(char *s)
{
   QMessageBox::information(0, "Quat: Debug", s, "OK", 0, 0, 0, 0);
   return;
}

void QCmds::QT_eol(int line)
{
   char s[stringlength];

   sprintf(s, "%i lines of %s", line, (const char*)act_name);
   QT_Change_Name(s);
}

void QCmds::QT_eol_4(int line)
{
   if ((line-1) % 4 == 0 || line==cmds->view.yres) QT_eol(line);
}

int main(int argc, char **argv)
{
   QApplication::setColorSpec(QApplication::ManyColor);
   QApplication a(argc, argv);
   QCmds* cmds = new QCmds(&a, "");
   QFont font("Helvetica", 12, 0);
   char s[stringlength], Error[stringlength];

   if (argc > 2)
   {
      printf("Syntax: xquat <png file | zpn file | ini file>\n"); fflush(stdout);
      return(0);
   }
   menu = new QuatMenu(0, "mymenu", cmds);
   menu->TitleImage();
   a.setMainWidget(menu);
   a.setFont(font, TRUE);
   cmds->ImageAdjustWindow();
   sprintf(s, "%s", (const char *)act_name);
   QCmds::QT_Change_Name(s);
   menu->show();
   menu->ChangeState(FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE);
   if (argc==2)
   {
      menu->setAllDisabled();
      menu->setCursor(waitCursor);
      strcpy(s, argv[1]);
      if (strstr(s, ".ini") || strstr(s, ".INI")) 
      {
         if(ParseINI(s, Error, &cmds->frac, &cmds->view, &cmds->realpal, 
            cmds->colscheme, cmds->cutbuf))
            QMessageBox::warning(0, "Quat: Error", Error, "OK", 0, 0, 0, 0);
      }
      else if (strstr(s, ".png") || strstr(s, ".PNG"))
         cmds->DoImgOpen(s, 0);
      else if (strstr(s, ".zpn") || strstr(s, ".ZPN"))
         cmds->DoImgOpen(s, 1);
      cmds->SetMenuState();
      menu->setCursor(arrowCursor);
   }
   return a.exec();
}

