//
//
//
//
//
//
//
//
//
//
// Microsoft Windows 95/98/NT Version 
//
//Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Tim Little & Chuck Walbourn
//
// esprtgen.cpp
//
// Contains the code for the EschParticleGenerator class which creates
// particles based on stochastic parameters.
//
//

//
//
//                                Includes
//
//

#include "escher.hpp"

//
//
//                                 Macros
//
//

// Returns a uniform psuedo random number between -1.0 and 1.0
#define RANDOM() (float(flx_rand().flx) / 2147483647.0f)

//
//
//                                 Code
//
//

//
//  Constructors/Destructors  
//

//Ŀ
// EschParticleGenerator - Constructor                                      
//
EschParticleGenerator::EschParticleGenerator() :
    EschParticleSystem(),
    func(esch_generate_particle),
    data(0),
    pal(0),
    rot_rate(0),
    var_rot_rate(0),
    rot_rate_low(0),
    rot_rate_high(0),
    init_dir(0,1,0)
{
    dtyp = ESCH_DRWT_PGENERATOR;
};

EschParticleGenerator::EschParticleGenerator(ulong mc) :
    EschParticleSystem(),
    func(esch_generate_particle),
    data(0),
    pal(0),
    rot_rate(0),
    var_rot_rate(0),
    rot_rate_low(0),
    rot_rate_high(0),
    init_dir(0,1,0)
{
    dtyp = ESCH_DRWT_PGENERATOR;

    init(mc);
};


//Ŀ
// EschParticleGenerator - Destructor                                       
//
EschParticleGenerator::~EschParticleGenerator()
{
};



//
//  Operations  
//

//Ŀ
// EschParticleGenerator - init                                             
//
esch_error_codes EschParticleGenerator::init(ulong mc)
{
    part_accum = 0;

// Call base
    esch_error_codes err = EschParticleSystem::init(mc);
    if (err)
        return err;

// Set default values
    world.reset();                      // Align to world space
    set_sphere(1);                      // Sphere generation area of size 1
    set_parts(1);                       // 1 particle per second
    set_speed(1);                       // 1 unit per second speed
    set_size(0);                        // Size of 0
    set_lifetime(10);                   // 10 frame lifetime
    set_color(1);                       // Color index 1
    set_alpha(255);                     // Alpha level 255

    return ESCH_ERR_NONE;
}


//Ŀ
// EschParticleGenerator - animate                                          
//                                                                          
// This calls the base animate after generating new particles.              
//
void EschParticleGenerator::animate()
{
    if (!(flags & ESCH_PRTG_NOAUTOGENERATE))
        generate();

    EschParticleSystem::animate();
}


//Ŀ
// EschParticleGenerator - generate                                         
//                                                                          
// This call generates new particles.                                       
//
ulong EschParticleGenerator::generate()
{
// Determine how many particles to generate this frame based on
// distribution as either frame-based or time-based rates
    if (flags & ESCH_PRTG_FRAMEBASED)
    {
        if (var_parts > 0)
            part_accum += (mean_parts + RANDOM()*var_parts);
        else
            part_accum += mean_parts;
    }
    else
    {
        if (var_parts > 0)
            part_accum += (mean_parts + RANDOM()*var_parts) * interval;
        else
            part_accum += mean_parts * interval;
    }

// Create new particles, if any
    for(ulong count=0; part_accum >= 1; part_accum -= 1.0f, count++)
    {
        EschParticleGenParams parms(this,data);

        // Compute position/directory based on generation area in local
        if (flags & ESCH_PRTG_CIRCLE)
        {
            // Circular generation area on x/z plane of local coords
            assertMyth("EschParticleGenerator::generate needs valid circle radius",
                       gsize1 > 0);

            parms.pos.x = RANDOM() * gsize1;
            parms.pos.y = 0;
            parms.pos.z = RANDOM() * gsize1;

            parms.velocity.i = init_dir.i;
            parms.velocity.j = init_dir.j;
            parms.velocity.k = init_dir.k;

            assertMyth("EschParticleGenerator::generate needs valid max angle",
                       maxangle >= 0);

            if (maxangle > 0)
            {
                parms.velocity.rotatex(RANDOM()*maxangle);
                parms.velocity.rotatez(RANDOM()*maxangle);
            }
        }
        else if (flags & ESCH_PRTG_RECTANGLE)
        {
            // Rectangular generation area on x/z plane of local coords
            assertMyth("EschParticleGenerator::generate needs valid rectangle size",
                       (gsize1 > 0) && (gsize2 > 0));

            parms.pos.x = RANDOM() * gsize1;
            parms.pos.y = 0;
            parms.pos.z = RANDOM() * gsize2;

            parms.velocity.i = init_dir.i;
            parms.velocity.j = init_dir.j;
            parms.velocity.k = init_dir.k;

            assertMyth("EschParticleGenerator::generate needs valid max angle",
                       maxangle >= 0);

            if (maxangle > 0)
            {
                parms.velocity.rotatex(RANDOM()*maxangle);
                parms.velocity.rotatez(RANDOM()*maxangle);
            }
        }
        else
        {
            // Spherical generation area within local coords
            assertMyth("EschParticleGenerator::generate needs valid sphere radius",
                       gsize1 > 0);

            parms.pos.x = RANDOM() * gsize1;
            parms.pos.y = RANDOM() * gsize1;
            parms.pos.z = RANDOM() * gsize1;

            parms.velocity.i = parms.pos.x;
            parms.velocity.j = parms.pos.y;
            parms.velocity.k = parms.pos.z;

            float mag = parms.velocity.magnitude();
            if (mag > 0)
            {
                parms.velocity /= mag;
            }
            else
            {
                parms.velocity.i = init_dir.i;
                parms.velocity.j = init_dir.j;
                parms.velocity.k = init_dir.k;
            }
        }

        // Transform from local to world for position/velocity
        parms.pos.transform(&world);
        parms.velocity.transform(&world);

        // Compute speed and set into velocity
        if (var_speed > 0)
            parms.velocity *= mean_speed + RANDOM()*var_speed;
        else
            parms.velocity *= mean_speed;

        // Compute size
        if (var_size > 0)
        {
            parms.size = mean_size + RANDOM()*var_size;
            if (parms.size <= 0)
                parms.size = mean_size;
        }
        else
            parms.size = mean_size;

        // Compute color
        if (pal)
        {
            float           t;
            VngoColor24bit  clr;

            if (var_color_r > 0)
                t = float(mean_color_r) + RANDOM()*float(var_color_r);
            else
                t = float(mean_color_r);

            if (t > 255)
                t = 255;
            else if (t < 0)
                t = 0;

            clr.r = byte(t);

            if (var_color_g > 0)
                t = float(mean_color_g) + RANDOM()*float(var_color_g);
            else
                t = float(mean_color_g);

            if (t > 255)
                t = 255;
            else if (t < 0)
                t = 0;

            clr.g = byte(t);

            if (var_color_b > 0)
                t = float(mean_color_b) + RANDOM()*float(var_color_b);
            else
                t = float(mean_color_b);

            if (t > 255)
                t = 255;
            else if (t < 0)
                t = 0;

            clr.b = byte(t);

            if (pal->flags & (VNGO_15BIT|VNGO_16BIT))
            {
                parms.color = clr.compress24() | VNGO_COLOR_24BIT;
            }
            else
            {
                parms.color = pal->get_index(clr);
            }
        }
        else
        {
            parms.color = cindex;
        }

        // Compute lifetime
        if (var_life > 0)
        {
            parms.lifetime = mean_life + RANDOM()*var_life;
            if (parms.lifetime <= 0)
                parms.lifetime = mean_life;
        }
        else
            parms.lifetime = mean_life;

        // Compute alpha
        if (var_alpha > 0)
        {
            float a = float(mean_alpha) + RANDOM()*float(var_alpha);
            if (a > 255)
                a = 255;
            else if (a < 0)
                a = 0;

            parms.alpha = byte(a);
        }
        else
            parms.alpha = mean_alpha;

        if (var_rot_rate > 0)
        {
            float rr = rot_rate + RANDOM()*float(var_rot_rate);
        }
        else if (rot_rate_high || rot_rate_low)
        {
            parms.rot_rate = rot_rate + (rot_rate_high - (RANDOM()*float(rot_rate_high-rot_rate_low)));
        }
        else
            parms.rot_rate = rot_rate;



        // Create new particle with callback
        EschParticle *p = func(&parms);
        if (!p)
            break;

        // Add to system
        add(p);
    }

    return count;
}



//
//  Utility Routines  
//

//Ŀ
// EschParticleGenerator - set_callback                                     
//                                                                          
// This sets the callback function for the generator.                       
//
void EschParticleGenerator::set_callback(EschParticleGenCallback f, void *d)
{
    if (!f)
        func = esch_generate_particle;
    else
        func = f;

    data=d;
}



//
// Callback routines 
//

//
// esch_generate_particle
//
// Default callback routine which creates an instance of EschParticle
//
EschParticle *esch_generate_particle(EschParticleGenParams *parms)
{
    assert(parms != 0 && parms->generator != 0);

    EschParticle *p = new EschParticle(&parms->pos, &parms->velocity,
                                       parms->color, parms->lifetime);
    if (!p)
        return 0;

    if (!(parms->generator->flags & ESCH_PRTG_FRAMEBASED))
        p->set_flags(p->flags | ESCH_PRT_TIMEBASED);

    return p;
}


//
// esch_generate_pyramid
//
// Callback routine which creates an instance of EschParticlePyramid
//
EschParticle *esch_generate_pyramid(EschParticleGenParams *parms)
{
    assert(parms != 0 && parms->generator != 0);

    EschParticlePyramid *p = new EschParticlePyramid(parms->size,
                                                     &parms->pos,
                                                     &parms->velocity,
                                                     parms->color,
                                                     parms->lifetime);
    if (!p)
        return 0;

    if (!(parms->generator->flags & ESCH_PRTG_FRAMEBASED))
        p->set_flags(p->flags | ESCH_PRT_TIMEBASED);

    return p;
}


//
// esch_generate_sprite
//
// Callback routine which creates an instance of EschParticleSprite and
// an associated EschSprite instance.
//
EschParticle *esch_generate_sprite(EschParticleGenParams *parms)
{
    assert(parms != 0 && parms->generator != 0 && parms->data != 0);

    EschSprite *sprite = new EschSprite(*((EschSprite*)parms->data));
    if (!sprite)
        return 0;

    sprite->set_position(&parms->pos);
    sprite->set_alpha(parms->alpha);
    sprite->set_flags(sprite->flags & ~ESCH_DRW_SKIP);

    EschParticleSprite *p = new EschParticleSprite(sprite,
                                                   &parms->velocity,
                                                   parms->lifetime);
    if (!p)
    {
        delete sprite;
        return 0;
    }

    if (!(parms->generator->flags & ESCH_PRTG_FRAMEBASED))
        p->set_flags(p->flags | ESCH_PRT_TIMEBASED);

    p->set_rot_rate(parms->rot_rate);

    return p;
}

//
// esch_generate_line
//
// Callback routine which creates an instance of EschParticleLine
//
EschParticle *esch_generate_line(EschParticleGenParams *parms)
{
    assert(parms != 0 && parms->generator != 0);

    EschParticleLine *p = new EschParticleLine(&parms->pos, &parms->velocity,
                                               parms->color, parms->lifetime);
    if (!p)
        return 0;

    if (!(parms->generator->flags & ESCH_PRTG_FRAMEBASED))
        p->set_flags(p->flags | ESCH_PRT_TIMEBASED);

    return p;
}


// End of module - esprtgen.cpp 

