#ifndef H_CON_MATH3D
#define H_CON_MATH3D

/** CON_Math3D.h:  This file defines the Vector3D, Plane, Quat,
    Matrix3, Matrix3D and Transformable types (structures) and their operators.
*/

#include <CON_Utils.h>

struct Quat;
struct Matrix3;
struct Matrix3D;

struct Vector3D {
   float x;
   float y;
   float z;

// =====================================
// Constructors
// =====================================

   Vector3D() {}
   Vector3D(const Vector3D& v)               { x=v.x; y=v.y; z=v.z; }
   Vector3D(const float f)                   { x = y = z = f; }
   Vector3D(const float _x, const float _y, const float _z) { x = _x; y = _y; z = _z; }
   Vector3D(const float f[3])                { x = f[0]; y = f[1]; z = f[2]; }

#ifdef _D3DTYPES_H_
   Vector3D(const D3DVECTOR& v)              { x = v.x; y = v.y; z = v.z; }
#endif
      
// =====================================
// Access grants
// =====================================

   const float& operator[](const int i) const { return (&x)[i]; }
   float& operator[](const int i)             { return (&x)[i]; }

// =====================================
// Assignment operators
// =====================================

   Vector3D& operator+=(const Vector3D& v)   { x += v.x; y += v.y; z += v.z; return *this; }
   Vector3D& operator-=(const Vector3D& v)   { x -= v.x; y -= v.y; z -= v.z; return *this; }
   Vector3D& operator*=(const Vector3D& v)   { x *= v.x; y *= v.y; z *= v.z; return *this; }
   Vector3D& operator*=(const float s)       { x *= s;   y *= s;   z *= s;   return *this; }
   Vector3D& operator/=(const Vector3D& v)   { x /= v.x; y /= v.y; z /= v.z; return *this; }
   Vector3D& operator/=(const float s)       { x /= s;   y /= s;   z /= s;   return *this; }
   Vector3D& operator*=(const Matrix3& m)    { return *this = (*this) * m; }
   Vector3D& operator*=(const Matrix3D& m)   { return *this = (*this) * m; }

// =====================================
// Unary operators
// =====================================

   const Vector3D& operator+() const         { return *this; }
   Vector3D  operator-() const               { return +Vector3D(-x, -y, -z); }

// =====================================
// Binary operators
// =====================================

// Addition and subtraction
   friend Vector3D operator+(const Vector3D&, const Vector3D&);
   friend Vector3D operator-(const Vector3D&, const Vector3D&);

// Scalar multiplication and division
   friend Vector3D operator*(const Vector3D&, const float);
   friend Vector3D operator*(const float, const Vector3D&);
   friend Vector3D operator/(const Vector3D&, const float);

// Equality
   friend int operator==(const Vector3D&, const Vector3D&);
   friend int operator!=(const Vector3D&, const Vector3D&);

// Vector dominance
   friend int operator< (const Vector3D&, const Vector3D&);
   friend int operator<=(const Vector3D&, const Vector3D&);
   friend int operator> (const Vector3D&, const Vector3D&);
   friend int operator>=(const Vector3D&, const Vector3D&);

// Dot Product
   friend float operator*(const Vector3D&, const Vector3D&);

// Transformation
   friend Vector3D operator*(const Vector3D&, const Matrix3&);
// the following assumes m(0,3)-m(3,3) == {0,0,0,1}  see Transform for full version
   friend Vector3D operator*(const Vector3D&, const Matrix3D&);
   friend Vector3D rotateVector(const Vector3D&, const Matrix3D&);
   friend Vector3D Transform(const Vector3D&, const Matrix3D&);
// rotate a vector through a unit quaternion
   friend Vector3D operator*(const Vector3D, const Quat&);


   friend Vector3D CrossProduct(const Vector3D u, const Vector3D& v);

// Length-related functions
   float SquareMagnitude() const             { return x*x + y*y + z*z; }
   float Magnitude() const                   { return (float) sqrt(SquareMagnitude()); }

// Returns vector with same direction and unit length
   Vector3D Normalized() const               { return (*this / Magnitude()); }
   Vector3D& Normalize()                     { return *this /= Magnitude(); }

// Return min/max component of the input vector
   float Min() const			     { return ::Min(::Min(x,y),z); }
   float Max() const			     { return ::Max(::Max(x,y),z); }

// Return memberwise min/max of input vectors
   friend Vector3D Minimize(const Vector3D&, const Vector3D&);
   friend Vector3D Maximize(const Vector3D&, const Vector3D&);


   static const Vector3D Zero;  // ( 0, 0, 0)   zero vector
   static const Vector3D I;     // ( 1, 0, 0)   unit vector in +x direction
   static const Vector3D J;     // ( 0, 1, 0)   unit vector in +y direction
   static const Vector3D K;     // ( 0, 0, 1)   unit vector in +z direction
   static const Vector3D _I;    // (-1, 0, 0)   unit vector in -x direction
   static const Vector3D _J;    // ( 0,-1, 0)   unit vector in -y direction
   static const Vector3D _K;    // ( 0, 0,-1)   unit vector in -z direction
};


struct Plane {
   Vector3D normal;
   float d;

   Plane() { }
   Plane(const Vector3D& _n, const float _d) { normal = _n; d = _d; }
   Plane(const Vector3D a, const Vector3D b, const Vector3D c)
      { SetFromTri(a, b, c); }

   Plane& SetFromTri(const Vector3D a, const Vector3D b, const Vector3D c);

   float Classify(const Vector3D v) const { return v * normal + d; }
};



struct Quat {  // quaternions
   float x;
   float y;
   float z;
   float w;

// Constructors
   Quat() { }
   Quat(const float _x, const float _y, const float _z, const float _w)
      { x = _x; y = _y; z = _z; w = _w; }

// construct from a normalized axis and angle to a unit quaternion.
   Quat(const Vector3D& axis, const float theta) {
      setAxisAngle(axis, theta);
   }

// construct from euler angles to a unit quaternion.
   Quat(const float Yaw, const float Pitch, const float Roll) {
      setAngles(Yaw, Pitch, Roll);
   }

// convert from a normalized axis and angle to a unit quaternion.
   Quat& setAxisAngle(const Vector3D& axis, const float theta);

// convert to a normalized axis and angle from a unit quaternion.
   void getAxisAngle(Vector3D& axis, float &theta) const
   {
      theta = (float) (acos(w) * 2);
      float s = (float) sin(theta / 2);
      axis.x = x / s;
      axis.y = y / s;
      axis.z = z / s;
   }

// Convert euler angles to a unit quaternion.
   Quat& setAngles(const float Yaw, const float Pitch, const float Roll);

// Convert a unit quaternion into a rotation matrix.
   Matrix3 getMatrix() const;

// Convert a rotation matrix into a unit quaternion.
   Quat& setMatrix(const Matrix3&);
   Quat& setMatrix(const Matrix3D&);

   Vector3D& imaginaryPart()        { return (Vector3D&) x; }

// assignment operators
   Quat& operator+=(const Quat& q)  { x+=q.x; y+=q.y; z+=q.z; w+=q.w; return *this; }
   Quat& operator-=(const Quat& q)  { x-=q.x; y-=q.y; z-=q.z; w-=q.w; return *this; }
   Quat& operator*=(const Quat& q)  { return *this = (*this) * q; }
   Quat& operator*=(const float s)  { x*=s;   y*=s;   z*=s;   w*=s;   return *this; }
   Quat& operator/=(const float s)  { x/=s;   y/=s;   z/=s;   w/=s;   return *this; }

// unary operators
   Quat operator-() const          { return Quat(-x,-y,-z,-w);    } // unary minus
   const Quat& operator+() const   { return *this; } // unary plus

// quaternion addition and subtraction
   friend Quat operator+(const Quat&, const Quat&);
   friend Quat operator-(const Quat&, const Quat&);

// scalar multiplication and division
   friend Quat operator*(const Quat&, const float);
   friend Quat operator*(const float, const Quat&);
   friend Quat operator/(const Quat&, const float);

// quaternion multiplication
   friend Quat operator*(const Quat&, const Quat&);
   

// rotate quaternion with by a rotataion matrix
   friend Quat operator*(const Quat&, const Matrix3&);

// rotate a vector through a unit quaternion
   friend Vector3D operator*(const Vector3D&, const Quat&);

   friend int operator==(const Quat&, const Quat&);
   friend int operator!=(const Quat&, const Quat&);

   Quat  Conjugate()  const { return Quat(-x,-y,-z,w);     }
   float Norm()       const { return (x+y+z+w)*2;          }
   Quat  InverseGen() const { return Conjugate() / Norm(); }
   Quat  Inverse()    const { return Conjugate();          }

   Quat  Normalized() const;
   Quat& Normalize()        { return *this = Normalized(); }
      // for unit quaternion, the Norm == 1

// Compute a quaternion which is the spherical linear interpolation
// between two other quaternions by alpha.
   friend Quat QuatSlerp(Quat q, Quat r, const float alpha);

   static const Quat Zero;      // (0, 0, 0, 0)  Zero quaternion
   static const Quat Identity;  // (0, 0, 0, 1)  Identity quaternion
};


struct Matrix3 {
   union {
      struct {
         float _11, _12, _13;
         float _21, _22, _23;
         float _31, _32, _33;
      };
      float m[3][3];
   };

// Constructors
   Matrix3() { }
   Matrix3(const float f) 
   {
      _11 = _12 = _13 =
      _21 = _22 = _23 =
      _31 = _32 = _33 = f;
   }

   Matrix3(const float f11, const float f12, const float f13, 
           const float f21, const float f22, const float f23,
           const float f31, const float f32, const float f33)
   {
      _11 = f11; _12 = f12; _13 = f13;
      _21 = f21; _22 = f22; _23 = f23;
      _31 = f31; _32 = f32; _33 = f33;
   }

// construct with rotation matrix from a Matrix3D
   Matrix3(const Matrix3D&);

// construct from a quaternion
   Matrix3(const Quat q) { *this = q.getMatrix(); }

// access grants
   float& operator()(const int row, const int col) { return m[row][col]; }
   const float& operator()(const int row, const int col) const { return m[row][col]; }

   Vector3D& baseX() { return (Vector3D&)_11; } // I (rotation space)
   Vector3D& baseY() { return (Vector3D&)_21; } // J
   Vector3D& baseZ() { return (Vector3D&)_31; } // K

// operators
   Matrix3& operator+=(const Matrix3&);
   Matrix3& operator-=(const Matrix3&);
   Matrix3& operator*=(const Matrix3& m) { return *this = *this * m; }
   Matrix3& operator*=(const float);

   friend int operator==(const Matrix3&, const Matrix3&);
   friend int operator!=(const Matrix3&, const Matrix3&);

   Matrix3 operator-() const; // unary minus
   const Matrix3& operator+() const { return *this; } // unary plus

   friend Matrix3 operator*(const Matrix3&, const Matrix3&);
   friend Matrix3 operator+(const Matrix3&, const Matrix3&);
   friend Matrix3 operator-(const Matrix3&, const Matrix3&);

   friend Matrix3 operator*(const Matrix3&, const float);
   friend Matrix3 operator*(const float, const Matrix3&);
   friend Matrix3 operator/(const Matrix3&, const float);

// Transformation
   friend Vector3D operator*(const Vector3D&, const Matrix3&);

// misc functions
   Matrix3 Transpose() const
   {
      return Matrix3(_11, _21, _31,
                     _12, _22, _32,
                     _13, _23, _33);
   }

   int  isSingular() const;
   float Determinant() const;

   Matrix3 Inverse() const;
   Matrix3 Adjoint() const;

   Matrix3& Reorthogonalize(int limit);


   Matrix3& Identify()  { return *this = Identity; }

   friend const Matrix3& IdentityMatrix3();
   friend const Matrix3& ZeroMatrix3();

   friend Matrix3 ScaleMatrix3(const float s);
   friend Matrix3 ScaleMatrix3(const Vector3D s);

   friend Matrix3 PitchMatrix3(const float theta);
   friend Matrix3 YawMatrix3(const float theta);
   friend Matrix3 RollMatrix3(const float theta);

   friend Matrix3 PitchMatrix3(const float cos_theta, const float sin_theta);
   friend Matrix3 YawMatrix3(const float cos_theta, const float sin_theta);
   friend Matrix3 RollMatrix3(const float cos_theta, const float sin_theta);

   /** generates a rotation matrix oriented to point in the given direction. */
   friend Matrix3 DirectionRotateMatrix3(const Vector3D& Dir);


   static const Matrix3 Zero;      // Zero matrix
   static const Matrix3 Identity;  // Identity matrix
};


struct Matrix3D {
   union {
      struct {
         float _11, _12, _13, _14;
         float _21, _22, _23, _24;
         float _31, _32, _33, _34;
         float _41, _42, _43, _44;
      };
      float m[4][4];
   };

// Constructors
   Matrix3D()  { }
   Matrix3D(const float f) 
   {
      _11 = _12 = _13 = _14 =
      _21 = _22 = _23 = _24 = 
      _31 = _32 = _33 = _34 = 
      _41 = _42 = _43 = _44 = f;
   }

   Matrix3D(const float f11, const float f12, const float f13, const float f14,
            const float f21, const float f22, const float f23, const float f24,
            const float f31, const float f32, const float f33, const float f34,
            const float f41, const float f42, const float f43, const float f44)
   {
      _11 = f11; _12 = f12; _13 = f13; _14 = f14;
      _21 = f21; _22 = f22; _23 = f23; _24 = f24;
      _31 = f31; _32 = f32; _33 = f33; _34 = f34;
      _41 = f41; _42 = f42; _43 = f43; _44 = f44;
   }

   Matrix3D(const Matrix3& n)
   {
      _11 = n._11; _12 = n._12; _13 = n._13;
      _21 = n._21; _22 = n._22; _23 = n._23;
      _31 = n._31; _32 = n._32; _33 = n._33;
      _14 = _24 = _34 = _41 = _42 = _43 = 0.0f;
      _44 = 1.0f;
   }

   Matrix3D(const Matrix3& n, const Vector3D v)
   {
      _11 = n._11; _12 = n._12; _13 = n._13;
      _21 = n._21; _22 = n._22; _23 = n._23;
      _31 = n._31; _32 = n._32; _33 = n._33;
      _41 = v.x; _42 = v.y; _43 = v.z; _44 = 1.0f;
      _14 = _24 = _34 = 0.0f;
   }

   Matrix3D(const Quat q) { *this = q.getMatrix(); }

#ifdef _D3DTYPES_H_
   Matrix3D(const D3DMATRIX& n)
   {
      _11 = n._11; _12 = n._12; _13 = n._13; _14 = n._14;
      _21 = n._21; _22 = n._22; _23 = n._23; _24 = n._24;
      _31 = n._31; _32 = n._32; _33 = n._33; _34 = n._34;
      _41 = n._41; _42 = n._42; _43 = n._43; _44 = n._44;
   }
#endif

// access grants
   float& operator()(const int row, const int col) { return m[row][col]; }
   const float& operator()(const int row, const int col) const { return m[row][col]; }

   Vector3D& baseX() { return (Vector3D&)_11; } // I (rotation space)
   Vector3D& baseY() { return (Vector3D&)_21; } // J
   Vector3D& baseZ() { return (Vector3D&)_31; } // K
   Vector3D& baseT() { return (Vector3D&)_41; } // translation
   
// operators
   Matrix3D& operator+=(const Matrix3D&);
   Matrix3D& operator-=(const Matrix3D&);
   Matrix3D& operator*=(const Matrix3D& m)  { return *this = (*this) * m; }
   Matrix3D& operator*=(const float);

   friend int operator==(const Matrix3D&, const Matrix3D&);
   friend int operator!=(const Matrix3D&, const Matrix3D&);

   Matrix3D operator-() const;   // unary minus
   const Matrix3D& operator+() const { return *this; } // unary plus

   friend Matrix3D operator*(const Matrix3D&, const Matrix3D&);
   friend Matrix3D operator+(const Matrix3D&, const Matrix3D&);
   friend Matrix3D operator-(const Matrix3D&, const Matrix3D&);

   friend Matrix3D operator*(const Matrix3D&, const float s);
   friend Matrix3D operator*(const float s, const Matrix3D& m);
   friend Matrix3D operator/(const Matrix3D&, const float s);

// Transformation
// the following assumes m(1,4)-m(4,4) == {0.0f,0.0f,0.0f,1}  see Transform for full version
   friend Vector3D operator*(const Vector3D&, const Matrix3D&);

   friend Vector3D Transform(const Vector3D&, const Matrix3D&);

// misc functions
   Matrix3D Transpose() const
   {
      return Matrix3D(_11, _21, _31, _41,
                      _12, _22, _32, _42,
                      _13, _23, _33, _43,
                      _14, _24, _34, _44);
   }

   int  isSingular() const;
   float Determinant() const;

   Matrix3D Inverse() const;
   Matrix3D Adjoint() const;

   Matrix3D& Reorthogonalize(int limit);


   Matrix3D& Identify()    { return *this = IdentityMatrix3D(); }

   friend const Matrix3D& IdentityMatrix3D();
   friend const Matrix3D& ZeroMatrix3D();

   friend Matrix3D TranslateMatrix3D(const Vector3D& v);
   friend Matrix3D ScaleMatrix3D(const float s);
   friend Matrix3D ScaleMatrix3D(const Vector3D& s);

   friend Matrix3D PitchMatrix3D(const float theta);
   friend Matrix3D YawMatrix3D(const float theta);
   friend Matrix3D RollMatrix3D(const float theta);

   friend Matrix3D PitchMatrix3D(const float cos_theta, const float sin_theta);
   friend Matrix3D YawMatrix3D(const float cos_theta, const float sin_theta);
   friend Matrix3D RollMatrix3D(const float cos_theta, const float sin_theta);


   /** generates a rotation matrix oriented to point in the given direction. */
   friend Matrix3D DirectionRotateMatrix3D(const Vector3D& Dir);

// Helper functions

   /** sets _41, _42 and _43 to v.x, v.y, v.z respectively  */
   void setPosition(const Vector3D& v);

   /** returns _41, _42, and _43 as a Vector3D */
   const Vector3D& getPosition() const;

   /** sets the orientation portion of the matrix such that forward()
       will move toward dir and that the up vector points toward up.
       Neither dir nor up must be normalized. */
   void setDirection(const Vector3D& dir, const Vector3D& up);

   /** fetch the forward direction of the orientation as a vector scaled by Size. */
   Vector3D getDirection(const float Size=1.0f) const;

   /** fetch the up direction of the orientation scaled by Size. */
   Vector3D getDirectionUp(const float Size=1.0f) const;

   /** fetch the "right" direction of the orientation as a vector scaled by Size. */
   Vector3D getDirectionRight(const float Size=1.0f) const;

   /** relocate the position d units along the forward vector. */
   void forward(const float d);

   /** translate the position by v.  */
   void relocate(const Vector3D& v);

   /** rotate the matrix by the euler angles given in v.  */
   void rotate(const Vector3D& v);

   /** scale the matrix by s. */
   void scale(const float s);

   /** align this matrix with the matrix given by m.  (simply assigns m to this). */
   void align(const Matrix3D& m);


   static const Matrix3D Zero;      // Zero matrix
   static const Matrix3D Identity;  // Identity matrix
};


class Transformable {
public:
   Transformable(const Matrix3D& m=IdentityMatrix3D(), const float scale=1.0f) : m_Transform(m), m_Scale(scale) {}
   Transformable(const Transformable& t1, const Transformable& t2) : 
                m_Transform(t2.m_Transform * t1.m_Transform), m_Scale(t2.m_Scale * t1.m_Scale) {}
   Matrix3D m_Transform;
   float    m_Scale;      // use scale() for scaling changes to the Transform
                          // this is for compatibility with RAPID

  /** Fetches a reference to the transformation matrix. */
   const Matrix3D& getTransform() const;

// Helper functions  (positioning and orientation)

   /** sets _41, _42 and _43 to v.x, v.y, v.z respectively  */
   void     setPosition(const Vector3D& v);

   /** returns _41, _42, and _43 as a Vector3D. */
   Vector3D getPosition() const;

   /** sets the orientation portion of the matrix such that forward()
       will move toward dir and that the up vector points toward up.
       Neither dir nor up must be normalized. */
   void     setDirection(const Vector3D& dir, const Vector3D& up);

   /** fetch the forward direction of the orientation as a vector scaled by Size. */
   Vector3D getDirection(const float Size=1.0f) const;

   /** fetch the up direction of the orientation scaled by Size. */
   Vector3D getDirectionUp(const float Size=1.0f) const;

   /** fetch the "right" direction of the orientation as a vector scaled by Size. */
   Vector3D getDirectionRight(const float Size=1.0f) const;

   /** relocate the position d units along the forward vector. */
   void forward(const float d);

   /** translate the position by v.  */
   void relocate(const Vector3D& v);

   /** rotate the matrix by the euler angles given in v.  */
   void rotate(const Vector3D& v);

   /** scale the matrix by s. */
   void scale(const float s);

   /** retrieve m_Scale */
   float getScale() const;

   /** align this matrix with the matrix within t.  (simply assigns t.m to this->m). */
   void align(const Transformable* t);

   /* Same as Matrix3D *, but also mulitiples m_scale */
   friend Transformable operator*(const Transformable&, const Transformable&);

   /* Transfrom this Transformable by another (almost same as Matrix3D *=, but also mulitiples m_scale. */
   Transformable& operator*=(const Transformable&);
};


// *************************************
// inline Implemetations
// *************************************

inline Vector3D
operator+(const Vector3D& u, const Vector3D& v)
{
   return Vector3D(u.x+v.x, u.y+v.y, u.z+v.z);
}


inline Vector3D
operator-(const Vector3D& u, const Vector3D& v)
{
   return Vector3D(u.x-v.x, u.y-v.y, u.z-v.z); 
}


inline Vector3D
operator*(const Vector3D& v, const float s)
{
   return Vector3D(v.x*s, v.y*s, v.z*s); 
}


inline Vector3D
operator*(const float s, const Vector3D& v)
{
   return Vector3D(s*v.x, s*v.y, s*v.z);
}

inline Vector3D
operator/(const Vector3D& v, const float s)
{
   return Vector3D(v.x/s, v.y/s, v.z/s); 
}


inline int
operator==(const Vector3D& u, const Vector3D& v)
{
   return ISEQ(u.x, v.x) && ISEQ(u.y, v.y) && ISEQ(u.z, v.z); 
}


inline int
operator!=(const Vector3D& u, const Vector3D& v)
{
   return !(u == v);
}


inline int
operator<(const Vector3D& u, const Vector3D& v)
{
   return u.x < v.x && u.y < v.y && u.z < v.z; 
}


inline int
operator<=(const Vector3D& u, const Vector3D& v)
{
   return u.x <= v.x && u.y <= v.y && u.z <= v.z; 
}


inline int
operator>(const Vector3D& u, const Vector3D& v)
{
   return !(u <= v);
}


inline int
operator>=(const Vector3D& u, const Vector3D& v)
{
   return !(u < v);
}


inline float
operator*(const Vector3D& u, const Vector3D& v)
{
   return u.x*v.x + u.y*v.y + u.z*v.z; 
}


inline Vector3D
CrossProduct(const Vector3D u, const Vector3D& v)
{
   return Vector3D(u.y * v.z - u.z * v.y,
                   u.z * v.x - u.x * v.z,
                   u.x * v.y - u.y * v.x);
}

inline Vector3D 
Minimize(const Vector3D& u, const Vector3D& v)
{
   return Vector3D(Min(u.x, v.x), Min(u.y, v.y), Min(u.z, v.z));
}

inline Vector3D 
Maximize(const Vector3D& u, const Vector3D& v)
{
   return Vector3D(Max(u.x, v.x), Max(u.y, v.y), Max(u.z, v.z));
}


inline int
operator==(const Quat& u, const Quat& v)
{
   return ISEQ(u.x, v.x) && ISEQ(u.y, v.y) && ISEQ(u.z, v.z) && ISEQ(u.w, v.w); 
}


inline int
operator!=(const Quat& u, const Quat& v)
{
   return !(u == v);
}


inline Quat
operator+(const Quat& q, const Quat& r)
{
   return Quat(q.x+r.x, q.y+r.y, q.z+r.z, q.w+r.w);
}


inline Quat
operator-(const Quat& q, const Quat& r)
{
   return Quat(q.x-r.x, q.y-r.y, q.z-r.z, q.w-r.w);
}


inline Quat
operator*(const Quat& q, const float s)
{
   return Quat(q.x * s, q.y * s, q.z * s, q.w * s);
}


inline Quat
operator*(const float s, const Quat& q)
{
   return Quat(q.x * s, q.y * s, q.z * s, q.w * s);
}


inline Quat
operator/(const Quat& q, const float s)
{
   return Quat(q.x / s, q.y / s, q.z / s, q.w / s);
}



inline Matrix3::Matrix3(const Matrix3D& n)
{
   _11 = n._11; _12 = n._12; _13 = n._13;
   _21 = n._21; _22 = n._22; _23 = n._23;
   _31 = n._31; _32 = n._32; _33 = n._33;
}


inline const Matrix3&
IdentityMatrix3()
{
   return Matrix3::Identity;
}


inline const Matrix3&
ZeroMatrix3()
{
   return Matrix3::Zero;
}


inline Matrix3
ScaleMatrix3(const Vector3D& v)
{
   return Matrix3(v.x,0.0f,0.0f, 0.0f,v.y,0.0f, 0.0f,0.0f,v.z);
}


inline Matrix3
ScaleMatrix3(const float s)
{
   return ScaleMatrix3(Vector3D(s,s,s));
}


inline Matrix3
PitchMatrix3(const float c, const float s)
{
   return Matrix3(1.0f,0.0f,0.0f, 0.0f,c,-s, 0.0f,s,c);
}


inline Matrix3
PitchMatrix3(const float theta)
{
   return PitchMatrix3((float) cos(theta), (float) sin(theta));
}


inline Matrix3
YawMatrix3(const float c, const float s)
{
   return Matrix3(c,0.0f,s, 0.0f,1.0f,0.0f, -s,0.0f,c);
}


inline Matrix3
YawMatrix3(const float theta)
{
   return YawMatrix3((float) cos(theta), (float) sin(theta));
}


inline Matrix3
RollMatrix3(const float c, const float s)
{
   return Matrix3(c,-s,0.0f, s,c,0.0f, 0.0f,0.0f,1.0f);
}


inline Matrix3
RollMatrix3(const float theta)
{
   return RollMatrix3((float) cos(theta), (float) sin(theta));
}


inline const Matrix3D&
IdentityMatrix3D()
{
   return Matrix3D::Identity;
}


inline const Matrix3D&
ZeroMatrix3D()
{
   return Matrix3D::Zero;
}


inline Matrix3D
TranslateMatrix3D(const Vector3D& v)
{
  return Matrix3D(0.0f,0.0f,0.0f,0.0f,
                  0.0f,0.0f,0.0f,0.0f,
                  0.0f,0.0f,0.0f,0.0f,
                   v.x, v.y, v.z,0.0f);
}


inline Matrix3D
ScaleMatrix3D(const Vector3D& v)
{
   return Matrix3D( v.x,0.0f,0.0f,0.0f,
                   0.0f, v.y,0.0f,0.0f,
                   0.0f,0.0f, v.z,0.0f,
                   0.0f,0.0f,0.0f,1.0f);
}


inline Matrix3D
ScaleMatrix3D(const float s)
{
   return ScaleMatrix3D(Vector3D(s,s,s));
}


inline Matrix3D
PitchMatrix3D(const float c, const float s)
{
   return Matrix3D(1.0f, 0.0f, 0.0f, 0.0f,
                   0.0f,    c,   -s, 0.0f,
                   0.0f,    s,    c, 0.0f,
                   0.0f, 0.0f, 0.0f, 1.0f);
}


inline Matrix3D
PitchMatrix3D(const float theta)
{
   return PitchMatrix3D((float) cos(theta), (float) sin(theta));
}


inline Matrix3D
YawMatrix3D(const float c, const float s)
{
   return Matrix3D(   c, 0.0f,    s, 0.0f,
                   0.0f, 1.0f, 0.0f, 0.0f,
                     -s, 0.0f,    c, 0.0f,
                   0.0f, 0.0f, 0.0f, 1.0f);
}


inline Matrix3D
YawMatrix3D(const float theta)
{
   return YawMatrix3D((float) cos(theta), (float) sin(theta));
}


inline Matrix3D
RollMatrix3D(const float c, const float s)
{
   return Matrix3D(c,   -s,    0.0f, 0.0f,
                   s,    c,    0.0f, 0.0f,
                   0.0f, 0.0f, 1.0f, 0.0f,
                   0.0f, 0.0f, 0.0f, 1.0f);
}


inline Matrix3D
RollMatrix3D(const float theta)
{
   return RollMatrix3D((float) cos(theta), (float) sin(theta));
}


inline Matrix3D
DirectionRotateMatrix3D(const Vector3D& Dir)
{
   return Matrix3D(DirectionRotateMatrix3(Dir));
}


inline void
Matrix3D::setPosition(const Vector3D& v)
{
   _41=v.x; _42=v.y; _43=v.z;
}


inline const Vector3D&
Matrix3D::getPosition() const
{
   return (const Vector3D&)((Matrix3D&)(*this)).baseT();
}


inline Vector3D
Matrix3D::getDirection(const float Size) const
{
   return ((Matrix3D&)(*this)).baseZ() * Size;
}


inline Vector3D
Matrix3D::getDirectionUp(const float Size) const
{
   return ((Matrix3D&)(*this)).baseY() * Size;
}


inline Vector3D
Matrix3D::getDirectionRight(const float Size) const
{
   return ((Matrix3D&)(*this)).baseX() * Size;
}


inline void
Matrix3D::forward(const float d)
{
   relocate(getDirection(d));
}


inline void
Matrix3D::relocate(const Vector3D& v)
{
  _41+=v.x; _42+=v.y; _43+=v.z;
}


inline void
Matrix3D::rotate(const Vector3D& v)
{
   if (v.x!=0.0f) *this = PitchMatrix3D(v.x) * (*this);
   if (v.y!=0.0f) *this = YawMatrix3D  (v.y) * (*this);
   if (v.z!=0.0f) *this = RollMatrix3D (v.z) * (*this);
}


inline void
Matrix3D::scale(const float s)
{
  *this *= ScaleMatrix3D(s);
}


inline void
Matrix3D::align(const Matrix3D& m)
{
  *this=m;
}


inline const Matrix3D&
Transformable::getTransform() const
{
   return m_Transform;
}


inline void
Transformable::setPosition(const Vector3D& v)
{
   m_Transform.setPosition(v);
}


inline Vector3D
Transformable::getPosition() const
{
   return m_Transform.getPosition();
}


inline void
Transformable::setDirection(const Vector3D& dir, const Vector3D& up)
{
   m_Transform.setDirection(dir, up);
}


inline Vector3D
Transformable::getDirection(const float Size) const
{
   return m_Transform.getDirection(Size);
}


inline Vector3D
Transformable::getDirectionUp(const float Size) const
{
   return m_Transform.getDirectionUp(Size);
}


inline Vector3D
Transformable::getDirectionRight(const float Size) const
{
   return m_Transform.getDirectionRight(Size);
}


inline void
Transformable::forward(const float d)
{
   m_Transform.relocate(m_Transform.getDirection(d));
}


inline void
Transformable::relocate(const Vector3D& v)
{
  m_Transform.relocate(v);
}


inline void
Transformable::rotate(const Vector3D& v)
{
   m_Transform.rotate(v);
}


inline void
Transformable::scale(const float s)
{
  m_Transform.scale(s);
  m_Scale *=s;
}

inline float
Transformable::getScale() const
{
   return m_Scale;
}

inline void
Transformable::align(const Transformable* t)
{
  m_Transform = t->m_Transform;
}


inline Transformable&
Transformable::operator*=(const Transformable& t)
{
   return *this = *this * t;
}

inline Transformable
operator*(const Transformable& t1, const Transformable& t2)
{
   return Transformable(t1.m_Transform * t2.m_Transform, t1.m_Scale * t2.m_Scale);
}



#endif // H_CON_MATH3D
