// Prop.cpp : implements the Properties tabbed dialog box
//

// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <MultiMon.h>
#include <math.h>
#include <float.h>

#include <mbstring.h>

#include "HexEdit.h"
#include "MainFrm.h"
#include "HexEditDoc.h"
#include "HexEditView.h"
#include "Misc.h"
#include "resource.hm"          // For control help IDs

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CPropSheet

IMPLEMENT_DYNAMIC(CPropSheet, CPropertySheet)

CPropSheet::CPropSheet(UINT iSelectPage /*=0*/, CHexEditView *pv /*=NULL*/)
        :CPropertySheet(_T("Properties"), NULL, iSelectPage)
{
    AddPage(&prop_file);
    AddPage(&prop_char);
    AddPage(&prop_dec);
    AddPage(&prop_float);
    AddPage(&prop_ibmfp);

    Create(AfxGetMainWnd());

    // Move window to previous position (but not completely off screen)
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (aa->prop_y_ != -30000)
    {
        CRect rr;               // Rectangle where we will put the dialog
        GetWindowRect(&rr);

        // Move to where it was when it was last closed
        rr.OffsetRect(aa->prop_x_ - rr.left, aa->prop_y_ - rr.top);

        CRect scr_rect;         // Rectangle that we want to make sure the window is within

        // Get the rectangle that contains the screen work area (excluding system bars etc)
        if (aa->mult_monitor_)
        {
            HMONITOR hh = MonitorFromRect(&rr, MONITOR_DEFAULTTONEAREST);
            MONITORINFO mi;
            mi.cbSize = sizeof(mi);
            if (hh != 0 && GetMonitorInfo(hh, &mi))
                scr_rect = mi.rcWork;  // work area of nearest monitor
            else
            {
                // Shouldn't happen but if it does use the whole virtual screen
                ASSERT(0);
                scr_rect = CRect(::GetSystemMetrics(SM_XVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_XVIRTUALSCREEN) + ::GetSystemMetrics(SM_CXVIRTUALSCREEN),
                    ::GetSystemMetrics(SM_YVIRTUALSCREEN) + ::GetSystemMetrics(SM_CYVIRTUALSCREEN));
            }
        }
        else if (!::SystemParametersInfo(SPI_GETWORKAREA, 0, &scr_rect, 0))
        {
            // I don't know if this will ever happen since the Windows documentation
            // is pathetic and does not say when or why SystemParametersInfo might fail.
            scr_rect = CRect(0, 0, ::GetSystemMetrics(SM_CXFULLSCREEN),
                                   ::GetSystemMetrics(SM_CYFULLSCREEN));
        }

        if (rr.left > scr_rect.right - 20)              // off right edge?
            rr.OffsetRect(scr_rect.right - (rr.left+rr.right)/2, 0);
        if (rr.right < scr_rect.left + 20)              // off left edge?
            rr.OffsetRect(scr_rect.left - (rr.left+rr.right)/2, 0);
        if (rr.top > scr_rect.bottom - 20)              // off bottom?
            rr.OffsetRect(0, scr_rect.bottom - (rr.top+rr.bottom)/2);
        // This is not analogous to the above since we don't the window off
        // the top at all, otherwise you can get to the drag bar to move it.
        if (rr.top < scr_rect.top)                      // off top at all?
            rr.OffsetRect(0, scr_rect.top - rr.top);

        MoveWindow(&rr);
    }
}

CPropSheet::~CPropSheet()
{
}

BOOL CPropSheet::OnNcCreate(LPCREATESTRUCT lpCreateStruct) 
{
        if (!CPropertySheet::OnNcCreate(lpCreateStruct))
                return FALSE;
        
        ModifyStyleEx(0, WS_EX_CONTEXTHELP);
        
        return TRUE;
}

void CPropSheet::Update(CHexEditView *pv /*=NULL*/, long address /*=-1*/)
{
    // Get currently active property page
    CPropUpdatePage *pp = dynamic_cast<CPropUpdatePage *>(GetActivePage());

    // Set pages members from view and update in displayed page
    pp->Update(pv, address);
}

BEGIN_MESSAGE_MAP(CPropSheet, CPropertySheet)
        //{{AFX_MSG_MAP(CPropSheet)
        ON_WM_DESTROY()
        ON_WM_NCCREATE()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropSheet message handlers

void CPropSheet::OnDestroy() 
{
    CPropertySheet::OnDestroy();
        
    // Save current page to restore when reopened
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CRect rr;
    GetWindowRect(&rr);
    aa->prop_x_ = rr.left;
    aa->prop_y_ = rr.top;
    aa->prop_page_ = GetActiveIndex();
}

void CPropSheet::PostNcDestroy() 
{
    CPropertySheet::PostNcDestroy();

    // Signal that there is no longer a Properties property sheet
    dynamic_cast<CMainFrame *>(AfxGetMainWnd())->pprop_ = NULL;
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_close);

    // Delete the class instance since this could be caused by the user hitting
    // the close button and this is the simplest way to avoid memory leaks.
    // Note: this assumes that CPropSheet always created on heap (ie. with new).
    ASSERT(m_hWnd == NULL);
    delete this;
}

//===========================================================================
/////////////////////////////////////////////////////////////////////////////
// CPropUpdatePage stuff (most is inline in Prop.h)
IMPLEMENT_DYNAMIC(CPropUpdatePage, CPropertyPage)

//===========================================================================
/////////////////////////////////////////////////////////////////////////////
// CPropFilePage property page

IMPLEMENT_DYNCREATE(CPropFilePage, CPropUpdatePage)

CPropFilePage::CPropFilePage() : CPropUpdatePage(CPropFilePage::IDD)
{
        //{{AFX_DATA_INIT(CPropFilePage)
        file_name_ = _T("");
        file_path_ = _T("");
        file_size_ = _T("");
        file_hidden_ = FALSE;
        file_readonly_ = FALSE;
        file_system_ = FALSE;
        file_type_ = _T("");
        file_modified_ = _T("");
        //}}AFX_DATA_INIT
}

CPropFilePage::~CPropFilePage()
{
}

void CPropFilePage::DoDataExchange(CDataExchange* pDX)
{
        CPropUpdatePage::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CPropFilePage)
        DDX_Text(pDX, IDC_FILE_NAME, file_name_);
        DDX_Text(pDX, IDC_FILE_PATH, file_path_);
        DDX_Text(pDX, IDC_FILE_SIZE, file_size_);
        DDX_Check(pDX, IDC_FILE_HIDDEN, file_hidden_);
        DDX_Check(pDX, IDC_FILE_READONLY, file_readonly_);
        DDX_Check(pDX, IDC_FILE_SYSTEM, file_system_);
        DDX_Text(pDX, IDC_FILE_TYPE, file_type_);
        DDX_Text(pDX, IDC_FILE_MODIFIED, file_modified_);
        //}}AFX_DATA_MAP
}

BEGIN_MESSAGE_MAP(CPropFilePage, CPropUpdatePage)
        //{{AFX_MSG_MAP(CPropFilePage)
        ON_WM_HELPINFO()
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropFilePage message handlers


BOOL CPropFilePage::OnSetActive() 
{
    Update(GetView());
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_file);
    return CPropUpdatePage::OnSetActive();
}

BOOL CPropFilePage::OnHelpInfo(HELPINFO* pHelpInfo) 
{
//      return CPropUpdatePage::OnHelpInfo(pHelpInfo);
    static DWORD id_pairs[] = { 
        IDC_FILE_NAME, HIDC_FILE_NAME,
        IDC_FILE_TYPE, HIDC_FILE_TYPE,
        IDC_FILE_PATH, HIDC_FILE_PATH,
        IDC_FILE_MODIFIED, HIDC_FILE_MODIFIED,
        IDC_FILE_SIZE, HIDC_FILE_SIZE,
        IDC_FILE_READONLY, HIDC_FILE_READONLY,
        IDC_FILE_HIDDEN, HIDC_FILE_HIDDEN,
        IDC_FILE_SYSTEM, HIDC_FILE_SYSTEM,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

void CPropFilePage::Update(CHexEditView *pv, long /*not used*/)
{
    // Clear fields
    file_name_ = "";
    file_path_ = "";
    file_type_ = "";
    file_size_ = "";
    file_modified_ = "";
    file_readonly_ = 0;
    file_hidden_ = 0;
    file_system_ = 0;

    if (pv == NULL)
    {
        UpdateData(FALSE);
        return;
    }

    file_size_ = NumScale(dynamic_cast<CHexEditDoc *>(pv->GetDocument())->length()) + "bytes";

    CFile *pf = &dynamic_cast<CHexEditDoc *>(pv->GetDocument())->file_;
    if (pf->m_hFile == CFile::hFileNull)
        return;

    CFileStatus status;
    size_t path_len;            // Length of path (excluding file name)

    file_name_ = pf->GetFileName();
    file_path_ = pf->GetFilePath();
    if ( (path_len = file_path_.ReverseFind('\\')) != -1 ||
         (path_len = file_path_.ReverseFind(':')) != -1 )
        file_path_ = file_path_.Left(path_len+1);
    else
        file_path_ = "";

    file_type_ = get_type();

    pf->GetStatus(status);
    file_modified_ = status.m_mtime.Format(/*"%d %B %y, %H:%M:%S"*/"%c");
    file_readonly_ = (status.m_attribute & CFile::readOnly) != 0;
    file_hidden_ = (status.m_attribute & CFile::hidden) != 0;
    file_system_ = (status.m_attribute & CFile::system) != 0;

    // Enable the check boxes so they can be set appropriately
    CWnd *pwnd = GetDlgItem(IDC_FILE_READONLY);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(TRUE);
    pwnd = GetDlgItem(IDC_FILE_HIDDEN);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(TRUE);
    pwnd = GetDlgItem(IDC_FILE_SYSTEM);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(TRUE);

    UpdateData(FALSE);

    // Disable the check boxes again so the user doesn't think he can change them
    pwnd = GetDlgItem(IDC_FILE_READONLY);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(FALSE);
    pwnd = GetDlgItem(IDC_FILE_HIDDEN);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(FALSE);
    pwnd = GetDlgItem(IDC_FILE_SYSTEM);
    ASSERT(pwnd != NULL);
    pwnd->EnableWindow(FALSE);

    // The call to UpdateWindow() is necessary during macro replay since (even
    // if property refresh is on) no changes are seen until the macro stops.
    UpdateWindow();
}

const char *CPropFilePage::get_type()
{
    int ext_pos = file_name_.ReverseFind('.');
    if (ext_pos == -1) return "None";

    CString file_ext = file_name_.Mid(ext_pos);
    long buf_len;

    char type_name[128];
    buf_len = sizeof(type_name);
    if (::RegQueryValue(HKEY_CLASSES_ROOT, file_ext, type_name, &buf_len) != ERROR_SUCCESS)
        return "Unknown";

    static char buf[128];
    buf_len = sizeof(buf);
    if (::RegQueryValue(HKEY_CLASSES_ROOT, type_name, buf, &buf_len) != ERROR_SUCCESS)
        return "Registry Error";

    return buf;
}
//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CPropCharPage property page

IMPLEMENT_DYNCREATE(CPropCharPage, CPropUpdatePage)

CPropCharPage::CPropCharPage() : CPropUpdatePage(CPropCharPage::IDD)
{
        //{{AFX_DATA_INIT(CPropCharPage)
        char_ascii_ = _T("");
        char_binary_ = _T("");
        char_ebcdic_ = _T("");
        char_hex_ = _T("");
        char_octal_ = _T("");
        char_dec_ = _T("");
        //}}AFX_DATA_INIT
        char_unicode_[0] = L'\0';
}

CPropCharPage::~CPropCharPage()
{
}

void CPropCharPage::DoDataExchange(CDataExchange* pDX)
{
        CPropUpdatePage::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CPropCharPage)
        DDX_Text(pDX, IDC_CHAR_ASCII, char_ascii_);
        DDV_MaxChars(pDX, char_ascii_, 5);
        DDX_Text(pDX, IDC_CHAR_BINARY, char_binary_);
        DDV_MaxChars(pDX, char_binary_, 8);
        DDX_Text(pDX, IDC_CHAR_EBCDIC, char_ebcdic_);
        DDV_MaxChars(pDX, char_ebcdic_, 5);
        DDX_Text(pDX, IDC_CHAR_HEX, char_hex_);
        DDV_MaxChars(pDX, char_hex_, 2);
        DDX_Text(pDX, IDC_CHAR_OCTAL, char_octal_);
        DDV_MaxChars(pDX, char_octal_, 3);
        DDX_Text(pDX, IDC_CHAR_DEC, char_dec_);
        DDV_MaxChars(pDX, char_dec_, 3);
        //}}AFX_DATA_MAP

        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        if (!pDX->m_bSaveAndValidate && aa->is_nt_)
        {
                CWnd *pedit = GetDlgItem(IDC_CHAR_UNICODE);
                ASSERT(pedit != NULL);
                ::SetWindowTextW(pedit->m_hWnd, char_unicode_);
        }
}


BEGIN_MESSAGE_MAP(CPropCharPage, CPropUpdatePage)
        //{{AFX_MSG_MAP(CPropCharPage)
        ON_WM_HELPINFO()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropCharPage message handlers

BOOL CPropCharPage::OnInitDialog() 
{
	CPropUpdatePage::OnInitDialog();

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->is_nt_)
    {
        CWnd *ptext = GetDlgItem(IDC_UNICODE_DESC);
        ASSERT(ptext != NULL);
        ptext->EnableWindow(FALSE);
        CWnd *pedit = GetDlgItem(IDC_CHAR_UNICODE);
        ASSERT(pedit != NULL);
        pedit->EnableWindow(FALSE);
    }
	else
	{
		static LOGFONTW lf;
		static HFONT hf = HFONT(0);
		if (hf == HFONT(0))
		{
//			::GetObjectW(::GetStockObject(SYSTEM_FONT), sizeof(lf), &lf);
		    memset((void *)&lf, '\0', sizeof(lf));
//			wcscpy(lf.lfFaceName, L"Arial");
	        wcscpy(lf.lfFaceName, L"Lucida Sans Unicode");
			lf.lfHeight = 12;
			lf.lfWeight = 0;
			lf.lfCharSet = DEFAULT_CHARSET;

			hf = ::CreateFontIndirectW(&lf);
		}
        HWND hw = ::GetDlgItem(m_hWnd, IDC_CHAR_UNICODE);
        ASSERT(hw != HWND(0));
		::SendMessageW(hw, WM_SETFONT, WPARAM(hf), 0L);
	}

	return TRUE;
}

BOOL CPropCharPage::OnSetActive() 
{
    Update(GetView());
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_char);
    return CPropUpdatePage::OnSetActive();
}

BOOL CPropCharPage::OnHelpInfo(HELPINFO *pHelpInfo)
{
    static DWORD id_pairs[] = { 
        IDC_CHAR_HEX, HIDC_CHAR_HEX,
        IDC_CHAR_DEC, HIDC_CHAR_DEC,
        IDC_CHAR_OCTAL, HIDC_CHAR_OCTAL,
        IDC_CHAR_BINARY, HIDC_CHAR_BINARY,
        IDC_CHAR_ASCII, HIDC_CHAR_ASCII,
        IDC_CHAR_UNICODE, HIDC_CHAR_UNICODE,
        IDC_CHAR_EBCDIC, HIDC_CHAR_EBCDIC,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

void CPropCharPage::Update(CHexEditView *pv, long address /*=-1*/)
{
    static char *ascii_control_char[] =
    {
        "NUL", "SOH", "STX", "ETX", "EOT", "ENQ", "ACK", "BEL",
        "BS",  "HT",  "LF",  "VT",  "FF",  "CR",  "SO",  "SI",
        "DLE", "DC1", "DC2", "DC3", "DC4", "NAK", "SYN", "ETB",
        "CAN", "EM",  "SUB", "ESC", "FS",  "GS",  "RS",  "US",
        "space", "**ERROR"
    };

    static char *ebcdic_control_char[] =
    {
        "NUL", "SOH", "STX", "ETX", "PF",  "HT",  "LC",  "DEL",
        "GE",  "RLF", "SMM", "VT",  "FF",  "CR",  "SO",  "SI",
        "DLE", "DC1", "DC2", "TM",  "RES", "NL",  "BS",  "IL",
        "CAN", "EM",  "CC",  "CU1", "IFS", "IGS", "IRS", "IUS",
        "DS",  "SOS", "FS",  "none","BYP", "LF",  "ETB", "ESC",
        "none","none","SM",  "CU2", "none","ENQ", "ACK", "BEL",
        "none","none","SYN", "none","PN",  "RS",  "UC",  "EOT",
        "none","none","none", "CU3", "DC4", "NAK", "none", "SUB",
        "space","**ERROR"
    };

    // Set fields to empty in case of no view, caret at EOF etc
    char_ascii_ = _T("");
    char_binary_ = _T("");
    char_ebcdic_ = _T("");
    char_hex_ = _T("");
    char_octal_ = _T("");
    char_dec_ = _T("");
    char_unicode_[0] = L'\0';

    // Get active view (if not given) and current address (if not given)
    if (pv == NULL)
    {
        UpdateData(FALSE);
        return;
    }
    if (address == -1)
        address = pv->GetPos();

    unsigned char cc[2];
    size_t got = dynamic_cast<CHexEditDoc *>(pv->GetDocument())->GetData(cc, 2, address);

    if (got > 0)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // Work out binary, octal, decimal, hex values
        for (int ii = 0; ii < 8; ++ii)
            char_binary_ += (cc[0] & (0x80>>ii)) ? '1' : '0';
        char_octal_.Format("%03o", cc[0]);
        char_dec_.Format("%d", cc[0]);
        if (aa->hex_ucase_)
            char_hex_.Format("%02X", cc[0]);
        else
            char_hex_.Format("%02x", cc[0]);

        // Work out display values (ASCII, EBCDIC, MBCS)
        if (cc[0] <= 0x20)      // Control char or space?
            char_ascii_ = ascii_control_char[cc[0]];
        else
            char_ascii_= cc[0];
        if (cc[0] <= 0x40)
            char_ebcdic_ = ebcdic_control_char[cc[0]];
        else if (e2a_tab[cc[0]] != '\0')
            char_ebcdic_ = e2a_tab[cc[0]];
        else
            char_ebcdic_ = "none";
    }
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (got > 1 && aa->is_nt_)
    {
        // Create unicode string if we got 2 bytes and we're running NT
        char_unicode_[0] = cc[0] + 256*cc[1];
        char_unicode_[1] = L'\0';
        if (char_unicode_[0] <= 0x20)
            VERIFY(MultiByteToWideChar(CP_ACP, MB_ERR_INVALID_CHARS,
                ascii_control_char[cc[0]], -1,
                char_unicode_, sizeof(char_unicode_)/sizeof(char_unicode_[0])));
        else if (char_unicode_[0] == 0x2028)
            VERIFY(MultiByteToWideChar(CP_ACP, MB_ERR_INVALID_CHARS,
                "newline", -1,
                char_unicode_, sizeof(char_unicode_)/sizeof(char_unicode_[0])));
        else if (char_unicode_[0] == 0x2029)
            VERIFY(MultiByteToWideChar(CP_ACP, MB_ERR_INVALID_CHARS,
                "para", -1,
                char_unicode_, sizeof(char_unicode_)/sizeof(char_unicode_[0])));
        else if (char_unicode_[0] == 0xFFFE)
            VERIFY(MultiByteToWideChar(CP_ACP, MB_ERR_INVALID_CHARS,
                "order", -1,
                char_unicode_, sizeof(char_unicode_)/sizeof(char_unicode_[0])));
        else if (char_unicode_[0] == 0xFFFF)
            VERIFY(MultiByteToWideChar(CP_ACP, MB_ERR_INVALID_CHARS,
                "spec.", -1,
                char_unicode_, sizeof(char_unicode_)/sizeof(char_unicode_[0])));
    }
    UpdateData(FALSE);

    // The call to UpdateWindow() is necessary during macro replay since (even
    // if property refresh is on) no changes are seen until the macro stops.
    UpdateWindow();
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CPropDecPage property page

IMPLEMENT_DYNCREATE(CPropDecPage, CPropUpdatePage)

CPropDecPage::CPropDecPage() : CPropUpdatePage(CPropDecPage::IDD)
{
        //{{AFX_DATA_INIT(CPropDecPage)
        dec_64bit_ = _T("");
        dec_16bit_ = _T("");
        dec_32bit_ = _T("");
        dec_8bit_ = _T("");
        signed_ = -1;
        big_endian_ = FALSE;
        //}}AFX_DATA_INIT
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        big_endian_ = aa->prop_dec_endian_;
        signed_ = aa->prop_dec_signed_;
}

CPropDecPage::~CPropDecPage()
{
    // Save current state for next invocation of property dialog or save settings
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    aa->prop_dec_signed_ = signed_;
    aa->prop_dec_endian_ = big_endian_;
}

void CPropDecPage::DoDataExchange(CDataExchange* pDX)
{
        CPropUpdatePage::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CPropDecPage)
        DDX_Text(pDX, IDC_DEC_64BIT, dec_64bit_);
        DDV_MaxChars(pDX, dec_64bit_, 28);
        DDX_Text(pDX, IDC_DEC_16BIT, dec_16bit_);
        DDV_MaxChars(pDX, dec_16bit_, 7);
        DDX_Text(pDX, IDC_DEC_32BIT, dec_32bit_);
        DDV_MaxChars(pDX, dec_32bit_, 14);
        DDX_Text(pDX, IDC_DEC_8BIT, dec_8bit_);
        DDV_MaxChars(pDX, dec_8bit_, 4);
        DDX_Radio(pDX, IDC_DEC_UNSIGNED, signed_);
        DDX_Check(pDX, IDC_BIG_ENDIAN, big_endian_);
        //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CPropDecPage, CPropUpdatePage)
        //{{AFX_MSG_MAP(CPropDecPage)
        ON_WM_HELPINFO()
        ON_BN_CLICKED(IDC_DEC_UNSIGNED, OnChangeFormat)
        ON_BN_CLICKED(IDC_BIG_ENDIAN, OnChangeFormat)
        ON_BN_CLICKED(IDC_DEC_2COMP, OnChangeFormat)
        ON_BN_CLICKED(IDC_DEC_1COMP, OnChangeFormat)
        ON_BN_CLICKED(IDC_DEC_SIGN_MAG, OnChangeFormat)
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropDecPage message handlers


void CPropDecPage::OnChangeFormat() 
{
    if (UpdateData(TRUE))       // Update big_endian_, signed_ from buttons
        Update(GetView());      // Recalc values based on new format
}

BOOL CPropDecPage::OnSetActive()
{
    Update(GetView());
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_dec);
    return CPropUpdatePage::OnSetActive();
}

BOOL CPropDecPage::OnHelpInfo(HELPINFO *pHelpInfo) 
{
//      return CPropUpdatePage::OnHelpInfo(pHelpInfo);
    static DWORD id_pairs[] = { 
        IDC_DEC_8BIT, HIDC_DEC_8BIT,
        IDC_DEC_16BIT, HIDC_DEC_16BIT,
        IDC_DEC_32BIT, HIDC_DEC_32BIT,
        IDC_DEC_64BIT, HIDC_DEC_64BIT,
//      IDC_DEC_LITTLE_ENDIAN, HIDC_DEC_LITTLE_ENDIAN,
//      IDC_DEC_BIG_ENDIAN, HIDC_DEC_BIG_ENDIAN,
        IDC_BIG_ENDIAN, HIDC_BIG_ENDIAN,
        IDC_DEC_UNSIGNED, HIDC_DEC_UNSIGNED,
//      IDC_DEC_SIGNED, HIDC_DEC_SIGNED,
        IDC_DEC_2COMP, HIDC_DEC_2COMP,
        IDC_DEC_1COMP, HIDC_DEC_1COMP,
        IDC_DEC_SIGN_MAG, HIDC_DEC_SIGN_MAG,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

void CPropDecPage::Update(CHexEditView *pv, long address /*=-1*/)
{
    // Set fields to empty in case no file, caret at EOF etc
    dec_8bit_ = _T("");
    dec_16bit_ = _T("");
    dec_32bit_ = _T("");
    dec_64bit_ = _T("");

    // Get active view (if not given) and current address (if not given)
    if (pv == NULL)
    {
        UpdateData(FALSE);
        return;
    }
    if (address == -1)
        address = pv->GetPos();

    unsigned char buf[8];
    size_t got = dynamic_cast<CHexEditDoc *>(pv->GetDocument())->GetData(buf, sizeof(buf), address);

    if (got >= 1)
    {
        switch (signed_)
        {
        case 0:
            dec_8bit_.Format("%u", buf[0]);
            break;
        case 1:
            dec_8bit_.Format("%d", signed char(buf[0]));
            break;
        case 2:
            if ((buf[0]&0x80) != 0)
                dec_8bit_.Format("-%u", ~buf[0]&0x7F);
            else
                dec_8bit_.Format("%u", buf[0]);
            break;
        case 3:
            if ((buf[0]&0x80) != 0)
                dec_8bit_.Format("-%u", buf[0]&0x7F);
            else
                dec_8bit_.Format("%u", buf[0]);
            break;
        default:
            ASSERT(0);
        }
    }
    if (got >= 2)
    {
        unsigned char *pp, tt[2];
        if (!big_endian_)
        {
            tt[0] = buf[1];
            tt[1] = buf[0];
            pp = tt;
        }
        else
            pp = buf;

        switch (signed_)
        {
        case 0:
            dec_16bit_.Format("%u", (pp[0]<<8) + pp[1]);
            break;
        case 1:
            dec_16bit_.Format("%d", short((pp[0]<<8) + pp[1]));
            break;
        case 2:
            if ((pp[0]&0x80) != 0)
                dec_16bit_.Format("-%u", ~((pp[0]<<8) + pp[1])&0x7FFF);
            else
                dec_16bit_.Format("%u", (pp[0]<<8) + pp[1]);
            break;
        case 3:
            if ((pp[0]&0x80) != 0)
                dec_16bit_.Format("-%u", ((pp[0]<<8) + pp[1])&0x7FFF);
            else
                dec_16bit_.Format("%u", (pp[0]<<8) + pp[1]);
            break;
        default:
            ASSERT(0);
        }
        AddCommas(dec_16bit_);
    }
    if (got >= 4)
    {
        unsigned char *pp, tt[4];
        if (!big_endian_)
        {
            tt[0] = buf[3];
            tt[1] = buf[2];
            tt[2] = buf[1];
            tt[3] = buf[0];
            pp = tt;
        }
        else
            pp = buf;

        switch (signed_)
        {
        case 0:
            dec_32bit_.Format("%lu",   pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0])<<24));
            break;
        case 1:
            dec_32bit_.Format("%ld",   pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0])<<24));
            break;
        case 2:
            if ((pp[0]&0x80) != 0)
                dec_32bit_.Format("-%lu",   ~(pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0])<<24))&0x7FFFFFFFUL);
            else
                dec_32bit_.Format("%lu",   pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0])<<24));
            break;
        case 3:
            if ((pp[0]&0x80) != 0)
                dec_32bit_.Format("-%lu",   pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0]&0x7F)<<24));
            else
                dec_32bit_.Format("%lu",   pp[3] + (long(pp[2])<<8) + 
                                (long(pp[1])<<16) + (long(pp[0])<<24));
            break;
        default:
            ASSERT(0);
        }
        AddCommas(dec_32bit_);
    }
    if (got >= 8)
    {
        unsigned char *pp, tt[8];
        if (!big_endian_)
        {
            tt[0] = buf[7];
            tt[1] = buf[6];
            tt[2] = buf[5];
            tt[3] = buf[4];
            tt[4] = buf[3];
            tt[5] = buf[2];
            tt[6] = buf[1];
            tt[7] = buf[0];
            pp = tt;
        }
        else
            pp = buf;

        // Note: %I64d does not seem to work with CString::Format() so use sprintf
        char aa[22];                    // temp buf where we sprintf

        switch (signed_)
        {
        case 0:
            sprintf(aa, "%I64u",  pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0])<<56));
            break;
        case 1:
            sprintf(aa, "%I64d",  pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0])<<56));
            break;
        case 2:
            if ((pp[0]&0x80) != 0)
                sprintf(aa, "-%I64u",  ~(pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0])<<56))&0x7FFFFFFFFFFFFFFF);
            else
                sprintf(aa, "%I64u",  pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0])<<56));
            break;
        case 3:
            if ((pp[0]&0x80) != 0)
                sprintf(aa, "-%I64u",  pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0]&0x7F)<<56));
            else
                sprintf(aa, "%I64u",  pp[7] + (__int64(pp[6])<<8) +
                             (__int64(pp[5])<<16) + (__int64(pp[4])<<24) + 
                             (__int64(pp[3])<<32) + (__int64(pp[2])<<40) + 
                             (__int64(pp[1])<<48) + (__int64(pp[0])<<56));
            break;
        default:
            ASSERT(0);
        }

        dec_64bit_ = aa;
        AddCommas(dec_64bit_);
    }
    UpdateData(FALSE);

    // The call to UpdateWindow() is necessary during macro replay since (even
    // if property refresh is on) no changes are seen until the macro stops.
    UpdateWindow();
}

//===========================================================================

/////////////////////////////////////////////////////////////////////////////
// CPropFloatPage property page

IMPLEMENT_DYNCREATE(CPropFloatPage, CPropUpdatePage)

CPropFloatPage::CPropFloatPage() : CPropUpdatePage(CPropFloatPage::IDD)
{
        //{{AFX_DATA_INIT(CPropFloatPage)
        exp_ = _T("");
        mant_ = _T("");
        val_ = _T("");
        format_ = -1;
        big_endian_ = FALSE;
        //}}AFX_DATA_INIT
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        big_endian_ = aa->prop_fp_endian_;
        format_ = aa->prop_fp_format_;
    // Use _fpclass CString::Format to get value, frexp to get mantissa/exp
}

CPropFloatPage::~CPropFloatPage()
{
    // Save current state for next invocation of property dialog or save settings
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    aa->prop_fp_format_ = format_;
    aa->prop_fp_endian_ = big_endian_;
}

void CPropFloatPage::DoDataExchange(CDataExchange* pDX)
{
        CPropUpdatePage::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CPropFloatPage)
        DDX_Text(pDX, IDC_FP_EXP, exp_);
        DDV_MaxChars(pDX, exp_, 5);
        DDX_Text(pDX, IDC_FP_MANT, mant_);
        DDV_MaxChars(pDX, mant_, 20);
        DDX_Text(pDX, IDC_FP_VAL, val_);
        DDV_MaxChars(pDX, val_, 25);
        DDX_Radio(pDX, IDC_FP_32BIT, format_);
        DDX_Check(pDX, IDC_BIG_ENDIAN, big_endian_);
        //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CPropFloatPage, CPropUpdatePage)
        //{{AFX_MSG_MAP(CPropFloatPage)
        ON_WM_HELPINFO()
        ON_BN_CLICKED(IDC_FP_32BIT, OnChangeFormat)
        ON_BN_CLICKED(IDC_FP_64BIT, OnChangeFormat)
        ON_BN_CLICKED(IDC_BIG_ENDIAN, OnChangeFormat)
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropFloatPage message handlers

void CPropFloatPage::OnChangeFormat() 
{
    if (UpdateData(TRUE))       // Update big_endian_, (32/64 bit) format_ from buttons
        Update(GetView());      // Recalc values based on new format
}

BOOL CPropFloatPage::OnSetActive()
{
    Update(GetView());
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_float);
    return CPropUpdatePage::OnSetActive();
}

BOOL CPropFloatPage::OnHelpInfo(HELPINFO* pHelpInfo) 
{
//      return CPropUpdatePage::OnHelpInfo(pHelpInfo);
    static DWORD id_pairs[] = { 
        IDC_FP_VAL, HIDC_FP_VAL,
        IDC_FP_MANT, HIDC_FP_MANT,
        IDC_FP_EXP, HIDC_FP_EXP,
        IDC_BIG_ENDIAN, HIDC_BIG_ENDIAN,
        IDC_FP_32BIT, HIDC_FP_32BIT,
        IDC_FP_64BIT, HIDC_FP_64BIT,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath, HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

void CPropFloatPage::Update(CHexEditView *pv, long address /*=-1*/)
{
    // Set fields to empty in case no view, not enough bytes to EOF etc
    val_ = _T("");
    mant_ = _T("");
    exp_ = _T("");

    // Get active view (if not given) and current address (if not given)
    if (pv == NULL)
    {
        UpdateData(FALSE);
        return;
    }
    if (address == -1)
        address = pv->GetPos();

    unsigned char buf[8];
    size_t got = dynamic_cast<CHexEditDoc *>(pv->GetDocument())->GetData(buf, sizeof(buf), address);

    // If not enough byte to display FP value just display empty fields
    if (format_ == 0 && got < 4 || format_ == 1 && got < 8)
    {
        UpdateData(FALSE);
        return;
    }

    double dd;                          // The value to be displayed

    // Convert the appropriate bytes to a double
    if (format_ == 0)
    {
        float ff;
        if (big_endian_)
        {
            unsigned char cc = buf[0]; buf[0] = buf[3]; buf[3] = cc;
            cc = buf[1]; buf[1] = buf[2]; buf[2] = cc;
        }
        memcpy(&ff, buf, 4);
        dd = ff;
    }
    else if (format_ == 1)
    {
        if (big_endian_)
        {
            unsigned char cc = buf[0]; buf[0] = buf[7]; buf[7] = cc;
            cc = buf[1]; buf[1] = buf[6]; buf[6] = cc;
            cc = buf[2]; buf[2] = buf[5]; buf[5] = cc;
            cc = buf[3]; buf[3] = buf[4]; buf[4] = cc;
        }
        memcpy(&dd, buf, 8);
    }
    else
        ASSERT(0);                      // IEEE 80 bit format no yet handled

    // Work out what to display
    int exp;
    double mant = frexp(dd, &exp);
    exp_.Format("%d", exp);
    mant_.Format(format_ ? "%.16f" : "%.7f", mant);
    switch (_fpclass(dd))
    {
    case _FPCLASS_SNAN:
    case _FPCLASS_QNAN:
        val_ = "NaN";
        break;
    case _FPCLASS_NINF:
        val_ = "-Inf";
        break;
    case _FPCLASS_PINF:
        val_ = "+Inf";
        break;
    default:
        val_.Format(format_ ? "%.16g" : "%.7g", dd);
        break;
    }

    UpdateData(FALSE);

    // The call to UpdateWindow() is necessary during macro replay since (even
    // if property refresh is on) no changes are seen until the macro stops.
    UpdateWindow();
}

/////////////////////////////////////////////////////////////////////////////
// CPropIBMFloatPage property page

IMPLEMENT_DYNCREATE(CPropIBMFloatPage, CPropUpdatePage)

CPropIBMFloatPage::CPropIBMFloatPage() : CPropUpdatePage(CPropIBMFloatPage::IDD)
{
        //{{AFX_DATA_INIT(CPropIBMFloatPage)
        big_endian_ = FALSE;
        exp_ = _T("");
        mant_ = _T("");
        val_ = _T("");
        format_ = -1;
        //}}AFX_DATA_INIT
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        big_endian_ = aa->prop_ibmfp_endian_;
        format_ = aa->prop_ibmfp_format_;
}

CPropIBMFloatPage::~CPropIBMFloatPage()
{
    // Save current state for next invocation of property dialog or save settings
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    aa->prop_ibmfp_format_ = format_;
    aa->prop_ibmfp_endian_ = big_endian_;
}

void CPropIBMFloatPage::DoDataExchange(CDataExchange* pDX)
{
        CPropUpdatePage::DoDataExchange(pDX);
        //{{AFX_DATA_MAP(CPropIBMFloatPage)
        DDX_Check(pDX, IDC_BIG_ENDIAN, big_endian_);
        DDX_Text(pDX, IDC_IBMFP_EXP, exp_);
        DDV_MaxChars(pDX, exp_, 4);
        DDX_Text(pDX, IDC_IBMFP_MANT, mant_);
        DDV_MaxChars(pDX, mant_, 20);
        DDX_Text(pDX, IDC_IBMFP_VAL, val_);
        DDV_MaxChars(pDX, val_, 25);
        DDX_Radio(pDX, IDC_IBMFP_32BIT, format_);
        //}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CPropIBMFloatPage, CPropUpdatePage)
        //{{AFX_MSG_MAP(CPropIBMFloatPage)
        ON_WM_HELPINFO()
        ON_BN_CLICKED(IDC_BIG_ENDIAN, OnChangeFormat)
        ON_BN_CLICKED(IDC_IBMFP_32BIT, OnChangeFormat)
        ON_BN_CLICKED(IDC_IBMFP_64BIT, OnChangeFormat)
        //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CPropIBMFloatPage message handlers

void CPropIBMFloatPage::OnChangeFormat() 
{
    if (UpdateData(TRUE))       // Update big_endian_, (32/64 bit) format_ from buttons
        Update(GetView());      // Recalc values based on new format
}

BOOL CPropIBMFloatPage::OnSetActive() 
{
    Update(GetView());
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_prop_ibmfp);
    return CPropUpdatePage::OnSetActive();
}

BOOL CPropIBMFloatPage::OnHelpInfo(HELPINFO* pHelpInfo) 
{
    static DWORD id_pairs[] = { 
        IDC_IBMFP_VAL, HIDC_IBMFP_VAL,
        IDC_IBMFP_MANT, HIDC_IBMFP_MANT,
        IDC_IBMFP_EXP, HIDC_IBMFP_EXP,
        IDC_BIG_ENDIAN, HIDC_BIG_ENDIAN,
        IDC_IBMFP_32BIT, HIDC_IBMFP_32BIT,
        IDC_IBMFP_64BIT, HIDC_IBMFP_64BIT,
        0,0 
    }; 
 
    CWinApp* pApp = AfxGetApp();
    ASSERT_VALID(pApp);
    ASSERT(pApp->m_pszHelpFilePath != NULL);

    CWaitCursor wait;

    if (!::WinHelp((HWND)pHelpInfo->hItemHandle, pApp->m_pszHelpFilePath,
                   HELP_WM_HELP, (DWORD) (LPSTR) id_pairs))
            ::HMessageBox(AFX_IDP_FAILED_TO_LAUNCH_HELP);
    return TRUE;
}

void CPropIBMFloatPage::Update(CHexEditView *pv, long address /*=-1*/)
{
    // Set fields to empty in case no view, not enough bytes to EOF etc
    val_ = _T("");
    mant_ = _T("");
    exp_ = _T("");

    // Get active view (if not given) and current address (if not given)
    if (pv == NULL)
    {
        UpdateData(FALSE);
        return;
    }
    if (address == -1)
        address = pv->GetPos();

    unsigned char buf[8];
    size_t got = dynamic_cast<CHexEditDoc *>(pv->GetDocument())->GetData(buf, sizeof(buf), address);

    // If not enough byte to display FP value just display empty fields
    if (format_ == 0 && got < 4 || format_ == 1 && got < 8)
    {
        UpdateData(FALSE);
        return;
    }

    long double value, mantissa;                        // The value to be displayed
    int exponent;

    // Convert the bytes from IBM format to a long double (and get mantissa/exponent)
    if (format_ == 0)
        value = ibm_fp32(buf, &exponent, &mantissa, !big_endian_);
    else if (format_ == 1)
        value = ibm_fp64(buf, &exponent, &mantissa, !big_endian_);
    else
        ASSERT(0);                      // IEEE 80 bit format no yet handled

    exp_.Format("%d", exponent);
    mant_.Format(format_ ? "%.16f" : "%.7f", mantissa);
    val_.Format(format_ ? "%.17g" : "%.7g", value);

    UpdateData(FALSE);

    // The call to UpdateWindow() is necessary during macro replay since (even
    // if property refresh is on) no changes are seen until the macro stops.
    UpdateWindow();
}

long double CPropIBMFloatPage::ibm_fp32(const unsigned char *pp, int *pexp /*=NULL*/,
                long double *pmant /*=NULL*/, bool little_endian /*=false*/)
{
    unsigned char tt[4];                // Used to store bytes in little-endian order
    int exponent;                       // Base 16 exponent of IBM FP value
    long mantissa;                      // Unsigned integer mantissa

    if (little_endian)
    {
        tt[0] = pp[3];
        tt[1] = pp[2];
        tt[2] = pp[1];
        tt[3] = pp[0];
        pp = tt;
    }

    // Work out the binary exponent:
    // - remove the high (sign) bit from the first byte
    // - subtract the bias of 64
    // - convert hex exponent to binary [note 16^X == 2^(4*X)]
    exponent = ((int)(pp[0] & 0x7F) - 64);
    if (pexp != NULL) *pexp = exponent;
    // - convert hex exponent to binary
    //   note that we multiply the exponent by 4 since 16^X == 2^(4*X)
    exponent *= 4;

    // Get the mantissa and return zero if there are no bits on
    mantissa = ((long)pp[1]<<16) + ((long)pp[2]<<8) + pp[3];
    if (mantissa == 0)
    {
        if (pmant != NULL) *pmant = 0.0;
        return 0.0;
    }

    if (pmant != NULL)
    {
        if ((pp[0] & 0x80) == 0)
            *pmant = mantissa / 16777216.0L;
        else
            *pmant = -(mantissa / 16777216.0L);
    }
    
    // Check sign bit and return appropriate result
    if ((pp[0] & 0x80) == 0)
        return (mantissa / 16777216.0L) * powl(2, exponent);
    else
        return -(mantissa / 16777216.0L) * powl(2, exponent);
}

long double CPropIBMFloatPage::ibm_fp64(const unsigned char *pp, int *pexp /*=NULL*/,
                long double *pmant /*=NULL*/, bool little_endian /*=false*/)
{
    unsigned char tt[8];                // Used to store bytes in little-endian order
    int exponent;                       // Base 16 exponent of IBM FP value
    __int64 mantissa;                   // Unsigned integral mantissa
    const long double two_pow56 = 72057594037927936.0L;

    if (little_endian)
    {
        tt[0] = pp[7];
        tt[1] = pp[6];
        tt[2] = pp[5];
        tt[3] = pp[4];
        tt[4] = pp[3];
        tt[5] = pp[2];
        tt[6] = pp[1];
        tt[7] = pp[0];
        pp = tt;
    }

    // Work out the binary exponent:
    // - remove the high (sign) bit from the first byte
    // - subtract the bias of 64
    exponent = ((int)(pp[0] & 0x7F) - 64);
    if (pexp != NULL) *pexp = exponent;
    // - convert hex exponent to binary
    //   note that we multiply the exponent by 4 since 16^X == 2^(4*X)
    exponent *= 4;

    // Get the mantissa and return zero if there are no bits on
    mantissa =  ((__int64)pp[1]<<48) + ((__int64)pp[2]<<40) + ((__int64)pp[3]<<32) +
                ((__int64)pp[4]<<24) + ((__int64)pp[5]<<16) + ((__int64)pp[6]<<8) + pp[7];
    if (mantissa == 0)
    {
        if (pmant != NULL) *pmant = 0.0;
        return 0.0;
    }

    if (pmant != NULL)
    {
        if ((pp[0] & 0x80) == 0)
            *pmant = mantissa / two_pow56;
        else
            *pmant = -(mantissa / two_pow56);
    }

    // Check sign bit and return appropriate result
    if ((pp[0] & 0x80) == 0)
        return (mantissa / two_pow56) * powl(2, exponent);
    else
        return -(mantissa / two_pow56) * powl(2, exponent);
}

