// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: dfileb.cpp
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 01/25/2000
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The disk file class is a general-purpose base class used handle
file and directory functions.
*/
// ----------------------------------------------------------- // 

// Define the __DOS_INCLUDES__ macro to use DOS path separators
#if defined(__DOS_INCLUDES__) && !defined(__UNIX__)
#include <sys\types.h>
#include <sys\stat.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#endif

// DOS and WIN32 include files for file/directory functions
#if defined(__DOS__) || defined(__WIN32__)
#include <direct.h>
#include <io.h>
#endif

// Include files needed to compile with DJGPP 2.7.2.1
#if defined(__DJGPP2721__)
#include <unistd.h>
#include <fcntl.h>
#endif

// UNIX include files for file/directory functions
#if defined(__UNIX__)
#include <unistd.h> // UNIX standard function definitions 
#endif

#include <stdio.h>  // For rename() and remove() functions
#include <string.h>
#include "dfileb.h"

const char *df_FILE_ERROR_MESSAGES[df_MAX_FILE_ERRORS] = {
  "Disk file exception: No errors reported", // df_NO_ERROR
  "Disk file exception: Invalid error code", // df_INVALID_ERROR_CODE 

  // File error exception strings
  "Disk file exception: Error closing the file",                     // CLOSE
  "Disk file exception: A fatal error occurred while copying",       // COPY
  "Disk file exception: Could not create the specified file",        // CREATE
  "Disk file exception: End of file error occurred",                 // EOF   
  "Disk file exception: The file/directory does not exist",          // EXIST
  "Disk file exception: Error flushing the file buffers",            // FLUSH
  "Disk file exception: An bad/invalid access mode was specified",   // MODE
  "Disk file exception: Cannot open the specified file",             // OPEN
  "Disk file exception: A fatal file read error occcurred",          // READ
  "Disk file exception: Could remove the specified file",            // REMOVE
  "Disk file exception: Could not rename the specified file",        // RENAME
  "Disk file exception: Error rewinding the file",                   // REWIND
  "Disk file exception: An error occcurred during a seek operation", // SEEK
  "Disk file exception: A fatal file write error occcurred",         // WRITE  

  // Directory error exception strings
  "Disk file exception: Could not change directory",               // CHDIR
  "Disk file exception: Could not create the specified directory", // MKDIR
  "Disk file exception: Could not remove the specified directory"  // RMDIR
};

char *DiskFileB::df_GetFileError(int error_code)
// Returns a null terminated string that
// represents the specified error code.
{
  if(error_code > (int)df_MAX_FILE_ERRORS)
    error_code = DiskFileB::df_INVALID_ERROR_CODE;
  return (char *)df_FILE_ERROR_MESSAGES[error_code];
}

const char *DiskFileB::DiskFileExceptionMessage()
// Returns a null terminated string that can
// be used to log or print a disk file exception.
{
  if(df_last_error > df_MAX_FILE_ERRORS)
    df_last_error = DiskFileB::df_INVALID_ERROR_CODE;
  return df_FILE_ERROR_MESSAGES[df_last_error];
}

char *DiskFileB::df_GetFileError()
// Returns a null terminated string that
// represents the last file error.
{
  if(df_last_error > df_MAX_FILE_ERRORS)
    df_last_error = DiskFileB::df_INVALID_ERROR_CODE;
  return (char *)df_FILE_ERROR_MESSAGES[df_last_error];
}

DiskFileB::DiskFileB()
{ 
  // Reset the file status and operation members
  df_fptr = 0;
  df_file_exists = df_file_open = 0;
  df_is_ok = df_ready_for_reading = df_ready_for_writing = 0;
  df_last_error = DiskFileB::df_NO_ERROR;
  df_last_operation = DiskFileB::df_NO_OPERATION;
  df_file_length = 0;
  df_ClearNameBuffer();
}

DiskFileB::DiskFileB(const char *fname, 
		 int mode, int create, int truncate, int share)
{
  df_Open(fname, mode, create, truncate, share);
}

DiskFileB::~DiskFileB()
{
  df_Close();
}

void DiskFileB::df_Copy(const DiskFileB &ob)
// Private member function used to copy DiskFileB objects.
// Copying or assignment should never be allowed unless
// each copy is accounted for by reference counting or 
// some other method.
{
  df_fptr = ob.df_fptr;
  df_file_exists = ob.df_file_exists;
  df_file_open = ob.df_file_open;
  df_is_ok = ob.df_is_ok; 
  df_ready_for_reading = ob.df_ready_for_reading;
  df_ready_for_writing = ob.df_ready_for_writing;
  df_last_error = ob.df_last_error;
  df_last_operation = ob.df_last_operation; 
  df_LoadNameBuffer((const char *)ob.df_open_file_name);
  df_file_length = ob.df_file_length;
}

void DiskFileB::df_LoadNameBuffer(const char *s)
// Private member function used to load the
// open file name into the name buffer.
{
  df_ClearNameBuffer();
  unsigned len = strlen(s);

  // Prevent an overflow if the maximum name length is exceeded
  if(len > df_MAX_NAME_LENGTH) len = df_MAX_NAME_LENGTH; 

  // Load the name buffer
  memmove(df_open_file_name, s, len);
  df_open_file_name[len] = '\0'; // Ensure null termination
}

void DiskFileB::df_ClearNameBuffer()
// Private member function used to clear the name buffer.
{
  for(unsigned i = 0; i < df_MAX_NAME_LENGTH; i++)
    df_open_file_name[i] = '\0'; // Clear the name buffer;
}

int DiskFileB::df_Open(const char *fname, 
		     int mode, int create, int truncate, int share,
		     fstream *f)
// Open the specified file. The "mode", "create", "truncate", and
// "share" values must correspond to one of the numerical values 
// defined in the DiskFileB access/open mode enumeration. If no
// fstream pointer is specified an internal fstream object will
// be used. Returns df_NO_ERROR if no errors occur or a DiskFileB
// error code if an error is encountered during an open call.
{
  if(!f) df_fptr = &df_iofile;

  if(df_IsOpen()) df_Close(); // Close the file before each open operation

  // Reset the file status and opeartion members
  df_file_exists = df_file_open = 0;
  df_is_ok = df_ready_for_reading = df_ready_for_writing = 0;
  df_last_error = df_NO_ERROR;
  df_last_operation = df_NO_OPERATION;

  int rv;
  df_file_exists = df_Exists(fname);

  if(!df_file_exists) {
    switch(create) {
      case DiskFileB::df_NO_CREATE :
	return df_last_error = DiskFileB::df_EXIST_ERROR;
      case DiskFileB::df_CREATE :
	rv = df_CreateFile(fname);
	if(rv != df_NO_ERROR) {
	  return df_last_error = rv;
	}
	break;
      default:
	return df_last_error = DiskFileB::df_EXIST_ERROR;
    }
  }
  
  // Get the length of the file before opening it
  df_file_length = df_FileSize(fname);

#if (defined(__DOS__) || defined(__WIN32__)) && \
(!defined(__MSVC_FILE_SHARING__))
  // Open the file in binary mode without using the file sharing modes
  // supported in MSVC
  if(mode == df_READONLY) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::in|ios::binary);
    else
      df_fptr->open(fname, ios::in|ios::binary);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_TRUNCATE)) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::out|ios::trunc|ios::binary);
    else
      df_fptr->open(fname, ios::out|ios::trunc|ios::binary);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_APPEND)) {
    if(share == df_SHARE)    
      df_fptr->open(fname, ios::out|ios::app|ios::binary);
    else
      df_fptr->open(fname, ios::out|ios::app|ios::binary);
  }
  else if((mode == df_READWRITE) && (truncate == df_TRUNCATE)) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::out|ios::in|ios::trunc|ios::binary); 
    else
      df_fptr->open(fname, ios::out|ios::in|ios::trunc|ios::binary);
  }
  else if((mode == df_READWRITE) && (truncate == df_APPEND)) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::out|ios::in|ios::app|ios::binary); 
    else
      if(share == df_SHARE)
	df_fptr->open(fname, ios::out|ios::in|ios::app|ios::binary); 
  }
  else {
    return df_last_error = DiskFileB::df_MODE_ERROR;
  }

#elif (defined(__DOS__) || defined(__WIN32__)) && \
(defined(__MSVC_FILE_SHARING__))
  // Open the file in binary mode using the MSVC file sharing modes
  // NOTE: filebuf::sh_read|filebuf::sh_write will allow file sharing.
  // For exclusive access the nProt bit is set to filebuf::sh_none
  if(mode == df_READONLY) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::in|ios::binary, filebuf::sh_read);
    else
      df_fptr->open(fname, ios::in|ios::binary, filebuf::sh_none);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_TRUNCATE)) {
    if(share == df_SHARE)
      df_fptr->open(fname, 
		     ios::out|ios::trunc|ios::binary, filebuf::sh_write);
    else
      df_fptr->open(fname, 
		     ios::out|ios::trunc|ios::binary, filebuf::sh_none);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_APPEND)) {
    if(share == df_SHARE)    
      df_fptr->open(fname, 
		     ios::out|ios::app|ios::binary, filebuf::sh_write);
    else
      df_fptr->open(fname, 
		     ios::out|ios::app|ios::binary, filebuf::sh_none);
  }
  else if((mode == df_READWRITE) && (truncate == df_TRUNCATE)) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::out|ios::in|ios::trunc|ios::binary, 
		     filebuf::sh_read|filebuf::sh_write);
    else
      df_fptr->open(fname, ios::out|ios::in|ios::trunc|ios::binary, 
		     filebuf::sh_read|filebuf::sh_none);
  }
  else if((mode == df_READWRITE) && (truncate == df_APPEND)) {
    if(share == df_SHARE)
      df_fptr->open(fname, ios::out|ios::in|ios::app|ios::binary, 
		     filebuf::sh_read|filebuf::sh_write);
    else
      if(share == df_SHARE)
	df_fptr->open(fname, ios::out|ios::in|ios::app|ios::binary, 
		       filebuf::sh_read|filebuf::sh_none);
  }
  else {
    return df_last_error = DiskFileB::df_MODE_ERROR;
  }
#elif defined(__UNIX__) 
  // In UNIX there is only one file type
  if(mode == df_READONLY) {
    df_fptr->open(fname, ios::in);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_TRUNCATE)) {
    df_fptr->open(fname, ios::out|ios::trunc);
  }
  else if((mode == df_WRITEONLY) && (truncate == df_APPEND)) {
    df_fptr->open(fname, ios::out|ios::app);
  }
  else if((mode == df_READWRITE) && (truncate == df_TRUNCATE)) {
    df_fptr->open(fname, ios::out|ios::in|ios::trunc);
  }
  else if((mode == df_READWRITE) && (truncate == df_APPEND)) {
    df_fptr->open(fname, ios::out|ios::in|ios::app);
  }
  else {
    return df_last_error = DiskFileB::df_MODE_ERROR;
  }
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  // Could not open the specified file
  if(!df_iofile) {
    return df_last_error = DiskFileB::df_OPEN_ERROR;
  }

  df_LoadNameBuffer(fname);
  df_file_open = df_is_ok = 1;
  
  switch(mode) {
    case df_READONLY :
      df_ready_for_reading = 1;
      break;
    case df_WRITEONLY :
      df_ready_for_writing = 1;
      break;
    case df_READWRITE :
      df_ready_for_reading = 1;
      df_ready_for_writing = 1;
      break;
    default:
      return df_last_error = DiskFileB::df_MODE_ERROR;
  }

  df_Rewind(); // Reset the input/output position offsets to 0 
  return DiskFileB::df_NO_ERROR;
}
   
int DiskFileB::df_CreateFile(const char *fname)
// Create the specified file. Returns zero
// if the file was created with no errors.
{
#if defined(__DOS__) || defined (__WIN32__)
  fstream tmp(fname, ios::out|ios::trunc|ios::binary);
#elif defined(__UNIX__) 
  fstream tmp(fname, ios::out|ios::trunc);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

    if(!tmp) {
      return df_last_error = DiskFileB::df_CREATE_ERROR;
    }

    tmp.close();
    return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_Exists(const char *name)
// Tests to see if the specified directory of file name 
// exists. Returns ture if the file or directory exists.

{
  struct stat buf;
  return stat(name, &buf) == 0;
}

int DiskFileB::df_Close() 
// Close the open file. Returns a df_CLOSE_ERROR
// error code if an error was encountered.
{
  if(!df_IsOpen()) { 
    return df_last_error = DiskFileB::df_CLOSE_ERROR;
  }
  df_ClearNameBuffer();
  df_file_open = 0; // Reset the open flag
  df_fptr->clear();
  df_fptr->close();
  df_ready_for_reading = df_ready_for_writing = 0;
  df_file_exists = 0;
  df_last_operation = df_NO_OPERATION;
  df_last_error = df_NO_ERROR; // Reset the last file error
  return df_last_error;
}

int DiskFileB::df_Rewind()
// Rewind the file pointers to zero and
// flush any open disk buffers. Returns a
// df_REWIND_ERROR error code if an error
// was encountered.
{
  if(!df_IsOK()) { 
    return df_last_error = DiskFileB::df_REWIND_ERROR;
  }
  df_fptr->seekg(0, ios::beg);
  df_fptr->seekp(0, ios::beg);
  df_last_operation = DiskFileB::df_REWIND;
  df_Flush();
  df_fptr->clear(); // Reset the EOF bit
  df_last_error = df_NO_ERROR; // Reset the last file error
  return df_last_error;
}

int DiskFileB::df_Flush()
// Flush any open disk buffers. Returns a
// df_FLUSH_ERROR error code if an error
// was encountered.
{
  if((!df_IsOK()) || (!df_ReadyForWriting())) {
    return df_last_error = DiskFileB::df_FLUSH_ERROR;
  }
  df_fptr->flush();
  df_last_error = df_NO_ERROR; // Reset the last file error
  return df_last_error;
}

df_StreamPos DiskFileB::df_FileSize(const char *fname)
// Returns the file size of the specified file. 
{
  struct stat buf;
  int result = stat(fname, &buf);
  if(result != 0) return (df_StreamPos)0;
  return (df_StreamPos)buf.st_size;
}

int DiskFileB::df_Read(void *buf, unsigned bytes, df_StreamPos position)
// Read a specified number of bytes into a buffer. Returns df_NO_ERROR
// if no errors occur or a DiskFileB error code if an error is
// encountered during a read call.
{
  if((df_IsOK()) && (df_ReadyForReading())) {
    if(position != df_CurrPosition) { 
      // Seek to the file position unless the position variable is set to -1, 
      // which means from the current position
      df_SeekTo(position, DiskFileB::df_READ);

      // Removed 03/15/2000: This code will interpret the file 
      // position always starting from the beginning of the file.
      // int rv = df_Seek(position, DiskFileB::df_SEEK_BEG, 
      //                  DiskFileB::df_READ); 
      // if(rv != DiskFileB::df_NO_ERROR){
      //	return df_last_error = DiskFileB::df_SEEK_ERROR;
      // }
    }
    if(df_fptr->eof()) { // At the end of the file
      return df_last_error = DiskFileB::df_EOF_ERROR;
    }

    df_fptr->read((char *)buf, bytes);

    if(!df_fptr->good()) { // An I/O error occurred
      if(df_fptr->bad()) { // The I/O error was fatal
	return df_last_error = DiskFileB::df_READ_ERROR;
      }
      // Continue processing if the error was not fatal
    }
  }
  else { // Cannot read from the file
    return df_last_error = DiskFileB::df_READ_ERROR;
  }

  df_last_operation = DiskFileB::df_READ;
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_Write(const void *buf, unsigned bytes, df_StreamPos position)
// Write a specified number of bytes from a buffer to the open file. 
// Returns df_NO_ERROR if no errors occur or a DiskFileB error code 
// if an error is encountered during a write call.
{
  //  int rv;
  if((df_IsOK()) && (df_ReadyForWriting())) {
    if(position != df_CurrPosition) { 
      // Seek to the file position unless the position variable is set to -1, 
      // which means from the current position
      df_SeekTo(position, DiskFileB::df_WRITE);

      // Removed 03/15/2000: This code will interpret the file 
      // position always starting from the beginning of the file.
      // rv = df_Seek(position, DiskFileB::df_SEEK_BEG, DiskFileB::df_WRITE); 
      // if(rv != DiskFileB::df_NO_ERROR){
      //   return df_last_error = DiskFileB::df_SEEK_ERROR;
      // }
    }

    df_fptr->write((const char *)buf, bytes);

    if(!df_fptr->good()) { // An I/O error occurred
      if(df_fptr->eof()) { // End of the file error
	return df_last_error = DiskFileB::df_EOF_ERROR;
      }
      if(df_fptr->bad()) { // The I/O error was fatal
	return df_last_error = DiskFileB::df_WRITE_ERROR;
      }
      // Continue processing if the error was not fatal
    }
  }
  else { // Cannot write to the file
    return df_last_error = DiskFileB::df_WRITE_ERROR;
  }

  df_file_length += (df_StreamPos)bytes; // Increment the file_length variable
  df_last_operation = DiskFileB::df_WRITE;
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_Seek(df_StreamPos position, int seek_mode, int operation)
// Move the file offset to the specified file position. Returns a
// df_SEEK_ERROR if an error occurred during a file seek. If no
// operation is specified both the get and put offsets will be moved
// to the specified position.
{
  if(!df_IsOK()) {
    return df_last_error = DiskFileB:: df_SEEK_ERROR;
  }

  switch(operation) {
    case df_READ :
      if(seek_mode == DiskFileB::df_SEEK_BEG)
	df_fptr->seekg(position, ios::beg);
      if(seek_mode == DiskFileB::df_SEEK_CUR)
	df_fptr->seekg(position, ios::cur);
      if(seek_mode == DiskFileB::df_SEEK_END)
	df_fptr->seekg(position, ios::end);
      break;
    case df_WRITE :
      if(seek_mode == DiskFileB::df_SEEK_BEG)
	df_fptr->seekp(position, ios::beg);
      if(seek_mode == DiskFileB::df_SEEK_CUR)
	df_fptr->seekp(position, ios::cur);
      if(seek_mode == DiskFileB::df_SEEK_END)
	df_fptr->seekp(position, ios::end);
      break;
    case df_NO_OPERATION :
      if(seek_mode == DiskFileB::df_SEEK_BEG) {
	df_fptr->seekg(position, ios::beg);
	df_fptr->seekp(position, ios::beg);
      }
      if(seek_mode == DiskFileB::df_SEEK_CUR) {
	df_fptr->seekg(position, ios::cur);
	df_fptr->seekp(position, ios::cur);
      }
      if(seek_mode == DiskFileB::df_SEEK_END) {
	df_fptr->seekg(position, ios::end);
	df_fptr->seekp(position, ios::end);
      }
      break;
    default: // An invalid operation was specified 
      return df_last_error = DiskFileB:: df_SEEK_ERROR;
  }
  return DiskFileB::df_NO_ERROR;
}

df_StreamPos DiskFileB::df_FilePosition(int operation)
// Returns the current file position after a read, write,
// or rewind operation. Returns -1 if an error occurred
// or an invalid operation was performed.
{
  if(!df_IsOK()) return -1;
  if(operation == df_NO_OPERATION) operation = df_last_operation;
  df_StreamPos pos;
  switch(operation) {
    case DiskFileB::df_READ :
      pos = (df_StreamPos)df_fptr->tellg();
      break;
    case DiskFileB::df_WRITE :
      pos = (df_StreamPos)df_fptr->tellp();
      break;
    case DiskFileB::df_REWIND :
      pos = (df_StreamPos)0;
      break;
    default:
      pos = (df_StreamPos)df_CurrPosition;
  }
  return pos;
}

int DiskFileB::df_ReOpen(fstream *f, int mode)
// Connects the file object pointer to another pointer.
// This function assumes the file has been opened by the
// calling function.  The calling function must tell this 
// function if the file was opened for read, write, or
// read/write access.
{
  if(df_IsOpen()) df_Close(); // Close the open file first
  df_fptr = f;
  df_file_exists = df_file_open = 1;

  df_file_length = 0; // The file length must be set by the derived class
  df_is_ok = df_ready_for_reading = df_ready_for_writing = 0;
  df_last_error = df_NO_ERROR;
  df_last_operation = df_NO_OPERATION;

  switch(mode) {
    case DiskFileB::df_READONLY :
      df_ready_for_reading = 1;
      break;
    case DiskFileB::df_WRITEONLY :
      df_ready_for_writing = 1;
      break;
    case DiskFileB::df_READWRITE :
      df_ready_for_reading = 1;
      df_ready_for_writing = 1;
      break;
    default:
      return df_last_error = DiskFileB::df_MODE_ERROR;
  }

  df_is_ok = 1;
  return DiskFileB::df_NO_ERROR;
}

df_StreamPos DiskFileB::df_SeekTo(df_StreamPos position, int operation)
// Seek to the specified position, optimizing the seek
// operation by moving the file position indicator based
// on the current stream position. Returns the current
// file position after performing the seek to operation.
{
  // Get the current stream position
  df_StreamPos pos = df_FilePosition(operation);

  if(position == df_CurrPosition) { // Do not perform a seek operation
    return pos;
  }
  else if(position > pos) { // Seek forward to the specified address
    df_StreamPos offset = position - pos;
    df_Seek(offset, df_SEEK_CUR, operation);
  }
  else if(position < pos) { // Seek backward to the specified address
    df_Seek(position, df_SEEK_BEG, operation);
  }
  
  // Return current file position after seeking
  return df_FilePosition(operation); 
}

int DiskFileB::df_GetLine(void *buf, unsigned bytes)
// Get a line of text from the open file. Returns df_NO_ERROR
// if no errors occur or a DiskFileB error code if an error is
// encountered during a getline call. NOTE: This function
// assumes that a buffer equal to the size of the "bytes"
// variable has already been allocated.
{
  if((df_IsOK()) && (df_ReadyForReading())) {
    if(df_fptr->eof()) { // At the end of the file
      return df_last_error = DiskFileB::df_EOF_ERROR;
    }
    df_fptr->getline((char *)buf, bytes);
    
    if(!df_fptr->good()) { // An I/O error occurred
      if(df_fptr->bad()) { // The I/O error was fatal
	return df_last_error = DiskFileB::df_READ_ERROR;
      }
      // Continue processing if the error was not fatal
    }
  }
  else { // Cannot read from the file
    return df_last_error = DiskFileB::df_READ_ERROR;
  }

  df_last_operation = DiskFileB::df_READ;
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_Get(char &c)
// Get a single character from the open file.
// Returns df_NO_ERROR if no errors occur or a
// DiskFileB error code if an error is encountered
// during a get call.
{
  if((df_IsOK()) && (df_ReadyForReading())) {
    if(df_fptr->eof()) { // At the end of the file
      return df_last_error = DiskFileB::df_EOF_ERROR;
    }
    df_fptr->get(c);
    
    if(!df_fptr->good()) { // An I/O error occurred 
      if(df_fptr->bad()) { // The I/O error was fatal
	return df_last_error = DiskFileB::df_READ_ERROR; 
      }
      // Continue processing if the error was not fatal
    }
  }
  else { // Cannot read from the file
    return df_last_error = DiskFileB::df_READ_ERROR;
  }

  df_last_operation = DiskFileB::df_READ;
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_Put(const char &c)
// Put a single character into the open file.
// Returns df_NO_ERROR if no errors occur or a
// DiskFileB error code if an error is encountered
// during a put call.

{
  if((df_IsOK()) && (df_ReadyForWriting())) {

    df_fptr->put(c);

    if(!df_fptr->good()) { // An I/O error occurred
      if(df_fptr->eof()) { // End of the file error
	return df_last_error = DiskFileB::df_EOF_ERROR;
      }
      if(df_fptr->bad()) { // The I/O error was fatal
	return df_last_error = DiskFileB::df_WRITE_ERROR;
      }
      // Continue processing if the error was not fatal
    }
  }
  else { // Cannot write to the file
    return df_last_error = DiskFileB::df_WRITE_ERROR;
  }

  df_file_length++; // Increment the file_length variable
  df_last_operation = DiskFileB::df_WRITE;
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_IsDirectory(const char *dir_name) 
// Returns true if the specified name is a directory.
{
  struct stat buf;

#ifndef lstat
  if (stat(dir_name, &buf) != 0) return 0;
#else
  // Use lstat() rather than stat() so that symbolic links
  // pointing to directories can be identified correctly. 
  if (lstat(fname, &buf) != 0) return 0;
#endif

#if defined (_S_IFMT) && defined(_S_IFDIR)
  if((buf.st_mode & _S_IFMT) == _S_IFDIR) return 1;
#else
  if((buf.st_mode & S_IFMT) == S_IFDIR) return 1;
#endif
  
 return 0;
}

int DiskFileB::df_IsFile(const char *fname)
// Returns true if the specified name is any kind of file
// or false if this is a directory.
{
 return df_IsDirectory(fname) == 0;
}

void DiskFileB::df_MakeDOSPath(char *path)
// Make a DOS directory path by changing all
// forward slash path separators with back
// slashes.
{
  unsigned len = strlen(path);
  while(len--) {
    char c = *path;
    switch(c) {
      case '/' :
	*path = '\\';
	break;
      default:
	break;
    }
    path++;
  }
}

void DiskFileB::df_MakeUNIXPath(char *path)
// Make a UNIX directory path by changing all
// back slash path separators with forward
// slashes.
{
  unsigned len = strlen(path);
  while(len--) {
    char c = *path;
    switch(c) {
      case '\\' :
	*path = '/';
	break;
      default:
	break;
    }
    path++;
  }
}

int DiskFileB::df_chdir(const char *dir_name)
// Change directory. Returns zero if successful.
{
  if(!df_Exists(dir_name)) {
    return df_last_error = DiskFileB::df_EXIST_ERROR; 
  }

  char *path = (char *)dir_name;
#if defined(__DOS__) || defined(__WIN32__)
  df_MakeDOSPath(path);
#elif defined(__UNIX__)
  df_MakeUNIXPath(path);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  if(chdir(path) != 0) {
    return df_last_error = DiskFileB::df_CHDIR_ERROR;
  }
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_rmdir(const char *dir_name)
// Remove the specified directory. Returns zero if successful.
// The directory must be empty and not be the current working
// directory or the root directory.
{
  if(!df_Exists(dir_name)) {
    return df_last_error = DiskFileB::df_EXIST_ERROR; 
  }

  char *path = (char *)dir_name;
#if defined(__DOS__) || defined(__WIN32__)
  df_MakeDOSPath(path);
#elif defined(__UNIX__)
  df_MakeUNIXPath(path);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  if(rmdir(path) != 0) {
    return df_last_error = DiskFileB::df_RMDIR_ERROR;
  }
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_HasDriveLetter(const char *dir_name, char &letter)
// DOS/WIN32 function used to parse the drive letter from the
// specified directory. Returns false if the path does not
// contain a drive letter. If a drive letter is found it will
// be passed back in the "letter" variable. 
{
  char *s = (char *)dir_name;
  letter = 0; // Reset the letter variable
  s++; if(*s != ':') return 0; // No drive letter found 
  
  // Get the drive letter following the first colon
  s--; letter = *s;
  return 1;
}

int DiskFileB::df_mkdir(const char *dir_name)
// Make the specified directory if it does not exist.
// Returns a non-zero value if an error occurs. UNIX
// file systems will use 755 permissions when new
// directories are created.
{
  if(df_Exists(dir_name)) { // Directory exists, return with no error reported
    return df_last_error = DiskFileB::df_NO_ERROR;
  }
  char *path = (char *)dir_name;
  
  char path_sep;
#if defined(__DOS__) || defined(__WIN32__)
  char drive_letter;
  int has_drive_letter = df_HasDriveLetter(dir_name, drive_letter);
  df_MakeDOSPath(path);
  path_sep = '\\';
#elif defined(__UNIX__)
  df_MakeUNIXPath(path);
  path_sep = '/';
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  // The mkdir function can only create one new directory per call
  // so the complete directory path can only be created one sub-
  // driectory at a time.
  char sbuf[df_MAX_DIR_LENGTH]; unsigned i = 0; int rv;
  for(i = 0; i < df_MAX_DIR_LENGTH; i++) sbuf[i] = '\0';

  unsigned len = strlen(path); char *s = (char *)path;
  if(len > df_MAX_DIR_LENGTH) len = df_MAX_DIR_LENGTH;

  for(i = 0; i < len; i++, s++) {

    // Get the whole directory path
    if(i == (len-1)) { memmove(sbuf, (char *)path, len); break; }

#if defined(__DOS__) || defined(__WIN32__)
    if((has_drive_letter) && (i == 1)) { // Skip over the drive letter
      i++; s++; 
      if(*s == path_sep) { i++; s++; }
    }
#endif

    if((*s == path_sep) && (i > 1)) { // Step past the root directory
      memmove(sbuf, path, i);
      if(!df_Exists(sbuf)) {
#if defined(__DJGPP2721__)
	rv = mkdir(sbuf, 
		   S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#elif (defined(__DOS__) || defined(__WIN32__)) && (!defined(__DJGPP2721__))
	rv = mkdir(sbuf);
#elif defined(__UNIX__)
	rv = mkdir(sbuf,
		   S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif
	if(rv != 0) {
	  return df_last_error = DiskFileB::df_MKDIR_ERROR;
	}
      }      
    }
  }

  // Get rid of any terminating path separators
  if(sbuf[len-1] == path_sep) sbuf[len-1] = '\0';

  // Make the complete directory path if needed
  if(!df_Exists(sbuf)) {
#if defined(__DJGPP2721__)
	rv = mkdir(sbuf,
		   S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#elif (defined(__DOS__) || defined(__WIN32__)) && (!defined(__DJGPP2721__))
	rv = mkdir(sbuf);
#elif defined(__UNIX__)
    rv = mkdir(sbuf,
	       S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif
    if(rv != 0) {
      return df_last_error = DiskFileB::df_MKDIR_ERROR;
    }
  }      
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_remove(const char *fname)
{
  if(!df_Exists(fname)) {
    return df_last_error = DiskFileB::df_EXIST_ERROR; 
  }
  if(remove(fname) != 0) { 
   return df_last_error = DiskFileB::df_REMOVE_ERROR;
  }
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_rename(const char *oldname, const char *newname)
{
  if(!df_Exists(oldname)) {
    return df_last_error = DiskFileB::df_EXIST_ERROR; 
  }
  if(rename(oldname, newname) != 0) {
    return df_last_error = DiskFileB::df_RENAME_ERROR;
  }
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_copy(const char *from, const char *to)
{
#if defined(__DOS__) || defined (__WIN32__)  
  ifstream in(from, ios::in | ios::nocreate | ios::binary);
#elif defined(__UNIX__) 
  ifstream in(from, ios::in | ios::nocreate);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  if(!in) {
    return df_last_error = DiskFileB::df_COPY_ERROR;
  }

  ofstream out;

#if defined(__DOS__) || defined (__WIN32__)
  out.open(to, ios::out | ios::noreplace | ios::binary);
#elif defined(__UNIX__) 
  out.open(to, ios::out | ios::noreplace);
#else
#error You must define a file system: __DOS__ __WIN32__ or __UNIX__
#endif

  if(!out) {
    return df_last_error = DiskFileB::df_COPY_ERROR;
  }

  char ch;
  while(in.get(ch)) out.put(ch);
  if(!in.eof() || out.bad()) {
    return df_last_error = DiskFileB::df_COPY_ERROR;
  }
  
  return DiskFileB::df_NO_ERROR;
}

int DiskFileB::df_chmod(const char *fname, int pmode)
// Change the file-permission settings. The "pmode"
// variable should be equal to df_READONLY, df_WRITEONLY,
// or df_READWRITE. Returns a non-zero value if an 
// error occurs.
{
  if(!df_Exists(fname)) return DiskFileB::df_EXIST_ERROR;

  switch(pmode) {
    case DiskFileB::df_READONLY :
#if defined (_S_IREAD)
      if(chmod(fname, _S_IREAD) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#else
      if(chmod(fname, S_IREAD) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#endif
      break;

    case DiskFileB::df_WRITEONLY :
#if defined (_S_IWRITE)
      if(chmod(fname, _S_IWRITE) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#else
      if(chmod(fname, S_IWRITE) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#endif
      break;

    case DiskFileB::df_READWRITE :
#if defined (_S_IREAD) && defined (_S_IWRITE)
      if(chmod(fname, _S_IREAD | _S_IWRITE) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#else
      if(chmod(fname, S_IREAD | S_IWRITE) != 0) {
	return df_last_error = DiskFileB::df_MODE_ERROR;
      }
#endif
      break;

    default:
      return df_last_error = DiskFileB::df_MODE_ERROR;
  }

  return DiskFileB::df_NO_ERROR;
}

void DiskFileB::df_PathSimplify(const char *path, char *new_path,
				char path_sep)
// Function used to canonicalize a path and return a new path.
// The path separator should either be a forward slash for UNIX
// file systems or a backslash for DOS/WIN32 file systems. If no
// path separator is specified a forward slash will be used by
// default. Multiple path separators will be collapsed to a single
// path separator. Leading `./' paths and trailing `/.' paths will
// be removed. Trailing path separators will be removed. All non-
// leading `../' paths and trailing `..' paths are handled by 
// removing portions of the path. NOTE: This function assumes that
// the necessary memory has already been allocated for the "new_path"
// variable by the calling function.
{
  if(!*path) return;

  // Copy the complete path into the new_path variable
  unsigned len = strlen(path);
  memmove(new_path, path, len);
  new_path[len] = '\0'; // Ensure null termination

  int i, start;
  int ddot; // Directory immediately above the current directory position
  char stub_char = (*new_path == path_sep) ? path_sep : '.';

  // Remove all `./' paths preceding the string.  If `../' paths
  // precede, put `/' in front and remove them as well
  i = ddot = start = 0;
  while (1) {
    if(new_path[i] == '.' && new_path[i + 1] == path_sep)
      i += 2;
    else if (new_path[i] == '.' && new_path[i + 1] == '.' && \
	     new_path[i + 2] == path_sep) { 
      i += 3;
      ddot = 1;
    }
    else
      break;
  }
  if(i) strcpy(new_path, new_path + i - ddot);

  // Replace single `.' or `..' with `/'.  
  if((new_path[0] == '.' && new_path[1] == '\0') 
     || (new_path[0] == '.' && new_path[1] == '.' && new_path[2] == '\0')) {
    new_path[0] = stub_char;
    new_path[1] = '\0';
    return;
  }

  // Walk along the path looking for paths to compact  
  i = 0;
  while(1) {
    if(!new_path[i]) break;

    while(new_path[i] && new_path[i] != path_sep)  i++;

    start = i++;

    // If we didn't find any slashes, then there is nothing left to do
    if(!new_path[start]) break;

    // Handle multiple path separators in a row
    while(new_path[i] == path_sep) i++;

    if((start + 1) != i) {
      strcpy(new_path + start + 1, new_path + i);
      i = start + 1;
    }

    // Check for trailing path separator
    if(start && !new_path[i]) {
      new_path[--i] = '\0';
      break;
    }

    // Check for `../' paths, `./' paths or a trailing `.' by itself
    if(new_path[i] == '.') {
      // Handle trailing a `.' path 
      if(!new_path[i + 1]) {
	new_path[--i] = '\0';
	break;
      }

      // Handle `./' paths
      if(new_path[i + 1] == path_sep) {
	strcpy(new_path + i, new_path + i + 1);
	i =(start < 0) ? 0 : start;
	continue;
      }

      // Handle `../' paths or a trailing `..' path by itself
      if(new_path[i + 1] == '.' &&
	 (new_path[i + 2] == path_sep || !new_path[i + 2])) {
	while(--start > -1 && new_path[start] != path_sep);
	strcpy(new_path + start + 1, new_path + i + 2);
	i = (start < 0) ? 0 : start;
	continue;
      }
    }
  }

  if(!*new_path) { // Nothing was left after the path was simplifed
     *new_path = stub_char;
    new_path[1] = '\0';
  }

  // Ensure null termination
  new_path[strlen(new_path)] = '\0';
}

int DiskFileB::df_GenOutputFileName(const char *current_file, char *out_file, 
				    const char *extension)
// Generate a name for the output file using the "current_file" name with 
// the specified dot extension. NOTE: This function assumes that the 
// necessary memory has already been allocated for the "out_file"
// variable by the calling function. Returns a non-zero value if any
// errors occur.
{
  if(!out_file) return 1;
  unsigned i = 0;
  for(i = 0; i < df_MAX_NAME_LENGTH; i++) out_file[i] = '\0';
  char *p = (char *)current_file;
  unsigned len = strlen(p);
  for(i = 0; i < len && i != df_MAX_NAME_LENGTH; i++, p++) {
    if(*p == '.') break;
    out_file[i] = *p;
  }
  if((strlen(out_file) + strlen(extension)) > (df_MAX_NAME_LENGTH - 1)) 
    return 1;
  strcat(out_file, extension); // Add the file extension (.xxx)
  return 0;
}

int DiskFileB::df_pwd(char *dir, unsigned max_len)
// Passes back the present working directory in the "dir"
// variable. Returns 0 if an error occurs. NOTE: This
// function assumes that the required amount of memory
// has already been allocated for the "dir" pointer. The
// "max_len" value must be at least one greater than the
// length of the pathname to be returned.
{
  // Clear the path name buffer
  unsigned i = 0;
  for(i = 0; i < max_len; i++) dir[i] = '\0';

  if(getcwd(dir, max_len) == 0) {
    return df_last_error = DiskFileB::df_EXIST_ERROR;
  }

  // Ensure null termination
  dir[strlen(dir)] = '\0';
  return DiskFileB::df_NO_ERROR;
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
