//============================================================
// PRINT.C - Prepackaged print routines.
//============================================================
#include "windows.h"
#include "commdlg.h"
#include "print.h"

#include "stdlib.h"
#include "stdio.h"
#include "string.h"
#include "time.h"

//-----------------------------------------------------------
// Generic defines and data types
//-----------------------------------------------------------
//
// Private defines
//
#define     PRO_PRINT     "Printing"
#define     PRO_HEADER    "Header"
#define     PRO_FOOTER    "Footer"
#define     PRO_MLEFT     "MargLeft"
#define     PRO_MTOP      "MargTop"
#define     PRO_MRIGHT    "MargRight"
#define     PRO_MBOTTOM   "MargBottom"
#define     PRO_LINES     "PageLines"
#define INT       int
#define UINT      WORD
#define APIENTRY  PASCAL
#define WNDPROC   FARPROC

#include "myprint.h"

//
// Private structures used by this module
//
typedef struct {
	char szHText[128];
	char szFText[128];
	LPSTR lpName;
	RECT rect;
	HFONT hFont;
	BOOL fPrint;
	HWND hWndStat;
	INT sTabs;
	INT sLines;
} MYPRNSTRUCT;	
typedef MYPRNSTRUCT *PMYPRNSTRUCT;

typedef struct {
	char szHeader[80];
	char szFooter[80];
	LONG left;
	LONG top;
	LONG right;
	LONG bottom;
	INT sLines;
} MYPAGESTRUCT;	
typedef MYPAGESTRUCT *PMYPAGESTRUCT;

INT PrintPage (HDC, PMYPRNSTRUCT, LPSTR *, UINT);
HDC MyPrintDlg (HWND, PRINTDLG *, DWORD);
HDC GetPrnhDC (HWND, PRINTDLG *);
LPSTR MakeHeadStr (LPSTR, LPSTR, INT, LPSTR, INT);
INT HeaderOut (HDC, RECT *, LPSTR);
BOOL CALLBACK PrintPageSetupDlgProc (HWND, UINT, UINT, LONG);

//
// Global Data
//
HINSTANCE hInst;
PRINTDLG pd;
MYPAGESTRUCT mp;
BOOL fContinue;
FARPROC lpfnAbortProc;
DLGPROC lpfnPageSetupProc;

char szDebug[512];

//------------------------------------------------------------
// PrnYield - Yields control to other programs, but returns
// if Windows is idle.
//------------------------------------------------------------
BOOL PrnYield (void) {
   MSG	msg;
   BOOL	bCont;
   
   bCont = TRUE;
	while (PeekMessage (&msg, NULL, 0, 0, PM_NOREMOVE)) {
	   if (msg.message == WM_QUIT) 
	      return FALSE;

		GetMessage (&msg, NULL, 0, 0);
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
	return bCont;
}
//------------------------------------------------------------
// PrnAbortProc - Abort proc for print process
//------------------------------------------------------------
BOOL FAR PASCAL PrnAbortProc (HDC hdcPrn, int nCode) {

	if (nCode) {
//wsprintf (szDebug, "Abort proc printer error: %d,\n", nCode);
//OutputDebugString (szDebug);
		if (nCode == SP_OUTOFDISK) {
			if (!PrnYield())
				fContinue = FALSE;
		} else	
			return FALSE;
	}		
	return fContinue;
}
//------------------------------------------------------------ 
// MyPrintInit - Printer init function.
//------------------------------------------------------------ 
INT MyPrintInit (HINSTANCE hInstance, HWND hWnd, LPSTR lpszProfile) {
	INT rc;
	char szTemp[64];

	hInst = hInstance;

	// Init page structure
	GetPrivateProfileString (PRO_PRINT, PRO_HEADER, "&f", mp.szHeader,
	                         sizeof (mp.szHeader), lpszProfile);
	GetPrivateProfileString (PRO_PRINT, PRO_FOOTER, "Page &p", mp.szFooter,
	                         sizeof (mp.szFooter), lpszProfile);

	GetPrivateProfileString (PRO_PRINT, PRO_MLEFT, "10800", szTemp,
	                         sizeof (szTemp), lpszProfile);
	mp.left = atol (szTemp);
	GetPrivateProfileString (PRO_PRINT, PRO_MTOP, "14400", szTemp,
	                         sizeof (szTemp), lpszProfile);
	mp.top = atol (szTemp);
	GetPrivateProfileString (PRO_PRINT, PRO_MRIGHT, "10800", szTemp,
	                         sizeof (szTemp), lpszProfile);
	mp.right = atol (szTemp);
	GetPrivateProfileString (PRO_PRINT, PRO_MBOTTOM, "7200", szTemp,
	                         sizeof (szTemp), lpszProfile);
	mp.bottom = atol (szTemp);
	mp.sLines = GetPrivateProfileInt (PRO_PRINT, PRO_LINES, 60, 
	                                  lpszProfile);
	fContinue = FALSE;

	lpfnAbortProc = MakeProcInstance (PrnAbortProc, hInst);
	lpfnPageSetupProc = MakeProcInstance((FARPROC)PrintPageSetupDlgProc, hInst);

	// Init common dlg print structure
	memset(&pd, 0, sizeof(PRINTDLG));
	rc = MyPrintDlg (hWnd, &pd, PD_RETURNDEFAULT);
	if (rc == 0)
		return 1;
	return 0;	
}	
//------------------------------------------------------------ 
// MyPrintTerm - Printer cleanup function.
//------------------------------------------------------------ 
INT MyPrintTerm (HINSTANCE hInstance, HWND hWnd, LPSTR lpszProfile) {
	char szTemp[64];

	// Save page structure
	if (lpszProfile) {
		WritePrivateProfileString (PRO_PRINT, PRO_HEADER, mp.szHeader,
		                           lpszProfile);
		WritePrivateProfileString (PRO_PRINT, PRO_FOOTER, mp.szFooter,
		                           lpszProfile);
		wsprintf (szTemp, "%ld", mp.left);
		WritePrivateProfileString (PRO_PRINT, PRO_MLEFT, szTemp, lpszProfile);
		wsprintf (szTemp, "%ld", mp.top);
		WritePrivateProfileString (PRO_PRINT, PRO_MTOP, szTemp, lpszProfile);
		wsprintf (szTemp, "%ld", mp.right);
		WritePrivateProfileString (PRO_PRINT, PRO_MRIGHT, szTemp, lpszProfile);
		wsprintf (szTemp, "%ld", mp.bottom);
		WritePrivateProfileString (PRO_PRINT, PRO_MBOTTOM, szTemp, lpszProfile);
		wsprintf (szTemp, "%d", mp.sLines);
		WritePrivateProfileString (PRO_PRINT, PRO_LINES, szTemp, lpszProfile);
	}
	// Free the abort proc thunk
	FreeProcInstance (lpfnAbortProc); 
	FreeProcInstance (lpfnPageSetupProc); 

	// Free print structures
	if (pd.hDevMode != NULL)
		GlobalFree(pd.hDevMode);
	if (pd.hDevNames != NULL)
		GlobalFree(pd.hDevNames);

	return 0;	
}	
//------------------------------------------------------------ 
// MyPrintSetup - Display Printer Setup Dialog
//------------------------------------------------------------ 
INT MyPrintSetup (HWND hWnd) {
	INT rc;

	// Call print dialog
	pd.lStructSize = sizeof(PRINTDLG);
	pd.hwndOwner = hWnd;
	pd.Flags = PD_PRINTSETUP;
	rc = PrintDlg(&pd);
	if (rc == 0)
		return 1;

	return 0;
}	
//------------------------------------------------------------ 
// MyPrintPageSetup - Display Page setup dialog
//------------------------------------------------------------ 
INT MyPrintPageSetup (HWND hWnd) {
	INT rc;

	rc = DialogBoxParam (hInst, "PrintPageSetupDlg", hWnd, 
	                     lpfnPageSetupProc, 0);
	return 0;
}	
//------------------------------------------------------------ 
// MyPrintCancel - Cancel Printing of a file.
//------------------------------------------------------------ 
INT MyPrintCancel (HWND hWnd, LPSTR lpFileName) {

//OutputDebugString ("MyPrintCancel\n");		
	fContinue = FALSE;
	return 0;
}
//------------------------------------------------------------ 
// MyPrintPrint - Print a file.
//------------------------------------------------------------ 
INT MyPrintPrint (HWND hWnd, LPSTR lpFileName, HFONT hFont, 
                  INT sTabs, PGDCBFUNC pGetData) {
	INT rc, sPage;
	HDC hdcPrn;
	UINT wLen;
	LONG lFileOffset;
	LPSTR lpBuff, lpData;
	HFONT hOldFont;
	LOGFONT lf;
	HGLOBAL hBuff;
	DOCINFO di;
	MYPRNSTRUCT mps;

//OutputDebugString ("Start of MyPrintPrint\n");		
	PrnYield();

	if (fContinue)
		return 0;
	fContinue = TRUE;
	
	// Store handle to status window.
	mps.hWndStat = hWnd;

	// Create printer DC using names from prev. Print setup call
	hdcPrn = GetPrnhDC (hWnd, &pd);
	if (hdcPrn == 0) {
		fContinue = FALSE;
		return PERR_BADPRNDC;
	}
	// Alloc temp buffer for print data
	hBuff = GlobalAlloc (GHND, 0x8000);
	lpBuff = GlobalLock (hBuff);
	if (!lpBuff) {
		DeleteDC(hdcPrn);
		fContinue = FALSE;
		return PERR_OUTOFMEMORY;
	}	
	// Reinitialize the DC.
	SetMapMode (hdcPrn, MM_TWIPS);

	// Compute the printable area, then correct for margin settings.
	mps.rect.right = MulDiv (GetDeviceCaps (hdcPrn, HORZRES), 1440,
	                         GetDeviceCaps (hdcPrn, LOGPIXELSX));

	mps.rect.bottom = MulDiv (GetDeviceCaps (hdcPrn, VERTRES), 1440,
	                          GetDeviceCaps (hdcPrn, LOGPIXELSY));
	// Add in margins provided by user.  Insure .1 inch margin.									  
	mps.rect.left = max ((INT)(mp.left/10), 144);
	mps.rect.top = max ((INT)(mp.top/10), 144);
	mps.rect.right -= max ((INT)(mp.right/10), 144);
	mps.rect.bottom -= max ((INT)(mp.bottom/10), 144);

	// Create printer font that relates to screen font.
	GetObject (hFont, sizeof (lf), &lf);
	rc = mp.sLines - 1;
	// If header or footer, shrink font to still fit in lines.
	if (lstrlen (mp.szHeader))
		rc += 3;
	if (lstrlen (mp.szFooter))
		rc += 3;
	lf.lfHeight = (mps.rect.bottom - mps.rect.top)/rc;
	lf.lfWidth = 0;
	lf.lfOutPrecision = OUT_TT_ONLY_PRECIS;   //Force truetype to insure sizing
	mps.hFont = CreateFontIndirect (&lf);
	if (mps.hFont == 0) {
		DeleteDC(hdcPrn);
		GlobalUnlock (hBuff);
		GlobalFree (hBuff);
		fContinue = FALSE;
		return PERR_NOFONT;
	}  
	sPage = 1;
	rc = 0;
	lFileOffset = 0;
	mps.sTabs = sTabs;
	mps.sLines = mp.sLines;
	
	for (lpData = lpFileName + lstrlen (lpFileName); 
	     (lpData > lpFileName) && (*lpData != '\\'); 
		  lpData--);
	if (*lpData == '\\')
		lpData++;
	mps.lpName = lpData;	
	mps.fPrint = TRUE;
	hOldFont = SelectObject (hdcPrn, mps.hFont);

	SetAbortProc (hdcPrn, lpfnAbortProc);

	// Init docinfo struct
	di.cbSize = sizeof (di);
	di.lpszDocName = mps.lpName;
	di.lpszOutput = 0;

//OutputDebugString ("StartDoc start\n");		
	
	if (StartDoc (hdcPrn, &di) > 0) {
//OutputDebugString ("StartDoc end good\n");		
		while ((rc == 0) && fContinue) {
			// Yield to the other apps.
			if (!PrnYield())
				fContinue = FALSE;
			//
			// Reinitialize the DC.
			//
			SetTextColor (hdcPrn, RGB (0, 0, 0));
			SetTextAlign (hdcPrn, TA_NOUPDATECP);
			SetMapMode (hdcPrn, MM_TWIPS);
			SetBkMode (hdcPrn, TRANSPARENT);
			SelectObject (hdcPrn, mps.hFont);

			//
			// Read text from the file into the buffer.
			//
			wLen = (*pGetData)(hWnd, lFileOffset, lpBuff, 0x8000);
			lpData = lpBuff;
			if (wLen == 0) {
				rc = PERR_EOF;
			} else {
				//
				// Print a page
				//
				MakeHeadStr (mp.szHeader, mps.szHText, sizeof (mps.szHText),
				             mps.lpName, sPage);
				MakeHeadStr (mp.szFooter, mps.szFText, sizeof (mps.szFText),
				             mps.lpName, sPage);
				rc = PrintPage (hdcPrn, &mps, &lpData, wLen); 
				lFileOffset += lpData - lpBuff;
				sPage++;
//if (rc) {				
//wsprintf (szDebug, "Print File Error: %d,\n", rc);
//OutputDebugString (szDebug);
//}
			}	
		}
		if (!fContinue && !rc)
			rc = PERR_CANCELED;
			
		if (rc == PERR_EOF)
			rc = 0;
		if (rc)
			AbortDoc (hdcPrn);
		else {	
//OutputDebugString ("EndDoc Start\n");		
			sPage = EndDoc (hdcPrn);
//wsprintf (szDebug ,"EndDoc End rc = %d\n", sPage);		
//OutputDebugString (szDebug);		
		}
	} else
		rc = PERR_BADPRINT;
	SelectObject (hdcPrn, hOldFont);
	DeleteObject (mps.hFont);
	DeleteDC(hdcPrn);
	GlobalUnlock (hBuff);
	GlobalFree (hBuff);
	fContinue = FALSE;
//OutputDebugString ("End of MyPrintPrint\n");		
	return rc;
}	
//------------------------------------------------------------ 
//------------------------------------------------------------ 
INT DrawRect (HDC hdc, RECT *rect) {
	HPEN hPen, hOldPen;
	
	hPen = CreatePen (PS_SOLID, 1, RGB (0, 0, 0));	
	hOldPen = SelectObject (hdc, hPen);
	MoveTo (hdc, rect->left, rect->top);
	LineTo (hdc, rect->right, rect->top);
	LineTo (hdc, rect->right, rect->bottom);
	LineTo (hdc, rect->left, rect->bottom);
	LineTo (hdc, rect->left, rect->top);
	SelectObject (hdc, hOldPen);
	DeleteObject (hPen);
	return 0;
}	
//------------------------------------------------------------ 
// PrintPage - Prints one page to the printer
//------------------------------------------------------------ 
INT PrintPage (HDC hdcPrn, PMYPRNSTRUCT pmps, LPSTR *lpStart, UINT wLen) {
	DWORD dwDim;
	LPSTR lpIn, lpLine;
	INT i, rc, x, y, NewX, NewY, dy, sTabLen, sCharCnt, sChk, sLCnt;
	RECT rect;
	BOOL fPrintText;
	TEXTMETRIC tm;

	// Init num chars before width check
	GetTextMetrics (hdcPrn, &tm);
	sChk = (pmps->rect.right - pmps->rect.left) / tm.tmMaxCharWidth;

	// Init line height and tab width
	x = pmps->rect.left;
	y = -pmps->rect.top;
	dwDim = GetTextExtent (hdcPrn, "                          ", pmps->sTabs);
	dy = -(INT)HIWORD (dwDim);
	sTabLen = (INT)LOWORD (dwDim);
	rc = 0;
	sCharCnt = 0;
	fPrintText = FALSE;

	if (pmps->fPrint) 
		if (StartPage (hdcPrn) <= 0)
			return PERR_BADPAGE;

//rect.left = pmps->rect.left;	
//rect.top = -pmps->rect.top;
//rect.right = pmps->rect.right;
//rect.bottom = -pmps->rect.bottom;
//DrawRect (hdcPrn, &rect);

	// Print header if needed
	if (*(LPLONG)pmps->szHText) {
		rect.left = pmps->rect.left;	
		rect.top = -pmps->rect.top + dy;
		rect.right = pmps->rect.right;
		rect.bottom = -pmps->rect.top;
		HeaderOut (hdcPrn, &rect, pmps->szHText);
		y += dy * 3;
	}
	SetTextAlign (hdcPrn, TA_LEFT);
	// Init vars
	lpIn = *lpStart;
	lpLine = lpIn;
	NewX = x;
	NewY = y;
	sLCnt = 0;

	// Print character loop	
	while ((rc == 0) && wLen && fContinue && (sLCnt < pmps->sLines) && 
	                                         (y > -pmps->rect.bottom)) {
		// Yield to the other apps.
		if (!PrnYield())
			fContinue = FALSE;
			
		// Get character and process
		wLen--;
		switch (*lpIn++) {
			case 0x0a:              //Line Feed
				fPrintText = TRUE;
				dwDim = GetTextExtent (hdcPrn, lpLine, sCharCnt);
				NewX = x + LOWORD (dwDim);
				NewY = y + dy;
				sLCnt++;
				break;

			case 0x0d:              //Carriage Return
				fPrintText = TRUE;
				NewX = pmps->rect.left;
				NewY = y;
				break;

			case 0x09:              //Tab
				fPrintText = TRUE;
				dwDim = GetTextExtent (hdcPrn, lpLine, sCharCnt);
				// Compute tab stop
				i = (x + LOWORD (dwDim) - rect.left) / sTabLen;
				// Compute location of next stop
				NewX = rect.left + (i + 1) * sTabLen;
				NewY = y;
				break;

			case 0x0c:              //Form Feed
//OutputDebugString ("Form Feed\n");
				wLen = 0;
				break;

			case 0x1a:              //End of File
//OutputDebugString ("End of file\n");
				fPrintText = TRUE;
				rc = PERR_EOF;
				break;

			default:                //All other characters
				sCharCnt++;
				if (sCharCnt > sChk) {
					dwDim = GetTextExtent (hdcPrn, lpLine, sCharCnt);

					if ((x + (INT)LOWORD (dwDim)) >= pmps->rect.right) {
						// Find last word break
						while ((lpIn > lpLine) && *--lpIn > ' ')
							wLen++;
						// See if entire line
						if ((lpIn != lpLine) || (x != pmps->rect.left)) {
							lpIn++;      // Put break on end of curr line
							sCharCnt = lpIn - lpLine;
						} else {        //If one word line, print anyway.
							wLen -= sCharCnt;
							lpIn += sCharCnt;
						}	
						fPrintText = TRUE;
						NewX = pmps->rect.left;
						NewY = y + dy;
						sLCnt++;
					} else             //Reset char count until next margin check
						sChk = sCharCnt + (pmps->rect.right-(x + (INT)LOWORD (dwDim))) / 
						                   tm.tmMaxCharWidth;
				}	
				break;
		}
		if ((fPrintText) || (wLen == 0)) {
			// Print text string
			if ((pmps->fPrint) && sCharCnt) {
				TextOut (hdcPrn, x, y, lpLine, sCharCnt);

//wsprintf (szDebug, "len: %d cnt: %d x: %d y:%d -->", wLen, sCharCnt, x, y);
//OutputDebugString (szDebug);
//lstrcpyn (szDebug, lpLine, min (sCharCnt+1, sizeof (szDebug) - 2));
//OutputDebugString (szDebug);
//OutputDebugString ("<--\n\n");
			
			} 
			// Reset text ptr
			lpLine = lpIn;	
			sCharCnt = 0;
			x = NewX;
			y = NewY;
			fPrintText = FALSE;		
			sChk = (pmps->rect.right - x) / tm.tmMaxCharWidth;
		}
	}			
//wsprintf (szDebug, "Page done. len: %d x: %d y:%d \n", wLen, x, y);
//OutputDebugString (szDebug);
	// Print footer if needed
	if (*(LPLONG)pmps->szFText) {
		rect.left = pmps->rect.left;	
//		rect.top = -pmps->rect.bottom + (3 * dy);
		rect.top = -pmps->rect.bottom;
		rect.right = pmps->rect.right;
//		rect.bottom = -pmps->rect.bottom + (2 * dy);
		rect.bottom = -pmps->rect.bottom - dy;
		HeaderOut (hdcPrn, &rect, pmps->szFText);
	}
	// Page cleanup	
//OutputDebugString ("EndPage Start\n");		
	if (pmps->fPrint) {
		x = EndPage (hdcPrn);
//wsprintf (szDebug ,"EndPage End rc = %d\n", x);		
//OutputDebugString (szDebug);		
		if (x < 0) {
//			OutputDebugString ("Printer Error\n");
			switch (x) {
				case SP_APPABORT:
				case SP_USERABORT:
					rc = PERR_CANCELED;  
					break;
				case SP_OUTOFDISK:
					rc = PERR_NODISK;    
					break;
				case SP_OUTOFMEMORY:
					rc = PERR_OUTOFMEMORY;
					break;
				default:	
					rc = PERR_BADPAGE;
					break;
			}		
		} 
	}
	*lpStart = lpIn;
	return rc;
}	
//------------------------------------------------------------ 
// MyPrintDlg - Returns the device contex for the printer
//------------------------------------------------------------ 
HDC MyPrintDlg (HWND hWnd, PRINTDLG *ppd, DWORD fFlags) {

	// Init common dlg print structure
	ppd->Flags = fFlags;
	// Free print structures if using the default printer
	if (fFlags & PD_RETURNDEFAULT) {
		if (ppd->hDevMode != NULL)
			GlobalFree(ppd->hDevMode);
		if (ppd->hDevNames != NULL)
			GlobalFree(ppd->hDevNames);
		ppd->hDevMode = 0;
		ppd->hDevNames = 0;
	}		
	ppd->lStructSize = sizeof(PRINTDLG);
	ppd->hwndOwner = hWnd;
	if (PrintDlg(ppd) == 0)
		return 0;

	return ppd->hDC;
}	
//------------------------------------------------------------ 
// GetPrnhDC - Returns the device contex for the printer
//------------------------------------------------------------ 
HDC GetPrnhDC (HWND hWnd, PRINTDLG *ppd) {
	LPDEVNAMES lpdn;
	LPDEVMODE lpdm;
	LPSTR lpDriver, lpDevice, lpOutput;
	HDC hdc;
	
	lpdn = (LPDEVNAMES)GlobalLock (ppd->hDevNames);
	lpdm = (LPDEVMODE)GlobalLock (ppd->hDevMode);

	// If using the default printer, check for current default
	if (lpdn->wDefault & DN_DEFAULTPRN) {
		GlobalUnlock (ppd->hDevNames);
		return MyPrintDlg (hWnd, ppd, PD_RETURNDC | PD_RETURNDEFAULT);
	} 
	// Else, use old info to create printer dc
	lpDriver = (LPSTR)lpdn + lpdn->wDriverOffset;
	lpDevice = (LPSTR)lpdn + lpdn->wDeviceOffset;
	lpOutput = (LPSTR)lpdn + lpdn->wOutputOffset;
	hdc = CreateDC (lpDriver, lpDevice, lpOutput, lpdm);
	
	GlobalUnlock (ppd->hDevNames);
	GlobalUnlock (ppd->hDevMode);
	return hdc;
}	
//------------------------------------------------------------ 
// MakeHeadStr - Returns a pointer to 3 asciiz strings that
// are to be aligned left, mid and right.
//------------------------------------------------------------ 
LPSTR MakeHeadStr (LPSTR lpIn, LPSTR lpOut, INT sLen, 
                   LPSTR lpFileName, INT sPage) {
	LPSTR lpOStart;
	char szAdd[64];
	char szLeft[80], szMid[80], szRight[80];
	char *pszActive;
	LONG lTime;
	struct tm *now;
	INT i;

	lpOStart = lpOut;
	szLeft[0] = '\0';
	szMid[0] = '\0';
	szRight[0] = '\0';
	pszActive = szMid;

	// Generate time and date strings
	time (&lTime);
	now = localtime (&lTime);

	// Scan and convert input string	
	while (*lpIn) {
		szAdd[0] = '\0';
		if (*lpIn == '&') {
			lpIn++;
			switch (*lpIn) {
				case 'l':
				case 'L':
					pszActive = szLeft;
					break;
					
				case 'r':
				case 'R':
					pszActive = szRight;
					break;
					
				case 'c':
				case 'C':
					pszActive = szMid;
					break;
					
				case 'd':
				case 'D':
					// Create date string
					if (now)
						wsprintf (szAdd, "%d/%d/%d", now->tm_mon+1, 
						          now->tm_mday, now->tm_year);
					break;
					
				case 't':
				case 'T':
					// Create time string
					if (now) {
						if (now->tm_hour < 12) {
							if (now->tm_hour == 0)
								now->tm_hour = 12;
							wsprintf (szAdd, "%d:%02d AM",now->tm_hour, now->tm_min);
						} else {
							if (now->tm_hour > 12)
								now->tm_hour -= 12;
							wsprintf (szAdd, "%d:%02d PM",now->tm_hour, now->tm_min);
						}
					}
					break;
					
				case 'p':
				case 'P':
					wsprintf (szAdd, "%d", sPage);
					break;
					
				case 'f':
				case 'F':
					lstrcpyn (szAdd, lpFileName, sizeof (szAdd));
					break;

				default:
					lpIn--;
					szAdd[0] = '&';
					szAdd[1] = '\0';
					break;
			}
		} else {
			szAdd[0] = *lpIn;
			szAdd[1] = '\0';
		}	
		lpIn++;
		if (szAdd[0]) {
			i = lstrlen (pszActive);
			if ((i + lstrlen (szAdd)) < 79)
				lstrcat (pszActive, szAdd);
		}
	}
	// Make sure sLen has room for three term zeros
	if (sLen < 4)
		return 0;
	sLen -= 3;
	*(LPLONG)lpOut = 0;	
	// Add left string
	i = lstrlen (szLeft);
	if (i && (i < sLen)) {
		lstrcpy (lpOut, szLeft);
		lpOut += i+1;
		sLen -= i+1;
	} else {
		*lpOut++ = '\0';	
		sLen--;
	}	
	// Add mid string
	i = lstrlen (szMid);
	if (i && (i < sLen)) {
		lstrcpy (lpOut, szMid);
		lpOut += i+1;
		sLen -= i+1;
	} else {
		*lpOut++ = '\0';	
		sLen--;
	}	
	// Add right string
	i = lstrlen (szRight);
	if (i && (i < sLen)) 
		lstrcpy (lpOut, szRight);
	else
		*lpOut = '\0';	
	return lpOStart;
}	
//------------------------------------------------------------ 
// HeaderOut - Prints the header and footer strings
//------------------------------------------------------------ 
INT HeaderOut (HDC hdcPrn, RECT *rect, LPSTR lpLine) {
	DWORD dwDim;
	INT sLen;
//OutputDebugString ("Printing header/footer\n");			 
	
	sLen = lstrlen (lpLine);
	if (*lpLine) {
		dwDim = GetTextExtent (hdcPrn, lpLine, sLen);
		TextOut (hdcPrn, rect->left, rect->bottom, lpLine, sLen);

//		OutputDebugString ("Left: ");
//		OutputDebugString (lpLine);
//		OutputDebugString ("\n");
	}
	lpLine += sLen + 1;
	sLen = lstrlen (lpLine);
	if (*lpLine) {
		dwDim = GetTextExtent (hdcPrn, lpLine, sLen);
		TextOut (hdcPrn, rect->left + (rect->right - rect->left)/2 - LOWORD(dwDim)/2, 
		rect->bottom, lpLine, sLen);

//		OutputDebugString ("Mid: ");
//		OutputDebugString (lpLine);
//		OutputDebugString ("\n");
	}
	lpLine += sLen + 1;
	sLen = lstrlen (lpLine);
	if (*lpLine) {
		dwDim = GetTextExtent (hdcPrn, lpLine, sLen);
		TextOut (hdcPrn, rect->right - LOWORD(dwDim), rect->bottom, 
		         lpLine, sLen);

//		OutputDebugString ("Right: ");
//		OutputDebugString (lpLine);
//		OutputDebugString ("\n");
	}
//wsprintf (szDebug, "header/footer: left %d top %d right %d bottom %d\n",
//          rect->left, rect->top, rect->right, rect->bottom);
//OutputDebugString (szDebug);			 

//DrawRect (hdcPrn, rect);

	return 0;
}
//------------------------------------------------------------
// SetMarginBox - Formats a margin value for an edit box
//------------------------------------------------------------
INT SetMarginBox (HWND hWnd, INT sID, LONG lVal) {
	char szText[64];
	LONG d, h;

	d = lVal / 14400;
	h = (lVal - (d * 14400)) / 144;
			
	wsprintf (szText, "%ld.%02ld", d, h);
	SetDlgItemText (hWnd, sID, szText);
	return 0;
}
//------------------------------------------------------------
// GetMarginBox - Unformats a margin value      
//------------------------------------------------------------
INT GetMarginBox (HWND hWnd, INT sID, LONG *plVal, LONG lMax) {
	char szText[64];
	INT i, sLen, h;
	LONG d, ht;

	sLen = GetDlgItemText (hWnd, sID, szText, sizeof (szText));
	d = 0;
	h = -1;
	ht = 0;
	for (i = 0; i < sLen; i++) {
		if (szText[i] == '.') {
			if (h != -1)
				return 1;
			h = 100;
		} else if ((szText[i] >= '0') && (szText[i] <= '9')) {
			if (h == -1)
				d = (d * 10) + szText[i] - '0';
			else {
				h /= 10;
				ht = ht + (h * (szText[i] - '0'));
			}	
		} else if (szText[i] == '"') {
			if (szText[i+1] != '\0')
				return 1;
		} else		
			return 1;
	}
	*plVal = d * 14400 + (INT)(((LONG)ht * 144));
	if (*plVal > lMax)
		return 1;
	return 0;
}
//============================================================  
// PrintPageSetupDlgProc - Page Setup Dialog proc
//============================================================  
BOOL CALLBACK PrintPageSetupDlgProc (HWND hWnd, UINT wMsg, 
                                     UINT wParam, LONG lParam) {
	INT rc;
	LONG cx, cy;
	HDC hdcPrn;
	
	switch (wMsg) {                              
		case WM_INITDIALOG:
			SetDlgItemText (hWnd, IDD_HEADER, mp.szHeader);
			SetDlgItemText (hWnd, IDD_FOOTER, mp.szFooter);

			SetMarginBox (hWnd, IDD_MLEFT, mp.left);		
			SetMarginBox (hWnd, IDD_MTOP, mp.top);		
			SetMarginBox (hWnd, IDD_MRIGHT, mp.right);		
			SetMarginBox (hWnd, IDD_MBOTTOM, mp.bottom);
			SetDlgItemInt (hWnd, IDD_LINES, mp.sLines, FALSE);
			return TRUE;
			
		case WM_COMMAND:
			switch (wParam) {
				case IDD_PRNINFO:
					MessageBox (hWnd, "Valid header and footer macros:\n\n"\
					                  "&l	Left align following text.\n"\
					                  "&c	Center following text.\n"\
					                  "&r	Right align following text.\n\n"\
					                  "&d	Insert current date.\n"\
					                  "&f	Insert file name.\n"\
					                  "&p	Insert page number.\n"\
					                  "&t	Insert current time.\n\n"\
					                  "Margins are computed from inside the printer's\n"\
					                  "printable area.  Values are in inches.", 
					                  "Page Setup", MB_OK);
					return TRUE;						
					
				case IDOK:
					GetDlgItemText (hWnd, IDD_HEADER, mp.szHeader, sizeof (mp.szHeader));
					GetDlgItemText (hWnd, IDD_FOOTER, mp.szFooter, sizeof (mp.szFooter));
					mp.sLines = (INT)GetDlgItemInt (hWnd, IDD_LINES, &rc, FALSE);
					if ((!rc) || (mp.sLines < 1)) {
						MessageBox (hWnd, "The Lines field does not contain a"\
						            " valid number.", "Error", MB_ICONSTOP | MB_OK);
						return TRUE;
					}

					// Create printer DC using names from prev. Print setup call
					hdcPrn = GetPrnhDC (hWnd, &pd);
					if (hdcPrn) {
						// Compute the printable area, then correct for margin settings.
						cx = MulDiv (GetDeviceCaps (hdcPrn, HORZRES), 1440,
	   	                      GetDeviceCaps (hdcPrn, LOGPIXELSX));
						cy = MulDiv (GetDeviceCaps (hdcPrn, VERTRES), 1440,
	            	              GetDeviceCaps (hdcPrn, LOGPIXELSY));
						DeleteDC(hdcPrn);
					} else {
						cx = 4 * 1440;
						cy = 5 * 1440;
					}
					rc = GetMarginBox (hWnd, IDD_MLEFT, &mp.left, cx * 10);
					if (!rc)
						rc = GetMarginBox (hWnd, IDD_MTOP, &mp.top, cy * 10);
					if (!rc)
						rc = GetMarginBox (hWnd, IDD_MRIGHT, &mp.right, cx * 10 - mp.left);
					if (!rc)
						rc = GetMarginBox (hWnd, IDD_MBOTTOM, &mp.bottom, cy * 10 - mp.top);
					if (rc) {
						MessageBox (hWnd, "The margin values are not valid or are larger"\
						            " than the printed region.", "Error", MB_ICONSTOP | MB_OK);
						return TRUE;
					}
					EndDialog (hWnd, 1);
					return TRUE;

				case IDCANCEL:
					EndDialog (hWnd, 0);
					return TRUE;
			}
			break;
	}
	return FALSE;
}	

