/* TABX.C - tab expansion utility */
/* Written by  Baruch Nissenbaum  - Tel-Aviv University, Israel 2/1990  (c)  */
/* E-mail: BARUCH@TAUNIVM.BITNET (BARUCH@TAUNIVM.TAU.AC.IL) or BARUCH@TAUENG */
/* Comments, Suggestions and Ideas are welcomed. */
/* Pleas do not modify the above note or the 'FREE DISTRIBUTION ONLY' note */

/* This program should work for a file of any size, and any line length.  */
/* Compiled and tested with TURBO-C, should work with any other compiler. */
/* See usage() for details of operation. */

/* Options are linked in a kind of link list manner, where the list is
   made up from functions calling each other via function pointers.
   This unique arrangement makes the program very modular, Anyone can
   add new filters very easily following the example below. Note, however,
   that the price is execution time. Each option adds one function call for
   each character in the output file. */


#include <stdio.h>
#include <ctype.h>
#include <string.h>

typedef (filter)(int);  /* a filter function */
typedef filter *fptr;   /* a pointer to a filter function */

/* each filter function has its corresponding output variable */
int     fputchar(int);
int     no_trail(int);         /* a function */
fptr    put_tr = &fputchar;    /* its output pointer */
int     conv_up(int);          /* etc. ... */
fptr    put_up = &fputchar;
int     conv_low(int);
fptr    put_low = &fputchar;
int     capit(int);
fptr    put_cap = &fputchar;
int     strip(int);
fptr    put_nud = &fputchar;
int     number(int);
fptr    put_num;

fptr put = &fputchar; /* output function called from main function */

/* output filter chaining */
typedef struct {
   fptr funct;     /* the filter function itself */
   char flag;      /* flag that activates this function */
   fptr *chain;    /* chain to the next filer */
} filter_data;

filter_data  filters[]={  /* table of function and activating flags */
  { no_trail, 'e', &put_tr },
  { conv_up,  'u', &put_up },
  { conv_low, 'l', &put_low },
  { capit,    'c', &put_cap },
  { strip,    's', &put_cap },
  { number,   'n', &put_num },
  { fputchar, '\0', NULL }}; /* terminator */

int insert_filter(filter_data *f)
/* insert a filter before the current filter list */
{
   if(f->chain==NULL || f->flag=='\0')
      return(1); /* error */
   *(f->chain)=put; /* set the output */
   put= f->funct;   /* link the input to function f */
   return(0); /* ok */
}

usage()
{
   fprintf(stderr,"      TABX  - tab expansion utility\n");
   fprintf(stderr,"      This program expands tab characters to spaces.\n");
   fprintf(stderr,"usage:   TABX  [-flags] [tab_spec]  < source_file  > dest_file\n");
   fprintf(stderr,"      tab_spec can be an integer for setting fixed tab spacing (default is 8),\n");
   fprintf(stderr,"      tab_spec can also be a string as in:\n");
   fprintf(stderr,"        TABX ---T---T-T---T  < in_file > out_file\n");
   fprintf(stderr,"      In this case tab stops will match the position of the 'T' characters,\n");
   fprintf(stderr,"      in the above example - tabs will be set to 4 8 10 and 14\n");
   fprintf(stderr,"      There are no limits on file size or line length\n\n");
   fprintf(stderr,"    Flags:\n");
   fprintf(stderr,"       -e  Blank character are remover from END of lines.\n");
   fprintf(stderr,"       -n  insert line numbers to the beginning of lines.\n");
   fprintf(stderr,"       -u  Convert all character to upper case.\n");
   fprintf(stderr,"       -l  Convert all character to lower.\n");
   fprintf(stderr,"       -c  Capitalize words in the file.\n");
   fprintf(stderr,"       -s  strip bit 8 from all characters.\n");
   fprintf(stderr,"    Any combination of options and tab specs may be specified at any order\n\n");
   fprintf(stderr,"  Written by Baruch Nissenbaum,  Israel,  27 Feb 90 (c)\n");
   fprintf(stderr,"  E-Mail: BARUCH@TAUNIVM.BITNET (TAUNIVM.TAU.AC.IL)\n\n");
   fprintf(stderr,"  Not for sale,  FREE distribution only!!\n\n");
}

fputchar( int c )
/* put char function */
{
   putchar( c );
}

no_trail( int c )
/* print a character but filter out trailing spaces */
{
   static state=0;
   static int n;

   switch( state ) {
   case 0:
      if( isspace(c) )
         state = n=1;
      else
         putchar( c );
      break;
   case 1:
      if(c=='\n') {
         putchar(c);
         state=0;
      }
      else if( isspace(c) )
         n++;
      else {
         for(;n;n--)
            putchar(' ');
         putchar( c );
         state=0;
      }
      break;
   }
}

main(argc,argv)
int argc;
char **argv;
{
   int c,col=0,tab=8,dlen=0,k,i,j;
   char *d;

   for(k=1;k<argc;k++) { /* parse arguments */
      if(strchr(argv[k],'T') != NULL || strchr(argv[k],'t') )
         dlen=strlen(d=argv[k]);
      else if(argv[k][0]=='-') {  /* chain filters according to flags */
         for(j=strlen(argv[k])-1; j>0; j--)
            for(i=0,c=tolower(argv[k][j]); filters[i].flag!='\0'; i++)
               if( filters[i].flag==c)
                  insert_filter( &filters[i] );
      }
      else
         tab=atoi(argv[k]);
      if(tab<1) {
         usage();
         exit(0);
      }
   }
   if(tab<1) {
      usage();
      exit(0);
   }

   while((c=getchar()) != EOF)
      if(c=='\t') {              /* expand tabs */
         put(' ');  col++;   /* tab will generate at least one space */
         for(; col<dlen && d[col]!='t' && d[col]!='T'; col++)
            put(' ');        /* expansion of tabs (tab string) */
         if( col>=dlen )
            for(; col%tab; col++)
               put(' ');     /* expansion of tabs (fixed spacing) */
      }
      else if(c=='\n' || c=='\r') {
         put(c);
         col=0;
     }
     else {                      /* any other character */
        put(c);
        col++;
     }
}

int conv_up( int c )
/* convert a character to upper case */
{
   return( put_up( toupper(c)));
}

int conv_low( int c )
/* convert characters to lower case */
{
   return( put_low( tolower(c)));
}

int capit( int c )
/* capitalize sentences */
{
   static state=0;  /* start of a sentence */

   if(c=='.' || c==':' || c==';')  /* sentence terminators */
      state=0;
   else if(state==0 && !isspace(c)) {
      if(islower(c))
         c=toupper(c); /* capitalize */
      state=1;   /* inside a sentence */
   }
   return( put_cap(c) );
}

int strip( int c )
/* strip bit 8 */
{
   return( put_nud(c & 0x7f) );
}

int number( int c )
/* add line numbers to beginning of lines */
{
   static long line_no=1;
   static new_line=1;
   char buff[20];
   int i;


   if(new_line && c!=EOF) {
      sprintf(buff,"%4ld ",line_no++);
      for(i=0; buff[i]!='\0'; i++)
         put_num(buff[i]);
      new_line=0;
   }
   if(c=='\n')
      new_line=1;
   return( put_num(c) );
}
