/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/


/*$ab ----------------------------- abstract ----------------------------------
  pilgen.c   	Contains routines to generate the Publisher's Interchange
  				Language from the data structures defined in pilstruc.h.
  				
  				NO MEMORY IS ALLOCATED OR FREED BY THE FUNCTIONS IN THIS MODULE!
  				
  				NO I/O (FILE OPENS, CLOSES, READS, OR WRITES, ERROR PRINTING)
  				IS DONE DIRECTLY BY THE FUNCTIONS IN THIS MODULE.

  				ERRORS ARE COMMUNICATED TO THE APPLICATION BY RETURNING AN
  				ERROR CODE AND SETTING A GLOBAL ERROR WORD WITH THAT ERROR CODE. 
  				
  				The functions in this module are called by pil_put_component
  				to generate pil "code" from the pil data structures. The
  				functions here call pil_put_string, in pilapi.c, to write to
  				a file opened by the application that called pil_put_component.
  				
  				
    Contains:
		pil_nextline					End the current line and indent the next		
		pil_put_asa						Write an application specific attribute
		pil_put_asi						Write an application specific item
		pil_put_canvas					Write a canvas
		pil_put_clipper					Write a clipping shape for canvas, object
		pil_put_content_data			Write content for a pil-content entity
		pil_put_content_end				End a pil-content entity
		pil_put_content_hdr				Write the header for a pil-content entity
		pil_put_content_start			Write the start of a pil-content entity
		pil_put_data_code				Write data type codes for app. spec. items
		pil_put_group					Write a group
		pil_put_kw						Translate a token to a keyword & write it
		pil_put_layout_end				End a layout entity, write it
		pil_put_layout_start			Write the start of a layout entity
		pil_put_nametbleentry			Write a single entry for a nametable
		pil_put_name_table				Write the whole nametable
		pil_put_nested_begblk			Write a begin block & adjust indents
		pil_put_nested_endblk			Write an end block & adjust indents
		pil_put_object					Write a pil object
		pil_put_objid					Write the object id
		pil_put_pathpt					Write a single point for a path
		pil_put_point					Write a single pil_point, e.g. origin
		pil_put_polydata				Write all points for polygon or polyline
		pil_put_polypt					Write single point for polygon or polyline
		pil_put_pvalue					Write a single value for app. spec. items
		pil_put_rectdata				Write the origin, width, height for rectangle
		pil_put_string					Write a string to the open pil file
		pil_put_text_flow				Write a text flow component
		pil_set_dflt_hdr				Set up the default header for pil content

    Calls:		pil_set_error			in pilapi.c



---------------------------------------------------------------------------- */


/*$au ------------------------------ audits -----------------------------------
    Author: Tom Arnold
            Quark, Inc.
            300 South Jackson
            Denver, Colorado 80209
            (303) 934-2211

    Ver     Date       Who  Etc
    v01.02  24-apr-91  tla	Updated for PIL 5.0
    v01.01  03-apr-91  tla	Fixed prototype for pil_put_color, and minor bug.
    	Fixed bug in pil_put_pathdata that treated arc radius as unsigned int.
    v01.00  12-feb-91  tla	Initial version
---------------------------------------------------------------------------- */

/*$ep --------------------------- header files ----------------------------- */

#include "pildefs.h"	/* everything */
#ifdef THINK_DA
#include <SANE.h>		/* num2str (Apple's SANE floating point to ascii) */
#else
#include <stdio.h>		/* for sprintf() */
#endif

/*$ed ------------------------ public global data -------------------------- */

#ifdef PIL_ANSI_PROTOTYPES
extern int (FPTRTYPE *pil_putchar)(PIL_UINT8);	/* pointer to Appl PutChar function */
#else
extern int (FPTRTYPE *pil_putchar)();	/* pointer to Appl PutChar function */
#endif
extern char PTRTYPE *pil_kws[];			/* Keyword pointer array			*/

/*$ld --------------------------- static  data ----------------------------- */

static char hex[16] = {'0','1','2','3','4','5','6','7',
					   '8','9','A','B','C','D','E','F'};
					   
static char outbuf[80];				/* used by pil_put_ui32, etc */
static int indentlev = 0;			/* current level of indentation */

static 	pil_content_hdr currenthdr = {
	 (PIL_INT32)PIL_CONTENT_HDR_C,	/* component type */
	 (PIL_INT8)PIL_ASCII_CONTENT,	/* encoding */
	 (PIL_INT8)PIL_BIG_ENDIAN,		/* byte order(big-endian is like Motorola)*/
	 (PIL_UINT8)PIL_DFT_CNT_ESC,	/* escape character */
	 (PIL_UINT8)PIL_DFT_CNT_BEG,	/* begin block character */
	 (PIL_UINT8)PIL_DFT_CNT_END		/* end block character */
};


/*$co ---------------------------- code start ------------------------------ */
		
/*=======================================================================*\
  pil_put_component

	Writes a pil component to the open pil file.
	APPLICATION MUST OPEN THE FILE FIRST, AND CALL pil_set_putchar
 
	Entry:	Pointer to the pil component

	Exit:	Error code

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_component(ptr)
FAST	pil_component PTRTYPE *ptr;
{
	/* make sure there is a putchar function to call ! */
	if (pil_putchar == NULL) {
		return(pil_set_error(PIL_NO_W_FUNC));
	}
	if (ptr->type == PIL_CANVAS_C) {
		return (pil_put_canvas((pil_canvas PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_NAME_TABLE_C) {
		return (pil_put_name_table((pil_name_table PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_OBJECT_C) {
		return (pil_put_object((pil_object PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_TEXT_FLOW_C) {
		return (pil_put_text_flow((pil_text_flow PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_GROUP_C) {
		return (pil_put_group((pil_group PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_LAYOUT_START_C) {
		return (pil_put_layout_start((pil_layout_start PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_LAYOUT_END_C) {
		return (pil_put_layout_end((pil_layout_end PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_CONTENT_START_C) {
		return (pil_put_content_start((pil_content_start PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_CONTENT_HDR_C) {
		return (pil_put_content_hdr((pil_content_hdr PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_CONTENT_DATA_C) {
		return (pil_put_content_data((pil_content_data PTRTYPE *)ptr));
	}
	if (ptr->type == PIL_CONTENT_END_C) {
		return (pil_put_content_end((pil_content_end PTRTYPE *)ptr));
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_comment

	Writes a comment string to the open pil file.
	APPLICATION MUST OPEN THE FILE FIRST.
	This function does not break the comment into lines, it just writes
	out whatever it is passed.
	To select C or C++ commenting conventions, choos the appropriate
	definitions of PIL_BEGIN_COMMENT and PIL_END_COMMENT in pil_struct.h
	
	CAUTION! A legal PIL file must start with an entity, not a comment!
	Therefore do not call pil_put_comment until after you've written a 
	pil_layout_start or a pil_content_start to the file.

	Entry:	Pointer to the string.

	Exit:	No return value. Writes comment via pil_put_string

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_comment(ptr)
char PTRTYPE *ptr;
{
	pil_put_string(PIL_BEGIN_COMMENT);
	pil_put_string(ptr);
	pil_put_string(PIL_END_COMMENT);
	pil_nextline();
}

/*=======================================================================*\
  pil_put_string

	Writes a c string to the open pil file.
	APPLICATION MUST OPEN THE FILE FIRST, AND CALL pil_set_putchar, or 
	provide a putchar function by calling pil_pg_init.
	
	NOTE: This function, and all functions that call pil_putchar, assume
	that all writes succeed. Therefore the write function itself must have
	some capability of detecting and correcting errors.
	
	Entry:	Pointer to the string

	Exit:	None.

\*-----------------------------------------------------------------------*/

PIL_VOID pil_put_string(str)
char PTRTYPE *str;
{
	while (*str) {
		(pil_putchar)(*str++);
	}
}

/*=======================================================================*\
  pil_put_layout_start

	Subroutine for pil_put_component.
	Writes a pil_layout_start to the open pil file.
 
	Entry:	Pointer to a pil_layout_start  

	Exit:	Error code is returned

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_layout_start(lsptr)
pil_layout_start PTRTYPE *lsptr;
{
	if (!PIL_RIGHT_CMPNT(lsptr,PIL_LAYOUT_START_C))
		return (pil_set_error(PIL_BAD_CMPNT_TYPE));
	pil_put_kw_no_sep(PIL_KW_LAYOUT);
	pil_put_literal(lsptr->version);
	pil_put_quoted_str(lsptr->name);
	pil_put_nested_begblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_name_table

	Subroutine for pil_put_component.
	Writes a pil_name_table to the open pil file.
 
	Entry:	Pointer to a pil_name_table

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_name_table(nptr)
pil_name_table PTRTYPE *nptr;
{
	FAST pil_name_table_entry PTRTYPE *nte;
	
	pil_put_named_begblk(PIL_KW_NAME_TABLE);
	if ((nte = nptr->entries) != NULL) {
		pil_nextline();
		do {
			if (pil_put_nametbleentry(nte) != PIL_OK)
				return (pil_last_error());
			if ((nte = nte->next)!= NULL)
				pil_nextline();
		} while (nte != NULL);
	}
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_canvas

	Subroutine for pil_put_component.
	Writes a pil_canvas to the open pil file.
 
	Entry:	Pointer to a pil_canvas

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_canvas(cptr)
pil_canvas PTRTYPE *cptr;
{
	pil_put_named_begblk(PIL_KW_CANVAS);
	if (cptr->username) {
		pil_nextline();
		pil_put_named_str(PIL_KW_USER_NAME,cptr->username);
	}
	if (cptr->cantype) {
		pil_nextline();
		pil_put_named_str(PIL_KW_TYPE,cptr->cantype);
	}
	if (cptr->units <= 0) return (pil_set_error(PIL_BAD_UNITS));
	pil_nextline();
	pil_put_named_dbl(PIL_KW_UNITS,cptr->units);
	pil_nextline();
	if (pil_put_dimensions(cptr->width,cptr->height) != PIL_OK)
		return (pil_last_error());
	if (cptr->clipshape.type != PIL_NO_SHAPE) {
		pil_nextline();
		if (pil_put_clipper(&cptr->clipshape) != PIL_OK) 
			return (pil_last_error());
	}
	if (cptr->asi != NULL) {
		pil_nextline();
		if (pil_put_asi(cptr->asi) != PIL_OK) return (pil_last_error());
	}
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_object

	Subroutine for pil_put_component.
	Writes a pil_object to the open pil file.
 
	Entry:	Pointer to a pil_object

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_object(optr)
pil_object PTRTYPE *optr;
{	
	PIL_RC_TYPE rctype;
	
	pil_put_named_begblk(PIL_KW_OBJECT);
	pil_nextline();
	pil_put_kw(PIL_KW_ORIGIN);
	pil_put_point(&optr->dest_rect.ul);
	pil_nextline();
	if (pil_put_dimensions(optr->dest_rect.width, 
		optr->dest_rect.height) != PIL_OK)
		return (pil_last_error());
	if (optr->id) {
		pil_nextline();
		pil_put_named_str(PIL_KW_ID,optr->id);
	}
	if (optr->units > 0.0) {
		pil_nextline();
		pil_put_named_dbl(PIL_KW_UNITS,optr->units);
	}
	if (optr->username) {
		pil_nextline();
		pil_put_named_str(PIL_KW_USER_NAME,optr->username);
	}
	if (optr->objtype != PIL_OBJ_UNKNOWN) {
		pil_nextline();
		pil_put_kw(PIL_KW_TYPE);
		switch (optr->objtype) {
			case PIL_OBJ_TEXT:
				pil_put_kw(PIL_KW_OBJ_TEXT);
				break;
			case PIL_OBJ_PRIMITIVE:
				pil_put_kw(PIL_KW_OBJ_PRIMITIVE);
				break;
			case PIL_OBJ_PICTURE:
				pil_put_kw(PIL_KW_OBJ_PICTURE);
		}
	}
	if (optr->rotation_angle != 0.0) {
		pil_nextline();
		pil_put_named_dbl(PIL_KW_ROT_ANGLE,optr->rotation_angle);
		if (optr->rotation_point.x != 0 && optr->rotation_point.y != 0) {
			pil_nextline();
			pil_put_kw(PIL_KW_ROT_POINT);
			pil_put_point(&optr->rotation_point);
		}
	}
	if (optr->clipshape.type != PIL_NO_SHAPE) {
		pil_nextline();
		if (pil_put_clipper(&optr->clipshape) != PIL_OK)
			return (pil_last_error());
	}
	if (optr->container.type != PIL_NO_SHAPE) {
		pil_nextline();
		if (pil_put_container(&optr->container) != PIL_OK)
			return (pil_last_error());
	}
	if (optr->graphic.shape.type != PIL_NO_SHAPE) {
		pil_nextline();
		if (pil_put_graphic(&optr->graphic) != PIL_OK)
			return (pil_last_error());
	}
	if (optr->rctype != PIL_RC_UNKNOWN){
		pil_nextline();
		rctype = optr->rctype;
		/* PLEASE MAKE ME A SUBROUTINE!  */
		pil_put_kw(PIL_KW_RC_TYPE);
		pil_put_string(BEGINLIST);
		if (rctype & PIL_RC_TEXT) {
			pil_put_kw(PIL_KW_RC_TYPE_TEXT); 
		}
		if (rctype & PIL_RC_GEOMETRY) {
			pil_put_kw(PIL_KW_RC_TYPE_GEOMETRY); 
		}
		if (rctype & PIL_RC_LINEART) {
			pil_put_kw(PIL_KW_RC_TYPE_LINEART); 
		}
		if (rctype & PIL_RC_IMAGE) {
			pil_put_kw(PIL_KW_RC_TYPE_IMAGE); 
		}
		pil_put_string(ENDBLOCK);
	}
	if (optr->rcname) {
		pil_nextline();
		pil_put_named_str(PIL_KW_RC_NAME,optr->rcname);
	}

	if (optr->src_rect.width != 0 && optr->src_rect.height != 0) {
		pil_nextline();
		pil_put_kw(PIL_KW_BBOX);
		pil_put_string(BEGINLIST);
		pil_put_dbl(optr->src_rect.ul.x);
		pil_put_dbl(optr->src_rect.ul.y);
		pil_put_dbl(optr->src_rect.width);
		pil_put_dbl(optr->src_rect.height);
		pil_put_string(ENDBLOCK);
		pil_nextline();
	}
	if (optr->asi != NULL) {
		pil_nextline();
		pil_put_asi(optr->asi);
	}
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_text_flow

	Subroutine for pil_put_component.
	Writes a pil_text_flow to the open pil file.

	Entry:	Pointer to a pil_text_flow

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_text_flow(tfptr)
pil_text_flow PTRTYPE *tfptr;
{
	pil_put_named_begblk(PIL_KW_TEXT_FLOW);
	if (tfptr->label) {
		pil_nextline();
		pil_put_named_str(PIL_KW_FLOW_LABEL,tfptr->label);
	}
	if (tfptr->from_layout) {
		pil_nextline();
		pil_put_kw(PIL_KW_FLOW_FROM);
		pil_put_string(BEGINLIST);
		pil_put_quoted_str(tfptr->from_layout);
		if (tfptr->from_label) pil_put_quoted_str(tfptr->from_label);
		pil_put_string(ENDBLOCK);		
	}
	if (tfptr->to_layout) {
		pil_nextline();
		pil_put_kw(PIL_KW_FLOW_TO);
		pil_put_string(BEGINLIST);
		pil_put_quoted_str(tfptr->to_layout);
		if (tfptr->to_label) pil_put_quoted_str(tfptr->to_label);
		pil_put_string(ENDBLOCK);
	}
	if (tfptr->objects) {
		pil_nextline();
		pil_put_kw(PIL_KW_FLOW_OBJECTS);
		if (pil_put_objid(tfptr->objects) != PIL_OK)
			return (pil_last_error());
	}
	if (tfptr->content) {
		pil_nextline();
		pil_put_kw(PIL_KW_FLOW_CONTENT);
		if (pil_put_objid(tfptr->content) != PIL_OK)
			return (pil_last_error());
	}
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_group

	Subroutine for pil_put_component.
	Writes a pil_group to the open pil file.
 
	Entry:	Pointer to a pil_group

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_group(gptr)
pil_group PTRTYPE *gptr;
{
	pil_put_named_begblk(PIL_KW_GROUP);
	pil_nextline();
	if (gptr->name) {
		pil_put_kw(PIL_KW_ID);
		pil_put_quoted_str(gptr->name);
		pil_nextline();
	}
	if (gptr->objects) {
		pil_put_kw(PIL_KW_GROUP_OBJECTS);
		if (pil_put_objid(gptr->objects) != PIL_OK)
			return (pil_last_error());
	}
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_layout_end

	Subroutine for pil_put_component.
	Writes a pil_layout_end to the open pil file.
 
	Entry:	Pointer to a pil_layout_end

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_layout_end(leptr)
pil_layout_end PTRTYPE *leptr;
{
	if (!PIL_RIGHT_CMPNT(leptr,PIL_LAYOUT_END_C))
		return (pil_set_error(PIL_BAD_CMPNT_TYPE));
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_content_start

	Subroutine for pil_put_component.
	Writes a pil_content_start to the open pil file.

	Entry:	Pointer to a pil_content_start

	Exit:	Err code returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_content_start(csptr)
pil_content_start PTRTYPE *csptr;
{
	/* set up the content header defaults for this entity */
	if (!PIL_RIGHT_CMPNT(csptr,PIL_CONTENT_START_C))
		return (pil_set_error(PIL_BAD_CMPNT_TYPE));
	pil_set_dflt_hdr();
	pil_put_kw_no_sep(PIL_KW_CONTENT);
	pil_put_literal(csptr->version);
	pil_put_quoted_str(csptr->name);
	pil_put_nested_begblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_content_hdr

	Subroutine for pil_put_component.
	Writes a header for content data to the open pil file.
	If called with a pointer to a content header, it writes that header.
	Can also be called with a null pointer, in which case it writes out
	the current (last used) content header (but does not reset it to
	the defaults).
 
	Entry:	Pointer to a pil_content_hdr

	Exit:	Err code returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_content_hdr(chptr)
FAST pil_content_hdr PTRTYPE *chptr;
{
	/* Set up new header data, if we were passed a valid header */
	if (chptr != NULL) {
		if (!PIL_RIGHT_CMPNT(chptr,PIL_CONTENT_HDR_C))
			return (pil_set_error(PIL_BAD_CMPNT_TYPE));
		if (chptr->encoding != PIL_ASCII_CONTENT &&
			chptr->encoding != PIL_BINARY_CONTENT) {
			return (pil_set_error(PIL_UNKN_HDR_ENC));
		}
		if (chptr->byteorder != PIL_LITTLE_ENDIAN &&
			chptr->byteorder != PIL_BIG_ENDIAN) {
			return (pil_set_error(PIL_UNKN_HDR_BORD));
		}
		if (chptr->endblkchar == chptr->escapechar) {
			return (pil_set_error(PIL_BAD_HEADER));
		}
		/* NOTE: for binary data, the escape, begin block and end block */
		/*	characters can be ignored. Therefore, we check here and set */
		/*	them to the language "defaults" if we have binary data */
		currenthdr.encoding = chptr->encoding;
		currenthdr.byteorder = chptr->byteorder;
		if (chptr->encoding == PIL_BINARY_CONTENT) {
			currenthdr.escapechar = PIL_DFT_CNT_ESC;
			currenthdr.begblkchar = PIL_DFT_CNT_BEG;
			currenthdr.endblkchar = PIL_DFT_CNT_END;
		}
		else {
			currenthdr.escapechar = chptr->escapechar;
			currenthdr.begblkchar = chptr->begblkchar;
			currenthdr.endblkchar = chptr->endblkchar;
		}
	}
	pil_put_kw(PIL_KW_HEADER);
	pil_put_string(BEGINLIST);
	if (currenthdr.byteorder == PIL_LITTLE_ENDIAN) {
		pil_put_kw(PIL_KW_LITTLE_ENDIAN);
	}
	else {
		pil_put_kw(PIL_KW_BIG_ENDIAN);
	}
	if (currenthdr.encoding == PIL_ASCII_CONTENT) {
		pil_put_kw(PIL_KW_ASCII);
	}
	else {
		pil_put_kw(PIL_KW_BINARY);
	}
	pil_put_ui32((PIL_UINT32)currenthdr.escapechar);
	pil_put_ui32((PIL_UINT32)currenthdr.begblkchar);
	pil_put_ui32((PIL_UINT32)currenthdr.endblkchar);
	pil_put_string(ENDBLOCK);
	pil_nextline();
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_content_data

	Subroutine for pil_put_component.
	Writes a pil_content_start to the open pil file.
 
	Entry:	Pointer to a pil_content_start 

	Exit:	Err code returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_content_data(cdptr)
pil_content_data PTRTYPE *cdptr;
{
FAST PIL_UINT8	end,esc,c;
char lilbuf[4];
static char linebuf[60];
static int	bufpos = 0;
FAST char PTRTYPE *p;
PIL_UINT32	l;
	if (!PIL_RIGHT_CMPNT(cdptr,PIL_CONTENT_DATA_C))
		return (pil_set_error(PIL_BAD_CMPNT_TYPE));
	pil_put_kw(PIL_KW_DATA);
	(pil_putchar)(currenthdr.begblkchar);
	p = cdptr->data;
	if ((l = cdptr->length) > 0) {
		if (currenthdr.encoding == PIL_ASCII_CONTENT) {
			end = currenthdr.endblkchar;
			esc = currenthdr.escapechar;
			while (l--) {
			/* if an endblock character or non-printing character occurs in */
			/* the data, escape it */
				c = *p++;
				if (c == esc || c == end || c < ' ' || c > '~')
				{
					lilbuf[0] = esc;
					lilbuf[1] = hex[c >> 4];
					lilbuf[2] = hex[c & 0x0F];
					lilbuf[3] = '\0';
					pil_put_string(lilbuf);
				}
				else {
					(pil_putchar)(c);
				}
			}
		}
		else {				/* binary data, just pump it out neatly */
			pil_nextline();	/* whitespace is tolerated */
			while (l--) {
				c = *p++;
				if (bufpos > sizeof(linebuf) - 4) {
					linebuf[bufpos] = '\0';
					pil_put_string(linebuf);
					pil_nextline();
					bufpos = 0;
					linebuf[0] = '\0';
				}
				linebuf[bufpos++] = hex[c >> 4];
				linebuf[bufpos++] = hex[c & 0x0F];
				linebuf[bufpos++] = ' ';
			}
			if (bufpos > 0) {
				linebuf[bufpos] = '\0';
				pil_put_string(linebuf);
				pil_nextline();
				bufpos = 0;
				linebuf[0] = '\0';
			}
		}
	}
	(pil_putchar)(currenthdr.endblkchar);
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_content_end

	Subroutine for pil_put_component.
	Writes a pil_content_end to the open pil file.
 
	Entry:	Pointer to a pil_content_end

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_content_end(ceptr)
pil_content_end PTRTYPE *ceptr;
{
	if (!PIL_RIGHT_CMPNT(ceptr,PIL_CONTENT_END_C))
		return (pil_set_error(PIL_BAD_CMPNT_TYPE));
	pil_put_nested_endblk();
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_dimensions

	Subroutine for pil_put_canvas, pil_put_object.
	Writes a dimensions to the open pil file.
 
	Entry:	width, height (UIN32)

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_dimensions(width, height)
PIL_UINT32 width,height;
{
	if (width == 0 || height == 0) return (pil_set_error(PIL_ZERO_DIMENSION));
	pil_put_kw(PIL_KW_DIMENSIONS);
	pil_put_string(BEGINBLOCK);
	pil_put_ui32(width);
	pil_put_ui32(height);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}
	
/*=======================================================================*\
  pil_put_clipper

	Subroutine for pil_put_canvas, pil_put_object.
	Writes a pil_clipshape to the open pil file.
 
	Entry:	Pointer to a pil_clipper

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_clipper(cptr)
pil_clipper PTRTYPE *cptr;
{
	pil_put_named_begblk(PIL_KW_CLIPPER);
	pil_nextline();
	if (IS_OPEN_SHAPE(cptr->type)) return (pil_set_error(PIL_OPEN_CLIPPER));
	if (pil_put_closed_shape(cptr) != PIL_OK) return (pil_last_error());
	pil_put_nested_endblk();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_container

	Subroutine for pil_put_object.
	Writes a pil_container to the open pil file.
 
	Entry:	Pointer to a pil_clipper

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_container(cptr)
pil_container PTRTYPE *cptr;
{
	pil_put_named_begblk(PIL_KW_CONTAINER);
	pil_nextline();
	if (IS_OPEN_SHAPE(cptr->type)) return (pil_set_error(PIL_OPEN_CONTAINER));
	if (pil_put_closed_shape(cptr) != PIL_OK) return (pil_last_error());
	pil_put_nested_endblk();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_graphic

	Subroutine for pil_put_object.
	Writes a pil_graphic to the open pil file.
 
	Entry:	Pointer to a pil_graphic

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_graphic(gptr)
pil_graphic PTRTYPE *gptr;
{
	static int colortokens[] = {
		PIL_KW_DEV_GRAY,			/*	PIL_DEV_GRAY_CM		0x0000 */
		PIL_KW_CAL_GRAY,			/*	PIL_CAL_GRAY_CM		0x0001 */
		PIL_KW_DEV_RGB,				/*	PIL_DEV_RGB_CM		0x0002 */
		PIL_KW_CAL_RGB,				/*	PIL_CAL_RGB_CM		0x0003 */
		PIL_KW_DEV_CMYK,			/*	PIL_DEV_CMYK_CM		0x0004 */
		PIL_KW_PANTONE,				/*	PIL_PANTONE_CM		0x0005 */
		PIL_KW_SPOT					/*	PIL_SPOT_CM			0x0006 */
	};
	static int renderops[] = {
		0,
		PIL_KW_FILL,				/* PIL_FILL_RENDER_OP			0x0001 */
		PIL_KW_STROKE,				/* PIL_STROKE_RENDER_OP			0x0002 */
		PIL_KW_FILL_STROKE			/* PIL_FILL_STROKE_RENDER_OP	0x0003 */
	};
	
	pil_put_named_begblk(PIL_KW_GRAPHIC);
	/* put the color model */
	if (gptr->color_model > PIL_SPOT_CM) {
		return (pil_set_error(PIL_UNKN_CM));
	}
	else {
		pil_nextline();
		pil_put_kw(PIL_KW_COLOR_MODEL); 
		pil_put_kw(colortokens[gptr->color_model]); 
	}
	
	/* put the renderop */
	if (gptr->renderop < PIL_FILL_RENDER_OP || 
		gptr->renderop > PIL_FILL_STROKE_RENDER_OP) 
		return (pil_set_error(PIL_UNKN_RENDEROP));
	pil_nextline();
	pil_put_kw(PIL_KW_RENDER_OP);
	pil_put_kw(renderops[gptr->renderop]);
	pil_nextline();
	
	/* then the render attributes */
	pil_put_named_begblk(PIL_KW_RENDER_ATTR); 
	if (PIL_IS_STROKE_OP(gptr->renderop)) {		/* the stroke attributes */
		/* Write out the stroke color */
		pil_nextline();
		pil_put_kw(PIL_KW_STROKE_COLOR);
		if (pil_put_color(gptr->color_model,&gptr->stroke_color) != PIL_OK)
			return (pil_last_error());

		/* Write out the line position, if not the default */
		if (gptr->lineposition != PIL_CENTERED) {
			pil_nextline();
			pil_put_kw(PIL_KW_POSITION); 
			if (gptr->lineposition == PIL_INSIDE) {
				pil_put_kw(PIL_KW_INSIDE); 
			}
			else if (gptr->lineposition == PIL_OUTSIDE){
				pil_put_kw(PIL_KW_OUTSIDE); 
			}
			else {
				return (pil_set_error(PIL_UNKN_RENDEROP));
			}
		}
		
		/* Write out the line width, if not the default */
		if (gptr->linewidth != 10) {
			pil_nextline();
			pil_put_kw(PIL_KW_WIDTH); 
			pil_put_ui32(gptr->linewidth);
		}
		
		/* Write out the line cap, if not the default */
		if (gptr->linecap != PIL_BUTT_CAP) {
			pil_nextline();
			pil_put_kw(PIL_KW_CAP); 
			if (gptr->linecap == PIL_ROUND_CAP) {
				pil_put_kw(PIL_KW_ROUND_CAP); 
			}
			else if (gptr->linecap == PIL_PROJECTING_CAP){
				pil_put_kw(PIL_KW_PROJECTING_CAP); 
			}
			else {
				return (pil_set_error(PIL_UNKN_RENDEROP));
			}
		}
		
		/* Write out the line join, if not the default */
		if (gptr->linejoin != PIL_MITER_JOIN) {
			pil_nextline();
			pil_put_kw(PIL_KW_JOIN); 
			if (gptr->linejoin == PIL_BEVEL_JOIN) {
				pil_put_kw(PIL_KW_BEVEL_JOIN); 
			}
			else if (gptr->linejoin == PIL_ROUND_JOIN){
				pil_put_kw(PIL_KW_ROUND_JOIN); 
			}
			else {
				return (pil_set_error(PIL_UNKN_RENDEROP));
			}
		}
		
		/* Write out the miter limit, if not the default */
		if (gptr->miter_limit != 10.0) {
			if (gptr->miter_limit < 1.0) return (pil_set_error(PIL_BAD_MITERLIM));
			pil_nextline();
			pil_put_kw(PIL_KW_MITER_LIMIT); 
			pil_put_dbl(gptr->miter_limit);
		}
		
	}
	if (PIL_IS_FILL_OP(gptr->renderop)) {				/* do the fill attributes */
		/* Write out the fill color */
		pil_nextline();
		pil_put_kw(PIL_KW_FILL_COLOR);
		if (pil_put_color(gptr->color_model,&gptr->fill_color) != PIL_OK)
			return (pil_last_error());
		if (gptr->fill_rule != PIL_EVEN_ODD) {
			pil_nextline();
			pil_put_kw(PIL_KW_FILL_RULE); 
			if (gptr->fill_rule == PIL_NON_ZERO_WINDING){
				pil_put_kw(PIL_KW_NZ_WINDING); 
			}
			else {
				return (pil_set_error(PIL_UNKN_FILLRULE));
			}
		}
	}
	pil_put_nested_endblk();
	pil_nextline();
	/*	Check for 2 special "pseudoshapes" which are only used for the 
		object graphic shape, and hence are not included directly in 
		pil_put_shape: PIL_KW_CLIPPER_SHAPE and PIL_KW_CONTAINER_SHAPE */
	if (gptr->shape.type == PIL_CLIPPER_SHAPE) {
		pil_put_kw(PIL_KW_CLIPPER_SHAPE);
	}
	else if (gptr->shape.type == PIL_CONTAINER_SHAPE) {
		pil_put_kw(PIL_KW_CONTAINER_SHAPE);
	}
	else {
		if (pil_put_shape(&gptr->shape) != PIL_OK) return (pil_last_error());
	}
	pil_put_nested_endblk();
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_color

	Subroutine for pil_put_clipper, pil_put_object.
	Writes a pil_primitive to the open pil file.
 
	Entry:	Pointer to a pil_primitive

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_color(model,color)
PIL_UINT16	model;
pil_color	PTRTYPE *color;
{
	switch (model) {
		case PIL_DEV_GRAY_CM:
		case PIL_CAL_GRAY_CM:
			if (color->devgray.value < 0.0 || color->devgray.value > 1.0)
				return (PIL_BAD_COLOR);
			pil_put_dbl(color->devgray.value);
			break;
		case PIL_DEV_RGB_CM:
		case PIL_CAL_RGB_CM:
			if (color->devrgb.r < 0.0 || color->devrgb.r > 1.0  ||
				color->devrgb.g < 0.0 || color->devrgb.g > 1.0  ||
				color->devrgb.b < 0.0 || color->devrgb.b > 1.0)
				return (PIL_BAD_COLOR);
			pil_put_string(BEGINLIST);
			pil_put_dbl(color->devrgb.r);
			pil_put_dbl(color->devrgb.g);
			pil_put_dbl(color->devrgb.b);
			pil_put_string(ENDBLOCK);
 			break;
 		case PIL_DEV_CMYK_CM:
			if (color->devcmyk.c < 0.0 || color->devcmyk.c > 1.0 ||
				color->devcmyk.m < 0.0 || color->devcmyk.m > 1.0 ||
				color->devcmyk.y < 0.0 || color->devcmyk.y > 1.0 ||
				color->devcmyk.k < 0.0 || color->devcmyk.k > 1.0)
				return (PIL_BAD_COLOR);
			pil_put_string(BEGINLIST);
			pil_put_dbl(color->devcmyk.c);
			pil_put_dbl(color->devcmyk.m);
			pil_put_dbl(color->devcmyk.y);
			pil_put_dbl(color->devcmyk.k);
			pil_put_string(ENDBLOCK);
 			break;
		case PIL_PANTONE_CM:
		case PIL_SPOT_CM:
			pil_put_quoted_str(color->pantone.name);
			break;
		default:
			return (pil_set_error(PIL_UNKN_CM));	
	}
	pil_nextline();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_closed_shape

	Subroutine for pil_put_clipper, pil_put_object.
	Writes a pil_primitive to the open pil file.
 
	Entry:	Pointer to a pil_primitive

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_closed_shape(cptr)
pil_closed_primitive PTRTYPE *cptr;
{
	PIL_ERROR rc;

	if (pil_put_shape_code(cptr->type) != PIL_OK) return (pil_last_error());
	switch (cptr->type) {
		case	PIL_RECTANGLE_SHAPE:
			rc = pil_put_rectdata(&cptr->shape.rect);
			break;
		case	PIL_CIRCLE_SHAPE:
			rc = pil_put_circledata(&cptr->shape.circle);
			break;
		case	PIL_ELLIPSE_SHAPE:
			rc = pil_put_rectdata((pil_rectangle PTRTYPE *)&cptr->shape.ellipse);
			break;
		case	PIL_POLYGON_SHAPE:
			rc = pil_put_polydata(&cptr->shape.polygon);
			break;
		case	PIL_CLOSEDPATH_SHAPE:
			rc = pil_put_pathdata(&cptr->shape.closedpath);
			break;
		default:
			rc = pil_set_error(PIL_UNKN_SHAPE);
			break;
	}
	return(rc);
}
/*=======================================================================*\
  pil_put_shape

	Subroutine for pil_put_clipper, pil_put_object, pil_put_graphic.
	Writes a pil_primitive to the open pil file.
 
	Entry:	Pointer to a pil_primitive

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_shape(cptr)
pil_primitive PTRTYPE *cptr;
{
	PIL_ERROR rc;

	if (pil_put_shape_code(cptr->type) != PIL_OK) 
		return (pil_last_error());
	switch (cptr->type) {
		case	PIL_LINE_SHAPE:
			rc = pil_put_line(&cptr->shape.line);
			break;
		case	PIL_ARC_SHAPE:
			rc = pil_put_arc(&cptr->shape.arc);
			break;
		case	PIL_BEZIER_SHAPE:
			rc = pil_put_bezier(&cptr->shape.bezier);
			break;
		case	PIL_RECTANGLE_SHAPE:
			rc = pil_put_rectdata(&cptr->shape.rect);
			break;
		case	PIL_CIRCLE_SHAPE:
			rc = pil_put_circledata(&cptr->shape.circle);
			break;
		case	PIL_ELLIPSE_SHAPE:
			rc = pil_put_rectdata((pil_rectangle PTRTYPE *)&cptr->shape.ellipse);
			break;
		case	PIL_POLYGON_SHAPE:
		case	PIL_POLYLINE_SHAPE:
			rc = pil_put_polydata(&cptr->shape.polygon);
			break;
		case	PIL_CLOSEDPATH_SHAPE:
		case	PIL_OPENPATH_SHAPE:
			rc = pil_put_pathdata(&cptr->shape.closedpath);
			break;
		default:
			rc = pil_set_error(PIL_UNKN_SHAPE);
			break;
	}
	return (rc);
}
/*=======================================================================*\
  pil_put_line

	Subroutine for pil_put_shape.
	Writes a pil_line to the open pil file.
 
	Entry:	Pointer to a pil_primitive

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_line(ptr)
pil_line PTRTYPE *ptr;
{
	if (SAMEPOINT(ptr->start,ptr->end)) 
		return (pil_set_error(PIL_ZERO_LENGTH));
	pil_put_string(BEGINBLOCK);
	pil_put_point_data(&ptr->start);
	pil_put_point_data(&ptr->end);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_arc

	Subroutine for pil_put_shape.
	Writes a pil_arc to the open pil file.
 
	Entry:	Pointer to a pil_arc

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_arc(ptr)
pil_arc PTRTYPE *ptr;
{
	if (ptr->arc_square.length == 0 ||
		(ptr->start_angle == ptr->end_angle)) {
		return (pil_set_error(PIL_ZERO_SHAPE));
	}
	pil_put_string(BEGINBLOCK);
	pil_put_point_data(&ptr->arc_square.ul);
	pil_put_ui32(ptr->arc_square.length);
	pil_put_dbl(ptr->start_angle);
	pil_put_dbl(ptr->end_angle);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_bezier

	Subroutine for pil_put_shape.
	Writes a pil_bezier to the open pil file.
 
	Entry:	Pointer to a pil_bezier

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_bezier(ptr)
pil_bezier PTRTYPE *ptr;
{
	pil_put_string(BEGINBLOCK);
	pil_put_point(&ptr->start);
	pil_put_point(&ptr->start_ctrl_pt);
	pil_put_point(&ptr->end_ctrl_pt);
	pil_put_point(&ptr->end);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_rectdata

	Subroutine for pil_put_shape
	Writes a pil_rectangle to the open pil file.
 
	Entry:	Pointer to a pil_rectangle

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_rectdata(ptr)
pil_rectangle PTRTYPE *ptr;
{
	if (ptr->height == 0 || ptr->width == 0) 
		return (pil_set_error(PIL_ZERO_SHAPE));
	pil_put_string(BEGINBLOCK);
	pil_put_point_data(&ptr->ul);
	pil_put_ui32(ptr->width);
	pil_put_ui32(ptr->height);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}
/*=======================================================================*\
  pil_circledata

	Subroutine for pil_put_shape.
	Writes a pil_circle to the open pil file.
 
	Entry:	Pointer to a pil_circle

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_circledata(ptr)
pil_circle PTRTYPE *ptr;
{
	if (ptr->length == 0) return (pil_set_error(PIL_ZERO_SHAPE));
	pil_put_string(BEGINBLOCK);
	pil_put_i32(ptr->ul.x);
	pil_put_i32(ptr->ul.y);
	pil_put_ui32(ptr->length);
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_pathdata

	Subroutine for pil_put_shape.
	Writes the points of a pil_path to the open pil file.
 
	Entry:	Pointer to a pil_polygon

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_pathdata(ptr)
pil_path_pt PTRTYPE *ptr;
{
	pil_put_nested_begblk();
	pil_nextline();
	if (ptr->type != PIL_MOVE_TO) return (pil_set_error(PIL_NEED_MOVE_TO));
	while (ptr != NULL) {
		switch(ptr->type) {
			case PIL_MOVE_TO:
				pil_put_kw(PIL_KW_MOVE_TO); 
				pil_put_point(&ptr->p.mt.pt);
				break;
			case PIL_LINE_TO:
				pil_put_kw(PIL_KW_LINE_TO); 
				pil_put_point(&ptr->p.lt.pt);
				break;
			case PIL_CURVE_TO:
				pil_put_kw(PIL_KW_CURVE_TO); 
				pil_put_string(BEGINBLOCK);
				pil_put_point(&ptr->p.ct.ctrl_pt_1);
				pil_put_point(&ptr->p.ct.ctrl_pt_2);
				pil_put_point(&ptr->p.ct.end_pt);
				pil_put_string(ENDBLOCK);
				break;
			case PIL_ARC_TO:
				pil_put_kw(PIL_KW_ARC_TO); 
				pil_put_string(BEGINBLOCK);
				pil_put_point(&ptr->p.at.ctrl_pt_1);
				pil_put_point(&ptr->p.at.ctrl_pt_2);
				pil_put_i32(ptr->p.at.radius);
				pil_put_string(ENDBLOCK);
				break;
			default:
				return (pil_set_error(PIL_UNKN_PATHPT));
		}
		if ((ptr = ptr->next) != NULL) {
			pil_nextline();
		}
		else {
			break;
		}
	}
	pil_put_nested_endblk();
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_polydata

	Subroutine for pil_put_shape.
	Writes the points of a pil_polygon or pil_polyline to the open pil file.
 
	Entry:	Pointer to a pil_polygon

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_polydata(ptr)
pil_poly_pt PTRTYPE *ptr;
{
	pil_put_nested_begblk();
	pil_nextline();
	for (;;) {
		pil_put_point(&ptr->p);
		if ((ptr = ptr->next) == NULL)
			break;
		else 
			pil_nextline();
	}
	pil_put_nested_endblk();
	return (PIL_OK);
}
/*=======================================================================*\
  pil_put_point

	Subroutine to write a pil_point to the open pil file.
 
	Entry:	Pointer to a pil_point

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_point(ptr)
pil_point PTRTYPE *ptr;
{
	pil_put_string(BEGINBLOCK);
	pil_put_point_data(ptr);
	pil_put_string(ENDBLOCK);
}
/*=======================================================================*\
  pil_put_point_data

	Subroutine to write coordinates of a pil_point to the open pil file.
 
	Entry:	Pointer to a pil_point

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_point_data(ptr)
pil_point PTRTYPE *ptr;
{
	pil_put_i32(ptr->x);
	pil_put_i32(ptr->y);
}

/*=======================================================================*\
  pil_put_nametblentry

	Writes a pil_name_table_entry to the open pil file.
 
	Entry:	Pointer to a pil_name_table *

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_nametbleentry(ptr)
FAST pil_name_table_entry PTRTYPE *ptr;
{
	if (ptr->name == NULL) return (pil_set_error(PIL_BAD_NTE));
	pil_put_quoted_str(ptr->name);
	if (pil_put_ctype_code(ptr->content_type_code, ptr->content_type_string) 
		!= PIL_OK) return (pil_last_error());
	if (pil_put_domain_code(ptr->domain_type_code, ptr->domain_type_string) 
		!= PIL_OK) return (pil_last_error());
	switch (ptr->value_type_code) {
		case	PIL_VAL_INLINE:
			pil_put_kw(PIL_KW_INLINE);
			break;
		case	PIL_VAL_UNKNOWN:
			pil_put_kw(PIL_KW_UNKNOWN);
			break;
		case	PIL_VAL_OTHER:
			if (ptr->value_type_string == NULL)
				return  (pil_set_error(PIL_BAD_NTE));
			else
				pil_put_quoted_str(ptr->value_type_string);
			break;
		default:
			return (pil_set_error(PIL_BAD_NTE));
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_ctype_code

	Writes a content type code to the open pil file. This routine depends
	on the fact that the content type codes as defined in pildefs.h are
	sequential integers which can be used as an array index. The array
	declared below puts the appropriate keyword value for each content
	type at the appropriate place in the array.
 
	Entry:	UINT16 that is a pil content type code

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_ctype_code(code,string)
PIL_NT_TYPE code;
char PTRTYPE *string;
{
	static int ctypetokens[] = {/* This column has the content type code */
		0,						/* PIL_CNT_OTHER				0 */
		PIL_KW_UNKNOWN ,		/* PIL_CNT_UNKNOWN				1 */
		PIL_KW_ASCII_TEXT ,		/* PIL_CNT_ASCII_TEXT			2 */
		PIL_KW_ATEX_ARIS ,		/* PIL_CNT_ATEX_ARIS			3 */
		PIL_KW_ATEX_ITF ,		/* PIL_CNT_ATEX_ITF				4 */
		PIL_KW_EPS ,			/* PIL_CNT_EPS					5 */
		PIL_KW_ICL ,			/* PIL_CNT_ICL					6 */
		PIL_KW_IT8_1 ,			/* PIL_CNT_IT8_1				7 */
		PIL_KW_IT8_2 ,			/* PIL_CNT_IT8_2				8 */
		PIL_KW_MS_RTF ,			/* PIL_CNT_MS_RTF				9 */
		PIL_KW_MS_WORD_3 ,		/* PIL_CNT_MS_WORD_3_0			10 */
		PIL_KW_MS_WORD_4 ,		/* PIL_CNT_MS_WORD_4_0			11 */
		PIL_KW_MIF ,			/* PIL_CNT_MIF					12 */
		PIL_KW_PICT ,			/* PIL_CNT_PICT					13 */
		PIL_KW_PIL_LAYOUT_NTET,	/* PIL_CNT_PIL_LAYOUT			14 */
		PIL_KW_POSTSCRIPT ,		/* PIL_CNT_POSTSCRIPT			15 */
		PIL_KW_QUARK_XPRESSTAGS,/* PIL_CNT_QUARK_XP_TAGS		16 */
		PIL_KW_SCITEX_CT ,		/* PIL_CNT_SCITEX_CT			17 */
		PIL_KW_SCITEX_HANDSHAKE,/* PIL_CNT_SCITEX_HS			18 */
		PIL_KW_SCITEX_LW ,		/* PIL_CNT_SCITEX_LW			19 */
		PIL_KW_SGML ,			/* PIL_CNT_SGML					20 */
		PIL_KW_TIFF ,			/* PIL_CNT_TIFF					21 */
		PIL_KW_WP_100 ,			/* PIL_CNT_WP_1_00				22 */
		PIL_KW_WP_102 ,			/* PIL_CNT_WP_1_02				23 */
		PIL_KW_XYQUEST_XYWRITE	/* PIL_CNT_XYQUEST_XYWRITE		24 */
	};
	/* check for out of range content type code */
	if (code > PIL_CNT_XYQUEST_XYWRITE)
		return (pil_set_error(PIL_BAD_NTE));
	/* check for no content type */
	if (code == PIL_CNT_OTHER) {
		if (string == NULL) 
			return (pil_set_error(PIL_BAD_NTE));
		else
			pil_put_quoted_str(string);
	}
	else {
		pil_put_kw(ctypetokens[code]);
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_domain_code

	Writes a domain type code to the open pil file. This routine depends
	on the fact that the domain type codes as defined in pildefs.h are
	sequential integers which can be used as an array index. The array
	declared below puts the appropriate keyword value for each content
	type at the appropriate place in the array.
 
	Entry:	UINT16 that is a pil domain type code

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_domain_code(code,string)
PIL_NT_DMN code;
char PTRTYPE *string;
{
	static int dtypetokens[] = {
		0,								/* PIL_DMN_OTHER			0 */
		PIL_KW_MAC_FILENAME,			/* PIL_DMN_MAC_FILE			1 */
		PIL_KW_UNIX_FILENAME,			/* PIL_DMN_UNIX_FILE		2 */
		PIL_KW_MSDOS_FILENAME,			/* PIL_DMN_MSDOS_FILE		3 */
		PIL_KW_OS2_FILENAME,			/* PIL_DMN_OS2_FILE			4 */
		PIL_KW_VMS_FILENAME,			/* PIL_DMN_VMS_FILE			5 */
		PIL_KW_VM_FILENAME,				/* PIL_DMN_VM_FILE			6 */
		PIL_KW_LCD_FILENAME,			/* PIL_DMN_LCD_FILE			7 */
		PIL_KW_INLINE,					/* PIL_DMN_INLINE			8 */
		PIL_KW_UNKNOWN,					/* PIL_DMN_UNKNOWN			9 */
	};
	/* check for out of range domain type code */
	if (code > PIL_DMN_UNKNOWN)
		return (pil_set_error(PIL_BAD_NTE));
	/* check for no content type */
	if (code == PIL_DMN_OTHER) {
		if (string == NULL)
			return (pil_set_error(PIL_BAD_NTE));
		else
			pil_put_quoted_str(string);
	}
	else {
		pil_put_kw(dtypetokens[code]);
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_shape_code

	Writes the name of a shape to the open pil file.
 
	Entry:	UINT16 that is a pil shape type

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_shape_code(code)
PIL_UINT16 code;
{
	static int openshapes[] = {
		0,							/*  */
		PIL_KW_LINE,				/* PIL_LINE_SHAPE 		0x0001 */
		PIL_KW_ARC,					/* PIL_ARC_SHAPE		0x0002 */
		PIL_KW_BEZIER,				/* PIL_BEZIER_SHAPE		0x0003  */
		PIL_KW_POLY_LINE,			/* PIL_POLYLINE_SHAPE	0x0004  */
		PIL_KW_OPEN_PATH			/* PIL_OPENPATH_SHAPE	0x0005  */
	};
	static int closedshapes[] = {
		0,							/*  */
		PIL_KW_RECTANGLE,			/* PIL_RECTANGLE_SHAPE	0x0100 */
		PIL_KW_CIRCLE,				/* PIL_CIRCLE_SHAPE		0x0200 */
		PIL_KW_ELLIPSE,				/* PIL_ELLIPSE_SHAPE	0x0300 */
		PIL_KW_POLYGON,				/* PIL_POLYGON_SHAPE	0x0400 */
		PIL_KW_CLOSED_PATH			/* PIL_CLOSEDPATH_SHAPE	0x0500 */
	};
	if (IS_CLOSED_SHAPE(code)) {
		if (code < PIL_RECTANGLE_SHAPE || code > PIL_CLOSEDPATH_SHAPE)
			return (pil_set_error(PIL_UNKN_SHAPE));
		pil_put_kw(closedshapes[code>>8]);
	}
	else {
		if (code < PIL_LINE_SHAPE || code > PIL_OPENPATH_SHAPE)
			return (pil_set_error(PIL_UNKN_SHAPE));
		pil_put_kw(openshapes[code]);
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_asi

	Writes a pil_APPL_SPECIFIC_ITEM to the open pil file. 
 
	Entry:	Pointer to a pil_asi *

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/


PIL_ERROR pil_put_asi(ptr)
pil_asi PTRTYPE *ptr;
{
	while (ptr != NULL) {
		pil_put_named_begblk(PIL_KW_APPLICATION);
		pil_nextline();
		pil_put_named_str(PIL_KW_APP_NAME,ptr->asi_name);
		pil_nextline();
		if (pil_put_asa(ptr->asa) != PIL_OK)
			return (pil_last_error());
		pil_put_nested_endblk();
		if ((ptr = ptr->next) != NULL) pil_nextline();
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_asa

	Writes a pil_asa list to the open pil file.
	The pil_asa is put in the buffer at the next 
	available location. It is given a name a list of values as passed
	to this function.
 
	Entry:	Pointer to a pil_asa *

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_asa(ptr)
pil_asa PTRTYPE *ptr;
{
	while (ptr != NULL) {
		if (pil_put_data_code(ptr->type & 0x0FFF) != PIL_OK)
			return (pil_last_error());
		pil_put_string(BEGINLIST);
		pil_put_string(ptr->attr_name);
		pil_put_string(SEPARATOR);
		if (pil_put_pvalue(ptr->value,ptr->type) != PIL_OK)
			return (pil_last_error());
		pil_put_string(ENDBLOCK);
		if ((ptr = ptr->next) != NULL) pil_nextline();
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_data_code

	Writes the token for a data type to the open pil file. NOTE: the
	array of tokens stored here corresponds to the definition of the 
	pil data type codes in pildefs.h, so don't change one without changing
	the other!
 
	Entry:	type code

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_data_code(type)
PIL_ASA_VAL_TYPE  type;
{
	static int datatokens[] = {
		PIL_KW_APP_INT8 ,
		PIL_KW_APP_INT16 ,
		PIL_KW_APP_INT32 ,
		PIL_KW_APP_UINT8 ,
		PIL_KW_APP_UINT16 ,
		PIL_KW_APP_UINT32 ,
		PIL_KW_APP_DOUBLE ,
		PIL_KW_APP_STRING
	};
	if (type > PIL_STRING_CODE)
		return (pil_set_error(PIL_BAD_VALUE_TYPE));
	pil_put_kw(datatokens[type]);
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_pvalue

	Writes a pil_value to the open pil file.
 
	Entry:	Pointer to a pil_value, data type of the value

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/
PIL_ERROR pil_put_pvalue(ptr,type)
pil_value PTRTYPE *ptr;
PIL_ASA_VAL_TYPE type;
{
	if (IS_PIL_VALUE_LIST(type)) {
		pil_put_string(BEGINLIST);
	}
	while (ptr != NULL) {
		switch (type) {
			case 	PIL_DOUBLE_CODE:
			case 	PIL_DOUBLE_LIST_CODE:
				pil_put_dbl((PIL_DOUBLE)ptr->data.dbl);
				break;
			case 	PIL_INT8_CODE:
			case 	PIL_INT8_LIST_CODE:
				pil_put_i32((PIL_INT32)((PIL_INT8)ptr->data.int8));
				break;
			case 	PIL_INT16_CODE:
			case 	PIL_INT16_LIST_CODE:
				pil_put_i32((PIL_INT32)((PIL_INT16)ptr->data.int16));
				break;
			case 	PIL_INT32_CODE:
			case 	PIL_INT32_LIST_CODE:
				pil_put_i32((PIL_INT32)ptr->data.int32);
				break;
			case 	PIL_UINT8_CODE:
			case 	PIL_UINT8_LIST_CODE:
				pil_put_ui32((PIL_UINT32)((PIL_UINT8)ptr->data.uint8));
				break;
			case 	PIL_UINT16_CODE:
			case 	PIL_UINT16_LIST_CODE:
				pil_put_ui32((PIL_UINT32)((PIL_UINT16)ptr->data.uint16));
				break;
			case 	PIL_UINT32_CODE:
			case 	PIL_UINT32_LIST_CODE:
				pil_put_ui32((PIL_UINT32)ptr->data.int32);
				break;
			case 	PIL_STRING_CODE:
			case 	PIL_STRING_LIST_CODE:
				pil_put_quoted_str((char PTRTYPE *)ptr->data.string);
				break;
			default:
				return (pil_set_error(PIL_BAD_VALUE_TYPE));
		}
		ptr = ptr->next;
	}
	if (IS_PIL_VALUE_LIST(type)) {
		pil_put_string(ENDBLOCK);
		pil_put_string(" ");
	}
	return (PIL_OK);
}

/*=======================================================================*\
  pil_put_objid

	Writes a pil_object_id_list to the open pil file.
 
	Entry:	Pointer to a pil_object_id_list *

	Exit:	Error code is returned.

\*-----------------------------------------------------------------------*/

PIL_ERROR pil_put_objid(ptr)
pil_object_id_list PTRTYPE *ptr;
{
	pil_put_string(BEGINLIST);
	while (ptr != NULL) {
		if (ptr->id == NULL) return (pil_set_error(PIL_BAD_OBJID));
		pil_put_quoted_str(ptr->id);
		ptr = ptr->next;
	}
	pil_put_string(ENDBLOCK);
	return (PIL_OK);
}

/*=======================================================================*\
  pil_set_dflt_hdr

	Subroutine for pil_put_content_start
	Resets the content header defaults for the start of a new content
	entity. If the application does not call pil_put_content_hdr, then
	the defaults set here will be used when writing the pil_content_data.

	Entry:	None

	Exit:	None

\*-----------------------------------------------------------------------*/

PIL_VOID pil_set_dflt_hdr()
{
	currenthdr.encoding = (PIL_INT8)PIL_ASCII_CONTENT;
	currenthdr.byteorder = (PIL_INT8)PIL_LITTLE_ENDIAN;
	currenthdr.escapechar = (PIL_UINT8)'\\';
	currenthdr.begblkchar = (PIL_UINT8)'{';
	currenthdr.endblkchar = (PIL_UINT8)'}';
}
/*=======================================================================*\
  pil_put_nested_begblk

	Subroutine for pil_put_component. Starts a nested block, set indent
	level.
 
	Entry:	none 

	Exit:	none

\*-----------------------------------------------------------------------*/

PIL_VOID pil_put_nested_begblk()
{
	indentlev++;
	pil_put_string(BEGINBLOCK);
}
/*=======================================================================*\
  pil_put_nested_endblk

	Subroutine for pil_put_component. Ends a nested block, sets indent
	level and indents the next line.
 
	Entry:	none 

	Exit:	none

\*-----------------------------------------------------------------------*/

PIL_VOID pil_put_nested_endblk()
{
	pil_put_string(NEWLINE);
	if (indentlev > 0) indentlev--;
	pil_put_string(pil_gen_indents((int) indentlev));
	pil_put_string(ENDBLOCK);
}

/*=======================================================================*\
  pil_put_literal
  
	Writes a string and a separator.
 
	Entry:	pointer to string 

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_literal(str)
char PTRTYPE *str;
{
	pil_put_string(str); 
	pil_put_string(SEPARATOR);
}
/*=======================================================================*\
  pil_put_named_str
  
	Writes a token, a separator, and a string in quotes, for exmaple,
		user-name "Page 1"

	Entry:	token, and string pointer 

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_named_str(token, str)
int token;
char PTRTYPE *str;
{
	pil_put_kw(token); 
	pil_put_quoted_str(str); 
}

/*=======================================================================*\
  pil_put_named_begblk
  
	Writes a token, a separator,nested begin block (indents line and 
	puts a newline)

	Entry:	token

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_named_begblk(token)
int token;
{
	pil_put_kw(token); 
	pil_put_nested_begblk(); 
}


/*=======================================================================*\
  pil_put_named_i32
  
	Writes a token, a separator, and an integer, for example,
		width 500

	Entry:	token, and a PIL_INT32

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_named_i32(token, n)
int token;
PIL_INT32 n;
{
	pil_put_kw(token); 
	pil_put_i32(n); 
}

/*=======================================================================*\
  pil_put_named_ui32
  
	Writes a token, a separator, and an integer, for example,
		width 500

	Entry:	token, and a PIL_UINT32

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_named_ui32(token, n)
int token;
PIL_UINT32 n;
{
	pil_put_kw(token); 
	pil_put_ui32(n); 
}

/*=======================================================================*\
  pil_put_named_dbl
  
	Writes a token, a separator, and an integer, for example,
		rot-angle -354.2

	Entry:	token, and a PIL_INT32

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_named_dbl(token, n)
int token;
PIL_DOUBLE n;
{
	pil_put_kw(token); 
	pil_put_dbl(n); 
}


/*=======================================================================*\
  pil_put_quoted_str
  
	Writes a string in quotes, followed by a separator.
 
	Entry:	string pointer 

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_quoted_str(str)
char PTRTYPE *str;
{
	pil_put_string("\""); 
	pil_put_string(str); 
	pil_put_string("\"");
	pil_put_string(SEPARATOR);
}

/*=======================================================================*\
  pil_nextline
  
	Subroutine for pil_put_component. Ends a line and indents the next one.
 
	Entry:	none 

	Exit:	none

\*-----------------------------------------------------------------------*/

PIL_VOID pil_nextline()
{
	pil_put_string(NEWLINE);
	pil_put_string(pil_gen_indents ((int) indentlev));
}

/*=======================================================================*\
  pil_put_kw
  
	Subroutine for pil_put_xxxxxx.
	Calls pil_put_string to write the keyword corresponding to the integer
	token, to the open pil file.
 
	Entry:	integer token 

	Exit:	none
	
\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_kw(token)
int token;
{
	pil_put_string(pil_token_to_str(token));
	pil_put_string(SEPARATOR);
}
/*=======================================================================*\
  pil_put_kw_no_sep
  
	Subroutine for pil_put_xxxxxx.
	Calls pil_put_string to write the keyword corresponding to the integer
	token, to the open pil file.
 
	Entry:	integer token 

	Exit:	none

\*-----------------------------------------------------------------------*/
PIL_VOID pil_put_kw_no_sep(token)
int token;
{
	pil_put_string(pil_token_to_str(token));
}


/*=======================================================================*\
  pil_token_to_str
  
	Subroutine for pil_put_kw.Converts an integer token to string.
 
	Entry:	integer token 

	Exit:	pointer to a string

\*-----------------------------------------------------------------------*/
char PTRTYPE *pil_token_to_str(token)
FAST int token; 
{
	if (token < PIL_MIN_KW || token > PIL_SIZE_KW_ARRAY-1) return ("");
	return (pil_kws[token]);	 
}


/*=======================================================================*\
  pil_i32toa
  
	Convert a signed 32 bit int to a string.
 
	Entry:	number, string buffer pointer

	Exit:	no return value,
			ascii representation of the number is in the buffer.

\*-----------------------------------------------------------------------*/
PIL_VOID pil_i32toa(n,s)
FAST PIL_INT32 n;
FAST char PTRTYPE *s;
{
	PIL_INT32 i, j, sign;
	char c;
	
	if ((sign=n) < 0)
		n = -n;
	i = 0;
	do {
		s[i++] = (char)(n % 10 + '0');
	} while ((n /= 10) > 0);

	if (sign < 0)
		s[i++] = '-';

	s[i] = '\0';
	/* reverse the string in place */
	for (j=0,i--; j<i; j++,i--) {
		c = s[j];
		s[j] = s[i];
		s[i] = c;
	}
}
/*=======================================================================*\
  pil_iu32toa
  
	Convert an unsigned 32 bit int to a string.
 
	Entry:	number, string buffer pointer

	Exit:	no return value,
			ascii representation of the number is in the buffer.

\*-----------------------------------------------------------------------*/

PIL_VOID pil_ui32toa(n,s)
PIL_UINT32 n;
FAST char PTRTYPE *s;
{
	int i,j;
	char c;
	i = 0;
	do {
		s[i++] = (char)(n % 10 + '0');
	} while ((n /= 10) > 0);
	s[i] = '\0';
	/* reverse the string in place */
	for (j=0,i--; j<i; j++,i--) {
		c = s[j];
		s[j] = s[i];
		s[i] = c;
	}
}
/*=======================================================================*\
  pil_ftoa
  
	Convert a PIL_DOUBLE (IEEE-754 floating point number) to ascii.
 
	Entry:	number, string buffer pointer

	Exit:	no return value,
			ascii representation of the number is in the buffer.

\*-----------------------------------------------------------------------*/

PIL_VOID pil_ftoa(n,s)
PIL_DOUBLE	n;
FAST char PTRTYPE *s;
{
#ifdef THINK_DA
	FAST int	i;
	decform	dform;				/* need to do %f or %g */
	/* convert floating point value to string */
	if (n > -1.0e9 && n < 1.0e9) dform.style = FIXEDDECIMAL;
	else dform.style = FLOATDECIMAL;
	dform.digits = 17;	/* 17 digits after the decimal point  */
	num2str(&dform,n,s);
	for (i = (int)*s; i > 0 ; i--, s++)
		*s = *(s+1);
	*s = '\0';
	/* PtoCstr(s); */
#else
#ifdef WINDOWS3
	/* stdlib routines require near pointers, so put string in a local buf */
	char localbuf[32];
	sprintf(localbuf,"%g",n);
	pil_strcpy(s,(char PTRTYPE *)localbuf);
#else
	sprintf(s,"%g",n);
#endif
#endif
}

/*=======================================================================*\
  pil_put_xxxxx 
  
	Various routines to print out numbers
	 
	Entry:	number

	Exit:	no return value

\*-----------------------------------------------------------------------*/

PIL_VOID pil_put_i32(n)
PIL_INT32 n;
{
	pil_i32toa(n,outbuf);
	pil_put_string(outbuf);
	pil_put_string(SEPARATOR);
}
PIL_VOID pil_put_ui32(n)
PIL_UINT32 n;
{
	pil_ui32toa(n,outbuf);
	pil_put_string(outbuf);
	pil_put_string(SEPARATOR);
}

PIL_VOID pil_put_dbl(n)
PIL_DOUBLE n;
{
	pil_ftoa(n,outbuf);
	pil_put_string(outbuf);
	pil_put_string(SEPARATOR);
}

/*=======================================================================*\
  pil_gen_indents 
	 
	Entry:	indent level

	Exit:	char pointer to your desired indenting string

\*-----------------------------------------------------------------------*/

char PTRTYPE *pil_gen_indents (level)
int level;
{
	switch (level)
	{
		case 0:		return ("");
		case 1:		return ("\t");
		case 2:		return ("\t\t");
		case 3:		return ("\t\t\t");
		case 4:		return ("\t\t\t\t");
		case 5:		return ("\t\t\t\t\t");
		case 6:		return ("\t\t\t\t\t\t");
		case 7:		return ("\t\t\t\t\t\t\t");
		case 8:		return ("\t\t\t\t\t\t\t\t");
		case 9:		return ("\t\t\t\t\t\t\t\t\t");
		case 10:	return ("\t\t\t\t\t\t\t\t\t\t");
		default:	return ("\t\t\t\t\t\t\t\t\t\t\t");
	}
}

/****************************End of pilgen.c******************************/
