page	58,132
title	TestBBL	Test algorithms for Big Block Letter device driver

Stak	SEGMENT PARA STACK 'STACK'
	DB	64 DUP ('STACK   ')
Stak	ENDS


CharWidth	EQU	08h		;width of chars in pixels
CharHeight	EQU	08h		;height of chars in pixels
DispChar	EQU	219		;solid block to construct display
CharAttr	EQU	0fh		;attribute of char to display

BELChar	EQU	07h
BSChar	EQU	08h
HTChar	EQU	09h
LFChar	EQU	0ah
VTChar	EQU	0bh
FFChar	EQU	0ch
CRChar	EQU	0dh
SOChar	EQU	0eh
SIChar	EQU	0fh

S$Init	EQU	0		;State 0 ==> Initial
S$VP	EQU	1		;State 1 ==> Vertical position
S$HP	EQU	2		;State 2 ==> Horizontal position
S$SM	EQU	3		;State 3 ==> Set Mode (BBL display mode)

TT$NOND	EQU	0		;Token type 0 ==> non-operation, non-digit
TT$Dig	EQU	1		;Token type 1 ==> digit
TT$CM	EQU	2		;Token type 2 ==> cursor movement
TT$HP	EQU	3		;Token type 3 ==> horizontal position
TT$VP	EQU	4		;Token type 4 ==> vertical position
TT$SM	EQU	5		;Token type 5 ==> set mode
NumTokTyp EQU	6


ModeDesc	STRUC
	Mag		db	?	;magnification for bit pattern
	MaxRows		db	?	;max rows with this mode
	MaxCols		db	?	;max cols with this mode
	RowOff		db	?	;row offset on phys screen 
	ColOff		db	?	;col offset on phys screen
ModeDesc	ENDS

StateDesc 	STRUC
	ActionProc	DW	?	;action this token invokes in this state
	NextState	DB	?	;next state
StateDesc	ENDS

;	The token classes are stored in the table "TC", and the
;	default is TT$NOND, non-op, non-digit.
;	We build the token table exceptions by processing a table
;	of exceptions, labeled with the characters (tokens), into the
;	TC table, addressed by char value.  First, define
;	the structure we use to build the exceptions table.


TokDef STRUC
	char	DB	?	;character being classified ("typed")
	ctype	DB	?	;type of that char
TokDef ENDS


DSEG	SEGMENT PARA PUBLIC 'DATA'

strng	db	"01.23",0dh,"45.67",08h,"8",0ch,"abc",0ah,"def"
	db	0ah,0dh,"ghi",0ah,0dh,"jkl",0

CurXY	DW	0			;row,,col of cur virtual position
CurPhys	DW	0			;row,,col of cur phys position
CurMode	db	1			;default to mode 1 on startup
CurChar	db	0			;space for char being written
CurPage	db	0			;current video page on entry
CurState db	0			;current state of input char processing

Mode	ModeDesc	3 DUP(<1,3,10,0,0>,<2,1,5,4,0>,<3,1,3,0,0>)
					;describe layout for each of
					; the three display sizes
BitMask	dw	0
ft_ptr	dd	0


;	The following table describes the actions and next-states
;	that result when a token is given to BBL.  There are four 
;	states (rows) and six token types (columns) in the table 
;	(in version 1).  The table is addressed as a matrix ... 
;	State[Current_State,Type_of_Incoming_Token], so the number of
;	token types is needed to compute the address for a given
;	(state, toktype) entry.

States	StateDesc	<ShowChar,S$Init>,<ShowChar,S$Init>,<Move,S$Init> ;state 0
	StateDesc		<Noop,S$HP>,<Noop,S$VP>,<Noop,S$SM>	
	StateDesc	<Beep,S$Init>,<VertPos,S$Init>,<Beep,S$Init>	;state 1
	StateDesc		<Beep,S$Init>,<Beep,S$Init>,<Beep,S$Init>
	StateDesc	<Beep,S$Init>,<HorPos,S$Init>,<Beep,S$Init>	;state 2
	StateDesc		<Beep,S$Init>,<Beep,S$Init>,<Beep,S$Init>
	StateDesc	<Beep,S$Init>,<SetMode,S$Init>,<Beep,S$Init>	;state 3
	StateDesc		<Beep,S$Init>,<Beep,S$Init>,<Beep,S$Init>

;	Now set up the table with default values
;	Bytes contain token type of chars <even,,odd>, with token type
;	stored as a 4-bit nibble

TC	db	128 dup(0)	; <TT$NOND<<4 + TT$NOND> ))

;	Next define the exceptions ... these all happen at assembly time

NDT	TokDef	<FFChar,TT$CM>
	irp	x,<'0','1','2','3','4','5','6','7','8','9'>
	TokDef	<x,TT$Dig>		
	endm
	TokDef	<BSCHar,TT$CM>
	TokDef	<LFChar,TT$CM>
	TokDef	<CRChar,TT$CM>
	TokDef	<HTChar,TT$HP>
	TokDef	<VTChar,TT$VP>
	TokDef	<SOChar,TT$SM>
NumTokDefs = ($-NDT)/SIZE TokDef

DSEG	ENDS

CSeg	SEGMENT	PARA PUBLIC 'CODE'
	ASSUME	CS:CSeg,DS:DSeg, SS:Stak

BLDemo 	PROC	FAR

	push	ds
	sub	ax,ax
	push	ax
	mov	ax,dseg
	mov	ds,ax

	call	FillTT			;init token table with non-defaults
	call	SetFontPtr		;point to 8x8 font

	lea	si,strng		;point to string
Lup:	mov	dl,[si]			;get next char
	inc	si			;  and prepare for following
	cmp	dl,0			;is it the NULL?
	 je	Done			;yes ... done
	push	si			;save the only thing we care about
	call	ProcessChar		;no ... process it
	pop	si
	jmp	Lup
Done:	ret
BLDemo	endp

SetFontPtr PROC	NEAR
	mov	ax,1130h		;get char font address
	mov	bh,03h
	int	10h
	mov	word ptr ft_ptr,bp	;save seg:off
	mov	word ptr ft_ptr+2,es
	ret
SetFontPtr	ENDP

ProcessChar PROC NEAR
;	Process the character in dl
;	Preserve nothing but leave char in dl and pointer
;	 struc describing this mode upon dispatch
	mov	al,CurMode		; for this mode
	mov	bl,SIZE ModeDesc	;size of mode record
	mul	bl			
	lea	si,Mode
	add	si,ax			;si now points to the right block

	mov	CurChar,dl		;keep this char
	xor	bh,bh			;clear high byte for addressing
	mov	bl,dl			;get char & compute token-class
	shr	bl,1			; table address
	mov	ch,TC[bx]		;get the byte containing the class
	 jc	TCxtr			;odd-valued chars are in right nibble
	mov	cl,4
	shr	ch,cl
TCxtr:	and	ch,0fh			;mask out that nibble
					;ch contains the token class of this char
	xor	ax,ax			;now compute the addr of the state-table
	mov	al,CurState		; entry for the current state
	mov	bl,NumTokTyp		; matrix address is
	mul	bl			; States + (CurState*NumTokTypes
	add	al,ch			;            + TokenType) 
	mov	bl,SIZE StateDesc
	mul	bl			;  [adjusted for size of entries]
	mov	bx,ax
	lea	bx,States[bx]
	mov	al,[bx].NextState	;get next state from this transition
	mov	CurState,al
	mov	bx,[bx].ActionProc
	jmp	bx			;and act on this token/state
					; dl still contains input char

ProcessChar ENDP


Beep	PROC	NEAR
	mov	ax,0e07h		;write a bell in teletype mode
	xor	bx,bx
	int	10h
	ret
Beep	ENDP

Noop	PROC	NEAR
	ret
Noop	ENDP

VertPos	PROC	NEAR
	call	Digit			;convert char in dl to integer in dl
	mov	cl,[si].MaxRows	
	cmp	dl,cl			;too large for this mode?
	 jle	VPSet			; no ..
	xchg	dl,cl			; yes ... set to max
VPSet:	mov	BYTE PTR CurXY+1,dl	;store into row-byte of position word
	ret
VertPos	ENDP

HorPos	PROC	NEAR
	call	Digit			;convert char in dl to integer in dl
	mov	cl,[si].MaxCols
	cmp	dl,cl			;too large for this mode?
	 jle	HPSet			; no ...
	xchg	dl,cl			; yes ... set to max
HPSet:	mov	BYTE PTR CurXY,dl	;store into col-byte of position word
	ret
HorPos	ENDP

SetMode	PROC	NEAR
	call	Digit			;convert char in dl to integer in dl
	mov	CurMode,dl		;store into current mode byte
	call	FF			;clear screen & reset CurXY
	ret
SetMode	ENDP

Digit	PROC	NEAR
	sub	dl,'0'
	ret
Digit	ENDP

Move	PROC	NEAR
; Assumptions:
; on entry, dl contains character to display
; no regs will have been preserved upon exit
; si points to struc that describes the display parameters in this mode
; Preserves: nothing

	cmp	dl,CRChar		;is CurChar a CR?
	 je	CR			; yes
	cmp	dl,LFChar		;no, is it a LF?
	 je	LF			; yes
	cmp	dl,BSChar		;no, is it a BackSpace?
	 je	BS			; yes
	cmp	dl,FFChar		;no, is it a FF?
	 jne	Beep			; no -- beep & ignore

					;yes, do screen clear
FF:	mov	ah,0fh			;get current mode
	int	10h			; mode left in al
	mov	ah,00h			;where we use it now
	int	10h			;reset in same mode clears
	xor	ax,ax			;reset our virtual pointer
	mov	CurXY,ax
	ret

CR:	mov	ax,CurXY		;reset pointer to col 0
	xor	al,al
	mov	CurXY,ax
	ret

BS:	mov	ax,CurXY		;decrement virtual col position by 1
	dec	al
	mov	CurXY,ax
	ret

LF:	mov	dx,CurXY		;inc virtual row if we can
	inc	dh
	cmp	dh,[si].MaxRows		;can we?
	 jl	LFDone			; yes! Done
	call	ScrollUp		;no -- scroll up screen
	ret				;note that we haven't changed CurXY
LFDone:	mov	CurXY,dx		;but we did here
	ret
ScrollUp:
	mov	ch,[si].RowOff		;rows down to top of scroll region
	mov	cl,[si].ColOff		;cols to left side of scroll region

	mov	al,[si].MaxRows		;compute bottom addr of scroll region
	xor	ah,ah
	mov	bl,CharHeight
	mul	bl
	mov	bl,[si].mag
	mul	bl
	mov	dh,al
	add	dh,ch			;dh now has addr of last row of scroll reg

	mov	al,[si].MaxCols		;compute right col of scroll reg
	xor	ah,ah
	mov	bl,CharWIdth
	mul	bl
	mov	bl,[si].mag
	mul	bl
	mov	dl,al
	add	dl,cl			;dl now has addr of las col of scroll reg

	mov	al,CharHeight		;finally, compute scroll distance
	xor	ah,ah
	mov	bl,[si].Mag
	mul	bl			;al now has # rows to scroll
	mov	bh,CharAttr
	mov	ah,06h			;function 6 scrolls up
	int	10h			;scroll that region
	ret
Move	ENDP

ShowChar PROC	NEAR
; Assumptions:
; si points to struc that describes the display parameters in this mode
; Preserves: nothing
	mov	dx,CurXY		;see if there's room on this line
	cmp	dl,[si].MaxCols
	 jl	Proceed			;yes, go on with it
	call	CR			;no, move back
	call	LF			; and down

Proceed:
	mov	ah,0fh			;get current video page
	int	10h
	xor	bl,bl			;[hdt] zero for now 
	mov	CurPage,bl		;and save for future use

	mov	al,CurChar		;now get pointer to our char bit pattern
	mov	bl,CharWidth
	mul	bl
	les	bx,ft_ptr
	add	bx,ax
	mov	di,bx			;es:di points to the pattern

 	mov	dx,CurXY		;compute phys row/col starting position
	mov	al,dh			;first, row position
	mov	bl,CharHeight
	mul	bl
	mov	bl,[si].Mag
	mul	bl
	add	al,[si].RowOff
	mov	dh,al			;store back in our active pointer
	mov	al,dl			;now compute col position
	mov	bl,CharWidth
	mul	bl
	mov	bl,[si].Mag
	mul	bl
	add	al,[si].ColOff
	mov	dl,al			;store col back into pointer

	mov	cx,CharHeight		;set up counter to count down rows
					;of char's pixel pattern

; assumptions in this section:
; si points to descriptor block for this mode
; es:di points to next row of pixels for this character
; dx contains row,,col (dh,,dl) of PHYSICAL location to write next on screen

RowLoop:				;loop over all the rows of the pattern
	push	cx			;save row counter
					;get a word from the pattern ...
	mov	bp,word ptr es:[di]	; even if we use only 8 bits
	add	di,CharWidth/8		;and bump the pointer appropriately

	xor	cx,cx
	mov	cl,si.[mag]		;display each row "mag" times
					;for magnification effect
MagLoop:
	push	cx			;save magnification count
					;# of screen rows written with this
					; character pixel row
	push	dx			;save phys screen position

	mov	bx,0080h		;starting mask for copying this row
	mov	BitMask,bx

	mov	cx,CharWidth		;set up counter to copy pixels

ColLoop:
	push	cx			;save column (pixel) counter

	mov	bh,CurPage		;set position on page
	mov	ah,02h			; dx already points to position
	int	10h

	mov	bx,BitMask		;get the mask & check the char
	test	bp,bx			;now is the pixel bit on?
	 jnz	LoadBlock		; yes
	mov	al,' '			;no ... write a blank
	jmp	DispIt			; go do it
LoadBlock:
	mov	al,DispChar		;write the block char
DispIt:	
	ror	bx,1			;ready to mask the next pixel
	mov	BitMask,bx

					;write out the designated disp char
	mov	ah,0Ah			; al already has char to write
	mov	bh,CurPage		; to this video page
	mov	bl,CharAttr		; with appropriate video attribute
	xor	cx,cx
	mov	cl,[si].Mag		; with appropriate magnification for
	int	10h			;  this BL mode

	add	dl,[si].Mag		;update col position
	pop	cx			;retrieve our pixel-width counter
	loop	ColLoop			;and process next pixel, if any

	pop	dx			;done with that row.  Restore phys
	inc	dh			; postion and update phys row
	pop	cx			;and repeat "mag" times
	loop	MagLoop

	pop	cx			;retrieve counter of # of rows of pattern
	loop	RowLoop			;and process next row of pattern
	mov	cx,CurXY		;note change in virtual col
	inc	cl
	mov	CurXY,cx
	ret

ShowChar ENDP

;	The following code gets executed once during startup to fill in
;	the non-default table values from the non-default token table 

FillTT	PROC	NEAR
	mov	cx,NumTokDefs
	xor	bx,bx
	lea	si,NDT			;point to the non-def token table

FillLoop:
	mov	al,[si][bx].char	;get char
	mov	dl,[si][bx].ctype	;and its type
	shr	al,1			;compute address of char in token class table
	jc	StoreType		;if char was even, goes to left half
	shl	dx,1			; of the byte we're ready to write
	shl	dx,1
	shl	dx,1
	shl	dx,1
StoreType:
	xor	ah,ah
	mov	di,ax			;ready to point to char in TokenClass table
	xor	BYTE PTR TC[di],dl
	add	bx,SIZE TokDef
	loop	FillLoop

	ret
FILLTT	ENDP

CSeg	ENDS

	END	BLDemo
