(* streams.ml: translation of streams *)

#open "const";;
#open "syntax";;
#open "prim";;
#open "lambda";;
#open "match";;
#open "tr_env";;

(* The following constants must be kept in sync with ../lib/stream.ml *)

let sempty_tag = ConstrRegular(0,5)
and scons_tag  = ConstrRegular(1,5)
and sapp_tag   = ConstrRegular(2,5)
and sfunc_tag  = ConstrRegular(4,5)
and not_found_tag = 20                  (* Revise!!! *)
;;

(* Translation of stream expressions *)

let translate_stream translate_expr env stream_comp_list =
  let rec transl_stream env = function
    [] ->
      Lconst(SCblock(sempty_tag, []))
  | [Znonterm e] ->
      translate_expr env e
  | component :: rest ->
      let tag =
        match component with Zterm _ -> scons_tag | Znonterm _ -> sapp_tag in
      let e =
        match component with Zterm e -> e | Znonterm e -> e in
      Lprim(Pmakeblock(sfunc_tag, [
        [Lfunction(Lprim(Pmakeblock(ConstrRegular(tag,5)),
                         [translate_expr (Treserved env) e;
                          translate_stream (Treserved env) rest]));
         Lconst(const_unit)] in
  translate_stream env stream_comp_list
;;

(* Translation of stream parsers *)

let stream_oper name =
  Lprim(Pget_global {qual="stream"; id=name}, [])
;;

let stream_raise name =
  Lprim(Praise,
        [Lconst(SCblock(ConstrExtensible {qual="stream"; id=name}, []))])
;;

let raise_not_found = stream_raise "Not_found"
and raise_syntax_error = stream_raise "Syntax_error"
;;

let catch_not_found() =
  Lifthenelse(
    Lprim(Ptest Peq_test,
          [Lprim(Ptagof, [Lvar 0]); Lconst(SCatom(ACint tag_not_found))]),
    Lstaticfail,
    Lprim(Praise, [Lvar 0]))
;;

let rec divide_term_parsing = function
    (Ztermpat pat :: spatl, act) :: rest ->
      let (pat_case_list, parsing) = divide_term_parsing rest in
        (pat, (spatl, act)) :: pat_case_list, parsing
  | parsing ->
        ([], parsing)
;;

let access_stream (* env *) =
  translate_access "%stream" (* env *)
;;

let translate_parser translate_expr env loc case_list =

  let env = Tenv(["%stream", Path_root], env) in

  let rec transl_inner env (patl, act) =
    match patl with
      [] ->
        translate_expr env act
    | Ztermpat pat :: rest ->
        let (new_env, add_lets) = add_pat_to_env env pat in
          Llet([Lapply(stream_oper "stream_get", [access_stream env])],
               translate_matching
                 loc
                 (fun tsb -> raise_syntax_error)
                 [[pat], add_lets(transl_inner new_env (rest,act))])
    | Znontermpat(parsexpr, pat) :: rest ->
        let (new_env, add_lets) = add_pat_to_env env pat in
          Llet([Lapply(stream_oper "parser_require",
                       [translate_expr env parsexpr; access_stream env])],
               translate_matching
                 loc
                 (fun tsb -> raise_syntax_error)
                 [[pat], add_lets(transl_inner new_env (rest,act))]) in

  let rec transl_top parsing =
    match parsing with
      (Ltermpat _ :: _, _) :: _ ->
        let translate_line (pat, case) =
          let (new_env, add_lets) = add_pat_to_env env pat in
            ([pat],
             add_lets(Lsequence(Lapply(stream_oper "stream_junk",
                                                  [access_stream new_env]),
                                transl_inner new_env case))) in
        begin match divide_term_parsing parsing with
          (pat_case_list, []) ->
            Llet([Lapply(stream_oper "stream_peek", [access_stream env])],
                 translate_matching
                   loc
                   (fun tsb -> raise_not_found)
                   (map translate_line pat_case_list))
        | (pat_case_list, rest) ->
            Lstatichandle(
              Llet(
                [Lhandle(Lapply(stream_oper "stream_peek", [access_stream env]),
                         catch_not_found())],
                 translate_matching
                   loc
                   (fun tsb -> Lstaticfail)
                   (map translate_line pat_case_list)),
              transl_top rest)
        end
    | (Lnontermpat(parsexpr, pat) :: spatl, act) :: [] ->
        let (new_env, add_lets) = add_pat_to_env env pat in
          Llet([Lapply(parsexpr, [access_stream env])],
               translate_matching
                 loc
                 (fun tsb -> raise_not_found)
                 [[pat], add_lets(transl_inner new_env (spatl,act))]
    | (Lnontermpat(parsexpr, pat) :: spatl, act) :: rest ->
        let (new_env, add_lets) = add_pat_to_env env pat in
          Lstatichandle(
            Llet([Lhandle(Lapply(parsexpr, [access_stream env]),
                          catch_not_found())],
                 translate_matching
                   loc
                   (fun tsb -> Lstaticfail)
                   [[pat], add_lets(transl_inner new_env (spatl,act))]),
            transl_top rest)
    | [] ->
        raise_not_found in

  Lfunction(transl_top case_list)
;;
