//
// DLGDEMO3 - Notepad Clone #3 demonstrating use of Win 3.1 Common Dialogs
// Copyright (C) 1992 Ray Duncan
// Ziff Davis Publishing Co. * PC Magazine
//

#define WIN31
#define dim(x) (sizeof(x) / sizeof(x[0]))   // returns no. of elements
#define EXENAMESIZE 256                     // max length of path+filename
#define BUFSIZE 65520                       // max length of file data

#include "string.h"
#include "windows.h"
#include "commdlg.h"
#include "dlgdemo3.h"

HANDLE hInst;                               // module instance handle
HWND hFrame;                                // handle for frame window
HWND hEdit;                                 // handle for edit window
char szFileName[EXENAMESIZE+1];             // name of current file 
char szTemp[EXENAMESIZE+1];                 // filename scratch buffer
int hFile;                                  // handle for current file
HANDLE hBuff;                               // handle for file I/O buffer
LPSTR lpBuff;                               // far pointer to file buffer
DWORD dwColor = RGB(0, 0, 0);               // user-selected colorref
            
                                            // for FindText & ReplaceText
FINDREPLACE fr;                             // common dialog data structure
HWND hFindDlg = (HWND) 0;                   // nomodal dialog handle
char szFind[256];                           // text to find
char szReplace[256];                        // text to replace

char szShortAppName[] = "DlgDemo";          // short application name
char szAppName[] = "Common Dialog Demo #3"; // long application name
char szMenuName[] = "DlgDemoMenu";          // name of menu resource
char szDefName[] = "UNTITLED";              // default filename
char szDefExt[] = "TXT";                    // default extension

char *szFilter[] = {                        // filters for Open and
    "ASCII Text (*.TXT)", "*.TXT",          // SaveAs common dialogs
    "All Files (*.*)", "*.*",
    "" };

struct decodeWord {                         // structure associates
    WORD Code;                              // messages or menu IDs
    LONG (*Fxn)(HWND, WORD, WORD, LONG); }; // with a function

//
// Table of window messages supported by FrameWndProc()
// and the functions which correspond to each message.
//
struct decodeWord messages[] = {
    0, DoFindReplace,
    WM_CREATE, DoCreate,
    WM_INITMENU, DoInitMenu,
    WM_SETFOCUS, DoSetFocus,
    WM_SIZE, DoSize,
    WM_COMMAND, DoCommand,
    WM_DESTROY, DoDestroy, 
    WM_CTLCOLOR, DoSetColor, } ;

//
// Table of menubar item IDs and their corresponding functions.
//
struct decodeWord menuitems[] = {
    IDM_NEW, DoMenuNew,
    IDM_OPEN, DoMenuOpen,
    IDM_SAVE, DoMenuSave,
    IDM_SAVEAS, DoMenuSaveAs,
    IDM_EXIT, DoMenuExit, 
    IDM_UNDO, DoMenuUndo,
    IDM_CUT, DoMenuCut,
    IDM_COPY, DoMenuCopy,
    IDM_PASTE, DoMenuPaste,
    IDM_DELETE, DoMenuDelete, 
    IDM_FIND, DoMenuFind,
    IDM_REPLACE, DoMenuReplace,
    IDM_FONT, DoMenuFont,
    IDM_COLOR, DoMenuColor, } ;

//
// WinMain -- entry point for this application from Windows.
//
int PASCAL WinMain(HANDLE hInstance,
    HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;

    hInst = hInstance;                      // save this instance handle

    if(!hPrevInstance)                      // if first instance,
        if(!InitApplication(hInstance))     // register window class
        {
            MessageBox(hFrame, "Can't initialize application!", szAppName, 
                MB_ICONSTOP|MB_OK);
            return(FALSE);
        }

    if(!InitInstance(hInstance, nCmdShow))  // create this instance's window
    {
        MessageBox(hFrame, "Can't initialize instance!", szAppName, 
            MB_ICONSTOP|MB_OK);
        return(FALSE);
    }

    while(GetMessage(&msg, NULL, 0, 0))     // while message != WM_QUIT
    {
        if((hFindDlg == 0) || !IsDialogMessage(hFindDlg, &msg))
        {
            TranslateMessage(&msg);         // translate virtual key codes
            DispatchMessage(&msg);          // dispatch message to window
        }
    }

    TermInstance(hInstance);                // clean up for this instance
    return(msg.wParam);                     // return code = WM_QUIT value
}

//
// InitApplication --- global initialization code for this application.
//
BOOL InitApplication(HANDLE hInstance)
{
    WNDCLASS  wc;

    // set parameters for frame window class
    wc.style = CS_HREDRAW|CS_VREDRAW;       // class style
    wc.lpfnWndProc = FrameWndProc;          // class callback function
    wc.cbClsExtra = 0;                      // extra per-class data
    wc.cbWndExtra = 0;                      // extra per-window data
    wc.hInstance = hInstance;               // handle of class owner
    wc.hIcon = LoadIcon(hInst, "DlgDemoIcon");      // application icon
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);       // default cursor
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); // background color 
    wc.lpszMenuName =  szMenuName;          // name of menu resource
    wc.lpszClassName = szShortAppName;      // name of window class

    return(RegisterClass(&wc));             // register class, return status
}

//
// InitInstance --- instance initialization code for this application.
//
BOOL InitInstance(HANDLE hInstance, WORD nCmdShow)
{
    hFrame = CreateWindow(                  // create frame window
        szShortAppName,                     // window class name
        szAppName,                          // text for title bar
        WS_OVERLAPPEDWINDOW,                // window style
        CW_USEDEFAULT, CW_USEDEFAULT,       // default position
        CW_USEDEFAULT, CW_USEDEFAULT,       // default size
        NULL,                               // no parent window
        NULL,                               // use class default menu
        hInstance,                          // window owner
        NULL);                              // unused pointer

    if(!hFrame) return(FALSE);              // error, can't create window

    hBuff = GlobalAlloc(GMEM_MOVEABLE, BUFSIZE);    // allocate memory
    if(!hBuff)                                      // abort if no memory
    {
        MessageBox(hFrame, "Can't allocate memory!", szAppName, 
            MB_ICONSTOP|MB_OK);
        return(0);
    }

    lpBuff = GlobalLock(hBuff);             // get far pointer to memory

    // register message for FindText() and ReplaceText() common dialogs
    messages[0].Code = RegisterWindowMessage((LPSTR) FINDMSGSTRING);

    ShowWindow(hFrame, nCmdShow);           // make frame window visible
    UpdateWindow(hFrame);                   // force WM_PAINT message
    SendMessage(hFrame, WM_COMMAND,         // force new (empty) file by 
        IDM_NEW, 0L);                       // simulating File-New command
    return(TRUE);                           // return success flag
}

//
// TermInstance -- instance termination code for this application.
//
BOOL TermInstance(HANDLE hinstance)
{
    GlobalUnlock(hBuff);                    // unlock the memory buffer
    GlobalFree(hBuff);                      // release the buffer
    return(TRUE);                           // return success flag
}

//
// FrameWndProc --- callback function for application frame window.
//
LONG FAR PASCAL FrameWndProc(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    int i;                                  // scratch variable

    for(i = 0; i < dim(messages); i++)      // decode window message and
    {                                       // run corresponding function
        if(wMsg == messages[i].Code)
            return((*messages[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoCreate -- process WM_CREATE message for frame window by
// creating a multiline edit control that will fill the frame window.
// 
LONG DoCreate(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    hEdit = CreateWindow("edit",            // class name
                NULL,                       // text for title bar 
                WS_CHILD|WS_VISIBLE|WS_HSCROLL|     
                WS_VSCROLL|WS_BORDER|ES_LEFT|ES_MULTILINE| 
                ES_AUTOHSCROLL|ES_AUTOVSCROLL,  // window style
                0, 0,                       // window position
                0, 0,                       // window size
                hWnd,                       // parent window
                IDE_MLE,                    // edit control ID
                hInst,                      // window owner
                NULL);                      // unused pointer
    return(0);
}

//
// DoSetFocus -- process WM_SETFOCUS message for frame window.
// 
LONG DoSetFocus(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SetFocus(hEdit);                        // toss the focus to 
    return(0);                              // multiline edit control
}

//
// DoSize -- process WM_SIZE message for frame window by resizing
// the multiline edit control to completely fill the client area.
// 
LONG DoSize(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    MoveWindow(hEdit, 0, 0, LOWORD(lParam), HIWORD(lParam), TRUE);
    return(0);
}

//
// DoDestroy -- process WM_DESTROY message for frame window.
// 
LONG DoDestroy(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    PostQuitMessage(0);                     // force WM_QUIT message to
    return(0);                              // terminate the event loop
}

//
// DoSetColor -- process WM_CTLCOLOR message for edit window.
// 
LONG DoSetColor(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SetTextColor((HDC) wParam, dwColor);    // set text color from results
    return(GetStockObject(WHITE_BRUSH));    // return background brush handle
}

//
// DoFindReplace -- process WM_FINDMSGSTRING message from nonmodal
// FindText() AND ReplaceText() common dialogs.  Dispatch DoFind()
// or DoReplace() according to flags in FINDREPLACE structure.
//
LONG DoFindReplace(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    if(fr.Flags & FR_DIALOGTERM)            // is dialog being destroyed?
    {
        hFindDlg = 0;                       // yes, reset window handle
        return(0);                          // allowing another dialog
    }                                       // be created

    if(fr.Flags & FR_FINDNEXT)              // no, perform find or replace
        return(DoFind(hWnd, wMsg, wParam, lParam));
    else if(fr.Flags & (FR_REPLACE | FR_REPLACEALL))
        return(DoReplace(hWnd, wMsg, wParam, lParam));
    else
        return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoFind -- process find command from modeless FindText dialog
//
LONG DoFind(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    if(!FindString())                       // search for string
        MessageBox(hFrame,                  // complain if no match
            "Can't find string!", szAppName, MB_ICONSTOP | MB_OK);
    SetFocus(hEdit);                        // restore input focus
    return(0);
}

//
// DoReplace -- process replace or replace-all command from 
// modeless FindReplace dialog
//
LONG DoReplace(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    int i = 0;                              // replacement counter
    char buff[80];                          // scratch buffer

    if(fr.Flags & FR_REPLACEALL)            // replace multiple?
    {
        while(FindString())                 // yes, while another match 
        {                                   // exists, replace it
            SendMessage(hEdit, EM_REPLACESEL, 0, (LPSTR) szReplace);
            i++;                            // and count replacements   
            
        }

        wsprintf(buff, "%d replacements made.", i);
        MessageBox(hFrame,                  // show counter to user
                buff, szAppName, MB_ICONEXCLAMATION | MB_OK);

    }
    else                                    // this is single replace
    {
        if(!FindString())                   // search for string
            MessageBox(hFrame,              // complain if no match
                "Can't find string!", szAppName, MB_ICONSTOP | MB_OK);
        else                                // otherwise replace it
            SendMessage(hEdit, EM_REPLACESEL, 0, (LPSTR) szReplace);
    }

    SetFocus(hEdit);                        // restore input focus
    return(0);
}

//
// FindString -- search edit control from current selection point for
// the string in szFind[].  If string is present, set selection to include
// match and return TRUE, otherwise return FALSE.
//
BOOL FindString(VOID)
{
    LONG sel;                               // current selection
    int begSel, endSel, cText, cSch;        // scratch variables
    int schDir = 1;                         // search direction
    PSTR pText, pWork;                      // scratch pointers
    HANDLE hText;                           // local heap handle
    int (*matcher)(PSTR, PSTR, WORD);       // pointer to match function

    // get handle for memory block, convert to pointer, get text length
    hText = (HANDLE) SendMessage(hEdit, EM_GETHANDLE, 0, 0L);
    pText = LocalLock(hText);
    cText = (WORD) SendMessage (hEdit, WM_GETTEXTLENGTH, 0, 0L);

    if(cText == 0) return(FALSE);           // bail out if no text at all

    // get offset of beginning and end of current selection
    sel = SendMessage(hEdit, EM_GETSEL, 0, 0L);
    begSel = LOWORD(sel);
    endSel = HIWORD(sel);

    // set search direction from flags in FINDREPLACE structure
    schDir = fr.Flags & FR_DOWN ? 1 : -1 ;  

    // select case-sensitive or case-insensitive match function
    matcher = fr.Flags & FR_MATCHCASE ? strncmp : strnicmp;

    // calculate search starting point within text block
    pWork = pText + begSel + schDir;    

    // calculate length of text to search
    cSch = schDir<1 ? begSel : max(cText-begSel+1-strlen(szFind), 0);

    while(cSch > 0)                         // search until match found
    {                                       // or text is exhausted
        if((*matcher)(pWork, szFind, strlen(szFind)) == 0)
        {
            LocalUnlock(hText);             // found a match, select it
            begSel = pWork - pText;
            endSel = begSel + strlen(szFind);
            SendMessage(hEdit, EM_SETSEL, 0, MAKELONG(begSel, endSel));
            return(TRUE);                   // return success flag
        }

        cSch--;                             // no match yet, adjust text
        pWork += schDir;                    // address & length remaining
    }

    LocalUnlock(hText);                     // unlock edit control text
    return(FALSE);                          // return match failed flag
}

//
// DoInitMenu - initialize the items on menu bar according to the 
// state of the multiline edit control.  If nothing is selected, the 
// edit-cut/copy/delete items are greyed out.  If nothing has 
// been changed since the last file read or write, the file-save item
// is greyed out.  If no text is in the clipboard, the edit-paste
// item is greyed out.
//
LONG DoInitMenu(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    LONG selection;
    
    selection = SendMessage(hEdit, EM_GETSEL, 0, 0);

    if(HIWORD(selection) != LOWORD(selection))          // set cut/copy/
    {                                                   // delete status
        EnableMenuItem(wParam, IDM_CUT, MF_ENABLED);    
        EnableMenuItem(wParam, IDM_COPY, MF_ENABLED);   
        EnableMenuItem(wParam, IDM_DELETE, MF_ENABLED); 
    }
    else
    {
        EnableMenuItem(wParam, IDM_CUT, MF_GRAYED); 
        EnableMenuItem(wParam, IDM_COPY, MF_GRAYED);    
        EnableMenuItem(wParam, IDM_DELETE, MF_GRAYED);  
    }

    if(SendMessage(hEdit, EM_CANUNDO, 0, 0))            // set undo status
        EnableMenuItem(wParam, IDM_UNDO, MF_ENABLED);   
    else
        EnableMenuItem(wParam, IDM_UNDO, MF_GRAYED);    

    if(IsClipboardFormatAvailable(CF_TEXT))             // set paste status
        EnableMenuItem(wParam, IDM_PASTE, MF_ENABLED);  
    else
        EnableMenuItem(wParam, IDM_PASTE, MF_GRAYED);   

    if(SendMessage(hEdit, EM_GETMODIFY, 0, 0))          // set save status
        EnableMenuItem(wParam, IDM_SAVE, MF_ENABLED);   
    else
        EnableMenuItem(wParam, IDM_SAVE, MF_GRAYED);    
    
    // disable find & replace if modeless dialog already active
    if(hFindDlg)                                           
    {                                                   
        EnableMenuItem(wParam, IDM_FIND, MF_GRAYED);    
        EnableMenuItem(wParam, IDM_REPLACE, MF_GRAYED); 
    }                                                   
    else
    {                                                   
        EnableMenuItem(wParam, IDM_FIND, MF_ENABLED);    
        EnableMenuItem(wParam, IDM_REPLACE, MF_ENABLED);    
    }

    return(0);
}

//
// DoCommand -- process WM_COMMAND message for frame window by
// decoding the menubar item with the menuitems[] array, then
// running the corresponding function to process the command.
// 
LONG DoCommand(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    int i;                                  // scratch variable

    if((wParam == IDE_MLE) && (HIWORD(lParam) == EN_ERRSPACE))
    {
        MessageBox(hWnd, "Out of memory!", "Common Dialog Demo", 
            MB_OK | MB_ICONSTOP);
        return(0);
    }

    for(i = 0; i < dim(menuitems); i++)     // decode menu command and
    {                                       // run corresponding function
        if(wParam == menuitems[i].Code)
            return((*menuitems[i].Fxn)(hWnd, wMsg, wParam, lParam));
    }

    return(DefWindowProc(hWnd, wMsg, wParam, lParam));
}

//
// DoMenuNew -- process File-New command from menu bar.
// 
LONG DoMenuNew(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    QueryWriteFile(hEdit);                  // check for dirty buffer
    NewFile(hEdit);                         // empty the text window
    return(0);
}

//
// DoMenuOpen -- process File-Open command from menu bar.
//
LONG DoMenuOpen(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    OPENFILENAME ofn;                       // used by common dialogs

    QueryWriteFile(hEdit);                  // check for dirty buffer

    szTemp[0]  = '\0';                      // init filename buffer

    ofn.lStructSize = sizeof(OPENFILENAME); // length of structure
    ofn.hwndOwner = hWnd;                   // handle for owner window
    ofn.lpstrFilter = szFilter[0];          // address of filter list
    ofn.lpstrCustomFilter = NULL;           // custom filter buffer address
    ofn.nFilterIndex = 1;                   // use *.TXT filter
    ofn.lpstrFile = szTemp;                 // buffer for path+filename
    ofn.nMaxFile = EXENAMESIZE;             // length of buffer
    ofn.lpstrFileTitle = NULL;              // buffer for filename only
    ofn.lpstrInitialDir = NULL;             // initial directory for dialog
    ofn.lpstrTitle = NULL;                  // title for dialog box
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
    ofn.lpstrDefExt = NULL;                 // default extension 

    if(GetOpenFileName(&ofn))               // display open dialog, 
        ReadFile(hEdit);                    // read data from file

    return(0);
}

//
// DoMenuSave -- Process File-Save command from menu bar.  If
// filename is default (UNTITLED), ask user for a different one.
//
LONG DoMenuSave(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    int hFile;                              // scratch file handle

    if(strcmp(szFileName, szDefName) == 0)  // if default name, use SaveAs
        SendMessage(hFrame, WM_COMMAND, IDM_SAVEAS, 0);
    else
        WriteFile(hEdit);                   // otherwise write data to file

    return(0);
}

//
// DoMenuSaveAs -- process File-SaveAs command from menu bar.
// 
LONG DoMenuSaveAs(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    OPENFILENAME ofn;                       // used by common dialogs

    strcpy(szTemp, szFileName);             // get default filename

    ofn.lStructSize = sizeof(OPENFILENAME); // length of structure
    ofn.hwndOwner = hWnd;                   // handle of owner window
    ofn.lpstrFilter = szFilter[0];          // address of filter list
    ofn.lpstrCustomFilter = NULL;           // custom filter buffer address
    ofn.nFilterIndex = 1L;                  // use *.TXT filter
    ofn.lpstrFile = szTemp;                 // buffer for path+filename
    ofn.nMaxFile = EXENAMESIZE;             // size of buffer
    ofn.lpstrFileTitle = NULL;              // buffer for filename only
    ofn.lpstrInitialDir = NULL;             // initial directory for dialog
    ofn.lpstrTitle = NULL;                  // title for dialog box
    ofn.Flags = OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_NOTESTFILECREATE;
    ofn.lpstrDefExt = szDefExt;             // default extension

    if(GetSaveFileName(&ofn))               // display save-as dialog,
        WriteFile(hEdit);                   // write data to file

    return(0);                          
}

//
// DoMenuExit -- process File-Exit command from menu bar.
// 
LONG DoMenuExit(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    QueryWriteFile(hEdit);                  // check for dirty buffer
    SendMessage (hWnd, WM_CLOSE, 0, 0L);    // send window close message    
    return(0);                              // to shut down the app
}

//
// DoMenuUndo -- process Edit-Undo command from menu bar.
// 
LONG DoMenuUndo(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SendMessage(hEdit, WM_UNDO, 0, 0);      // tell the edit control
    return(0);
}

//
// DoMenuCut -- process Edit-Cut command from menu bar.
// 
LONG DoMenuCut(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SendMessage(hEdit, WM_CUT, 0, 0);       // tell the edit control
    return(0);
}

//
// DoMenuCopy -- process Edit-Copy command from menu bar.
// 
LONG DoMenuCopy(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SendMessage(hEdit, WM_COPY, 0, 0);      // tell the edit control
    return(0);
}

//
// DoMenuPaste -- process Edit-Paste command from menu bar.
// 
LONG DoMenuPaste(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SendMessage(hEdit, WM_PASTE, 0, 0);     // tell the edit control
    return(0);
}

//
// DoMenuDelete -- process Edit-Delete command from menu bar.
// 
LONG DoMenuDelete(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    SendMessage(hEdit, WM_CLEAR, 0, 0);     // tell the edit control
    return(0);
}

//
// DoMenuFind -- process Search-Find command from menu bar.
// 
LONG DoMenuFind(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    fr.lStructSize = sizeof(FINDREPLACE);   // length of structure
    fr.hwndOwner = hWnd;                    // handle of owner window
    fr.Flags = FR_DOWN | FR_HIDEWHOLEWORD;  // option flags
    fr.lpstrFindWhat = szFind;              // string to find
    fr.wFindWhatLen = sizeof(szFind);       // length of find buffer

    hFindDlg = FindText(&fr);               // activate FindText dialog
    return(0);
}

//
// DoMenuReplace -- process Search-Replace command from menu bar.
// 
LONG DoMenuReplace(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    fr.lStructSize = sizeof(FINDREPLACE);   // length of structure
    fr.hwndOwner = hWnd;                    // handle of owner window
    fr.Flags = FR_DOWN | FR_HIDEWHOLEWORD;  // option flags
    fr.lpstrFindWhat = szFind;              // string to find
    fr.wFindWhatLen = sizeof(szFind);       // length of find buffer
    fr.lpstrReplaceWith = szReplace;        // replacement string
    fr.wReplaceWithLen = sizeof(szReplace); // length of replace buffer

    hFindDlg = ReplaceText(&fr);            // activate ReplaceText dialog
    return(0);
}

//
// DoMenuFont -- process Font command from menu bar.
// 
LONG DoMenuFont(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    LOGFONT lf;                             // logical font data structure
    CHOOSEFONT chf;                         // font dialog data structure
    HFONT hfont;                            // logical font handle

    chf.lStructSize = sizeof(CHOOSEFONT);   // size of structure    
    chf.hwndOwner = hWnd;                   // window handle of owner
    chf.lpLogFont = &lf;                    // logical font structure
    chf.Flags = CF_SCREENFONTS|CF_EFFECTS;  // option flags
    chf.rgbColors = RGB(0,0,0);             // default color = black

    if(ChooseFont(&chf))                    // display ChooseFont dialog    
    {
        hfont = CreateFontIndirect(&lf);    // save font handle, then send 
        SendMessage(hEdit, WM_SETFONT, hfont, TRUE);  // it to edit control
    }

    return(0);
}

//
// DoMenuColor -- process Color command from menu bar.
// 
LONG DoMenuColor(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
    CHOOSECOLOR chc;                        // color dialog data structure
    DWORD dwCustColors[16];                 // array of custom colorrefs
    int i;                                  // scratch variable

    for(i = 0; i < 16; i++)                 // init custom colors to white
        dwCustColors[i] = RGB(255, 255, 255);

    chc.lStructSize = sizeof(CHOOSECOLOR);  // size of structure
    chc.hwndOwner = hWnd;                   // window handle of owner
    chc.rgbResult = dwColor;                // initial color to select
    chc.lpCustColors = dwCustColors;        // addr of custom color array
    chc.Flags = CC_FULLOPEN | CC_RGBINIT;   // option flags

    if(ChooseColor(&chc))                   // display ChooseColor dialog
    {
        dwColor = chc.rgbResult;            // save new text color
        InvalidateRect(hEdit, NULL, TRUE);  // then force edit window to
        UpdateWindow(hEdit);                // be redrawn with new color
    }

    return(0);
}

//
// NewFile -- set empty text window with default filename.
//
VOID NewFile(HANDLE hEdit)
{
    lpBuff[0] = '\0';                       // empty the edit control
    SetWindowText(hEdit, lpBuff);           // put text into window
    strcpy(szFileName, szDefName);          // set default filename
    SetWindowCaption(szFileName);           // update title bar
    SendMessage(hEdit, EM_SETMODIFY, 0, 0); // clear change flag
}

//
// ReadFile -- read text from specified file to window, close file.
//
VOID ReadFile(HANDLE hEdit)
{
    int length;                             // scratch variable
    int hFile;                              // scratch file handle

    hFile = _lopen(szTemp, OF_READ);        // try and open the file
    if(hFile == -1)                         // bail out if no such file
    {
        MessageBox(hFrame, "Can't open file!", szAppName, MB_ICONSTOP|MB_OK);
        return;
    }

    strcpy(szFileName, szTemp);             // save new filename
    length = _lread(hFile, lpBuff, BUFSIZE);    // read the file
    lpBuff[length] = '\0';                  // make text ASCIIZ
    SetWindowText(hEdit, lpBuff);           // put text into window
    _lclose(hFile);                         // close the file
    SetWindowCaption(szFileName);           // update title bar
    SendMessage(hEdit, EM_SETMODIFY, 0, 0); // clear change flag
}

//
// WriteFile -- write text from specified window to file, close file.
//
VOID WriteFile(HANDLE hEdit)
{
    int length;                             // scratch variable
    int hFile;                              // scratch file handle

    hFile = _lcreat(szTemp, 0);             // try and create the file
    if(hFile == -1)                         // bail out if create failed
    {
        MessageBox(hFrame, "Can't create file!", szAppName, 
            MB_ICONSTOP|MB_OK);
        return;
    }

    strcpy(szFileName, szTemp);             // save new filename
    length = GetWindowTextLength(hEdit);    // chars in edit control
    GetWindowText(hEdit, lpBuff, length+1); // retrieve text
    _lwrite(hFile, lpBuff, length);         // write text to file
    _lclose(hFile);                         // close the file
    SetWindowCaption(szFileName);           // update title bar
    SendMessage(hEdit, EM_SETMODIFY, 0, 0); // clear change flag
}

//
// QueryWriteFile -- check if buffer has been changed, and if
// so prompt the user to write the file to disk.  
//
VOID QueryWriteFile(HANDLE hEdit)
{
    if(SendMessage(hEdit, EM_GETMODIFY, 0, 0))  // was buffer changed?
    {
        if(MessageBox(hFrame, "File has been changed.  Write file?", 
            szAppName, MB_ICONQUESTION|MB_YESNO) == IDYES)
        {
            SendMessage(hFrame, WM_COMMAND, IDM_SAVEAS, 0);
        }
    }
}

//
// SetWindowCaption -- concatenate the filename with the application
// name, then update the frame window's title bar.
//
VOID SetWindowCaption(char * szFilename)
{
    char szTemp[EXENAMESIZE+1];             // filename scratch buffer

    strcpy(szTemp, szAppName);              // get application name
    strcat(szTemp, " - ");                  // add separator
    strcat(szTemp, szFileName);             // add filename
    SetWindowText(hFrame, szTemp);          // put result into title bar
}

