// Song code file (SONG.CPP)
//
// Written 1996, 1997 by Roland Acton - public domain.
//
// This file is part of the Game Music System 1.1 distribution.

#define COMPILING_SONG

#include "globals.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifdef TARGET_WIN95
  #include <dos.h>
#else
  #ifdef TARGET_MSDOS
    #include <dos.h>
  #else
    #include "msonly.h"
  #endif
#endif
#ifdef USING_BORLAND
  #include <alloc.h>
#endif
#ifdef USING_WATCOM
  #include <conio.h>
#endif
#include <assert.h>
#include "block.h"
#include "inst.h"
#include "song.h"
#include "musdrv.h"
#ifdef EDITOR_HOOKS
  #include "intrface.h"
  #include "gengad.h"
  #include "statbar.h"
  #include "specgad1.h"
#endif



// Sets up the song class's internal variables.
void song::set_up_class() {
  int stepper;

  name[0] = 0;
  info_page = (char *) malloc(14);
  memset(info_page, 0, 14);
  pattern_list[0] = 0;
  highest_pattern = 0;
  highest_stereo_left = 3;
  lowest_stereo_right = 4;
  highest_track = 8;
  soundsystem = adlib;
  percussion_mode = NO;
  overall_am_vib.byte = 0; 
  tempo = 130;
  secondary_tempo = 6;
  next_effect_id = 0;
  block_pointer[0] = new block(64);
  highest_block = 0;
  for (stepper = 0;stepper < MAX_INSTRUMENTS;stepper++)
    instrument_pointer[stepper] = new instrument;
// Instrument 0 is a "silent" synth instrument.
  instrument_pointer[0]->has_synth_component = YES;
  instrument_pointer[0]->a_d1.byte = 0xFF;
  instrument_pointer[0]->a_d2.byte = 0xFF;
  instrument_pointer[0]->s_r1.byte = 0xFF;
  instrument_pointer[0]->s_r2.byte = 0xFF;
  position_jump(0);
}



// Returns the song's allocated memory to the system.
void song::shut_down_class() {

  wipe_old_song();
  delete instrument_pointer[0];
  free(info_page);
}



// Sets the position pointers to a new place in the song.
void song::position_jump(unsigned int new_pattern, unsigned int new_line) {

  music_driver::forbid_playing();
  if (new_pattern > highest_pattern)
    new_pattern = highest_pattern; 
  block_jump(pattern_list[new_pattern], new_line);
  current_pattern = new_pattern;
  music_driver::permit_playing();
}



// Sets the position pointers to a new place in the block.
void song::block_jump(unsigned int new_block, unsigned int new_line) {
  static int stepper, stepper2;

  music_driver::forbid_playing();
  if (new_block > highest_block)
    new_block = highest_block;
  for (stepper = highest_track;stepper >= 0;stepper--)
    current_note[stepper] =
      block_pointer[new_block]->track_pointer[stepper];
  if (new_line > block_pointer[new_block]->highest_line)
    new_line = block_pointer[new_block]->highest_line;
  for (stepper = highest_track;stepper >= 0;stepper--)
    for (stepper2 = new_line;stepper2 > 0;stepper2--)
      skip_note(current_note[stepper]);
  current_block = new_block;
  current_line = new_line;
  music_driver::reset_accumulated_ticks();
  music_driver::permit_playing();
}



/* Advance the position pointers to the next line, and possibly to the next
   pattern. */
void song::advance_line() {
  int stepper;

  if (++current_line > block_pointer[current_block]->highest_line) {
#ifdef EDITOR_HOOKS
    if (music_driver::play_block_mode)
      block_jump(current_block);
    else {
#endif
      if (++current_pattern > highest_pattern)
        current_pattern = 0;
      position_jump(current_pattern);
#ifdef EDITOR_HOOKS
    }
#endif
  }
  else
    for (stepper = highest_track;stepper >= 0;stepper--)
      skip_note(current_note[stepper]); 
}



// Diverts to a music subroutine.
void song::start_subroutine(unsigned int subroutine_pattern,
  unsigned int restart_block_on_finish) {

  music_driver::forbid_playing();
  if (!in_subroutine) {
    in_subroutine = YES;
    subroutine_pattern_memory = current_pattern;
    if (restart_block_on_finish)
      subroutine_line_memory = 0;
    else
      subroutine_line_memory = current_line;
  }
  position_jump(subroutine_pattern);
  music_driver::permit_playing();
}



// Cancels the currently playing subroutine.
void song::cancel_subroutine() {

  if (in_subroutine) {
    music_driver::forbid_playing();
    position_jump(subroutine_pattern_memory, subroutine_line_memory);
    in_subroutine = NO;
    music_driver::permit_playing();
  }
}



// Starts an effect in parallel with the music.
unsigned int song::start_effect(unsigned int effect_pattern) {
  effect_info *new_effect;

  music_driver::forbid_playing();
  new_effect = new effect_info;
  new_effect->effect_id = next_effect_id++;
  new_effect->pointers_valid = NO;
  new_effect->jump_pattern = effect_pattern;
  new_effect->next_effect = effect_list;
  effect_list = new_effect;
  music_driver::permit_playing();
  return new_effect->effect_id;
}



// Cancels a previously started effect.
void song::cancel_effect(unsigned int id_to_cancel) {
  effect_info **effect_step;
  effect_info *this_effect;

  music_driver::forbid_playing();
  effect_step = &effect_list;
  while (*effect_step) {
    this_effect = *effect_step;
    if (this_effect->effect_id == id_to_cancel) {
      *effect_step = this_effect->next_effect;
      delete this_effect;
      break;
    }
    effect_step = &(this_effect->next_effect);
  }
  music_driver::permit_playing();
}



// Cancels all active effects.
void song::cancel_all_effects() {
  effect_info *this_effect;
  effect_info *next_effect;

  music_driver::forbid_playing();
  this_effect = effect_list;
  while (this_effect) {
    next_effect = this_effect->next_effect;
    delete this_effect;
    this_effect = next_effect;
  }
  effect_list = 0;
  music_driver::permit_playing();
}



// Removes the currently loaded song from memory, except for instrument zero.
void song::wipe_old_song() {
  int stepper;

  while (highest_block) {
    delete block_pointer[highest_block];
    block_pointer[highest_block] = 0;   // For debugging.
    highest_block--;
  }
  delete block_pointer[0];
  block_pointer[0] = 0;
  for (stepper = MAX_INSTRUMENTS - 1;stepper > 0;stepper--) {
    instrument_pointer[stepper]->name[0] = 0;
    instrument_pointer[stepper]->delete_synth_component();
    instrument_pointer[stepper]->delete_sample_component();
  }
}



// Loads a GMS module from a file, using the given pointer.
gms_function_return_codes song::load_gms(FILE *module) {
  static char tracker_id_reader[20];
  static unsigned int inbyte;
  static unsigned int counter, stepper;
  static char *char_read;
  static unsigned int track_memory[MAX_BLOCKS][MAX_TRACKS];
  static unsigned char *track_data[MAX_BLOCKS * MAX_TRACKS];
  static unsigned int track_counter, track_max;
  static unsigned int track_lines;
  static unsigned int track_start, track_end;
  static unsigned char module_version;
  static unsigned int highest_inst_used;
  static unsigned int track_bytes;
  static unsigned char *old_track;
#ifdef EDITOR_HOOKS
  char inst_error_text[255];
#endif
#ifdef DEBUG_DISPLAYS
  FILE *debug_file;
#endif
// Variables made static for Watcom compiler bug workaround.

#ifdef DEBUG_DISPLAYS
  debug_file = fopen("debugld.gms", "wt");
  fprintf(debug_file, "Loading song...\n");
#endif
  fread(tracker_id_reader, 1, tracker_id_length[GMS_MOD_OFFSET], module);
  if (memcmp(tracker_id_reader, tracker_id_string[GMS_MOD_OFFSET],
    tracker_id_length[GMS_MOD_OFFSET])) {
#ifdef EDITOR_HOOKS
    status_bar_obj.message_output("This is not a GMS module.");
#endif
    return gms_function_failure;
  }
  else {
    module_version = fgetc(module);
    if (module_version <= 1) {
#ifdef EDITOR_HOOKS
      if (module_version == 1)
        status_bar_obj.message_output("This is a GMS 1.1 module.");
      else
        status_bar_obj.message_output("This is a GMS 1.0 module.");
#endif
// Load song info.
      stepper = 0;
      while ((name[stepper++] = fgetc(module)));
      free(info_page);
      info_page = (char *) malloc(784);
      char_read = info_page;
      for (counter = 0;counter < 14;counter++)
        while ((*(char_read++) = fgetc(module)));
      info_page = (char *) realloc(info_page, char_read - info_page);
      overall_am_vib.byte = fgetc(module);
#ifdef EDITOR_HOOKS
      tracks_obj.change_num_tracks(fgetc(module) + 1);    
         // Handles clipboard.
#else
      highest_track = fgetc(module);
#endif
      tempo = fgetc(module);
      secondary_tempo = fgetc(module);
      soundsystem = (sound_cards) fgetc(module);
      highest_stereo_left = fgetc(module);
      if (module_version == 1)
        lowest_stereo_right = fgetc(module);
      else
        lowest_stereo_right = highest_stereo_left + 1;
      if (module_version == 1)
        percussion_mode = fgetc(module);
      else
        percussion_mode = NO;
      wipe_old_song();   // Must do this after change_num_tracks().
// Load instruments.
      if (module_version == 1)
        highest_inst_used = MAX_INSTRUMENTS - 1;
/* Important: changing MAX_INSTRUMENTS after the 1.1 release will break the
   load routine unless this line is updated! */
      else
        highest_inst_used = fgetc(module);
      for (counter = 1;counter <= highest_inst_used;counter++)
        if (do_gmi_load(module, instrument_pointer[counter],
          module_version) == gms_function_failure) {
#ifdef EDITOR_HOOKS
          sprintf(inst_error_text, "WARNING: Insufficient memory to load instrument %d.",
            counter);
          status_bar_obj.message_output(inst_error_text);
#endif
        }
// Load pattern-list data.
      highest_pattern = fgetc(module);
      for (counter = 0;counter <= highest_pattern;counter++)
        pattern_list[counter] = fgetc(module);
// Load blocks.
      highest_block = fgetc(module);
// Read track mappings, create blocks.
#ifdef DEBUG_DISPLAYS
      fprintf(debug_file, "\nReading track mappings...\n");
#endif
      track_max = 0;
      for (counter = 0;counter <= highest_block;counter++) {
#ifdef DEBUG_DISPLAYS
        fprintf(debug_file, "Block %u:   ", counter);
#endif
        block_pointer[counter] = new block(fgetc(module) + 1);
        for (stepper = 0;stepper <= highest_track;stepper++) {
          inbyte = fgetc(module);
          track_memory[counter][stepper] = (inbyte << 8) + fgetc(module);
#ifdef DEBUG_DISPLAYS
          fprintf(debug_file, "%u ", track_memory[counter][stepper]);
#endif
          if (track_memory[counter][stepper] > track_max)
            track_max = track_memory[counter][stepper];
        }
#ifdef DEBUG_DISPLAYS
        fprintf(debug_file, "\n");
#endif
      }
// Read tracks.
#ifdef DEBUG_DISPLAYS
      fprintf(debug_file, "\nReading tracks...\n");
#endif
      for (track_counter = 1;track_counter <= track_max;track_counter++) {
        if (module_version == 0) {
/* GMS 1.0 did not store the number of bytes used by the track, so it must
   be found out the hard way. */
          track_lines = 9999;
          for (counter = 0;counter <= highest_block;counter++)
            for (stepper = 0;stepper <= highest_track;stepper++)
              if (track_memory[counter][stepper] == track_counter)
                track_lines = block_pointer[counter]->highest_line;
          assert(track_lines != 9999);
          track_start = ftell(module);
          for (counter = 0;counter <= track_lines;counter++) {
            if (fgetc(module) & 128)
              if (fgetc(module) & 128)
                do {
                  inbyte = fgetc(module);
                  fgetc(module);
                } while (inbyte & 128);
          }
          track_end = ftell(module);
          fseek(module, track_start, SEEK_SET);
          track_bytes = track_end - track_start;
        }
        else {
          inbyte = fgetc(module);
          track_bytes = (inbyte << 8) + fgetc(module);
        }
#ifdef DEBUG_DISPLAYS
        fprintf(debug_file, "Track %u is %u bytes.\n",
          track_counter, track_bytes);
#endif
        track_data[track_counter] = (unsigned char *) malloc(track_bytes);
        fread(track_data[track_counter], 1, track_bytes, module);
      }
// Map tracks onto blocks.
#ifdef DEBUG_DISPLAYS
      fprintf(debug_file, "\nMapping tracks...\n");
#endif
      for (counter = 0;counter <= highest_block;counter++)
        for (stepper = 0;stepper <= highest_track;stepper++) {
          if (track_memory[counter][stepper]) {
#ifdef DEBUG_DISPLAYS
            fprintf(debug_file, "Block %u track %u is track number %u.\n",
              counter, stepper, track_memory[counter][stepper]);
#endif
            old_track = block_pointer[counter]->swap_track
              (track_data[track_memory[counter][stepper]], stepper);
          }
          else {
#ifdef DEBUG_DISPLAYS
            fprintf(debug_file, "Block %u track %u is blank.\n",
              counter, stepper);
#endif
            old_track = block_pointer[counter]->swap_track(0, stepper);
          }
          free(old_track);
        }
#ifdef EDITOR_HOOKS
      editor_reformat();
#endif
      position_jump(0);
#ifdef DEBUG_DISPLAYS
      fclose(debug_file);
#endif
      return gms_function_success;
    }
    else {
#ifdef EDITOR_HOOKS
      status_bar_obj.message_output("Unknown GMS format. You need a newer version of GMS.");
#endif
      return gms_function_failure;
    }
  }
}



/* Loads the GMI core data from the position in the given file, placing it
   in the given instrument. instrument_version tells the routine which
   instrument format is used by the file. */
gms_function_return_codes song::do_gmi_load(FILE *inst_file,
  instrument *load_inst, unsigned char instrument_version) {
  unsigned int stepper;
#ifdef USING_BORLAND
  signed char huge *mem_grabber;
#else
  signed char *mem_grabber;
#endif

  stepper = 0;
  while ((load_inst->name[stepper++] = fgetc(inst_file)));
  if (instrument_version == 0
    || (instrument_version == 1 && fgetc(inst_file))) {
    load_inst->delete_synth_component();
    load_inst->has_synth_component = YES;
    load_inst->am_vib1.byte = fgetc(inst_file);
    load_inst->scaling_volume1.byte = fgetc(inst_file);
    load_inst->a_d1.byte = fgetc(inst_file);
    load_inst->s_r1.byte = fgetc(inst_file);
    load_inst->wave1.byte = fgetc(inst_file);
    load_inst->am_vib2.byte = fgetc(inst_file);
    load_inst->scaling_volume2.byte = fgetc(inst_file);
    load_inst->a_d2.byte = fgetc(inst_file);
    load_inst->s_r2.byte = fgetc(inst_file);
    load_inst->wave2.byte = fgetc(inst_file);
    load_inst->op1mod_feedback.byte = fgetc(inst_file);
  }
  if (instrument_version == 1 && fgetc(inst_file)) {
    load_inst->delete_sample_component();
    load_inst->has_sample_component = YES;
    load_inst->in_signed_form = YES;
    load_inst->presence = (instrument::presence_types) fgetc(inst_file);
    load_inst->sample_length = fgetc(inst_file) << 24;
    load_inst->sample_length += fgetc(inst_file) << 16;
    load_inst->sample_length += fgetc(inst_file) << 8;
    load_inst->sample_length += fgetc(inst_file);
    load_inst->repeat_start = fgetc(inst_file) << 24;
    load_inst->repeat_start += fgetc(inst_file) << 16;
    load_inst->repeat_start += fgetc(inst_file) << 8;
    load_inst->repeat_start += fgetc(inst_file);
    load_inst->repeat_length = fgetc(inst_file) << 24;
    load_inst->repeat_length += fgetc(inst_file) << 16;
    load_inst->repeat_length += fgetc(inst_file) << 8;
    load_inst->repeat_length += fgetc(inst_file);
    load_inst->sample_freq = fgetc(inst_file) << 8;
    load_inst->sample_freq += fgetc(inst_file);
#ifdef USING_BORLAND
    mem_grabber = (signed char huge *) farmalloc(load_inst->sample_length);
#else
    mem_grabber = (signed char *) malloc(load_inst->sample_length);
#endif
    if (!mem_grabber) {
      load_inst->has_sample_component = NO;
      return gms_function_failure;
    }
#ifdef USING_BORLAND
    load_inst->sample_address_RM = mem_grabber;
#else
    load_inst->sample_address_PM = mem_grabber;
#endif
    fread(mem_grabber, 1, load_inst->sample_length, inst_file);
  }
  return gms_function_success;
}



// Moves the sample components of all instruments into conventional memory.
gms_function_return_codes song::prepare_all_samples() {
  unsigned int stepper;
  int prepare_successful;

  prepare_successful = YES;
  for (stepper = 0;stepper < MAX_INSTRUMENTS;stepper++)
    if (instrument_pointer[stepper]->prepare_sample() == gms_function_failure)
      prepare_successful = NO;
  if (prepare_successful)
    return gms_function_success;
  else
    return gms_function_failure;
}



/* Removes the sample components of all instruments from conventional
   memory. */
void song::unprepare_all_samples() {
  unsigned int stepper;

  for (stepper = 0;stepper < MAX_INSTRUMENTS;stepper++)
    instrument_pointer[stepper]->unprepare_sample();
}
