/*
         This program will generate a three dimensional maze suitable for
    printing on a Hewlett-Packard LaserJet III printer.  A different random
    number seed will produce a different maze.

         Written by on September 20, 1992 by James L. Dean
                                             406 40th Street
                                             New Orleans, LA 70124-1532
*/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include <graph.h>
#include <process.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>
#include <setjmp.h>

typedef struct
          {
            int   x;
            int   y;
          } box_rec;

typedef struct
          {
            long          color;
            unsigned char outside_maze;
            int           x_division_index;
            int           y_division_index;
            float         x;
            float         y;
            float         z;
          } prime_rec;

typedef struct stack_rec_record
          {
            char index_1;
            int  index_2;
          } stack_rec;

typedef struct
          {
            double x;
            double y;
            double z;
          } vertex_rec;

#define TRUE  1
#define FALSE 0

/* printer constants */
#define PIXELS_PER_INCH    300
#define WIDTH_IN_INCHES    7.5
#define LENGTH_IN_INCHES   10.6
#define PIXELS_PER_STROKE  1    /* 300/PIXELS_PER_INCH */
#define NUM_X_BYTES        397  /* (int) (LENGTH_IN_INCHES*PIXELS_PER_INCH/8) */
#define X_BYTE_MAX         396  /* NUM_X_BYTES-1 */
#define NUM_X_PIXELS       3176 /* 8*NUM_X_BYTES */
#define X_PIXEL_MAX        3175 /* NUM_X_PIXELS-1 */
#define NUM_Y_PIXELS       2250 /* (int) (WIDTH_IN_INCHES*PIXELS_PER_INCH) */
#define Y_PIXEL_MAX        2249 /* NUM_Y_PIXELS-1 */
#define ASPECT_RATIO       1.0
#define LOG2_NUM_X_PIXELS  12   /* 2^LOG2_NUM_X_PIXELS >= NUM_X_PIXELS */
#define LOG2_NUM_Y_PIXELS  12   /* 2^LOG2_NUM_Y_PIXELS >= 2*NUM_Y_PIXELS */
#define NUM_COLORS         16777216
                                /* 2^(LOG2_NUM_X_PIXELS+LOG2_NUM_Y_PIXELS) */

typedef struct 
          {
            unsigned char column [NUM_X_BYTES];
          } row_rec;

/* maze constants */
#define WIDTH_OF_ROOM            15  /* millimeters */
#define PIXELS_PER_ROOM         177  /* 10*PIXELS_PER_INCH*WIDTH_OF_ROOM/254 */
#define RESOLUTION                4  /* larger numbers give more detail
                                      but take more time and (virtual) memory */

/* virtual memory constants */
#define NUM_PRIME_BUFFERS         4
#define PRIME_REC_COUNT        1560  /* 32768/sizeof(prime_rec) */
#define NUM_ROW_BUFFERS           2
#define ROW_REC_COUNT           165  /* 32768/sizeof(row_rec) */

static void      adjust_perspective(int,int,double,double,double,double,
                  double);
static void      draw_line_on_page(char **,int,int,int,int);
static void      evaluate_and_transform(double,double,double,double,int,int,
                  double,double,double *,double *,double *,double *,double *,
                  vertex_rec *);
static int       external_to_maze(double,double);
static double    f(double,double);
static void      free_memory(char ***,int,char ***,int,stack_rec **,int,
                  prime_rec ***,int,row_rec ***,int);
static void      generate_maze(int *,char **,char **,int,int,stack_rec *,int,
                  int,int,int,int);
static void      get_cursor(unsigned char *,unsigned char *,unsigned char *,
                  unsigned char *);
static void      hash(int *,int *,int *,int *,int *,int *,int *,int *);
static void      increment(int *,int *,int *,int *,int *,int *,int *,int *);
       int       main(int,char **);
static int       memory_allocated(char ***,int,int,char ***,int,int,
                  stack_rec **,int,prime_rec ***,int,int,row_rec ***,int,int);
static void      plot(int,int,long,double,double,double,double,double);
static prime_rec *prime_addr(long);
static void      pset(int,int,long);
static void      quicksort(long,long);
static void      rearrange(long,long,long *);
static row_rec   *row_addr(int);
static void      set_cursor_position(unsigned char,unsigned char);
static void      set_cursor_size(unsigned char,unsigned char);
static void      set_point_on_page(char **,int,int);
static void      shade(int,int,vertex_rec *,long);
static void      solve_maze(stack_rec *,char **,int *,int *,int,int);
static void      sort_back_to_front(long);
static void      titillate(void);
static void      write_outfile(int *);

static char          **base_page;
static unsigned char black [8] = {128, 64, 32, 16, 8, 4, 2, 1};
static unsigned char cursor_column;
static unsigned char cursor_row;
static unsigned char cursor_start;
static unsigned char cursor_stop;
static int           delta_x [6] [720];
static int           delta_y [6] [720];
static int           max_x;
static int           max_y;
static char          **page;
static char          titillator [4] = { '|', '/', '-', '\\' };
static int           titillator_index;
static long          prime_access [NUM_PRIME_BUFFERS];
static long          prime_access_num;
static prime_rec     **prime_buffer;
static int           prime_buffer_count;
static long          prime_start [NUM_PRIME_BUFFERS];
       FILE          *prime_vm;
static long          row_access [NUM_PRIME_BUFFERS];
static long          row_access_num;
static row_rec       **row_buffer;
static int           row_buffer_count;
static int           row_start [NUM_PRIME_BUFFERS];
       FILE          *row_vm;
static unsigned char white [8] = {127, 191, 223, 239, 247, 251, 253, 254};
static int           x_dot_max;
static int           y_dot_max;

int main(
  int argc,
  char *argv[])
    {
      register int        column_num;
      static   int        fatal_error;
      static   vertex_rec light;
      static   int        max_x_plus_1;
      static   int        max_y_plus_1;
      static   int        num_columns;
      static   long       num_primes;
      static   int        num_rooms_in_maze;
      static   int        num_rows;
      static   int        num_x_divisions;
      static   int        num_x_dots;
      static   int        num_y_divisions;
      static   int        num_y_dots;
      static   int        prime_index;
      static   long       prime_num;
      static   int        r_n [8];
      static   int        r_n_index_1;
      static   int        r_n_index_2;
      static   double     rotation;
      register int        row_index;
      static   int        row_num;
      static   int        seed_length;
      static   stack_rec  *stack;
      static   double     tilt;
      static   double     x_max;
      static   double     x_min;
      static   double     x_prime_max;
      static   double     y_max;
      static   double     y_min;
      static   double     y_prime_max;
      static   double     y_prime_min;
      static   double     z_prime_max;
      static   double     z_prime_min;
  
      fatal_error=FALSE;
      if (argc == 2)
        {
          seed_length=strlen(argv[1]);
          for (r_n_index_1=0;
           ((r_n_index_1 < seed_length) && (r_n_index_1 < 8));
           ++r_n_index_1)
            r_n[r_n_index_1]=(int) (argv[1][r_n_index_1] % 10);
          r_n_index_2=7;
          while (r_n_index_1 > 0)
            {
               r_n_index_1--;
               r_n[r_n_index_2]=r_n[r_n_index_1];
               r_n_index_2--;
            }
          while (r_n_index_2 >= 0)
            {
              r_n[r_n_index_2]=0;
              r_n_index_2--;
            }
          num_columns=2*NUM_X_PIXELS/(3*PIXELS_PER_ROOM)-1;
          num_columns=2*num_columns+1;
          max_x=8*(num_columns/2)+6;
          max_x_plus_1=max_x+1;
          num_x_dots=RESOLUTION*max_x_plus_1;
          x_dot_max=num_x_dots-1;
          num_rows=(int) ((2.0/sqrt(3.0))*ASPECT_RATIO*((double) NUM_Y_PIXELS)
           /((double) PIXELS_PER_ROOM));
          max_y=4*num_rows;
          max_y_plus_1=max_y+1;
          num_y_dots=RESOLUTION*max_y_plus_1;
          y_dot_max=num_y_dots-1;
          num_rooms_in_maze=num_rows*num_columns-(num_columns/2);
          num_x_divisions=num_y_dots+2;
          num_y_divisions=num_x_dots+2;
          num_primes=(long) num_x_divisions;
          num_primes*=((long) num_y_divisions);
          if (memory_allocated(&base_page,max_x_plus_1,max_y_plus_1,&page,
           num_x_dots,num_y_dots,&stack,num_rooms_in_maze,&prime_buffer,
           NUM_PRIME_BUFFERS,PRIME_REC_COUNT,&row_buffer,NUM_ROW_BUFFERS,
           ROW_REC_COUNT))
            {
              get_cursor(&cursor_row,&cursor_column,&cursor_start,&cursor_stop);
              set_cursor_size((unsigned char) 32,(unsigned char) 32);
              titillator_index=0;
              generate_maze(&r_n[0],base_page,page,max_x,max_y,stack,x_dot_max,
               y_dot_max,num_rooms_in_maze,num_columns,num_rows);
              x_min=(double) 1.0;
              x_max=(double) num_x_divisions;
              y_min=(double) 1.0;
              y_max=(double) num_y_divisions;
              light.x=0.0;
              light.y=-1.0;
              light.z=3.0;
              prime_vm=fopen("HP3DMAZE.PAG","w+b");
              prime_buffer_count=0;
              prime_access_num=0l;
              for (prime_num=0l; ((! fatal_error) && (prime_num < num_primes));
               prime_num+=PRIME_REC_COUNT)
                {
                  if (prime_buffer_count < NUM_PRIME_BUFFERS)
                    {
                      prime_access_num++;
                      for (prime_index=0; prime_index < PRIME_REC_COUNT;
                       prime_index++)
                        {
                          (prime_buffer[prime_buffer_count][prime_index]).color
                           =0l;
                          (prime_buffer[prime_buffer_count][
                           prime_index]).outside_maze=(unsigned char) '\0';
                          (prime_buffer[prime_buffer_count][
                           prime_index]).x_division_index=0;
                          (prime_buffer[prime_buffer_count][
                           prime_index]).y_division_index=0;
                          (prime_buffer[prime_buffer_count][prime_index]).x
                           =(float) 0.0;
                          (prime_buffer[prime_buffer_count][prime_index]).y
                           =(float) 0.0;
                          (prime_buffer[prime_buffer_count][prime_index]).z
                           =(float) 0.0;
                        }
                      prime_access[prime_buffer_count]=prime_access_num;
                      prime_start[prime_buffer_count]=prime_num;
                      prime_buffer_count++;
                    }
                  fatal_error=(fwrite(&(prime_buffer[0][0]),
                   sizeof(prime_rec),PRIME_REC_COUNT,prime_vm) != PRIME_REC_COUNT);
                }
              if (fatal_error)
                printf("\n     Fatal error:  not enough disk space.\n");
              else
                {
                  row_vm=fopen("HP3DMAZE.ROW","w+b");
                  row_buffer_count=0;
                  row_access_num=0l;
                  for (row_num=0; 
                   ((! fatal_error) && (row_num < NUM_Y_PIXELS));
                   row_num+=ROW_REC_COUNT)
                    {
                      if (row_buffer_count < NUM_ROW_BUFFERS)
                        {
                          row_access_num++;
                          for (row_index=0; row_index < ROW_REC_COUNT;
                           row_index++)
                            for (column_num=0; column_num < NUM_X_BYTES;
                             column_num++)
                              (row_buffer[row_buffer_count][row_index]).column[
                               column_num]=(unsigned char) '\0';
                          row_access[row_buffer_count]=row_access_num;
                          row_start[row_buffer_count]=row_num;
                          row_buffer_count++;
                        }
                      fatal_error=(fwrite(&(row_buffer[0][0]),
                       sizeof(row_rec),ROW_REC_COUNT,row_vm)
                       != ROW_REC_COUNT);
                    }
                  if (fatal_error)
                    printf("\n     Fatal error:  not enough disk space.\n");
                  else
                    {
                      titillator_index=0;
                      x_min=-1.0;
                      x_max=(double) num_y_dots;
                      y_min=-1.0;
                      y_max=(double) num_x_dots;
                      rotation=0.0;
                      tilt=30.0;
                      light.x=1.5;
                      light.y=-1.0;
                      light.z=2.6;
                      evaluate_and_transform(x_min,x_max,y_min,y_max,
                       num_x_divisions,num_y_divisions,rotation,tilt,
                       &x_prime_max,&y_prime_min,&y_prime_max,&z_prime_min,
                       &z_prime_max,&light);
                      shade(num_x_divisions,num_y_divisions,&light,num_primes);
                      adjust_perspective(num_x_divisions,num_y_divisions,
                       x_prime_max,y_prime_min,y_prime_max,z_prime_min,
                       z_prime_max);
                      sort_back_to_front(num_primes);
                      plot(num_x_divisions,num_y_divisions,num_primes,
                       y_prime_min,y_prime_max,z_prime_min,z_prime_max,
                       ASPECT_RATIO);
                      write_outfile(&fatal_error);
                      set_cursor_position(cursor_row,cursor_column);
                      putchar((int) ' ');
                      if (! fatal_error)
                        printf(
                    "\nIf LPT1: is your Laserjet, \"COPY /B HP3DMAZE LPT1:\".\n");
                      set_cursor_size(cursor_start,cursor_stop);
                      fclose(row_vm);
                      unlink("HP3DMAZE.ROW");
                    }
                  fclose(prime_vm);
                  unlink("HP3DMAZE.PAG");
                }
              free_memory(&base_page,max_y_plus_1,&page,num_y_dots,&stack,
               num_rooms_in_maze,&prime_buffer,NUM_PRIME_BUFFERS,&row_buffer,
               NUM_ROW_BUFFERS);
            }
          else
            {
              fatal_error=TRUE;
              printf("\n     Fatal error:  out of memory.\n");
            }
        }
      else
        {
          printf(
"     This program will generate a three dimensional maze suitable for\n");
          printf(
"printing on a Laserjet printer.  It writes the maze to the file\n");
          printf(
"\"HP3DMAZE\".\n");
          printf(
"     If LPT1: is your Laserjet, \"COPY /B HP3DMAZE LPT1:\" to print\n");
          printf(
"the maze.\n");
          printf(
"     A different random number seed will produce a different maze.\n\n");
          printf(
"     Usage:  HP3DMAZE <random number seed>\n");
          printf(
"     Example:  HP3DMAZE 7455303\n");
        }
      return(fatal_error);
    }

static void write_outfile(
  int *fatal_error)
    {
      static   unsigned char ascii_equivalent [6];
      static   int           cursor_x;
      static   char          cursor_x_prefix [3] = {'\033', '\052', '\160'};
      static   char          cursor_x_suffix = '\130';
      static   int           cursor_y;
      static   char          cursor_y_prefix [3] = {'\033', '\052', '\160'};
      static   char          cursor_y_suffix = '\131';
      static   char          dpi_prefix [3] = {'\033', '\052', '\164'};
      static   char          dpi_suffix = '\122';
      register int           ending_column_num;
      static   char          formfeed = '\014';
      static   char          landscape [5]
                              = {'\033', '\046', '\154', '\061', '\117'};
      static   char          logical_mode [5]
                              = {'\033', '\052', '\162', '\060', '\106'};
               FILE          *maze;
      static   int           non_null_found;
      static   int           num_bytes;
      static   char          raster_prefix [5]
                              = {'\033', '\052', '\162', '\061', '\101'};
      static   char          raster_suffix [4]
                              = {'\033', '\052', '\162', '\102'};
      static   row_rec       *row;
      static   char          row_prefix [3] = {'\033', '\052', '\142'};
      static   char          row_suffix = '\127';
      static   int           starting_column_num;
      register int           y;

      if ((maze=fopen("HP3DMAZE","wb")) == NULL)
        {
          *fatal_error=TRUE;
          printf("\n     Fatal error:  cannot open HP3DMAZE.\n");
        }
      else
        {
          *fatal_error
           =(fwrite(landscape,1,sizeof(landscape),maze) != sizeof(landscape));
          if (! *fatal_error)
            *fatal_error=(fwrite(dpi_prefix,1,sizeof(dpi_prefix),maze) 
             != sizeof(dpi_prefix));
          if (! *fatal_error)
            {
              sprintf((char *) ascii_equivalent,"%d",PIXELS_PER_INCH);
              *fatal_error=(fwrite(&(ascii_equivalent[0]),1,
               strlen((char *) ascii_equivalent),maze) 
               != strlen((char *) ascii_equivalent));
            }
          if (! *fatal_error)
            *fatal_error=(fwrite(&dpi_suffix,1,sizeof(dpi_suffix),maze)
             != sizeof(dpi_suffix));
          if (! *fatal_error)
            *fatal_error=(fwrite(logical_mode,1,sizeof(logical_mode),maze)
             != sizeof(logical_mode));
          for (y=0; ((! *fatal_error) && (y < NUM_Y_PIXELS)); y++)
            {
              titillate();
              row=row_addr(y);
              starting_column_num=0;
              non_null_found=FALSE;
              while ((starting_column_num <= X_BYTE_MAX)
              &&     (! non_null_found))
                if ((*row).column[starting_column_num] 
                 == 0)
                  starting_column_num++;
                else
                  non_null_found=TRUE;
              if (non_null_found)
                {
                  non_null_found=FALSE;
                  ending_column_num=X_BYTE_MAX;
                  while ((ending_column_num >= 0)
                   && (! non_null_found))
                    if ((*row).column[ending_column_num]
                     == 0)
                      ending_column_num--;
                    else
                      non_null_found=TRUE;
                  *fatal_error
                   =(fwrite(cursor_x_prefix,1,sizeof(cursor_x_prefix),maze)
                   != sizeof(cursor_x_prefix));
                  if (! *fatal_error)
                    {
                      cursor_x=PIXELS_PER_STROKE*8*starting_column_num;
                      sprintf((char *) ascii_equivalent,"%d",cursor_x);
                      *fatal_error=(fwrite(&(ascii_equivalent[0]),1,
                       strlen(ascii_equivalent),maze)
                       != strlen(ascii_equivalent));
                    }
                  if (! *fatal_error)
                    *fatal_error=(fwrite(&cursor_x_suffix,1,
                     sizeof(cursor_x_suffix),maze) != sizeof(cursor_x_suffix));
                  if (! *fatal_error)
                    *fatal_error=(fwrite(cursor_y_prefix,1,
                     sizeof(cursor_y_prefix),maze) != sizeof(cursor_y_prefix));
                  if (! *fatal_error)
                    {
                      cursor_y=PIXELS_PER_STROKE*y;
                      sprintf(ascii_equivalent,"%d",cursor_y);
                      *fatal_error=(fwrite(&(ascii_equivalent[0]),1,
                       strlen(ascii_equivalent),maze)
                       != strlen(ascii_equivalent));
                    }
                  if (! *fatal_error)
                    *fatal_error=(fwrite(&cursor_y_suffix,1,
                     sizeof(cursor_y_suffix),maze) != sizeof(cursor_y_suffix));
                  if (! *fatal_error)
                    *fatal_error=(fwrite(raster_prefix,1,sizeof(raster_prefix),
                     maze) != sizeof(raster_prefix));
                  if (! *fatal_error)
                    *fatal_error=(fwrite(row_prefix,1,sizeof(row_prefix),maze)
                     != sizeof(row_prefix));
                  if (! *fatal_error)
                    {
                      num_bytes=ending_column_num-starting_column_num+1;
                      sprintf(ascii_equivalent,"%d",num_bytes);
                      *fatal_error=(fwrite(&(ascii_equivalent[0]),1,
                       strlen(ascii_equivalent),maze) 
                       != strlen(ascii_equivalent));
                    }
                  if (! *fatal_error)
                    *fatal_error=(fwrite(&row_suffix,1,sizeof(row_suffix),maze)
                     != sizeof(row_suffix));
                  if (! *fatal_error)
                    *fatal_error=(fwrite(&((*row).column[starting_column_num]),
                     1,num_bytes,maze) != (unsigned int) num_bytes);
                  if (! *fatal_error)
                    *fatal_error=(fwrite(raster_suffix,1,sizeof(raster_suffix),
                     maze) != sizeof(raster_suffix));
                }
            }
          *fatal_error=(fwrite(&formfeed,1,sizeof(formfeed),maze)
           != sizeof(formfeed));
          if (*fatal_error)
            printf("\n     Fatal error:  not enough disk space.\n");
          fclose(maze);
        }
    }
  
static int memory_allocated(
  char      ***base_page,
  int       max_x_plus_1,
  int       max_y_plus_1,
  char      ***page,
  int       num_x_dots,
  int       num_y_dots,
  stack_rec **stack,
  int       num_rooms_in_maze,
  prime_rec ***prime_buffer,
  int       num_prime_buffers,
  int       prime_rec_count,
  row_rec   ***row_buffer,
  int       num_row_buffers,
  int       row_rec_count)
    {
      static   int result;
      register int prime_buffer_num;
      register int row_buffer_num;
      static   int y;

      if (((*base_page)=(char **) malloc(max_y_plus_1*sizeof(char *))) == NULL)
        result=FALSE;
      else
        {
          result=TRUE;
          for (y=0; ((result) && (y < max_y_plus_1)); y++)
           result=(((*base_page)[y]=malloc(max_x_plus_1*sizeof(char))) != NULL);
          if (! result)
            {
              --y;
              while (y > 0)
                free((void *) (*base_page)[--y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          if (((*page)=(char **) malloc(num_y_dots*sizeof(char *)))
           == NULL)
            {
              result=FALSE;
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          for (y=0; ((result) && (y < num_y_dots)); y++)
            result=(((*page)[y]=malloc(num_x_dots*sizeof(char))) != NULL);
          if (! result)
            {
              --y;
              while (y > 0)
                free((void *) (*page)[--y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          if ((*stack=(stack_rec *) malloc(
           ((unsigned int) num_rooms_in_maze)*sizeof(stack_rec))) == NULL)
            {
              result=FALSE;
              for (y=0; y < num_y_dots; y++)
               free((void *) (*page)[y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          if (((*prime_buffer)=(prime_rec **)
           malloc(num_prime_buffers*sizeof(prime_rec *))) == NULL)
            {
              result=FALSE;
              free((void *) *stack);
              for (y=0; y < num_y_dots; y++)
               free((void *) (*page)[y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          for (prime_buffer_num=0; 
           ((result) && (prime_buffer_num < num_prime_buffers));
           prime_buffer_num++)
            result=(((*prime_buffer)[prime_buffer_num]
             =malloc(prime_rec_count*sizeof(prime_rec))) != NULL);
          if (! result)
            {
              --prime_buffer_num;
              while (prime_buffer_num > 0)
                free((void *) (*prime_buffer)[--prime_buffer_num]);
              free((void *) *prime_buffer);
              free((void *) *stack);
              for (y=0; y < num_y_dots; y++)
               free((void *) (*page)[y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          if (((*row_buffer)=(row_rec **)
           malloc(num_row_buffers*sizeof(row_rec *))) == NULL)
            {
              result=FALSE;
              for (prime_buffer_num=0; prime_buffer_num < num_prime_buffers;
               prime_buffer_num++)
                free((void *) (*prime_buffer)[prime_buffer_num]);
              free((void *) *stack);
              for (y=0; y < num_y_dots; y++)
               free((void *) (*page)[y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      if (result)
        {
          for (row_buffer_num=0; 
           ((result) && (row_buffer_num < num_row_buffers));
           row_buffer_num++)
            result=(((*row_buffer)[row_buffer_num]
             =malloc(row_rec_count*sizeof(row_rec))) != NULL);
          if (! result)
            {
              --row_buffer_num;
              while (row_buffer_num > 0)
                free((void *) (*row_buffer)[--row_buffer_num]);
              free((void *) *row_buffer);
              for (prime_buffer_num=0; prime_buffer_num < num_prime_buffers;
               prime_buffer_num++)
                free((void *) (*prime_buffer)[prime_buffer_num]);
              free((void *) *stack);
              for (y=0; y < num_y_dots; y++)
               free((void *) (*page)[y]);
              free((void *) *page);
              for (y=0; y < max_y_plus_1; y++)
               free((void *) (*base_page)[y]);
              free((void *) *base_page);
            }
        }
      return(result);
    }

static void free_memory(
  char      ***base_page,
  int       max_y_plus_1,
  char      ***page,
  int       num_y_dots,
  stack_rec **stack,
  int       num_rooms_in_maze,
  prime_rec ***prime_buffer,
  int       num_prime_buffers,
  row_rec   ***row_buffer,
  int       num_row_buffers)
    {
      register int prime_buffer_num;
      register int row_buffer_num;
      static   int y;

      for (row_buffer_num=0; row_buffer_num < num_row_buffers;
       row_buffer_num++)
        free((void *) (*row_buffer)[row_buffer_num]);
      free((void *) *row_buffer);
      for (prime_buffer_num=0; prime_buffer_num < num_prime_buffers;
       prime_buffer_num++)
        free((void *) (*prime_buffer)[prime_buffer_num]);
      free((void *) *prime_buffer);
      free((void *) *stack);
      for (y=0; y < num_y_dots; y++)
        free((void *) (*page)[y]);
      free((void *) *page);
      for (y=0; y < max_y_plus_1; y++)
        free((void *) (*base_page)[y]);
      free((void *) *base_page);
      return;
    }

static prime_rec *prime_addr(
  long prime_num)
    {
      static   int       buffer_found;
      register int       buffer_num;
      static   long      earliest_access;
      static   int       oldest_buffer_num;
      static   prime_rec *result;

      prime_access_num++;
      buffer_found=FALSE;
      oldest_buffer_num=0;
      earliest_access=prime_access[0];
      for (buffer_num=0; ((! buffer_found) && (buffer_num < prime_buffer_count));
       buffer_num++)
        if ((prime_num >= prime_start[buffer_num])
        &&  (prime_num < ((long) PRIME_REC_COUNT)+(prime_start[buffer_num])))
          buffer_found=TRUE;
        else
          {
            if (prime_access[buffer_num] < earliest_access)
              {
                earliest_access=prime_access[buffer_num];
                oldest_buffer_num=buffer_num;
              }
          }
      if (buffer_found)
        {
          buffer_num--;
          result=&(prime_buffer[buffer_num][prime_num-prime_start[buffer_num]]);
          prime_access[buffer_num]=prime_access_num;
        }
      else
        {
          fseek(prime_vm,prime_start[oldest_buffer_num]*((long) sizeof(prime_rec)),
           SEEK_SET);
          fwrite(&(prime_buffer[oldest_buffer_num][0]),sizeof(prime_rec),
           PRIME_REC_COUNT,prime_vm);
          prime_start[oldest_buffer_num]=prime_num/((long) PRIME_REC_COUNT);
          (prime_start[oldest_buffer_num])*=((long) PRIME_REC_COUNT);
          fseek(prime_vm,prime_start[oldest_buffer_num]*((long) sizeof(prime_rec)),
           SEEK_SET);
          fread(&(prime_buffer[oldest_buffer_num][0]),sizeof(prime_rec),
           PRIME_REC_COUNT,prime_vm);
          result=&(prime_buffer[oldest_buffer_num][
           prime_num-prime_start[oldest_buffer_num]]);
          prime_access[oldest_buffer_num]=prime_access_num;
        }
      return(result);
    }

static row_rec *row_addr(
  int row_num)
    {
      static   int     buffer_found;
      register int     buffer_num;
      static   long    earliest_access;
      static   int     oldest_buffer_num;
      static   row_rec *result;

      row_access_num++;
      buffer_found=FALSE;
      oldest_buffer_num=0;
      earliest_access=row_access[0];
      for (buffer_num=0; ((! buffer_found) && (buffer_num < row_buffer_count));
       buffer_num++)
        if ((row_num >= row_start[buffer_num])
        &&  (row_num < ROW_REC_COUNT+(row_start[buffer_num])))
          buffer_found=TRUE;
        else
          {
            if (row_access[buffer_num] < earliest_access)
              {
                earliest_access=row_access[buffer_num];
                oldest_buffer_num=buffer_num;
              }
          }
      if (buffer_found)
        {
          buffer_num--;
          result=&(row_buffer[buffer_num][row_num-row_start[buffer_num]]);
          row_access[buffer_num]=row_access_num;
        }
      else
        {
          fseek(row_vm,
           ((long) (row_start[oldest_buffer_num]))*((long) sizeof(row_rec)),
           SEEK_SET);
          fwrite(&(row_buffer[oldest_buffer_num][0]),sizeof(row_rec),
           ROW_REC_COUNT,row_vm);
          row_start[oldest_buffer_num]=row_num/ROW_REC_COUNT;
          (row_start[oldest_buffer_num])*=ROW_REC_COUNT;
          fseek(row_vm,
           ((long) (row_start[oldest_buffer_num]))*((long) sizeof(row_rec)),
           SEEK_SET);
          fread(&(row_buffer[oldest_buffer_num][0]),sizeof(row_rec),
           ROW_REC_COUNT,row_vm);
          result=&(row_buffer[oldest_buffer_num][
           row_num-row_start[oldest_buffer_num]]);
          row_access[oldest_buffer_num]=row_access_num;
        }
      return(result);
    }

static void get_cursor(
  unsigned char *cursor_row,
  unsigned char *cursor_column,
  unsigned char *cursor_start,
  unsigned char *cursor_stop)
    {
      static union REGS in;
      static union REGS out;

      in.h.ah=(unsigned char) 3;
      in.h.bh=(unsigned char) 0;
      int86(0x10,&in,&out);
      *cursor_row=out.h.dh;
      *cursor_column=out.h.dl;
      *cursor_start=out.h.ch;
      *cursor_stop=out.h.cl;
      return;
    }

static void set_cursor_position(
  unsigned char cursor_row,
  unsigned char cursor_column)
    {
      static union REGS in;
      static union REGS out;

      in.h.ah=(unsigned char) 2;
      in.h.dh=cursor_row;
      in.h.dl=cursor_column;
      int86(0x10,&in,&out);
      return;
    }

static void set_cursor_size(
  unsigned char cursor_start,
  unsigned char cursor_stop)
    {
      static union REGS in;
      static union REGS out;

      in.h.ah=(unsigned char) 1;
      in.h.ch=cursor_start;
      in.h.cl=cursor_stop;
      int86(0x10,&in,&out);
      return;
    }

static void titillate()
    {
      set_cursor_position(cursor_row,cursor_column);
      titillator_index++;
      if (titillator_index > 3)
        titillator_index=0;
      putchar((int) titillator[titillator_index]);
      return;
    }

static void set_point_on_page(
  char **page,
  int  x,
  int  y)
    {
      static int x_offset;
      static int x_out;
      static int y_offset;
      static int y_out;

      for (x_offset = 0; x_offset < RESOLUTION; x_offset++)
        {
          x_out=x+x_offset;
          for (y_offset=0; y_offset < RESOLUTION; y_offset++)
            {
              y_out=y+y_offset;
              page[y_out][x_out]='W';
            }
        }
      return;
    }

static void draw_line_on_page(
  char **page,
  int  x1,
  int  y1,
  int  x2,
  int  y2)
    {
      static int error;
      static int error_prime_x;
      static int error_prime_y;
      static int permissible_delta_x;
      static int permissible_delta_y;
      static int x;
      static int x_diff;
      static int y;
      static int y_diff;

      error=0;
      if (x2 >= x1)
        permissible_delta_x=1;
      else
        permissible_delta_x=-1;
      if (y2 >= y1)
        permissible_delta_y=1;
      else
        permissible_delta_y=-1;
      x=x1;
      y=y1;
      x_diff=x2-x1;
      y_diff=y2-y1;
      set_point_on_page(page,x,y);
      while ((x != x2) || (y != y2))
        {
          error_prime_x=error+permissible_delta_x*y_diff;
          error_prime_y=error-permissible_delta_y*x_diff;
          if (abs(error_prime_x) <= abs(error_prime_y))
            {
              x+=permissible_delta_x;
              error=error_prime_x;
            }
          else
            {
              y+=permissible_delta_y;
              error=error_prime_y;
            }
          set_point_on_page(page,x,y);
        }
      return;
    }

static void solve_maze(
  stack_rec *stack,
  char      **base_page,
  int       *num_rooms_in_solution,
  int       *adjacency,
  int       max_x,
  int       max_y)
    {
      int delta_index;
      int passage_found;
      int stack_head;
      int x;
      int x_next;
      int y;
      int y_next;

      *num_rooms_in_solution=1;
      *adjacency=0;
      x=3;
      y=2;
      stack_head=-1;
      base_page[y][x]='S';
      do
        {
          delta_index=0;
          passage_found=FALSE;
          do
            {
              while ((delta_index < 6) && (! passage_found))
                {
                  x_next=x+delta_x[delta_index][0];
                  y_next=y+delta_y[delta_index][0];
                  if (base_page[y_next][x_next] == ' ')
                    passage_found=TRUE;
                  else
                    delta_index++;
                }
              if (! passage_found)
                {
                  delta_index=(int) (stack[stack_head].index_1);
                  base_page[y][x]=' ';
                  x-=delta_x[delta_index][0];
                  y-=delta_y[delta_index][0];
                  base_page[y][x]=' ';
                  x-=delta_x[delta_index][0];
                  y-=delta_y[delta_index][0];
                  stack_head--;
                  delta_index++;
                }
            }
          while (! passage_found);
          base_page[y_next][x_next]='S';
          x_next+=delta_x[delta_index][0];
          y_next+=delta_y[delta_index][0];
          if (y_next <= max_y)
            {
              stack_head++;
              stack[stack_head].index_1=(char) delta_index;
              base_page[y_next][x_next]='S';
              x=x_next;
              y=y_next;
            }
        }
      while (y_next < max_y);
      x=max_x-3;
      y=max_y-2;
      *adjacency=0;
      while (stack_head >= 0)
        {
          for (delta_index=0; delta_index < 6; delta_index++)
            {
              x_next=x+delta_x[delta_index][0];
              y_next=y+delta_y[delta_index][0];
              if (base_page[y_next][x_next] != 'S')
                {
                  if (base_page[y_next][x_next] == 'W')
                    {
                      x_next+=delta_x[delta_index][0];
                      y_next+=delta_y[delta_index][0];
                      if (x_next < 0)
                        (*adjacency)++;
                      else
                        if (x_next > max_x)
                          (*adjacency)++;
                        else
                          if (y_next < 0)
                            (*adjacency)++;
                          else
                            if (y_next > max_y)
                              (*adjacency)++;
                            else
                              {
                                if (base_page[y_next][x_next] == 'S')
                                  (*adjacency)++;
                              }
                    }
                }
            }
          x-=(2*delta_x[stack[stack_head].index_1][0]);
          y-=(2*delta_y[stack[stack_head].index_1][0]);
          stack_head--;
          (*num_rooms_in_solution)++;
        }
      for (delta_index=0; delta_index < 6; delta_index++)
        {
          x_next=x+delta_x[delta_index][0];
          y_next=y+delta_y[delta_index][0];
          if (base_page[y_next][x_next] != ' ')
            {
              if (base_page[y_next][x_next] == 'W')
                {
                  x_next+=delta_x[delta_index][0];
                  y_next+=delta_y[delta_index][0];
                  if (x_next < 0)
                    (*adjacency)++;
                  else
                    if (x_next > max_x)
                      (*adjacency)++;
                    else
                      if (y_next < 0)
                        (*adjacency)++;
                      else
                        if (y_next > max_y)
                          (*adjacency)++;
                        else
                          {
                            if (base_page[y_next][x_next] == 'S')
                              (*adjacency)++;
                          }
                }
            }
        }
      return;
    }

static void generate_maze(
  int       *r_n,
  char      **base_page,
  char      **page,
  int       max_x,
  int       max_y,
  stack_rec *stack,
  int       x_dot_max,
  int       y_dot_max,
  int       num_rooms_in_maze,
  int       num_columns,
  int       num_rows)
    {
      static   int  adjacency;
      static   int  age;
      static   int  column_num;
      static   int  counter_0;
      static   int  counter_1;
      static   int  counter_2;
      static   int  counter_3;
      static   int  counter_4;
      static   int  counter_5;
      static   int  counter_6;
      static   int  counter_7;
      static   int  delta_index_1a;
      static   int  delta_index_1b;
      static   int  delta_index_1c;
      static   int  delta_index_1d;
      static   int  delta_index_1e;
      static   int  delta_index_1f;
      static   int  delta_index_2;
      static   int  num_rooms_in_solution;
      static   int  passage_found;
      register int  r_n_index_1;
      register int  r_n_index_2;
      static   int  row_num;
      static   int  search_complete;
               FILE *solution;
      static   int  stack_head;
      static   int  tem_int;
      static   int  trial_num_mod_600;
      static   int  x;
      static   int  x_mod_8;
      static   int  x_next;
      static   int  y;
      static   int  y_mod_4;
      static   int  y_next;
      static   int  y_previous;

      counter_0=r_n[0];
      counter_1=r_n[1];
      counter_2=r_n[2];
      counter_3=r_n[3];
      counter_4=r_n[4];
      counter_5=r_n[5];
      counter_6=r_n[6];
      counter_7=r_n[7];
      hash(&counter_0,&counter_1,&counter_2,&counter_3,&counter_4,&counter_5,
       &counter_6,&counter_7);
      delta_y[0][0]=-1;
      delta_x[0][0]=-2;
      delta_y[1][0]=1;
      delta_x[1][0]=-2;
      delta_y[2][0]=-2;
      delta_x[2][0]=0;
      delta_y[3][0]=2;
      delta_x[3][0]=0;
      delta_y[4][0]=-1;
      delta_x[4][0]=2;
      delta_y[5][0]=1;
      delta_x[5][0]=2;
      delta_index_2=0;
      for (delta_index_1a=0; delta_index_1a < 6; delta_index_1a++)
        for (delta_index_1b=0; delta_index_1b < 6; delta_index_1b++)
          if (delta_index_1a != delta_index_1b)
           for (delta_index_1c=0; delta_index_1c < 6; delta_index_1c++)
             if ((delta_index_1a != delta_index_1c)
             &&  (delta_index_1b != delta_index_1c))
               for (delta_index_1d=0; delta_index_1d < 6; delta_index_1d++)
                  if ((delta_index_1a != delta_index_1d)
                  &&  (delta_index_1b != delta_index_1d)
                  &&  (delta_index_1c != delta_index_1d))
                    for (delta_index_1e=0; delta_index_1e < 6;
                     delta_index_1e++)
                      if ((delta_index_1a != delta_index_1e)
                      &&  (delta_index_1b != delta_index_1e)
                      &&  (delta_index_1c != delta_index_1e)
                      &&  (delta_index_1d != delta_index_1e))
                        for (delta_index_1f=0; delta_index_1f < 6;
                         delta_index_1f++)
                          if ((delta_index_1a != delta_index_1f)
                          &&  (delta_index_1b != delta_index_1f)
                          &&  (delta_index_1c != delta_index_1f)
                          &&  (delta_index_1d != delta_index_1f)
                          &&  (delta_index_1e != delta_index_1f))
                            {
                              delta_x[delta_index_1a][delta_index_2]
                               =delta_x[0][0];
                              delta_y[delta_index_1a][delta_index_2]
                               =delta_y[0][0];
                              delta_x[delta_index_1b][delta_index_2]
                               =delta_x[1][0];
                              delta_y[delta_index_1b][delta_index_2]
                               =delta_y[1][0];
                              delta_x[delta_index_1c][delta_index_2]
                               =delta_x[2][0];
                              delta_y[delta_index_1c][delta_index_2]
                               =delta_y[2][0];
                              delta_x[delta_index_1d][delta_index_2]
                               =delta_x[3][0];
                              delta_y[delta_index_1d][delta_index_2]
                               =delta_y[3][0];
                              delta_x[delta_index_1e][delta_index_2]
                               =delta_x[4][0];
                              delta_y[delta_index_1e][delta_index_2]
                               =delta_y[4][0];
                              delta_x[delta_index_1f][delta_index_2]
                               =delta_x[5][0];
                              delta_y[delta_index_1f][delta_index_2]
                               =delta_y[5][0];
                              delta_index_2++;
                            };
      age=3;
      trial_num_mod_600=0;
      do
        {
          titillate();
          r_n[0]=counter_0+1;
          r_n[1]=counter_1+1;
          r_n[2]=counter_2+1;
          r_n[3]=counter_3+1;
          r_n[4]=counter_4+1;
          r_n[5]=counter_5+1;
          r_n[6]=counter_6+1;
          r_n[7]=counter_7+1;
          y_mod_4=1;
          for (y=0; y <= max_y; y++)
            {
              if (y_mod_4 == 1)
                {
                  x_mod_8=1;
                  for (x=0; x <= max_x; x++)
                    {
                      if (((x_mod_8 == 0)
                        && (y != 0)
                        && (y != max_y))
                      ||  (x_mod_8 == 3)
                      ||  (x_mod_8 == 4)
                      ||  (x_mod_8 == 5))
                        base_page[y][x]='W';
                      else
                        base_page[y][x]=' ';
                      x_mod_8++;
                      if (x_mod_8 >= 8)
                        x_mod_8=0;
                    }
                }
              else
                {
                  if (y_mod_4 == 0 || y_mod_4 == 2)
                    {
                      x_mod_8=1;
                      for (x=0; x <= max_x; x++)
                        {
                          if ((x_mod_8 == 2) || (x_mod_8 == 6))
                            base_page[y][x]='W';
                          else
                            base_page[y][x]=' ';
                          x_mod_8++;
                          if (x_mod_8 >= 8)
                            x_mod_8 = 0;
                        }
                    }
                  else
                    {
                      x_mod_8=1;
                      for (x=0; x <= max_x; x++)
                        {
                          if ((x_mod_8 == 0)
                          ||  (x_mod_8 == 1)
                          ||  (x_mod_8 == 4)
                          ||  (x_mod_8 == 7))
                            base_page[y][x]='W';
                          else
                            base_page[y][x]=' ';
                          x_mod_8++;
                          if (x_mod_8 >= 8)
                            x_mod_8=0;
                      }
                    }
                }
              y_mod_4++;
              if (y_mod_4 >= 4)
                y_mod_4=0;
            }
          column_num=r_n[0];
          r_n_index_1=0;
          r_n_index_2=1;
          while (r_n_index_2 < 8)
            {
              tem_int=r_n[r_n_index_2];
              r_n[r_n_index_1]=tem_int;
              column_num+=tem_int;
              if (column_num >= 727)
                column_num-=727;
              r_n_index_1=r_n_index_2;
              r_n_index_2++;
            }
          r_n[7]=column_num;
          column_num%=num_columns;
          x=4*column_num+3;
          row_num=r_n[0];
          r_n_index_1=0;
          r_n_index_2=1;
          while (r_n_index_2 < 8)
            {
              tem_int=r_n[r_n_index_2];
              r_n[r_n_index_1]=tem_int;
              row_num+=tem_int;
              if (row_num >= 727)
                row_num-=727;
              r_n_index_1=r_n_index_2;
              r_n_index_2++;
            }
          r_n[7]=row_num;
          if (column_num%2)
            {
              row_num%=(num_rows-1);
              y=4*row_num+4;
            }
          else
            {
              row_num%=num_rows;
              y=4*row_num+2;
            }
          base_page[y][x]=' ';
          stack_head=-1;
          do
            {
              delta_index_1a=0;
              do
                {
                  delta_index_2=r_n[0];
                  r_n_index_1=0;
                  r_n_index_2=1;
                  while (r_n_index_2 < 8)
                    {
                      tem_int=r_n[r_n_index_2];
                      r_n[r_n_index_1]=tem_int;
                      delta_index_2+=tem_int;
                      if (delta_index_2 >= 727)
                        delta_index_2-=727;
                      r_n_index_1=r_n_index_2;
                      r_n_index_2++;
                    }
                  r_n[7]=delta_index_2;
                }
              while (delta_index_2 >= 720);
              passage_found=FALSE;
              search_complete=FALSE;
              while (! search_complete)
                {
                  while ((delta_index_1a < 6) && (! passage_found))
                    {
                      x_next=x+2*delta_x[delta_index_1a][delta_index_2];
                      if (x_next <= 0)
                        delta_index_1a++;
                      else
                        if (x_next > max_x)
                          delta_index_1a++;
                        else
                          {
                            y_next=y+2*delta_y[delta_index_1a][delta_index_2];
                            if (y_next <= 0)
                              delta_index_1a++;
                            else
                              if (y_next > max_y)
                                delta_index_1a++;
                              else
                                if (base_page[y_next][x_next] == 'W')
                                  passage_found=TRUE;
                                else
                                  delta_index_1a++;
                          }
                    }
                  if (! passage_found)
                    {
                      if (stack_head >= 0)
                        {
                          delta_index_1a=(int) (stack[stack_head].index_1);
                          delta_index_2=stack[stack_head].index_2;
                          x-=2*delta_x[delta_index_1a][delta_index_2];
                          y-=2*delta_y[delta_index_1a][delta_index_2];
                          stack_head--;
                          delta_index_1a++;
                        }
                    }
                  search_complete=((passage_found)
                   || ((stack_head == -1) && (delta_index_1a >= 6)));
                }
              if (passage_found)
                {
                  stack_head++;
                  stack[stack_head].index_1=(char) delta_index_1a;
                  stack[stack_head].index_2=delta_index_2;
                  base_page[y_next][x_next]=' ';
                  base_page[(y+y_next)/2][(x+x_next)/2]=' ';
                  x=x_next;
                  y=y_next;
                }
            }
          while (stack_head != -1);
          base_page[0][3]='S';
          base_page[max_y][max_x-3]=' ';
          solve_maze(stack,base_page,&num_rooms_in_solution,&adjacency,max_x,
           max_y);
          increment(&counter_0,&counter_1,&counter_2,&counter_3,&counter_4,
           &counter_5,&counter_6,&counter_7);
          trial_num_mod_600++;
          if (trial_num_mod_600 >= 600)
            {
              trial_num_mod_600=0;
              age++;
            }
        }
      while ((3*num_rooms_in_solution < num_rooms_in_maze)
      ||     (2*adjacency > age*num_rooms_in_solution));
      solution=fopen("HP3DMAZE.SOL","w");
      for (y=0; y <= max_y; y++)
        {
          for (x=0; x <= max_x; x++)
            switch (base_page[y][x])
              {
                case 'S':
                  fputc(250,solution);
                  break;
                case 'W':
                  fputc(219,solution);
                  break;
                default:
                  fputc((int) ' ',solution);
                  break;
              };
          fputc((int) '\n',solution);
        }
      fclose(solution);
      for (x=0; x <= x_dot_max; x++)
        for (y=0; y <= y_dot_max; y++)
          page[y][x]=' ';
      y_previous=-1;
      y_next=1;
      for (y = 0; y <= max_y; y++)
        {
          x=0;
          for (x_next = 1; x_next <= max_x; x_next++)
            {
              if (base_page[y][x] == 'W')
                {
                  if (base_page[y][x_next] == 'W')
                    draw_line_on_page(page,RESOLUTION*x,RESOLUTION*y,
                     RESOLUTION*x_next,RESOLUTION*y);
                  if (y_previous >= 0)
                    {
                      if (base_page[y_previous][x_next] == 'W')
                        draw_line_on_page(page,RESOLUTION*x,RESOLUTION*y,
                         RESOLUTION*x_next,RESOLUTION*y_previous);
                    }
                  if (y_next <= max_y)
                    {
                      if (base_page[y_next][x_next] == 'W')
                        draw_line_on_page(page,RESOLUTION*x,RESOLUTION*y,
                         RESOLUTION*x_next,RESOLUTION*y_next);
                    }
                }
              x=x_next;
            }
          y_previous=y;
          y_next++;
        }
      return;
    }

static int substitution_high [100] =
             { 4,1,2,8,8,9,9,6,5,7,2,1,2,9,8,8,6,3,5,1,9,5,4,4,9,8,6,0,8,0,
               6,0,2,4,1,9,2,0,7,4,7,3,0,0,2,6,8,9,4,0,8,3,2,3,2,5,2,4,6,9,
               7,9,1,3,5,7,1,1,4,5,8,1,6,0,5,7,8,2,3,3,7,3,5,1,7,5,4,0,3,6,
               3,7,7,1,9,4,0,5,6,6
             };
static int substitution_low [100] =
             { 1,2,2,1,5,5,4,6,4,6,4,4,5,6,6,3,0,9,6,5,7,2,0,9,3,4,2,3,9,1,
               9,9,9,3,8,9,3,4,1,5,0,5,2,7,0,8,8,0,4,5,0,3,6,8,1,7,8,8,7,1,
               3,2,7,7,1,8,0,3,7,5,2,6,4,0,9,9,7,7,4,6,2,0,0,1,7,3,6,6,1,1,
               2,4,5,9,8,2,8,8,3,5
             };
static void hash(
  int *counter_0,
  int *counter_1,
  int *counter_2,
  int *counter_3,
  int *counter_4,
  int *counter_5,
  int *counter_6,
  int *counter_7)
    {
      register int iteration;
      static   int seed_0;
      static   int seed_1;
      static   int seed_2;
      static   int seed_3;
      static   int seed_4;
      static   int seed_5;
      static   int seed_6;
      static   int seed_7;
      register int substitution_index;
      static   int tem_0;
      static   int tem_1;
      static   int tem_2;

      seed_0=(*counter_0);
      seed_1=(*counter_1);
      seed_2=(*counter_2);
      seed_3=(*counter_3);
      seed_4=(*counter_4);
      seed_5=(*counter_5);
      seed_6=(*counter_6);
      seed_7=(*counter_7);
      for (iteration=1; iteration <= 8; iteration++)
        {
          substitution_index=10*seed_1+seed_0;
          tem_0=substitution_low[substitution_index];
          tem_1=substitution_high[substitution_index];
          substitution_index=10*seed_3+seed_2;
          seed_0=substitution_low[substitution_index];
          tem_2=substitution_high[substitution_index];
          substitution_index=10*seed_5+seed_4;
          seed_2=substitution_low[substitution_index];
          seed_1=substitution_high[substitution_index];
          substitution_index=10*seed_7+seed_6;
          seed_5=substitution_low[substitution_index];
          seed_7=substitution_high[substitution_index];
          seed_3=tem_0;
          seed_6=tem_1;
          seed_4=tem_2;
        }
      (*counter_0)=seed_0;
      (*counter_1)=seed_1;
      (*counter_2)=seed_2;
      (*counter_3)=seed_3;
      (*counter_4)=seed_4;
      (*counter_5)=seed_5;
      (*counter_6)=seed_6;
      (*counter_7)=seed_7;
      return;
    }

static void increment(
  int *counter_0,
  int *counter_1,
  int *counter_2,
  int *counter_3,
  int *counter_4,
  int *counter_5,
  int *counter_6,
  int *counter_7)
    {
      register tem;

      tem=(*counter_0)+1;
      if (tem <= 9)
        (*counter_0)=tem;
      else
        {
          (*counter_0)=0;
          tem=(*counter_1)+1;
          if (tem <= 9)
            (*counter_1)=tem;
          else
            {
              (*counter_1)=0;
              tem=(*counter_2)+1;
              if (tem <= 9)
                (*counter_2)=tem;
              else
                {
                  (*counter_2)=0;
                  tem=(*counter_3)+1;
                  if (tem <= 9)
                    (*counter_3)=tem;
                  else
                    {
                      (*counter_3)=0;
                      tem=(*counter_4)+1;
                      if (tem <= 9)
                        (*counter_4)=tem;
                      else
                        {
                          (*counter_4)=0;
                          tem=(*counter_5)+1;
                          if (tem <= 9)
                            (*counter_5)=tem;
                          else
                            {
                              (*counter_5)=0;
                              tem=(*counter_6)+1;
                              if (tem <= 9)
                                (*counter_6)=tem;
                              else
                                {
                                  (*counter_6)=0;
                                  tem=(*counter_7)+1;
                                  if (tem <= 9)
                                    (*counter_7)=tem;
                                  else
                                    (*counter_7)=0;
                                }
                            }
                        }
                    }
                }
            }
        }
      return;
    }

static void evaluate_and_transform(
  double     x_min,
  double     x_max,
  double     y_min,
  double     y_max,
  int        num_x_divisions,
  int        num_y_divisions,
  double     rotation,
  double     tilt,
  double     *x_prime_max,
  double     *y_prime_min,
  double     *y_prime_max,
  double     *z_prime_min,
  double     *z_prime_max,
  vertex_rec *light)
    {
      static   double    cos_rotation;
      static   double    cos_tilt;
      static   double    magnitude;
      static   prime_rec *prime;
      static   long      prime_num;
      static   double    radians;
      static   double    radians_per_degree;
      static   double    sin_rotation;
      static   double    sin_tilt;
      static   double    x;
      static   double    x_delta;
      register int       x_division_num;
      static   double    x_rotated;
      static   double    y;
      static   double    y_delta;
      register int       y_division_num;
      static   double    z;

      radians_per_degree=atan(1.0)/45.0;
      radians=tilt*radians_per_degree;
      cos_tilt=cos(radians);
      sin_tilt=sin(radians);
      radians=rotation*radians_per_degree;
      cos_rotation=cos(radians);
      sin_rotation=sin(radians);
      z=f(x_min,y_min);
      x_rotated=x_min*cos_rotation+y_min*sin_rotation;
      *y_prime_min=-x_min*sin_rotation+y_min*cos_rotation;
      *z_prime_min=-x_rotated*sin_tilt+z*cos_tilt;
      *y_prime_max=*y_prime_min;
      *z_prime_max=*z_prime_min;
      *x_prime_max=x_rotated*cos_tilt+z*sin_tilt;
      x_delta=(double) (num_x_divisions-1);
      x_delta=(x_max-x_min)/x_delta;
      y_delta=(double) (num_y_divisions-1);
      y_delta=(y_max-y_min)/y_delta;
      x=x_min;
      prime_num=0l;
      for (x_division_num=0; x_division_num < num_x_divisions; x_division_num++)
        {
          titillate();
          y=y_min;
          for (y_division_num=0; y_division_num < num_y_divisions;
           y_division_num++)
            {
              z=f(x,y);
              prime=prime_addr(prime_num);
              if ((z == 0.0)
              &&  (external_to_maze(x,y)))
                prime->outside_maze=(unsigned char) '\1';
              else
                prime->outside_maze=(unsigned char) '\0';
              prime->x_division_index=x_division_num;
              prime->y_division_index=y_division_num;
              x_rotated=x*cos_rotation+y*sin_rotation;
              prime->y=(float) (-x*sin_rotation+y*cos_rotation);
              prime->x=(float) (x_rotated*cos_tilt+z*sin_tilt);
              prime->z=(float) (-x_rotated*sin_tilt+z*cos_tilt);
              if (((double) (prime->x)) > *x_prime_max)
                *x_prime_max=(double) (prime->x);
              if (((double) (prime->y)) < *y_prime_min)
                *y_prime_min=(double) (prime->y);
              if (((double) (prime->y)) > *y_prime_max)
                *y_prime_max=(double) (prime->y);
              if (((double) (prime->z)) < *z_prime_min)
                *z_prime_min=(double) (prime->z);
              if (((double) (prime->z)) > *z_prime_max)
                *z_prime_max=(double) (prime->z);
              y+=y_delta;
              prime_num++;
            }
          x+=x_delta;
        }
      magnitude=(*light).x*(*light).x;
      magnitude+=((*light).y*(*light).y);
      magnitude+=((*light).z*(*light).z);
      magnitude=sqrt(magnitude);
      (*light).x/=magnitude;
      (*light).y/=magnitude;
      (*light).z/=magnitude;
      return;
    }

static void shade(
  int        num_x_divisions,
  int        num_y_divisions,
  vertex_rec *light,
  long       num_primes)
    {
      static   double     magnitude;
      static   vertex_rec normal;
      static   prime_rec  *prime;
      static   long       prime_num;
      static   vertex_rec vertex [4];
      register int        x_division_num;
      register int        y_division_num;

      prime_num=num_primes;
      for (x_division_num=num_x_divisions-1; x_division_num >= 0;
       --x_division_num)
        {
          titillate();
          for (y_division_num=num_y_divisions-1; y_division_num >= 0;
           --y_division_num)
            {
              prime_num--;
              prime=prime_addr(prime_num);
              vertex[0].x=(double) (prime->x);
              vertex[0].y=(double) (prime->y);
              vertex[0].z=(double) (prime->z);
              if (x_division_num < (num_x_divisions-1))
                if (y_division_num < (num_y_divisions-1))
                  {
                    prime_num+=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[1].x=(double) (prime->x);
                    vertex[1].y=(double) (prime->y);
                    vertex[1].z=(double) (prime->z);
                    prime_num++;
                    prime=prime_addr(prime_num);
                    vertex[2].x=(double) (prime->x);
                    vertex[2].y=(double) (prime->y);
                    vertex[2].z=(double) (prime->z);
                    prime_num-=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[3].x=(double) (prime->x);
                    vertex[3].y=(double) (prime->y);
                    vertex[3].z=(double) (prime->z);
                    prime_num--;
                  }
                else
                  {
                    prime_num--;
                    prime=prime_addr(prime_num);
                    vertex[1].x=(double) (prime->x);
                    vertex[1].y=(double) (prime->y);
                    vertex[1].z=(double) (prime->z);
                    prime_num+=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[2].x=(double) (prime->x);
                    vertex[2].y=(double) (prime->y);
                    vertex[2].z=(double) (prime->z);
                    prime_num++;
                    prime=prime_addr(prime_num);
                    vertex[3].x=(double) (prime->x);
                    vertex[3].y=(double) (prime->y);
                    vertex[3].z=(double) (prime->z);
                    prime_num-=((long) num_y_divisions);
                  }
              else
                if (y_division_num < (num_y_divisions-1))
                  {
                    prime_num++;
                    prime=prime_addr(prime_num);
                    vertex[1].x=(double) (prime->x);
                    vertex[1].y=(double) (prime->y);
                    vertex[1].z=(double) (prime->z);
                    prime_num-=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[2].x=(double) (prime->x);
                    vertex[2].y=(double) (prime->y);
                    vertex[2].z=(double) (prime->z);
                    prime_num--;
                    prime=prime_addr(prime_num);
                    vertex[3].x=(double) (prime->x);
                    vertex[3].y=(double) (prime->y);
                    vertex[3].z=(double) (prime->z);
                    prime_num+=((long) num_y_divisions);
                  }
                else
                  {
                    prime_num-=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[1].x=(double) (prime->x);
                    vertex[1].y=(double) (prime->y);
                    vertex[1].z=(double) (prime->z);
                    prime_num--;
                    prime=prime_addr(prime_num);
                    vertex[2].x=(double) (prime->x);
                    vertex[2].y=(double) (prime->y);
                    vertex[2].z=(double) (prime->z);
                    prime_num+=((long) num_y_divisions);
                    prime=prime_addr(prime_num);
                    vertex[3].x=(double) (prime->x);
                    vertex[3].y=(double) (prime->y);
                    vertex[3].z=(double) (prime->z);
                    prime_num++;
                  }
              normal.x
               =(vertex[1].y-vertex[0].y)*(vertex[3].z-vertex[0].z)
               -(vertex[3].y-vertex[0].y)*(vertex[1].z-vertex[0].z)
               +(vertex[2].y-vertex[1].y)*(vertex[0].z-vertex[1].z)
               -(vertex[0].y-vertex[1].y)*(vertex[2].z-vertex[1].z)
               +(vertex[3].y-vertex[2].y)*(vertex[1].z-vertex[2].z)
               -(vertex[1].y-vertex[2].y)*(vertex[3].z-vertex[2].z)
               +(vertex[0].y-vertex[3].y)*(vertex[2].z-vertex[3].z)
               -(vertex[2].y-vertex[3].y)*(vertex[0].z-vertex[3].z);
              normal.y
               =(vertex[3].x-vertex[0].x)*(vertex[1].z-vertex[0].z)
               -(vertex[1].x-vertex[0].x)*(vertex[3].z-vertex[0].z)
               +(vertex[0].x-vertex[1].x)*(vertex[2].z-vertex[1].z)
               -(vertex[2].x-vertex[1].x)*(vertex[0].z-vertex[1].z)
               +(vertex[1].x-vertex[2].x)*(vertex[3].z-vertex[2].z)
               -(vertex[3].x-vertex[2].x)*(vertex[1].z-vertex[2].z)
               +(vertex[2].x-vertex[3].x)*(vertex[0].z-vertex[3].z)
               -(vertex[0].x-vertex[3].x)*(vertex[2].z-vertex[3].z);
              normal.z
               =(vertex[1].x-vertex[0].x)*(vertex[3].y-vertex[0].y)
               -(vertex[3].x-vertex[0].x)*(vertex[1].y-vertex[0].y)
               +(vertex[2].x-vertex[1].x)*(vertex[0].y-vertex[1].y)
               -(vertex[0].x-vertex[1].x)*(vertex[2].y-vertex[1].y)
               +(vertex[3].x-vertex[2].x)*(vertex[1].y-vertex[2].y)
               -(vertex[1].x-vertex[2].x)*(vertex[3].y-vertex[2].y)
               +(vertex[0].x-vertex[3].x)*(vertex[2].y-vertex[3].y)
               -(vertex[2].x-vertex[3].x)*(vertex[0].y-vertex[3].y);
              prime=prime_addr(prime_num);
              if (normal.x < 0.0)
                prime->color=NUM_COLORS;
              else
                {
                  magnitude
                   =sqrt(normal.x*normal.x+normal.y*normal.y+normal.z*normal.z);
                  if (magnitude == 0.0)
                    prime->color=0l;
                  else
                    {
                      prime->color
                       =(long) ((((float) NUM_COLORS)/2.0)*(1.0
                       +((*light).x*normal.x+(*light).y*normal.y
                       +(*light).z*normal.z)/magnitude));
                      if (prime->color >= NUM_COLORS)
                        prime->color=NUM_COLORS-1l;
                    }
                }
            }
        }
      return;
    }

static void adjust_perspective(
  int    num_x_divisions,
  int    num_y_divisions,
  double x_prime_max,
  double y_prime_min,
  double y_prime_max,
  double z_prime_min,
  double z_prime_max)
    {
      static   prime_rec  *prime;
      static   long       prime_num;
      static   vertex_rec vertex [4];
      register int        x_division_num;
      static   double     x_eye;
      static   double     y_center;
      register int        y_division_num;
      static   double     z_center;

      if ((y_prime_max-y_prime_min) > (z_prime_max-z_prime_min))
        x_eye=1.1*(y_prime_max-y_prime_min)+x_prime_max;
      else
        x_eye=1.1*(z_prime_max-z_prime_min)+x_prime_max;
      if (((y_prime_max-y_prime_min) > (z_prime_max-z_prime_min))
      ||  (z_prime_max != z_prime_min))
        {
          y_center=(y_prime_max+y_prime_min)/2.0;
          z_center=(z_prime_max+z_prime_min)/2.0;
          prime_num=0l;
          for (x_division_num=0; x_division_num < num_x_divisions;
           x_division_num++)
            {
              titillate();
              for (y_division_num=0; y_division_num < num_y_divisions;
               y_division_num++)
                {
                  prime=prime_addr(prime_num);
                  vertex[0].x=(double) (prime->x);
                  vertex[0].y=(double) (prime->y);
                  vertex[0].z=(double) (prime->z);
                  if (x_division_num < (num_x_divisions-1))
                    if (y_division_num < (num_y_divisions-1))
                      {
                        prime_num+=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[1].x=(double) (prime->x);
                        vertex[1].y=(double) (prime->y);
                        vertex[1].z=(double) (prime->z);
                        prime_num++;
                        prime=prime_addr(prime_num);
                        vertex[2].x=(double) (prime->x);
                        vertex[2].y=(double) (prime->y);
                        vertex[2].z=(double) (prime->z);
                        prime_num-=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[3].x=(double) (prime->x);
                        vertex[3].y=(double) (prime->y);
                        vertex[3].z=(double) (prime->z);
                        prime_num--;
                      }
                    else
                      {
                        prime_num--;
                        prime=prime_addr(prime_num);
                        vertex[1].x=(double) (prime->x);
                        vertex[1].y=(double) (prime->y);
                        vertex[1].z=(double) (prime->z);
                        prime_num+=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[2].x=(double) (prime->x);
                        vertex[2].y=(double) (prime->y);
                        vertex[2].z=(double) (prime->z);
                        prime_num++;
                        prime=prime_addr(prime_num);
                        vertex[3].x=(double) (prime->x);
                        vertex[3].y=(double) (prime->y);
                        vertex[3].z=(double) (prime->z);
                        prime_num-=((long) num_y_divisions);
                      }
                  else
                    if (y_division_num < (num_y_divisions-1))
                      {
                        prime_num++;
                        prime=prime_addr(prime_num);
                        vertex[1].x=(double) (prime->x);
                        vertex[1].y=(double) (prime->y);
                        vertex[1].z=(double) (prime->z);
                        prime_num-=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[2].x=(double) (prime->x);
                        vertex[2].y=(double) (prime->y);
                        vertex[2].z=(double) (prime->z);
                        prime_num--;
                        prime=prime_addr(prime_num);
                        vertex[3].x=(double) (prime->x);
                        vertex[3].y=(double) (prime->y);
                        vertex[3].z=(double) (prime->z);
                        prime_num+=((long) num_y_divisions);
                      }
                    else
                      {
                        prime_num-=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[1].x=(double) (prime->x);
                        vertex[1].y=(double) (prime->y);
                        vertex[1].z=(double) (prime->z);
                        prime_num--;
                        prime=prime_addr(prime_num);
                        vertex[2].x=(double) (prime->x);
                        vertex[2].y=(double) (prime->y);
                        vertex[2].z=(double) (prime->z);
                        prime_num+=((long) num_y_divisions);
                        prime=prime_addr(prime_num);
                        vertex[3].x=(double) (prime->x);
                        vertex[3].y=(double) (prime->y);
                        vertex[3].z=(double) (prime->z);
                        prime_num++;
                      }
                  prime=prime_addr(prime_num);
                  prime->y=(float) (y_center
                   +(vertex[0].y-y_center)*(x_eye-x_prime_max)
                   /(x_eye-vertex[0].x));
                  prime->z=(float) (z_center
                   +(vertex[0].z-z_center)*(x_eye-x_prime_max)
                   /(x_eye-vertex[0].x));
                  prime->x=(float)
                   (-(vertex[0].x+vertex[1].x+vertex[2].x+vertex[3].x)/4.0);
                  prime_num++;
                }
            }
         }
      return;
    }

static void rearrange(
  long lower_bound,
  long upper_bound,
  long *j)
    {
      static   long      down;
      static   int       finished;
      static   prime_rec *prime_1;
      static   prime_rec *prime_2;
      static   long      up;
      static   float     x1;
      static   int       x_division_index1;
      static   int       y_division_index1;

      prime_1=prime_addr(lower_bound);
      x1=prime_1->x;
      x_division_index1=prime_1->x_division_index;
      y_division_index1=prime_1->y_division_index;
      *j=lower_bound;
      up=upper_bound;
      down=lower_bound;
      do
        {
          finished=FALSE;
          while (! finished)
            if (up <= down)
              finished=TRUE;
            else
              {
                prime_1=prime_addr(up);
                if (prime_1->x > x1)
                  finished=TRUE;
                else
                  up--;
              };
          *j=up;
          if (up != down)
            {
              prime_1=prime_addr(down);
              prime_2=prime_addr(up);
              prime_1->x=prime_2->x;
              prime_1->x_division_index=prime_2->x_division_index;
              prime_1->y_division_index=prime_2->y_division_index;
              finished=FALSE;
              while (! finished)
                if (down >= up)
                  finished=TRUE;
                else
                  {
                    prime_1=prime_addr(down);
                    if (prime_1->x < x1)
                      finished=TRUE;
                    else
                      down++;
                  };
              *j=down;
              if (down != up)
                {
                  prime_1=prime_addr(up);
                  prime_2=prime_addr(down);
                  prime_1->x=prime_2->x;
                  prime_1->x_division_index=prime_2->x_division_index;
                  prime_1->y_division_index=prime_2->y_division_index;
                }
            }
        }
      while (down != up);
      prime_1=prime_addr(*j);
      prime_1->x=x1;
      prime_1->x_division_index=x_division_index1;
      prime_1->y_division_index=y_division_index1;
      return;
    }

static void quicksort(
  long lower_bound,
  long upper_bound)
    {
      long j;

      if (lower_bound < upper_bound)
        {
          titillate();
          rearrange(lower_bound,upper_bound,&j);
          quicksort(lower_bound,j-1l);
          quicksort(j+1l,upper_bound);
        }
      return;
    }

static void sort_back_to_front(
  long   num_primes)
    {
      quicksort(0l,num_primes-1l);
      return;
    }

static void pset(
  int  x,
  int  y,
  long color)
    {
      register int     bit_num;
      register int     byte_num;
      static   long    inverse;
      static   int     quotient;
      static   int     remainder;
      static   row_rec *row;
      static   int     x_dividend;
      static   int     y_dividend;

      y_dividend=y;
      x_dividend=x;
      inverse=0l;
      for (bit_num=0; bit_num < LOG2_NUM_X_PIXELS; bit_num++)
        {
          quotient=x_dividend/2;
          remainder=(x_dividend-2*quotient);
          inverse+=inverse;
          inverse+=((long) remainder);
          x_dividend=quotient;
          quotient=y_dividend/2;
          inverse+=inverse;
          inverse+=((long) (remainder^(y_dividend-2*quotient)));
          y_dividend=quotient;
        }
      byte_num=x/8;
      bit_num=x-8*byte_num;
      row=row_addr(y);
      if (inverse > color)
        (*row).column[byte_num]
         =((*row).column[byte_num] | black[bit_num]);
      else
        (*row).column[byte_num]
         =((*row).column[byte_num] & white[bit_num]);
      return;
    }

static void plot(
  int    num_x_divisions,
  int    num_y_divisions,
  long   num_primes,
  double y_prime_min,
  double y_prime_max,
  double z_prime_min,
  double z_prime_max,
  double aspect_ratio)
    {
      static   double     box_delta_x;
      static   double     box_delta_y;
      static   box_rec    box [4];
      static   int        box_num_1;
      static   int        box_num_2;
      static   double     box_x_intercept;
      static   int        box_x1;
      static   int        box_x2;
      static   int        box_y_max;
      static   int        box_y_min;
      static   double     box_y_offset;
      static   int        box_y1;
      static   long       color_num;
      static   int        intercept_count_mod_2;
      register int        line_x1;
      register int        line_x2;
      static   int        outside;
      static   double     pixels_per_unit;
      static   prime_rec  *prime;
      static   long       prime_num;
      static   vertex_rec vertex [4];
      static   int        x_division_num;
      static   long       x_prime_num;
      static   long       x_prime_num_mod_50;
      static   int        y_division_num;
      static   double     y_offset;
      static   double     y_out_max;
      static   double     z_offset;
      static   double     z_out_max;

      y_out_max=(double) (NUM_X_PIXELS-1);
      z_out_max=(double) (NUM_Y_PIXELS-1);
      if (aspect_ratio*z_out_max*(y_prime_max-y_prime_min)
       > y_out_max*(z_prime_max-z_prime_min))
        {
          pixels_per_unit
           =y_out_max/(aspect_ratio*(y_prime_max-y_prime_min));
          y_offset=0.0;
          z_offset
           =-(z_out_max-pixels_per_unit*(z_prime_max-z_prime_min))/2.0;
        }
      else
        if (aspect_ratio*z_out_max*(y_prime_max-y_prime_min)
         < y_out_max*(z_prime_max-z_prime_min))
          {
            pixels_per_unit=z_out_max/(z_prime_max-z_prime_min);
            y_offset=(y_out_max
             -aspect_ratio*pixels_per_unit*(y_prime_max-y_prime_min))/2.0;
            z_offset=0.0;
          }
        else
          {
            pixels_per_unit=1.0;
            y_offset=y_out_max/2.0;
            z_offset=-z_out_max/2.0;
          };
      x_prime_num_mod_50=0;
      for (x_prime_num=0l; x_prime_num < num_primes; x_prime_num++)
        {
          x_prime_num_mod_50++;
          if (x_prime_num_mod_50 >= 50)
            {
              titillate();
              x_prime_num_mod_50=0;
            }
          prime=prime_addr(x_prime_num);
          x_division_num=prime->x_division_index;
          if (x_division_num < (num_x_divisions-1))
            {
              y_division_num=prime->y_division_index;
              if (y_division_num < (num_y_divisions-1))
                {
                  prime_num
                   =((long) num_y_divisions)*((long) x_division_num)
                   +((long) y_division_num);
                  prime=prime_addr(prime_num);
                  color_num=(prime->color);
                  if (color_num < NUM_COLORS)
                    {
                      outside=(prime->outside_maze == (unsigned char) '\1');
                      vertex[0].y=(double) (prime->y);
                      vertex[0].z=(double) (prime->z);
                      prime_num+=((long) num_y_divisions);
                      prime=prime_addr(prime_num);
                      if (outside)
                        outside=(prime->outside_maze == (unsigned char) '\1');
                      vertex[1].y=(double) (prime->y);
                      vertex[1].z=(double) (prime->z);
                      prime_num++;
                      prime=prime_addr(prime_num);
                      if (outside)
                        outside=(prime->outside_maze == (unsigned char) '\1');
                      vertex[2].y=(double) (prime->y);
                      vertex[2].z=(double) (prime->z);
                      prime_num-=((long) num_y_divisions);
                      prime=prime_addr(prime_num);
                      if (outside)
                        outside=(prime->outside_maze == (unsigned char) '\1');
                      vertex[3].y=(double) (prime->y);
                      vertex[3].z=(double) (prime->z);
                      if (outside)
                        color_num=NUM_COLORS-1l;
                      for (box_num_1=0; box_num_1 < 4; box_num_1++)
                        {
                          box[box_num_1].x=(int) (y_offset
                           +pixels_per_unit*aspect_ratio
                           *(vertex[box_num_1].y-y_prime_min));
                          box[box_num_1].y=(int) (z_offset+z_out_max
                           -pixels_per_unit
                           *(vertex[box_num_1].z-z_prime_min));
                        }
                      box_y_min=box[0].y;
                      box_y_max=box_y_min;
                      for (box_num_1=1; box_num_1 < 4; box_num_1++)
                        {
                          if (box[box_num_1].y < box_y_min)
                            box_y_min=box[box_num_1].y;
                          if (box[box_num_1].y > box_y_max)
                            box_y_max=box[box_num_1].y;
                        }
                      for (box_y1=box_y_min; box_y1 <= box_y_max; ++box_y1)
                        {
                          intercept_count_mod_2=0;
                          box_num_2=1;
                          for (box_num_1=0; box_num_1 < 4; ++box_num_1)
                            {
                              if (box[box_num_1].y >= box_y1)
                                {
                                  if (box_y1 > box[box_num_2].y)
                                    {
                                      box_delta_y=(double)
                                       (box[box_num_2].y-box[box_num_1].y);
                                      box_delta_x=(double)
                                       (box[box_num_2].x-box[box_num_1].x);
                                      box_y_offset=(double)
                                       (box_y1-box[box_num_1].y);
                                      box_x_intercept
                                       =(double) (box[box_num_1].x);
                                      box_x1
                                       =(int) ((box_delta_x*box_y_offset)
                                       /box_delta_y+box_x_intercept);
                                      if (intercept_count_mod_2 == 0)
                                        box_x2=box_x1;
                                      else
                                        {
                                          if (box_x1 < box_x2)
                                            {
                                              line_x1=box_x1;
                                              line_x2=box_x2;
                                            }
                                          else
                                            {
                                              line_x1=box_x2;
                                              line_x2=box_x1;
                                            }
                                          pset(line_x1,box_y1,color_num);
                                          while (line_x1 < line_x2)
                                            {
                                              line_x1++;
                                              pset(line_x1,box_y1,color_num);
                                            }
                                        }
                                      intercept_count_mod_2
                                       =1-intercept_count_mod_2;
                                    }
                                }
                              else
                                {
                                  if (box_y1 <= box[box_num_2].y)
                                    {
                                      box_delta_y=(double)
                                       (box[box_num_2].y-box[box_num_1].y);
                                      box_delta_x=(double)
                                       (box[box_num_2].x-box[box_num_1].x);
                                      box_y_offset=(double)
                                       (box_y1-box[box_num_1].y);
                                      box_x_intercept
                                       =(double) (box[box_num_1].x);
                                      box_x1
                                       =(int) ((box_delta_x*box_y_offset)
                                       /box_delta_y+box_x_intercept);
                                      if (intercept_count_mod_2 == 0)
                                        box_x2=box_x1;
                                      else
                                        {
                                          if (box_x1 < box_x2)
                                            {
                                              line_x1=box_x1;
                                              line_x2=box_x2;
                                            }
                                          else
                                            {
                                              line_x1=box_x2;
                                              line_x2=box_x1;
                                            }
                                          pset(line_x1,box_y1,color_num);
                                          while (line_x1 < line_x2)
                                            {
                                              line_x1++;
                                              pset(line_x1,box_y1,color_num);
                                            }
                                        }
                                      intercept_count_mod_2
                                       =1-intercept_count_mod_2;
                                    }
                                }
                              box_num_2++;
                              if (box_num_2 >= 4)
                                box_num_2=0;
                            }
                        }
                    }
                }
            }
        }
      return;
    }

static double f(
  double x,
  double y)
    {
      register int    x_out = 0;
      register int    y_out = 0;
      static   double z = 0.0;

      y_out=(int) x;
      if (y_out < 0)
        z=0.0;
      else
        if (y_out > y_dot_max)
          z=0.0;
        else
          {
            x_out=(int) y;
            if (x_out < 0)
              z=0.0;
            else
              if (x_out > x_dot_max)
                z=0.0;
              else
                if (page[y_out][x_out] == 'W')
                  z=((double) (5*RESOLUTION));
                else
                  z=0.0;
          }
      return(z);
    }

static int external_to_maze(
  double x,
  double y)
    {
      static   int result = 0;
      static   int x_out = 0;
      register int x_next = 0;
      static   int y_out = 0;
      register int y_next = 0;

      result=FALSE;
      y_out=(int) x;
      if (y_out < 0)
        result=TRUE;
      else
        if (y_out > y_dot_max)
          result=TRUE;
        else
          {
            x_out=(int) y;
            if (x_out < 0)
              result=TRUE;
            else
              if (x_out > x_dot_max)
                result=TRUE;
              else
                {
                  if ((x_out/RESOLUTION != 3)
                  &&  (x_out/RESOLUTION != max_x-3))
                    {
                      x_next=x_out;
                      result=TRUE;
                      while((result) && (x_next >= 0)) 
                        {
                          if (page[y_out][x_next] == ' ')
                            x_next--;
                          else
                            result=FALSE;
                        }
                      if (! result)
                        {
                          x_next=x_out;
                          result=TRUE;
                          while((result) && (x_next <= x_dot_max)) 
                            {
                              if (page[y_out][x_next] == ' ')
                                x_next++;
                              else
                                result=FALSE;
                            }
                        }
                      if (! result)
                        {
                          y_next=y_out;
                          result=TRUE;
                          while((result) && (y_next >= 0)) 
                            {
                              if (page[y_next][x_out] == ' ')
                                y_next--;
                              else
                                result=FALSE;
                            }
                        }
                      if (! result)
                        {
                          y_next=y_out;
                          result=TRUE;
                          while((result) && (y_next <= y_dot_max))
                            {
                              if (page[y_next][x_out] == ' ')
                                y_next++;
                              else
                                result=FALSE;
                            }
                        }
                    }
                }
          }
      return(result);
    }
